#include <QMenuBar>
#include <QDebug>
#include <QDesktopServices>
#include <QtGui>
#include <QtWebKit>
#include <QtMaemo5>

#include <QtGui/QX11Info>
#include <X11/Xlib.h>
#include <X11/Xatom.h>

#include "contentwindow.h"

/* Got ViewportItem and GraphicsView from maemobrowser in the qt examples. The
 * whole point of this is to get a finger friendly web widget */

class ViewportItem : public QGraphicsWidget, public QAbstractKineticScroller {
	Q_OBJECT

	public:
		ViewportItem() : QGraphicsWidget(), QAbstractKineticScroller(), m_widget(0), m_ignoreEvents(false) {
			setFlag(QGraphicsItem::ItemHasNoContents, true);
			setFlag(QGraphicsItem::ItemClipsChildrenToShape, true);
			setFlag(QGraphicsItem::ItemClipsToShape, true);
			setAttribute(Qt::WA_OpaquePaintEvent, true);
			setFiltersChildEvents(true);
		}

		void setWidget(QGraphicsWidget *widget) {
			if (m_widget) {
				m_widget->setParentItem(0);
				delete m_widget;
			}
			m_widget = widget;
			if (m_widget) {
				m_widget->setParentItem(this);
				m_widget->setAttribute(Qt::WA_OpaquePaintEvent, true);

				if (qgraphicsitem_cast<QGraphicsWebView *>(m_widget)) {
					connect(m_widget, SIGNAL(loadProgress(int)), this, SLOT(resizeWebViewToFrame()));
					connect(m_widget, SIGNAL(loadFinished(bool)), this, SLOT(resizeWebViewToFrame()));
					resizeWebViewToFrame();
				}
			}
		}

	signals:
		void showNextEntry();
		void showPrevEntry();
		void showOriginal();
		void toggleStarred();
		void toggleShared();
		void toggleRead();

	protected:
		bool sceneEventFilter(QGraphicsItem *i, QEvent *e) {
			bool res = false;
			if (i && (i == m_widget) && !m_ignoreEvents && m_widget->isEnabled()) {
				switch (e->type()) {
				case QEvent::GraphicsSceneMousePress:
				case QEvent::GraphicsSceneMouseMove:
				case QEvent::GraphicsSceneMouseRelease:
					res = handleMouseEvent(static_cast<QGraphicsSceneMouseEvent *>(e));
					break;
				default:
					break;
				}
			}
			// prevent text selection and image dragging
			if (e->type() == QEvent::GraphicsSceneMouseMove)
				return true;
			return res ? true : QGraphicsWidget::sceneEventFilter(i, e);
		}

		void keyPressEvent(QKeyEvent *e) {
			int x, y, w, h;

			switch(e->key()) {
			case Qt::Key_Down:
				y = m_widget->y() - 50;
				h = (m_widget->size() - size()).height();
				m_widget->setY(y < -h ? -h : y);
				break;

			case Qt::Key_Up:
				y = m_widget->y() + 50;
				m_widget->setY(y > 0 ? 0 : y);
				break;

			case Qt::Key_Right:
				x = m_widget->x() - 50;
				w = (m_widget->size() - size()).width();
				m_widget->setX(x < -w ? -w : x);
				break;

			case Qt::Key_Left:
				x = m_widget->x() + 50;
				m_widget->setX(x > 0 ? 0 : x);
				break;

			case Qt::Key_F7:
				if(qgraphicsitem_cast<QGraphicsWebView *>(m_widget)) {
					QGraphicsWebView *wv = (QGraphicsWebView *)m_widget;
					wv->setZoomFactor(wv->zoomFactor() * 1.25);
					resizeWebViewToFrame();
				}
				break;

			case Qt::Key_F8:
				if(qgraphicsitem_cast<QGraphicsWebView *>(m_widget)) {
					QGraphicsWebView *wv = (QGraphicsWebView *)m_widget;
					wv->setZoomFactor(wv->zoomFactor() / 1.25);
					resizeWebViewToFrame();
				}
				break;

			case Qt::Key_J:
			case Qt::Key_N:
				emit showNextEntry();
				break;

			case Qt::Key_K:
			case Qt::Key_P:
				emit showPrevEntry();
				break;

			case Qt::Key_Space:
				if(e->modifiers() & Qt::ShiftModifier) {
					y = m_widget->y() + size().height();
					m_widget->setY(y > 0 ? 0 : y);
				}
				else {
					y = m_widget->y() - size().height();;
					h = (m_widget->size() - size()).height();
					m_widget->setY(y < -h ? -h : y);
				}
				break;

			case Qt::Key_S:
				if(e->modifiers() & Qt::ShiftModifier)
					emit toggleShared();
				else
					emit toggleStarred();
				break;

			case Qt::Key_V:
				emit showOriginal();
				break;

			case Qt::Key_M:
				emit toggleRead();
				break;
			}
		}

		QSize viewportSize() const {
			return size().toSize();
		}

		QPoint maximumScrollPosition() const {
			QSizeF s = m_widget ? m_widget->size() - size() : QSize(0, 0);
			return QPoint(qMax(0, int(s.width())), qMax(0, int(s.height())));
		}

		QPoint scrollPosition() const {
			return m_widget ? -m_widget->pos().toPoint() + m_overShoot : QPoint();
		}

		void setScrollPosition(const QPoint &p, const QPoint &overShoot) {
			m_overShoot = overShoot;
			if (m_widget)
				m_widget->setPos(-p + m_overShoot);
		}

		void sendEvent(QGraphicsItem *i, QEvent *e) {
			m_ignoreEvents = true;
			scene()->sendEvent(i, e);
			m_ignoreEvents = false;
		}

	private slots:
		void resizeWebViewToFrame() {
			if (QGraphicsWebView *view = qgraphicsitem_cast<QGraphicsWebView *>(m_widget)) {
				if (view->page() && view->page()->mainFrame()) {
					QSizeF s = view->page()->mainFrame()->contentsSize();
					/* TODO: Figure out the proper way to
					 * get this 800 pixels. */
					QSizeF s2 = size();
					s2.setWidth(800);
					s = s.expandedTo(s2);
					view->setGeometry(QRectF(view->geometry().topLeft(), s));
				}
			}
		}

	private:
		QGraphicsWidget *m_widget;
		bool m_ignoreEvents;
		QPoint m_overShoot;
};

class GraphicsView : public QGraphicsView {
	Q_OBJECT
	
	public:
		GraphicsView() : QGraphicsView(new QGraphicsScene()), viewport(0) {
			setViewportUpdateMode(QGraphicsView::BoundingRectViewportUpdate);
			setOptimizationFlags(QGraphicsView::DontSavePainterState);

			setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
			setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

			setFrameShape(QFrame::NoFrame);
			setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

			viewport = new ViewportItem();
			scene()->addItem(viewport);
		}

		ViewportItem *viewportItem() const {
			return viewport;
		}

	protected:
		void resizeEvent(QResizeEvent *e) {
			QGraphicsView::resizeEvent(e);
			setUpdatesEnabled(false);

			if (!viewport)
				return;

			QRectF rect(QPointF(0, 0), size());
			scene()->setSceneRect(rect);

			viewport->setGeometry(rect);
			setUpdatesEnabled(true);
			update();
		}

	private:
		ViewportItem *viewport;
};

ContentWindow::ContentWindow(QWidget *parent, Entry *e) : QMainWindow(parent) {
	setAttribute(Qt::WA_Maemo5StackedWindow);

	QWebSettings::globalSettings()->setAttribute(QWebSettings::PluginsEnabled, true);

	starred = new QAction(tr("Starred"), this);
	starred->setCheckable(true);
	menuBar()->addAction(starred);

	shared = new QAction(tr("Shared"), this);
	shared->setCheckable(true);
	menuBar()->addAction(shared);

	keepUnread = new QAction(tr("Keep unread"), this);
	keepUnread->setCheckable(true);
	menuBar()->addAction(keepUnread);

	menuBar()->addAction(tr("See original"), this, SLOT(seeOriginal()));

	GraphicsView *gv = new GraphicsView();
	webview = new QGraphicsWebView();
	gv->viewportItem()->setWidget(webview);

	webview->settings()->setFontSize(QWebSettings::MinimumFontSize, 22);
	webview->page()->setLinkDelegationPolicy(QWebPage::DelegateAllLinks);

	connect(webview, SIGNAL(loadFinished(bool)), SLOT(loadFinished(bool)));
	connect(webview, SIGNAL(loadStarted()), SLOT(loadStarted()));
	connect(webview->page(), SIGNAL(linkClicked(const QUrl &)), SLOT(showLink(const QUrl &)));
	connect(gv->viewportItem(), SIGNAL(showNextEntry()), SIGNAL(showNextEntry()));
	connect(gv->viewportItem(), SIGNAL(showPrevEntry()), SIGNAL(showPrevEntry()));
	connect(gv->viewportItem(), SIGNAL(showOriginal()), SLOT(seeOriginal()));
	connect(gv->viewportItem(), SIGNAL(toggleStarred()), SLOT(toggleStarred()));
	connect(gv->viewportItem(), SIGNAL(toggleShared()), SLOT(toggleShared()));
	connect(gv->viewportItem(), SIGNAL(toggleRead()), SLOT(toggleRead()));

	setCentralWidget(gv);
	gv->viewportItem()->setFocus();
	gv->viewportItem()->grabKeyboard();

	grabZoomKeys(true);

	entry = NULL;

	showEntry(e);
}

ContentWindow::~ContentWindow() {
	delete(webview);
}

void ContentWindow::loadStarted() {
	setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
}

void ContentWindow::loadFinished(bool) {
	setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
}

void ContentWindow::showLink(const QUrl &url) {
	/* Attempt to launch external browser */
	if(!QDesktopServices::openUrl(url))
		webview->setUrl(url); /* Failed... Show inline */
}

void ContentWindow::seeOriginal() {
	showLink(entry->link);
}

void ContentWindow::toggleStarred() {
	starred->toggle();
	if(starred->isChecked()) {
		QMaemo5InformationBox::information(this, "Starred",
			QMaemo5InformationBox::DefaultTimeout);
	}
	else {
		QMaemo5InformationBox::information(this, "Star removed",
			QMaemo5InformationBox::DefaultTimeout);
	}
}

void ContentWindow::toggleShared() {
	shared->toggle();
	if(shared->isChecked()) {
		QMaemo5InformationBox::information(this, "Shared",
			QMaemo5InformationBox::DefaultTimeout);
	}
	else {
		QMaemo5InformationBox::information(this, "Unshared",
			QMaemo5InformationBox::DefaultTimeout);
	}
}

void ContentWindow::toggleRead() {
	if(!keepUnread->isEnabled()) {
		QMaemo5InformationBox::information(this, "Read state locked",
			QMaemo5InformationBox::DefaultTimeout);
		return;
	}

	keepUnread->toggle();
	if(keepUnread->isChecked()) {
		QMaemo5InformationBox::information(this, "Marked unread",
			QMaemo5InformationBox::DefaultTimeout);
	}
	else {
		QMaemo5InformationBox::information(this, "Marked read",
			QMaemo5InformationBox::DefaultTimeout);
	}
}

void ContentWindow::showEntry(Entry *e) {
	if(entry) {
		/* Store settings of previously shown entry */
		entry->markRead(!keepUnread->isChecked());
		entry->markStar(starred->isChecked());
		entry->markShared(shared->isChecked());
	}

	entry = e;

	starred->setChecked((entry->flags & ENTRY_FLAG_STARRED));
	shared->setChecked((entry->flags & ENTRY_FLAG_SHARED));
	keepUnread->setChecked(false);
	keepUnread->setEnabled((entry->flags & ENTRY_FLAG_LOCKED) == 0);

	setWindowTitle(entry->title);

	webview->setHtml(entry->content);
}

void ContentWindow::closeEvent(QCloseEvent *event) {
	grabZoomKeys(false);
	entry->markRead(!keepUnread->isChecked());
	entry->markStar(starred->isChecked());
	entry->markShared(shared->isChecked());
	QMainWindow::closeEvent(event);
}

void ContentWindow::grabZoomKeys(bool grab) {
	if(!winId())
		return;

	unsigned long val = (grab) ? 1 : 0;
	Atom atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", False);
	if(!atom)
		return;

	XChangeProperty (QX11Info::display(), winId(), atom, XA_INTEGER, 32, PropModeReplace,
		reinterpret_cast<unsigned char *>(&val), 1);
}

#include "contentwindow.moc"
