#include <QMenuBar>
#include <QDebug>
#include <QDesktopServices>
#include <QtGui>
#include <QtWebKit>
#include "contentwindow.h"

/* Got ViewportItem and GraphicsView from maemobrowser in the qt examples. The
 * whole point of this is to get a finger friendly web widget */

class ViewportItem : public QGraphicsWidget, public QAbstractKineticScroller {
	Q_OBJECT

	public:
		ViewportItem() : QGraphicsWidget(), QAbstractKineticScroller(), m_widget(0), m_ignoreEvents(false) {
			setFlag(QGraphicsItem::ItemHasNoContents, true);
			setFlag(QGraphicsItem::ItemClipsChildrenToShape, true);
			setFlag(QGraphicsItem::ItemClipsToShape, true);
			setAttribute(Qt::WA_OpaquePaintEvent, true);
			setFiltersChildEvents(true);
		}

		void setWidget(QGraphicsWidget *widget) {
			if (m_widget) {
				m_widget->setParentItem(0);
				delete m_widget;
			}
			m_widget = widget;
			if (m_widget) {
				m_widget->setParentItem(this);
				m_widget->setAttribute(Qt::WA_OpaquePaintEvent, true);

				if (qgraphicsitem_cast<QGraphicsWebView *>(m_widget)) {
					connect(m_widget, SIGNAL(loadProgress(int)), this, SLOT(resizeWebViewToFrame()));
					connect(m_widget, SIGNAL(loadFinished(bool)), this, SLOT(resizeWebViewToFrame()));
					resizeWebViewToFrame();
				}
			}
		}

	protected:
		bool sceneEventFilter(QGraphicsItem *i, QEvent *e) {
			bool res = false;
			if (i && (i == m_widget) && !m_ignoreEvents && m_widget->isEnabled()) {
				switch (e->type()) {
				case QEvent::GraphicsSceneMousePress:
				case QEvent::GraphicsSceneMouseMove:
				case QEvent::GraphicsSceneMouseRelease:
				/*case QEvent::GraphicsSceneMouseDoubleClick:*/ {
					res = handleMouseEvent(static_cast<QGraphicsSceneMouseEvent *>(e));
					break;
				}
				default:
					break;
				}
			}
			// prevent text selection and image dragging
			if (e->type() == QEvent::GraphicsSceneMouseMove)
				return true;
			return res ? true : QGraphicsWidget::sceneEventFilter(i, e);
		}

		QSize viewportSize() const {
			return size().toSize();
		}

		QPoint maximumScrollPosition() const {
			QSizeF s = m_widget ? m_widget->size() - size() : QSize(0, 0);
			return QPoint(qMax(0, int(s.width())), qMax(0, int(s.height())));
		}

		QPoint scrollPosition() const {
			return m_widget ? -m_widget->pos().toPoint() + m_overShoot : QPoint();
		}

		void setScrollPosition(const QPoint &p, const QPoint &overShoot) {
			m_overShoot = overShoot;
			if (m_widget)
				m_widget->setPos(-p + m_overShoot);
		}

		void cancelLeftMouseButtonPress(const QPoint & /*globalPressPos*/) {
		}

		void sendEvent(QGraphicsItem *i, QEvent *e) {
			m_ignoreEvents = true;
			scene()->sendEvent(i, e);
			m_ignoreEvents = false;
		}

	private slots:
		void resizeWebViewToFrame() {
			if (QGraphicsWebView *view = qgraphicsitem_cast<QGraphicsWebView *>(m_widget)) {
				if (view->page() && view->page()->mainFrame()) {
					QSizeF s = view->page()->mainFrame()->contentsSize();
					/* TODO: Figure out the proper way to
					 * get this 800 pixels. */
					QSizeF s2 = size();
					s2.setWidth(800);
					s = s.expandedTo(s2);
					view->setGeometry(QRectF(view->geometry().topLeft(), s));
				}
			}
		}

	private:
		QGraphicsWidget *m_widget;
		bool m_ignoreEvents;
		QPoint m_overShoot;
};

class GraphicsView : public QGraphicsView {
	Q_OBJECT
	
	public:
		GraphicsView() : QGraphicsView(new QGraphicsScene()), viewport(0) {
			setViewportUpdateMode(QGraphicsView::BoundingRectViewportUpdate);
			setOptimizationFlags(QGraphicsView::DontSavePainterState);

			setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
			setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

			setFrameShape(QFrame::NoFrame);
			setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

			viewport = new ViewportItem();
			scene()->addItem(viewport);
		}

		ViewportItem *viewportItem() const {
			return viewport;
		}

	protected:
		void resizeEvent(QResizeEvent *e) {
			QGraphicsView::resizeEvent(e);
			setUpdatesEnabled(false);

			if (!viewport)
				return;

			QRectF rect(QPointF(0, 0), size());
			scene()->setSceneRect(rect);

			viewport->setGeometry(rect);
			setUpdatesEnabled(true);
			update();
		}

	private:
		ViewportItem *viewport;
};

ContentWindow::ContentWindow(QWidget *parent, Entry *e) : QMainWindow(parent) {
	setAttribute(Qt::WA_Maemo5StackedWindow);

	QWebSettings::globalSettings()->setAttribute(QWebSettings::PluginsEnabled, true);

	entry = e;

	starred = new QAction(tr("Starred"), this);
	starred->setCheckable(true);
	starred->setChecked((entry->flags & ENTRY_FLAG_STARRED));
	menuBar()->addAction(starred);

	keepUnread = new QAction(tr("Keep unread"), this);
	keepUnread->setCheckable(true);
	keepUnread->setEnabled((entry->flags & ENTRY_FLAG_LOCKED) == 0);
	menuBar()->addAction(keepUnread);

	menuBar()->addAction(tr("See original"), this, SLOT(seeOriginal()));

	setWindowTitle(entry->title);

	GraphicsView *gv = new GraphicsView();
	webview = new QGraphicsWebView();
	gv->viewportItem()->setWidget(webview);

	/* TODO: Configurable text size ?? */
	webview->settings()->setFontSize(QWebSettings::MinimumFontSize, 22);

	connect(webview, SIGNAL(loadFinished(bool)), SLOT(loadFinished(bool)));
	connect(webview, SIGNAL(loadStarted()), SLOT(loadStarted()));

	webview->setHtml(entry->content);

	setCentralWidget(gv);
}

ContentWindow::~ContentWindow() {
	delete(webview);
}

void ContentWindow::loadStarted() {
	setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
}

void ContentWindow::loadFinished(bool) {
	setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
}

void ContentWindow::seeOriginal() {
	/* Attempt to launch external browser */
	if(!QDesktopServices::openUrl(entry->link))
		webview->setUrl(entry->link); /* Failed... Show inline */
}

void ContentWindow::closeEvent(QCloseEvent *event) {
	entry->markRead(!keepUnread->isChecked());
	entry->markStar(starred->isChecked());
	QMainWindow::closeEvent(event);
}

#include "contentwindow.moc"
