#include "contactsdatamodel.h"

ContactsDataModel::ContactsDataModel(QObject *parent)
     : QAbstractTableModel(parent)
{
    m_Contacts = NULL;
}

ContactsDataModel::ContactsDataModel(QObject *parent, QList<Contact*>* contacts)
     : QAbstractTableModel(parent)
{
    m_defaultImage = (new QPixmap(":/images/contacts.png"))->scaled(48, 48, Qt::KeepAspectRatio);
    m_Contacts = contacts;
}

ContactsDataModel::~ContactsDataModel()
{
    clear();
}

int ContactsDataModel::rowCount(const QModelIndex &) const
{
    if (m_Contacts)
        return m_Contacts->count();
    return 0;
}

int ContactsDataModel::columnCount(const QModelIndex &) const
{
    return 4;
}

QVariant ContactsDataModel::headerData(int, Qt::Orientation, int) const
{
     return QVariant();
}

QVariant ContactsDataModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid() || !m_Contacts)
        return QVariant();

    if (index.row() >= m_Contacts->count() || index.row() < 0)
        return QVariant();

    Contact* c = m_Contacts->at(index.row());
    if (!c)
        return QVariant();

    if (role == Qt::CheckStateRole){
        if (index.column() == 0)
            return c->Selected ? Qt::Checked : Qt::Unchecked;
    }else if (role == Qt::DecorationRole) {
        if (index.column() == 1){
            if (c->Thumbnail)
                return QPixmap::fromImage(*c->Thumbnail);
            else
                return m_defaultImage;
        }
    }else if (role == Qt::TextAlignmentRole) {
        if (index.column() == 1)
            return Qt::AlignCenter;
        else if (index.column() == 3)
            return Qt::AlignVCenter + Qt::AlignRight;
        return Qt::AlignVCenter + Qt::AlignLeft;
    }else if (role == Qt::DisplayRole) {
        if (index.column() == 2){
            //if (!m_Highlight.isEmpty())
            //    return c->Name.replace(m_Highlight, QString("<b>%1</b>").arg(m_Highlight));
            return c->Name;
        }else if (index.column() == 3)
            return c->Number;
    }
    return QVariant();
}

Qt::ItemFlags ContactsDataModel::flags(const QModelIndex& index) const
{
    if (!index.isValid() || !m_Contacts)
        return Qt::NoItemFlags;

    if (index.row() >= m_Contacts->count() || index.row() < 0)
        return Qt::NoItemFlags;

    if (index.column() == 0)
        return Qt::ItemIsSelectable | Qt::ItemIsUserCheckable | Qt::ItemIsEnabled;
    else
        return Qt::ItemIsSelectable | Qt::ItemIsEnabled;
}

bool ContactsDataModel::setData(const QModelIndex& index, const QVariant& value, int)
{
    if (!index.isValid() || !m_Contacts)
        return false;

    if (index.row() >= m_Contacts->count() || index.row() < 0)
        return false;

    Contact* c = m_Contacts->at(index.row());
    if (c){
        if (index.column() == 0){
            c->Selected = value.toBool();
            emit dataChanged(index, index);
            return true;
        }
    }
    return false;
}

void ContactsDataModel::clear()
{
    if (m_Contacts != NULL && m_Contacts->count() > 0){
        beginRemoveRows(QModelIndex(), 0, rowCount(QModelIndex()) - 1);
        m_Contacts->clear();
        endRemoveRows();
    }
}

void ContactsDataModel::toggleSelect(QModelIndex index)
{
    if (!index.isValid() || !m_Contacts)
        return;
    setData(this->index(index.row(), 0), !m_Contacts->at(index.row())->Selected);
}

void ContactsDataModel::selectAll(bool select)
{
    if (m_Contacts){
        for (int i=0; i<m_Contacts->count(); i++)
            setData(this->index(i, 0), select);
    }
}

bool ContactsDataModel::removeRows(int row, int count, const QModelIndex &)
{
    if (!m_Contacts)
        return false;

    if (row >= m_Contacts->count() || row < 0)
        return false;

    beginRemoveRows(QModelIndex(), row, row+count-1);
    for (int i=0; i<count; i++)
        m_Contacts->removeAt(row);
    endRemoveRows();

    return true;
}

QList<Contact*> ContactsDataModel::getSelected()
{    
    QList<Contact*> res;
    if (m_Contacts){
        foreach (Contact* c, *m_Contacts){
            if (c->Selected)
                res.append(c);
        }
    }
    return res;
}

Contact* ContactsDataModel::getContact(QModelIndex index)
{
    if (!index.isValid() || !m_Contacts)
        return NULL;

    return m_Contacts->at(index.row());
}

int ContactsDataModel::search(QTableView* tableView, QString text)
{
    m_Highlight = text;
    int row = 0;
    int count = 0;

    if (m_Contacts){
        foreach (Contact* c, *m_Contacts)
        {
            if (!c->Name.contains(text, Qt::CaseInsensitive)){
                tableView->setRowHidden(row, true);
            }else{
                tableView->setRowHidden(row, false);
                count++;
            }
            row++;
        }
    }
    return count;
}
