/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"
#include <math.h>

#if defined(USE_MAEMO) && (MAEMO_VERSION_MAJOR >= 5)
#include <hildon/hildon-entry.h>
#endif

#define STR_NAN  "----"

typedef struct {
  appdata_t *appdata;

  GtkWidget *lon1, *lat1, *dist1, *dir1;
  GtkWidget *lon2, *lat2;

  GtkWidget *distance, *proj_lat, *proj_lon;
} math_dialog_state_t;

static gboolean mark(GtkWidget *widget, gboolean valid) {
  gtk_widget_set_state(widget, valid?GTK_STATE_NORMAL:TAG_STATE);
  return valid;
}

/* a label that is colored red when being "active" */
static GtkWidget *gtk_red_label_new(char *str) {
  GdkColor color;
  GtkWidget *widget = left_label_new(str);
  gdk_color_parse("#ff0000", &color);
  gtk_widget_modify_fg(widget, TAG_STATE, &color);
  return widget;
}

static void on_calc_clicked(GtkButton *button, gpointer user_data) {
  math_dialog_state_t *state = (math_dialog_state_t*)user_data;
  pos_t pos1, pos2;
  gboolean pos1_ok = FALSE, pos2_ok = FALSE;
  float dist1;
  gboolean dist1_ok = FALSE;
  float dir1;
  gboolean dir1_ok = FALSE;

  /* parse input */
  pos1.lat = lat_entry_get(state->lat1);
  pos1.lon = lon_entry_get(state->lon1);
  if(!isnan(pos1.lat) && !isnan(pos1.lon)) pos1_ok = TRUE;

  pos2.lat = lat_entry_get(state->lat2);
  pos2.lon = lon_entry_get(state->lon2);
  if(!isnan(pos2.lat) && !isnan(pos2.lon)) pos2_ok = TRUE;

  dist1 = dist_entry_get(state->dist1, state->appdata->imperial);
  if(!isnan(dist1)) dist1_ok = TRUE;

  dir1 = angle_entry_get(state->dir1);
  if(!isnan(dir1)) dir1_ok = TRUE;

  /* ------------------- do all calculations ------------------- */


  /* ------------------- distance of coo1 and coo2  ------------------- */
  if(mark(state->distance, pos1_ok && pos2_ok)) {
    char str[32];
    float dist = gpx_pos_get_distance(pos1, pos2, state->appdata->imperial);  
    distance_str(str, sizeof(str), dist, state->appdata->imperial);

    gtk_label_set_text(GTK_LABEL(state->distance), str);
  } else
    gtk_label_set_text(GTK_LABEL(state->distance), STR_NAN);

  // N 53° 09.033'  W 001° 50.666' 100km / 30° = N 53° 55.616, W001° 04.850
  /* ------------------- coordinate projection ---------------- */
  mark(state->proj_lat, pos1_ok && dist1_ok && dir1_ok);
  if(mark(state->proj_lon, pos1_ok && dist1_ok && dir1_ok)) {
    pos_t pro;

    /* get great circle radius in miles/kilometers */
    float gcrad = state->appdata->imperial?3959.0:6371.0;

    // from: http://www.movable-type.co.uk/scripts/latlong.html
    pro.lat = asin(sin(pos1.lat/180*M_PI) * cos(dist1/gcrad) + 
		   cos(pos1.lat/180*M_PI) * sin(dist1/gcrad) * 
		   cos(dir1/180*M_PI) )/M_PI*180;
    pro.lon = pos1.lon + atan2(sin(dir1/180*M_PI)*sin(dist1/gcrad)*
			       cos(pos1.lat/180*M_PI), 
			       cos(dist1/gcrad)-sin(pos1.lat/180*M_PI)*
			       sin(pro.lat/180*M_PI))/M_PI*180;
    pro.lon = fmodf(pro.lon+180,360) - 180;  // normalise to -180...+180

    lat_label_set(state->proj_lat, pro.lat);
    lon_label_set(state->proj_lon, pro.lon);

    if(!isnan(pro.lat) && !isnan(pro.lon)) 
      state->appdata->geomath = pro;
  } else {
    gtk_label_set_text(GTK_LABEL(state->proj_lat), STR_NAN);
    gtk_label_set_text(GTK_LABEL(state->proj_lon), STR_NAN);
  }
}

static void table_attach_button(GtkWidget *table, GtkWidget *button, int x, int y) {
#ifndef FREMANTLE
  gtk_table_attach_defaults(GTK_TABLE(table), button, x, x+1, y, y+1);
#else
  gtk_table_attach(GTK_TABLE(table), button, x, x+1, y, y+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
#endif
}

void geomath_dialog(appdata_t *appdata) {
  static pos_t pos1 = { 0.0, 0.0 }, pos2 = { 0.0, 0.0 };
  static float dist1 = 0.0;
  static float dir1 = 0.0;
  static gboolean is_imperial = FALSE;

  math_dialog_state_t state;

  /* this is quite ugly. It would be nice to run the entire system on */
  /* one specific system (e.g. metric) and only convert for in- and output */
  if(!appdata->imperial && is_imperial)
    dist1 *= 6371.0/3959.0;   /* we just switched to metric */
  if(appdata->imperial && !is_imperial)
    dist1 *= 3959.0/6371.0;   /* we just switched to imperial */
  is_imperial = appdata->imperial;

  state.appdata = appdata;

#ifdef USE_MAEMO
  if(appdata->cur_cache) 
    printf("current cache is %s\n", appdata->cur_cache->id);
  else
    printf("no current cache\n");
#endif

  GtkWidget *dialog = gtk_dialog_new_with_buttons(_("Geomath"),
                          GTK_WINDOW(appdata->window),
			  //      GTK_DIALOG_NO_SEPARATOR | 
			  GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                          GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
                          NULL);

#if defined(USE_MAEMO) && defined(HILDON_HELP)
  hildon_help_dialog_help_enable(GTK_DIALOG(dialog), HELP_ID_GEOMATH, 
				 appdata->osso_context);
#endif

  gtk_window_set_default_size(GTK_WINDOW(dialog), DIALOG_WIDTH, DIALOG_HEIGHT);

  /* do geomath dialog */
  GtkWidget *hbox = gtk_hbox_new(FALSE, 0);

  /* ------------------------- input area ------------------------- */
  GtkWidget *table = gtk_table_new(5, 5, FALSE);
  gtk_table_set_col_spacing(GTK_TABLE(table), 0, 20);
  gtk_table_set_col_spacing(GTK_TABLE(table), 2, 20);

  gtk_table_attach_defaults(GTK_TABLE(table), left_label_new(_("Latitude:")),              0, 1, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), left_label_new(_("Longitude:")),             0, 1, 2, 3);

#ifndef FREMANTLE
  gtk_table_attach_defaults(GTK_TABLE(table), gtk_label_new(_("Coordinate 1")),           1, 3, 0, 1);
#endif
  table_attach_button(table, state.lat1 = lat_entry_new(pos1.lat),        1, 1);
  table_attach_button(table, state.lon1 = lon_entry_new(pos1.lon),        1, 2);
  table_attach_button(table, preset_coordinate_picker(appdata, state.lat1, state.lon1),  2,1);

#ifndef FREMANTLE
  gtk_table_attach_defaults(GTK_TABLE(table), gtk_label_new(_("Coordinate 2")),           3, 5, 0, 1);
#endif
  table_attach_button(table, state.lat2 = lat_entry_new(pos2.lat),        3, 1);
  table_attach_button(table, state.lon2 = lon_entry_new(pos2.lon),        3, 2);
  table_attach_button(table, preset_coordinate_picker(appdata, state.lat2, state.lon2),  4, 1);

  gtk_table_attach_defaults(GTK_TABLE(table), left_label_new(_("Distance:")),              0, 1, 3, 4);
  table_attach_button(table, state.dist1 = dist_entry_new(dist1, appdata->imperial), 1,3);

  gtk_table_attach_defaults(GTK_TABLE(table), left_label_new(_("Direction:")),             0, 1, 4, 5);
  table_attach_button(table, state.dir1 = angle_entry_new(dir1),          1,4);

  gtk_box_pack_start(GTK_BOX(hbox), table, TRUE, TRUE, 0);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, TRUE, TRUE, 0);

  /* ------------------------- do-it-button ------------------------- */

  GtkWidget *button = gtk_button_new_with_label(_("Calculate!"));
  g_signal_connect(button, "clicked", (GCallback)on_calc_clicked, &state);

#ifdef FREMANTLE
  hildon_gtk_widget_set_theme_size(button, 
		   (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
  gtk_table_attach(GTK_TABLE(table), button, 3,5,3,5, GTK_EXPAND, GTK_EXPAND, 0, 0);
#else
  /* in non-maemo5 the button has its own row */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, TRUE, FALSE, 0);
#endif  

  /* ------------------------- output area ------------------------- */
  
  table = gtk_table_new(3, 3, FALSE);

  gtk_table_attach_defaults(GTK_TABLE(table), left_label_new(_("Distance = ")),
			    0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), state.distance = gtk_red_label_new(STR_NAN),
			    1, 3, 0, 1);

  gtk_table_attach_defaults(GTK_TABLE(table), left_label_new(_("Projection = ")),
			    0, 1, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), state.proj_lat = gtk_red_label_new(STR_NAN),
			    1, 2, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), state.proj_lon = gtk_red_label_new(STR_NAN),
			    2, 3, 1, 2);
#if 0
  gtk_table_attach_defaults(GTK_TABLE(table), left_label_new(_("Middle = ")),
			    0, 1, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), state.middle_lat = gtk_red_label_new(STR_NAN),
			    1, 2, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), state.middle_lon = gtk_red_label_new(STR_NAN),
			    2, 3, 2, 3);
#endif

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), table, TRUE, TRUE, 0);
  

  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_CLOSE);

  gtk_widget_show_all(dialog);
  gtk_dialog_run(GTK_DIALOG(dialog));

  /* copy values back to local static variables so they re-appear if */
  /* the dialog is re-opened, convert illegal values (NAN) to 0 */

  pos1.lat = lat_entry_get(state.lat1); if(isnan(pos1.lat)) pos1.lat=0; 
  pos1.lon = lon_entry_get(state.lon1); if(isnan(pos1.lon)) pos1.lon=0; 
  pos2.lat = lat_entry_get(state.lat2); if(isnan(pos2.lat)) pos2.lat=0; 
  pos2.lon = lon_entry_get(state.lon2); if(isnan(pos2.lon)) pos2.lon=0; 
  dist1 = dist_entry_get(state.dist1, state.appdata->imperial); 
  if(isnan(dist1)) dist1=0; 
  dir1 = angle_entry_get(state.dir1); if(isnan(dir1)) dir1=0; 

  gtk_widget_destroy(dialog);
}
