/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"
#include <math.h>
#include <string.h>

#if MAEMO_VERSION_MAJOR == 5
#include <gdk/gdkx.h>
#include <X11/Xatom.h>
#endif

static GtkWidget *cache_description(appdata_t *appdata, cache_t *cache) {
  return html_view(appdata, cache->long_description, 
	   cache->long_is_html?HTML_HTML:HTML_PLAIN_TEXT, TRUE, cache, NULL);
}

#ifndef USE_MAEMO  // maemos touchscreen doesn't support tooltips
static const char *cache_type_tip[] = {
  "Traditional Cache", "Multicache",    "Mystery/Unknown Cache", 
  "Virtual Cache",     "Webcam Cache",  "Event Cache", 
  "Letterbox Hybrid",  "Earthcache",    "Wherigo Cache", 
  "Mega-Event Cache",  "Cache-In-Trash-Out Cache"
};

static const char *cache_size_tip[] = {
  "Regular Container", "Small Container", "Micro", 
  "Other Container", "Container not chosen", "Large Container", 
  "Virtual Container" 
};
#endif

static const char *cache_size_name[] = {
  "Regular", "Small", "Micro", "Other", "Not chosen", "Large", "Virtual" 
};

void bearing_fill_hbox(GtkWidget *hbox, 
		       appdata_t *appdata, pos_t refpos, pos_t pos) {
  char str[32];

  if(!isnan(pos.lat) && !isnan(pos.lon)) {
    gtk_box_pack_start(GTK_BOX(hbox), gtk_image_new_from_pixbuf(
	       		icon_bearing(refpos, pos)),TRUE,FALSE,0);

    if(!isnan(refpos.lat) && !isnan(refpos.lon)) {
      snprintf(str, sizeof(str), _("%.1f°"), 
	       gpx_pos_get_bearing(refpos, pos));
      gtk_box_pack_start_defaults(GTK_BOX(hbox), GTK_LABEL_SMALL(str));
      gpx_pos_get_distance_str(str, sizeof(str), 
			     refpos, pos, appdata->imperial);
      gtk_box_pack_start_defaults(GTK_BOX(hbox), GTK_LABEL_SMALL("  "));
      gtk_box_pack_start(GTK_BOX(hbox), 
          gtk_label_attrib(str, SIZE_SMALL, STRIKETHROUGH_NONE),TRUE,FALSE,0);
    } else
      gtk_box_pack_start(GTK_BOX(hbox), 
		 gtk_label_attrib(_("(no position)"), 
		 SIZE_SMALL, STRIKETHROUGH_NONE),TRUE,FALSE,0);
  } else
    gtk_box_pack_start(GTK_BOX(hbox), 
		       gtk_label_attrib(_("(invalid position in notes)"), 
		SIZE_SMALL, STRIKETHROUGH_NONE),TRUE,FALSE,0);
}

/* this function sets everything related to the coordinate. used to */
/* cope with the user setting a new "note coordinate" */
void overview_coordinate_update(cache_context_t *context) {
  if(!context->notes.modified)
    return;
  
  /* update position labels */
  int strike = notes_get_override(context)?STRIKETHROUGH:STRIKETHROUGH_NONE;
  lat_label_attrib_set(context->pos_lat_label, 
		       context->cache->pos.lat, SIZE_BIG, strike);
  lon_label_attrib_set(context->pos_lon_label, 
		       context->cache->pos.lon, SIZE_BIG, strike);
  
  /* remove enire hbox and build a new one */
  gtk_container_foreach(GTK_CONTAINER(context->bearing_hbox), 
			(GtkCallback)gtk_widget_destroy, NULL);
  
  /* update distance/etc */
  if(!isnan(context->cache->pos.lat) && 
     !isnan(context->cache->pos.lon)) 
    bearing_fill_hbox(context->bearing_hbox, context->appdata, 
		      *get_pos(context->appdata), notes_get_pos(context)); 
  
  gtk_widget_show_all(context->bearing_hbox);
}

static void gcvote_set(cache_context_t *context, vote_t *vote) {
  if(!vote) return;
  
  if(context->quality) {
    gtk_widget_destroy(context->quality);
    context->quality = NULL;
  } 
  
  if(context->votes) {
    gtk_widget_destroy(context->votes);
    context->votes = NULL;
  } 
  
  /* update/draw the voting */
#ifndef USE_MAEMO
  GtkTooltips *tips = gtk_tooltips_new();
#endif
  
  char *votes_str = g_strdup_printf(_("Quality (%d %s):"), vote->votes,
				    (vote->votes == 1)?_("vote"):_("votes"));
  context->votes = GTK_LABEL_SMALL(votes_str);
  gtk_box_pack_start(GTK_BOX(context->votebox), 
		     context->votes, FALSE, FALSE, 0);
  g_free(votes_str);
  context->quality = icon_get_widget(ICON_STARS, (int)(vote->quality*2-2));
  gtk_box_pack_start(GTK_BOX(context->votebox), context->quality,
		     FALSE, FALSE, 0);
  
#ifndef USE_MAEMO
  char *str = g_strdup_printf(_("Quality: %d"), vote->quality);
  gtk_tooltips_set_tip(tips, context->votebox, str, NULL);
  g_free(str);
#endif
  
  gtk_widget_show_all(context->votebox);

  g_free(vote);
}

static void gcvote_callback(vote_t *vote, gpointer data) {
  cache_context_t *context = (cache_context_t*)data;
  
  /* no vote returned: request failed, just cleanup */
  if(!vote) {
    printf("gcvote: callback for failed request\n"); 
    
    gcvote_request_free(context->gcvote_request);
    context->gcvote_request = NULL;
    return;
  }
  
  printf("gcvote: callback is being called with a %d/%d\n", 
	 vote->quality, vote->votes);
  
  gcvote_set(context, vote);
  
  gcvote_save(context->appdata, context->cache, &context->gcvote_request->mem);
  
  gcvote_request_free(context->gcvote_request);
  context->gcvote_request = NULL;
}

static void pack_vcentered(GtkWidget *vbox, GtkWidget *child) {
  GtkWidget *align = gtk_alignment_new(0.5, 0.5, 0.0, 0.0);
  gtk_container_add(GTK_CONTAINER(align), child);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), align);
}

#ifdef FREMANTLE
#define ICON_SIZE  ICON_CACHE_TYPE_2X
#else
#define ICON_SIZE  ICON_CACHE_TYPE_1_5X
#endif

static GtkWidget *cache_overview(cache_context_t *context) {
  GtkWidget *vbox, *ivbox;
  GtkWidget *colbox, *tip;
  char str[64];
#ifndef USE_MAEMO
  GtkTooltips *tips = gtk_tooltips_new();
#endif
  appdata_t *appdata = context->appdata;
  cache_t *cache = context->cache;
  
  vbox = gtk_vbox_new(FALSE, 4);
  
  /* hbox containing the four columns of cache details */
  GtkWidget *pbox = portrait_box_new(FALSE, 0);

  GtkWidget *ihbox = gtk_hbox_new(FALSE, 0);
  
  /* vbox containing leftmost column (icon and container) */
  colbox = gtk_vbox_new(FALSE, 0);
  
  if(cache->type != CACHE_TYPE_UNKNOWN) {
    gtk_box_pack_start_defaults(GTK_BOX(colbox), 
	tip = icon_get_widget(ICON_SIZE, cache->type));
    
#ifndef USE_MAEMO
    gtk_tooltips_set_tip(tips, tip, _(cache_type_tip[cache->type]), NULL);
#endif
  }
  
  /* ------------ box containing container info ------------ */
  if(cache->container != CACHE_CONT_UNKNOWN) {
    ivbox = gtk_vbox_new(FALSE, 0);
    sprintf(str, _("Size: %s"), _(cache_size_name[cache->container]));
    gtk_box_pack_start(GTK_BOX(ivbox), GTK_LABEL_SMALL(str),
		       FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(ivbox), 
		       icon_get_widget(ICON_CACHE_SIZE, cache->container),
		       FALSE, FALSE, 0);
    
    pack_vcentered(colbox, ivbox);
#ifndef USE_MAEMO
    gtk_tooltips_set_tip(tips, ivbox, _(cache_size_tip[cache->container]), NULL);
#endif
  }
  
  gtk_box_pack_start_defaults(GTK_BOX(ihbox), colbox);
  colbox = gtk_vbox_new(FALSE, 0);
  
  /* ----------------------- id ---------------------------- */
  if(cache->id) {
    int strike = cache->archived?STRIKETHROUGH_RED:
      (!cache->available?STRIKETHROUGH:STRIKETHROUGH_NONE);
    GtkWidget *lbl = link_button_attrib(context->appdata, 
					cache->id, context->cache->url,
					SIZE_BIG, strike);
    pack_vcentered(colbox, lbl);
  }

  /* --------------- box containing owner info ------------- */
  if(cache->owner) {
    gtk_box_pack_start_defaults(GTK_BOX(colbox), GTK_LABEL_SMALL(_("by")));

    static const char *owner_type = "profile/";
    pack_vcentered(colbox,
		link_button_by_id(appdata, cache->owner->name, 
				  owner_type, cache->owner->id));
  }

  gtk_box_pack_start_defaults(GTK_BOX(ihbox), colbox);

  /* pack left two columns into portrait box */
  gtk_box_pack_start_defaults(GTK_BOX(pbox), ihbox);
  ihbox = gtk_hbox_new(FALSE, 0);

  /* ----------- vbox containing all ratings ---------- */
  colbox = gtk_vbox_new(FALSE, 0);

  /* ----------- box containing difficulty rating ---------- */
  if(cache->difficulty != 0) {
    ivbox = gtk_vbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(ivbox), 
		       GTK_LABEL_SMALL(_("Difficulty:")),
		       FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(ivbox), 
		  icon_get_widget(ICON_STARS, (int)(cache->difficulty*2-2)),
		       FALSE, FALSE, 0);

    pack_vcentered(colbox, ivbox);
#ifndef USE_MAEMO
    sprintf(str, _("Difficulty: %.1f"), cache->difficulty);
    gtk_tooltips_set_tip(tips, ivbox, str, NULL);
#endif
  }
    
  /* ------------ box containing terrain rating ------------ */
  if(cache->terrain != 0) {
    ivbox = gtk_vbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(ivbox), GTK_LABEL_SMALL(_("Terrain:")), 
		       FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(ivbox), 
		       icon_get_widget(ICON_STARS, (int)(cache->terrain*2-2)),
		       FALSE, FALSE, 0);
    pack_vcentered(colbox, ivbox);
#ifndef USE_MAEMO
    sprintf(str, _("Terrain: %.1f"), cache->terrain);
    gtk_tooltips_set_tip(tips, ivbox, str, NULL);
#endif
  }

  /* --------------------- GCVote ------------------------ */
  if(!appdata->disable_gcvote) {
    vote_t *vote = gcvote_restore(appdata, cache);

    context->gcvote_request = 
      gcvote_request(appdata, gcvote_callback, cache->url, context);

    context->votebox = gtk_vbox_new(FALSE, 0);
    pack_vcentered(colbox, context->votebox);

    /* fill with vote if present on disk (will also free vote) */
    if(vote) 
      gcvote_set(context, vote);
  }

  gtk_box_pack_start_defaults(GTK_BOX(ihbox), colbox);

  /* ----------------------------------------------------- */


  /* ----------------- the two coordinates ----------------- */
  /* ----------------- and the heading/distance ------------ */
  colbox = gtk_vbox_new(FALSE, 0);

  pos_t *refpos = get_pos(appdata);

  ivbox = gtk_vbox_new(FALSE, 0);
  int strike = (cache->notes && cache->notes->override)?
                  STRIKETHROUGH:STRIKETHROUGH_NONE;

  /* the original coordinate is being displayed */
  gtk_box_pack_start(GTK_BOX(ivbox), 
       context->pos_lat_label = pos_lat(cache->pos.lat, SIZE_BIG, strike), FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(ivbox), 
       context->pos_lon_label = pos_lon(cache->pos.lon, SIZE_BIG, strike), FALSE, FALSE, 0);

  pack_vcentered(colbox, ivbox);

  /* but calculations may be done with respect to the overriden one */
  if(!isnan(cache->pos.lat) && !isnan(cache->pos.lon)) {
    context->bearing_hbox = gtk_hbox_new(FALSE, 0);
    bearing_fill_hbox(context->bearing_hbox, appdata, *refpos, 
		      gpx_cache_pos(cache));
    pack_vcentered(colbox, context->bearing_hbox);
  }

  gtk_box_pack_start_defaults(GTK_BOX(ihbox), colbox);
  gtk_box_pack_start_defaults(GTK_BOX(pbox), ihbox);
 
  /* ----------------------------------------------------- */

  gtk_box_pack_start(GTK_BOX(vbox), pbox, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), gtk_hseparator_new(), FALSE, FALSE, 0);

  /* ----------------------------------------------------- */

  if(cache->attribute) {
    GtkWidget *thbox = gtk_hbox_new(FALSE, 0);
    ihbox = gtk_hbox_new(FALSE, 0);

    attribute_t *att = cache->attribute;
    while(att) {
      if(att->inc)
	tip = icon_get_widget(ICON_ATT, att->id+1);
      else
	tip = icon_get_widget_ovl(ICON_ATT, att->id+1, 0);

      gtk_box_pack_start(GTK_BOX(ihbox), tip, FALSE, FALSE, 4);

#ifndef USE_MAEMO
      gtk_tooltips_set_tip(tips, tip, _(att->name), NULL);
#endif

      att = att->next;
    }

    gtk_box_pack_start(GTK_BOX(thbox), ihbox, TRUE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), thbox, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), gtk_hseparator_new(), FALSE, FALSE, 0);
  }

  /* ----------------------------------------------------- */

  if(cache->short_description)
    gtk_box_pack_start_defaults(GTK_BOX(vbox), 
	html_view(appdata, cache->short_description, 
	  cache->short_is_html?HTML_HTML:HTML_PLAIN_TEXT, TRUE, cache, NULL));

  return vbox;
}

/* slow but short, we don't need performance here ... */
static void rot13(char *t) {
  int braces = 0;

  while(*t) {
    if(!braces) {
      if(*t == '[') 
	braces++;
      else if(((*t >= 'a') && (*t < 'n')) ||
	      ((*t >= 'A') && (*t < 'N'))) *t += 13;
      else if(((*t >= 'n') && (*t <= 'z')) ||
	      ((*t >= 'N') && (*t <= 'Z'))) *t -= 13;
    } else {
      if(braces > 0 && *t == ']')
	braces--;
    }

    t++;
  }
}

static void on_decrypt(GtkWidget *widget, gpointer data) {
  /* data is a link to the textview */
  g_assert(GTK_IS_TEXT_VIEW(data));

  GtkTextIter start, end;
  GtkTextBuffer *buffer = gtk_text_view_get_buffer((GtkTextView*)data);

  gtk_text_buffer_get_start_iter(buffer, &start);
  gtk_text_buffer_get_end_iter(buffer, &end);
  char *text = gtk_text_buffer_get_text(buffer, &start, &end, FALSE);

  rot13(text);
  gtk_text_buffer_set_text(buffer, text, -1);

  free(text);
}

static GtkWidget *cache_hint(appdata_t *appdata, cache_t *cache) {
  /* encrypting/decrypting html is nothing we want to do */
  if(cache->hint_is_html) 
    return html_view(appdata, cache->hint, HTML_HTML, TRUE, NULL, NULL);

  /* we can now be sure that we are talking about pain text */
  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);

  char *hint = strdup(cache->hint);
  rot13(hint);
  GtkWidget *view = 
    html_view(appdata, hint, HTML_PLAIN_TEXT, TRUE, NULL, NULL);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), view);
  free(hint);

  GtkWidget *button = gtk_button_new_with_label(_("Encrypt/Decrypt"));
#if defined(USE_MAEMO) && (MAEMO_VERSION_MAJOR == 5)
  hildon_gtk_widget_set_theme_size(button, 
	   (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
#endif  
  gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
	   GTK_SIGNAL_FUNC(on_decrypt), gtk_bin_get_child(GTK_BIN(view)));

  return vbox;
}

static GtkWidget *cache_wpts(appdata_t *appdata, wpt_t *wpt) {
  pos_t *refpos = NULL;

#ifndef USE_PANNABLE_AREA
  GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
  				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
#else
  GtkWidget *pannable_area = hildon_pannable_area_new();
#endif

  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);

  /* four rows per waypoint */
  GtkWidget *table =  gtk_table_new(4*gpx_number_of_waypoints(wpt)-1,4, FALSE);

  refpos = get_pos(appdata);

  int wpt_row=0;
  while(wpt) {
    GtkWidget *ihbox, *tip;
    char str[32];

    /* ----------------------- icon/id ---------------------------- */
    ihbox = gtk_hbox_new(FALSE, 0);

    if(wpt->sym != WPT_SYM_UNKNOWN) {
      gtk_box_pack_start(GTK_BOX(ihbox),
	  tip = icon_get_widget(ICON_WPT, wpt->sym), 1,0,0);
    }

    if(wpt->id)
      gtk_box_pack_start(GTK_BOX(ihbox), GTK_LABEL_BIG(wpt->id), 1,0,0);

    gtk_table_attach_defaults(GTK_TABLE(table), ihbox, 0,1,wpt_row, wpt_row+1);

    /* ----------------- the two coordinates ----------------- */
    /* ----------------- and the heading/distance ------------ */
    gtk_table_attach_defaults(GTK_TABLE(table), 
		      pos_lat(wpt->pos.lat, SIZE_BIG, STRIKETHROUGH_NONE), 
		      1,2, wpt_row, wpt_row+1);
    gtk_table_attach_defaults(GTK_TABLE(table), 
		      pos_lon(wpt->pos.lon, SIZE_BIG, STRIKETHROUGH_NONE), 
		      2,3, wpt_row, wpt_row+1);

    if(refpos && !isnan(refpos->lat) && !isnan(refpos->lon)) {
      ihbox = gtk_hbox_new(FALSE, 0);
      gtk_box_pack_start(GTK_BOX(ihbox), gtk_image_new_from_pixbuf(
			   icon_bearing(*refpos, wpt->pos)),1,0,0);
      snprintf(str, sizeof(str), _("%.1f°"), 
	       gpx_pos_get_bearing(*refpos, wpt->pos));
      gtk_box_pack_start_defaults(GTK_BOX(ihbox), GTK_LABEL_SMALL(str));

      gpx_pos_get_distance_str(str, sizeof(str), 
			       *refpos, wpt->pos, appdata->imperial);
      gtk_box_pack_start(GTK_BOX(ihbox), GTK_LABEL_SMALL(str),1,0,0);
      
      gtk_table_attach_defaults(GTK_TABLE(table), ihbox, 3,4, 
				wpt_row+0, wpt_row+1);
    }

    /* ------------------ description ------------------------- */
    if(wpt->desc) 
      gtk_table_attach_defaults(GTK_TABLE(table), 
				simple_text_widget(wpt->desc), 0,4, 
				wpt_row+1, wpt_row+2);

    /* ------------------ comment ------------------------- */
    if(wpt->cmt) 
      gtk_table_attach_defaults(GTK_TABLE(table),
				simple_text_widget(wpt->cmt), 0,4, 
				wpt_row+2, wpt_row+3);

    /* --------------------- seperator -------------------------*/
    if(wpt->next) {
      gtk_table_set_row_spacing(GTK_TABLE(table), wpt_row+2, 8);
      gtk_table_attach_defaults(GTK_TABLE(table), gtk_hseparator_new(), 0,4, 
				wpt_row+3, wpt_row+4);
      gtk_table_set_row_spacing(GTK_TABLE(table), wpt_row+3, 8);
    }

    wpt_row+=4;
    wpt = wpt->next;
  }

  gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

#ifndef  USE_PANNABLE_AREA
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), 
					vbox);
  return scrolled_window;
#else
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(pannable_area),
					 vbox);
  return pannable_area;
#endif
}

static GtkWidget *cache_tbs(appdata_t *appdata, tb_t *tb) {
  pos_t *refpos = NULL;

#ifndef USE_PANNABLE_AREA
  GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
  				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
#else
  GtkWidget *pannable_area = hildon_pannable_area_new();
#endif

  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);

  /* four rows per waypoint */
  GtkWidget *table =  gtk_table_new(2*gpx_number_of_tbs(tb)-1,3, FALSE);

  refpos = get_pos(appdata);

  int tb_row=0;
  while(tb) {
    static const char *tb_type = "track/details.aspx";
    
    /* --------------------- icon/ref/name -------------------------*/
    GtkWidget *icon = NULL;
    if((strcasestr(tb->name, "coin") != 0) ||
       (strcasestr(tb->name, "muenze") != 0) ||
       (strcasestr(tb->name, "münze") != 0))
      icon = icon_get_widget(ICON_TB, 1);   /* coin icon */
    else
      icon = icon_get_widget(ICON_TB, 0);   /* tb icon */

    gtk_table_attach_defaults(GTK_TABLE(table), icon, 
			      0, 1, tb_row+0, tb_row+1);

    if(tb->ref) {
      GtkWidget *ref = link_button_by_id(appdata, tb->ref, tb_type, tb->id);
      gtk_table_attach_defaults(GTK_TABLE(table), ref,
				1, 2, tb_row+0, tb_row+1);
    }

    if(tb->name)
      gtk_table_attach_defaults(GTK_TABLE(table), GTK_LABEL_BIG(tb->name),
				2, 3, tb_row+0, tb_row+1);

    /* --------------------- seperator -------------------------*/
    if(tb->next)
      gtk_table_attach_defaults(GTK_TABLE(table), gtk_hseparator_new(), 0, 3, 
				tb_row+1, tb_row+2);
    tb_row+=2;
    tb = tb->next;
  }

  gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

#ifndef  USE_PANNABLE_AREA
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), 
					vbox);
  return scrolled_window;
#else
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(pannable_area),
					 vbox);
  return pannable_area;
#endif
}

#ifdef ENABLE_BROWSER_INTERFACE
static void on_gclink_clicked(GtkButton *button, gpointer data) {
  cache_context_t *context = (cache_context_t*)data;
  char *url = g_strdup_printf("http://www.geocaching.com/seek/log.aspx?wp=%s", context->cache->id);
  browser_url(context->appdata, url);
  g_free(url);
}
#endif

static GtkWidget *cache_logs(appdata_t *appdata, cache_context_t *context, 
			     log_t *log, int is_html) {
#ifndef  USE_PANNABLE_AREA
  /* put this inside a scrolled view */
  GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
  				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
#else
  GtkWidget *pannable_area = hildon_pannable_area_new();
#endif

#ifdef ENABLE_BROWSER_INTERFACE
  gboolean gc_link = strncmp(context->cache->id, "GC", 2) == 0;
#else
#define gc_link (FALSE)
#endif

  GtkWidget *vbox = gtk_vbox_new(FALSE, 6);

#ifdef ENABLE_BROWSER_INTERFACE
  if(gc_link) {
    GtkWidget *but =
      gtk_button_new_with_label(_("Post a new log entry for this geocache"));
#if defined(USE_MAEMO) && (MAEMO_VERSION_MAJOR == 5)
  hildon_gtk_widget_set_theme_size(but, 
	   (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
#endif  
    gtk_signal_connect(GTK_OBJECT(but), "clicked",
		       GTK_SIGNAL_FUNC(on_gclink_clicked), context);

    gtk_box_pack_start(GTK_BOX(vbox), but, FALSE, FALSE, 0);
  }
#endif

  int logs = gpx_number_of_logs(log);
  GtkWidget *table = gtk_table_new(2*logs-1, 2,FALSE);
  int log_cnt = 0;

  gtk_table_set_col_spacing(GTK_TABLE(table), 0, 8);
  
  /* add all logs to the vbox */
  while(log) {
    GtkWidget *ivbox = gtk_vbox_new(FALSE, 2);
    GtkWidget *ihbox = gtk_hbox_new(FALSE, 2);

    static const char *finder_type = "profile/";
    GtkWidget *finder = link_button_by_id(appdata, log->finder->name, 
					  finder_type, log->finder->id);

    /* if the finder is a button make sure it's the right size and */
    /* does not exceed the size limits */
    if(GTK_WIDGET_TYPE(finder) == GTK_TYPE_BUTTON) {
#if defined(USE_MAEMO) && (MAEMO_VERSION_MAJOR == 5)
      hildon_gtk_widget_set_theme_size(finder, 
		   (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
#endif  

    gtk_label_set_ellipsize(GTK_LABEL(gtk_bin_get_child(GTK_BIN(finder))), 
			      PANGO_ELLIPSIZE_END);
    } else
      gtk_label_set_ellipsize(GTK_LABEL(finder), PANGO_ELLIPSIZE_END);

    gtk_box_pack_start(GTK_BOX(ivbox), finder, FALSE, FALSE, 0);

    gtk_box_pack_start_defaults(GTK_BOX(ihbox), 
	      icon_get_widget(ICON_LOG, log->type));

    char date_str[32];
    if(log->day && log->month && log->year) {
      GDate *date = g_date_new_dmy(log->day, log->month, log->year);
      g_date_strftime(date_str, sizeof(date_str), "%x", date);
      g_date_free(date);
    } else
      strcpy(date_str, "---");

    gtk_box_pack_start_defaults(GTK_BOX(ihbox), gtk_label_new(date_str));

    gtk_box_pack_start(GTK_BOX(ivbox), ihbox, FALSE, FALSE, 0);

    gtk_table_attach(GTK_TABLE(table), ivbox, 0, 1,
		     2*log_cnt, 2*log_cnt+1, 0, GTK_EXPAND | GTK_FILL, 0, 0);

    if(log->text) {
      gtk_table_attach_defaults(GTK_TABLE(table),
	html_view(appdata, log->text, 
		  is_html?HTML_HTML:HTML_CUSTOM_MARKUP, FALSE, NULL, NULL),
				1, 2, 2*log_cnt, 2*log_cnt+1);    
    }

    if(log_cnt < logs-1) {
      gtk_table_set_row_spacing(GTK_TABLE(table), 2*log_cnt, 8);

      gtk_table_attach_defaults(GTK_TABLE(table), gtk_hseparator_new(), 
				0, 2, 2*log_cnt+1, 2*log_cnt+2);

      gtk_table_set_row_spacing(GTK_TABLE(table), 2*log_cnt+1, 8);
    }

    log = log->next;
    log_cnt++;
  }

  gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

#ifndef  USE_PANNABLE_AREA
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), 
					vbox);
  return scrolled_window;
#else
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(pannable_area),
					 vbox);
  return pannable_area;
#endif
}

#ifdef USE_MAEMO
/* this routine is called once a second as long as the "goto" tab is visible */
static gboolean screensaver_update(gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  if(appdata->goto_disable_screensaver)
    if (osso_display_blanking_pause(appdata->osso_context) != OSSO_OK) 
      fprintf(stderr, "error with display blank\n");

  return TRUE;  // fire again
}
#endif

static void on_notebook_page_change(GtkNotebook     *notebook,
				    GtkNotebookPage *page,
				    guint            page_num,
				    gpointer         user_data) {

  cache_context_t *context = (cache_context_t*)user_data;
  GtkWidget *w = gtk_notebook_get_nth_page(notebook, page_num);
  const char *name = gtk_notebook_get_tab_label_text(notebook, w);

#ifdef USE_MAEMO  
  if(context->handler_id)
    gtk_timeout_remove(context->handler_id);
#endif

  /* this is a workaround, around some bug in the gtktextwidget or so ... */
  /* i tried to get info on this and everybody agreed that this is a bug */
  /* in gtk but noone had a fix ready. so i came up with this. */
  /* seems to work ... */
  if(strcasecmp(name, _("Logs")) == 0) {
    gtk_widget_queue_resize(w);
  } else if(strcasecmp(name, _("Goto")) == 0) {
#ifdef USE_MAEMO  
    context->handler_id = gtk_timeout_add(1000, screensaver_update, 
					  context->appdata);
#endif

    goto_coordinate_update(context);
  }

  if(strcasecmp(name, _("Main")) == 0) {
    /* the notes page may have changed its "override" setting, thus the */
    /* striked out coordinate may need update */
    overview_coordinate_update(context);
  }
}

static void on_notebook_destroy(GtkWidget *widget, gpointer user_data ) {
  cache_context_t *context = (cache_context_t*)user_data;

  printf("destroying notebook\n");

  /* cancel a pending gcvote request */
  if(context->gcvote_request) {
    gcvote_request_free(context->gcvote_request);
    context->gcvote_request = NULL;
  }

  notes_destroy_event(NULL, context);
  goto_destroy_event(NULL, context);

#ifdef USE_MAEMO
  if(context->handler_id)
    gtk_timeout_remove(context->handler_id);
#endif

#ifdef USE_STACKABLE_WINDOW
  if(context->notes_have_been_changed) {
    printf("notes changed -> cachelist redraw\n");

    /* now the cachelist is visible again. so redraw it since it may */
    /* have changed */
    cachelist_redraw(context->appdata);
  }
#endif

  printf("freeing cache context\n");
  context->appdata->cache_context = NULL;
  g_free(context);
}

static GObject *notebook_object(GtkWidget *notebook) {
  return G_OBJECT(notebook_get_gtk_notebook(notebook));
}

GtkWidget *cache_view(appdata_t *appdata, cache_t *cache) {
  GtkWidget *notebook;

  cache_context_t *cache_context = g_new0(cache_context_t, 1);
  appdata->cache_context = cache_context;
  cache_context->appdata = appdata;
  cache_context->cache = cache;

#ifdef USE_MAEMO
#define TAB_DESC   _("Desc.")
#define TAB_WPTS   _("Wpts")
#else
#define TAB_DESC   _("Description")
#define TAB_WPTS   _("Waypoints")
#endif

  notebook = notebook_new();

  notebook_append_page(notebook, 
       cache_overview(cache_context), _("Main"));

  if(cache->long_description)
    notebook_append_page(notebook, 
	   cache_description(appdata, cache), TAB_DESC);

  if(cache->hint)
    notebook_append_page(notebook, 
	   cache_hint(appdata, cache), _("Hint"));

  // always display log page to allow user to post a new log
  if(cache->log
#ifdef ENABLE_BROWSER_INTERFACE
     || (strncmp(cache->id, "GC", 2) == 0)
#endif
     ) 
    notebook_append_page(notebook, 
     cache_logs(appdata, cache_context, cache->log, cache->logs_are_html), 
	   _("Logs"));

  if(cache->wpt)
    notebook_append_page(notebook, 
	     cache_wpts(appdata, cache->wpt), TAB_WPTS);

  if(cache->tb)
    notebook_append_page(notebook, 
	     cache_tbs(appdata, cache->tb), _("TBs"));

  /* the demo caches don't have coordinates, so avoid */
  /* having to deal with them */
  if(!isnan(cache->pos.lat) && !isnan(cache->pos.lon)) {
    notebook_append_page(notebook, 
			 cache_notes(cache_context), _("Notes"));

    notebook_append_page(notebook, 
			 goto_cache(cache_context), _("Goto"));
  }

  g_signal_connect(notebook_object(notebook), "switch-page", 
	   G_CALLBACK(on_notebook_page_change), cache_context);

  g_signal_connect(notebook_object(notebook), "destroy", 
	   G_CALLBACK(on_notebook_destroy), cache_context);

  return notebook;
}

#ifndef USE_MAEMO
void cache_dialog(appdata_t *appdata, cache_t *cache) {
  GtkWidget *dialog = gtk_dialog_new_with_buttons(cache->name, 
			  GTK_WINDOW(appdata->window),
		          GTK_DIALOG_NO_SEPARATOR | GTK_DIALOG_MODAL | 
				       GTK_DIALOG_DESTROY_WITH_PARENT,
		          GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
                          NULL);

  gtk_window_set_default_size(GTK_WINDOW(dialog), DIALOG_WIDTH, DIALOG_HEIGHT);

  /* create cache visualization widget */
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), 
		    cache_view(appdata, cache));

  gtk_widget_show_all(dialog);
  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);
}

#else
#ifdef USE_STACKABLE_WINDOW

#if !defined(__i386__)
/* get access to zoom buttons */
static void
on_window_realize(GtkWidget *widget, gpointer data) {
  if (widget->window) {
    unsigned char value = 1;
    Atom hildon_zoom_key_atom = 
      gdk_x11_get_xatom_by_name("_HILDON_ZOOM_KEY_ATOM"),
      integer_atom = gdk_x11_get_xatom_by_name("INTEGER");
    Display *dpy = 
      GDK_DISPLAY_XDISPLAY(gdk_drawable_get_display(widget->window));
    Window w = GDK_WINDOW_XID(widget->window);

    XChangeProperty(dpy, w, hildon_zoom_key_atom, 
		    integer_atom, 8, PropModeReplace, &value, 1);
  }
}
#endif

static void on_cache_destroy (GtkWidget *widget, appdata_t *appdata) {
  appdata->cur_cache = NULL;

  HildonWindowStack *stack = hildon_window_stack_get_default();
  appdata->window = HILDON_WINDOW(hildon_window_stack_peek(stack));

  /* restore cur_view */
  appdata->cur_view = g_object_get_data(G_OBJECT(widget), "cur_view");
}

void cache_dialog(appdata_t *appdata, cache_t *cache) {
  GtkWidget *window = hildon_stackable_window_new();
  appdata->window = HILDON_WINDOW(window);

#if !defined(__i386__)
  g_signal_connect(G_OBJECT(appdata->window), "realize", 
		   G_CALLBACK(on_window_realize), NULL);
#endif  

  g_signal_connect(G_OBJECT(appdata->window), "key_press_event",
		   G_CALLBACK(on_window_key_press), appdata);

  /* store last "cur_view" in window */
  g_object_set_data(G_OBJECT(window), "cur_view", appdata->cur_view);

  appdata->cur_cache = cache;
  gtk_window_set_title(GTK_WINDOW(window), cache->name);

  /* make sure window can control gps */
  g_signal_connect(G_OBJECT(window), "focus-in-event", 
		   G_CALLBACK(on_main_focus_change), appdata);
  
  g_signal_connect(G_OBJECT(window), "focus-out-event", 
		   G_CALLBACK(on_main_focus_change), appdata);

  /* create cache visualization widget */
  appdata->cur_view = cache_view(appdata, cache);
  gtk_container_add(GTK_CONTAINER(window), appdata->cur_view);

  hildon_window_set_app_menu(HILDON_WINDOW(window), 
			     menu_create(appdata, MENU_CACHE));

  g_signal_connect(G_OBJECT(window), "destroy", 
		   G_CALLBACK(on_cache_destroy), appdata);

  gtk_widget_show_all(window);
}
#endif // USE_STACKABLE_WINDOW

#endif // USE_MAEMO
