/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef GPX_H
#define GPX_H

#ifndef NAN
#define NAN (0.0/0.0)
#endif

#define WPT_SYM_UNKNOWN               -1
#define WPT_SYM_MULTISTAGE             0
#define WPT_SYM_PARKING                1
#define WPT_SYM_FINAL                  2
#define WPT_SYM_QUESTION               3
#define WPT_SYM_TRAILHEAD              4
#define WPT_SYM_REFPOINT               5
#define WPT_SYM_MAX                   (WPT_SYM_REFPOINT)

#define CACHE_TYPE_UNKNOWN            -1
#define CACHE_TYPE_TRADITIONAL         0
#define CACHE_TYPE_MULTI               1
#define CACHE_TYPE_MYSTERY             2
#define CACHE_TYPE_VIRTUAL             3
#define CACHE_TYPE_WEBCAM              4
#define CACHE_TYPE_EVENT               5
#define CACHE_TYPE_LETTERBOX           6
#define CACHE_TYPE_EARTHCACHE          7
#define CACHE_TYPE_WHERIGO             8
#define CACHE_TYPE_MEGA_EVENT          9
#define CACHE_TYPE_CITO               10
#define CACHE_TYPE_MAX                (CACHE_TYPE_CITO)

#define CACHE_CONT_UNKNOWN            -1
#define CACHE_CONT_REGULAR             0
#define CACHE_CONT_SMALL               1
#define CACHE_CONT_MICRO               2
#define CACHE_CONT_OTHER               3
#define CACHE_CONT_NOT_CHOSEN          4
#define CACHE_CONT_LARGE               5
#define CACHE_CONT_VIRTUAL             6
#define CACHE_CONT_MAX                (CACHE_CONT_VIRTUAL)

#define LOG_TYPE_UNKNOWN              -1
#define LOG_TYPE_FOUND                 0
#define LOG_TYPE_NOT_FOUND             1
#define LOG_TYPE_MAINTENANCE           2
#define LOG_TYPE_WRITE_NOTE            3
#define LOG_TYPE_REVIEWER_NOTE         4
#define LOG_TYPE_ENABLE_LISTING        5
#define LOG_TYPE_PUBLISH_LISTING       6
#define LOG_TYPE_WILL_ATTEND           7
#define LOG_TYPE_ATTENDED              8
#define LOG_TYPE_PHOTO                 9
#define LOG_TYPE_TEMP_DISABLE         10
#define LOG_TYPE_NEEDS_MAIN           11
#define LOG_TYPE_UPDATED_COORDINATES  12
#define LOG_TYPE_UNARCHIVE            13
#define LOG_TYPE_NEEDS_ARCHIVED       14
#define LOG_TYPE_ARCHIVE              15
#define LOG_TYPE_MAX                  (LOG_TYPE_ARCHIVE)

// gpx 1.0.1 attributes
#define ATT_UNKNOWN             0
#define ATT_DOGS                1
#define ATT_FEE                 2
#define ATT_RAPPELLING          3
#define ATT_BOAT                4
#define ATT_SCUBA               5
#define ATT_KIDS                6
#define ATT_ONEHOUR             7
#define ATT_SCENIC              8
#define ATT_HIKING              9
#define ATT_CLIMBING           10
#define ATT_WADING             11
#define ATT_SWIMMING           12
#define ATT_AVAILABLE          13
#define ATT_NIGHT              14
#define ATT_WINTER             15
#define ATT_UNKNOWN0           16  // <-- ???
#define ATT_POISONOAK          17
#define ATT_SNAKES             18
#define ATT_TICKS              19
#define ATT_MINE               20
#define ATT_CLIFF              21
#define ATT_HUNTING            22
#define ATT_DANGER             23
#define ATT_WHEELCHAIR         24
#define ATT_PARKING            25
#define ATT_PUBLIC             26
#define ATT_WATER              27
#define ATT_RESTROOMS          28
#define ATT_PHONE              29
#define ATT_PICNIC             30
#define ATT_CAMPING            31
#define ATT_BICYCLES           32
#define ATT_MOTORCYCLES        33
#define ATT_QUADS              34
#define ATT_JEEPS              35
#define ATT_SNOWMOBILES        36
#define ATT_HORSES             37
#define ATT_CAMPFIRES          38
#define ATT_THORN              39
#define ATT_STEALTH            40
#define ATT_STROLLER           41
#define ATT_FIRSTAID           42
#define ATT_COW                43
#define ATT_FLASHLIGHT         44
// august 2010 attributes
#define ATT_LANDF              45
#define ATT_UNKNOWN1           46  // <-- ???
#define ATT_FIELD_PUZZLE       47
#define ATT_UV                 48
#define ATT_SNOWSHOES          49
#define ATT_SKIIS              50
#define ATT_S_TOOL             51
#define ATT_NIGHTCACHE         52
#define ATT_PARKNGRAB          53
#define ATT_ABANDONEDBUILDING  54
#define ATT_HIKE_SHORT         55
#define ATT_HIKE_MED           56
#define ATT_HIKE_LONG          57
#define ATT_FUEL               58
#define ATT_FOOD               59
#define ATT_MAX                (ATT_FOOD)

typedef struct attribute {
  char *name;
  unsigned int id;
  gboolean inc;
  struct attribute *next;
} attribute_t;

typedef struct {
  char *name;
  unsigned int id;
} user_t;

typedef struct tb {
  char *name;
  char *ref;
  unsigned int id;
  struct tb *next;
} tb_t;

typedef struct log {
  int  day, month, year;
  int  type;
  user_t *finder;
  unsigned int id;
  char *text;

  struct log *next;
} log_t;

typedef struct pos {
  float lat;
  float lon;
} pos_t;

typedef struct wpt {
  char *id;
  pos_t pos;  
  char *cmt;
  char *desc;
  int sym;

  struct wpt *next;
} wpt_t;

typedef struct {
  char *text;
  pos_t pos;  
  gboolean override;
  gboolean found;  
  gboolean logged;  
  time_t ftime;
} notes_t;

typedef struct cache {
  /* cache information */
  char *id;
  char *name;

  user_t *owner;

  int  container;
  int  type;
  char *short_description, *long_description;
  int short_is_html, long_is_html;
  float difficulty, terrain;
  pos_t pos;
  char *hint;
  int hint_is_html;
  int logs_are_html;
  char *url;
  int available, archived;
  gboolean found;      /* determined by log */
  gboolean mine;

  notes_t *notes;           /* user provides data. Doesn't come from gpx file */
  wpt_t *wpt;               /* pointer to list of waypoints */
  log_t *log;               /* pointer to list of logs */
  tb_t *tb;                 /* pointer to list of travelbugs */
  attribute_t *attribute;   /* pointer to list of attributes */
  struct cache *next;       /* pointer to next cache in list */
} cache_t;

typedef struct gpx {
  char *filename; 
  char *name; 
  char *desc;
  int  day, month, year;
  int  notes_loaded;   /* notes are loaded on demand to keep reponse fast */
  int  closed;         /* true if entry is "closed" */
  cache_t *cache;

  /* pointer to next gpx in list */
  struct gpx *next;
} gpx_t;

typedef struct {
  GtkWidget *dialog;
  GtkWidget *label, *pbar;
} gpx_dialog_t;

#define GPX_SORT_BY_DISTANCE  0
#define GPX_SORT_BY_ID        1
#define GPX_SORT_BY_NAME      2

gpx_dialog_t *gpx_busy_dialog_new(GtkWidget *parent);
void gpx_busy_dialog_destroy(gpx_dialog_t *);

gpx_t *gpx_parse(gpx_dialog_t *dialog, char *filename, char *username);
gpx_t *gpx_parse_dir(gpx_dialog_t *dialog, char *dirname, char *username);
void  gpx_free(gpx_t *gpx);
void  gpx_free_all(gpx_t *gpx);
void gpx_free_caches(gpx_t *gpx);
void  gpx_display_all(gpx_t *gpx);
int   gpx_total_caches(gpx_t *gpx);
int   gpx_total_caches_global(gpx_t *gpx);
void  gpx_sort(gpx_t *gpx, int by, pos_t *refpos);
float gpx_pos_get_bearing(pos_t p1, pos_t p2);
float gpx_pos_get_distance(pos_t p1, pos_t p2, int miles);
int   gpx_number_of_waypoints(wpt_t *wpt);
void  gpx_pos_get_distance_str(char *str, int l, pos_t p1, pos_t p2, int mil);
int   gpx_number_of_logs(log_t *log);
int   gpx_number_of_tbs(tb_t *tb);
gpx_t *gpx_cache2gpx(gpx_t *gpx, cache_t *cache);
pos_t gpx_cache_pos(cache_t *cache);

#endif /* GPX_H */
