/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"
#include <sqlite3.h>

#include <libxml/parser.h>
#include <libxml/tree.h>

#if !defined(LIBXML_TREE_ENABLED) || !defined(LIBXML_OUTPUT_ENABLED)
#error "libxml doesn't support required tree or output"
#endif

typedef struct {
  GtkWidget *dialog;
  appdata_t *appdata;
  GtkWidget *ignore_found;
  GtkWidget *info_label;
  gpx_t *gpx;
} export_context_t;

/* add a cache to the chain of caches to be exported */
static void chain_cache(gpx_t *gpx, cache_t *cache) {
  cache_t **cur = &gpx->cache;

  /* search end of chain */
  while(*cur) {
    if(strcmp(cache->id, (*cur)->id) == 0) 
      return;

    cur = &(*cur)->next;
  }

  *cur = malloc(sizeof(cache_t));
  if(!(*cur)) return;

  memcpy(*cur, cache, sizeof(cache_t));
  (*cur)->next = NULL;
}

static gpx_t *export_list_create(appdata_t *appdata) {
  gpx_t *gpx = g_new0(gpx_t,1);
  if(!gpx) return gpx;

  gpx_t *lgpx = appdata->gpx;
  while(lgpx) {
    /* make sure all notes are loaded */
    if(!lgpx->notes_loaded) {
      notes_load_all(appdata, lgpx);
      lgpx->notes_loaded = TRUE;
    }

    cache_t *cache = lgpx->cache;
    while(cache) {
      if(cache->notes && cache->notes->override)
	chain_cache(gpx, cache); 

      cache = cache->next; 
    }
    lgpx = lgpx->next;
  }

  return gpx;
}

static void export_list_free(gpx_t *gpx) {
  printf("freeing export list\n");

  cache_t *cache = gpx->cache;
  while(cache) {
    cache_t *tmp = cache;
    cache = cache->next;
    free(tmp);
  }
  free(gpx);
}

static int export_list_count(appdata_t *appdata, gpx_t *gpx) {
  int cnt = 0;

  cache_t *cache = gpx->cache;
  while(cache) {
    if(!appdata->garmin_ign_found || 
       (appdata->garmin_ign_found && !cache->notes->found))
      cnt++;
    
    cache = cache->next;
  }
  return cnt;
}

/* Our usual callback function */
static void export_update(GtkWidget *widget, gpointer data) {
  export_context_t *context = (export_context_t *)data;

  if(check_button_get_active(context->ignore_found)) 
    context->appdata->garmin_ign_found = TRUE;
  else
    context->appdata->garmin_ign_found = FALSE;

  char str[256];
  snprintf(str, sizeof(str), 
	   _("This will export the overridden coordinates of %d "
	     "caches into a GPX waypoint file suitable for garmin "
	     "GPS units."), export_list_count(context->appdata, context->gpx));

  gtk_label_set_text(GTK_LABEL(context->info_label), str);
}

void garmin_export(appdata_t *appdata) {
  char *old_garmin_path = strdup(appdata->garmin_path);

  export_context_t context;
  memset(&context, 0, sizeof(export_context_t));
  context.appdata = appdata;
  context.gpx = export_list_create(appdata);

  printf("export garmin data\n");

  /* --------- do confirmation dialog ----------- */
  context.dialog = gtk_dialog_new_with_buttons(_("Garmin waypoint export"),
	  GTK_WINDOW(appdata->window), GTK_DIALOG_MODAL,
	  GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
          GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
          NULL);

#if defined(USE_MAEMO) && defined(HILDON_HELP)
  hildon_help_dialog_help_enable(GTK_DIALOG(context.dialog), 
		 HELP_ID_GARMIN, appdata->osso_context);
#endif 

  GtkWidget *vbox = gtk_vbox_new(FALSE,2);

  gtk_box_pack_start_defaults(GTK_BOX(vbox),
	      context.ignore_found = check_button_new_with_label(
	     _("Ignore found caches")));
  check_button_set_active(context.ignore_found, 
			   appdata->garmin_ign_found);
  /* Connect the "clicked" signal of the button to our callback */
  gtk_signal_connect (GTK_OBJECT(context.ignore_found), "clicked",
		      GTK_SIGNAL_FUNC(export_update), (gpointer)&context);

  context.info_label = gtk_label_new("");
  gtk_label_set_line_wrap_mode(GTK_LABEL(context.info_label),PANGO_WRAP_WORD);
  gtk_label_set_line_wrap(GTK_LABEL(context.info_label), TRUE);
  gtk_misc_set_alignment(GTK_MISC(context.info_label), 0.f, 0.5f);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), context.info_label);

  export_update(NULL, &context);

  /* ------------------ path/file ------------------ */
  gtk_box_pack_start_defaults(GTK_BOX(vbox), gtk_hseparator_new());

  gtk_box_pack_start_defaults(GTK_BOX(vbox), 
      export_file(_("Save garmin GPX"), &appdata->garmin_path));

  /* ----------------------------------- */

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(context.dialog)->vbox), vbox);

  gtk_widget_show_all(context.dialog);

  if(GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(context.dialog))) {
    /* remove existing database */
    remove(appdata->garmin_path);

    if(checkdir(appdata->garmin_path) != 0) 
      errorf(_("Unable to access or create output directory!"));
    else {

      if(checkdir(appdata->garmin_path) != 0) {
	errorf(_("Unable to create export path!"));
      } else {

	LIBXML_TEST_VERSION;
    
	xmlDocPtr doc = xmlNewDoc(BAD_CAST "1.0");
	xmlNodePtr root_node = xmlNewNode(NULL, BAD_CAST "gpx");
	xmlDocSetRootElement(doc, root_node);
  
	/* add all waypoints */
	cache_t *cache = context.gpx->cache;
	while(cache) {
	  notes_t *notes = cache->notes;
	  g_assert(notes);

	  if(!appdata->garmin_ign_found || 
	     (appdata->garmin_ign_found && !notes->found)) {
	    
	    xmlNodePtr wpt_node = 
	      xmlNewChild(root_node, NULL, BAD_CAST "wpt", NULL);
	    
	    /* make sure no invalid position gets saved */
	    char str[32];
	    g_ascii_dtostr(str, sizeof(str), notes->pos.lat);
	    xmlNewProp(wpt_node, BAD_CAST "lat", BAD_CAST str);
	    g_ascii_dtostr(str, sizeof(str), notes->pos.lon);
	    xmlNewProp(wpt_node, BAD_CAST "lon", BAD_CAST str);
	    
	    int len = strlen(cache->id) + strlen(" - ") +
	      strlen(cache->name) + 1;
	    char *name = malloc(len);
	    snprintf(name, len, "%s - %s", cache->id, cache->name);
	    xmlNewChild(wpt_node, NULL, BAD_CAST "name", BAD_CAST name);
	    free(name);
	    xmlNewChild(wpt_node, NULL, BAD_CAST "sym", BAD_CAST "Pin, Blue");
	    xmlNewChild(wpt_node, NULL, BAD_CAST "desc", BAD_CAST notes->text);
	  }
	    
	  cache = cache->next;
	}
  
	/* write everything and free it */
	printf("writing %s\n", appdata->garmin_path);
	xmlSaveFormatFileEnc(appdata->garmin_path, doc, "UTF-8", 1);
	xmlFreeDoc(doc);
	xmlCleanupParser();
      }
    }
  } else {
    /* restore old garmin_path, in case it has been altered */
    /* but not been used */
    free(appdata->garmin_path);
    appdata->garmin_path = strdup(old_garmin_path);
  }
  
  gtk_widget_destroy(context.dialog);
  export_list_free(context.gpx);
  free(old_garmin_path);
}
