/*
 * Copyright (C) 2009 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"

#ifndef FREMANTLE
#define LINK_COLOR "blue"
#else
#define LINK_COLOR "lightblue"
#define CUSTOM_NOTEBOOK
#endif

#ifdef ENABLE_BROWSER_INTERFACE
static gboolean on_link_clicked(GtkWidget *widget, GdkEventButton *event,
				gpointer user_data) {

  const char *str = 
    gtk_label_get_text(GTK_LABEL(gtk_bin_get_child(GTK_BIN(widget))));
  browser_url((appdata_t*)user_data, (char*)str);
  return TRUE;
}
#endif

static GtkWidget *link_new(appdata_t *appdata, const char *url) {
#ifdef ENABLE_BROWSER_INTERFACE
  if(appdata) {
    GtkWidget *label = gtk_label_new("");
    char *str = g_strdup_printf("<span color=\"" LINK_COLOR 
				"\"><u>%s</u></span>", url);
    gtk_label_set_markup(GTK_LABEL(label), str);
    g_free(str);
    
    GtkWidget *eventbox = gtk_event_box_new();
    gtk_container_add(GTK_CONTAINER(eventbox), label);
    
    g_signal_connect(eventbox, "button-press-event", 
		     G_CALLBACK(on_link_clicked), appdata); 
    return eventbox;
  }
#endif
  GtkWidget *label = gtk_label_new("");
  char *str = g_strdup_printf("<span color=\"" LINK_COLOR "\">%s</span>", url);
  gtk_label_set_markup(GTK_LABEL(label), str);
  g_free(str);
  return label;
}

#ifdef ENABLE_BROWSER_INTERFACE
void on_paypal_button_clicked(GtkButton *button, appdata_t *appdata) {
  //  gtk_dialog_response(GTK_DIALOG(context->dialog), GTK_RESPONSE_ACCEPT); 
  browser_url(appdata, 
	      "https://www.paypal.com/cgi-bin/webscr"
	      "?cmd=_s-xclick&hosted_button_id=7400558");
}
#endif

static GtkWidget *notebook_new(void) {
#ifdef CUSTOM_NOTEBOOK
  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);

  GtkWidget *notebook =  gtk_notebook_new();

  /* solution for fremantle: we use a row of ordinary buttons instead */
  /* of regular tabs */

  /* hide the regular tabs */
  gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook), FALSE);

  gtk_box_pack_start_defaults(GTK_BOX(vbox), notebook);

  /* store a reference to the notebook in the vbox */
  g_object_set_data(G_OBJECT(vbox), "notebook", (gpointer)notebook);

  /* create a hbox for the buttons */
  GtkWidget *hbox = gtk_hbox_new(TRUE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  g_object_set_data(G_OBJECT(vbox), "hbox", (gpointer)hbox);

  return vbox;
#else
  return gtk_notebook_new();
#endif
}

#ifdef CUSTOM_NOTEBOOK
static void on_notebook_button_clicked(GtkWidget *button, gpointer data) {
  GtkNotebook *nb = 
    GTK_NOTEBOOK(g_object_get_data(G_OBJECT(data), "notebook"));

  gint page = (gint)g_object_get_data(G_OBJECT(button), "page");
  gtk_notebook_set_current_page(nb, page);
}
#endif

static void notebook_append_page(GtkWidget *notebook, 
				 GtkWidget *page, char *label) {
#ifdef CUSTOM_NOTEBOOK
  GtkNotebook *nb = 
    GTK_NOTEBOOK(g_object_get_data(G_OBJECT(notebook), "notebook"));

  gint page_num = gtk_notebook_append_page(nb, page, gtk_label_new(label));
  GtkWidget *button = NULL;

  /* select button for page 0 by default */
  if(!page_num) {
    button = gtk_radio_button_new_with_label(NULL, label);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
    g_object_set_data(G_OBJECT(notebook), "group_master", (gpointer)button);
  } else {
    GtkWidget *master = g_object_get_data(G_OBJECT(notebook), "group_master");
    button = gtk_radio_button_new_with_label_from_widget(
				 GTK_RADIO_BUTTON(master), label);
  }

  gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(button), FALSE);
  g_object_set_data(G_OBJECT(button), "page", (gpointer)page_num);

  gtk_signal_connect(GTK_OBJECT(button), "clicked",
	   GTK_SIGNAL_FUNC(on_notebook_button_clicked), notebook);

#if defined(USE_MAEMO) && (MAEMO_VERSION_MAJOR == 5)
  hildon_gtk_widget_set_theme_size(button, 
	   (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
#endif  

  gtk_box_pack_start_defaults(
     GTK_BOX(g_object_get_data(G_OBJECT(notebook), "hbox")),
     button);

#else
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, gtk_label_new(label));
#endif
}

GtkWidget *label_big(char *str) {
  GtkWidget *label = gtk_label_new("");
  char *markup = 
    g_markup_printf_escaped("<span size='x-large'>%s</span>", str);
  gtk_label_set_markup(GTK_LABEL(label), markup);
  g_free(markup);
  return label;
}

GtkWidget *label_xbig(char *str) {
  GtkWidget *label = gtk_label_new("");
  char *markup = 
    g_markup_printf_escaped("<span size='xx-large'>%s</span>", str);
  gtk_label_set_markup(GTK_LABEL(label), markup);
  g_free(markup);
  return label;
}

static void
on_label_realize(GtkWidget *widget, gpointer user_data)  {
  /* get parent size (which is a container) */
  gtk_widget_set_size_request(widget, widget->parent->allocation.width, -1);
}

GtkWidget *label_wrap(char *str) {
  GtkWidget *label = gtk_label_new(str);
  gtk_label_set_line_wrap_mode(GTK_LABEL(label), PANGO_WRAP_WORD);
  gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);

  g_signal_connect(G_OBJECT(label), "realize",
                   G_CALLBACK(on_label_realize), NULL);

  return label;
}

GtkWidget *license_page_new(void) {
  char *name = g_strdup(ICONPATH "COPYING");

  GtkWidget *label = label_wrap("");

  /* load license into buffer */
  FILE *file = fopen(name, "r");
  g_free(name);

  if(!file) {
    /* loading from installation path failed, try to load */
    /* from local directory (for debugging) */
    name = g_strdup("./data/COPYING");
    file = fopen(name, "r");
    g_free(name);
  }

  if(file) {
    fseek(file, 0l, SEEK_END);
    int flen = ftell(file);
    fseek(file, 0l, SEEK_SET);

    char *buffer = g_malloc(flen+1);
    fread(buffer, 1, flen, file);
    fclose(file);

    buffer[flen]=0;

    gtk_label_set_text(GTK_LABEL(label), buffer);

    g_free(buffer);
  } else
    gtk_label_set_text(GTK_LABEL(label), _("Load error"));

#ifndef USE_PANNABLE_AREA
  GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
  				 GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), 
					label);
  gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW(scrolled_window),
				       GTK_SHADOW_IN);
  return scrolled_window;
#else
  GtkWidget *pannable_area = hildon_pannable_area_new();
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(pannable_area), 
					label);
  return pannable_area;
#endif
}

GtkWidget *copyright_page_new(appdata_t *appdata) {
  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);

  /* ------------------------ */
  GtkWidget *ivbox = gtk_vbox_new(FALSE, 0);

  GtkWidget *hbox = gtk_hbox_new(FALSE, 0);
  GtkWidget *ihbox = gtk_hbox_new(FALSE, 20);
  gtk_box_pack_start(GTK_BOX(ihbox), icon_get_widget(ICON_MISC, 5),
		     FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(ihbox), label_xbig("GPXView"), 
		     FALSE, FALSE, 0);

  gtk_box_pack_start(GTK_BOX(hbox), ihbox, TRUE, FALSE, 0);
  gtk_box_pack_start_defaults(GTK_BOX(ivbox), hbox);

  gtk_box_pack_start_defaults(GTK_BOX(ivbox), 
		      label_big(_("Geocaching with Maemo")));

  gtk_box_pack_start(GTK_BOX(vbox), ivbox, TRUE, FALSE, 0);

  /* ------------------------ */
  ivbox = gtk_vbox_new(FALSE, 0);

  gtk_box_pack_start(GTK_BOX(ivbox), 
		      gtk_label_new("Version " VERSION), FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(ivbox), 
		      gtk_label_new(__DATE__ " " __TIME__), FALSE, FALSE, 0);

  gtk_box_pack_start(GTK_BOX(vbox), ivbox, TRUE, FALSE, 0);

  /* ------------------------ */
  ivbox = gtk_vbox_new(FALSE, 0);

  gtk_box_pack_start(GTK_BOX(ivbox), 
	      gtk_label_new(_("Copyright 2008-2009")), FALSE, FALSE, 0);

  gtk_box_pack_start(GTK_BOX(ivbox), 
      link_new(appdata, "http://www.harbaum.org/till/maemo#gpxview"),
			      FALSE, FALSE, 0);

  gtk_box_pack_start(GTK_BOX(vbox), ivbox, TRUE, FALSE, 0);

  return vbox;
}

/* a label that is left aligned */
GtkWidget *left_label(char *str) {
  GtkWidget *widget = gtk_label_new(str);
  gtk_misc_set_alignment(GTK_MISC(widget), 0.0f, 0.5f);
  return widget;
}

static void author_add(GtkWidget *box, char *str) {
  gtk_box_pack_start(GTK_BOX(box), left_label(str), FALSE, FALSE, 0);
}

GtkWidget *authors_page_new(void) {
  GtkWidget *ivbox, *vbox = gtk_vbox_new(FALSE, 16);

  /* -------------------------------------------- */
  ivbox = gtk_vbox_new(FALSE, 0);
  author_add(ivbox, _("Main developer:"));
  author_add(ivbox, "Till Harbaum <till@harbaum.org>");
  gtk_box_pack_start(GTK_BOX(vbox), ivbox, TRUE, FALSE, 0);

  /* -------------------------------------------- */
  ivbox = gtk_vbox_new(FALSE, 0);
  author_add(ivbox, _("Original map widget by:"));
  author_add(ivbox, "John Stowers <john.stowers@gmail.com>");
  gtk_box_pack_start(GTK_BOX(vbox), ivbox, TRUE, FALSE, 0);

  /* -------------------------------------------- */
  ivbox = gtk_vbox_new(FALSE, 0);
  author_add(ivbox, _("GCVote service provided by:"));
  author_add(ivbox, "Guido Wegener <guido.wegener@gmx.de>");
  gtk_box_pack_start(GTK_BOX(vbox), ivbox, TRUE, FALSE, 0);

  /* -------------------------------------------- */
  ivbox = gtk_vbox_new(FALSE, 0);
  author_add(ivbox, _("Additional translations by:"));
  author_add(ivbox, "Marko Vertainen <marko.vertainen@iki.fi>");
  gtk_box_pack_start(GTK_BOX(vbox), ivbox, TRUE, FALSE, 0);

  /* -------------------------------------------- */
  ivbox = gtk_vbox_new(FALSE, 0);
  author_add(ivbox, _("Testers:"));
  author_add(ivbox, "Uwe Koch <asys3@yahoo.com>");
  author_add(ivbox, "Tanja Harbaum <tanja@harbaum.org>");
  gtk_box_pack_start(GTK_BOX(vbox), ivbox, TRUE, FALSE, 0);

#ifndef USE_PANNABLE_AREA
  GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
  				 GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), 
					vbox);
  gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW(scrolled_window),
				       GTK_SHADOW_IN);
  return scrolled_window;
#else
  GtkWidget *pannable_area = hildon_pannable_area_new();
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(pannable_area), 
					vbox);
  return pannable_area;
#endif
}  

GtkWidget *donate_page_new(appdata_t *appdata) {
  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);

  gtk_box_pack_start_defaults(GTK_BOX(vbox), 
      label_wrap(_("If you like GPXView and want to support its future development "
		   "please consider donating to the developer. You can either "
		   "donate via paypal to")));
  
  gtk_box_pack_start_defaults(GTK_BOX(vbox), 
			      link_new(NULL, "till@harbaum.org"));
  
  gtk_box_pack_start_defaults(GTK_BOX(vbox), 
      label_wrap(_("or you can just click the button below which will open "
		   "the appropriate web page in your browser.")));

  GtkWidget *ihbox = gtk_hbox_new(FALSE, 0);
  GtkWidget *button = gtk_button_new();
  gtk_button_set_image(GTK_BUTTON(button), 
		       icon_get_widget(ICON_MISC, 3));
  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
  g_signal_connect(button, "clicked", 
		   G_CALLBACK(on_paypal_button_clicked), appdata); 
  gtk_box_pack_start(GTK_BOX(ihbox), button, TRUE, FALSE, 0);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), ihbox);

  return vbox;
}  

GtkWidget *bugs_page_new(appdata_t *appdata) {
  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);

  gtk_box_pack_start_defaults(GTK_BOX(vbox), 
      label_wrap(_("Please report bugs or feature requests via the GPXView "
		   "bug tracker. This bug tracker can directly be reached via "
		   "the following link:")));

  gtk_box_pack_start_defaults(GTK_BOX(vbox), 
       link_new(appdata, "https://garage.maemo.org/tracker/?group_id=973"));

  gtk_box_pack_start_defaults(GTK_BOX(vbox), 
      label_wrap(_("You might also be interested in joining the mailing lists "
		   "or the forum:")));

  gtk_box_pack_start_defaults(GTK_BOX(vbox), 
	      link_new(appdata, "http://garage.maemo.org/projects/gpxview/"));

  gtk_box_pack_start_defaults(GTK_BOX(vbox), 
      label_wrap(_("Thank you for contributing!")));

  return vbox;
}  

void about_box(appdata_t *appdata) {
  GtkWidget *dialog = gtk_dialog_new_with_buttons(_("About GPXView"),
	  GTK_WINDOW(appdata->window), GTK_DIALOG_MODAL,
          GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);

#ifdef USE_MAEMO
  gtk_window_set_default_size(GTK_WINDOW(dialog), 640, 480);
#else
  gtk_window_set_default_size(GTK_WINDOW(dialog), 400, 200);
#endif

  GtkWidget *notebook = notebook_new();

  notebook_append_page(notebook, copyright_page_new(appdata), _("Copyright"));
  notebook_append_page(notebook, license_page_new(),          _("License"));
  notebook_append_page(notebook, authors_page_new(),          _("Authors"));
  notebook_append_page(notebook, donate_page_new(appdata),    _("Donate"));
  notebook_append_page(notebook, bugs_page_new(appdata),      _("Bugs"));

  gtk_box_pack_start_defaults(GTK_BOX((GTK_DIALOG(dialog))->vbox),
			      notebook);

  gtk_widget_show_all(dialog);

  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);
}
