/*
 * Copyright (C) 2009 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"

#define __USE_GNU
#include <string.h>

#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <errno.h>
#include <math.h>

#if defined(USE_MAEMO) && (MAEMO_VERSION_MAJOR >= 5)
#include <hildon/hildon-entry.h>
#endif

#define GEOTOAD "/usr/bin/geotoad"

#define COLOR_ERR     "red"
#define COLOR_OK      "darkgreen"
#define COLOR_SYSTEM  "darkblue"

#define BUFFER_SIZE  256

typedef enum {
  GT_STATE_ILLEGAL = 0,
  GT_STATE_STARTED,
  GT_STATE_PREMATURE_END,
  GT_STATE_SAVE_FOUND,
} gt_state_t;

typedef struct {
  appdata_t *appdata;
  
  GPtrArray *argv;
  gchar *info, *color;
  GMutex *update_mutex;
  GCond *update_cond;

  gt_state_t state;

  GtkWidget *dialog;

  long pid;

  struct log_s {
    GtkTextBuffer *buffer;
    GtkWidget *view;
  } log;

  GtkWidget *username, *password;
  GtkWidget *lat, *lon, *dst;
  char *filename;

  int use_cnt;
  
} gt_context_t;

static void arg_free(gpointer data, gpointer user_data) {
  if(data) g_free(data);
}

static void context_free(gt_context_t *context) {
  context->use_cnt--;
    
  if(context->use_cnt > 0)
    printf("still in use by %d, keeping context\n", context->use_cnt);
  else {
    printf("freeing context\n");

    if(context->info) g_free(context->info);

    if(context->argv) {
      g_ptr_array_foreach(context->argv, arg_free, NULL);
      g_ptr_array_free (context->argv, TRUE);
    }

    if(context->filename) g_free(context->filename);

    g_free(context);
  }
}

static void appendf(struct log_s *log, char *colname, 
		    const char *fmt, ...) {
  va_list args;
  va_start( args, fmt );
  char *buf = g_strdup_vprintf(fmt, args);
  va_end( args );

  //  printf("append: %s", buf);

  GtkTextTag *tag = NULL;
  if(colname)
    tag = gtk_text_buffer_create_tag(log->buffer, NULL,
				     "foreground", colname,
				     NULL);
  
  GtkTextIter end;
  gtk_text_buffer_get_end_iter(log->buffer, &end);
  if(tag) 
    gtk_text_buffer_insert_with_tags(log->buffer, &end, buf, -1, tag, NULL);
  else    
    gtk_text_buffer_insert(log->buffer, &end, buf, -1);

  g_free(buf);

  gtk_text_view_scroll_to_iter(GTK_TEXT_VIEW(log->view),
			       &end, 0.0, TRUE, 1.0, 1.0); 
}

// This function receives a requst from a worker thread asking to
// update the gui with the required info.
gboolean cb_update_job(gt_context_t *context) {
  if (context->info) {

    /* check if client reports "saved to ..." */
    if(strstr(context->info, "Saved to "))
      context->state = GT_STATE_SAVE_FOUND;

    appendf(&context->log, context->color, context->info);
    
    g_free(context->info);
    context->info = NULL;
  }
  
  // Indicate that the update is done
  g_mutex_lock(context->update_mutex);
  g_cond_signal(context->update_cond);
  g_mutex_unlock(context->update_mutex);
  
  return FALSE;
}

// A helper function run in the job thread receiving the string that
// should be displayed in the textview.
void append_parentf(gt_context_t *context, char *colname, 
	     const char *fmt, ...) {
  va_list args;
  va_start( args, fmt );
  context->info = g_strdup_vprintf(fmt, args);
  va_end( args );

  if(colname)
    context->color = colname;
  else
    context->color = NULL;

  // Lock mutex to make sure that we will receive the condition signal
  g_mutex_lock(context->update_mutex);

  g_idle_add((GSourceFunc)cb_update_job, context);
  
  // Wait for cb_update_job to tell me that the update is done
  g_cond_wait(context->update_cond, context->update_mutex);
  g_mutex_unlock(context->update_mutex);
}

/* custom version of popen to get access to the pid and to be able */
/* to slaughter the child process */
static FILE *gt_popen(gt_context_t *context, char **args, 
		      const char *type, long *tid) {
  int   p[2];
  FILE *fp;
  
  if (*type != 'r' && *type != 'w')
    return NULL;
  
  if (pipe(p) < 0)
    return NULL;
  
  if ((*tid = fork()) > 0) { /* then we are the parent */
    if (*type == 'r') {
      close(p[1]);
      fp = fdopen(p[0], type);
    } else {
      close(p[0]);
      fp = fdopen(p[1], type);
    }
    
    return fp;
  } else if (*tid == 0) {  /* we're the child */

    /* make our thread id the process group leader */
    setpgid(0, 0);
    
    if (*type == 'r') {
      fflush(stdout);
      fflush(stderr);
      close(1);
      if (dup(p[1]) < 0)
        perror("dup of write side of pipe failed");
      close(2);
      if (dup(p[1]) < 0)
        perror("dup of write side of pipe failed");
    } else {
      close(0);
      if (dup(p[0]) < 0)
        perror("dup of read side of pipe failed");
    }
    
    close(p[0]); /* close since we dup()'ed what we needed */
    close(p[1]);
    
    execve(args[0], args, NULL);
    
    /* this printf will actually be redirected through the pipe */
    printf("Error: Failed to execute!\n");
    exit(1);
  } else {         /* we're having major problems... */
    close(p[0]);
    close(p[1]);

    /* this printf will actually be redirected through the pipe??? No! */
    printf("Error: Failed to fork!\n");
  }
  
  return NULL;
}

// The thread entry point. It will do the job, send the data to the
// GUI and self destruct when it is done.
static gpointer thread_worker(gt_context_t *context)
{
  FILE *fh;
  GIOStatus status;
  GError *error = NULL;
  gsize length;
  gsize terminator_pos;
  gchar *str_return;

  fh = gt_popen(context, (char**)(context->argv->pdata),"r", &context->pid);
  if(!fh) {
    printf("fail free\n");
    context_free(context);
    //    g_thread_exit(NULL);
    return NULL;
  }

  /* the client is running */
  printf("client inc use_cnt\n");
  context->use_cnt++;

  /* switch to line buffered mode */
  if(setvbuf(fh, NULL, _IOLBF, 0))
    perror("setvbuf(_IOLBF)");
  
  GIOChannel *gh = g_io_channel_unix_new(fileno(fh));

  while( (status = g_io_channel_read_line(gh,
					  &str_return,
					  &length,
					  &terminator_pos,
					  &error)) == G_IO_STATUS_NORMAL) {
    char *color = NULL;
    if(strstr(str_return, "Saved to "))  color = COLOR_OK;
    if(strcasestr(str_return, "error"))  color = COLOR_ERR;
    
    append_parentf(context, color, str_return);
    g_free(str_return);
  }
  
  g_io_channel_unref(gh);
  pclose(fh);
  append_parentf(context, COLOR_SYSTEM, "Job done!");

  context_free(context);
  
  g_thread_exit(NULL);
  return NULL;
}


static void arg_dsp(gpointer data, gpointer user_data) {
  gt_context_t *context = (gt_context_t*)user_data;
  
  if(data)
    appendf(&context->log, COLOR_SYSTEM, "%s\n", data);
}

static void run(gt_context_t *context) {
  GError *error = NULL;
  char str[8];

  /* build list of arguments to call geotoad */
  context->argv = g_ptr_array_new();
  g_ptr_array_add (context->argv, g_strdup_printf(GEOTOAD));
  g_ascii_dtostr(str, sizeof(str), context->appdata->gt.distance);
  g_ptr_array_add (context->argv, 
	     g_strdup_printf("--distanceMax=%s", str));
  g_ptr_array_add (context->argv, 
	     g_strdup_printf("--output=%s", context->appdata->gt.filename));
  g_ptr_array_add (context->argv, 
	     g_strdup_printf("--password=%s", context->appdata->gt.password));
  g_ptr_array_add (context->argv, 
	     g_strdup_printf("--queryType=coord"));
  g_ptr_array_add (context->argv,
	     g_strdup_printf("--user=%s", context->appdata->username));

  /* check if we need to add proxy config */
  char *proxy = NULL;

  if(context->appdata->proxy && context->appdata->proxy->host) {
    if(context->appdata->proxy->use_authentication &&
       context->appdata->proxy->authentication_user &&
       context->appdata->proxy->authentication_password)
      proxy = g_strdup_printf("--proxy=http://%s:%s@%s:%d", 
                              context->appdata->proxy->authentication_user,
                              context->appdata->proxy->authentication_password,
                              context->appdata->proxy->host,
                              context->appdata->proxy->port);
    else
      proxy = g_strdup_printf("--proxy=http://%s:%d", 
                              context->appdata->proxy->host,
                              context->appdata->proxy->port);
    
  } else {
    /* use environment settings if preset (for scratchbox) */
    const char *proxy_env = g_getenv("http_proxy");
    if(proxy_env)
      proxy = g_strdup_printf("--proxy=%s", proxy_env);
  }
  
  if(proxy)
    g_ptr_array_add (context->argv, proxy);

  /* convert coordinates into simple ascii format */
  char n = (context->appdata->gt.lat >= 0)?'N':'S';
  char e = (context->appdata->gt.lon >= 0)?'E':'W';
  float lat = fabs(context->appdata->gt.lat);
  float lon = fabs(context->appdata->gt.lon);
  float lat_mint, lat_int, lat_frac = modff(lat, &lat_int);
  float lon_mint, lon_int, lon_frac = modff(lon, &lon_int);
  lat_frac = modff(lat_frac*60.0, &lat_mint);
  lon_frac = modff(lon_frac*60.0, &lon_mint);

  g_ptr_array_add (context->argv, 
	   g_strdup_printf("%c%02u %02u.%03u %c%03u %02u.%03u",
	   n, (int)lat_int, (int)lat_mint, (int)(lat_frac*1000.0+0.5),
	   e, (int)lon_int, (int)lon_mint, (int)(lon_frac*1000.0+0.5)));

  g_ptr_array_add (context->argv, NULL);

  /* show all entries */
  g_ptr_array_foreach(context->argv, arg_dsp, context);

  g_thread_create((GThreadFunc)thread_worker, context, FALSE, &error);
  if (error) {
    appendf(&context->log, COLOR_ERR, "Error: %s\n", error->message);
    g_error_free(error);
  }
}
	
/* show text window and display output of running geotoad */
static void gui_run(gt_context_t *context) {
  GtkWidget *dialog = gtk_dialog_new_with_buttons(_("GeoToad - Run"),
                          GTK_WINDOW(context->appdata->window),
			  GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                          GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
                          NULL);

  gtk_window_set_default_size(GTK_WINDOW(dialog), 640, 480);

#ifndef USE_PANNABLE_AREA
  GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
  				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
#else
  GtkWidget *pannable_area = hildon_pannable_area_new();
#endif
  
  context->log.buffer = gtk_text_buffer_new(NULL);

#ifndef USE_HILDON_TEXT_VIEW
  context->log.view = gtk_text_view_new_with_buffer(context->log.buffer);
#else
  context->log.view = hildon_text_view_new();
  hildon_text_view_set_buffer(HILDON_TEXT_VIEW(context->log.view), 
			      context->log.buffer);
#endif

#ifndef USE_PANNABLE_AREA
  gtk_container_add(GTK_CONTAINER(scrolled_window), context->log.view);
  gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW(scrolled_window),
				       GTK_SHADOW_IN);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), 
		     scrolled_window, TRUE, TRUE, 0);
#else
  gtk_container_add(GTK_CONTAINER(pannable_area), context->log.view);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), 
		     pannable_area, TRUE, TRUE, 0);
#endif

  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_CLOSE);

  gtk_widget_show_all(dialog);

  appendf(&context->log, COLOR_SYSTEM, "Running GeoToad\n");
  run(context);

  gtk_dialog_run(GTK_DIALOG(dialog));

  gtk_widget_destroy(dialog);
}

static void table_attach(GtkWidget *table, GtkWidget *child, int left, int top) {
  gtk_table_attach_defaults(GTK_TABLE(table), child, left, left+1, top, top+1);
}


static gboolean gui_setup(gt_context_t *context) {
  appdata_t *appdata = context->appdata;
  gboolean ok = FALSE;

  /* if no filename has been setup yet, create one */
  if(!appdata->gt.filename && appdata->path) {
    printf("creating path\n");
    appdata->gt.filename = 
      g_strdup_printf("%s/gtoad.gpx", appdata->path);
  }

  if(appdata->gt.filename)
    context->filename = g_strdup(appdata->gt.filename);

  context->dialog = gtk_dialog_new_with_buttons(_("GeoToad - Setup"),
			GTK_WINDOW(appdata->window),
			GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OK,     GTK_RESPONSE_OK,
			NULL);

  GtkWidget *table = gtk_table_new(5, 2, FALSE);

  /* ------------------- Coordinates ------------------------- */

  table_attach(table, left_label_new(_("Position:")), 0, 0);
  table_attach(table, left_label_new(_("Distance:")), 0, 1);

  /* setup default positions */
  pos_t *refpos = get_pos(appdata);
  if((isnan(appdata->gt.lat) || isnan(appdata->gt.lat)) && refpos) {
    appdata->gt.lat = refpos->lat;
    appdata->gt.lon = refpos->lon;
  }

  GtkWidget *hbox = gtk_hbox_new(FALSE, 0);
  context->lat = lat_entry_new(appdata->gt.lat);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context->lat);
  context->lon = lon_entry_new(appdata->gt.lon);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context->lon);
  GtkWidget *popup = preset_coordinate_picker(appdata, context->lat, context->lon);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), popup);
  table_attach(table, hbox, 1, 0);

  float dst = appdata->gt.distance;  // distance is given in kilometers
  if(appdata->imperial) dst /= 1.609344;
  context->dst = dist_entry_new(dst, appdata->imperial);
  table_attach(table, context->dst, 1, 1);

  /* ------------------- Username/Password ------------------------- */
  table_attach(table, left_label_new(_("Username:")), 0, 2);
  table_attach(table, left_label_new(_("Password:")), 0, 3);
  
  context->username = entry_new();
  context->password = entry_new();
  gtk_entry_set_visibility(GTK_ENTRY(context->password), FALSE);

  /* set saved defaults */
  if(appdata->username)
    gtk_entry_set_text(GTK_ENTRY(context->username), 
		       appdata->username);

  if(appdata->gt.password)
    gtk_entry_set_text(GTK_ENTRY(context->password), 
		       appdata->gt.password);

  table_attach(table, context->username, 1, 2);
  table_attach(table, context->password, 1, 3);

  /* ------------------- file name ------------------------- */
  gtk_table_attach_defaults(GTK_TABLE(table), 
	    export_file(_("Save GPX file"), &context->filename),
	       0, 2, 4, 5);

  /* ---------------------------------------------------------------- */
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(context->dialog)->vbox), 
			      table);

  gtk_dialog_set_default_response(GTK_DIALOG(context->dialog), 
				  GTK_RESPONSE_OK);

  gtk_widget_show_all(context->dialog);

  if(gtk_dialog_run(GTK_DIALOG(context->dialog)) == GTK_RESPONSE_OK) {

    /* parse coordinates */
    appdata->gt.lat = lat_entry_get(context->lat);
    appdata->gt.lon = lon_entry_get(context->lon);

    /* save values */
    if(appdata->username) g_free(appdata->username);
    appdata->username = 
      g_strdup(gtk_entry_get_text(GTK_ENTRY(context->username)));
					     
    if(appdata->gt.password) g_free(appdata->gt.password);
    appdata->gt.password = 
      g_strdup(gtk_entry_get_text(GTK_ENTRY(context->password)));

    if(appdata->gt.filename) g_free(appdata->gt.filename);
    if(context->filename)
      appdata->gt.filename = g_strdup(context->filename);

    /* get distance in kilometers */
    appdata->gt.distance = dist_entry_get(context->dst, FALSE);


    /* check for valid entries */
    if(isnan(appdata->gt.lat) || isnan(appdata->gt.lon) ||
       isnan(appdata->gt.distance) || !appdata->gt.filename ||
       !appdata->username || !appdata->gt.password) 
      errorf(_("The GeoToad setup is not complete."));
    else
      ok = TRUE;
  }

  gtk_widget_destroy(context->dialog);

  return ok;
}

void geotoad(appdata_t *appdata) {
  if(!geotoad_available()) {
    errorf(_("GeoToad is not installed on this device.\n"
	     "You need to install it in order to be able to use it."));
    return;
  }

  gt_context_t *context = g_new0(gt_context_t, 1);
  context->appdata = appdata;
  context->use_cnt++;          // parent still uses this

  context->update_mutex = g_mutex_new();
  context->update_cond = g_cond_new();

  if(gui_setup(context))
    gui_run(context);

  /* continue to process if something has actually been saved */
  if(context->state == GT_STATE_SAVE_FOUND) {
    /* download seems to be successful. Make sure the GUI is */
    /* updated if required */

    gpx_t **gpx = &(appdata->gpx);
    while(*gpx && strcmp((*gpx)->filename, appdata->gt.filename))
      gpx = &(*gpx)->next;

    /* return main GUI to GPX list */
#ifdef USE_BREAD_CRUMB_TRAIL
    while(appdata->cur_gpx)
      hildon_bread_crumb_trail_pop(HILDON_BREAD_CRUMB_TRAIL(appdata->bct));
#elif defined(BCT)
    while(appdata->cur_gpx)
      bct_pop(appdata->bct);
#else
    HildonWindowStack *stack = hildon_window_stack_get_default();
    gint num = hildon_window_stack_size(stack)-1;
    while(num--) {
      GtkWidget *top = hildon_window_stack_peek(stack);
      gtk_widget_destroy(top);
    }
#endif

    /* replace an existing entry or add to end of list */
    if(*gpx) {
      GtkTreeIter iter;
      g_assert(gpxlist_find(appdata, &iter, *gpx));

      gpx_t *next = (*gpx)->next;

      gpx_free(*gpx);
      *gpx = gpx_parse(NULL, appdata->gt.filename, appdata->username);
      (*gpx)->next = next;

      /* update gpxlist */
      gpxlist_set(appdata->gpxstore, &iter, *gpx);

      /* select that row */
      GtkTreeSelection *selection = 
	gtk_tree_view_get_selection(GTK_TREE_VIEW(appdata->gpxview));
      gtk_tree_selection_select_iter(selection, &iter);
      GtkTreePath *path = 
	gtk_tree_model_get_path(GTK_TREE_MODEL(appdata->gpxstore), &iter);
      gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(appdata->gpxview), 
				   path, NULL, TRUE, 0.0, 0.0);
      gtk_tree_path_free(path);
    } else {
      gpx_t *new = gpx_parse(NULL, appdata->gt.filename, appdata->username);
      if(new) gpxlist_add(appdata, new);
    }
  }

  printf("main context free\n");
  context_free(context);
}

gboolean geotoad_available(void) {
  /* before doing anything make sure geotoad is installed */
  return g_file_test(GEOTOAD, G_FILE_TEST_IS_EXECUTABLE);
}


