/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

/* TODO:
*/

#include <math.h>
#include <string.h>
#include <ctype.h>

#include <glib.h>
#include <glib/gstdio.h>

#include "gpxview.h"

#ifdef ENABLE_OSM_GPS_MAP
#include "osm-gps-map.h"
#endif

#if defined(USE_MAEMO) && (MAEMO_VERSION_MAJOR >= 5)
#include <hildon/hildon-entry.h>
#include <hildon/hildon-touch-selector.h>
#include <hildon/hildon-picker-button.h>
#include <hildon/hildon-picker-dialog.h>
#include <hildon/hildon-check-button.h>
#endif

float roundf(float x);

/* Enable special fremantle UI elements. These particular widgets */
/* don't use any hildon/fremantle specific parts and can thus even */
/* be used under plain gtk. This is mainly for testing and developemt */
#ifdef FREMANTLE
#define PRESET_PICKER_DIALOG
#define COORDINATE_PICKER
#endif

char strlastchr(char *str) {
  return str[strlen(str)]-1;
}

/* make sure the entire path "dir" exists and create it if not */
int checkdir(char *dir) {
  struct stat filestat;
  char *p = dir, tmp;

  /* don't try to create root dir */
  if(p[0] == '/') p++;

  do {
    while(*p && *p != '/') p++;
    tmp = *p;
    *p = 0;

    int err = stat(dir, &filestat);
    if(err) {
      if(mkdir(dir, S_IRWXU) != 0) {
	perror("mkdir()");
	*p++ = tmp;
	return -1;
      }
    } else {
      if(!filestat.st_mode & S_IFDIR) {
	printf("File %s exists and is _no_ directory\n", dir);
	*p++ = tmp;
	return -1;
      }
    }
    
    *p++ = tmp;
  } while(tmp && strchr(p, '/'));

  return 0;
}

void pos_lat_str(char *str, int len, float latitude) {
  char *c = _("N");
  float integral, fractional;

  if(isnan(latitude)) 
    str[0] = 0;
  else {
    if(latitude < 0) { latitude = fabs(latitude); c = _("S"); }
    fractional = modff(latitude, &integral);

    snprintf(str, len, "%s %02d° %06.3f'", c, (int)integral, fractional*60.0);
  }
}

GtkWidget *pos_lat(float latitude, int size, int strikethrough) {
  char str[32];

  pos_lat_str(str, sizeof(str), latitude);
  return gtk_label_attrib(str, size, strikethrough);
}

void pos_lon_str(char *str, int len, float longitude) {
  char *c = _("E");
  float integral, fractional;

  if(isnan(longitude)) 
    str[0] = 0;
  else {
    if(longitude < 0) { longitude = fabs(longitude); c = _("W"); }
    fractional = modff(longitude, &integral);

    snprintf(str, len, "%s %03d° %06.3f'", c, (int)integral, fractional*60.0);
  }
}

GtkWidget *pos_lon(float longitude, int size, int strikethrough) {
  char str[32];

  pos_lon_str(str, sizeof(str), longitude);
  return gtk_label_attrib(str, size, strikethrough);
}

float pos_parse_lat(char *str) {
  int integral_int;
  float fractional;
  char c;

  if(sscanf(str, "%c %d° %f'", &c, &integral_int, &fractional) == 3) {
    c = toupper(c);
    
    if(c != 'S' && c != 'N')
      return NAN;
    
    /* prevent -0.0 */
    if(!integral_int && (fractional == 0.0))
      return 0.0;

    return ((c == 'S')?-1:+1) * (integral_int + fractional/60.0);
  }
  
  return NAN;
}

float pos_parse_lon(char *str) {
  int integral_int;
  float fractional;
  char c;

  if(sscanf(str, "%c %d° %f'", &c, &integral_int, &fractional) == 3) {
    c = toupper(c);

    /* O is german "Ost" for "East" */
    if(c != 'E' && c != 'W' && c != 'O')
      return NAN;

    /* prevent -0.0 */
    if(!integral_int && (fractional == 0.0))
      return 0.0;

    return ((c == 'W')?-1:+1) * (integral_int + fractional/60.0);
  }

  return NAN;
}

const char *pos_get_bearing_str(pos_t from, pos_t to) {
  static const char *bear_str[]={ 
    "N", "NE", "E", "SE", "S", "SW", "W", "NW", "" };

  float bearing = gpx_pos_get_bearing(from, to);
  if(!isnan(bearing)) {
    int idx = (bearing+22.5)/45.0;
    /* make sure we stay in icon bounds */
    while(idx < 0) idx += 8;
    while(idx > 7) idx -= 8;
    return _(bear_str[idx]);
  }

  return bear_str[8];  // empty string
}

/* the maemo font size is quite huge, so we adjust some fonts */
/* differently on maemo and non-maemo. Basically "BIG" does nothing */
/* on maemo and "SMALL" only does something on maemo */
#ifdef USE_MAEMO
#define MARKUP_SMALL "<span size='small'>%s</span>"
GtkWidget *gtk_label_small(char *str) {
  GtkWidget *label = gtk_label_new("");
  char *markup = g_markup_printf_escaped(MARKUP_SMALL, str);
  gtk_label_set_markup(GTK_LABEL(label), markup);
  g_free(markup);
  return label;
}
#else
#define MARKUP_BIG "<span size='x-large'>%s</span>"
GtkWidget *gtk_label_big(char *str) {
  GtkWidget *label = gtk_label_new("");
  char *markup = g_markup_printf_escaped(MARKUP_BIG, str);
  gtk_label_set_markup(GTK_LABEL(label), markup);
  g_free(markup);
  return label;
}
#endif

void gtk_label_attrib_set(GtkWidget *label, 
			  char *str, int size, int strikethrough) {
  char format[80];

  snprintf(format, sizeof(format), "<span%s%s%s>%%s</span>", 
#ifdef USE_MAEMO
   (size==SIZE_SMALL)?" size='small'":"",
#else
   (size==SIZE_BIG)?" size='x-large'":"",
#endif
   strikethrough?" strikethrough='yes'":"",
   (strikethrough==STRIKETHROUGH_RED)?" strikethrough_color='red'":"");

  char *markup = g_markup_printf_escaped(format, str);
  //  printf("markup = %s\n", markup);
  gtk_label_set_markup(GTK_LABEL(label), markup);
  g_free(markup);
}

GtkWidget *gtk_label_attrib(char *str, int size, int strikethrough) {
  GtkWidget *label = gtk_label_new("");
  gtk_label_attrib_set(label, str, size, strikethrough);
  return label;
}

GtkWidget *gtk_button_attrib(char *str, int size, int strikethrough) {
  GtkWidget *button = gtk_button_new_with_label("");
  gtk_label_attrib_set(gtk_bin_get_child(GTK_BIN(button)), 
		       str, size, strikethrough);
  return button;
}

void textbox_disable(GtkWidget *widget) {
  gtk_editable_set_editable(GTK_EDITABLE(widget), FALSE);
  gtk_widget_set_sensitive(widget, FALSE);
}

void textbox_enable(GtkWidget *widget) {
  gtk_widget_set_sensitive(widget, TRUE);
  gtk_editable_set_editable(GTK_EDITABLE(widget), TRUE);
}

pos_t *get_pos(appdata_t *appdata) {
  pos_t *pos = &appdata->home;

  if(appdata->active_location) {
    int i = appdata->active_location-1;
    location_t *loc = appdata->location;
    while(i--) loc = loc->next;
    pos = &loc->pos;
  }

  if(appdata->use_gps) {
    pos = gps_get_pos(appdata);

    if(!pos) pos = &appdata->gps;   /* use saved position */
    else     appdata->gps = *pos;   /* save position */
  }
  return pos;
}

void distance_str(char *str, int len, float dist, gboolean imperial) {
  if(isnan(dist))
    snprintf(str, len, "---");
  else if(imperial) {
    /* 1 mil = 1760 yd = 5280 ft ... */
    if(dist<0.018)      snprintf(str, len, "%.1f ft", dist*5280.0);
    else if(dist<0.055) snprintf(str, len, "%.1f yd", dist*1760.0);
    else if(dist<0.55)  snprintf(str, len, "%.0f yd", dist*1760.0);
    else if(dist<10.0)  snprintf(str, len, "%.2f mi", dist);
    else if(dist<100.0) snprintf(str, len, "%.1f mi", dist);
    else                snprintf(str, len, "%.0f mi", dist);
  } else {
    if(dist<0.01)       snprintf(str, len, "%.2f m",  dist*1000.0);
    else if(dist<0.1)   snprintf(str, len, "%.1f m",  dist*1000.0);
    else if(dist<1.0)   snprintf(str, len, "%.0f m",  dist*1000.0);
    else if(dist<100.0) snprintf(str, len, "%.1f km", dist);
    else                snprintf(str, len, "%.0f km", dist);
  }
}

/* return distance in miles or kilometers */
float distance_parse(char *str, gboolean imperial) {
  char unit[4];
  float val = NAN;

  if(sscanf(str, "%f %3s", &val, unit) == 2) {
    gboolean fimp = FALSE;

    if(strcasecmp(unit, "ft") == 0)      { fimp = TRUE;  val /= 5280.0; }
    else if(strcasecmp(unit, "yd") == 0) { fimp = TRUE;  val /= 1760.0; }
    else if(strcasecmp(unit, "mi") == 0) { fimp = TRUE;  }
    else if(strcasecmp(unit, "m")  == 0) { fimp = FALSE; val /= 1000.0; }
    else if(strcasecmp(unit, "km") == 0) { fimp = FALSE; }
    else val = NAN;

    /* found imperial and metric requested? convert miles into kilometers */
    if(fimp & !imperial) val *= 1.609344;

    /* found metric and imperial requested? convert kilometers into miles */
    if(!fimp & imperial) val /= 1.609344;
  }
  return val;
}

/* ------------------ coordinate picker tool --------------------------- */

#ifndef COORDINATE_PICKER
static gboolean mark(GtkWidget *widget, gboolean valid) {
  gtk_widget_set_state(widget, valid?GTK_STATE_NORMAL:TAG_STATE);
  return valid;
}

GtkWidget *red_entry_new_with_text(char *str) {
  GdkColor color;

  GtkWidget *widget = entry_new();
  gdk_color_parse("#ff0000", &color);
  gtk_widget_modify_text(widget, TAG_STATE, &color);
  if(str) gtk_entry_set_text(GTK_ENTRY(widget), str);

  return widget;
}

static void callback_modified_lat(GtkWidget *widget, gpointer data ) {
  float i = pos_parse_lat((char*)gtk_entry_get_text(GTK_ENTRY(widget)));
  mark(widget, !isnan(i));
}
#else
static void picker_center_iter(GtkWidget *view, GtkTreeIter *iter) {
  /* select new iter */
  GtkTreeSelection *selection = 
    gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
  GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(view));
  gtk_tree_selection_select_iter(selection, iter);
  GtkTreePath *path = 
    gtk_tree_model_get_path(model, iter);
  gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(view), 
			       path, NULL, TRUE, 0.5, 0.5);
  gtk_tree_path_free(path);
}

static void on_picker_activated(GtkTreeView        *treeview,
				GtkTreePath        *path,
				GtkTreeViewColumn  *col,
				gpointer            userdata) {
  gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(treeview), 
			       path, NULL, TRUE, 0.5, 0.5);
}

#define DUMMY_NUM  1

static void dummy_append(GtkListStore *store, int n) {
  GtkTreeIter iter;
 
  while(n--) {
    gtk_list_store_append (store, &iter);
    gtk_list_store_set(store, &iter, 0, "", 1, -1, 2, FALSE, -1);
  }
}

static GtkWidget *string_picker_create(const char *str[], int sel) {
#ifndef FREMANTLE
  GtkWidget *view = gtk_tree_view_new();
#else
  GtkWidget *view = hildon_gtk_tree_view_new(HILDON_UI_MODE_EDIT);
#endif

  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), FALSE); 
  GtkTreeSelection *selection = 
    gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
  
  /* --- "char" column --- */
  GtkCellRenderer *renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "xalign", 0.5, NULL );
  gtk_tree_view_insert_column_with_attributes(
      GTK_TREE_VIEW(view), -1, "str", renderer, 
      "text", 0,
      "sensitive", 2, 
      NULL);

  GtkListStore *store = 
    gtk_list_store_new(3, G_TYPE_STRING, G_TYPE_INT, G_TYPE_BOOLEAN); 

  dummy_append(store, DUMMY_NUM);

  /* add strings */
  GtkTreeIter iter, siter;
  int i;
  for(i=0;*str;str++,i++) {
    gtk_list_store_append (store, &iter);
    gtk_list_store_set(store, &iter, 0, _(*str), 1, i, 2, TRUE, -1);
    if(i == sel) siter = iter;
  }

  dummy_append(store, DUMMY_NUM);

  gtk_tree_view_set_model(GTK_TREE_VIEW(view), GTK_TREE_MODEL(store));
  g_object_unref(store);

#ifndef FREMANTLE
  g_signal_connect(view, "row-activated", 
		   (GCallback)on_picker_activated, NULL);
#else
  g_signal_connect(view, "hildon-row-tapped", 
		   (GCallback)on_picker_activated, NULL);
#endif

  /* select right character */
  gtk_tree_selection_select_iter(selection, &siter);
  picker_center_iter(view, &siter);
  
  /* put this inside a scrolled view */
#ifndef USE_PANNABLE_AREA
  GtkWidget *scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
				 GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scrolled_window), view);
  return scrolled_window;
#else
  GtkWidget *pannable_area = hildon_pannable_area_new();
  gtk_container_add(GTK_CONTAINER(pannable_area), view);
  return pannable_area;
#endif
}

static int picker_get(GtkWidget *widget) {
  GtkWidget *view = gtk_bin_get_child(GTK_BIN(widget));
  
  GtkTreeSelection *sel =
    gtk_tree_view_get_selection(GTK_TREE_VIEW(view));

  GtkTreeModel *model;
  GtkTreeIter iter;

  /* there should never be an unseletced column. But if */
  /* it is, we count it as zero */
  if(!gtk_tree_selection_get_selected(sel, &model, &iter))
    return 0;

  int retval = 0;
  gtk_tree_model_get(model, &iter, 1, &retval, -1);
  return retval;
}

static GtkWidget *digit_picker_create(int min, int max, int sel) {
#ifndef FREMANTLE
  GtkWidget *view = gtk_tree_view_new();
#else
  GtkWidget *view = hildon_gtk_tree_view_new(HILDON_UI_MODE_EDIT);
#endif

  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), FALSE); 
  GtkTreeSelection *selection = 
    gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
  
  /* --- "digit" column --- */
  GtkCellRenderer *renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "xalign", 0.5, NULL );
  gtk_tree_view_insert_column_with_attributes(
      GTK_TREE_VIEW(view), -1, "digit", renderer, 
      "text", 0,
      "sensitive", 2, 
      NULL);

  GtkListStore *store = 
    gtk_list_store_new(3, G_TYPE_STRING, G_TYPE_INT, G_TYPE_BOOLEAN); 

  dummy_append(store, DUMMY_NUM);

  /* add digits */
  int i;
  GtkTreeIter siter;
  for(i=min;i<=max;i++) {
    char str[2] = { '0'+i, 0 };
    GtkTreeIter     iter;
    /* Append a row and fill in some data */
    gtk_list_store_append (store, &iter);
    gtk_list_store_set(store, &iter, 0, str, 1, i, 2, TRUE, -1);

    if(i == sel) siter = iter;
  }
    
  dummy_append(store, DUMMY_NUM);

  gtk_tree_view_set_model(GTK_TREE_VIEW(view), GTK_TREE_MODEL(store));
  g_object_unref(store);

#ifndef FREMANTLE
  g_signal_connect(view, "row-activated", 
		   (GCallback)on_picker_activated, NULL);
#else
  g_signal_connect(view, "hildon-row-tapped", 
		   (GCallback)on_picker_activated, NULL);
#endif

  gtk_tree_selection_select_iter(selection, &siter);
  picker_center_iter(view, &siter);

  /* put this inside a scrolled view */
#ifndef USE_PANNABLE_AREA
  GtkWidget *scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
				 GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scrolled_window), view);
  return scrolled_window;
#else
  GtkWidget *pannable_area = hildon_pannable_area_new();
  gtk_container_add(GTK_CONTAINER(pannable_area), view);
  return pannable_area;
#endif
}

static gint on_lat_picker_button_press(GtkWidget *button, 
		   GdkEventButton *event, gpointer data) {

  if(event->type == GDK_BUTTON_PRESS) {
    GtkWidget *dialog = 
      gtk_dialog_new_with_buttons(_("Latitude"),
	  GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(button))), 
				  GTK_DIALOG_MODAL,
          GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, 
	  _("Done"),        GTK_RESPONSE_ACCEPT, 
	  NULL);

    gtk_window_set_default_size(GTK_WINDOW(dialog), 400, 210);

    int i, lat = (int)g_object_get_data(G_OBJECT(button), "latitude");

    /* parse latitude into components */
    int sign = (lat>=0)?0:1;
    lat = abs(lat);
    int deg = lat / 60000;
    int min = (lat /1000)%60;
    int minfrac = lat % 1000;

    GtkWidget *signw, *degw[2], *minw[2], *minfracw[3];
    

    /* create N/S 89° 99.999 */
    GtkWidget *hbox = gtk_hbox_new(FALSE, 0);
    static const char *ns[] = { "N", "S", NULL };
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		signw = string_picker_create(ns, sign));

    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		degw[0] = digit_picker_create(0,8, deg/10));
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		degw[1] = digit_picker_create(0,9, deg%10));
    gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new("°"), FALSE, FALSE, 0);

    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		minw[0] = digit_picker_create(0,5, min/10));
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		minw[1] = digit_picker_create(0,9, min%10));
    gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new("."), FALSE, FALSE, 0);

    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		minfracw[0] = digit_picker_create(0,9, minfrac/100));
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
	        minfracw[1] = digit_picker_create(0,9, (minfrac/10)%10));
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		minfracw[2] = digit_picker_create(0,9, minfrac%10));

    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox);

    gtk_widget_show_all(dialog);
    if(gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {

      /* parse degrees ... */
      for(deg=0,i=0;i<2;i++) 
	deg = 10 * deg + picker_get(degw[i]);

      /* ... minutes ... */
      for(min=0,i=0;i<2;i++) 
	min = 10 * min + picker_get(minw[i]);

      /* ... and fractions of minutes */
      for(minfrac=0,i=0;i<3;i++) 
	minfrac = 10 * minfrac + picker_get(minfracw[i]);

      /* parse N/S */
      if(deg | min | minfrac)
	sign = picker_get(signw)?-1:1;
      else
	sign = 1;   // theres no S 00 00.000

      float latitude = sign * (deg + min/60.0 + minfrac/60000.0); 
      lat_entry_set(button, latitude);
    }

    gtk_widget_destroy(dialog);

    return TRUE;
  }
  return FALSE;
}
#endif

/* a entry that is colored red when being "active" */
GtkWidget *lat_entry_new(float lat) {
  char str[32];
  pos_lat_str(str, sizeof(str), lat);

#ifndef COORDINATE_PICKER
  GtkWidget *widget = red_entry_new_with_text(str);

  g_signal_connect(G_OBJECT(widget), "changed",
		   G_CALLBACK(callback_modified_lat), NULL);
 
#else
  GtkWidget *widget = gtk_button_new_with_label(str);

#ifdef FREMANTLE
  hildon_gtk_widget_set_theme_size(widget, 
	(HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
#endif
  int lat_int = (int)roundf(lat * 60000);
  g_object_set_data(G_OBJECT(widget), "latitude", (gpointer)lat_int); 
  gtk_signal_connect(GTK_OBJECT(widget), "button-press-event",
  		     (GtkSignalFunc)on_lat_picker_button_press, NULL);
#endif
  
  return widget;
}

#ifndef COORDINATE_PICKER
static void callback_modified_lon(GtkWidget *widget, gpointer data ) {
  float i = pos_parse_lon((char*)gtk_entry_get_text(GTK_ENTRY(widget)));
  mark(widget, !isnan(i));
}
#else
static gint on_lon_picker_button_press(GtkWidget *button, 
		   GdkEventButton *event, gpointer data) {

  if(event->type == GDK_BUTTON_PRESS) {
    GtkWidget *dialog = 
      gtk_dialog_new_with_buttons(_("Longitude"),
	  GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(button))), 
				  GTK_DIALOG_MODAL,
          GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, 
	  _("Done"),        GTK_RESPONSE_ACCEPT, 
	  NULL);

    gtk_window_set_default_size(GTK_WINDOW(dialog), 400, 210);

    int i, lat = (int)g_object_get_data(G_OBJECT(button), "longitude");

    /* parse latitude into components */
    int sign = (lat>=0)?0:1;
    lat = abs(lat);
    int deg = lat / 60000;
    int min = (lat /1000)%60;
    int minfrac = lat % 1000;

    GtkWidget *signw, *degw[3], *minw[2], *minfracw[3];

    /* create E/W 179° 99.999 */
    GtkWidget *hbox = gtk_hbox_new(FALSE, 0);
    static const char *ew[] = { "E", "W", NULL };
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		signw = string_picker_create(ew, sign));

    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		degw[0] = digit_picker_create(0,1, deg/100));
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		degw[1] = digit_picker_create(0,9, (deg/10)%10));
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		degw[2] = digit_picker_create(0,9, deg%10));
    gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new("°"), FALSE, FALSE, 0);

    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		minw[0] = digit_picker_create(0,5, min/10));
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		minw[1] = digit_picker_create(0,9, min%10));
    gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new("."), FALSE, FALSE, 0);

    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		minfracw[0] = digit_picker_create(0,9, minfrac/100));
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
	        minfracw[1] = digit_picker_create(0,9, (minfrac/10)%10));
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		minfracw[2] = digit_picker_create(0,9, minfrac%10));

    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox);

    gtk_widget_show_all(dialog);
    if(gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {

      /* parse degrees ... */
      for(deg=0,i=0;i<3;i++) 
	deg = 10 * deg + picker_get(degw[i]);

      /* ... minutes ... */
      for(min=0,i=0;i<2;i++) 
	min = 10 * min + picker_get(minw[i]);

      /* ... and fractions of minutes */
      for(minfrac=0,i=0;i<3;i++) 
	minfrac = 10 * minfrac + picker_get(minfracw[i]);

      if(deg | min | minfrac)
	sign = picker_get(signw)?-1:1;
      else
	sign = 1;   // theres no W 00 00.000

      float longitude = sign * (deg + min/60.0 + minfrac/60000.0); 
      lon_entry_set(button, longitude);
    }

    gtk_widget_destroy(dialog);

    return TRUE;
  }
  return FALSE;
}
#endif

/* a entry that is colored red when filled with invalid coordinate */
GtkWidget *lon_entry_new(float lon) {
  char str[32];
  pos_lon_str(str, sizeof(str), lon);

#ifndef COORDINATE_PICKER
  GtkWidget *widget = red_entry_new_with_text(str);
  g_signal_connect(G_OBJECT(widget), "changed",
		   G_CALLBACK(callback_modified_lon), NULL);
 
#else
  GtkWidget *widget = gtk_button_new_with_label(str);

#ifdef FREMANTLE
  hildon_gtk_widget_set_theme_size(widget, 
	(HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
#endif
  int lon_int = (int)roundf(lon * 60000);
  g_object_set_data(G_OBJECT(widget), "longitude", (gpointer)lon_int); 
  gtk_signal_connect(GTK_OBJECT(widget), "button-press-event",
  		     (GtkSignalFunc)on_lon_picker_button_press, NULL);
#endif

  return widget;
}

float lat_entry_get(GtkWidget *widget) {
#ifndef COORDINATE_PICKER
  char *p = (char*)gtk_entry_get_text(GTK_ENTRY(widget));
#else
  char *p = (char*)gtk_button_get_label(GTK_BUTTON(widget));
#endif
  return pos_parse_lat(p);
}

float lon_entry_get(GtkWidget *widget) {
#ifndef COORDINATE_PICKER
  char *p = (char*)gtk_entry_get_text(GTK_ENTRY(widget));
#else
  char *p = (char*)gtk_button_get_label(GTK_BUTTON(widget));
#endif
  return pos_parse_lon(p);
}

void lat_entry_set(GtkWidget *widget, float lat) {
  char str[32];
  pos_lat_str(str, sizeof(str)-1, lat); 
#ifndef COORDINATE_PICKER
  gtk_entry_set_text(GTK_ENTRY(widget), str);
#else
  gtk_button_set_label(GTK_BUTTON(widget), str);
  int lat_int = (int)roundf(lat * 60000);
  g_object_set_data(G_OBJECT(widget), "latitude", (gpointer)lat_int); 
  g_signal_emit_by_name(widget, "changed");
#endif
}

void lon_entry_set(GtkWidget *widget, float lon) {
  char str[32];
  pos_lon_str(str, sizeof(str)-1, lon); 
#ifndef COORDINATE_PICKER
  gtk_entry_set_text(GTK_ENTRY(widget), str);
#else
  gtk_button_set_label(GTK_BUTTON(widget), str);
  int lon_int = (int)roundf(lon * 60000);
  g_object_set_data(G_OBJECT(widget), "longitude", (gpointer)lon_int); 
  g_signal_emit_by_name(widget, "changed");
#endif
}

void lat_label_set(GtkWidget *widget, float lat) {
  char str[32];
  pos_lat_str(str, sizeof(str)-1, lat); 
  gtk_label_set(GTK_LABEL(widget), str);
}

void lon_label_set(GtkWidget *widget, float lon) {
  char str[32];
  pos_lon_str(str, sizeof(str)-1, lon); 
  gtk_label_set(GTK_LABEL(widget), str);
}

void lat_label_attrib_set(GtkWidget *widget, float lat,
			  int size, int strikethrough) {
  char str[32];
  pos_lat_str(str, sizeof(str)-1, lat); 
  gtk_label_attrib_set(widget, str, size, strikethrough);
}

void lon_label_attrib_set(GtkWidget *widget, float lon,
			  int size, int strikethrough) {
  char str[32];
  pos_lon_str(str, sizeof(str)-1, lon); 
  gtk_label_attrib_set(widget, str, size, strikethrough);
}

#ifndef COORDINATE_PICKER
static void callback_modified_dist(GtkWidget *widget, gpointer data ) {
  /* don't care for metric/imperial since we only want to know if this */
  /* is parseable at all */
  float i = 
    distance_parse((char*)gtk_entry_get_text(GTK_ENTRY(widget)), FALSE);
  mark(widget, !isnan(i));
}
#else
static gint on_dist_picker_button_press(GtkWidget *button, 
		   GdkEventButton *event, gpointer data) {

  if(event->type == GDK_BUTTON_PRESS) {
    GtkWidget *dialog = 
      gtk_dialog_new_with_buttons(_("Distance"),
	  GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(button))), 
				  GTK_DIALOG_MODAL,
          GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, 
	  _("Done"),        GTK_RESPONSE_ACCEPT, 
	  NULL);

    gtk_window_set_default_size(GTK_WINDOW(dialog), 400, 210);

    /* distance is given in m or ft (depending on mil) */
    int i, dist = (int)g_object_get_data(G_OBJECT(button), "distance");
    gboolean mil = (gboolean)g_object_get_data(G_OBJECT(button), "mil");
    int unit = 0;

    /* parse distance into components */
    if(mil) {
      /* 1 mil = 1760 yd = 5280 ft. 1yd = 3 ft */
      if(dist<95)        { unit = 0; dist *= 100;  }
      else if(dist<2904) { unit = 1; dist = 100 * dist / 3;  }
      else               { unit = 2; dist = 5 * dist / 264; }
    } else {
      if(dist<1000)      { unit = 3; dist *= 100;  }
      else               { unit = 4; dist /= 10; }
    }

    GtkWidget *distw[4], *fracw[2], *unitw;

    /* create xxxx.x unit */
    GtkWidget *hbox = gtk_hbox_new(FALSE, 0);

    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		distw[0] = digit_picker_create(0,9, (dist/100000)%10));
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		distw[1] = digit_picker_create(0,9, (dist/10000)%10));
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		distw[2] = digit_picker_create(0,9, (dist/1000)%10));
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		distw[3] = digit_picker_create(0,9, (dist/100)%10));
    gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(" . "), FALSE, FALSE, 0);
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		fracw[0] = digit_picker_create(0,9, (dist/10)%10));
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		fracw[1] = digit_picker_create(0,9, (dist/1)%10));

    static const char *units[] = { "ft", "yd", "mi", "m", "km", NULL };
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		unitw = string_picker_create(units, unit));

    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox);

    gtk_widget_show_all(dialog);
    if(gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {

      /* parse distance */
      for(dist=0,i=0;i<4;i++) 
	dist = 10 * dist + picker_get(distw[i]);

      for(i=0;i<2;i++) 
	dist = 10 * dist + picker_get(fracw[i]);

      unit = picker_get(unitw);
      if(unit == 0)      { dist /= 100; }         // ft
      else if(unit == 1) { dist = 3*dist/100; }   // yd
      else if(unit == 2) { dist = 528*dist/10; }  // mi
      else if(unit == 3) { dist /= 100; }         // m
      else if(unit == 4) { dist *= 10; }          // km

      /* user may have switched between metric and imperial */
      float distance;
      if(unit <= 2) {
	distance = dist / 5280.0;
	if(!mil) distance *=  1.609344;
      } else {
	distance = dist / 1000.0;
	if( mil) distance /=  1.609344;
      }

      dist_entry_set(button, distance, mil);
    }

    gtk_widget_destroy(dialog);

    return TRUE;
  }
  return FALSE;
}
#endif

/* a entry that is colored red when filled with invalid distance */
GtkWidget *dist_entry_new(float dist, gboolean mil) {
  char str[32];
  distance_str(str, sizeof(str), dist, mil);

#ifndef COORDINATE_PICKER
  GtkWidget *widget = red_entry_new_with_text(str);
  g_signal_connect(G_OBJECT(widget), "changed",
		   G_CALLBACK(callback_modified_dist), NULL);
 
#else
  GtkWidget *widget = gtk_button_new_with_label(str);

#ifdef FREMANTLE
  hildon_gtk_widget_set_theme_size(widget, 
	(HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
#endif
  int dist_int = (int)roundf(dist * 1000);        // km -> m 
  if(mil) dist_int = (int)roundf(dist * 5280.0);  // mi -> ft
 
  g_object_set_data(G_OBJECT(widget), "distance", (gpointer)dist_int); 
  g_object_set_data(G_OBJECT(widget), "mil", (gpointer)mil); 
  gtk_signal_connect(GTK_OBJECT(widget), "button-press-event",
   		     (GtkSignalFunc)on_dist_picker_button_press, NULL);
#endif

  return widget;
}

float dist_entry_get(GtkWidget *widget, gboolean mil) {
#ifndef COORDINATE_PICKER
  char *p = (char*)gtk_entry_get_text(GTK_ENTRY(widget));
#else
  char *p = (char*)gtk_button_get_label(GTK_BUTTON(widget));
#endif
  return distance_parse(p, mil);
}

void dist_entry_set(GtkWidget *widget, float dist, gboolean mil) {
  char str[32];
  distance_str(str, sizeof(str), dist, mil);
 
#ifndef COORDINATE_PICKER
  gtk_entry_set_text(GTK_ENTRY(widget), str);
#else
  gtk_button_set_label(GTK_BUTTON(widget), str);
  int dist_int = (int)roundf(dist * 1000);        // km -> m 
  if(mil) dist_int = (int)roundf(dist * 5280.0);  // mi -> ft
  g_object_set_data(G_OBJECT(widget), "distance", (gpointer)dist_int); 
  g_object_set_data(G_OBJECT(widget), "mil", (gpointer)mil); 
  g_signal_emit_by_name(widget, "changed");
#endif
}

#ifndef USE_MAEMO
#ifdef ENABLE_BROWSER_INTERFACE
#include <libgnome/gnome-url.h>

int browser_url(appdata_t *appdata, char *url) {
  /* taken from gnome-open, part of libgnome */
  GError *err = NULL;
  gnome_url_show(url, &err);
  return 0;
}
#endif
#endif

/* recursively remove an entire file system */
void rmdir_recursive(char *path) {
  GDir *dir = g_dir_open(path, 0, NULL);
  if(dir) {
    const char *name = g_dir_read_name(dir);
    while(name) {
      char *fullname = g_strdup_printf("%s/%s", path, name);
      //      printf("deleting %s\n", fullname);
      
      if(g_file_test(fullname, G_FILE_TEST_IS_DIR)) 
	rmdir_recursive(fullname);
      else if(g_file_test(fullname, G_FILE_TEST_IS_REGULAR)) 
	g_remove(fullname);

      g_free(fullname);
      name = g_dir_read_name(dir);
    }

    g_dir_close(dir);
  }
  g_rmdir(path);
}

#ifdef ENABLE_BROWSER_INTERFACE
static void on_link_clicked(GtkButton *button, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  char *url = g_object_get_data(G_OBJECT(button), "url"); 
  if(url) browser_url(appdata, url);
}
#endif

/* a button containing a weblink */
GtkWidget *link_button_attrib(appdata_t *appdata, char *str, char *url, 
		       int size, int strikethrough) {

#ifdef ENABLE_BROWSER_INTERFACE
  if(url) {
    GtkWidget *button = gtk_button_attrib(str, size, strikethrough);
    g_object_set_data(G_OBJECT(button), "url", url); 
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       (GtkSignalFunc)on_link_clicked, appdata);

    return button;
  }
#endif
  return gtk_label_attrib(str, size, strikethrough);
}

#ifdef ENABLE_BROWSER_INTERFACE
static void on_link_id_clicked(GtkButton *button, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  unsigned int id = (unsigned int)g_object_get_data(G_OBJECT(button), "id"); 
  char *type = g_object_get_data(G_OBJECT(button), "type"); 

  char *url = g_strdup_printf("http://www.geocaching.com/%s?id=%u", 
			      type, id);

  if(url) {
    browser_url(appdata, url);
    g_free(url);
  }
}
#endif

GtkWidget *link_button_by_id(appdata_t *appdata, char *str, 
			     const char *type, int id) {

#ifdef ENABLE_BROWSER_INTERFACE
  if(id) {
    GtkWidget *ref = gtk_button_new_with_label(str);
#if defined(USE_MAEMO) && (MAEMO_VERSION_MAJOR == 5)
    //    hildon_gtk_widget_set_theme_size(ref, 
    //	       (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
#endif  
    g_object_set_data(G_OBJECT(ref), "id", (gpointer)id);
    g_object_set_data(G_OBJECT(ref), "type", (gpointer)type);
    gtk_signal_connect(GTK_OBJECT(ref), "clicked",
		       GTK_SIGNAL_FUNC(on_link_id_clicked), appdata);
    
    return ref;
  }
#endif
  return gtk_label_new(str);
}


GtkWidget *link_icon_button_by_id(appdata_t *appdata, GtkWidget *icon, 
			     const char *type, int id) {

#ifdef ENABLE_BROWSER_INTERFACE
  if(id) {
    GtkWidget *ref = gtk_button_new();
    gtk_button_set_image(GTK_BUTTON(ref), icon);

#if defined(USE_MAEMO) && (MAEMO_VERSION_MAJOR == 5)
    //    hildon_gtk_widget_set_theme_size(ref, 
    //	       (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
#endif  
    g_object_set_data(G_OBJECT(ref), "id", (gpointer)id);
    g_object_set_data(G_OBJECT(ref), "type", (gpointer)type);
    gtk_signal_connect(GTK_OBJECT(ref), "clicked",
		       GTK_SIGNAL_FUNC(on_link_id_clicked), appdata);
    
    return ref;
  }
#endif
  return icon;
}

/* left aligned, word wrapped multiline widget */
GtkWidget *simple_text_widget(char *text) {
  GtkWidget *label = gtk_label_new(text);
  
  gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
  gtk_label_set_line_wrap_mode(GTK_LABEL(label), PANGO_WRAP_WORD);
  gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);

  return label;
}


/* a label that is left aligned */
GtkWidget *left_label_new(char *str) {
  GtkWidget *widget = gtk_label_new(str);
  gtk_misc_set_alignment(GTK_MISC(widget), 0.0f, 0.5f);
  return widget;
}

/* ------------- preset coordinate picker tool ----------------- */

static void pos_set(GtkWidget *item, float lat, float lon) {

  GtkWidget *lat_entry = g_object_get_data(G_OBJECT(item), "lat_entry");
  lat_entry_set(lat_entry, lat);

  GtkWidget *lon_entry = g_object_get_data(G_OBJECT(item), "lon_entry");
  lon_entry_set(lon_entry, lon);
}

static void cb_gps(GtkWidget *item, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  gint id = (gint)g_object_get_data(G_OBJECT(item), "id");
  pos_t *pos = NULL;

  if(!id) 
    pos = gps_get_pos(appdata);
  else if(id == 1)
    pos = &appdata->home;
  else {
    location_t *location = appdata->location;
    while(location && id > 2) {
      location = location->next;
      id--;
    }
      
    if(id == 2)
      pos = &location->pos;
  }

  if(!pos) pos_set(item, NAN, NAN);
  else     pos_set(item, pos->lat, pos->lon);
}

static void cb_geomath(GtkWidget *item, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  pos_set(item, appdata->geomath.lat, appdata->geomath.lon);
}

#ifdef ENABLE_OSM_GPS_MAP
static void cb_map(GtkWidget *item, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  pos_set(item, appdata->map.pos.lat, appdata->map.pos.lon);
}
#endif

#ifdef ENABLE_MAEMO_MAPPER
static void cb_mm(GtkWidget *item, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  pos_set(item, appdata->mmpos.lat, appdata->mmpos.lon);
}
#endif

static void cb_cache(GtkWidget *item, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  
  cache_t *cache = appdata->cur_cache;
  g_assert(cache);

  gint id = (gint)g_object_get_data(G_OBJECT(item), "id");

  if(!id)
    pos_set(item, cache->pos.lat, cache->pos.lon);
  else if(id == 1) {
    /* fetch position out of notes dialog since they probably */
    /* haven't been saved yet */
    pos_t pos = notes_get_pos(appdata->cache_context);
    pos_set(item, pos.lat, pos.lon);
  } else {
    wpt_t *wpt = cache->wpt;
    while(wpt && id > 2) {
      wpt = wpt->next;
      id--;
    }
      
    if(id == 2)
      pos_set(item, wpt->pos.lat, wpt->pos.lon);
  }
}

#ifndef PRESET_PICKER_DIALOG
static GtkWidget *menu_add(GtkWidget *menu, appdata_t *appdata, 
			   GtkWidget *icon, char *menu_str, 
			   void(*func)(GtkWidget*, gpointer), gint id,
			   GtkWidget *lon_entry, GtkWidget *lat_entry) {

  GtkWidget *item = gtk_image_menu_item_new_with_label(menu_str);

  if(icon)
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(item), icon);

  g_object_set_data(G_OBJECT(item), "lat_entry", (gpointer)lat_entry);
  g_object_set_data(G_OBJECT(item), "lon_entry", (gpointer)lon_entry);
  g_object_set_data(G_OBJECT(item), "id", (gpointer)id);

  if(func)
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       (GtkSignalFunc)func, appdata);

  gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);      

  return item;
}

static GtkWidget *popup_menu_create(appdata_t *appdata, 
		    GtkWidget *lat_entry, GtkWidget *lon_entry) {
  GtkWidget *menu = gtk_menu_new();

  if(pos_valid(gps_get_pos(appdata)))
    menu_add(menu, appdata, icon_get_widget(ICON_POS, 18),
	     _("GPS position"), cb_gps, 0, lon_entry, lat_entry);

  if(pos_valid(&appdata->home))
    menu_add(menu, appdata, icon_get_widget(ICON_POS, 21),
	     _("Home"), cb_gps, 1, lon_entry, lat_entry);

  location_t *location = appdata->location;
  gint id = 2;
  while(location) {
    if(pos_valid(&location->pos))
      menu_add(menu, appdata, icon_get_widget(ICON_POS, 17),
	       location->name, cb_gps, id, lon_entry, lat_entry);

    id++;
    location = location->next;
  }

  if(pos_valid(&appdata->geomath))
    menu_add(menu, appdata, icon_get_widget(ICON_POS, 19),
	     _("Geomath projection"), cb_geomath, 0, lon_entry, lat_entry);

#ifdef ENABLE_OSM_GPS_MAP
  if(pos_valid(&appdata->map.pos))
    menu_add(menu, appdata, icon_get_widget(ICON_POS, 20), _("Map position"), 
	     cb_map, 0, lon_entry, lat_entry);
#endif

#ifdef ENABLE_MAEMO_MAPPER
  if(appdata->mmpos_valid) {
    menu_add(menu, appdata, icon_get_widget(ICON_POS, 24), 
	     _("Maemo Mapper position"), cb_mm, 0, lon_entry, lat_entry);
  }
#endif

  if(appdata->cur_cache) {
    cache_t *cache = appdata->cur_cache;

    char *name = cache->name;
    if(!name) name = cache->id;

    /* original cache position */
    if(pos_valid(&cache->pos))
      menu_add(menu, appdata, icon_get_widget(ICON_POS, cache->type + 6),
	       name, cb_cache, 0, lon_entry, lat_entry);

    /* overwritten cache position */
    if(appdata->cache_context && notes_get_override(appdata->cache_context))
      menu_add(menu, appdata, icon_get_widget(ICON_POS, cache->type + 6),
	       _("Modified coordinate"), cb_cache, 1, lon_entry, lat_entry);

    wpt_t *wpt = cache->wpt;
    gint id = 2;
    while(wpt) {
      if(pos_valid(&wpt->pos)) {
	GtkWidget *icon = NULL;
	if(wpt->sym != WPT_SYM_UNKNOWN)
	  icon = icon_get_widget(ICON_POS, wpt->sym);

	char *name = wpt->desc;
	if(!name) name = wpt->cmt;
	if(!name) name = wpt->id;
	
	menu_add(menu, appdata, icon, name, cb_cache, id, 
		 lon_entry, lat_entry);
      }

      id++;
      wpt = wpt->next;
    }
  }

  gtk_widget_show_all(menu);

  return menu;
}

static gint on_popup_button_press(GtkWidget *button, GdkEventButton *event, 
				  gpointer data) {

  appdata_t *appdata = (appdata_t*)data;

  if(event->type == GDK_BUTTON_PRESS) {
    GtkWidget *menu = g_object_get_data(G_OBJECT(button), "menu");

    if(menu)
      gtk_widget_destroy(menu);

    gpointer lat_entry = g_object_get_data(G_OBJECT(button), "lat_entry");
    g_assert(lat_entry);
    gpointer lon_entry = g_object_get_data(G_OBJECT(button), "lon_entry");
    g_assert(lon_entry);

    menu = popup_menu_create(appdata, lat_entry, lon_entry);
    g_object_set_data(G_OBJECT(button), "menu", (gpointer)menu);

    /* draw a popup menu */
    gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
		   event->button, event->time);
    return TRUE;
  }
  return FALSE;
}

static void on_popup_destroy(GtkWidget *widget, gpointer user_data ) {
  GtkWidget *menu = g_object_get_data(G_OBJECT(widget), "menu");
  if(menu) gtk_widget_destroy(menu);
}
#endif

#ifdef PRESET_PICKER_DIALOG

enum {
  PRESET_PICKER_COL_ICON = 0,
  PRESET_PICKER_COL_NAME,
  PRESET_PICKER_COL_ID,
  PRESET_PICKER_COL_CB,
  PRESET_PICKER_NUM_COLS
};

static void preset_picker_add(GtkListStore *store,  appdata_t *appdata, 
		       GdkPixbuf *icon, char *menu_str, 
		       void(*func)(GtkWidget*, gpointer), gint id) {
  GtkTreeIter     iter;

  /* Append a row and fill in some data */
  gtk_list_store_append (store, &iter);

  gtk_list_store_set(store, &iter,
		     PRESET_PICKER_COL_ICON, icon,
		     PRESET_PICKER_COL_NAME, menu_str,
		     PRESET_PICKER_COL_ID, id,
		     PRESET_PICKER_COL_CB, func,
		     -1);
}

static void on_preset_picker_activated(GtkTreeView        *treeview,
				GtkTreePath        *path,
				GtkTreeViewColumn  *col,
				gpointer            userdata) {
  GtkTreeIter   iter;
  GtkTreeModel *model = gtk_tree_view_get_model(treeview);

  if(gtk_tree_model_get_iter(model, &iter, path)) {
    gint id;
    void(*func)(GtkWidget*, gpointer);
    gtk_tree_model_get(model, &iter, 
		       PRESET_PICKER_COL_ID, &id, 
		       PRESET_PICKER_COL_CB, &func, 
		       -1);

    /* set id on widget as callbacks expect it this way */
    g_object_set_data(G_OBJECT(treeview), "id", (gpointer)id);
    func(GTK_WIDGET(treeview), userdata);

    gtk_dialog_response(GTK_DIALOG(gtk_widget_get_toplevel(
                    GTK_WIDGET(treeview))), GTK_RESPONSE_ACCEPT);

  }
}

static GtkWidget *preset_picker_create(appdata_t *appdata, 
				GtkWidget *lat_entry, GtkWidget *lon_entry) {
  GtkCellRenderer *renderer;
  GtkListStore    *store;

  GtkWidget *view = gtk_tree_view_new();
  
  g_object_set_data(G_OBJECT(view), "lat_entry", (gpointer)lat_entry); 
  g_object_set_data(G_OBJECT(view), "lon_entry", (gpointer)lon_entry); 

  /* --- "Icon" column --- */
  renderer = gtk_cell_renderer_pixbuf_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(view),
      -1, "Icon", renderer, "pixbuf", PRESET_PICKER_COL_ICON, NULL);

  /* --- "Name" column --- */
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "ellipsize", PANGO_ELLIPSIZE_END, NULL );
  GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes(
		 "Name", renderer, "text", PRESET_PICKER_COL_NAME, NULL);
  gtk_tree_view_column_set_expand(column, TRUE);
  gtk_tree_view_insert_column(GTK_TREE_VIEW(view), column, -1);

  store = gtk_list_store_new(PRESET_PICKER_NUM_COLS, 
			     GDK_TYPE_PIXBUF,
			     G_TYPE_STRING,
			     G_TYPE_INT,
			     G_TYPE_POINTER);

  if(pos_valid(gps_get_pos(appdata)))
    preset_picker_add(store, appdata, icon_get(ICON_POS, 18),
		      _("GPS position"), cb_gps, 0);

  if(pos_valid(&appdata->home))
    preset_picker_add(store, appdata, icon_get(ICON_POS, 21),
		      _("Home"), cb_gps, 1);

  location_t *location = appdata->location;
  gint id = 2;
  while(location) {
    if(pos_valid(&location->pos))
      preset_picker_add(store, appdata, icon_get(ICON_POS, 17),
			location->name, cb_gps, id);

    id++;
    location = location->next;
  }

  if(pos_valid(&appdata->geomath))
    preset_picker_add(store, appdata, icon_get(ICON_POS, 19),
		      _("Geomath projection"), cb_geomath, 0);

#ifdef ENABLE_OSM_GPS_MAP
  if(pos_valid(&appdata->map.pos))
    preset_picker_add(store, appdata, icon_get(ICON_POS, 20),
		      _("Map position"), cb_map, 0);
#endif

#ifdef ENABLE_MAEMO_MAPPER
  if(appdata->mmpos_valid) {
    preset_picker_add(store, appdata, icon_get(ICON_POS, 24), 
		      _("Maemo Mapper position"), cb_mm, 0);
  }
#endif

  if(appdata->cur_cache) {
    cache_t *cache = appdata->cur_cache;

    char *name = cache->name;
    if(!name) name = cache->id;
    
    /* original cache position */
    if(pos_valid(&cache->pos))
      preset_picker_add(store, appdata, icon_get(ICON_POS, cache->type + 6),
			name, cb_cache, 0);

    /* overwritten cache position */
    if(appdata->cache_context && notes_get_override(appdata->cache_context))
      preset_picker_add(store, appdata, icon_get(ICON_POS, cache->type + 6),
			_("Modified coordinate"), cb_cache, 1);

    wpt_t *wpt = cache->wpt;
    gint id = 2;
    while(wpt) {
      if(pos_valid(&wpt->pos)) {
	GdkPixbuf *icon = NULL;
	if(wpt->sym != WPT_SYM_UNKNOWN)
	  icon = icon_get(ICON_POS, wpt->sym);

	char *name = wpt->desc;
	if(!name) name = wpt->cmt;
	if(!name) name = wpt->id;

	preset_picker_add(store, appdata, icon, name, cb_cache, id);
      }
      id++;
      wpt = wpt->next;
    }
  }

  gtk_tree_view_set_model(GTK_TREE_VIEW(view), GTK_TREE_MODEL(store));
  g_object_unref(store);

  /* make list react on clicks */
  g_signal_connect(view, "row-activated", 
		   (GCallback)on_preset_picker_activated, appdata);

  /* put this inside a scrolled view */
#ifndef USE_PANNABLE_AREA
  GtkWidget *scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
				 GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scrolled_window), view);
  return scrolled_window;
#else
  GtkWidget *pannable_area = hildon_pannable_area_new();
  gtk_container_add(GTK_CONTAINER(pannable_area), view);
  return pannable_area;
#endif
}

static gint on_preset_picker_button_press(GtkWidget *button, 
		   GdkEventButton *event, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  
  gpointer lat_entry = g_object_get_data(G_OBJECT(button), "lat_entry");
  gpointer lon_entry = g_object_get_data(G_OBJECT(button), "lon_entry");

  if(event->type == GDK_BUTTON_PRESS) {
    GtkWidget *dialog = 
      gtk_dialog_new_with_buttons(_("Use coordinate"),
	  GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(button))), 
				  GTK_DIALOG_MODAL,
          GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, 
	  NULL);

    gtk_window_set_default_size(GTK_WINDOW(dialog), 400, 210);

    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox),
			preset_picker_create(appdata, lat_entry, lon_entry));

    gtk_widget_show_all(dialog);
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);

    return TRUE;
  }
  return FALSE;
}
#endif


GtkWidget *preset_coordinate_picker(appdata_t *appdata, 
		     GtkWidget *lat_entry, GtkWidget *lon_entry) {

  GtkWidget *button = gtk_button_new();

  gtk_button_set_image(GTK_BUTTON(button), icon_get_widget(ICON_POS, 22));

#ifndef USE_MAEMO
  gtk_widget_set_tooltip_text(button, _("Use coordinate"));
#endif

  g_object_set_data(G_OBJECT(button), "lat_entry", (gpointer)lat_entry); 
  g_object_set_data(G_OBJECT(button), "lon_entry", (gpointer)lon_entry); 

#ifndef PRESET_PICKER_DIALOG
  gtk_signal_connect(GTK_OBJECT(button), "button-press-event",
		     (GtkSignalFunc)on_popup_button_press, appdata);

  gtk_signal_connect(GTK_OBJECT(button), "destroy", 
		     (GtkSignalFunc)on_popup_destroy, appdata);
#else
#ifdef FREMANTLE
  hildon_gtk_widget_set_theme_size(button, 
	(HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
#endif

  gtk_signal_connect(GTK_OBJECT(button), "button-press-event",
		     (GtkSignalFunc)on_preset_picker_button_press, appdata);
#endif
  
  return button;
}

#if defined(ENABLE_MAEMO_MAPPER) || defined(ENABLE_OSM_GPS_MAP)
static pos_t goto_pos_get(GtkWidget *item) {
  pos_t pos;

  GtkWidget *lat_entry = g_object_get_data(G_OBJECT(item), "lat_entry");
  pos.lat = lat_entry_get(lat_entry);

  GtkWidget *lon_entry = g_object_get_data(G_OBJECT(item), "lon_entry");
  pos.lon = lon_entry_get(lon_entry);

  return  pos;
}

#if defined(ENABLE_MAEMO_MAPPER) && defined(ENABLE_OSM_GPS_MAP)
#ifdef ENABLE_MAEMO_MAPPER
static void cb_mm_set(GtkWidget *item, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  pos_t pos = goto_pos_get(item);
  if(!isnan(pos.lat) && !isnan(pos.lon))
    dbus_mm_set_position(appdata, &pos);
}
#endif

#ifdef ENABLE_OSM_GPS_MAP
static void cb_map_set(GtkWidget *item, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  pos_t pos = goto_pos_get(item);
  if(!isnan(pos.lat) && !isnan(pos.lon)) {
    map(appdata);
    osm_gps_map_set_center(OSM_GPS_MAP(appdata->map.context->widget), 
			     pos.lat, pos.lon);
  }
}
#endif

#ifndef PRESET_PICKER_DIALOG
static GtkWidget *goto_popup_menu_create(appdata_t *appdata, 
		    GtkWidget *lat_entry, GtkWidget *lon_entry) {
  GtkWidget *menu = gtk_menu_new();

#if defined(ENABLE_OSM_GPS_MAP)
  menu_add(menu, appdata, icon_get_widget(ICON_POS, 20),
	   _("Internal map"), cb_map_set, 0, lon_entry, lat_entry);
#endif

#if defined(ENABLE_MAEMO_MAPPER)
  menu_add(menu, appdata, icon_get_widget(ICON_POS, 24),
	   _("Maemo Mapper"), cb_mm_set, 0, lon_entry, lat_entry);
#endif

  gtk_widget_show_all(menu);

  return menu;
}
#else
static GtkWidget *goto_picker_create(appdata_t *appdata, 
		     GtkWidget *lat_entry, GtkWidget *lon_entry) {
  GtkCellRenderer *renderer;
  GtkListStore    *store;

  GtkWidget *view = gtk_tree_view_new();
  
  g_object_set_data(G_OBJECT(view), "lat_entry", (gpointer)lat_entry); 
  g_object_set_data(G_OBJECT(view), "lon_entry", (gpointer)lon_entry); 

  /* --- "Icon" column --- */
  renderer = gtk_cell_renderer_pixbuf_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(view),
      -1, "Icon", renderer, "pixbuf", PRESET_PICKER_COL_ICON, NULL);

  /* --- "Name" column --- */
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "ellipsize", PANGO_ELLIPSIZE_END, NULL );
  GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes(
		 "Name", renderer, "text", PRESET_PICKER_COL_NAME, NULL);
  gtk_tree_view_column_set_expand(column, TRUE);
  gtk_tree_view_insert_column(GTK_TREE_VIEW(view), column, -1);

  store = gtk_list_store_new(PRESET_PICKER_NUM_COLS, 
			     GDK_TYPE_PIXBUF,
			     G_TYPE_STRING,
			     G_TYPE_INT,
			     G_TYPE_POINTER);

#if defined(ENABLE_OSM_GPS_MAP)
  preset_picker_add(store, appdata, icon_get(ICON_POS, 20),
		    _("Internal map"), cb_map_set, 0);
#endif

#if defined(ENABLE_MAEMO_MAPPER)
  preset_picker_add(store, appdata, icon_get(ICON_POS, 24),
		    _("Maemo Mapper"), cb_mm_set, 0);
#endif

  gtk_tree_view_set_model(GTK_TREE_VIEW(view), GTK_TREE_MODEL(store));
  g_object_unref(store);

  /* make list react on clicks */
  g_signal_connect(view, "row-activated", 
		   (GCallback)on_preset_picker_activated, appdata);

  /* put this inside a scrolled view */
#ifndef USE_PANNABLE_AREA
  GtkWidget *scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
				 GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scrolled_window), view);
  return scrolled_window;
#else
  GtkWidget *pannable_area = hildon_pannable_area_new();
  gtk_container_add(GTK_CONTAINER(pannable_area), view);
  return pannable_area;
#endif
}
#endif
#endif

static gint on_goto_button_press(GtkWidget *button, 
				 GdkEventButton *event, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  
  if(event->type == GDK_BUTTON_PRESS) {

#if defined(ENABLE_MAEMO_MAPPER) && !defined(ENABLE_OSM_GPS_MAP)
    /* only maemo mapper is being used */
    pos_t pos = goto_pos_get(button);
    if(!isnan(pos.lat) && !isnan(pos.lon)) 
      dbus_mm_set_position(appdata, &pos);
#elif !defined(ENABLE_MAEMO_MAPPER) && defined(ENABLE_OSM_GPS_MAP)
    /* only internal map is being used */
    pos_t pos = goto_pos_get(button);
    if(!isnan(pos.lat) && !isnan(pos.lon)) {
      map(appdata);
      osm_gps_map_set_center(OSM_GPS_MAP(appdata->map.context->widget), 
			     pos.lat, pos.lon);
    }
#else

    gpointer lat_entry = g_object_get_data(G_OBJECT(button), "lat_entry");
    g_assert(lat_entry);
    gpointer lon_entry = g_object_get_data(G_OBJECT(button), "lon_entry");
    g_assert(lon_entry);

#ifndef PRESET_PICKER_DIALOG
    GtkWidget *menu = g_object_get_data(G_OBJECT(button), "menu");
    if(!menu) { 
      menu = goto_popup_menu_create(appdata, lat_entry, lon_entry);
      g_object_set_data(G_OBJECT(button), "menu", (gpointer)menu);
    }

    /* draw a popup menu */
    gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
		   event->button, event->time);
#else
    GtkWidget *dialog = 
      gtk_dialog_new_with_buttons(_("Show coordinate on map"),
	  GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(button))), 
				  GTK_DIALOG_MODAL,
          GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, 
	  NULL);

    gtk_window_set_default_size(GTK_WINDOW(dialog), 400, 210);

    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox),
			goto_picker_create(appdata, lat_entry, lon_entry));

    gtk_widget_show_all(dialog);
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
#endif

#endif
    
    return TRUE;
  }
  return FALSE;
}

/* a button which makes the map/maemo mapper go to the associated */
/* position */
GtkWidget *goto_coordinate(appdata_t *appdata, 
		     GtkWidget *lat_entry, GtkWidget *lon_entry) {

  GtkWidget *button = gtk_button_new();

  gtk_button_set_image(GTK_BUTTON(button), icon_get_widget(ICON_POS, 23));

#ifndef USE_MAEMO
  gtk_widget_set_tooltip_text(button, _("Show coordinate on map"));
#endif

  g_object_set_data(G_OBJECT(button), "lat_entry", (gpointer)lat_entry); 
  g_object_set_data(G_OBJECT(button), "lon_entry", (gpointer)lon_entry); 

  gtk_signal_connect(GTK_OBJECT(button), "button-press-event",
		     (GtkSignalFunc)on_goto_button_press, appdata);

#ifndef PRESET_PICKER_DIALOG
  gtk_signal_connect(GTK_OBJECT(button), "destroy", 
		     (GtkSignalFunc)on_popup_destroy, appdata);
#else
#ifdef FREMANTLE
  hildon_gtk_widget_set_theme_size(button, 
	(HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
#endif
#endif
  
  return button;
}
#else
/* no map installed */
GtkWidget *goto_coordinate(appdata_t *appdata, 
		     GtkWidget *lat_entry, GtkWidget *lon_entry) {
  return gtk_label_new("");
}
#endif

GtkWidget *entry_new(void) {
#if !defined(USE_MAEMO) || (MAEMO_VERSION_MAJOR < 5)
  return gtk_entry_new();
#else
  return hildon_entry_new(HILDON_SIZE_AUTO);
#endif
}

gboolean pos_differ(pos_t *pos1, pos_t *pos2) {
  int lat1 = (60000 * pos1->lat)+0.5, lon1 = (60000 * pos1->lon)+0.5;
  int lat2 = (60000 * pos2->lat)+0.5, lon2 = (60000 * pos2->lon)+0.5;

  return((lat1 != lat2) || (lon1 != lon2));
}

gboolean pos_valid(pos_t *pos) {
  if(!pos) return FALSE;
  return(!isnan(pos->lat) && !isnan(pos->lon));
}

void misc_init(void) {
  g_signal_new ("changed", GTK_TYPE_BUTTON,
		G_SIGNAL_RUN_FIRST, 0, NULL, NULL,
		g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
}

void angle_str(char *str, int len, float angle) {
  snprintf(str, len, _("%.1f°"), angle);
}

float angle_parse(char *str) {
  float val;

  if(sscanf(str, _("%f°"), &val) != 1)
    val = NAN;
  
  return val;
}

#ifndef COORDINATE_PICKER
static void callback_modified_angle(GtkWidget *widget, gpointer data ) {
  float i = angle_parse((char*)gtk_entry_get_text(GTK_ENTRY(widget)));
  mark(widget, !isnan(i));
}
#else
static gint on_angle_picker_button_press(GtkWidget *button, 
		   GdkEventButton *event, gpointer data) {

  if(event->type == GDK_BUTTON_PRESS) {
    GtkWidget *dialog = 
      gtk_dialog_new_with_buttons(_("Direction"),
	  GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(button))), 
				  GTK_DIALOG_MODAL,
          GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, 
	  _("Done"),        GTK_RESPONSE_ACCEPT, 
	  NULL);

    gtk_window_set_default_size(GTK_WINDOW(dialog), 400, 210);

    int i, angle = (int)g_object_get_data(G_OBJECT(button), "angle");

    GtkWidget *anglew[3], *fracw;

    /* create xxx.x° */
    GtkWidget *hbox = gtk_hbox_new(FALSE, 0);

    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		anglew[0] = digit_picker_create(0,3, (angle/1000)%10));
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		anglew[1] = digit_picker_create(0,9, (angle/100)%10));
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		anglew[2] = digit_picker_create(0,9, (angle/10)%10));
    gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(" . "), FALSE, FALSE, 0);
    gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		fracw = digit_picker_create(0,9, (angle/1)%10));
    gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new("°"), FALSE, FALSE, 0);

    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox);

    gtk_widget_show_all(dialog);
    if(gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {

      /* parse angle */
      for(angle=0,i=0;i<3;i++) 
	angle = 10 * angle + picker_get(anglew[i]);

      angle = 10 * angle + picker_get(fracw);

      angle_entry_set(button, angle/10.0);
    }

    gtk_widget_destroy(dialog);

    return TRUE;
  }
  return FALSE;
}
#endif

GtkWidget *angle_entry_new(float angle) {
  char str[32];
  angle_str(str, sizeof(str), angle);

#ifndef COORDINATE_PICKER
  GtkWidget *widget = red_entry_new_with_text(str);
  g_signal_connect(G_OBJECT(widget), "changed",
		   G_CALLBACK(callback_modified_angle), NULL);
#else
  GtkWidget *widget = gtk_button_new_with_label(str);

#ifdef FREMANTLE
  hildon_gtk_widget_set_theme_size(widget, 
	(HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
#endif
  int angle_int = (int)roundf(angle*10.0);
  g_object_set_data(G_OBJECT(widget), "angle", (gpointer)angle_int); 
  gtk_signal_connect(GTK_OBJECT(widget), "button-press-event",
  		     (GtkSignalFunc)on_angle_picker_button_press, NULL);
#endif
  return widget;
}
 
float angle_entry_get(GtkWidget *widget) {
#ifndef COORDINATE_PICKER
  char *p = (char*)gtk_entry_get_text(GTK_ENTRY(widget));
#else
  char *p = (char*)gtk_button_get_label(GTK_BUTTON(widget));
#endif
  return angle_parse(p);
}

void angle_entry_set(GtkWidget *widget, float angle) {
  char str[32];
  angle_str(str, sizeof(str)-1, angle); 
#ifndef COORDINATE_PICKER
  gtk_entry_set_text(GTK_ENTRY(widget), str);
#else
  gtk_button_set_label(GTK_BUTTON(widget), str);
  int angle_int = (int)roundf(angle * 10.0);
  g_object_set_data(G_OBJECT(widget), "angle", (gpointer)angle_int); 
  g_signal_emit_by_name(widget, "changed");
#endif
}
 
GtkWidget *toggle_button_new_with_label(char *label) {
#if !defined(USE_MAEMO) || (MAEMO_VERSION_MAJOR < 5)
  return gtk_check_button_new_with_label(label);
#else
  GtkWidget *cbut = gtk_toggle_button_new_with_label(label);
  hildon_gtk_widget_set_theme_size(cbut, 
	   (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
  gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(cbut), FALSE);
  return cbut;
#endif
}

void toggle_button_set_active(GtkWidget *button, gboolean active) {
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), active);
}

gboolean toggle_button_get_active(GtkWidget *button) {
  return gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));
}

GtkWidget *check_button_new_with_label(char *label) {
#if !defined(USE_MAEMO) || (MAEMO_VERSION_MAJOR < 5)
  return gtk_check_button_new_with_label(label);
#else
  GtkWidget *cbut = 
    hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT | 
			    HILDON_SIZE_AUTO_WIDTH);
  gtk_button_set_label(GTK_BUTTON(cbut), label);
  return cbut;
#endif
}

void check_button_set_active(GtkWidget *button, gboolean active) {
#if !defined(USE_MAEMO) || (MAEMO_VERSION_MAJOR < 5)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), active);
#else
  hildon_check_button_set_active(HILDON_CHECK_BUTTON(button), active);
#endif
}

gboolean check_button_get_active(GtkWidget *button) {
#if !defined(USE_MAEMO) || (MAEMO_VERSION_MAJOR < 5)
  return gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));
#else
  return hildon_check_button_get_active(HILDON_CHECK_BUTTON(button));
#endif
}

GtkWidget *button_new(void) {
  GtkWidget *button = gtk_button_new();
#if defined(USE_MAEMO) && (MAEMO_VERSION_MAJOR == 5)
  hildon_gtk_widget_set_theme_size(button, 
	   (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
#endif  
  return button;
}

GtkWidget *button_new_with_label(char *label) {
  GtkWidget *button = gtk_button_new_with_label(label);
#if defined(USE_MAEMO) && (MAEMO_VERSION_MAJOR == 5)
  hildon_gtk_widget_set_theme_size(button, 
	   (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
#endif  
  return button;
}

static void on_browse(GtkWidget *widget, gpointer data) {
  GtkWidget *dialog;

  char **fileptr = g_object_get_data(G_OBJECT(widget), "fileptr");

#ifdef USE_MAEMO
  dialog = hildon_file_chooser_dialog_new(
	       GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(widget))), 
	       GTK_FILE_CHOOSER_ACTION_SAVE);
#else
  dialog = gtk_file_chooser_dialog_new(
	       g_object_get_data(G_OBJECT(widget), "sel_title"),
	       GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(widget))), 
	       GTK_FILE_CHOOSER_ACTION_SAVE,
               GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	       GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
	       NULL);
#endif

  if(*fileptr) {
    printf("set filename <%s>\n", *fileptr);

    if(!g_file_test(*fileptr, G_FILE_TEST_EXISTS)) {
      char *last_sep = strrchr(*fileptr, '/');
      if(last_sep) {
	*last_sep = 0;  // seperate path from file 
	
	/* the user just created a new document */
	gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), 
					    *fileptr);
	gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog), 
					  last_sep+1);
	
	/* restore full filename */
	*last_sep = '/';
      }
    } else 
      gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(dialog), *fileptr);
  }

  if (gtk_dialog_run (GTK_DIALOG(dialog)) == GTK_FM_OK) {
    gchar *name = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
    if(name) {
      if(*fileptr) g_free(*fileptr);
      *fileptr = g_strdup(name);

#ifndef FREMANTLE
      GtkWidget *label = g_object_get_data(G_OBJECT(widget), "label");
      gtk_label_set_text(GTK_LABEL(label), *fileptr);
#else
      hildon_button_set_value(HILDON_BUTTON(widget), *fileptr);
#endif
    }
  }
  
  gtk_widget_destroy (dialog);
}

GtkWidget *export_file(char *sel_title, char **filename) {
  /* non-fremantle and classic gtk use a rather ugly and */
  /* complex layout */
#ifndef FREMANTLE
  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);
  GtkWidget *hbox = gtk_hbox_new(FALSE, 0);
  GtkWidget *label = gtk_label_new(_("Export to"));
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE,0);
  gtk_misc_set_alignment(GTK_MISC(label), 0.f, 0.5f);
  GtkWidget *button = button_new_with_label(_("Browse"));
  g_object_set_data(G_OBJECT(button), "sel_title", sel_title);
  g_object_set_data(G_OBJECT(button), "fileptr", filename);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
  		     GTK_SIGNAL_FUNC(on_browse), NULL);
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE,0);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), hbox);

  label = gtk_label_new((*filename)?*filename:"");
  g_object_set_data(G_OBJECT(button), "label", label);
  gtk_misc_set_alignment(GTK_MISC(label), 0.f, 0.5f);
  gtk_label_set_ellipsize(GTK_LABEL(label), PANGO_ELLIPSIZE_MIDDLE);

  gtk_box_pack_start_defaults(GTK_BOX(vbox), label);
  return vbox;
#else
  GtkWidget *button = hildon_button_new_with_text(
	  (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH),
	  HILDON_BUTTON_ARRANGEMENT_VERTICAL,
	  _("Export to"), *filename);
  hildon_button_set_title_alignment(HILDON_BUTTON(button), 0.5, 0.5);
  hildon_button_set_value_alignment(HILDON_BUTTON(button), 0.5, 0.5);
  g_object_set_data(G_OBJECT(button), "sel_title", sel_title);
  g_object_set_data(G_OBJECT(button), "fileptr", filename);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
  		     GTK_SIGNAL_FUNC(on_browse), NULL);
  return button;
#endif
}


