/*
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <string.h>
#include <math.h>

#include <time.h>
#include <sys/time.h>

#include "gpxview.h"
#include "custom_size_renderer.h"
#include "custom_rating_renderer.h"
#include "custom_type_renderer.h"

#ifdef USE_MAEMO
#include <hildon/hildon-banner.h>
#endif

#include <locale.h>

extern char *strcasestr (__const char *__haystack, __const char *__needle);

#ifdef USE_BREAD_CRUMB_TRAIL
static void crumb_add(appdata_t *appdata, char *name, int level, 
		      gpointer user_data);

enum {
  CRUMB_GPXLIST = 0,
  CRUMB_CACHELIST,
  CRUMB_SEARCH_GLOBAL,
  CRUMB_SEARCH_GPX,
  CRUMB_CACHE,
};
#endif

/* size of first buffer malloc; start small to exercise grow routines */
#define	FIRSTSIZE	1

void errorf(const char *fmt, ...) {
  va_list args;
  char *buf = NULL;
  size_t bufsize;
  char *newbuf;
  size_t nextsize = 0;
  int outsize;
  
  bufsize = 0;
  for (;;) {
    if (bufsize == 0) {
      if ((buf = (char *)malloc(FIRSTSIZE)) == NULL) 
	return;

      bufsize = 1;
    } else if ((newbuf = (char *)realloc(buf, nextsize)) != NULL) {
      buf = newbuf;
      bufsize = nextsize;
    } else {
      free(buf);
      return;
    }
    
    va_start(args, fmt);
    outsize = vsnprintf(buf, bufsize, fmt, args);
    va_end(args);
    
    if (outsize == -1) {
      nextsize = bufsize * 2;
    } else if (outsize == bufsize) {
      nextsize = bufsize * 2;
    } else if (outsize > bufsize) {
      nextsize = outsize + 2;  // Linux!
    } else if (outsize == bufsize - 1) {
      nextsize = bufsize * 2;
    } else {
      /* Output was not truncated */
      break;
    }
  }

  GtkWidget *dialog = gtk_message_dialog_new(
			   GTK_WINDOW(NULL),
			   GTK_DIALOG_DESTROY_WITH_PARENT,
			   GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
			   buf);

  gtk_window_set_title(GTK_WINDOW(dialog), _("ERROR"));

  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);

  free(buf);
}

gpx_t *choose_file(appdata_t *appdata, gboolean whole_dir) {
  GtkWidget *dialog;
  gpx_t *gpx = NULL;
  
#ifdef USE_MAEMO
  dialog = hildon_file_chooser_dialog_new(GTK_WINDOW(appdata->window), 
			whole_dir?GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER :
					GTK_FILE_CHOOSER_ACTION_OPEN);
  
#ifdef HILDON_HELP
  hildon_help_dialog_help_enable(GTK_DIALOG(dialog), 
		 HELP_ID_IMPORT, appdata->osso_context);
#endif
#else
  dialog = gtk_file_chooser_dialog_new (whole_dir?_("Import directory"):
					_("Import file"),
			GTK_WINDOW(appdata->window),
			whole_dir?GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER :
					GTK_FILE_CHOOSER_ACTION_OPEN,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
			NULL);
#endif
  
  /* use path if one is present */
  if(appdata->path) 
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), 
					appdata->path);

  gtk_widget_show_all (GTK_WIDGET(dialog));
  if (gtk_dialog_run (GTK_DIALOG(dialog)) == GTK_FM_OK) {
    char *filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));

    gpx_dialog_t *dialog = gpx_busy_dialog_new(GTK_WIDGET(appdata->window));

    if(!whole_dir) 
      gpx = gpx_parse(dialog, filename);
    else {
      /* cur trailing '/' if present */
      if(strlastchr(filename) == '/')
	filename[strlen(filename)] = 0;

      gpx = gpx_parse_dir(dialog, filename);
    }

    gpx_busy_dialog_destroy(dialog);

    /* save path if gpx was successfully loaded */
    if(gpx) {
      char *r = strrchr(filename, '/');

      /* there is a delimiter, use everything left of it as path */
      if(r && !whole_dir) {
	*r = 0;
	if(appdata->path) free(appdata->path);
	appdata->path = strdup(filename);
	/* restore path ... just in case ... */
	*r = '/';
      }

      if(whole_dir)
	appdata->path = strdup(filename);
    } else
      errorf(_("Load error"));

    g_free (filename);
  }
  
  gtk_widget_destroy (dialog);

  return gpx;
}

/******************** begin of cachelist ********************/

enum {
  CACHELIST_COL_TYPE = 0,
  CACHELIST_COL_ID,
  CACHELIST_COL_NAME,
  CACHELIST_COL_SIZE,
  CACHELIST_COL_RATING,
  CACHELIST_COL_BEARING,
  CACHELIST_COL_DISTANCE,
  CACHELIST_COL_DATA,
  CACHELIST_COL_AVAIL,
  CACHELIST_COL_ARCHIVE,
  CACHELIST_NUM_COLS
} ;

void cachelist_view_onRowActivated(GtkTreeView        *treeview,
				   GtkTreePath        *path,
				   GtkTreeViewColumn  *col,
				   gpointer            userdata) {
  appdata_t *appdata = (appdata_t*)userdata;
  GtkTreeIter   iter;
  GtkTreeModel *model = gtk_tree_view_get_model(treeview);

#ifdef USE_MAEMO
  /* check if a cache is already selected and ignore click if yes */
  /* (was probably a double click) */
  if(appdata->cur_cache) return;
#endif

  if(gtk_tree_model_get_iter(model, &iter, path)) {
    cache_t *cache;
    gtk_tree_model_get(model, &iter, CACHELIST_COL_DATA, &cache, -1);
#ifndef USE_BREAD_CRUMB_TRAIL
    cache_dialog(appdata, cache);
#else
    gtk_container_remove(GTK_CONTAINER(appdata->vbox), appdata->cur_view);
    appdata->cur_view = cache_view(appdata, cache);
    gtk_box_pack_start_defaults(GTK_BOX(appdata->vbox), appdata->cur_view);
    gtk_widget_show_all(appdata->vbox);

    crumb_add(appdata, cache->name, CRUMB_CACHE, cache);
#endif
  }
}

typedef struct {
  appdata_t *appdata;
  GtkTreePath *path;  
  gboolean done;
} cachelist_expose_t;

static gboolean cachelist_expose(GtkWidget *widget, GdkEventExpose *event,
				 gpointer user_data) {
  cachelist_expose_t *ce = (cachelist_expose_t*)user_data;

  if(event->type == GDK_EXPOSE) {
    if(ce->path && !ce->done) {
      gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(widget), 
				   ce->path, NULL, TRUE, 0.5, 0.5);
      gtk_tree_path_free(ce->path);
      ce->done = TRUE;
    }
  }

  return FALSE;
}

static void cachelist_destroy(GtkWidget *widget, gpointer user_data) {
  cachelist_expose_t *ce = (cachelist_expose_t*)user_data;

  printf("cachelist timer removed\n");
  g_assert(ce->appdata->cachelist_handler_id);
  gtk_timeout_remove(ce->appdata->cachelist_handler_id);
  ce->appdata->cachelist_handler_id = 0;

  free(user_data);
}

#define CACHELIST_UPDATE_TIMEOUT (30000)

static GtkWidget *cachelist_create(appdata_t *appdata, gpx_t *gpx,
				   cache_t *sel_cache);

void cachelist_redraw(appdata_t *appdata) {
#ifndef USE_MAEMO
  //  gtk_container_remove(GTK_CONTAINER(appdata->vbox), appdata->cur_view);
  //  appdata->cur_view = gpxlist_create_view_and_model(appdata, NULL);
  //  gtk_box_pack_start_defaults(GTK_BOX(appdata->vbox), appdata->cur_view);
  // gtk_widget_show_all(appdata->vbox);
#else

  if(!appdata->cur_view) {
    printf("cachelist redraw: no active view\n");
    return;
  }

  g_assert(!appdata->cur_cache);
  int redraw = 0;
  if(appdata->search_results)
    redraw = 1;
  else {
    if(appdata->cur_gpx) 
      redraw = 2;  // redraw cachelist
  }

  if(redraw) {
    GtkWidget *container = appdata->vbox;

    printf("redraw %d\n", redraw);

#ifdef USE_STACKABLE_WINDOW
    HildonWindowStack *stack = hildon_window_stack_get_default();
    container = hildon_window_stack_peek(stack);
#endif

    gtk_container_remove(GTK_CONTAINER(container), appdata->cur_view);
    switch(redraw) {
    case 1:
      appdata->cur_view = cachelist_create(appdata, 
					   appdata->search_results, NULL);
      break;
    case 2:
      appdata->cur_view = cachelist_create(appdata, 
					   appdata->cur_gpx, NULL);
      break;
    }

#ifdef USE_STACKABLE_WINDOW
    if(container != appdata->vbox)
      gtk_container_add(GTK_CONTAINER(container), appdata->cur_view); 
    else
#endif
      gtk_box_pack_start_defaults(GTK_BOX(container), appdata->cur_view);
    
    gtk_widget_show_all(container);
  }
#endif
}


static gboolean cachelist_update(gpointer data) {

  printf("cachelist timer fired!\n");

#ifdef USE_MAEMO
  appdata_t *appdata = (appdata_t*)data;

  if(appdata->cur_cache)
    return TRUE;

  if(appdata->cachelist_disable_screensaver)
    if (osso_display_blanking_pause(appdata->osso_context) != OSSO_OK) 
      fprintf(stderr, "error with display blank\n");

  if(appdata->cachelist_update)
    cachelist_redraw(appdata);
#endif

  return TRUE;
}

static void cachelist_timer_reset(appdata_t *appdata) {
  printf("cachelist timer reset\n");
  g_assert(appdata->cachelist_handler_id);
  gtk_timeout_remove(appdata->cachelist_handler_id);
  appdata->cachelist_handler_id = 
    gtk_timeout_add(CACHELIST_UPDATE_TIMEOUT, cachelist_update, appdata);  
}

static gboolean cachelist_update_reset0(GtkWidget *widget,
					GdkEventButton *event,
					gpointer user_data) {
  cachelist_timer_reset((appdata_t*)user_data);
  return FALSE;
}

static void cachelist_update_reset1(GtkAdjustment *adj,
				    gpointer user_data) {
  cachelist_timer_reset((appdata_t*)user_data);
}

static GtkWidget *cachelist_create(appdata_t *appdata, gpx_t *gpx,
				   cache_t *sel_cache) {
  GtkCellRenderer *renderer;
  GtkWidget       *view;
  GtkListStore    *store;
  GtkTreeIter     iter;

  if(!gpx->notes_loaded) {
    notes_load_all(appdata, gpx);
    gpx->notes_loaded = TRUE;
  }

  appdata->cur_items = appdata->cachelist_items;

  /* first sort the caches */
  pos_t *refpos = get_pos(appdata);
  gpx_sort(gpx, GPX_SORT_BY_DISTANCE, refpos);

  view = gtk_tree_view_new();
  
  /* --- "Type" column --- */
  renderer = custom_cell_renderer_type_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(view),
      -1, "Type", renderer, "type", CACHELIST_COL_TYPE, NULL);
  
  /* --- "Id" column --- */
  if(appdata->cachelist_items & CACHELIST_ITEM_ID) {
    renderer = gtk_cell_renderer_text_new();
    gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(view),
      -1, "Id", renderer, "text", CACHELIST_COL_ID, NULL);
  }

  /* --- "Name" column --- */
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "ellipsize", PANGO_ELLIPSIZE_END, NULL );

  GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes(
		 "Name", renderer, "text", CACHELIST_COL_NAME, NULL);
  gtk_tree_view_column_set_expand(column, TRUE);
  gtk_tree_view_insert_column(GTK_TREE_VIEW(view), column, -1);

  g_object_set(renderer, "foreground", "#ff0000", NULL );
  gtk_tree_view_column_add_attribute(column, renderer, "strikethrough", 
				     CACHELIST_COL_AVAIL);
  gtk_tree_view_column_add_attribute(column, renderer, 
			     "foreground-set", CACHELIST_COL_ARCHIVE);

  /* --- "Size" column --- */
  if(appdata->cachelist_items & CACHELIST_ITEM_SIZE) {
    renderer = custom_cell_renderer_size_new();
    gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(view),
      -1, "Size", renderer, "size", CACHELIST_COL_SIZE, NULL);
  }

  /* --- "Rating" column --- */
  if(appdata->cachelist_items & CACHELIST_ITEM_RATING) {
    renderer = custom_cell_renderer_rating_new();
    gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(view),
      -1, "Rating", renderer, "rating", CACHELIST_COL_RATING, NULL);
  }

  /* --- "Bearing" column --- */
  renderer = gtk_cell_renderer_pixbuf_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(view),
      -1, "Bearing", renderer, "pixbuf", CACHELIST_COL_BEARING, NULL);

  /* --- "Distance" column --- */
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(view),
      -1, "Distance", renderer, "text", CACHELIST_COL_DISTANCE, NULL);

  store = gtk_list_store_new(CACHELIST_NUM_COLS, G_TYPE_INT, 
			     G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT, 
			     G_TYPE_INT, GDK_TYPE_PIXBUF, G_TYPE_STRING, 
			     G_TYPE_POINTER, G_TYPE_BOOLEAN, 
			     G_TYPE_BOOLEAN);

  GtkTreeSelection *sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
  
  GtkTreeIter sel_iter;
  gboolean sel_iter_valid = FALSE;
  GtkTreePath *path = NULL; 
  cache_t *cache = gpx->cache;
  while(cache) {
    char str[32];
    gpx_pos_get_distance_str(str, sizeof(str), 
		     *refpos, gpx_cache_pos(cache), appdata->imperial);

    int dint = (int)(cache->difficulty*2-2);
    if(dint < 0) dint = 0;
    if(dint > 8) dint = 8;

    int tint = (int)(cache->terrain*2-2);
    if(tint < 0) tint = 0;
    if(tint > 8) tint = 8;

    /* cache type includes "solved" flag in lowest bit */
    int type = (cache->type<<8) +
      (cache->notes?4:0) +
      ((cache->notes && cache->notes->override)?1:0) +
      ((cache->notes && cache->notes->found)?2:0);

    if((!(type & 2)) || !appdata->cachelist_hide_found) {

      /* Append a row and fill in some data */
      gtk_list_store_append (store, &iter);

      gtk_list_store_set(store, &iter,
			 CACHELIST_COL_TYPE, type,
			 CACHELIST_COL_ID,   cache->id,
			 CACHELIST_COL_NAME, cache->name,
			 CACHELIST_COL_SIZE, cache->container,
			 CACHELIST_COL_RATING, 100 * dint + tint,
			 CACHELIST_COL_BEARING, 
			 icon_bearing(*refpos, gpx_cache_pos(cache)),
			 CACHELIST_COL_DISTANCE, str,
			 CACHELIST_COL_DATA, cache,
			 CACHELIST_COL_AVAIL, !cache->available || 
			 cache->archived,
			 CACHELIST_COL_ARCHIVE, cache->archived,
			 -1);

      if(cache == sel_cache) {
	sel_iter = iter;
	sel_iter_valid = TRUE;
      }
    }

    cache = cache->next;
  }

  gtk_tree_view_set_model(GTK_TREE_VIEW(view), GTK_TREE_MODEL(store));
  g_object_unref(store);

  if(sel_iter_valid) {
    gtk_tree_selection_select_iter(sel, &sel_iter);
    path = gtk_tree_model_get_path(GTK_TREE_MODEL(store), &sel_iter);
  }

  /* make list react on clicks */
  g_signal_connect(view, "row-activated", 
		   (GCallback)cachelist_view_onRowActivated, appdata);

  cachelist_expose_t *ce = malloc(sizeof(cachelist_expose_t));
  ce->appdata = appdata;
  ce->path = path;
  ce->done = FALSE;

  g_signal_connect(view, "expose-event", 
		   (GCallback)cachelist_expose, ce);
  g_signal_connect(view, "destroy",
		   (GCallback)cachelist_destroy, ce);

  /* put this inside a scrolled view */
#ifndef USE_PANNABLE_AREA
  GtkWidget *scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
				 GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scrolled_window), view);
#else
  GtkWidget *pannable_area = hildon_pannable_area_new();
  
  gtk_container_add(GTK_CONTAINER(pannable_area), view);
#endif

  /* add a timer for automatic update */
  g_assert(!appdata->cachelist_handler_id);
  appdata->cachelist_handler_id = 
    gtk_timeout_add(CACHELIST_UPDATE_TIMEOUT, cachelist_update, appdata);

  /* update timer is being reset if the user scrolls or selects */
  g_signal_connect(view, "button-press-event", 
		   (GCallback)cachelist_update_reset0, appdata);

#ifndef USE_PANNABLE_AREA
  g_signal_connect(gtk_scrolled_window_get_vadjustment(
		       GTK_SCROLLED_WINDOW(scrolled_window)), 
			   "value-changed", 
			   (GCallback)cachelist_update_reset1, appdata);

  return scrolled_window;
#else
  g_signal_connect(hildon_pannable_area_get_vadjustment(
		       HILDON_PANNABLE_AREA(pannable_area)), 
			   "value-changed", 
			   (GCallback)cachelist_update_reset1, appdata);


  return pannable_area;
#endif
}

#ifndef USE_MAEMO
void cachelist_dialog(appdata_t *appdata, gpx_t *gpx) {
  GtkWidget *dialog = 
    gtk_dialog_new_with_buttons(gpx->name, GTK_WINDOW(appdata->window),
				GTK_DIALOG_NO_SEPARATOR | GTK_DIALOG_MODAL | 
				GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
				NULL);
  
  gtk_window_set_default_size(GTK_WINDOW(dialog), DIALOG_WIDTH, DIALOG_HEIGHT);

  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), 
		    cachelist_create(appdata, gpx, NULL));

  gtk_widget_show_all(dialog);

  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);
}
#else
#ifdef USE_STACKABLE_WINDOW
static void search_result_free(gpx_t *result);

void on_cachelist_destroy(GtkWidget *widget, appdata_t *appdata) {
  if(appdata->search_results) {
    search_result_free(appdata->search_results);
    appdata->search_results = NULL;
  }
  appdata->cur_gpx = NULL;

  /* restore cur_view */
  appdata->cur_view = g_object_get_data(G_OBJECT(widget), "cur_view");
}

void cachelist_dialog(appdata_t *appdata, gpx_t *gpx) {
  GtkWidget *window = hildon_stackable_window_new();

  /* store last "cur_view" in window */
  g_object_set_data(G_OBJECT(window), "cur_view", appdata->cur_view);

  appdata->cur_gpx = gpx;
  char *title = g_strdup_printf("%s - GPXView", gpx->name);
  gtk_window_set_title(GTK_WINDOW(window), title);
  g_free(title);

  appdata->cur_view = cachelist_create(appdata, gpx, NULL);
  gtk_container_add(GTK_CONTAINER(window), appdata->cur_view);
		    

  hildon_window_set_app_menu(HILDON_WINDOW(window), 
			     menu_create(appdata, MENU_CACHELIST));

  g_signal_connect(G_OBJECT(window), "destroy", 
		   G_CALLBACK(on_cachelist_destroy), appdata);

  gtk_widget_show_all(window);
}
#endif
#endif

/******************** end of cachelist ********************/

/******************** begin of gpxlist ********************/

enum {
  GPXLIST_COL_ICON = 0,
  GPXLIST_COL_FILENAME,
  GPXLIST_COL_NAME,
  GPXLIST_COL_DATE,
  GPXLIST_COL_CACHES,
  GPXLIST_COL_OPEN,
#ifdef USE_PANNABLE_AREA
  GPXLIST_COL_DELETE,
#endif
  GPXLIST_COL_DATA,
  GPXLIST_NUM_COLS
} ;

static GdkPixbuf *gpx_icon_get(gpx_t *gpx) {
  if(gpx->filename && g_file_test(gpx->filename, G_FILE_TEST_IS_DIR)) 
    return icon_get(ICON_FILE, 1);

  if(gpx->filename&& !strcasecmp(gpx->filename+strlen(gpx->filename)-4,".zip")) 
    return icon_get(ICON_FILE, 2);

  return icon_get(ICON_FILE, 0);
}

static void gpxlist_set(GtkListStore *store, GtkTreeIter *iter, gpx_t *gpx) {
  char date_str[32], cnum[32];

  if(gpx->year && gpx->month && gpx->day) {
    GDate *date = g_date_new_dmy(gpx->day, gpx->month, gpx->year);
    g_date_strftime(date_str, sizeof(date_str), "%x", date);
    g_date_free(date);
  } else
    strcpy(date_str, "---");

  char *fname = strrchr(gpx->filename, '/');
  if(!fname) fname = gpx->filename;
  else       fname++;  /* skip '/' */
  
  snprintf(cnum, sizeof(cnum), "%d", gpx_total_caches(gpx));
  
  /* Append a row and fill in some data */
  gtk_list_store_set(store, iter,
		     GPXLIST_COL_ICON, gpx_icon_get(gpx),
		     GPXLIST_COL_FILENAME, fname,
		     GPXLIST_COL_NAME, gpx->name,
		     GPXLIST_COL_DATE, gpx->closed?NULL:date_str,
		     GPXLIST_COL_OPEN, !gpx->closed,
		     GPXLIST_COL_CACHES, gpx->closed?NULL:cnum,
#ifdef USE_PANNABLE_AREA
		     GPXLIST_COL_DELETE, icon_get(ICON_MISC, 7),
#endif
		     GPXLIST_COL_DATA, gpx,
		     -1);
}

static void gpxlist_remove(appdata_t *appdata, 
			   GtkListStore *store, GtkTreeIter *iter, 
			   gpx_t *gpx) {

  printf("removing %s\n", gpx->name);
  
  /* de-chain */
  gpx_t **prev = &appdata->gpx;
  while(*prev != gpx) prev = &((*prev)->next);
  *prev = gpx->next;
  
  /* remove gconf entry if file was closed */
  gconf_remove_closed_name(appdata, gpx->filename);
  
  /* free gpx itself */
  gpx_free(gpx);
  
  /* and remove from store */
  gtk_list_store_remove(store, iter);
}

static void gpxlist_close(appdata_t *appdata,
			  GtkListStore *store, GtkTreeIter *iter, 
			  gpx_t *gpx) {
  
  printf("closing %s\n", gpx->name);

  g_assert(!gpx->closed);
  gpx->closed = TRUE;
  
  /* free all associated caches */
  gpx_free_caches(gpx);
  
  /* update entry */
  gpxlist_set(store, iter, gpx);
  
  /* save name in gconf so we know this has been closed */
  gconf_save_closed_name(appdata, gpx->filename, gpx->name);
}

static void gpxlist_view_onRowActivated(GtkTreeView *treeview,
			  GtkTreePath        *path,
			  GtkTreeViewColumn  *col,
			  gpointer            userdata) {
  appdata_t *appdata = (appdata_t*)userdata;
  GtkTreeIter   iter;
  GtkTreeModel *model = gtk_tree_view_get_model(treeview);
  
#ifdef USE_MAEMO
  /* check if a cache is already selected and ignore click if yes */
  /* (was probably a double click) */
  if(appdata->cur_gpx) return;
#endif

  if (gtk_tree_model_get_iter(model, &iter, path)) {
    gpx_t *gpx;
    gtk_tree_model_get(model, &iter, GPXLIST_COL_DATA, &gpx, -1);

#ifdef USE_PANNABLE_AREA
    /* get name of column the user clicked on */
    const char *col_name = NULL;
    if(col) col_name = gtk_tree_view_column_get_title(col);

    if(col_name && !strcmp(col_name, "Del")) {
      printf("clicked delete\n");

      /* ask user what he wants */
      GtkWidget *dialog = gtk_message_dialog_new(
  	         GTK_WINDOW(appdata->window),
		 GTK_DIALOG_DESTROY_WITH_PARENT,
		 GTK_MESSAGE_QUESTION, GTK_BUTTONS_CANCEL,
		 _("Do you want to close this entry only or do "
		   "you want to remove it completely from the list?"));

      gtk_dialog_add_buttons(GTK_DIALOG(dialog), 
			     _("Remove"), 1,
			     _("Close"), 2,
			     NULL);

      if(gpx->closed)
	gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog), 2, FALSE);

      gtk_window_set_title(GTK_WINDOW(dialog), _("Remove entry?"));
      
      /* set the active flag again if the user answered "no" */
      switch(gtk_dialog_run(GTK_DIALOG(dialog))) {

      case 1:
	gpxlist_remove(appdata, GTK_LIST_STORE(model), &iter, gpx);
	break;

      case 2:
	gpxlist_close(appdata, GTK_LIST_STORE(model), &iter, gpx);
	break;

      default:
	break;
      }
      
      gtk_widget_destroy(dialog);

    } else
#endif
    {

      /* this gpx file may be closed. Since the user definitely wants */
      /* to use it, we just open it again */
      if(gpx->closed) {
	gpx_dialog_t *dialog = 
	  gpx_busy_dialog_new(GTK_WIDGET(appdata->window));
	gpx_t *new = NULL;
	
	if(g_file_test(gpx->filename, G_FILE_TEST_IS_DIR)) 
	  new = gpx_parse_dir(dialog, gpx->filename);
	else
	  new = gpx_parse(dialog, gpx->filename);
	
	if(new) {
	  gpx_t **prev = &(appdata->gpx);
	  while(*prev && *prev != gpx)
	    prev = &(*prev)->next;
	  
	  /* this entry _must_ be in the list */
	  g_assert(*prev);
	  
	  /* replace gpx entry with the new "open" one */
	  (*prev) = new;
	  new->next = gpx->next;
	  gpx->next = NULL;
	  
	  /* free old closed one */
	  gpx_free(gpx);
	  
	  gpx = new;
	  
	  /* finally update the visible list */
	  gpxlist_set(appdata->gpxstore, &iter, gpx);
	  
	  /* and remove gconf entry */
	  gconf_remove_closed_name(appdata, gpx->filename);
	  
#ifndef USE_PANNABLE_AREA
	  gtk_widget_set_sensitive(appdata->menu_close, TRUE);
#endif
	} else {
	  printf("unable to reopen file %s\n", gpx->filename);
	  return;
	}
      
	gpx_busy_dialog_destroy(dialog);
      }
#ifndef USE_BREAD_CRUMB_TRAIL
#ifdef USE_STACKABLE_WINDOW
      if(!appdata->cur_gpx)
#endif
	cachelist_dialog(appdata, gpx);
#ifdef USE_STACKABLE_WINDOW
      else
	printf("selected gpx, but cachelist window already present\n");
#endif
#else
      gtk_container_remove(GTK_CONTAINER(appdata->vbox), appdata->cur_view);
      appdata->cur_view = cachelist_create(appdata, gpx, NULL);
      gtk_box_pack_start_defaults(GTK_BOX(appdata->vbox), appdata->cur_view);
      gtk_widget_show_all(appdata->vbox);
      
      crumb_add(appdata, gpx->name, CRUMB_CACHELIST, gpx);
#endif
    }
  }
}

#ifndef USE_PANNABLE_AREA
static gboolean
view_selection_func(GtkTreeSelection *selection, GtkTreeModel *model,
		     GtkTreePath *path, gboolean path_currently_selected,
		     gpointer userdata) {
  appdata_t *appdata = (appdata_t*)userdata;
  GtkTreeIter iter;
    
  if(gtk_tree_model_get_iter(model, &iter, path)) {
    gpx_t *gpx;
    gtk_tree_model_get(model, &iter, GPXLIST_COL_DATA, &gpx, -1);

    gtk_widget_set_sensitive(appdata->menu_remove, !path_currently_selected);

    if(!gpx->closed)
      gtk_widget_set_sensitive(appdata->menu_close,  !path_currently_selected);
  }
  
  return TRUE; /* allow selection state to change */
}
#endif

static GtkWidget *gpxlist_create_view_and_model(appdata_t *appdata, 
						gpx_t *sel_gpx) {
  gpx_t               *gpx = appdata->gpx;
  GtkCellRenderer     *renderer;

  /* saved displayed items */
  appdata->cur_items = appdata->gpxlist_items;

#ifndef USE_PANNABLE_AREA
  /* nothing selected yet */
  gtk_widget_set_sensitive(appdata->menu_remove,  FALSE);
  gtk_widget_set_sensitive(appdata->menu_close,   FALSE);
#endif

  appdata->gpxview = gtk_tree_view_new ();

  GtkTreeSelection *selection = 
    gtk_tree_view_get_selection(GTK_TREE_VIEW(appdata->gpxview));
#ifndef USE_PANNABLE_AREA
  gtk_tree_selection_set_select_function(selection, view_selection_func, 
					 appdata, NULL);
#endif

  /* --- "Icon" column --- */
  renderer = gtk_cell_renderer_pixbuf_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(appdata->gpxview),
		  -1, "Icon", renderer, 
		  "pixbuf", GPXLIST_COL_ICON,
	      //	          "sensitive", GPXLIST_COL_OPEN, 
	          NULL);

  /* --- "FileName" column --- */
  if(appdata->gpxlist_items & GPXLIST_ITEM_FILENAME) {
    renderer = gtk_cell_renderer_text_new();
    gtk_tree_view_insert_column_with_attributes(
		GTK_TREE_VIEW(appdata->gpxview),
		-1, "Filename", renderer, 
		"text", GPXLIST_COL_FILENAME, 
		"sensitive", GPXLIST_COL_OPEN, 
		NULL);
  }

  /* --- "Name" column --- */
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "ellipsize", PANGO_ELLIPSIZE_END, NULL );

  GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes(
		 "Name", renderer, 
		 "text", GPXLIST_COL_NAME,
		 "sensitive", GPXLIST_COL_OPEN, 
		 NULL);
  gtk_tree_view_column_set_expand(column, TRUE);
  gtk_tree_view_insert_column(GTK_TREE_VIEW(appdata->gpxview), column, -1);

  /* --- "Date" column --- */
  if(appdata->gpxlist_items & GPXLIST_ITEM_DATE) {
    renderer = gtk_cell_renderer_text_new();
    g_object_set(renderer, "xalign", 1.0, NULL );
    gtk_tree_view_insert_column_with_attributes(
		GTK_TREE_VIEW(appdata->gpxview),
		-1, "Date", renderer, 
		"text", GPXLIST_COL_DATE,
		"sensitive", GPXLIST_COL_OPEN, 
		NULL);
  }

  /* --- "Number of caches" column --- */
  if(appdata->gpxlist_items & GPXLIST_ITEM_CNUM) {
    renderer = gtk_cell_renderer_text_new();
    g_object_set(renderer, "xalign", 1.0, NULL );
    gtk_tree_view_insert_column_with_attributes(
		GTK_TREE_VIEW(appdata->gpxview),
		-1, "#Caches", renderer, 
		"text", GPXLIST_COL_CACHES,
 		"sensitive", GPXLIST_COL_OPEN, 
		NULL);
  }

#ifdef USE_PANNABLE_AREA
  /* --- "Delete" column --- */
  renderer = gtk_cell_renderer_pixbuf_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(appdata->gpxview),
		  -1, "Del", renderer, 
		  "pixbuf", GPXLIST_COL_DELETE,
		  "sensitive", GPXLIST_COL_OPEN, 
	          NULL);
#endif

  /* build and fill the store */
  appdata->gpxstore = gtk_list_store_new(GPXLIST_NUM_COLS, GDK_TYPE_PIXBUF,
			 G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, 
			 G_TYPE_STRING, G_TYPE_BOOLEAN, 
#ifdef USE_PANNABLE_AREA
			 GDK_TYPE_PIXBUF, 
#endif
			 G_TYPE_POINTER);

  GtkTreePath *path = NULL; 
  GtkTreeIter sel_iter;
  gboolean sel_iter_valid = FALSE;
  while(gpx) {
    /* don't display entries that failed at load */
    if(!gpx->failed) {
      GtkTreeIter         iter;
      gtk_list_store_append(appdata->gpxstore, &iter);
      gpxlist_set(appdata->gpxstore, &iter, gpx);
      
      if(gpx == sel_gpx) {
	sel_iter = iter;
	sel_iter_valid = TRUE;
      }
    }

    gpx = gpx->next;
  }
  
  gtk_tree_view_set_model(GTK_TREE_VIEW(appdata->gpxview), 
			  GTK_TREE_MODEL(appdata->gpxstore));

  g_object_unref(appdata->gpxstore);

  if(sel_iter_valid) {
    gtk_tree_selection_select_iter(selection, &sel_iter);
    path = gtk_tree_model_get_path(GTK_TREE_MODEL(appdata->gpxstore), 
				   &sel_iter);
    gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(appdata->gpxview), 
				 path, NULL, TRUE, 0.0, 0.0);
    gtk_tree_path_free(path);
  }

  /* make list react on clicks */
  g_signal_connect(appdata->gpxview, "row-activated", 
		   (GCallback)gpxlist_view_onRowActivated, appdata);

  /* put this inside a scrolled view */
#ifndef USE_PANNABLE_AREA
  GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
				 GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scrolled_window), appdata->gpxview);

  return scrolled_window;
#else
  GtkWidget *pannable_area = hildon_pannable_area_new();
  gtk_container_add(GTK_CONTAINER(pannable_area), appdata->gpxview);

  return pannable_area;
#endif
}

/* add last entry in gpx list to visual representation */
static void gpxlist_add(appdata_t *appdata, gpx_t *new) {
  GtkTreeIter         iter;

  gtk_list_store_append(appdata->gpxstore, &iter);
  gpxlist_set(appdata->gpxstore, &iter, new);

  /* and attach entry to end of list */
  gpx_t **gpx = &appdata->gpx;
  while(*gpx) gpx = &((*gpx)->next);
  *gpx = new;
}

/******************** end of gpxlist ********************/

/******************** begin of menu *********************/

typedef struct {
  appdata_t *appdata;
  GtkWidget *dialog;
} about_context_t;

#ifdef ENABLE_BROWSER_INTERFACE
void on_paypal_button_clicked(GtkButton *button, about_context_t *context) {
  gtk_dialog_response(GTK_DIALOG(context->dialog), GTK_RESPONSE_ACCEPT); 
  browser_url(context->appdata, 
	      "https://www.paypal.com/cgi-bin/webscr"
	      "?cmd=_s-xclick&hosted_button_id=7400558");
}
#endif

static void 
cb_menu_about(GtkWidget *window, gpointer data) {
  about_context_t context;

  context.appdata = (appdata_t *)data;

#ifdef ENABLE_LIBLOCATION
  char *uses = "uses liblocation";
#elif defined(ENABLE_GPSBT)
  char *uses = "uses gpsbt and gpsd";
#else
  char *uses = "uses gpsd";
#endif
  
  const gchar *authors[] = {
    "Till Harbaum <till@harbaum.org>",
    "John Stowers <john.stowers@gmail.com>",
    NULL };

  context.dialog = g_object_new(GTK_TYPE_ABOUT_DIALOG,
			"name", "GPXView",
			"version", VERSION,
			"copyright", _("Copyright 2008-2009"),
			"authors", authors,		
			"website", _("http://www.harbaum.org/till/maemo"),
			"comments", _(uses),
			NULL);

#ifdef ENABLE_BROWSER_INTERFACE
  /* add a way to donate to the project */
  GtkWidget *alignment = gtk_alignment_new(0.5, 0, 0, 0);
  
  GtkWidget *hbox = gtk_hbox_new(FALSE, 8);
  gtk_box_pack_start(GTK_BOX(hbox),
		     gtk_label_new(_("Do you like GPXView?")), 
		     FALSE, FALSE, 0);

  GtkWidget *button = gtk_button_new();
  gtk_button_set_image(GTK_BUTTON(button), 
		       icon_get_widget(ICON_MISC, 8));
  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
  g_signal_connect(button, "clicked", 
		   G_CALLBACK(on_paypal_button_clicked), &context); 
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);

  gtk_container_add(GTK_CONTAINER(alignment), hbox);
  gtk_box_pack_start_defaults(GTK_BOX((GTK_DIALOG(context.dialog))->vbox),
			      alignment);

  gtk_widget_show_all(alignment);
#endif

  gtk_dialog_run(GTK_DIALOG(context.dialog));
  gtk_widget_destroy(context.dialog);
}

#if defined(USE_MAEMO) && defined(HILDON_HELP)
static void 
cb_menu_help(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  hildon_help_show(appdata->osso_context, HELP_ID_INTRO, 0);
}
#endif

static void 
cb_menu_add(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t *)data;

  gpx_t *new = choose_file(appdata, FALSE);
  if(new) gpxlist_add(appdata, new);
}

static void 
cb_menu_adddir(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t *)data;

  gpx_t *new = choose_file(appdata, TRUE);
  if(new) gpxlist_add(appdata, new);
}

#ifndef USE_PANNABLE_AREA
static void 
cb_menu_close(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t *)data;
  GtkTreeSelection *selection;
  GtkTreeModel     *model;
  GtkTreeIter       iter;
  
  printf("selected close\n");
  
  /* the entry cannot be closed again */
  gtk_widget_set_sensitive(appdata->menu_close,   FALSE);
  
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(appdata->gpxview));

  printf("gpxlist close\n");

  if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
    gpx_t *gpx = NULL;
    gtk_tree_model_get(model, &iter, GPXLIST_COL_DATA, &gpx, -1);

    if(gpx) gpxlist_close(appdata, GTK_LIST_STORE(model), &iter, gpx);
  } else {
    g_print ("no row selected.\n");
  }
}

static void 
cb_menu_remove(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t *)data;

  /* disable menu item */
  gtk_widget_set_sensitive(appdata->menu_remove,  FALSE);
  gtk_widget_set_sensitive(appdata->menu_close,   FALSE);

  GtkTreeModel     *model;
  GtkTreeIter       iter;
  GtkTreeSelection *selection =
    gtk_tree_view_get_selection(GTK_TREE_VIEW(appdata->gpxview));

  printf("gpxlist remove\n");

  if(gtk_tree_selection_get_selected(selection, &model, &iter)) {
    gpx_t *gpx = NULL;
    gtk_tree_model_get(model, &iter, GPXLIST_COL_DATA, &gpx, -1);
    
    if(gpx) gpxlist_remove(appdata, GTK_LIST_STORE(model), &iter, gpx);
  } else {
    g_print ("no row selected.\n");
  }
}

#endif // !USE_PANNABLE_AREA

static void search_result_free(gpx_t *result) {
  printf("freeing search results\n");    

  /* free found chain */
  cache_t *cache = result->cache;
  while(cache) {
    cache_t *next = cache->next;
    free(cache);
    cache = next;
  }
  free(result->name);
  free(result);
}

#define MAX_HITS 50

static time_t localize_time(time_t in) {
  time_t ret;
  char *tz;
  struct tm *tm = localtime(&in);
  
  tz = getenv("TZ");
  setenv("TZ", "", 1);
  tzset();
  ret = mktime(tm);
  if (tz)
    setenv("TZ", tz, 1);
  else
    unsetenv("TZ");
  tzset();
  return ret;
}

static int days_ago(time_t in) {
  int day_in  = localize_time(in) / (60*60*24);
  int day_now = localize_time(time(NULL)) / (60*60*24);

  return day_now - day_in;
}

gpx_t *search_do(appdata_t *appdata, gpx_t *gpx, char *phrase, 
		 int what, gboolean local) {
  /* walk through all caches */
      
  int hits = 0;
  gpx_t *found = malloc(sizeof(gpx_t));
  memset(found, 0, sizeof(gpx_t));
  cache_t **cacheP = &(found->cache);

  if(what & SEARCH_FINDS) {
    time_t loc_now = localize_time(time(NULL));
    printf("now: %ld days since 1/1/1970, days hour is %ld\n", 
	   loc_now/(60*60*24), loc_now%(60*60*24)/(60*60));
  }

  while(gpx && hits < MAX_HITS) {

    /* we need all notes ... */
    if(what & SEARCH_FINDS) {
      notes_load_all(appdata, gpx);
      gpx->notes_loaded = TRUE;      
    }

    cache_t *cache = gpx->cache;

    while(cache && hits < MAX_HITS) {
      gboolean hit = FALSE;

      if(what & SEARCH_FINDS) {
	if(cache->notes && cache->notes->found ) {
	  int days = days_ago(cache->notes->ftime);

	  if(cache->id)
	    printf("find of %s is %d days ago\n", cache->id, days);

	  if(days <= appdata->search_days)
	    hit = 1;
	}
      } else if(cache->id && (what & SEARCH_ID) && 
	 strcasestr(cache->id, phrase)) 
	hit = 1;
      else if(cache->name && (what & SEARCH_NAME) && 
	      strcasestr(cache->name, phrase))
	hit = 1;
      else if(cache->short_description && (what & SEARCH_DESC) && 
	      strcasestr(cache->short_description, phrase))
	hit = 1;
      else if(cache->long_description && (what & SEARCH_DESC) && 
	      strcasestr(cache->long_description, phrase))
	hit = 1;
      else if(cache->owner && (what & SEARCH_OWNER) && 
	      strcasestr(cache->owner, phrase))
	hit = 1;
      
      if(hit) {
	/* chain a copy of this cache structure into the found list */
	*cacheP = malloc(sizeof(cache_t));
	memcpy(*cacheP, cache, sizeof(cache_t));
	(*cacheP)->next = NULL;
	cacheP = &((*cacheP)->next);
	hits++;
      }
      cache = cache->next;
    }

    if(!local) gpx = gpx->next;
    else       gpx = NULL;       /* local search within one gpx only */
  }
  
  found->name = strdup(_("Search results"));

  return found;
}

typedef struct {
  appdata_t *appdata;
  GtkWidget *entry, *spinner;
  GtkWidget *in_id, *in_name, *in_desc, *in_owner, *in_finds;
} search_context_t;

static void callback_finds_toggled(GtkWidget *widget, gpointer data ) {
  search_context_t *context = (search_context_t*)data;

  gboolean in_finds = gtk_toggle_button_get_active(
	  GTK_TOGGLE_BUTTON(context->in_finds));

  gtk_widget_set_sensitive(context->entry,    !in_finds);
  gtk_widget_set_sensitive(context->in_id,    !in_finds);
  gtk_widget_set_sensitive(context->in_name,  !in_finds);
  gtk_widget_set_sensitive(context->in_desc,  !in_finds);
  gtk_widget_set_sensitive(context->in_owner, !in_finds);
  gtk_widget_set_sensitive(context->spinner,   in_finds);
}

static void 
cb_menu_search(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t *)data;

  search_context_t context;
  memset(&context, 0, sizeof(search_context_t));
  context.appdata = appdata;

  GtkWidget *dialog = gtk_dialog_new_with_buttons(_("Enter search phrase"),
  	    GTK_WINDOW(appdata->window), GTK_DIALOG_MODAL,
            GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
            GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
            NULL);

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox),
			      gtk_label_new(_("Search in:")));

  GtkWidget *table = gtk_table_new(2, 2, TRUE);
  gtk_table_set_col_spacing(GTK_TABLE(table), 0, 8);

  context.in_id = gtk_check_button_new_with_label(_("Waypoint IDs"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context.in_id), 
			       appdata->search & SEARCH_ID);
  gtk_table_attach_defaults(GTK_TABLE(table), context.in_id, 0, 1, 0, 1);

  context.in_name = gtk_check_button_new_with_label(_("Names"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context.in_name), 
			       appdata->search & SEARCH_NAME);
  gtk_table_attach_defaults(GTK_TABLE(table), context.in_name, 1, 2, 0, 1);

  context.in_desc = gtk_check_button_new_with_label(_("Descriptions"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context.in_desc), 
			       appdata->search & SEARCH_DESC);
  gtk_table_attach_defaults(GTK_TABLE(table), context.in_desc, 0, 1, 1, 2);

  context.in_owner = gtk_check_button_new_with_label(_("Owner"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context.in_owner), 
			       appdata->search & SEARCH_OWNER);
  gtk_table_attach_defaults(GTK_TABLE(table), context.in_owner, 1, 2, 1, 2);

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), table);

  /* -------------------------------------------------------------- */

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox),
			      gtk_label_new(_("Search for:")));
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox),
			      context.entry = gtk_entry_new());
  if(appdata->search_str)
    gtk_entry_set_text(GTK_ENTRY(context.entry), appdata->search_str);

  /* -------------------------------------------------------------- */

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), 
			      gtk_hseparator_new());

  GtkWidget *hbox = gtk_hbox_new(FALSE, 5);

  context.in_finds = gtk_check_button_new_with_label(_("Search finds for"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context.in_finds), 
			       appdata->search & SEARCH_FINDS);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context.in_finds);
  g_signal_connect(G_OBJECT(context.in_finds), "toggled",
		   G_CALLBACK(callback_finds_toggled), &context);

#ifndef USE_MAEMO
  GtkObject *adj = gtk_adjustment_new(appdata->search_days, 0, 99, 1, 10, 10);
  context.spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1, 0);
#else
  context.spinner = hildon_number_editor_new(0, 99);
  hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(context.spinner), 
				 appdata->search_days);
#endif
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context.spinner);

  gtk_box_pack_start_defaults(GTK_BOX(hbox), gtk_label_new(_("days")));

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox);

  /* -------------------------------------------------------------- */

  gtk_widget_show_all(dialog);
  callback_finds_toggled(NULL, &context);

  if(GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(dialog))) {
    char *p = strdup(gtk_entry_get_text(GTK_ENTRY(context.entry)));

    /* update saved search string */
    if(appdata->search_str) free(appdata->search_str);
    if(strlen(p) > 0)
      appdata->search_str = strdup(p);

#ifndef USE_MAEMO
    appdata->search_days = gtk_spin_button_get_value_as_int(
		    GTK_SPIN_BUTTON(context.spinner));
#else
    appdata->search_days = hildon_number_editor_get_value(
		    HILDON_NUMBER_EDITOR(context.spinner));
#endif

    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(context.in_finds)))
      appdata->search |=  SEARCH_FINDS;
    else
      appdata->search &= ~SEARCH_FINDS;

    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(context.in_id)))
      appdata->search |=  SEARCH_ID;
    else
      appdata->search &= ~SEARCH_ID;

    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(context.in_name)))
      appdata->search |=  SEARCH_NAME;
    else
      appdata->search &= ~SEARCH_NAME;

    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(context.in_desc)))
      appdata->search |=  SEARCH_DESC;
    else
      appdata->search &= ~SEARCH_DESC;
 
    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(context.in_owner)))
      appdata->search |=  SEARCH_OWNER;
    else
      appdata->search &= ~SEARCH_OWNER;
 
    gtk_widget_destroy(dialog);

    /* don't search if we are asked to search for nothing */
    if(((appdata->search & (SEARCH_ID|SEARCH_NAME|SEARCH_DESC|SEARCH_OWNER)) &&
	strlen(p) > 0) || (appdata->search & SEARCH_FINDS)) {

      printf("Search for %s (flags = %x)...\n", p, appdata->search);

#ifndef USE_BREAD_CRUMB_TRAIL
      gpx_t *found =
	search_do(appdata, appdata->gpx, p, appdata->search, FALSE);

      /* do search result dialog here ... */
      cachelist_dialog(appdata, found);
#ifndef USE_STACKABLE_WINDOW
      search_result_free(found);
#else
      appdata->search_results = found;
#endif
#else
      gpx_t *found = NULL;

      if(appdata->cur_gpx)
	found = search_do(appdata, appdata->cur_gpx, p, appdata->search, TRUE);
      else
	found = search_do(appdata, appdata->gpx, p, appdata->search, FALSE);

      gtk_container_remove(GTK_CONTAINER(appdata->vbox), appdata->cur_view);
      appdata->cur_view = cachelist_create(appdata, found, NULL);
      gtk_box_pack_start_defaults(GTK_BOX(appdata->vbox), appdata->cur_view);
      gtk_widget_show_all(appdata->vbox);
      crumb_add(appdata, found->name, 
		appdata->cur_gpx?CRUMB_SEARCH_GPX:CRUMB_SEARCH_GLOBAL, found);
#endif
    } else
      printf("No valid search: \"%s\" with flags %x!\n", p, appdata->search);

    free(p);
  } else 
    gtk_widget_destroy(dialog);
}

static void on_window_destroy (GtkWidget *widget, gpointer data);

#ifndef USE_MAEMO
static void 
cb_menu_quit(GtkWidget *window, gpointer data) {
  on_window_destroy(window, data);
}
#endif

#ifndef NO_COPY_N_PASTE
static void 
cb_menu_cut(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  if(appdata->active_buffer) {
    if(GTK_WIDGET_TYPE(appdata->active_buffer) == GTK_TYPE_TEXT_BUFFER) {
      gtk_text_buffer_cut_clipboard(GTK_TEXT_BUFFER(appdata->active_buffer), 
				    appdata->clipboard, TRUE);
    } else
      printf("cut: ERROR, not a text buffer\n");
  } else
    printf("cut: ERROR, no active buffer\n");
}

static void 
cb_menu_copy(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  if(appdata->active_buffer) {
    if(GTK_WIDGET_TYPE(appdata->active_buffer) == GTK_TYPE_TEXT_BUFFER) {
      gtk_text_buffer_copy_clipboard(GTK_TEXT_BUFFER(appdata->active_buffer), 
				     appdata->clipboard);
    } else if(GTK_WIDGET_TYPE(appdata->active_buffer) == gtk_html_get_type()) {
      printf("copy from html buffer\n");
      html_copy_to_clipboard(appdata);
    } else
      printf("copy: ERROR, not a text nor a html buffer\n");
  } else 
    printf("copy: ERROR, no active buffer\n");
}

static void 
cb_menu_paste(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  if(appdata->active_buffer) {
    if(GTK_WIDGET_TYPE(appdata->active_buffer) == GTK_TYPE_TEXT_BUFFER) {
      gtk_text_buffer_paste_clipboard(GTK_TEXT_BUFFER(appdata->active_buffer), 
				      appdata->clipboard, NULL, TRUE);
    } else
      printf("paste: ERROR, not a text buffer\n");
  } else
    printf("paste: ERROR, no active buffer\n");
}
#endif

static void 
cb_menu_export_log(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  notes_log_export(appdata);
}

#ifdef USE_MAEMO
static void 
cb_menu_export_mmpoi(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  mmpoi_export(appdata);
}
#endif

static void 
cb_menu_export_garmin(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  garmin_export(appdata);
}

#ifdef ENABLE_OSM_GPS_MAP
static void 
cb_menu_map(GtkWidget *window, gpointer data) {
  map((appdata_t *)data);
}
#endif

static void 
cb_menu_geomath(GtkWidget *window, gpointer data) {
  geomath_dialog((appdata_t *)data);
}

static void 
cb_menu_geotext(GtkWidget *window, gpointer data) {
  geotext_dialog((appdata_t *)data);
}

static void 
cb_menu_precpos(GtkWidget *window, gpointer data) {
  precise_position((appdata_t *)data);
}

#ifdef USE_STACKABLE_WINDOW
typedef struct {
  char *label, *desc;
  GtkSignalFunc activate_cb;
} menu_entry_t;

typedef struct {
  const char *title;
  const menu_entry_t *menu;
  int len;
} submenu_t;

#define COLUMNS  1

void on_submenu_entry_clicked(GtkButton *button, GtkWidget *menu) {

  /* force closing of submenu dialog */
  gtk_dialog_response(GTK_DIALOG(menu), GTK_RESPONSE_NONE);
  gtk_widget_hide(menu);
  
  /* let gtk clean up */
  while(gtk_events_pending()) 
    gtk_main_iteration();
}

static GtkWidget *app_submenu_create(appdata_t *appdata, 
				     const submenu_t *submenu) {

  /* create a oridinary dialog box */
  GtkWidget *dialog = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(dialog), _(submenu->title));
  gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
  gtk_window_set_transient_for(GTK_WINDOW(dialog), 
			       GTK_WINDOW(appdata->window));
  gtk_dialog_set_has_separator(GTK_DIALOG(dialog), FALSE);

  GtkWidget *table = gtk_table_new(submenu->len/COLUMNS, COLUMNS, TRUE);
  int x = 0, y = 0;

  const menu_entry_t *menu_entries = submenu->menu;
  while(menu_entries->label) {
    GtkWidget *button = NULL;

    button = hildon_button_new_with_text(
	    HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH,
	    HILDON_BUTTON_ARRANGEMENT_VERTICAL,
	    _(menu_entries->label), _(menu_entries->desc));

    /* try to center both texts */
    hildon_button_set_title_alignment(HILDON_BUTTON(button), 0.5, 0.5);
    hildon_button_set_value_alignment(HILDON_BUTTON(button), 0.5, 0.5);

    g_signal_connect(button, "clicked", 
		     G_CALLBACK(on_submenu_entry_clicked), dialog); 
    
    g_signal_connect(button, "clicked", 
		     menu_entries->activate_cb, appdata); 
    
    gtk_table_attach_defaults(GTK_TABLE(table),  button, x, x+1, y, y+1);
    
    x++;
    if(x == COLUMNS) { x = 0; y++; }

    menu_entries++;
  }

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), table);

  return dialog;
}

/* popup the dialog shaped submenu */
static void submenu_popup(GtkWidget *menu) {
  gtk_widget_show_all(menu);
  gtk_dialog_run(GTK_DIALOG(menu));
  gtk_widget_hide(menu);
}

static void submenu_cleanup(GtkWidget *menu) {
  gtk_widget_destroy(menu);
}

static const menu_entry_t submenu_export_entries[] = {
  { "Export to Maemo Mapper" , "Save a Maemo Mapper POI file",
    G_CALLBACK(cb_menu_export_mmpoi)  },
  { "Export Field Notes",      "Save a Garmin Field Notes file",
    G_CALLBACK(cb_menu_export_log)    },
  { "Export Garmin GPX",       "Save modified waypoints in GPX file",
    G_CALLBACK(cb_menu_export_garmin) },
  { NULL, NULL, NULL }
};

static const submenu_t submenu_export = {
  "Export", submenu_export_entries, 
  sizeof(submenu_export_entries)/sizeof(menu_entry_t)-1
};

/* the export submenu */
void on_export_clicked(GtkButton *button, appdata_t *appdata) {
  if(!appdata->export_menu)
    appdata->export_menu = app_submenu_create(appdata, &submenu_export);

  submenu_popup(appdata->export_menu);
}

static const menu_entry_t submenu_tools_entries[] = {
  { "Geomath",          "Geocoordinate calculation", 
    G_CALLBACK(cb_menu_geomath) },
  { "Geotext",          "Text analysis",             
    G_CALLBACK(cb_menu_geotext) },
  { "Precise Position", "Calculate a precise GPS position",  
    G_CALLBACK(cb_menu_precpos) },
  { NULL, NULL, NULL }
};

static const submenu_t submenu_tools = {
  "Tools", submenu_tools_entries, 
  sizeof(submenu_tools_entries)/sizeof(menu_entry_t)-1
};

/* the tools submenu */
void on_tools_clicked(GtkButton *button, appdata_t *appdata) {
  if(!appdata->tools_menu)
    appdata->tools_menu = app_submenu_create(appdata, &submenu_tools);

  submenu_popup(appdata->tools_menu);
}

HildonAppMenu *menu_create(appdata_t *appdata, int mode) { 
  GtkWidget *button;
  HildonAppMenu *menu = HILDON_APP_MENU(hildon_app_menu_new());

  /* ------- */
  button = gtk_button_new_with_label(_("About"));
  g_signal_connect_after(button, "clicked", 
			 G_CALLBACK(cb_menu_about), appdata);
  hildon_app_menu_append(menu, GTK_BUTTON(button));

  button = gtk_button_new_with_label(_("Settings"));
  g_signal_connect_after(button, "clicked", G_CALLBACK(cb_menu_settings), 
			 appdata);
  hildon_app_menu_append(menu, GTK_BUTTON(button));

  if(mode == MENU_GPXLIST) {
    button = gtk_button_new_with_label(_("Import file"));
    g_signal_connect_after(button, "clicked", 
			   G_CALLBACK(cb_menu_add), appdata);
    hildon_app_menu_append(menu, GTK_BUTTON(button));

    button = gtk_button_new_with_label(_("Import directory"));
    g_signal_connect_after(button, "clicked", 
			   G_CALLBACK(cb_menu_adddir), appdata);
    hildon_app_menu_append(menu, GTK_BUTTON(button));

    button = gtk_button_new_with_label(_("Export"));
    g_signal_connect_after(button, "clicked", 
			   G_CALLBACK(on_export_clicked), appdata);
    hildon_app_menu_append(menu, GTK_BUTTON(button));

    button = gtk_button_new_with_label(_("Search"));
    g_signal_connect_after(button, "clicked", 
			   G_CALLBACK(cb_menu_search), appdata);
    hildon_app_menu_append(menu, GTK_BUTTON(button));
  }

  button = gtk_button_new_with_label(_("Tools"));
  g_signal_connect_after(button, "clicked", 
			 G_CALLBACK(on_tools_clicked), appdata);
  hildon_app_menu_append(menu, GTK_BUTTON(button));

#ifdef ENABLE_OSM_GPS_MAP
  if((mode == MENU_GPXLIST) || (mode == MENU_CACHELIST)) {
    button = gtk_button_new_with_label(_("Map"));
    g_signal_connect_after(button, "clicked", 
			   G_CALLBACK(cb_menu_map), appdata);
    hildon_app_menu_append(menu, GTK_BUTTON(button));
  }
#endif

#ifdef HILDON_HELP
  button = gtk_button_new_with_label(_("Help"));
  g_signal_connect_after(button, "clicked", 
			 G_CALLBACK(cb_menu_help), appdata);
  hildon_app_menu_append(menu, GTK_BUTTON(button));
#endif

  gtk_widget_show_all(GTK_WIDGET(menu));

  return menu;
}
#else

void menu_create(appdata_t *appdata) { 
  GtkWidget *menu, *item;  
  menu = gtk_menu_new();

#ifdef USE_BREAD_CRUMB_TRAIL
  appdata->menu_import = 
#endif
  item = gtk_menu_item_new_with_label(_("Import"));
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  GtkWidget *submenu = gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);

  item = gtk_menu_item_new_with_label( _("File") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_add), appdata);

  item = gtk_menu_item_new_with_label( _("Directory") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_adddir), appdata);

#ifndef USE_PANNABLE_AREA
  gtk_menu_append(GTK_MENU_SHELL(submenu), gtk_separator_menu_item_new());

  appdata->menu_close = 
    item = gtk_menu_item_new_with_label( _("Close") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_close), appdata);

  appdata->menu_remove = 
    item = gtk_menu_item_new_with_label( _("Remove") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_remove), appdata);
#endif

#ifdef USE_BREAD_CRUMB_TRAIL
  appdata->menu_export = 
#endif
    item = gtk_menu_item_new_with_label(_("Export"));
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  submenu = gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);

#ifdef USE_MAEMO
  item = gtk_menu_item_new_with_label( _("Maemo Mapper POI") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", 
		   GTK_SIGNAL_FUNC(cb_menu_export_mmpoi), appdata);
#endif

  item = gtk_menu_item_new_with_label( _("Garmin Field Notes") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", 
		   GTK_SIGNAL_FUNC(cb_menu_export_log), appdata);

  item = gtk_menu_item_new_with_label( _("Garmin GPX") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", 
		   GTK_SIGNAL_FUNC(cb_menu_export_garmin), appdata);

#ifdef USE_BREAD_CRUMB_TRAIL
  appdata->menu_search = 
#endif
  item = gtk_menu_item_new_with_label( _("Search") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_search), appdata);

  gtk_menu_append(GTK_MENU_SHELL(menu), gtk_separator_menu_item_new());
 
#ifndef NO_COPY_N_PASTE
  /* ----------- copy'n paste submenu ----------------- */
  appdata->clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
  gtk_clipboard_set_can_store(appdata->clipboard, NULL, 0);

  item = gtk_menu_item_new_with_label(_("Edit"));
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  submenu = gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);

  appdata->menu_cut = item = gtk_menu_item_new_with_label( _("Cut") );
  gtk_widget_set_sensitive(item, FALSE);
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_cut), appdata);
  appdata->menu_copy = item = gtk_menu_item_new_with_label( _("Copy") );
  gtk_widget_set_sensitive(item, FALSE);
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_copy), appdata);
  appdata->menu_paste = item = gtk_menu_item_new_with_label( _("Paste") );
  gtk_widget_set_sensitive(item, FALSE);
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_paste), appdata);
#endif

  item = gtk_menu_item_new_with_label( _("Settings") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_settings), 
		   appdata);

  gtk_menu_append(GTK_MENU_SHELL(menu), gtk_separator_menu_item_new());

#ifdef ENABLE_OSM_GPS_MAP
  item = gtk_menu_item_new_with_label( _("Map") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", 
		   GTK_SIGNAL_FUNC(cb_menu_map), appdata);
#endif

  item = gtk_menu_item_new_with_label(_("Tools"));
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  submenu = gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);

  item = gtk_menu_item_new_with_label( _("Geomath") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", 
		   GTK_SIGNAL_FUNC(cb_menu_geomath), appdata);

  item = gtk_menu_item_new_with_label( _("Geotext") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", 
		   GTK_SIGNAL_FUNC(cb_menu_geotext), appdata);

  item = gtk_menu_item_new_with_label( _("Precise Position") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", 
		   GTK_SIGNAL_FUNC(cb_menu_precpos), appdata);

  gtk_menu_append(GTK_MENU_SHELL(menu), gtk_separator_menu_item_new());

#if defined(USE_MAEMO) && defined(HILDON_HELP)
  item = gtk_menu_item_new_with_label( _("Help") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_help), appdata);
#endif

  item = gtk_menu_item_new_with_label( _("About") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_about), appdata);

#ifndef USE_MAEMO
  item = gtk_menu_item_new_with_label( _("Quit") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_quit), appdata);
#endif

#ifdef USE_MAEMO
  hildon_window_set_menu(appdata->window, GTK_MENU(menu));
#else
  /* attach ordinary gtk menu */
  GtkWidget *menu_bar = gtk_menu_bar_new();

  GtkWidget *root_menu = gtk_menu_item_new_with_label (_("Menu"));
  gtk_widget_show(root_menu);

  gtk_menu_bar_append(menu_bar, root_menu); 
  gtk_menu_item_set_submenu(GTK_MENU_ITEM (root_menu), menu);

  gtk_widget_show(menu_bar);
  gtk_box_pack_start(GTK_BOX(appdata->vbox), menu_bar, 0, 0, 0);
#endif
}
#endif

/********************* end of menu **********************/

void cleanup(appdata_t *appdata) {
  gpx_free_all(appdata->gpx);
  if(appdata->path) free(appdata->path);
  if(appdata->image_path) free(appdata->image_path);
  if(appdata->search_str) free(appdata->search_str);

#ifdef USE_STACKABLE_WINDOW
  if(appdata->export_menu) submenu_cleanup(appdata->export_menu);
  if(appdata->tools_menu)  submenu_cleanup(appdata->tools_menu);
#endif

  gnome_vfs_shutdown();
  icons_free();
  gps_release(appdata);

#ifdef USE_MAEMO
  if(appdata->search_results) {
    printf("freeing pending search\n");
    search_result_free(appdata->search_results);
  }

  if(appdata->osso_context)
    osso_deinitialize(appdata->osso_context);

  appdata->program = NULL;
#endif

  /* free chain of locations */
  location_t *loc = appdata->location;
  while(loc) {
    location_t *next = loc->next;
    if(loc->name) free(loc->name);
    free(loc);
    loc = next;
  }

  puts("everything is gone");
}

static void on_window_destroy (GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  gconf_save_state(appdata);
  gtk_main_quit();
  appdata->window = NULL;
}

gboolean on_window_key_press(GtkWidget *widget, 
			 GdkEventKey *event, appdata_t *appdata) {
  int handled = FALSE;

  //  printf("key event %d\n", event->keyval);

  switch(event->keyval) {
#ifdef USE_MAEMO

#ifdef HILDON_HARDKEY_INCREASE
  case HILDON_HARDKEY_INCREASE:
    html_zoom(appdata, TRUE);
    handled = TRUE;
    break;
#endif
    
#ifdef HILDON_HARDKEY_DECREASE
  case HILDON_HARDKEY_DECREASE:
    html_zoom(appdata, FALSE);
    handled = TRUE;
    break;
#endif

#ifdef HILDON_HARDKEY_FULLSCREEN
  case HILDON_HARDKEY_FULLSCREEN:
    {
      appdata->fullscreen = !appdata->fullscreen;

      if(appdata->fullscreen)
	gtk_window_fullscreen(GTK_WINDOW(appdata->window));
      else
	gtk_window_unfullscreen(GTK_WINDOW(appdata->window));

      handled = TRUE;
    }
    break;
#endif

#else
  case '+':
    printf("zoom+\n");
    html_zoom(appdata, TRUE);
    handled = TRUE;
    break;
  case '-':
    printf("zoom-\n");
    html_zoom(appdata, FALSE);
    handled = TRUE;
    break;
#endif
  }
  
  return handled;
}

#ifdef USE_BREAD_CRUMB_TRAIL
typedef struct {
  int level;
  appdata_t *appdata;
  gpointer data;
} crumb_t;

static void
crumb_back(gpointer data) {
  crumb_t *crumb = (crumb_t*)data;
  printf("crumb_back called with %d\n", crumb->level);

  /* don't do anything if main window has already been destroyed */
  if(!crumb->appdata->window) {
    printf("Main window gone ...\n");
    return;
  }

  /* whatever is being displayed: we don't need it anymore */
  gtk_container_remove(GTK_CONTAINER(crumb->appdata->vbox), 
		       crumb->appdata->cur_view);

  /* back from cache to cachelist */
  if(crumb->level == CRUMB_CACHE) {
    gpx_t *gpx = crumb->appdata->search_results;

    if(!gpx) {
      gtk_widget_set_sensitive(crumb->appdata->menu_search,  TRUE);
      gtk_widget_set_sensitive(crumb->appdata->menu_export,  TRUE);
      printf("no search data found, return to gpx\n");
      gpx = crumb->appdata->cur_gpx;
    } else
      printf("returning to search result\n");
      
    g_assert(gpx != NULL);

    crumb->appdata->cur_view = cachelist_create(crumb->appdata, gpx,
						crumb->appdata->cur_cache);

    /* returning from cache view: invalidate cache reference */
    crumb->appdata->cur_cache = NULL;

    gtk_box_pack_start_defaults(GTK_BOX(crumb->appdata->vbox), 
				crumb->appdata->cur_view);
  }

  if(crumb->level == CRUMB_SEARCH_GPX) {
    printf("returning from a local search!\n");

    g_assert((gpx_t*)crumb->data == crumb->appdata->search_results);

    search_result_free((gpx_t*)crumb->data);
    crumb->appdata->search_results = NULL;

    gtk_widget_set_sensitive(crumb->appdata->menu_search,  TRUE);

    crumb->appdata->cur_view = cachelist_create(crumb->appdata, 
					crumb->appdata->cur_gpx, NULL);
    gtk_box_pack_start_defaults(GTK_BOX(crumb->appdata->vbox), 
				crumb->appdata->cur_view);
  }

  /* back from cachelist to gpxlist */
  if((crumb->level == CRUMB_CACHELIST) || 
     (crumb->level == CRUMB_SEARCH_GLOBAL)) {

    crumb->appdata->cur_view = gpxlist_create_view_and_model(
			      crumb->appdata, crumb->appdata->cur_gpx);

    /* returning from cachelist/global search view: */
    /* invalidate gpx reference */
    crumb->appdata->cur_gpx = NULL;

    gtk_box_pack_start_defaults(GTK_BOX(crumb->appdata->vbox), 
				crumb->appdata->cur_view);

    if((crumb->level == CRUMB_SEARCH_GLOBAL) ||
       (crumb->level == CRUMB_SEARCH_GPX)) {
      g_assert((gpx_t*)crumb->data == crumb->appdata->search_results);

      search_result_free((gpx_t*)crumb->data);
      crumb->appdata->search_results = NULL;
    }

    /* enable gpxlist related menu entries */
    gtk_widget_set_sensitive(crumb->appdata->menu_import,  TRUE);
    gtk_widget_set_sensitive(crumb->appdata->menu_search,  TRUE);
    gtk_widget_set_sensitive(crumb->appdata->menu_export,  TRUE);
  }

  gtk_widget_show_all(crumb->appdata->vbox);    
  g_free(data);
}

static void crumb_add(appdata_t *appdata, char *name, int level, 
		      gpointer user_data) {
  crumb_t *crumb = malloc(sizeof(crumb_t));
  crumb->level = level;
  crumb->appdata = appdata;
  crumb->data = user_data;

  printf("crumb_add with level %d\n", level);

  /* save that we are working on search results */
  if((level == CRUMB_SEARCH_GLOBAL) ||
     (level == CRUMB_SEARCH_GPX)) {
    appdata->search_results = (gpx_t*)user_data;

    /* searches cannot be nested */
    gtk_widget_set_sensitive(appdata->menu_search,  FALSE);
  }

  /* save "path" pointers in appdata */
  if(crumb->level == CRUMB_CACHELIST) 
    appdata->cur_gpx = (gpx_t*)user_data;

  if(crumb->level == CRUMB_CACHE) {
    appdata->cur_cache = (cache_t*)user_data;
    /* the cache view cannot be searched */
    gtk_widget_set_sensitive(appdata->menu_search,  FALSE);
    gtk_widget_set_sensitive(appdata->menu_export,  FALSE);
  }

  if(level != CRUMB_GPXLIST) {
    /* disable gpxlist related menu entries */
    gtk_widget_set_sensitive(appdata->menu_import,  FALSE);
#ifndef USE_PANNABLE_AREA
    gtk_widget_set_sensitive(appdata->menu_remove,  FALSE);
    gtk_widget_set_sensitive(appdata->menu_close,   FALSE);
#endif
  }

  hildon_bread_crumb_trail_push_text(HILDON_BREAD_CRUMB_TRAIL(appdata->bct),
				     name, crumb, (GDestroyNotify)crumb_back);
}
#endif // USE_BREAD_CRUMB_TRAIL

void main_after_settings_redraw(appdata_t *appdata, int flags) {
  printf("main after settings redraw\n");

  if(!appdata->cur_view) {
    printf("no active view\n");
    return;
  }

#ifndef USE_MAEMO
  // in non-maemo setup this can only affect the main screen as
  // the menu is blocked while a dialog is open. also the main
  // screen is always present
  if(appdata->gpxlist_items != appdata->cur_items) {
    /* re-do the main screen */
    gtk_container_remove(GTK_CONTAINER(appdata->vbox), appdata->cur_view);
    appdata->cur_view = gpxlist_create_view_and_model(appdata, NULL);
    gtk_box_pack_start_defaults(GTK_BOX(appdata->vbox), appdata->cur_view);
    gtk_widget_show_all(appdata->vbox);
  }
#else
  /* a cache screen cannot be changed from the settings and thus doesn't */
  /* need to be redrawn */
  if(appdata->cur_cache) {
    printf("No redraw in cache view required\n");
    return;
  }

  int redraw = 0;    // nothing to redraw

  if(appdata->search_results) {
    if((appdata->cur_items != appdata->cachelist_items) || flags)
      redraw = 1;
  } else {
    if(!appdata->cur_gpx) {
      if(appdata->cur_items != appdata->gpxlist_items) 
	redraw = 2;  // redraw gpxlist
    } else {
      if((appdata->cur_items != appdata->cachelist_items) || flags)
	redraw = 3;  // redraw cachelist
    }
  }

  if(redraw) {
    GtkWidget *container = appdata->vbox;

#ifdef USE_STACKABLE_WINDOW
    HildonWindowStack *stack = hildon_window_stack_get_default();
    container = hildon_window_stack_peek(stack);
#endif

    gtk_container_remove(GTK_CONTAINER(container), appdata->cur_view);
    switch(redraw) {
    case 1:
      appdata->cur_view = cachelist_create(appdata, 
					   appdata->search_results, NULL);
      break;
    case 2:
      appdata->cur_view = gpxlist_create_view_and_model(appdata, NULL);
      break;
    case 3:
      appdata->cur_view = cachelist_create(appdata, 
					   appdata->cur_gpx, NULL);
      break;
    }

#ifdef USE_STACKABLE_WINDOW
    if(container != appdata->vbox)
      gtk_container_add(GTK_CONTAINER(container), appdata->cur_view); 
    else
#endif
      gtk_box_pack_start_defaults(GTK_BOX(container), appdata->cur_view);
    
    gtk_widget_show_all(container);
  }
#endif // USE_MAEMO
}

int main(int argc, char *argv[]) {
  appdata_t appdata;

  /* init appdata */
  memset(&appdata, 0, sizeof(appdata));

  printf("Using locale for %s in %s\n", PACKAGE, LOCALEDIR);

  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE, LOCALEDIR);
  bind_textdomain_codeset(PACKAGE, "UTF-8");
  textdomain(PACKAGE);

  /* prepare thread system */
  g_thread_init(NULL);

  gtk_init (&argc, &argv);

#ifdef USE_MAEMO
  printf("Installing osso context for \"org.harbaum." APP "\"\n");
  appdata.osso_context = osso_initialize("org.harbaum."APP, 
					 VERSION, TRUE, NULL);
  if(appdata.osso_context == NULL) {
    fprintf(stderr, "error initiating osso context\n");
  }

  dbus_register(&appdata);
#endif

  icons_init();

  if(!gnome_vfs_init()) {
    g_error("Gnome VFS init failed\n");
  }

#ifdef USE_MAEMO
  /* Create the hildon program and setup the title */
  appdata.program = HILDON_PROGRAM(hildon_program_get_instance());
  g_set_application_name("GPXView");
  
  /* Create HildonWindow and set it to HildonProgram */
#ifdef USE_STACKABLE_WINDOW
  appdata.window = HILDON_WINDOW(hildon_stackable_window_new());
#else
  appdata.window = HILDON_WINDOW(hildon_window_new());
#endif
  hildon_program_add_window(appdata.program, appdata.window);
#else
  /* Create a Window. */
  appdata.window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  /* Set a decent default size for the window. */
  gtk_window_set_default_size(GTK_WINDOW(appdata.window), 500, 300);
#endif

#if MAEMO_VERSION_MAJOR == 5
  gtk_window_set_title(GTK_WINDOW(appdata.window), "GPXView");
#endif

  g_signal_connect(G_OBJECT(appdata.window), "destroy", 
		   G_CALLBACK(on_window_destroy), &appdata);

  g_signal_connect(G_OBJECT(appdata.window), "key_press_event",
		   G_CALLBACK(on_window_key_press), &appdata);

  appdata.vbox = gtk_vbox_new(FALSE, 2);
  gtk_container_add(GTK_CONTAINER(appdata.window), appdata.vbox);
#ifndef USE_STACKABLE_WINDOW
  menu_create(&appdata);
#else
  hildon_window_set_app_menu(HILDON_WINDOW(appdata.window), 
			     menu_create(&appdata, MENU_GPXLIST));
#endif

#ifdef USE_BREAD_CRUMB_TRAIL
  appdata.bct = hildon_bread_crumb_trail_new();

  gtk_box_pack_start(GTK_BOX(appdata.vbox), appdata.bct, FALSE,FALSE,0);

  hildon_bread_crumb_trail_clear(HILDON_BREAD_CRUMB_TRAIL(appdata.bct));
  crumb_add(&appdata, "GPX", CRUMB_GPXLIST, NULL);
#endif

  /* wait for main gui to appear */
  gtk_widget_show_all(GTK_WIDGET(appdata.window));
  while(gtk_events_pending()) 
    gtk_main_iteration();

  appdata.gconf_client = gconf_client_get_default();
  gconf_load_state(&appdata);
  gps_init(&appdata);

  appdata.cur_view = gpxlist_create_view_and_model(&appdata, NULL);
  gtk_box_pack_start_defaults(GTK_BOX(appdata.vbox), appdata.cur_view);

  gtk_widget_show_all(GTK_WIDGET(appdata.window));
  gtk_main();

  cleanup(&appdata);

  return 0;
}
