/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stddef.h>
#include <stdlib.h>
#include <ctype.h>
#include <math.h>     // for isnan
#include "gpxview.h"

#define GCONF_PATH         "/apps/gpxview/"
#define GCONF_KEY_GPX      GCONF_PATH "gpx%d"
#define GCONF_KEY_CNT      GCONF_PATH "entries"

#define GCONF_KEY_LOC_NAME GCONF_PATH "location%d/name"
#define GCONF_KEY_LOC_LAT  GCONF_PATH "location%d/latitude"
#define GCONF_KEY_LOC_LON  GCONF_PATH "location%d/longitude"
#define GCONF_KEY_LOC_CNT  GCONF_PATH "location_entries"

#define GCONF_KEY_CLOSED   GCONF_PATH "closed/%s"

#include <string.h>

enum {
  STORE_STRING, STORE_FLOAT, STORE_INT, STORE_BOOL,
};

typedef struct {
  char *key;
  int type;
  int offset;
} store_t;

#define OFFSET(a) offsetof(appdata_t, a)

static store_t store[] = {
  { "image_path",       STORE_STRING, OFFSET(image_path) },
  { "path",             STORE_STRING, OFFSET(path) },
  { "geotext/text",     STORE_STRING, OFFSET(geotext_text) },
  { "geotext/shift",    STORE_INT,    OFFSET(geotext_shift) },
  { "mmpoi_path",       STORE_STRING, OFFSET(mmpoi_path) },
  { "garmin_path",      STORE_STRING, OFFSET(garmin_path) },
  { "fnotes_path",      STORE_STRING, OFFSET(fieldnotes_path) },
  { "garmin_ign_found", STORE_BOOL,   OFFSET(garmin_ign_found) },
  { "active_location",  STORE_INT,    OFFSET(active_location) },
  { "mmpoi_use_radius", STORE_BOOL,   OFFSET(mmpoi_use_radius) },
  { "mmpoi_radius",     STORE_FLOAT,  OFFSET(mmpoi_radius) },
  { "mmpoi_ign_found",  STORE_BOOL,   OFFSET(mmpoi_dont_export_found) },
  { "mmpoi_ign_disabl", STORE_BOOL,   OFFSET(mmpoi_dont_export_disabled) },
  { "use_gps",          STORE_BOOL,   OFFSET(use_gps) },
  { "imperial",         STORE_BOOL,   OFFSET(imperial) },
  { "compass_locked",   STORE_BOOL,   OFFSET(compass_locked) },
  { "latitude",         STORE_FLOAT,  OFFSET(home.lat) },
  { "longitude",        STORE_FLOAT,  OFFSET(home.lon) },
  { "manual_goto_lat",  STORE_FLOAT,  OFFSET(manual_goto.lat) },
  { "manual_goto_lon",  STORE_FLOAT,  OFFSET(manual_goto.lon) },
  { "gps_lat",          STORE_FLOAT,  OFFSET(gps.lat) },
  { "gps_lon",          STORE_FLOAT,  OFFSET(gps.lon) },
  { "search_in",        STORE_INT,    OFFSET(search) },
  { "search_days",      STORE_INT,    OFFSET(search_days) },
  { "search_str",       STORE_STRING, OFFSET(search_str) },
  { "gpxlist_items",    STORE_INT,    OFFSET(gpxlist_items) },
  { "cachelist_items",  STORE_INT,    OFFSET(cachelist_items) },
  { "compass_damping",  STORE_INT,    OFFSET(compass_damping) },
  { "cachelist_hide_found", STORE_BOOL, OFFSET(cachelist_hide_found) },
#ifdef USE_MAEMO
  { "mmpoi_dontlaunch", STORE_BOOL,   OFFSET(mmpoi_dontlaunch) },
  { "cachelist_dss",    STORE_BOOL,   OFFSET(cachelist_disable_screensaver) },
  { "goto_dss",         STORE_BOOL,   OFFSET(goto_disable_screensaver) },
  { "cachelist_update", STORE_BOOL,   OFFSET(cachelist_update) },
#endif
#ifdef ENABLE_OSM_GPS_MAP
  { "map_lat",          STORE_FLOAT,  OFFSET(map.pos.lat) },
  { "map_lon",          STORE_FLOAT,  OFFSET(map.pos.lon) },
  { "map_zoom",         STORE_INT,    OFFSET(map.zoom) },
#endif
  { NULL, -1, -1 }
};

static char *get_basename(char *name) {
  char *p = strrchr(name, '/');
  if(!p) p = name;
  else   p = p+1;

  g_assert(*p);

  /* escape all non alnum characters */
  p = g_strdup(p);
  int i;
  for(i=0;i<strlen(p);i++)
    if(!isalnum(p[i]))
      p[i] = '_';

  return p;
}

void gconf_remove_closed_name(appdata_t *appdata, char *filename) {
  char *key = g_strdup_printf(GCONF_KEY_CLOSED, get_basename(filename));
  gconf_client_unset(appdata->gconf_client, key, NULL);
  g_free(key);
}

void gconf_save_closed_name(appdata_t *appdata, char *filename, char *name) {
  char *key = g_strdup_printf(GCONF_KEY_CLOSED, get_basename(filename));
  gconf_client_set_string(appdata->gconf_client, key, name, NULL);
  g_free(key);
}

char *gconf_restore_closed_name(appdata_t *appdata, char *filename) {
  char *key = g_strdup_printf(GCONF_KEY_CLOSED, get_basename(filename));
  char *ret = gconf_client_get_string(appdata->gconf_client, key, NULL);
  g_free(key);
  return ret;
}

void gconf_save_state(appdata_t *appdata) {
  int entries = 0;

  gpx_t *gpx = appdata->gpx;
  while(gpx) {
    char str[128];
    snprintf(str, sizeof(str), GCONF_KEY_GPX, entries++);
    gconf_client_set_string(appdata->gconf_client, str, gpx->filename, NULL);
    gpx = gpx->next;
  }

  gconf_client_set_int(appdata->gconf_client, GCONF_KEY_CNT, entries, NULL);

  /* -------------- save locations (excl. home location) --------------- */
  entries = 0;
  location_t *loc = appdata->location;
  while(loc) {
    char str[128];
    snprintf(str, sizeof(str), GCONF_KEY_LOC_NAME, entries);
    gconf_client_set_string(appdata->gconf_client, str, loc->name, NULL);
    snprintf(str, sizeof(str), GCONF_KEY_LOC_LAT, entries);
    gconf_client_set_float(appdata->gconf_client, str, loc->pos.lat, NULL);
    snprintf(str, sizeof(str), GCONF_KEY_LOC_LON, entries);
    gconf_client_set_float(appdata->gconf_client, str, loc->pos.lon, NULL);
    entries++;
    loc = loc->next;
  }

  gconf_client_set_int(appdata->gconf_client, GCONF_KEY_LOC_CNT, entries, NULL);

  /* store everything listed in the store table */
  store_t *st = store;
  while(st->key) {
    void **ptr = ((void*)appdata) + st->offset;
    char *key = g_strdup_printf(GCONF_PATH "%s", st->key);

    switch(st->type) {
    case STORE_STRING: 
      if((char*)(*ptr)) {
	gconf_client_set_string(appdata->gconf_client, key, (char*)(*ptr), NULL);
      }
      break;

    case STORE_BOOL: 
      gconf_client_set_bool(appdata->gconf_client, key, *((int*)ptr), NULL);
      break;

    case STORE_INT: 
      gconf_client_set_int(appdata->gconf_client, key, *((int*)ptr), NULL);
      break;

    case STORE_FLOAT: 
      if(!isnan(*((float*)ptr)))
	gconf_client_set_float(appdata->gconf_client, key, *((float*)ptr), NULL);
      break;

    default:
      printf("Unsupported type %d\n", st->type);
      break;
    }

    g_free(key);
    st++;
  }
}

void gconf_load_state(appdata_t *appdata) {
  gpx_t **gpx = &appdata->gpx;

  while(*gpx) gpx = &((*gpx)->next);

  gpx_dialog_t *dialog = NULL;

  /* default positions are invalid */
  appdata->home.lat = appdata->home.lon = NAN;
  appdata->manual_goto.lat = appdata->manual_goto.lon = NAN;
  appdata->gps.lat = appdata->gps.lon = NAN;

  int i, entries = gconf_client_get_int(appdata->gconf_client, 
				 GCONF_KEY_CNT, NULL);

  if(entries)
    dialog = gpx_busy_dialog_new(GTK_WIDGET(appdata->window));

  for(i=0;i<entries;i++) {
    char str[128];
    snprintf(str, sizeof(str), GCONF_KEY_GPX, i);
    char *fname = gconf_client_get_string(appdata->gconf_client, str, NULL);

    if(fname) {
      /* check if there's a valid name stored for this file. */
      /* if yes it's a "closed" file */
      char *name = gconf_restore_closed_name(appdata, fname);
      if(name) {
	*gpx = g_new0(gpx_t, 1);
	(*gpx)->filename = fname;
	(*gpx)->name = g_strdup(name);
	(*gpx)->closed = TRUE;
      } else {
	if(g_file_test(fname, G_FILE_TEST_IS_DIR)) 
	  *gpx = gpx_parse_dir(dialog, fname);
	else
	  *gpx = gpx_parse(dialog, fname);

	free(fname);
      }
    }

    /* use next gpx entry of this was loaded successfully */
    if(*gpx)
      gpx = &((*gpx)->next);
  }

  gpx_busy_dialog_destroy(dialog);

  /* ------------- load locations --------------------- */
  entries = gconf_client_get_int(appdata->gconf_client, 
				 GCONF_KEY_LOC_CNT, NULL);

  location_t **loc = &appdata->location;
  for(i=0;i<entries;i++) {
    *loc = g_new0(location_t, 1);
    if(*loc) {
      char str[128];
      snprintf(str, sizeof(str), GCONF_KEY_LOC_NAME, i);
      (*loc)->name = gconf_client_get_string(appdata->gconf_client, str, NULL);
      snprintf(str, sizeof(str), GCONF_KEY_LOC_LAT, i);
      (*loc)->pos.lat  = gconf_client_get_float(appdata->gconf_client, str, NULL);
      snprintf(str, sizeof(str), GCONF_KEY_LOC_LON, i);
      (*loc)->pos.lon  = gconf_client_get_float(appdata->gconf_client, str, NULL);
      
      loc = &((*loc)->next);
    }
  }

  /* restore everything listed in the store table */
  store_t *st = store;
  while(st->key) {
    void **ptr = ((void*)appdata) + st->offset;
    char *key = g_strdup_printf(GCONF_PATH "%s", st->key);

    /* check if key is present */
    GConfValue *value = gconf_client_get(appdata->gconf_client, key, NULL);
    if(value) {
      gconf_value_free(value); 
      
      switch(st->type) {
      case STORE_STRING: {
	char **str = (char**)ptr;
	*str = gconf_client_get_string(appdata->gconf_client, key, NULL);
      } break;
	
      case STORE_BOOL: 
	*((int*)ptr) = gconf_client_get_bool(appdata->gconf_client, key, NULL);
	break;
	
      case STORE_INT: 
	*((int*)ptr) = gconf_client_get_int(appdata->gconf_client, key, NULL);
	break;
	
      case STORE_FLOAT: 
	*((float*)ptr) = gconf_client_get_float(appdata->gconf_client, key, NULL);
	break;
	
      default:
	printf("Unsupported type %d\n", st->type);
	break;
      }
    }
    g_free(key);
    st++;
  }

  /* ----- set all kinds of defaults ------- */

  if(!appdata->compass_damping) appdata->compass_damping = 1;
  
  if(!appdata->mmpoi_radius) 
    appdata->mmpoi_radius = 100.0;  // 100 km

  if(!appdata->search)
    appdata->search = SEARCH_NAME | SEARCH_ID;

  if(!appdata->image_path) {

    /* use gps by default */
    appdata->use_gps = TRUE;

#ifndef USE_MAEMO
    char *p = getenv("HOME");
    if(p) {
      /* build image path in home directory */
      appdata->image_path =
	malloc(strlen(p)+strlen(DEFAULT_IMAGE_PATH_HOME)+2);
      strcpy(appdata->image_path, p);
      if(appdata->image_path[strlen(appdata->image_path)-1] != '/')
	strcat(appdata->image_path, "/");
      strcat(appdata->image_path, DEFAULT_IMAGE_PATH_HOME);
    } else
#endif
    appdata->image_path = strdup(DEFAULT_IMAGE_PATH);

  } else {
    /* some versions old versions messed up the path */
    if(appdata->image_path[strlen(appdata->image_path)-1] != '/') {
      printf("adjusting image path\n");
      appdata->image_path = realloc(appdata->image_path, 
				    strlen(appdata->image_path)+2);
      strcat(appdata->image_path, "/");
    }
  }

  if(!appdata->mmpoi_path) {
    char *p = getenv("HOME");
    if(p) {
      /* build mmpoi path in home directory */
      appdata->mmpoi_path = 
	malloc(strlen(p)+strlen(DEFAULT_MMPOI_PATH)+2);
      strcpy(appdata->mmpoi_path, p);
      if(appdata->mmpoi_path[strlen(appdata->mmpoi_path)-1] != '/')
	strcat(appdata->mmpoi_path, "/");
      strcat(appdata->mmpoi_path, DEFAULT_MMPOI_PATH);
    } else
      appdata->mmpoi_path = strdup(DEFAULT_MMPOI_PATH);
  }

  if(!appdata->fieldnotes_path) {
    char *p = getenv("HOME");
    if(p) {
      /* build fieldnotes path in home directory */
      appdata->fieldnotes_path = 
	malloc(strlen(p)+strlen(DEFAULT_FIELDNOTES_PATH)+2);
      strcpy(appdata->fieldnotes_path, p);
      if(appdata->fieldnotes_path[strlen(appdata->fieldnotes_path)-1] != '/')
	strcat(appdata->fieldnotes_path, "/");
      strcat(appdata->fieldnotes_path, DEFAULT_FIELDNOTES_PATH);
    } else
      appdata->fieldnotes_path = strdup(DEFAULT_FIELDNOTES_PATH);
  }

  if(!appdata->garmin_path) {
    char *p = getenv("HOME");
    if(p) {
      /* build image path in home directory */
      appdata->garmin_path = 
	malloc(strlen(p)+strlen(DEFAULT_GARMIN_PATH)+2);
      strcpy(appdata->garmin_path, p);
      if(appdata->garmin_path[strlen(appdata->garmin_path)-1] != '/')
	strcat(appdata->garmin_path, "/");
      strcat(appdata->garmin_path, DEFAULT_GARMIN_PATH);
    } else
      appdata->garmin_path = strdup(DEFAULT_GARMIN_PATH);
  }

  /* make sure image path actually exists */
  checkdir(appdata->image_path);

  if(!appdata->gpxlist_items)
    appdata->gpxlist_items = GPXLIST_ITEM_DEFAULT;

  if(!appdata->cachelist_items)
    appdata->cachelist_items = CACHELIST_ITEM_DEFAULT;

  /* if there are no entries in the main list, try to add the */
  /* "welcome" one */
  if(!appdata->gpx) {
    char *name = g_strdup(ICONPATH "welcome.gpx");
    dialog = gpx_busy_dialog_new(GTK_WIDGET(appdata->window));
    printf("No GPX file loaded, trying to load demo\n");
    appdata->gpx = gpx_parse(dialog, name);
    gpx_busy_dialog_destroy(dialog);
    g_free(name);
  }
}

