/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <string.h>

#include <glib/gunicode.h>

#include <libxml/parser.h>
#include <libxml/tree.h>

#include <math.h>

#if !defined(LIBXML_TREE_ENABLED) || !defined(LIBXML_OUTPUT_ENABLED)
#error "libxml doesn't support required tree or output"
#endif

#include "gpxview.h"

void gtk_text_buffer_set_can_paste_rich_text(GtkTextBuffer *buffer, gboolean);
void gtk_text_buffer_set_rich_text_format(GtkTextBuffer *buffer, const gchar *);

#define TAG_STATE  GTK_STATE_PRELIGHT

static notes_t *notes_load(appdata_t *appdata, cache_t *cache) {
  notes_t *notes = NULL;
  xmlDoc *doc = NULL;
  xmlNode *root_element = NULL;

  LIBXML_TEST_VERSION;

  /* build local path */
  int path_len = strlen(appdata->image_path) + 2 * strlen(cache->id) + 6;
  char *path = malloc(path_len);
  snprintf(path, path_len, "%s%s/%s.gpx", 
	   appdata->image_path, cache->id, cache->id);

  /* no such file? */
  if(!g_file_test(path, G_FILE_TEST_EXISTS)) {
    free(path);
    return NULL;
  }

  /* parse the file and get the DOM */
  doc = xmlReadFile(path, NULL, 0);
  
  if(doc == NULL) {
    printf("error: could not parse file %s\n", path);
    free(path);
    return NULL;
  }
  
  /* Get the root element node */
  root_element = xmlDocGetRootElement(doc);

  xmlNode *cur_node = NULL;
  for (cur_node = root_element; cur_node; cur_node = cur_node->next) {
    if (cur_node->type == XML_ELEMENT_NODE) {
      if(strcasecmp((char*)cur_node->name, "gpx") == 0) {
	xmlNode *wpt_node = cur_node->children;
	
	while(wpt_node != NULL) {
	  if(wpt_node->type == XML_ELEMENT_NODE) {
	    if(strcasecmp((char*)wpt_node->name, "wpt") == 0) {
	      
	      notes = malloc(sizeof(notes_t));
	      memset(notes, 0, sizeof(notes_t));
	      notes->pos = gpx_cache_pos(cache);

	      char *str;
	      if((str = (char*)xmlGetProp(wpt_node, BAD_CAST "override"))) {
		notes->override = (strcasecmp(str, "yes") == 0);
		xmlFree(str);
	      }

	      if((str = (char*)xmlGetProp(wpt_node, BAD_CAST "found"))) {
		/* check if there's a valid number -> found time */
		if(strtoul(str, NULL, 10) != 0) {
		  notes->found = TRUE;
		  notes->ftime = strtoul(str, NULL, 10);
		} else {
		  notes->found = (strcasecmp(str, "yes") == 0);
		  notes->ftime = 0;
		}
		xmlFree(str);
	      }

	      if((str = (char*)xmlGetProp(wpt_node, BAD_CAST "logged"))) {
		notes->logged = (strcasecmp(str, "yes") == 0);
		xmlFree(str);
	      }

	      if((str = (char*)xmlGetProp(wpt_node, BAD_CAST "lat"))) {
		notes->pos.lat = g_ascii_strtod(str, NULL);
		xmlFree(str);
	      }

	      if((str = (char*)xmlGetProp(wpt_node, BAD_CAST "lon"))) {
		notes->pos.lon = g_ascii_strtod(str, NULL);
		xmlFree(str);
	      }

	      xmlNode *sub_node = wpt_node->children;
	
	      while (sub_node != NULL) {
		if (sub_node->type == XML_ELEMENT_NODE) {
		  if(strcasecmp((char*)sub_node->name, "desc") == 0) 
		    notes->text = (char*)
		      xmlNodeListGetString(doc, sub_node->children, 1);
		}
		sub_node = sub_node->next;
	      }
	    }
	  }
	  wpt_node = wpt_node->next;
	}
      }
    }
  }

  xmlFreeDoc(doc);
  xmlCleanupParser();

  printf("got notes for cache %s\n", cache->id);

  free(path);
  return notes;
}

void notes_load_all(appdata_t *appdata, gpx_t *gpx) {
  printf("Load all notes for %s\n", gpx->name);

  cache_t *cache = gpx->cache;
  while(cache) {
    /* a note may actually already be there if this gpx file */
    /* is e.g. assembled from search results */
    if(!cache->notes) 
      cache->notes = notes_load(appdata, cache);

    cache = cache->next;
  }
}

static int notes_write_file(cache_context_t *context,
		      char *text, pos_t pos, 
		      gboolean override, gboolean found, 
		      time_t ftime, gboolean logged) {
  /* build local path */
  int path_len = strlen(context->appdata->image_path) +
    2 * strlen(context->cache->id) + 6;
  char *path = malloc(path_len);
  snprintf(path, path_len, "%s%s/%s.gpx",
	   context->appdata->image_path,
	   context->cache->id, context->cache->id);
  
  if(checkdir(path) != 0) {
    printf("unable to create notes path\n");
    free(path);
    return -1;
  }

  LIBXML_TEST_VERSION;
    
  xmlDocPtr doc = xmlNewDoc(BAD_CAST "1.0");
  xmlNodePtr root_node = xmlNewNode(NULL, BAD_CAST "gpx");
  xmlDocSetRootElement(doc, root_node);
  
  xmlNodePtr wpt_node = xmlNewChild(root_node, NULL, BAD_CAST "wpt", NULL);
  /* make sure no invalid position gets saved */
  if(!isnan(pos.lat) && !isnan(pos.lon)) { 
    if(override)
      xmlNewProp(wpt_node, BAD_CAST "override", BAD_CAST "yes");
    else
      xmlNewProp(wpt_node, BAD_CAST "override", BAD_CAST "no");

    if(logged)
      xmlNewProp(wpt_node, BAD_CAST "logged", BAD_CAST "yes");
    else
      xmlNewProp(wpt_node, BAD_CAST "logged", BAD_CAST "no");

    if(found) {
      if(ftime) {
	char str[32];
	snprintf(str, sizeof(str), "%lu", ftime);
	xmlNewProp(wpt_node, BAD_CAST "found", BAD_CAST str);
      } else
	xmlNewProp(wpt_node, BAD_CAST "found", BAD_CAST "yes");
    } else
      xmlNewProp(wpt_node, BAD_CAST "found", BAD_CAST "no");

    char str[32];
    g_ascii_dtostr(str, sizeof(str), pos.lat);
    xmlNewProp(wpt_node, BAD_CAST "lat", BAD_CAST str);
    g_ascii_dtostr(str, sizeof(str), pos.lon);
    xmlNewProp(wpt_node, BAD_CAST "lon", BAD_CAST str);
  }

  int len = strlen(context->cache->id) + strlen(" - ") +
    strlen(context->cache->name) + 1;
  char *name = malloc(len);
  snprintf(name, len, "%s - %s", context->cache->id, context->cache->name);
  xmlNewChild(wpt_node, NULL, BAD_CAST "name", BAD_CAST name);
  free(name);
  xmlNewChild(wpt_node, NULL, BAD_CAST "sym", BAD_CAST "Pin, Blue");
  xmlNewChild(wpt_node, NULL, BAD_CAST "desc", BAD_CAST text);
  
  /* write everything and free it */
  printf("writing %s\n", path);
  xmlSaveFormatFileEnc(path, doc, "UTF-8", 1);
  xmlFreeDoc(doc);
  xmlCleanupParser();
  free(path);

  return 0;
}

static void notes_save(cache_context_t *context) {
  /* only save if: there is a text which has been changed or */
  /* there is a position which differs from the original one */
  /* or has been changed */

  if(context->notes.modified) {
    printf("something has been modified, saving notes\n");

    GtkTextIter start;
    GtkTextIter end;
    gtk_text_buffer_get_start_iter(context->notes.buffer, &start);
    gtk_text_buffer_get_end_iter(context->notes.buffer, &end);
    char *text = gtk_text_buffer_get_text(context->notes.buffer, 
					  &start, &end, FALSE);
    
    pos_t pos;
    pos.lat = lat_get(context->notes.latw);
    pos.lon = lon_get(context->notes.lonw);

    gboolean override = 
      gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(context->notes.overridew));
    gboolean found = 
      gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(context->notes.foundw));
    gboolean logged =
      gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(context->notes.loggedw));

    /* required accuracy is 1/60000 degree */
    if(((int)(pos.lat * 60000 + .5) != 
	(int)(context->cache->pos.lat * 60000 + .5)) ||
       ((int)(pos.lon * 60000 + .5) != 
	(int)(context->cache->pos.lon * 60000 + .5)))
      printf("position is modified %f != %f / %f != %f\n",
	     pos.lat * 60000 + .5, context->cache->pos.lat * 60000 + .5,
	     pos.lon * 60000 + .5, context->cache->pos.lon * 60000 + .5);
    if(override || found)
      printf("flags are set\n");
    if(strlen(text))
      printf("text is present\n");

    /* check if the notes are empty */
    if(((int)(pos.lat * 60000 + .5) == 
	(int)(context->cache->pos.lat * 60000 + .5)) &&
       ((int)(pos.lon * 60000 + .5) == 
	(int)(context->cache->pos.lon * 60000 + .5)) &&
       !override && !found && !logged && (strlen(text) == 0)) {
      printf("notes are in default state, removing them if present\n");

      /* remove note */
      int path_len = strlen(context->appdata->image_path) + 
	2 * strlen(context->cache->id) + 6;
      char *path = malloc(path_len);
      snprintf(path, path_len, "%s%s/%s.gpx",
	       context->appdata->image_path,
	       context->cache->id, context->cache->id);
  
      printf("removing note %s\n", path);
      remove(path);
      free(path);

      /* search for matching caches and replace note there */
      gpx_t *gpx = context->appdata->gpx;
      while(gpx) {
	cache_t *cache = gpx->cache;
	while(cache) {
	  if(strcmp(cache->id, context->cache->id)==0) {
	    if(cache->notes) {
	      notes_free(cache->notes);
	      cache->notes = NULL;
	    }
	  }
	  cache = cache->next;
	}
	gpx = gpx->next;
      }

#ifdef USE_MAEMO
      /* update search results if present */
      if(context->appdata->search_results) {
	printf("Updating search results\n");
	
	/* add note to all matching search results */
	cache_t *cache = context->appdata->search_results->cache;
	while(cache) {
	  if(strcmp(cache->id, context->cache->id)==0) 
	    cache->notes = NULL;

	  cache = cache->next;
	}
      }
#endif


    } else {
      /* we have to do two things here: */
      /* - update the notes.xml file on disk */
      /* - update the notes entry in the loaded gpx tree */
      
      /* update file on disk */
      notes_write_file(context, text, pos, override, found, 
		       context->notes.ftime, logged);
    
      /* search for matching caches and replace note there */
      notes_t *note = NULL;
      gpx_t *gpx = context->appdata->gpx;
      while(gpx) {
	cache_t *cache = gpx->cache;
	while(cache) {
	  if(strcmp(cache->id, context->cache->id)==0) {
	    //	  printf("found %s in %s\n", cache->id, gpx->name);
	    
	    if(cache->notes)
	      notes_free(cache->notes);
	    
	    /* create a new note for this cache */
	    cache->notes = note = malloc(sizeof(notes_t));
	    memset(cache->notes, 0, sizeof(notes_t));
	    cache->notes->text = strdup(text);
	    cache->notes->pos = pos;
	    cache->notes->override = override;
	    cache->notes->found = found;
	    cache->notes->logged = logged;
	    cache->notes->ftime = context->notes.ftime;
	  }
	  cache = cache->next;
	}
	gpx = gpx->next;
      }

#ifdef USE_MAEMO
      /* update search results if present */
      if(context->appdata->search_results) {
	printf("Updating search results\n");
	
	/* add note to all matching search results */
	cache_t *cache = context->appdata->search_results->cache;
	while(cache) {
	  if(strcmp(cache->id, context->cache->id)==0) 
	    cache->notes = note;
	  
	  cache = cache->next;
	}
      }
#endif
    }

    if(text) free(text);
  }
}

/* this is called from the destroy event of the entire notebook */
gint notes_destroy_event(GtkWidget *widget, gpointer data ) {
  cache_context_t *context = (cache_context_t*)data;

  printf("about to destroy notes view\n");
  notes_save(context);

  return FALSE;
}

#ifndef NO_COPY_N_PASTE
static void on_destroy_textview(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  printf("destroying notes textview\n");

  /* only do this if main windows hasn't already been destroyed */
  if(!appdata->window) {
    printf("destroy notes textview: main window is gone\n");
    return;
  }

  if(!appdata->active_buffer)
    printf("There is no active buffer!\n");
  else {
    if(appdata->active_buffer == 
       gtk_text_view_get_buffer(GTK_TEXT_VIEW(widget))) {
      printf("This was the active buffer\n");

      appdata->active_buffer = NULL;

      gtk_widget_set_sensitive(appdata->menu_cut, FALSE);
      gtk_widget_set_sensitive(appdata->menu_copy, FALSE);
      gtk_widget_set_sensitive(appdata->menu_paste, FALSE);
    }
  }
}
#endif

static void ftime_update(GtkWidget *widget, cache_context_t *context, 
			 gboolean update) {
  /* check if it has been selected */
  if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
    printf("set active\n");

    if(update)
      context->notes.ftime = time(NULL);

    if(context->notes.ftime) {
      struct tm *loctime = localtime(&context->notes.ftime);
      char str[32];
      strftime(str, sizeof(str), "%x %X", loctime);
      
      gtk_label_set_text(GTK_LABEL(context->notes.datew), str);
    } else
      gtk_label_set_text(GTK_LABEL(context->notes.datew), "");

  } else {
    printf("invalidating time\n");
    context->notes.ftime = 0;
    gtk_label_set_text(GTK_LABEL(context->notes.datew), "");
  }
}

/* buffer edited */
static void callback_modified(GtkWidget *widget, gpointer data ) {
  cache_context_t *context = (cache_context_t*)data;
  //  printf("something has been edited\n");
  context->notes.modified = TRUE;

  if(widget == context->notes.foundw) {
    printf("was foundw\n");

    /* about to remove "found" flag -> ask for confirmation */
    if(!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
      GtkWidget *dialog = gtk_message_dialog_new(
		 GTK_WINDOW(context->appdata->window),
		 GTK_DIALOG_DESTROY_WITH_PARENT,
		 GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
		 _("Do you really want to remove the \"found\" flag? "
		   "This will void the recorded date of your find!"));
      
      gtk_window_set_title(GTK_WINDOW(dialog), _("Reset \"found\" flag?"));

      /* set the active flag again if the user answered "no" */
      if(GTK_RESPONSE_NO == gtk_dialog_run(GTK_DIALOG(dialog)))
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), TRUE);

      gtk_widget_destroy(dialog);
    }

    ftime_update(widget, context, TRUE);
  }

  if(widget == context->notes.loggedw) {
    printf("was loggedw\n");

    /* about to remove "found" flag -> ask for confirmation */
    if(!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
      GtkWidget *dialog = gtk_message_dialog_new(
		 GTK_WINDOW(context->appdata->window),
		 GTK_DIALOG_DESTROY_WITH_PARENT,
		 GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
		 _("Do you really want to remove the \"logged\" flag? "
		   "This may cause problems on your next Garmin Field "
		   "Notes upload!"));
      
      gtk_window_set_title(GTK_WINDOW(dialog), _("Reset \"logged\" flag?"));

      /* set the active flag again if the user answered "no" */
      if(GTK_RESPONSE_NO == gtk_dialog_run(GTK_DIALOG(dialog)))
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), TRUE);
      else {
	gtk_widget_set_sensitive(widget, FALSE);
	gtk_widget_set_sensitive(context->notes.foundw, TRUE);
      }

      gtk_widget_destroy(dialog);
    }
  }
}

#ifndef NO_COPY_N_PASTE
static gboolean focus_in(GtkWidget *widget, GdkEventFocus *event,
			 gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  printf("note focus in!\n");

  /* these buffers are read/write, thus all items are enabled */
  gtk_widget_set_sensitive(appdata->menu_cut, TRUE);
  gtk_widget_set_sensitive(appdata->menu_copy, TRUE);
  gtk_widget_set_sensitive(appdata->menu_paste, TRUE);

  if(GTK_WIDGET_TYPE(widget) == GTK_TYPE_TEXT_VIEW) {
    appdata->active_buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(widget));
  } else
    printf("not a text view\n");

  return FALSE;
}
#endif

static gboolean focus_out(GtkWidget *widget, GdkEventFocus *event,
			 gpointer data) {
  cache_context_t *context = (cache_context_t*)data;

  notes_save(context);
#if !defined(USE_MAEMO) && defined(ENABLE_OSM_GPS_MAP)
  map_update(context->appdata);
#endif

  return FALSE;
}

GtkWidget *cache_notes(cache_context_t *context) {
  cache_t *cache = context->cache;

  context->notes.modified = FALSE;

  if(context->cache->notes)
    context->notes.ftime = context->cache->notes->ftime;
  else 
    context->notes.ftime = 0;

  GtkWidget *vbox = gtk_vbox_new(FALSE, 2);

  /* -------------- custom coordinate ---------------- */

  GtkWidget *table = gtk_table_new(2, 4, FALSE);
  gtk_table_set_col_spacing(GTK_TABLE(table), 0, 16);
  gtk_table_set_col_spacing(GTK_TABLE(table), 2, 16);

  gtk_table_attach_defaults(GTK_TABLE(table),
		    gtk_label_new(_("New coordinate:")), 0, 1, 0, 1);  
  context->notes.overridew = gtk_check_button_new_with_label(_("Override"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context->notes.overridew), 
			       cache->notes && cache->notes->override);
  gtk_table_attach_defaults(GTK_TABLE(table),
			    context->notes.overridew, 0, 1, 1, 2);  

  GtkWidget *hbox = gtk_hbox_new(FALSE, 2);

  context->notes.foundw = gtk_check_button_new_with_label(_("Found"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context->notes.foundw), 
			       cache->notes && cache->notes->found);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context->notes.foundw);

  context->notes.loggedw = gtk_check_button_new_with_label(_("Logged"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context->notes.loggedw), 
			       cache->notes && cache->notes->logged);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context->notes.loggedw);
  
  gtk_table_attach_defaults(GTK_TABLE(table), hbox, 3, 4, 0, 1);  

  /* only found and logged caches have a log flag the user can change */
  if(!(cache->notes && cache->notes->found && cache->notes->logged))
    gtk_widget_set_sensitive(context->notes.loggedw, FALSE);
  else
    gtk_widget_set_sensitive(context->notes.foundw, FALSE);

  context->notes.datew = gtk_label_new("");
  gtk_misc_set_alignment(GTK_MISC(context->notes.datew), 0.0f, 0.5f);
  gtk_table_attach_defaults(GTK_TABLE(table),
			    context->notes.datew, 3, 4, 1, 2);  
  ftime_update(context->notes.foundw, context, FALSE);

  pos_t pos = gpx_cache_pos(cache);
  if(cache->notes) pos = cache->notes->pos;

  gtk_table_attach_defaults(GTK_TABLE(table), 
			    context->notes.latw = lat_entry_new(pos.lat), 2, 3, 0, 1);
  g_signal_connect(G_OBJECT(context->notes.latw), "focus-out-event", 
		   G_CALLBACK(focus_out), context);
  
  gtk_table_attach_defaults(GTK_TABLE(table), 
			    context->notes.lonw = lon_entry_new(pos.lon), 2, 3, 1, 2);
  g_signal_connect(G_OBJECT(context->notes.lonw), "focus-out-event", 
		   G_CALLBACK(focus_out), context);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), table, FALSE, FALSE, 0);  
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);  

#ifndef USE_PANNABLE_AREA
  GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
  				 GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
#else
  GtkWidget *pannable_area = hildon_pannable_area_new();
#endif

  context->notes.buffer = gtk_text_buffer_new(NULL);

  if(cache->notes && cache->notes->text)
    gtk_text_buffer_set_text(context->notes.buffer, cache->notes->text, -1);

#ifndef USE_HILDON_TEXT_VIEW
  GtkWidget *view = gtk_text_view_new_with_buffer(context->notes.buffer);
#else
  GtkWidget *view = hildon_text_view_new();
  hildon_text_view_set_buffer(HILDON_TEXT_VIEW(view), context->notes.buffer);
#endif

  gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(view), GTK_WRAP_WORD);
  gtk_text_view_set_editable(GTK_TEXT_VIEW(view), TRUE);
  gtk_text_view_set_left_margin(GTK_TEXT_VIEW(view), 2 );
  gtk_text_view_set_right_margin(GTK_TEXT_VIEW(view), 2 );

#ifdef USE_MAEMO 
  /* Enable Rich Text Support */
  gtk_text_buffer_set_can_paste_rich_text(context->notes.buffer, TRUE );
  gtk_text_buffer_set_rich_text_format(context->notes.buffer, "RTF" );
#endif  

#ifndef NO_COPY_N_PASTE
  g_signal_connect(G_OBJECT(view), "focus-in-event", 
		   G_CALLBACK(focus_in), context->appdata);
  g_signal_connect(G_OBJECT(view), "destroy", 
		   G_CALLBACK(on_destroy_textview), context->appdata);
#endif

#ifndef USE_PANNABLE_AREA
  gtk_container_add(GTK_CONTAINER(scrolled_window), view);

  gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW(scrolled_window),
				       GTK_SHADOW_IN);
  gtk_box_pack_start(GTK_BOX(vbox), scrolled_window, TRUE, TRUE, 0);
#else
  gtk_container_add(GTK_CONTAINER(pannable_area), view);
  gtk_box_pack_start(GTK_BOX(vbox), pannable_area, TRUE, TRUE, 0);
#endif

  gtk_text_view_place_cursor_onscreen(GTK_TEXT_VIEW(view));
  
  g_signal_connect(G_OBJECT(context->notes.buffer), "modified-changed",
		   G_CALLBACK(callback_modified), context);
  g_signal_connect(G_OBJECT(context->notes.buffer), "changed",
		   G_CALLBACK(callback_modified), context);
  g_signal_connect(G_OBJECT(context->notes.lonw), "changed",
		   G_CALLBACK(callback_modified), context);
  g_signal_connect(G_OBJECT(context->notes.latw), "changed",
		   G_CALLBACK(callback_modified), context);
  g_signal_connect(G_OBJECT(context->notes.overridew), "toggled",
		   G_CALLBACK(callback_modified), context);
  g_signal_connect(G_OBJECT(context->notes.foundw), "toggled",
		   G_CALLBACK(callback_modified), context);
  g_signal_connect(G_OBJECT(context->notes.loggedw), "toggled",
		   G_CALLBACK(callback_modified), context);

  return vbox;
}

void notes_free(notes_t *notes) {
  if(notes) {
    if(notes->text) xmlFree(notes->text);
    free(notes);
  }
}

pos_t notes_get_pos(cache_context_t *context) {
  pos_t pos = context->cache->pos;
  if(gtk_toggle_button_get_active(
		 GTK_TOGGLE_BUTTON(context->notes.overridew))) {
    pos.lat = lat_get(context->notes.latw);
    pos.lon = lon_get(context->notes.lonw);
  }
  return pos;
}

gboolean notes_get_override(cache_context_t *context) {
  /* get override value */
  return gtk_toggle_button_get_active(
		      GTK_TOGGLE_BUTTON(context->notes.overridew));

}

typedef struct {
  GtkWidget *info_label;
  GtkWidget *dialog;
  appdata_t *appdata;
  GtkWidget *path_label;
} export_context_t;

static void on_browse(GtkWidget *widget, gpointer data) {
  GtkWidget *dialog;

  export_context_t *context = (export_context_t*)data;

#ifdef USE_MAEMO
  dialog = hildon_file_chooser_dialog_new(GTK_WINDOW(context->dialog), 
					  GTK_FILE_CHOOSER_ACTION_SAVE);
#else
  dialog = gtk_file_chooser_dialog_new(_("Save POI database"),
				       GTK_WINDOW(context->dialog),
				       GTK_FILE_CHOOSER_ACTION_SAVE,
				       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				       GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
				       NULL);
#endif

  printf("set filename <%s>\n", context->appdata->fieldnotes_path);

  if(!g_file_test(context->appdata->fieldnotes_path, G_FILE_TEST_EXISTS)) {
    char *last_sep = strrchr(context->appdata->fieldnotes_path, '/');
    if(last_sep) {
      *last_sep = 0;  // seperate path from file 

      /* the user just created a new document */
      gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), 
					  context->appdata->fieldnotes_path);
      gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog), last_sep+1);

      /* restore full filename */
      *last_sep = '/';
    }
  } else 
    gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(dialog), 
				  context->appdata->fieldnotes_path);

  if (gtk_dialog_run (GTK_DIALOG(dialog)) == GTK_FM_OK) {
    gchar *name = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
    if(name) {
      free(context->appdata->fieldnotes_path);
      context->appdata->fieldnotes_path = strdup(name);
      gtk_label_set_text(GTK_LABEL(context->path_label), 
			 context->appdata->fieldnotes_path);
    }
  }
  
  gtk_widget_destroy (dialog);
}

typedef struct log_chain_s {
  cache_t *cache;
  struct log_chain_s *next;
} log_chain_t;

void notes_log_export(appdata_t *appdata) {
  gpx_t *gpx = appdata->gpx;
  log_chain_t *log = NULL, *llog, **clog = &log;

  export_context_t context;
  memset(&context, 0, sizeof(export_context_t));
  context.appdata = appdata;

  printf("export log\n");

  int logs2export = 0;

  /* make sure all notes are loaded */
  while(gpx) {
    if(!gpx->notes_loaded) {
      notes_load_all(appdata, gpx);
      gpx->notes_loaded = TRUE;
    }

    cache_t *cache = gpx->cache;
    while(cache) {
      if(cache->notes && cache->notes->found && !cache->notes->logged) {
	gboolean already_chained = FALSE;
	llog = log;
	while(llog) {
	  if(strcasecmp(llog->cache->id, cache->id) == 0)
	    already_chained = TRUE;

	  llog = llog->next;
	}

	if(!already_chained) {
	  logs2export++;

	  printf("chaining log for %s\n", cache->id);

	  *clog = g_new0(log_chain_t, 1);
	  (*clog)->cache = cache; 
	  clog = &((*clog)->next);
	} else
	  printf("dup for %s\n", cache->id);
	
      }

      cache = cache->next;
    }

    gpx = gpx->next;
  }


  /* ------------- confirmation dialog ---------------- */
  char *old_fieldnotes_path = strdup(appdata->fieldnotes_path);

  context.dialog = gtk_dialog_new_with_buttons(_("Garmin Field Notes Export"),
	  GTK_WINDOW(appdata->window), GTK_DIALOG_MODAL,
	  GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
          GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
          NULL);

#if defined(USE_MAEMO) && defined(HILDON_HELP)
  hildon_help_dialog_help_enable(GTK_DIALOG(context.dialog), 
		 HELP_ID_FIELDNOTES, appdata->osso_context);
#endif 

  GtkWidget *vbox = gtk_vbox_new(FALSE,0);

  /* ------------------ info text -------------- */

  char *msg = g_strdup_printf(_("This will export the notes of %d caches "
				"into the given file. This file can be "
				"uploaded to geocaching.com for logging."),
			      logs2export);

  context.info_label = gtk_label_new(msg);
  g_free(msg);
  gtk_label_set_line_wrap_mode(GTK_LABEL(context.info_label), PANGO_WRAP_WORD);
  gtk_label_set_line_wrap(GTK_LABEL(context.info_label), TRUE);
  gtk_misc_set_alignment(GTK_MISC(context.info_label), 0.f, 0.5f);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), context.info_label);

  /* ------------------ path/file ------------------ */
  gtk_box_pack_start_defaults(GTK_BOX(vbox), gtk_hseparator_new());

  GtkWidget *hbox = gtk_hbox_new(FALSE, 0);
  GtkWidget *label = gtk_label_new(_("Export to:"));
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE,0);
  gtk_misc_set_alignment(GTK_MISC(label), 0.f, 0.5f);
  GtkWidget *button = gtk_button_new_with_label(_("Browse"));
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(on_browse), (gpointer)&context);
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE,0);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), hbox);

  context.path_label = gtk_label_new(appdata->fieldnotes_path);
  gtk_misc_set_alignment(GTK_MISC(context.path_label), 0.f, 0.5f);
  gtk_label_set_ellipsize(GTK_LABEL(context.path_label), 
			  PANGO_ELLIPSIZE_MIDDLE);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), context.path_label);

  label = gtk_label_new(_("(a %s in the filename will be replaced by the "
			  "current date and time)"));
  gtk_label_set_line_wrap_mode(GTK_LABEL(label), PANGO_WRAP_WORD);
  gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
  gtk_misc_set_alignment(GTK_MISC(label), 0.f, 0.5f);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), label);

  /* ------------------ info ------------------ */

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(context.dialog)->vbox), vbox);

  gtk_widget_show_all(context.dialog);

  if(GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(context.dialog))) {

    /* ---------------- do actual export ------------------ */

    time_t now = time(NULL);
    struct tm *tm_now = localtime(&now);
    char now_str[32];
    strftime(now_str, sizeof(now_str)-1, "%F_%H:%M", tm_now);
    char *fname = g_strdup_printf(appdata->fieldnotes_path, now_str);
    printf("--- about to export logs to %s ---\n", fname);
    FILE *file = fopen(fname, "w");
    g_free(fname);

    if(file) {

      llog = log;
      while(llog) {
	printf("Exporting %s\n", llog->cache->id);

	/* ----------- build utc time string ----------- */
	
	char *tz = getenv("TZ");
	setenv("TZ", "", 1);
	tzset();
	struct tm *tm = localtime(&llog->cache->notes->ftime);
	char tstr[32];
	strftime(tstr, sizeof(tstr)-1, "%FT%H:%MZ", tm);
	if(tz) setenv("TZ", tz, 1);
	else   unsetenv("TZ");
	tzset();

	/* escape \" in text */
	char *text = NULL;
	if(llog->cache->notes->text) {
	  text = g_strdup(llog->cache->notes->text);
	  char *p = text;
	  while(*p) {
	    /* convert " to ' */
	    if(*p == '\"') *p = '\'';
	    p++;
	  }
	} else
	  text = g_strdup("");
	
	/* ----------- build complete log string ------------ */
	char *str = g_strdup_printf("%s,%s,Found it,\"%s\"\n", 
				    llog->cache->id, tstr, text);
	g_free(text);
	
	/* ----------- convert to unicode ------------ */
	glong written = 0;
	gunichar2 *uc = g_utf8_to_utf16(str, -1, NULL, &written, NULL);
	g_free(str);

	/* -------------- and write it --------------- */
	fwrite(uc, written, sizeof(gunichar2), file);
	g_free(uc);
	
	/* -------------- set "logged" flag in notes and update them ------ */
	llog->cache->notes->logged = TRUE;

	/* update flags in all copies of this cache */

	gpx_t *tgpx = appdata->gpx;
	while(tgpx) {
	  cache_t *tcache = tgpx->cache;
	  while(tcache) {
	    if((tcache != llog->cache) && 
	       (strcmp(tcache->id, llog->cache->id) == 0)) {
	      printf("found dup cache %s in %s\n", tcache->id, tgpx->name);

	      if(tcache->notes)
		notes_free(tcache->notes);

	      /* create a new note for this cache */
	      tcache->notes = malloc(sizeof(notes_t));
	      memset(tcache->notes, 0, sizeof(notes_t));
	      if(llog->cache->notes->text)
		tcache->notes->text = strdup(llog->cache->notes->text);
	      tcache->notes->pos = llog->cache->notes->pos;
	      tcache->notes->override = llog->cache->notes->override;
	      tcache->notes->found = llog->cache->notes->found;
	      tcache->notes->logged = llog->cache->notes->logged;
	      tcache->notes->ftime = llog->cache->notes->ftime;
	    }
	    tcache = tcache->next;
	  }
	  tgpx = tgpx->next;
	}
	
	/* finally write the notes file itself */
	cache_context_t ccontext;
	ccontext.appdata = appdata;
	ccontext.cache = llog->cache;

	notes_write_file(&ccontext,
		   llog->cache->notes->text, llog->cache->notes->pos, 
		   llog->cache->notes->override, llog->cache->notes->found, 
		   llog->cache->notes->ftime, llog->cache->notes->logged);

	llog = llog->next;
      }
    
      fclose(file);
    }
  } else {
    /* restore old path, in case it has been altered but not been used */
    free(appdata->fieldnotes_path);
    appdata->fieldnotes_path = strdup(old_fieldnotes_path);
  }

  gtk_widget_destroy(context.dialog);
  
  free(old_fieldnotes_path);

  /* free list */
  while(log) {
    log_chain_t *next = log->next;
    g_free(log);
    log = next;
  }
}
