/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"
#include <math.h>

typedef struct {
  GtkWidget *cbox_gps;
  GtkWidget *loc;
} settings_dialog_state_t;

/* Our usual callback function */
static void settings_update(GtkWidget *widget, gpointer data) {
  settings_dialog_state_t *hstate = (settings_dialog_state_t *)data;

  if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(hstate->cbox_gps))) 
    gtk_widget_set_sensitive(hstate->loc, FALSE);
  else
    gtk_widget_set_sensitive(hstate->loc, TRUE);
}

typedef struct {
  appdata_t *appdata;
  GtkWidget *settings_dialog;
  GtkWidget *view;
  GtkListStore *store;
  GtkWidget *but_add, *but_edit, *but_remove;
  gboolean changed;
} location_context_t;

enum {
  LOCATION_COL_NAME = 0,
  LOCATION_COL_LAT,
  LOCATION_COL_LON,
  LOCATION_COL_DATA,
  LOCATION_NUM_COLS
};

static void location_select(location_context_t *context) {
  GtkTreeSelection *selection = 
    gtk_tree_view_get_selection(GTK_TREE_VIEW(context->view));

  GtkTreePath *path = gtk_tree_path_new_from_indices(
			    context->appdata->active_location, -1);

  /* Modify a particular row */
  GtkTreeIter iter;
  gtk_tree_model_get_iter(GTK_TREE_MODEL(context->store), &iter, path);
  gtk_tree_selection_select_iter(selection, &iter);
  gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(context->view), 
  			       path, NULL, TRUE, 0, 0);
  gtk_tree_path_free(path);

  if(!context->appdata->active_location && context->but_remove) 
    gtk_widget_set_sensitive(context->but_remove,  FALSE);
}

#ifdef ENABLE_MAEMO_MAPPER
#include "dbus.h"

typedef struct {
  appdata_t *appdata;
  pos_t pos;
  GtkWidget *import_button;
  GtkWidget *export_button;
  GtkWidget *lat, *lon;
} mm_context_t;

static void on_mm_import_clicked(GtkButton *button, gpointer data) {
  char str[32];
  mm_context_t *context = (mm_context_t*)data;

  pos_lat_str(str, sizeof(str), context->appdata->mmpos.lat);
  gtk_entry_set_text(GTK_ENTRY(context->lat), str);
  pos_lon_str(str, sizeof(str), context->appdata->mmpos.lon);
  gtk_entry_set_text(GTK_ENTRY(context->lon), str);
}

static void on_mm_export_clicked(GtkButton *button, gpointer data) {
  mm_context_t *context = (mm_context_t*)data;

  /* update position from entries */
  pos_t pos;
  pos.lat = lat_get(context->lat);
  pos.lon = lon_get(context->lon);

  g_assert(!isnan(pos.lat) && !isnan(pos.lon));

  dbus_mm_set_position(context->appdata, &pos);
}

static void callback_modified_pos(GtkWidget *widget, gpointer data ) {
  mm_context_t *context = (mm_context_t*)data;

  gboolean valid =
    (!isnan(lat_get(context->lat))) && (!isnan(lon_get(context->lon)));

  gtk_widget_set_sensitive(context->export_button, valid);
}
#endif

static void on_location_edit(GtkWidget *button, location_context_t *context) {
  GtkWidget *dialog = gtk_dialog_new_with_buttons(_("Edit Location"),
	  GTK_WINDOW(context->settings_dialog), GTK_DIALOG_MODAL,
          GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
	  GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);

#if defined(USE_MAEMO) && defined(HILDON_HELP)
  hildon_help_dialog_help_enable(GTK_DIALOG(dialog), 
		 HELP_ID_LOCEDIT, context->appdata->osso_context);
#endif

  printf("edit, active = %d\n", context->appdata->active_location);

  location_t *loc = NULL;
  if(context->appdata->active_location) {
    loc = context->appdata->location;
    int i = context->appdata->active_location-1;
    while(i--) {
      g_assert(loc->next);
      loc = loc->next;
    }
    printf("location edit for %s\n", loc->name);
  } else
    printf("location edit for Home\n");

#ifdef ENABLE_MAEMO_MAPPER
  mm_context_t mm_context;
#else
  GtkWidget *lat, *lon;
#endif

  GtkWidget *label, *name;
  GtkWidget *table = gtk_table_new(2, 3, FALSE);

  gtk_table_attach_defaults(GTK_TABLE(table),
		    label = gtk_label_new(_("Name:")), 0, 1, 0, 1);
  gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
  gtk_table_attach_defaults(GTK_TABLE(table), 
		    name = gtk_entry_new(), 1, 2, 0, 1);

  pos_t pos;
  if(loc) pos = loc->pos;
  else    pos = context->appdata->home;

  /* avoid to use "nan" as the user will then not be displayed a nice */
  /* preset value to alter */
  if(isnan(pos.lat)) pos.lat = 0;
  if(isnan(pos.lon)) pos.lon = 0;

  gtk_table_attach_defaults(GTK_TABLE(table),
		   label = gtk_label_new(_("Latitude:")), 0, 1, 1, 2);
  gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
#ifdef ENABLE_MAEMO_MAPPER
  gtk_table_attach_defaults(GTK_TABLE(table), 
		    mm_context.lat = lat_entry_new(pos.lat), 1, 2, 1, 2);
  g_signal_connect(G_OBJECT(mm_context.lat), "changed",
		   G_CALLBACK(callback_modified_pos), &mm_context);
#else
  gtk_table_attach_defaults(GTK_TABLE(table), 
		    lat = lat_entry_new(pos.lat), 1, 2, 1, 2);
#endif

  
  gtk_table_attach_defaults(GTK_TABLE(table),
		    label = gtk_label_new(_("Longitude:")), 0, 1, 2, 3);
  gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
#ifdef ENABLE_MAEMO_MAPPER
  gtk_table_attach_defaults(GTK_TABLE(table),
		   mm_context.lon = lon_entry_new(pos.lon), 1, 2, 2, 3);
  g_signal_connect(G_OBJECT(mm_context.lon), "changed",
		   G_CALLBACK(callback_modified_pos), &mm_context);
#else
  gtk_table_attach_defaults(GTK_TABLE(table),
		   lon = lon_entry_new(pos.lon), 1, 2, 2, 3);
#endif

  if(loc) 
    gtk_entry_set_text(GTK_ENTRY(name), loc->name);
  else {
    gtk_entry_set_text(GTK_ENTRY(name), _("Home"));
    gtk_widget_set_sensitive(GTK_WIDGET(name), FALSE);
  }

#ifdef ENABLE_MAEMO_MAPPER
  mm_context.appdata = context->appdata;
  if(loc) mm_context.pos = loc->pos;
  else    mm_context.pos = context->appdata->home;

  mm_context.import_button = gtk_button_new();
  gtk_button_set_image(GTK_BUTTON(mm_context.import_button), 
		       icon_get_widget(ICON_MISC, 5));
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->action_area),
		    mm_context.import_button);
  gtk_signal_connect(GTK_OBJECT(mm_context.import_button), "clicked",
    		     (GtkSignalFunc)on_mm_import_clicked, &mm_context);
  
  if(!context->appdata->mmpos_valid) 
    gtk_widget_set_sensitive(mm_context.import_button,  FALSE);
  
  mm_context.export_button = gtk_button_new();
  gtk_button_set_image(GTK_BUTTON(mm_context.export_button), 
		       icon_get_widget(ICON_MISC, 0));
  gtk_signal_connect(GTK_OBJECT(mm_context.export_button), "clicked",
  		     (GtkSignalFunc)on_mm_export_clicked, &mm_context);
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->action_area), 
		    mm_context.export_button);
#endif

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), table);

  gtk_widget_show_all(dialog);

  if(GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(dialog))) {
    pos_t pos;

#ifdef ENABLE_MAEMO_MAPPER
    pos.lat = lat_get(mm_context.lat);
    pos.lon = lon_get(mm_context.lon);
#else
    pos.lat = lat_get(lat);
    pos.lon = lon_get(lon);
#endif

    if(isnan(pos.lat) || isnan(pos.lon)) 
      errorf(_("Ignoring invalid position"));
    else {
      char *p = (char*)gtk_entry_get_text(GTK_ENTRY(name));
      printf("%s is at %f/%f\n", p, pos.lat, pos.lon);

      /* now the list has to be re-done */
      GtkTreePath *path = gtk_tree_path_new_from_indices(
			 context->appdata->active_location, -1);
      
      /* Modify a particular row or create a new one if that doesn't exist */
      GtkTreeIter iter;
      if(!gtk_tree_model_get_iter(GTK_TREE_MODEL(context->store),&iter,path)) 
	gtk_list_store_append(context->store, &iter);
      
      char lat_str[32], lon_str[32];
      pos_lat_str(lat_str, sizeof(lat_str), pos.lat);
      pos_lon_str(lon_str, sizeof(lon_str), pos.lon);
      
      if(loc) {
	free(loc->name);
	loc->name = strdup(p);
	loc->pos.lat = pos.lat;
	loc->pos.lon = pos.lon;
	
	gtk_list_store_set(context->store, &iter,
			   LOCATION_COL_NAME, loc->name,
			   LOCATION_COL_LAT, lat_str,
			 LOCATION_COL_LON, lon_str,
			   LOCATION_COL_DATA, loc,
			   -1);
	
      } else {
	context->appdata->home.lat = pos.lat;
	context->appdata->home.lon = pos.lon;      
	
	gtk_list_store_set(context->store, &iter,
			   LOCATION_COL_LAT, lat_str,
			   LOCATION_COL_LON, lon_str,
			   -1);
      }
      context->changed = TRUE;
    }
  }
  gtk_widget_destroy(dialog);
}

static void on_location_add(GtkWidget *button, location_context_t *context) {
  location_t **loc = &context->appdata->location;
  int prev_active = context->appdata->active_location;

  int i = 1;
  while(*loc) {
    loc = &(*loc)->next;
    i++;
  }

  *loc = g_new0(location_t, 1);
  if(!*loc) {
    errorf(_("Out of memory"));
    return;
  }

  (*loc)->name = strdup(_("<new>"));
#if 0
  (*loc)->pos.lat = DEFAULT_LAT;
  (*loc)->pos.lon = DEFAULT_LON;
#endif

  context->changed = FALSE;
  context->appdata->active_location = i;
  on_location_edit(button, context);

  if(context->changed) 
    location_select(context);
  else {
    /* remove newly attached  entry and select previous one */
    location_t *tmp = *loc;
    *loc = NULL;
    free(tmp->name);
    free(tmp);

    context->appdata->active_location = prev_active;
  }
}

static void on_location_remove(GtkWidget *but, location_context_t *context) {
  GtkTreeSelection *selection;
  GtkTreeModel     *model;
  GtkTreeIter       iter;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(context->view));
  if(gtk_tree_selection_get_selected(selection, &model, &iter)) {
    location_t *loc;
    gtk_tree_model_get(model, &iter, LOCATION_COL_DATA, &loc, -1);

    g_assert(loc);

    /* de-chain */
    location_t **prev = &context->appdata->location;
    while(*prev != loc) prev = &((*prev)->next);
    *prev = loc->next;

    /* free location itself */
    if(loc->name) free(loc->name);
    free(loc);

    /* and remove from store */
    gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
  }
  
  /* disable remove button */
  gtk_widget_set_sensitive(context->but_remove,  FALSE);
  /* select the first entry */

  context->appdata->active_location = 0;
  location_select(context);
}

static gboolean
view_selection_func(GtkTreeSelection *selection, GtkTreeModel *model,
		     GtkTreePath *path, gboolean path_currently_selected,
		     gpointer userdata) {
  location_context_t *context = (location_context_t*)userdata;
  GtkTreeIter iter;
    
  if(gtk_tree_model_get_iter(model, &iter, path)) {
    g_assert(gtk_tree_path_get_depth(path) == 1);

    /* if the first entry has been selected */
    if(!path_currently_selected && context->but_remove) {
      context->appdata->active_location = gtk_tree_path_get_indices(path)[0];
      gtk_widget_set_sensitive(context->but_remove, 
			       context->appdata->active_location);
    }
  }
  
  return TRUE; /* allow selection state to change */
}

static GtkWidget *location_widget(location_context_t *context) {

  GtkWidget *vbox = gtk_vbox_new(FALSE,3);
  context->view = gtk_tree_view_new();

  gtk_tree_selection_set_select_function(
	 gtk_tree_view_get_selection(GTK_TREE_VIEW(context->view)), 
	 view_selection_func, 
	 context, NULL);

#ifndef USE_MAEMO
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(context->view), FALSE);
#endif

  /* --- "Name" column --- */
  GtkCellRenderer *renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "ellipsize", PANGO_ELLIPSIZE_END, NULL );
  GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes(
		 _("Name"), renderer, "text", LOCATION_COL_NAME, NULL);
  gtk_tree_view_column_set_expand(column, TRUE);
  gtk_tree_view_insert_column(GTK_TREE_VIEW(context->view), column, -1);

  /* --- "Latitude" column --- */
  renderer = gtk_cell_renderer_text_new();
  //  g_object_set(renderer, "xalign", 1.0, NULL );
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(context->view),
	-1, _("Latitude"), renderer, "text", LOCATION_COL_LAT, NULL);

  /* --- "Longitude" column --- */
  renderer = gtk_cell_renderer_text_new();
  //  g_object_set(renderer, "xalign", 1.0, NULL );
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(context->view),
	-1, _("Longitude"), renderer, "text", LOCATION_COL_LON, NULL);

  /* build and fill the store */
  context->store = gtk_list_store_new(LOCATION_NUM_COLS, 
		G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_POINTER);

  gtk_tree_view_set_model(GTK_TREE_VIEW(context->view), 
			  GTK_TREE_MODEL(context->store));

  char lat[32], lon[32];
  GtkTreeIter iter;
  
  /* add home position */
  pos_lat_str(lat, sizeof(lat), context->appdata->home.lat);
  pos_lon_str(lon, sizeof(lon), context->appdata->home.lon);
  gtk_list_store_append(context->store, &iter);
  gtk_list_store_set(context->store, &iter,
		     LOCATION_COL_NAME, _("Home"),
		     LOCATION_COL_LAT, lat,
		     LOCATION_COL_LON, lon,
		     LOCATION_COL_DATA, NULL,
		     -1);

  location_t *loc = context->appdata->location;
  while(loc) {
    pos_lat_str(lat, sizeof(lat), loc->pos.lat);
    pos_lon_str(lon, sizeof(lon), loc->pos.lon);

    /* Append a row and fill in some data */
    gtk_list_store_append(context->store, &iter);
    gtk_list_store_set(context->store, &iter,
		       LOCATION_COL_NAME, loc->name,
		       LOCATION_COL_LAT, lat,
		       LOCATION_COL_LON, lon,
		       LOCATION_COL_DATA, loc,
		       -1);
    loc = loc->next;
  }
  
  g_object_unref(context->store);

#if 0
  /* make list react on clicks */
  g_signal_connect(context->view, "row-activated", 
		   (GCallback)gpxlist_view_onRowActivated, appdata);
#endif

  /* select the "active" row */
  location_select(context);

  /* put it into a scrolled window */
  GtkWidget *scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
				 GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled_window), 
				      GTK_SHADOW_ETCHED_IN);
  //  gtk_container_set_border_width(GTK_CONTAINER(scrolled_window), 3);
  gtk_container_add(GTK_CONTAINER(scrolled_window), context->view);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), scrolled_window);

  /* ------- button box ------------ */

  GtkWidget *hbox = gtk_hbox_new(TRUE,3);
  context->but_add = gtk_button_new_with_label(_("Add"));
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context->but_add);
  gtk_signal_connect(GTK_OBJECT(context->but_add), "clicked",
		     GTK_SIGNAL_FUNC(on_location_add), context);

  context->but_edit = gtk_button_new_with_label(_("Edit"));
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context->but_edit);
  gtk_signal_connect(GTK_OBJECT(context->but_edit), "clicked",
		     GTK_SIGNAL_FUNC(on_location_edit), context);

  context->but_remove = gtk_button_new_with_label(_("Remove"));
  gtk_widget_set_sensitive(context->but_remove, 
			   context->appdata->active_location);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context->but_remove);
  gtk_signal_connect(GTK_OBJECT(context->but_remove), "clicked",
      GTK_SIGNAL_FUNC(on_location_remove), context);

  gtk_box_pack_start_defaults(GTK_BOX(vbox), hbox);
  return vbox;
}

void cb_menu_settings(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t *)data;
  GtkWidget *table, *label, *hbox, *notebook;
  GtkWidget *cbox_imperial;
  settings_dialog_state_t hstate;

  GtkWidget *dialog = gtk_dialog_new_with_buttons(_("Settings"),
	    GTK_WINDOW(appdata->window), GTK_DIALOG_MODAL,
            GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
            GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
            NULL);

#if defined(USE_MAEMO) && defined(HILDON_HELP)
  hildon_help_dialog_help_enable(GTK_DIALOG(dialog), 
		 HELP_ID_SETTINGS, appdata->osso_context);
  gtk_window_set_default_size(GTK_WINDOW(dialog), 550, 100);
#endif 

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
		     notebook = gtk_notebook_new(), TRUE, TRUE, 0);


  /* ------------------ the "home" widget ---------------------- */
  table = gtk_table_new(2, 2, FALSE);

  hstate.cbox_gps = gtk_check_button_new_with_label(_("Enable GPS"));
  gtk_table_attach(GTK_TABLE(table),
		   hstate.cbox_gps, 0, 2, 0, 1, GTK_FILL, 0, 2, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(hstate.cbox_gps), 
			       appdata->use_gps);

  location_context_t location_context;
  memset(&location_context, 0, sizeof(location_context_t));  
  location_context.appdata = appdata;
  location_context.settings_dialog = dialog;

  /* location widget */
  gtk_table_attach_defaults(GTK_TABLE(table), 
	    hstate.loc = location_widget(&location_context), 0, 2, 1, 2);

  settings_update(NULL, &hstate);

  /* Connect the "clicked" signal of the button to our callback */
  gtk_signal_connect (GTK_OBJECT (hstate.cbox_gps), "clicked",
		      GTK_SIGNAL_FUNC(settings_update), (gpointer)&hstate);

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table, 
			   gtk_label_new(_("GPS")));

  /* ---------------- misc old main menu entries ----------------- */

  table = gtk_table_new(2, 2, FALSE);

  cbox_imperial = gtk_check_button_new_with_label(
			 _("Imperial units"));
  gtk_table_attach(GTK_TABLE(table),
		   cbox_imperial, 0, 2, 0, 1, GTK_FILL, 0, 2, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cbox_imperial), 
			       appdata->imperial);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table, 
			   gtk_label_new(_("Misc")));

  /* ----------------- gpxlist settings ------------------- */

  table = gtk_table_new(1, 2, FALSE);

  hbox = gtk_hbox_new(FALSE,2);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), 
	    label = gtk_label_new(_("Visible items:")));
  gtk_misc_set_alignment(GTK_MISC(label), 0.f, 0.5f);

  GtkWidget *cbox_fname = gtk_check_button_new_with_label(_("Filename"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cbox_fname), 
		       appdata->gpxlist_items & GPXLIST_ITEM_FILENAME);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), cbox_fname);
  GtkWidget *cbox_date = gtk_check_button_new_with_label(_("Date"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cbox_date), 
		       appdata->gpxlist_items & GPXLIST_ITEM_DATE);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), cbox_date);
  GtkWidget *cbox_num = gtk_check_button_new_with_label(_("# Caches"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cbox_num), 
		       appdata->gpxlist_items & GPXLIST_ITEM_CNUM);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), cbox_num);

  gtk_table_attach(GTK_TABLE(table), hbox, 0, 2, 0, 1, GTK_FILL, 0, 2, 0);

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table, 
			   gtk_label_new(_("GPX list")));

  /* ----------------- cachelist settings ------------------- */

  table = gtk_table_new(4, 2, FALSE);

  hbox = gtk_hbox_new(FALSE,2);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), 
	    label = gtk_label_new(_("Visible items:")));
  gtk_misc_set_alignment(GTK_MISC(label), 0.f, 0.5f);

  GtkWidget *cbox_wpt = gtk_check_button_new_with_label(_("Wpt"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cbox_wpt), 
		       appdata->cachelist_items & CACHELIST_ITEM_ID);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), cbox_wpt);
  GtkWidget *cbox_size = gtk_check_button_new_with_label(_("Size"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cbox_size), 
		       appdata->cachelist_items & CACHELIST_ITEM_SIZE);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), cbox_size);
  GtkWidget *cbox_rate = gtk_check_button_new_with_label(_("Rating"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cbox_rate), 
		       appdata->cachelist_items & CACHELIST_ITEM_RATING);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), cbox_rate);

  gtk_table_attach(GTK_TABLE(table), hbox, 0, 2, 0, 1, GTK_FILL, 0, 2, 0);

  GtkWidget *cbox_cachelist_hidef = 
    gtk_check_button_new_with_label(_("Hide caches marked \"found\""));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cbox_cachelist_hidef), 
			       appdata->cachelist_hide_found);
  gtk_table_attach(GTK_TABLE(table), cbox_cachelist_hidef, 
			    0, 2, 1, 2, GTK_FILL, 0, 2, 0);


#ifdef USE_MAEMO
  GtkWidget *cbox_cachelist_dss = 
    gtk_check_button_new_with_label(_("Disable screen saver"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cbox_cachelist_dss), 
			       appdata->cachelist_disable_screensaver);
  gtk_table_attach(GTK_TABLE(table), cbox_cachelist_dss, 
		   0, 2, 2, 3, GTK_FILL, 0, 2, 0);
#endif

  GtkWidget *cbox_update = 
    gtk_check_button_new_with_label(_("Update every 30 sec"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cbox_update), 
			       appdata->cachelist_update);
  gtk_table_attach(GTK_TABLE(table), cbox_update, 
		   0, 2, 3, 4, GTK_FILL, 0, 2, 0);

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table, 
			   gtk_label_new(_("Cache list")));

  /* ----------------- cache settings ------------------- */

  table = gtk_table_new(2, 2, FALSE);

  hbox = gtk_hbox_new(FALSE,2);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), 
	    label = gtk_label_new(_("Compass damping:")));
  gtk_misc_set_alignment(GTK_MISC(label), 0.f, 0.5f);

  gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Min")), FALSE, FALSE,0);
  GtkWidget *scale = gtk_hscale_new_with_range(1, MAX_AVERAGE, 1);
  gtk_scale_set_value_pos(GTK_SCALE(scale), GTK_POS_LEFT);
  gtk_scale_set_draw_value(GTK_SCALE(scale), FALSE);
  gtk_range_set_value(GTK_RANGE(scale), appdata->compass_damping);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), scale);
  gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Max")), FALSE, FALSE,0);

  gtk_table_attach(GTK_TABLE(table), hbox, 0, 2, 0, 1, GTK_FILL | GTK_EXPAND, 0, 0, 0);

#ifdef USE_MAEMO
  GtkWidget *cbox_goto_dss = gtk_check_button_new_with_label(
			    _("Disable screen saver in \"goto\" view"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cbox_goto_dss), 
			       appdata->goto_disable_screensaver);
  gtk_table_attach(GTK_TABLE(table), cbox_goto_dss, 0, 2, 1, 2, GTK_FILL, 0, 2, 0);
#endif

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table, 
			   gtk_label_new(_("Cache")));


  /* -------------------------------------------------------- */

  gtk_widget_show_all(dialog);

  if(GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(dialog))) {
    gboolean prev_cachelist_hide_found = appdata->cachelist_hide_found;

    appdata->use_gps = gtk_toggle_button_get_active(
			    GTK_TOGGLE_BUTTON(hstate.cbox_gps));
    appdata->imperial = gtk_toggle_button_get_active(
			    GTK_TOGGLE_BUTTON(cbox_imperial));

    appdata->compass_damping = 0.5 + gtk_range_get_value(GTK_RANGE(scale));

    appdata->gpxlist_items = GPXLIST_ITEM_VALID;
    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(cbox_fname)))
      appdata->gpxlist_items |= GPXLIST_ITEM_FILENAME;
    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(cbox_date)))
      appdata->gpxlist_items |= GPXLIST_ITEM_DATE;
    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(cbox_num)))
      appdata->gpxlist_items |= GPXLIST_ITEM_CNUM;
      
    appdata->cachelist_items = CACHELIST_ITEM_VALID;
    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(cbox_wpt)))
      appdata->cachelist_items |= CACHELIST_ITEM_ID;
    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(cbox_size)))
      appdata->cachelist_items |= CACHELIST_ITEM_SIZE;
    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(cbox_rate)))
      appdata->cachelist_items |= CACHELIST_ITEM_RATING;      

    appdata->cachelist_hide_found = gtk_toggle_button_get_active(
			     GTK_TOGGLE_BUTTON(cbox_cachelist_hidef));

#ifdef USE_MAEMO
    appdata->goto_disable_screensaver = gtk_toggle_button_get_active(
			     GTK_TOGGLE_BUTTON(cbox_goto_dss));      
    appdata->cachelist_disable_screensaver = gtk_toggle_button_get_active(
			     GTK_TOGGLE_BUTTON(cbox_cachelist_dss));
#endif
    appdata->cachelist_update = gtk_toggle_button_get_active(
			     GTK_TOGGLE_BUTTON(cbox_update));

    /* build some additional flags that are used to decide whether a */
    /* redraw is necessary */
    int flags = CHANGE_FLAG_POS;

    if(prev_cachelist_hide_found != appdata->cachelist_hide_found)
      flags |= CHANGE_FLAG_MASK;   // visibility mask has changed 

    main_after_settings_redraw(appdata, flags);
  }
  gtk_widget_destroy(dialog);
}

