#include "launchdialog.h"

LaunchDialog::LaunchDialog(QFileInfo romFile, QIcon cover, QWidget *parent) :
    QDialog(parent),
    ui(new Ui::LaunchDialog)
{
    ui->setupUi(this);

    this->setAttribute(Qt::WA_DeleteOnClose);

    // Realign the start button
    ui->startLayout->removeWidget(ui->startButton);
    ui->startLayout->addWidget(ui->startButton, 1, 1, Qt::AlignBottom);

    this->romFile = romFile.absoluteFilePath();

    this->setWindowTitle(romFile.completeBaseName());

    // Set ROM info appearance
    QPalette palette = ui->romInfoLabel->palette();
    palette.setColor(QPalette::WindowText, QMaemo5Style::standardColor("SecondaryTextColor"));
    ui->romInfoLabel->setPalette(palette);

    // Set ROM info
    RomInfo info(romFile.absoluteFilePath());
    ui->romInfoLabel->setText(info.isAvailable() ?
                              QString("%1 / %2 / %3").arg(info.title(), info.code(), info.maker()) :
                              "(info unavailable)");

    // Prepare configurations
    QString configFile = romFile.absolutePath() + "/" + romFile.completeBaseName() + ".ini";
    coreConfig = CoreConfig(configFile);
    keysConfig = KeysConfig(configFile);
    accelerometerConfig = AccelerometerConfig(configFile);

    // Hide the image widget if there's nothing to show in it
    if (cover.availableSizes().isEmpty()) {
        ui->coverImage->hide();
    } else {
        ui->coverImage->setPixmap(cover.pixmap(160, 160));
        ui->coverImage->setContentsMargins(ui->startLayout->spacing(), 0, 0, 0);
    }

    // Connect buttons
    connect(ui->startButton, SIGNAL(clicked()), this, SLOT(onStartClicked()));
    connect(ui->coreConfigButton, SIGNAL(clicked()), this, SLOT(configureCore()));
    connect(ui->keysConfigButton, SIGNAL(clicked()), this, SLOT(configureKeys()));

    // Connect checkboxes
    connect(ui->customCoreBox, SIGNAL(stateChanged(int)), this, SLOT(onCoreToggled()));
    connect(ui->customKeysBox, SIGNAL(stateChanged(int)), this, SLOT(onKeysToggled()));

    // Tick checkboxes
    ui->customCoreBox->setChecked(coreConfig.exists());
    ui->customKeysBox->setChecked(keysConfig.exists());
}

LaunchDialog::~LaunchDialog()
{
    cleanup();

    delete ui;
}

// Remove unused custom configs
void LaunchDialog::cleanup()
{
    if (!ui->customCoreBox->isChecked()) {
        coreConfig.drop();
        coreConfig = CoreConfig(Path::cfgFile());
        coreConfig.unbind();
    }
    if (!ui->customKeysBox->isChecked()) {
        keysConfig.drop();
        keysConfig = KeysConfig(Path::cfgFile());
        coreConfig.unbind();
    }
}

void LaunchDialog::onCoreToggled()
{
    if (ui->customCoreBox->isChecked()) {
        ui->coreConfigButton->show();
        if (!coreConfig.exists())
            configureCore();
    } else {
        ui->coreConfigButton->hide();
    }
}

void LaunchDialog::onKeysToggled()
{
    if (ui->customKeysBox->isChecked()) {
        ui->keysConfigButton->show();
        if (!keysConfig.exists())
            configureKeys();
    } else {
        ui->keysConfigButton->hide();
    }
}

void LaunchDialog::onStartClicked()
{
    cleanup();

    Launcher(romFile, &coreConfig, &keysConfig, &accelerometerConfig).launch();

    this->close();
}

void LaunchDialog::configureCore()
{
    (new CoreDialog(&coreConfig, this))->exec();
}

void LaunchDialog::configureKeys()
{
    (new KeysDialog(&keysConfig, this))->exec();
}
