#include "accelerometerwidget.h"

AccelerometerWidget::AccelerometerWidget(QWidget *parent) :
    QWidget(parent),
    file("/sys/class/i2c-adapter/i2c-3/3-001d/coord")
{
    portrait = false;

    pitch = 0;
    roll  = 0;

    deadzone = 0;
    neutralPitch = 0;
    neutralRoll  = 0;

    this->setFixedSize(181, 361);

    background = QPixmap(this->width(), this->height());

    QPainter painter(&background);

    // Prepare the background
    painter.setPen(QMaemo5Style::standardColor("SecondaryTextColor"));
    painter.setBrush(QPalette().color(QPalette::Base));
    painter.drawRect(0,0,this->width()-1, this->height()-1);

    // Prepare direction areas on the background
    painter.translate(this->width()/2, this->height()/2);
    painter.setBrush(QMaemo5Style::standardColor("SecondaryTextColor"));
    painter.setRenderHints(QPainter::Antialiasing);
    for (int i = 0; i < 4; i++)
      drawDirectionArea(painter, i);

    readTimer = new QTimer(this);
    readTimer->setInterval(50);

    connect(readTimer, SIGNAL(timeout()), this, SLOT(read()));
}

void AccelerometerWidget::resume()
{
    readTimer->start();
}

void AccelerometerWidget::suspend()
{
    readTimer->stop();
    pitch = 0;
    roll  = 0;
    this->update();
}

void AccelerometerWidget::setPortrait(bool portrait)
{
    this->portrait = portrait;
}

void AccelerometerWidget::setDeadzone(double deadzone)
{
    this->deadzone = deadzone;
}

void AccelerometerWidget::setNeutralPitch(double pitch)
{
    neutralPitch = pitch;
}

void AccelerometerWidget::setNeutralRoll(double roll)
{
    neutralRoll = roll;
}

float AccelerometerWidget::absolutePitch()
{
    return neutralPitch - pitch;
}

float AccelerometerWidget::absoluteRoll()
{
    return neutralRoll - roll;
}

void AccelerometerWidget::read()
{
    int x, y, z;

    // Read raw values
    file.open(QIODevice::ReadOnly);
    if (portrait) {
        QTextStream(&file) >> y >> x >> z;
        x = -x;
    } else {
        QTextStream(&file) >> x >> y >> z;
    }
    file.close();

    // Convert to angular coordinates
    pitch = fmod(atan2(y, -z) * 180/M_PI + neutralPitch + 180, 360) - 180;
    roll  = atan2(x, sqrt(y*y + z*z)) * 180/M_PI + neutralRoll;

    this->update();
}

const QPoint AccelerometerWidget::directionPoints[8] = {
    QPoint(  90,  tan( -22.5 * M_PI/180) *  90),
    QPoint(  90,  tan(  22.5 * M_PI/180) *  90),
    QPoint( 180,  tan(  67.5 * M_PI/180) * 180),
    QPoint(-180, -tan( 112.5 * M_PI/180) * 180),
    QPoint( -90, -tan( 157.5 * M_PI/180) *  90),
    QPoint( -90, -tan(-157.5 * M_PI/180) *  90),
    QPoint(-180, -tan(-112.5 * M_PI/180) * 180),
    QPoint( 180,  tan( -67.5 * M_PI/180) * 180)
};

void AccelerometerWidget::drawDirectionArea(QPainter &painter, int index)
{
    QPoint points[3] = {
        QPoint(0, 0),
        directionPoints[index*2],
        directionPoints[index*2 + 1]
    };

    painter.drawPolygon(points, 3);
}

void AccelerometerWidget::paintEvent(QPaintEvent *)
{
    QPainter painter(this);

    // Background
    painter.drawPixmap(0, 0, background);

    painter.translate(this->width()/2, this->height()/2);

    painter.setRenderHints(QPainter::Antialiasing);
    painter.setPen(QMaemo5Style::standardColor("SecondaryTextColor"));

    // Active direction areas
    if (pitch*pitch + roll*roll > deadzone*deadzone) {
        float direction = atan2(pitch, roll);

        painter.setBrush(QMaemo5Style::standardColor("ActiveTextColor"));

        if (direction < M_PI*3/8 && direction > -M_PI*3/8)
            drawDirectionArea(painter, 2);
        else if (direction > M_PI*5/8 || direction < -M_PI*5/8)
            drawDirectionArea(painter, 0);

        if (direction > M_PI*1/8 && direction < M_PI*7/8)
            drawDirectionArea(painter, 3);
        else if (direction < -M_PI*1/8 && direction > -M_PI*7/8)
            drawDirectionArea(painter, 1);
    }

    // Deadzone
    painter.setBrush(QPalette().color(QPalette::Base));
    painter.drawEllipse(QPointF(0, 0), deadzone, deadzone);

    painter.resetTransform();

    // Current coordinates
    painter.setPen(QMaemo5Style::standardColor("DefaultTextColor"));
    painter.setBrush(Qt::transparent);
    painter.drawEllipse(QPoint(this->width()/2 - roll, this->height()/2 - pitch), 5, 5);
}
