#include "rominfo.h"

#define ROM_INFO_OFFSET 0xA0
#define READ_BUFFER_SIZE 4096

struct ZipDataDescriptor
{
  quint32 CRC32;
  quint32 CompressedSize;
  quint32 UncompressedSize;
} __attribute__((packed));

struct ZipEntryHeader
{
  quint32 Signature;
  quint16 VersionToExtract;
  quint16 GeneralPurposeBitFlag;
  quint16 CompressionMethod;
  quint16 FileLastModificationTime;
  quint16 FileLastModificationDate;
  ZipDataDescriptor DataDescriptor;
  quint16 FilenameLength;
  quint16 ExtraFieldLength;
  // quint8 Filename[FilenameLength]
  // quint8 ExtraField[ExtraFieldLength]
} __attribute__((packed));

inline void skip(QFile &file, qint64 bytes)
{
    file.seek(file.pos() + bytes);
}

bool RomInfo::isAvailable()
{
    return !m_title.isNull();
}

QString RomInfo::title()
{
    return m_title;
}

QString RomInfo::code()
{
    return m_code;
}

QString RomInfo::maker()
{
    return m_maker;
}

RomInfo::RomInfo(QString romFile)
{
    if (romFile.right(4).toLower() == ".zip")
        extractZip(romFile);
    else
        extractGba(romFile);
}

void RomInfo::parseRomInfo(const char *data)
{
    // Validate the data
    if (data[12+4+2] == 0x96) {
        // Seems valid, parse as ROM info
        m_title = QString::fromAscii(QByteArray(data, 12));
        m_code  = QString::fromAscii(QByteArray(data+12, 4));
        m_maker = QString::fromAscii(QByteArray(data+12+4, 2));
    }
}

void RomInfo::extractGba(QString romFile)
{
    QFile rom(romFile);
    rom.open(QIODevice::ReadOnly);

    char *romHeaderPart = new char[12+4+2+1];

    rom.seek(ROM_INFO_OFFSET);
    if (rom.read(romHeaderPart, 12+4+2+1) == 12+4+2+1)
        parseRomInfo(romHeaderPart);

    delete[] romHeaderPart;
}

void RomInfo::extractZip(QString zipFilename)
{
    ZipEntryHeader zipHeader;

    QFile zipFile(zipFilename);
    zipFile.open(QIODevice::ReadOnly);

    // Look for GBA ROMs, allow only zip headers on the way
    bool searching = true;
    while (searching &&
           sizeof(zipHeader) == zipFile.read((char*) &zipHeader, sizeof(zipHeader)) &&
           zipHeader.Signature == 0x04034b50) {

        // Check if the data descriptor was a dummy
        if (zipHeader.GeneralPurposeBitFlag & 0x8) {
            // Appended descriptor not supported by us, abort
            searching = false;
        } else {
            // Get the filename of the compressed file
            QString filename(zipFile.read(zipHeader.FilenameLength));

            // Skip the rest of the header
            skip(zipFile, zipHeader.ExtraFieldLength);

            // Only GBA ROMs are interesting, check by extension
            if (filename.right(4).toLower() == ".gba") {
                // There you are!
                searching = false;

                // Allocate space for the required part of ROM header
                char *romHeaderPart = new char[ROM_INFO_OFFSET+12+4+2+1];

                // Determine compression type
                if (zipHeader.CompressionMethod == 0) {
                    // No compression, simply copy the data
                    zipFile.seek(zipFile.pos() + ROM_INFO_OFFSET);
                    if (zipFile.read(romHeaderPart, 12+4+2+1) == 12+4+2+1)
                        parseRomInfo(romHeaderPart);
                } else { // zipHeader.CompressionMethod == 8, at least it should
                    // Allocate space for the deflated input
                    char *buffer = new char[READ_BUFFER_SIZE];

                    // Initialize the data stream
                    z_stream stream;
                    stream.next_in = (Bytef*) buffer;
                    stream.next_out = (Bytef*) romHeaderPart;
                    stream.avail_in = zipFile.read(buffer, READ_BUFFER_SIZE);
                    stream.avail_out = ROM_INFO_OFFSET+12+4+2+1;
                    stream.zalloc = Z_NULL;
                    stream.zfree = Z_NULL;

                    // Initialize for raw input
                    int error = inflateInit2(&stream, -MAX_WBITS);

                    if (error == Z_OK) {
                        // Decompress data until the whole header is filled
                        while (error == Z_OK && stream.avail_out > 0) {
                            // Do the actual inflation
                            error = inflate(&stream, Z_SYNC_FLUSH);

                            // If there is not enough input data, feed more
                            if (error == Z_BUF_ERROR) {
                                stream.avail_in = zipFile.read(buffer, READ_BUFFER_SIZE);
                                // There's a possibility that reading failed
                                if (stream.avail_in > 0) {
                                    // Reset the position in the buffer
                                    stream.next_in = (Bytef*) buffer;
                                    // Error is fixed, reset the notification
                                    error = Z_OK;
                                }
                            }
                        }

                        // Deinitialize the data stream
                        inflateEnd(&stream);

                        // Check for completeness of the header
                        if (stream.avail_out == 0)
                            parseRomInfo(romHeaderPart+0xA0);
                    }

                    delete[] buffer;
                }

                delete[] romHeaderPart;
            } else {
                // The file is not a GBA ROM, move on
                skip(zipFile, zipHeader.DataDescriptor.CompressedSize);
            }
        }
    }
}
