#include "mainwindow.h"

MainWindow::MainWindow() :
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    ui->searchCloseButton->setIcon(QIcon::fromTheme("general_close"));

    this->setAttribute(Qt::WA_Maemo5StackedWindow);
    this->setAttribute(Qt::WA_Maemo5AutoOrientation, true);

    // Initialize the ROM list model
    romModel = new QStandardItemModel(this);
    romProxyModel = new QSortFilterProxyModel(this);
    romProxyModel->setFilterCaseSensitivity(Qt::CaseInsensitive);
    romProxyModel->setSourceModel(romModel);
    ui->romList->setModel(romProxyModel);
    ui->romList->setItemDelegate(new RomDelegate(ui->romList));

    // Watch for changes in the directory with ROMs
    romWatcher = new QFileSystemWatcher(this);
    romWatcher->addPath(Path::romDir());
    connect(romWatcher, SIGNAL(directoryChanged(QString)), this, SLOT(listRoms()));

    // Search box
    connect(ui->searchEdit, SIGNAL(textChanged(QString)), this, SLOT(onSearchTextChanged(QString)));
    connect(ui->searchCloseButton, SIGNAL(clicked()), this, SLOT(closeSearch()));

    // Connect menu actions
    connect(ui->openAction, SIGNAL(triggered()), this, SLOT(openFile()));
    connect(ui->settingsAction, SIGNAL(triggered()), this, SLOT(openSettingsDialog()));
    connect(ui->coreAction, SIGNAL(triggered()), this, SLOT(openCoreDialog()));
    connect(ui->keysAction, SIGNAL(triggered()), this, SLOT(openKeysDialog()));

    // Connect ROM list
    connect(ui->romList, SIGNAL(activated(QModelIndex)), this, SLOT(onRomActivated(QModelIndex)));

    ui->romList->viewport()->installEventFilter(this);

    listRoms();
}

MainWindow::~MainWindow()
{
    delete ui;
}

// Rrefresh the list of roms
void MainWindow::listRoms()
{
    ui->romList->clearSelection();

    QStringList filters = QStringList() << "*.gba" << "*.zip";

    QFileInfoList entries = QDir(Path::romDir()).entryInfoList(filters,
                                                               QDir::Files,
                                                               QDir::Name | QDir::IgnoreCase | QDir::LocaleAware);

    if (entries.isEmpty()) {
        ui->romList->hide();
        ui->emptyInfo->show();
    } else {
        ui->emptyInfo->hide();
        ui->romList->show();
    }

    int rowDelta = entries.size() - romModel->rowCount();

    if (rowDelta > 0) {
        while (rowDelta-- > 0)
            romModel->appendRow(new QStandardItem());
    } else {
        while (rowDelta++ < 0)
            romModel->removeRow(romModel->rowCount()-1);
    }

    for (int i = 0; i < entries.size(); i++) {
        QString baseName = entries.at(i).completeBaseName();
        QIcon icon(entries.at(i).absolutePath() + "/" + baseName + ".jpg");
        romModel->item(i)->setText(baseName);
        romModel->item(i)->setIcon(icon.availableSizes().isEmpty() ? QIcon() : icon);
        romModel->item(i)->setStatusTip(entries.at(i).absoluteFilePath());
    }

    // Refresh the filter, in case text in some items was modified
    romProxyModel->setFilterFixedString(ui->searchEdit->text());
}

void MainWindow::keyPressEvent(QKeyEvent *e)
{
    switch (e->key()) {
        case Qt::Key_Left:
        case Qt::Key_Right:
        case Qt::Key_Enter:
        case Qt::Key_Space:
        case Qt::Key_Shift:
        case Qt::Key_Control:
        case Qt::Key_Backspace:
            break;

        case Qt::Key_Up:
        case Qt::Key_Down:
            ui->romList->setFocus();
            break;

        default:
            ui->romList->clearSelection();
            if (!ui->searchEdit->hasFocus()) {
                ui->searchWidget->show();
                ui->searchEdit->setText(ui->searchEdit->text() + e->text());
                ui->searchEdit->setFocus();
            }
            break;
    }
}

bool MainWindow::eventFilter(QObject *, QEvent *e)
{
    if (e->type() == QEvent::MouseButtonPress
    && static_cast<QMouseEvent*>(e)->y() > ui->romList->viewport()->height() - 25)
        ui->searchWidget->show();

    return false;
}

// Show the search box
void MainWindow::onSearchTextChanged(QString text)
{
    romProxyModel->setFilterFixedString(text);

    if (text.isEmpty())
        ui->searchWidget->hide();
}

// Close the search box
void MainWindow::closeSearch()
{
    if (ui->searchEdit->text().isEmpty())
        ui->searchWidget->hide();
    else
        ui->searchEdit->clear();
}

void MainWindow::openFile()
{
    // File picker can take a while to load, prevent multiple clicks
    this->setEnabled(false);

    QString romFile = QFileDialog::getOpenFileName(this, tr("Select ROM"), Path::romDir(), tr("GBA ROMs (*.gba *.zip)"));

    if (!romFile.isNull())
        (new LaunchDialog(romFile, QIcon(), this))->exec();

    this->setEnabled(true);
}

void MainWindow::openSettingsDialog()
{
    (new SettingsDialog(this))->exec();

    // Clear the watchlist and add the possibly modified path
    romWatcher->removePaths(romWatcher->directories());
    romWatcher->addPath(Path::romDir());

    listRoms();
}

void MainWindow::openCoreDialog()
{
    CoreConfig config(Path::cfgFile());

    (new CoreDialog(&config, this))->exec();
}

void MainWindow::openKeysDialog()
{
    KeysConfig config(Path::cfgFile());

    (new KeysDialog(&config, this))->exec();

    // Launcher creates its own file later, this one is for running without GUI
    QDir().mkpath(Path::emuDir());
    config.write(Path::emuDir() + "keys");
}

void MainWindow::onRomActivated(QModelIndex index)
{
    (new LaunchDialog(index.data(Qt::StatusTipRole).toString(),
                      qvariant_cast<QIcon>(index.data(Qt::DecorationRole)),
                      this))->exec();
}
