/*
  GPSJinni - show raw data from the GPS subsystem.
  Copyright (C) 2009  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Compass.h"
#include "Util.h"

#include <cmath>

#include <Lum/OS/Font.h>

Compass::Compass()
 : arrowColorRight(0.0,0.0,0.0,Lum::OS::Display::blackColor),
   arrowColorLeft( 1.0,0.0,0.0,Lum::OS::Display::blackColor),
   arrowShadowColor(0.6,0.6,0.6,Lum::OS::Display::whiteColor),
   refreshTimer(new Lum::Model::Action()),
   lastDirection(0.0)
{
  Observe(refreshTimer);
}

bool Compass::SetModel(Lum::Base::Model* model)
{
  this->model=dynamic_cast<Lum::Model::Number*>(model);

  Lum::Control::SetModel(this->model);

  return this->model.Valid();
}

void Compass::SetShowCourseDirection(Lum::Model::Boolean *arrowShowCourse)
{
  if (this->arrowShowCourseDirection.Valid()) {
    Forget(this->arrowShowCourseDirection.Get());
  }

  this->arrowShowCourseDirection=arrowShowCourse;

  if (this->arrowShowCourseDirection.Valid()) {
    Observe(this->arrowShowCourseDirection.Get());
  }
}

void Compass::CalcSize()
{
  width=25;
  height=25;

  minWidth=width;
  minHeight=height;

  Lum::Control::CalcSize();
}

void Compass::Draw(Lum::OS::DrawInfo* draw,
                   int x, int y, size_t w, size_t h)
{
  Lum::Control::Draw(draw,x,y,w,h);

  if (!Intersect(x,y,w,h)) {
    return;
  }

  /* --- */


  Lum::OS::FontRef          font=Lum::OS::display->GetFont();
  size_t                    directionBox=0;
  size_t                    area;
  size_t                    radius=0;
  std::vector<std::wstring> directions;
  bool                      arrowShowCourse=arrowShowCourseDirection.Valid() &&
                                            !arrowShowCourseDirection->IsNull() &&
                                            *arrowShowCourseDirection;

  directions.push_back(L"N");
  directions.push_back(L"E");
  directions.push_back(L"S");
  directions.push_back(L"W");

  for (size_t i=0; i<directions.size(); i++) {
    Lum::OS::FontExtent extent;

    font->StringExtent(directions[i],extent,Lum::OS::Font::bold);

    directionBox=std::max(directionBox,extent.width);
    directionBox=std::max(directionBox,extent.height);
  }

  area=std::min(width,height);

  radius=(area-
          2*std::max(Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceIntraObject),
                     Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceIntraObject))-
          2*directionBox)/2-1;

  int ax=this->x+(width-area)/2;
  int ay=this->y+(height-area)/2;
  int rx=ax+(area/2);
  int ry=ay+(area/2);

  // scale background
  draw->PushForeground(Lum::OS::Display::whiteColor);
  draw->FillArc(ax+(area-radius*2+1)/2,ay+(area-radius*2+1)/2,
                radius*2+1,radius*2+1,
                0,360*64);
  draw->PopForeground();

  // scale frame
#if 0
  // Hmm, this draws exactly on the circle border - almost,
  // so the border becomes somewhat messy. Skip it.
  draw->PushForeground(Lum::OS::Display::blackColor);
  draw->DrawArc(ax+(area-radius*2+1)/2,ay+(area-radius*2+1)/2,
                radius*2+1,radius*2+1,
                0,360*64/8);
  draw->PopForeground();
#endif

  double direction=lastDirection;
  double roseDirection=0;
  double deviation=0;

  // have the validness of direction in a separate flag "directionOK",
  // so this will not change while we calculate with direction value.
  // The value "model" points to might be somewhat volatile.
  bool directionOK=model.Valid() && !model->IsNull() && !isnan(model->GetDouble());
  if (directionOK) {
    // the model value is course, the travel direction on north-aligned map.
    direction=model->GetDouble();
    // hack to transport deviation to here:
    if (model->GetMinAsDouble()==0.0) {
      deviation=model->GetMaxAsDouble()-360.0;
    }
  }

  // the next block implements a smooth moving arrow.
  {
    double smoothDirection=CalculateWeightedDirection(lastDirection,direction,0.2,360);

    if (fabs(lastDirection-smoothDirection) > 0.3) {
      Lum::OS::display->AddTimer(0,200,refreshTimer);
      direction = smoothDirection;
    }
    else if (! refreshTimer->IsFinished()) {
      Lum::OS::display->RemoveTimer(refreshTimer);
    }

    lastDirection = direction;
  }

  // direction values of north and the course as well
  {
    draw->PushForeground(Lum::OS::Display::textColor);
    draw->PushFont(font);

    int posX=0;
    int posY=font->ascent;

    draw->DrawString(this->x+posX,
                     this->y+posY,
                     L"Course: "+DoubleToString(direction)+L"\u00b0");
    posY+=font->height;
    draw->DrawString(this->x+posX,
                     this->y+posY,
                     L"North: "+DoubleToString(direction==0 ? 0
                                             : 360-direction)+L"\u00b0");
    posY+=font->height;
    draw->DrawString(this->x+posX,
                     this->y+posY,
                     L"Devia.: \u00b1"+DoubleToString(deviation/2)+L"\u00b0");

    draw->PopFont();
    draw->PopForeground();
  }

  if (!arrowShowCourse) {
    // convert to true north direction.
    direction=360-direction;
    roseDirection=direction;
  }

  // give subtle visual information about deviation
  {
    radius += 4;
    draw->PushForeground(Lum::OS::ColorRef(1.0,0.0,0.0,Lum::OS::Display::blackColor));
    draw->DrawArc(ax+(area-radius*2+1)/2,ay+(area-radius*2+1)/2,
                  radius*2+1,radius*2+1,
                  0,
                  360*64);
    draw->PopForeground();

    draw->PushForeground(Lum::OS::Display::whiteColor);
    draw->DrawArc(ax+(area-radius*2+1)/2,ay+(area-radius*2+1)/2,
                  radius*2+1,radius*2+1,
                  round(64*(90-direction-deviation/2)),
                  round(64*deviation));
    draw->PopForeground();
    radius -= 4;
  }

  int ar=(area-directionBox)/2;

  ax+=ar;
  ay+=ar;

  Lum::OS::DrawInfo::Point points[5];

  // directions, tied to arrow unless arrowShowCourse which means
  // directions are aligned to the device e.g. map.
  draw->PushForeground(Lum::OS::Display::textColor);
  draw->PushFont(font,Lum::OS::Font::bold);

  points[0].x=ax+lround(ar*sin(((roseDirection+  0)*M_PI)/180));
  points[0].y=ay-lround(ar*cos(((roseDirection+  0)*M_PI)/180));
  points[1].x=ax+lround(ar*sin(((roseDirection+ 90)*M_PI)/180));
  points[1].y=ay-lround(ar*cos(((roseDirection+ 90)*M_PI)/180));
  points[2].x=ax+lround(ar*sin(((roseDirection+180)*M_PI)/180));
  points[2].y=ay-lround(ar*cos(((roseDirection+180)*M_PI)/180));
  points[3].x=ax+lround(ar*sin(((roseDirection+270)*M_PI)/180));
  points[3].y=ay-lround(ar*cos(((roseDirection+270)*M_PI)/180));

  for (size_t i=0; i<4; i++) {
    Lum::OS::FontExtent extent;

    font->StringExtent(directions[i],extent,Lum::OS::Font::bold);

    draw->DrawString(points[i].x+(directionBox-extent.width) /2-extent.left,
                     points[i].y+(directionBox-extent.height)/2+extent.ascent,
                     directions[i]);
  }

  draw->PopFont();
  draw->PopForeground();

  // rose, never turning, always aligned to the map
  draw->PushForeground(arrowColorRight);
  for (int rose=0; rose<360; rose+=15) {
    size_t rrose=100;

    switch (rose % 90) {
    case  0:
      rrose -= 11;   // FALL THRU
    case 45:
      rrose -= 11;   // FALL THRU
    default:
      rrose -= 11;   // FALL THRU
    }

    rrose*=radius;
    rrose/=100;

    points[0].x=rx+lround(radius*sin(((rose+0)*M_PI)/180));
    points[0].y=ry-lround(radius*cos(((rose+0)*M_PI)/180));
    points[1].x=rx+lround( rrose*sin(((rose+0)*M_PI)/180));
    points[1].y=ry-lround( rrose*cos(((rose+0)*M_PI)/180));

    draw->DrawPolygon(points, 2);
  }
  draw->PopForeground();

  // arrow, compass needle if arrowShowCourse or travel direction on aligned map otherwise
  points[0].x=rx+lround(radius*90/100*sin(((direction+  0)*M_PI)/180));
  points[0].y=ry-lround(radius*90/100*cos(((direction+  0)*M_PI)/180));
  points[1].x=rx+lround(radius*90/100*sin(((direction+160)*M_PI)/180));
  points[1].y=ry-lround(radius*90/100*cos(((direction+160)*M_PI)/180));
  points[2].x=rx+lround(radius*50/100*sin(((direction+180)*M_PI)/180));
  points[2].y=ry-lround(radius*50/100*cos(((direction+180)*M_PI)/180));
  points[3].x=rx+lround(radius*90/100*sin(((direction-160)*M_PI)/180));
  points[3].y=ry-lround(radius*90/100*cos(((direction-160)*M_PI)/180));
  points[4]=points[0];

  for (size_t i=0; i<5; i++) {
    points[i].x+=3;
    points[i].y+=3;
  }

  draw->PushForeground(arrowShadowColor);
  draw->FillPolygon(points,5);
  draw->PopForeground();

  for (size_t i=0; i<5; i++) {
    points[i].x-=3;
    points[i].y-=3;
  }

  if (!directionOK) {
    // no arrow in this case - only shadow :-)
    return;
  }

  draw->PushForeground(arrowColorLeft);
  draw->FillPolygon(points,5);
  draw->PopForeground();

  draw->PushForeground(arrowColorRight);
  points[3] = points[0];
  draw->FillPolygon(points,4);
  draw->PopForeground();
}

void Compass::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==this->model ||
      model==this->arrowShowCourseDirection ||
      model==this->refreshTimer) {
    Redraw();
  }

  Control::Resync(model,msg);
}

Compass* Compass::Create(Lum::Base::Model* model,
                         bool horizontalFlex,
                         bool verticalFlex,
                         Lum::Model::Boolean* arrowShowCourse)
{
  Compass *compass=new Compass();

  compass->SetFlex(horizontalFlex,verticalFlex);
  compass->SetModel(model);
  compass->SetShowCourseDirection(arrowShowCourse);

  return compass;
}

