/*
  GPSJinni - show raw data from the GPS subsystem.
  Copyright (C) 2010  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Tracks.h"

#include <netinet/in.h>

#include <cmath>
#include <iomanip>
#include <iostream>

#include <Lum/Base/Path.h>

#include <Lum/Dlg/File.h>
#include <Lum/Dlg/Msg.h>
#include <Lum/Dlg/Value.h>

#include <Lum/Manager/FileSystem.h>

#include <Lum/Model/String.h>

#include "config.h"

static unsigned long trackingFileId='G'*256*256*256+'J'*256*256+'T'*256+'\0';
static unsigned long trackingFileVersion=2;

struct DataPoint
{
  Lum::Base::SystemTime time;
  double                lat;
  double                lon;
  double                alt;
};

bool StartTracking(Lum::OS::Window* parent,
                   std::ofstream& trackFile)
{
  Lum::Model::StringRef trackName=new Lum::Model::String(L"");
  std::wstring          homeDir;
  Lum::Base::Path       fileName;

  if (!Lum::Dlg::TextInput::GetText(parent,
                                    L"Enter track name!",
                                    L"Please enter a name for the to be recorded track.",
                                    trackName) || trackName->Empty()) {
    return false;
  }

  if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDir,
      homeDir)) {
    return false;
  }

  fileName.SetNativeDir(homeDir);
  fileName.AppendDir(L".gpsjinni");

  Lum::Base::Path::CreateDir(fileName.GetPath());

  fileName.SetBaseName(trackName->Get()+L".gjt");

  if (Lum::Base::Path::Exists(fileName.GetPath())) {
    Lum::Dlg::Msg::ShowOk(parent,
                          L"Error while opening file!",
                          std::wstring(L"An error occured while opening file:\n")+
                          L"'"+fileName.GetPath()+L"'\n"+
                          L"Error:\n"+
                          L"'File already exists'");
    return false;
  }

  trackFile.open(Lum::Base::WStringToString(fileName.GetPath()).c_str(),
                 std::ios::out|std::ios::trunc|std::ios::binary);

  if (!trackFile) {
    Lum::Base::Status status;

    status.SetToCurrentErrno();
    Lum::Dlg::Msg::ShowOk(parent,
                          L"Error while opening file!",
                          std::wstring(L"An error occured while opening file:\n")+
                          L"'"+fileName.GetPath()+L"'\n"+
                          L"Error:\n"+
                          L"'"+status.GetDescription()+L"'");

    return false;
  }

  unsigned long tfid=htonl(trackingFileId);
  unsigned long tfv=htonl(trackingFileVersion);
  unsigned long tft=htonl(Lum::Base::SystemTime().GetTime());

  trackFile.write((const char*)&tfid,sizeof(tfid));
  trackFile.write((const char*)&tfv,sizeof(tfv));
  trackFile.write((const char*)&tft,sizeof(tft));

  return true;
}

bool AddTrackingRecord(std::ofstream& trackFile,
                       const GPSData& data)
{
  if (!data.HasFix()) {
    return false;
  }

  double                lon=0.0;
  double                lat=0.0;
  double                alt=0.0;
  Lum::Base::SystemTime tim;

  if (data.HasLongitude()) {
    lon=data.GetLongitude();
  }

  if (data.HasLatitude()) {
    lat=data.GetLatitude();
  }

  if (data.HasAltitude()) {
    alt=data.GetAltitude();
  }

  if (data.HasTime()) {
    tim=data.GetTime();
  }

  unsigned long flon=htonl((unsigned long)rint((lon+180.0)*10000000.0));
  unsigned long flat=htonl((unsigned long)rint((lat+90.0)*10000000.0));
  unsigned long falt=htonl((unsigned long)rint(alt*100));
  unsigned long ftim=htonl(tim.GetTime());

  trackFile.write((const char*)&flon,sizeof(flon));
  trackFile.write((const char*)&flat,sizeof(flat));
  trackFile.write((const char*)&falt,sizeof(falt));
  trackFile.write((const char*)&ftim,sizeof(ftim));

  return true;
}

unsigned long GetFileSize(std::ofstream& trackFile)
{
  return trackFile.tellp();
}

void RefreshTracks(TracksModel* tracks)
{
  Lum::Base::Path              dirName,fileName;
  Lum::Base::DirScannerRef     scanner;
  Lum::Base::DirScanner::Entry entry;
  Lum::Base::Status            status;
  std::wstring                 homeDir;

  tracks->Clear();
  tracks->Off();

  if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDir,
      homeDir)) {
    return;
  }

  dirName.SetNativeDir(homeDir);
  dirName.AppendDir(L".gpsjinni");

  scanner=Lum::Base::DirScanner::Create(dirName);

  while (scanner->GetNext(entry,status)) {
    if (entry.isDir) {
      continue;
    }

    std::cout << "Found possible track '" << Lum::Base::WStringToString(entry.name) << "'" << std::endl;

    if (entry.name.length()<4 ||
        entry.name.substr(entry.name.length()-4)!=L".gjt") {
      std::cerr << "wrong suffix!" << std::endl;
      continue;
    }

    std::ifstream file;

    fileName=dirName;
    fileName.SetBaseName(entry.name);

    file.open(Lum::Base::WStringToString(fileName.GetPath()).c_str(),
              std::ios::in|std::ios::binary);

    if (!file) {
      std::cerr << "Cannot open file!" << std::endl;
      continue;
    }

    unsigned long tfid,tfv,tft;

    file.read((char*)&tfid,sizeof(tfid));
    file.read((char*)&tfv,sizeof(tfv));
    file.read((char*)&tft,sizeof(tft));

    if (!file) {
      std::cerr << "Cannot read file header!" << std::endl;
      continue;
    }

    tfid=ntohl(tfid);
    tfv=ntohl(tfv);
    tft=ntohl(tft);

    if (tfid!=trackingFileId) {
      std::cerr << "Wrong magic!" << std::endl;
      continue;
    }

    /* We support the current version and version 1 */
    if (tfv!=trackingFileVersion && tfv!=1) {
      std::cerr << "Wrong file version!" << std::endl;
      continue;
    }

    tracks->Append(Track(entry.name.substr(0,entry.name.length()-4),
                         Lum::Base::SystemTime(tft)));
  }

  tracks->Sort(1);

  tracks->On();
}

bool DeleteTrack(Lum::OS::Window* parent,
                 const std::wstring& name)
{
  Lum::Base::Path   path;
  Lum::Base::Status status;
  std::wstring      homeDir;

  if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDir,
                                                      homeDir)) {
    return false;
  }

  path.SetNativeDir(homeDir);
  path.AppendDir(L".gpsjinni");
  path.SetBaseName(name+L".gjt");

  status=Lum::Base::Path::RemoveFile(path.GetPath());

  if (!status) {
    Lum::Dlg::Msg::ShowOk(parent,
                          L"Error while deleting file!",
                          std::wstring(L"An error occured while deletion of file:\n")+
                          L"'"+path.GetPath()+L"'\n"+
                          L"Error:\n"+
                          L"'"+status.GetDescription()+L"'");
    return false;
  }

  return true;
}

void ExportTrackToKML(Lum::OS::Window* parent,
                      const std::wstring& name)
{
  Lum::Base::Path                path;
  std::wstring                   file;
  Lum::Dlg::File::Options        *options=new Lum::Dlg::File::Options();
  Lum::Model::Dir::PatternFilter *filter;
  std::ifstream                  in;
  std::ofstream                  out;
  std::wstring                   homeDir;

  if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDir,
      homeDir)) {
    return;
  }

  path.SetNativeDir(homeDir);
  path.AppendDir(L".gpsjinni");
  path.SetBaseName(name+L".gjt");

  in.open(Lum::Base::WStringToString(path.GetPath()).c_str(),
          std::ios::in|std::ios::binary);

  if (!in) {
    Lum::Base::Status status;

    status.SetToCurrentErrno();
    Lum::Dlg::Msg::ShowOk(parent,
                          L"Error while opening file!",
                          std::wstring(L"An error occured while opening file:\n")+
                          L"'"+file+L"'\n"+
                          L"Error:\n"+
                          L"'"+status.GetDescription()+L"'");

    return;
  }

  unsigned long tfid,tfv,tft;

  in.read((char*)&tfid,sizeof(tfid));
  in.read((char*)&tfv,sizeof(tfv));
  in.read((char*)&tft,sizeof(tft));

  if (!in) {
    std::cerr << "Cannot read file header!" << std::endl;
    return;
  }

  tfid=ntohl(tfid);
  tfv=ntohl(tfv);
  tft=ntohl(tft);

  if (tfid!=trackingFileId) {
    Lum::Dlg::Msg::ShowOk(parent,
                          L"Error while opening file!",
                          std::wstring(L"An error occured while opening file:\n")+
                          L"'"+file+L"'\n"+
                          L"Error:\n"+
                          L"'File is not a stored track'");
    return;
  }

  /* We support the current version and version 1 */
  if (tfv!=trackingFileVersion && tfv!=1) {
    Lum::Dlg::Msg::ShowOk(parent,
                          L"Error while opening file!",
                          std::wstring(L"An error occured while opening file:\n")+
                          L"'"+file+L"'\n"+
                          L"Error:\n"+
                          L"'File format version not supported'");
    return;
  }

  std::cout << "Track data: " << std::hex << tfid << std::dec << " " << tfv << " " << tft << std::endl;

  std::wstring docsDir;

  if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDocumentDir,
                                                      docsDir)) {
    if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDir,
                                                        docsDir)) {
      return;
    }
  }

  path.SetNativeDirAndFile(docsDir,name+L".kml");

  file=path.GetPath();

  options->SetMode(Lum::Dlg::File::Options::modeFile);
  options->SetExistingOnly(false);

  filter=new Lum::Model::Dir::PatternFilter(L"KML Files");
  filter->AddPattern(L"*.kml");
  options->AddFilter(filter);

  if (!Lum::Dlg::File::SaveFile(parent,
                                L"Exporting track...",
                                file,
                                options)) {
    return;
  }

  out.open(Lum::Base::WStringToString(file).c_str(),
           std::ios::out|std::ios::trunc);

  if (!out) {
    Lum::Base::Status status;

    status.SetToCurrentErrno();
    Lum::Dlg::Msg::ShowOk(parent,
                          L"Error while opening file!",
                          std::wstring(L"An error occured while opening file:\n")+
                          L"'"+file+L"'\n"+
                          L"Error:\n"+
                          L"'"+status.GetDescription()+L"'");

    return;
  }

  std::list<DataPoint> data;

  while (in) {
    unsigned long flon;
    unsigned long flat;
    unsigned long falt;
    unsigned long ftim;

    in.read((char*)&flon,sizeof(flon));
    in.read((char*)&flat,sizeof(flat));
    in.read((char*)&falt,sizeof(falt));
    in.read((char*)&ftim,sizeof(ftim));

    if (in) {
      DataPoint dataPoint;

      flon=ntohl(flon);
      flat=ntohl(flat);
      falt=ntohl(falt);
      ftim=ntohl(ftim);

      //std::cout << flon/10000000.0 << " " << flat/10000000.0 << " " << falt/100.0 << " " << ftim << std::endl;

      if (tfv==1) {
        dataPoint.lon=flon/10000000.0-90.0;
        dataPoint.lat=flat/10000000.0-180.0;
      }
      else {
        dataPoint.lon=flon/10000000.0-180.0;
        dataPoint.lat=flat/10000000.0-90.0;
      }
      dataPoint.alt=falt/100.0;
      dataPoint.time.SetTime(ftim);

      data.push_back(dataPoint);
    }
  }

  out << std::fixed;

  out << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << std::endl;
  out << "<kml xmlns=\"http://www.opengis.net/kml/2.2\">" << std::endl;
  out << "  <Document>" << std::endl;
  out << "    <name>" << Lum::Base::WStringToUTF8(name) << ".kml" << "</name>" << std::endl;
  out << "    <open>1</open>" << std::endl;
  out << std::endl;
  out << "    <Style id=\"startpointStyle\">" << std::endl;
  out << "      <IconStyle>" << std::endl;
  out << "        <Icon>" << std::endl;
  out << "          <href>http://maps.google.com/mapfiles/kml/paddle/A.png</href>" << std::endl;
  out << "        </Icon>" << std::endl;
  out << "      </IconStyle>" << std::endl;
  out << "    </Style>" << std::endl;
  out << std::endl;
  out << "    <Style id=\"waypointStyle\">" << std::endl;
  out << "      <IconStyle>" << std::endl;
  out << "        <Icon>" << std::endl;
  out << "          <href>http://maps.google.com/mapfiles/kml/pushpin/ylw-pushpin.png</href>" << std::endl;
  out << "        </Icon>" << std::endl;
  out << "      </IconStyle>" << std::endl;
  out << "    </Style>" << std::endl;
  out << std::endl;
  out << "    <Style id=\"endpointStyle\">" << std::endl;
  out << "      <IconStyle>" << std::endl;
  out << "        <Icon>" << std::endl;
  out << "          <href>http://maps.google.com/mapfiles/kml/paddle/B.png</href>" << std::endl;
  out << "        </Icon>" << std::endl;
  out << "      </IconStyle>" << std::endl;
  out << "    </Style>" << std::endl;
  out << std::endl;

  out << "    <Folder>" << std::endl;
  out << "      <name>" << Lum::Base::WStringToUTF8(name) << " waypoints with timestamps</name>" << std::endl;
  out << "      <open>1</open>" << std::endl;
  out << std::endl;
  out << "      <Style>" << std::endl;
  out << "        <ListStyle>" << std::endl;
  out << "          <listItemType>checkHideChildren</listItemType>" << std::endl;
  out << "        </ListStyle>" << std::endl;
  out << "      </Style>" << std::endl;

  for (std::list<DataPoint>::const_iterator dataPoint=data.begin();
       dataPoint!=data.end();
       ++dataPoint) {
    out << std::endl;
    out << "      <Placemark>" << std::endl;
    out << "        <TimeStamp>" << std::endl;
    out << "          <when>" << Lum::Base::WStringToString(dataPoint->time.GetUTCISO8601DateTime()) << "</when>" << std::endl;
    out << "        </TimeStamp>" << std::endl;
    out << "        <styleUrl>#waypointStyle</styleUrl>" << std::endl;
    out << "        <Point>" << std::endl;
    out << "          <coordinates>" << std::setprecision(8) << dataPoint->lon << ",";
    out << dataPoint->lat << "," << std::setprecision(3) << dataPoint->alt << "</coordinates>" << std::endl;
    out << "        </Point>" << std::endl;
    out << "      </Placemark>" << std::endl;
  }
  out << "    </Folder>" << std::endl;

  out << std::endl;
  out << std::endl;

  out << "    <Folder>" << std::endl;
  out << "      <name>" << Lum::Base::WStringToUTF8(name) << " path waypoints</name>" << std::endl;
  out << "      <open>1</open>" << std::endl;
  out << std::endl;
  out << "      <Style>" << std::endl;
  out << "        <ListStyle>" << std::endl;
  out << "          <listItemType>checkHideChildren</listItemType>" << std::endl;
  out << "        </ListStyle>" << std::endl;
  out << "      </Style>" << std::endl;

  for (std::list<DataPoint>::const_iterator dataPoint=data.begin();
       dataPoint!=data.end();
       ++dataPoint) {
    out << std::endl;
    out << "      <Placemark>" << std::endl;
    out << "        <styleUrl>#waypointStyle</styleUrl>" << std::endl;
    out << "        <Point>" << std::endl;
    out << "          <coordinates>" << std::setprecision(8) << dataPoint->lon << ",";
    out << dataPoint->lat << "," << std::setprecision(3) << dataPoint->alt << "</coordinates>" << std::endl;
    out << "        </Point>" << std::endl;
    out << "      </Placemark>" << std::endl;
  }
  out << "    </Folder>" << std::endl;

  out << std::endl;
  out << std::endl;

  if (data.size()>0) {
    out << "    <Placemark>" << std::endl;
    out << "      <styleUrl>#startpointStyle</styleUrl>" << std::endl;
    out << "      <name>Start</name>" << std::endl;
    out << "      <Point>" << std::endl;
    out << "        <coordinates>" << std::setprecision(8) << data.begin()->lon << ",";
    out << data.begin()->lat << "," << std::setprecision(3) << data.begin()->alt << "</coordinates>" << std::endl;
    out << "      </Point>" << std::endl;
    out << "    </Placemark>" << std::endl;
    out << std::endl;
    out << "    <Placemark>" << std::endl;
    out << "      <styleUrl>#endpointStyle</styleUrl>" << std::endl;
    out << "      <name>End</name>" << std::endl;
    out << "      <Point>" << std::endl;
    out << "        <coordinates>" << std::setprecision(8) << data.rbegin()->lon << ",";
    out << data.rbegin()->lat << "," << std::setprecision(3) << data.rbegin()->alt << "</coordinates>" << std::endl;
    out << "      </Point>" << std::endl;
    out << "    </Placemark>" << std::endl;
    out << std::endl;
  }

  out << "    <Placemark>" << std::endl;
  out << "      <name>" << Lum::Base::WStringToUTF8(name) << " path</name>" << std::endl;
  out << std::endl;
  out << "      <Style>" << std::endl;
  out << "        <LineStyle>" << std::endl;
  out << "          <color>ff0000ff</color>" << std::endl;
  out << "          <width>2</width>" << std::endl;
  out << "        </LineStyle>" << std::endl;
  out << "      </Style>" << std::endl;
  out << std::endl;
  out << "      <LineString>" << std::endl;
  //out << "        <extrude>0</extrude>" << std::endl;
  out << "        <tessellate>1</tessellate>" << std::endl;
  out << "        <altitudeMode>clampToGround</altitudeMode>" << std::endl;
  out << "        <coordinates>" << std::endl;

  double lastLat=1000;
  double lastLon=1000;
  double lastAlt=1000;

  for (std::list<DataPoint>::const_iterator dataPoint=data.begin();
       dataPoint!=data.end();
       ++dataPoint) {
    if (dataPoint->lon!=lastLon ||
        dataPoint->lat!=lastLat ||
        dataPoint->alt!=lastAlt) {
      out << "          " << std::setprecision(8) << dataPoint->lon << ",";
      out << dataPoint->lat << "," << std::setprecision(3) << dataPoint->alt << std::endl;
      lastLon=dataPoint->lon;
      lastLat=dataPoint->lat;
      lastAlt=dataPoint->alt;
    }
  }

  out << "        </coordinates>" << std::endl;
  out << "      </LineString>" << std::endl;
  out << "    </Placemark>" << std::endl;
  out << "  </Document>" << std::endl;
  out << "</kml>" << std::endl;

  out.close();
  in.close();
}

void ExportTrackToGPX(Lum::OS::Window* parent,
                      const std::wstring& name)
{
  Lum::Base::Path                path;
  std::wstring                   file;
  Lum::Dlg::File::Options        *options=new Lum::Dlg::File::Options();
  Lum::Model::Dir::PatternFilter *filter;
  std::ifstream                  in;
  std::ofstream                  out;
  std::wstring                   homeDir;

  if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDir,
      homeDir)) {
    return;
  }

  path.SetNativeDir(homeDir);
  path.AppendDir(L".gpsjinni");
  path.SetBaseName(name+L".gjt");

  in.open(Lum::Base::WStringToString(path.GetPath()).c_str(),
          std::ios::in|std::ios::binary);

  if (!in) {
    Lum::Base::Status status;

    status.SetToCurrentErrno();
    Lum::Dlg::Msg::ShowOk(parent,
                          L"Error while opening file!",
                          std::wstring(L"An error occured while opening file:\n")+
                          L"'"+file+L"'\n"+
                          L"Error:\n"+
                          L"'"+status.GetDescription()+L"'");

    return;
  }

  unsigned long tfid,tfv,tft;

  in.read((char*)&tfid,sizeof(tfid));
  in.read((char*)&tfv,sizeof(tfv));
  in.read((char*)&tft,sizeof(tft));

  if (!in) {
    std::cerr << "Cannot read file header!" << std::endl;
    return;
  }

  tfid=ntohl(tfid);
  tfv=ntohl(tfv);
  tft=ntohl(tft);

  if (tfid!=trackingFileId) {
    Lum::Dlg::Msg::ShowOk(parent,
                          L"Error while opening file!",
                          std::wstring(L"An error occured while opening file:\n")+
                          L"'"+file+L"'\n"+
                          L"Error:\n"+
                          L"'File is not a stored track'");
    return;
  }

  /* We support the current version and version 1 */
  if (tfv!=trackingFileVersion && tfv!=1) {
    Lum::Dlg::Msg::ShowOk(parent,
                          L"Error while opening file!",
                          std::wstring(L"An error occured while opening file:\n")+
                          L"'"+file+L"'\n"+
                          L"Error:\n"+
                          L"'File format version not supported'");
    return;
  }

  std::cout << "Track data: " << std::hex << tfid << std::dec << " " << tfv << " " << tft << std::endl;

  std::wstring docsDir;

  if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDocumentDir,
                                                      docsDir)) {
    if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDir,
                                                        docsDir)) {
      return;
    }
  }

  path.SetNativeDirAndFile(docsDir,name+L".gpx");

  file=path.GetPath();

  options->SetMode(Lum::Dlg::File::Options::modeFile);
  options->SetExistingOnly(false);

  filter=new Lum::Model::Dir::PatternFilter(L"GPX Files");
  filter->AddPattern(L"*.gpx");
  options->AddFilter(filter);

  if (!Lum::Dlg::File::SaveFile(parent,
                                L"Exporting track...",
                                file,
                                options)) {
    return;
  }

  out.open(Lum::Base::WStringToString(file).c_str(),
           std::ios::out|std::ios::trunc);

  if (!out) {
    Lum::Base::Status status;

    status.SetToCurrentErrno();
    Lum::Dlg::Msg::ShowOk(parent,
                          L"Error while opening file!",
                          std::wstring(L"An error occured while opening file:\n")+
                          L"'"+file+L"'\n"+
                          L"Error:\n"+
                          L"'"+status.GetDescription()+L"'");

    return;
  }

  std::list<DataPoint> data;

  while (in) {
    unsigned long flon;
    unsigned long flat;
    unsigned long falt;
    unsigned long ftim;

    in.read((char*)&flon,sizeof(flon));
    in.read((char*)&flat,sizeof(flat));
    in.read((char*)&falt,sizeof(falt));
    in.read((char*)&ftim,sizeof(ftim));

    if (in) {
      DataPoint dataPoint;

      flon=ntohl(flon);
      flat=ntohl(flat);
      falt=ntohl(falt);
      ftim=ntohl(ftim);

      //std::cout << flon/10000000.0 << " " << flat/10000000.0 << " " << falt/100.0 << " " << ftim << std::endl;

      if (tfv==1) {
        dataPoint.lon=flon/10000000.0-90.0;
        dataPoint.lat=flat/10000000.0-180.0;
      }
      else {
        dataPoint.lon=flon/10000000.0-180.0;
        dataPoint.lat=flat/10000000.0-90.0;
      }
      dataPoint.alt=falt/100.0;
      dataPoint.time.SetTime(ftim);

      data.push_back(dataPoint);
    }
  }

  Lum::Base::SystemTime now;

  out << std::fixed;

  out << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << std::endl;
  out << "<gpx" << std::endl;
  out << " version=\"1.0\"" << std::endl;
  out << " creator=\"GPSJinni " << PACKAGE_VERSION << "\"" << std::endl;
  out << " xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"" << std::endl;
  out << " xmlns=\"http://www.topografix.com/GPX/1/0\"" << std::endl;
  out << " xsi:schemaLocation=\"http://www.topografix.com/GPX/1/0 http://www.topografix.com/GPX/1/0/gpx.xsd\">" << std::endl;
  out << std::endl;
  out << "  <name>" << Lum::Base::WStringToString(name) << "</name>" << std::endl;
  out << "  <time>" << Lum::Base::WStringToString(now.GetUTCISO8601DateTime()) << "</time>" << std::endl;
  out << std::endl;
  out << "  <trk>" << std::endl;
  out << "    <name>" << Lum::Base::WStringToString(name) << "</name>" << std::endl;
  out << "    <trkseg>" << std::endl;

  double lastLat=1000;
  double lastLon=1000;
  double lastAlt=1000;

  for (std::list<DataPoint>::const_iterator dataPoint=data.begin();
       dataPoint!=data.end();
       ++dataPoint) {
    if (dataPoint->lon!=lastLon ||
        dataPoint->lat!=lastLat ||
        dataPoint->alt!=lastAlt) {
      out << "      <trkpt lat=\"" << std::setprecision(8) << dataPoint->lat << "\"";
      out << " lon=\"" << std::setprecision(8) << dataPoint->lon << "\">" << std::endl;
      out << "        <ele>" << std::setprecision(3) << dataPoint->alt << "</ele>" << std::endl;
      out << "        <time>" << Lum::Base::WStringToString(dataPoint->time.GetUTCISO8601DateTime()) << "</time>" << std::endl;
      out << "      </trkpt>" << std::endl;

      lastLon=dataPoint->lon;
      lastLat=dataPoint->lat;
      lastAlt=dataPoint->alt;
    }
  }

  out << "    </trkseg>" << std::endl;
  out << "  </trk>" << std::endl;
  out << "</gpx>" << std::endl;

  out.close();
  in.close();
}

