/*
  GPSJinni - show raw data from the GPS subsystem.
  Copyright (C) 2009  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "GPSData.h"

#include <cassert>
#include <limits>

#include "config.h"

// maximum age of data in seconds until it is invalid
#if defined(HAVE_LIB_GPS)
// Using normal libgps we don't trust situation if GPS daemon is not
// responding in time
static long obsoleteDataAge=30;
#elif defined(HAVE_LIB_LOCATION)
// Using liblocation on the other hand we trust it to send diffs
// every time something relvant changes.
// Data does not age!
static long obsoleteDataAge=std::numeric_limits<long>::max();
#else
static long obsoleteDataAge=0;
#endif

GPSData::GPSData()
 : connectedLastData(0),
   onlineLastData(0),
   fixLastData(0),
   latitudeLastData(0),
   longitudeLastData(0),
   altitudeLastData(0),
   speedLastData(0),
   trackLastData(0),
   climbLastData(0),
   timeLastData(0),
   satellitesLastData(0)
{
  // no code
}

//
// Connected
//
void GPSData::SetConnected(bool connected)
{
  connectedLastData.SetToNow();
  this->connected=connected;
}

bool GPSData::HasConnected() const
{
  Lum::Base::SystemTime now;

  return now-connectedLastData<obsoleteDataAge;
}

bool GPSData::GetConnected() const
{
  assert(HasConnected());

  return connected;
}

//
// Online
//
void GPSData::SetOnline(bool online)
{
  onlineLastData.SetToNow();
  this->online=online;
}

bool GPSData::HasOnline() const
{
  Lum::Base::SystemTime now;

  return now-onlineLastData<obsoleteDataAge;
}

bool GPSData::GetOnline() const
{
  assert(HasOnline());

  return online;
}

//
// Fix
//
void GPSData::SetFix(bool fix)
{
  fixLastData.SetToNow();
  this->fix=fix;
}

bool GPSData::HasFix() const
{
  Lum::Base::SystemTime now;

  return now-fixLastData<obsoleteDataAge;
}

bool GPSData::GetFix() const
{
  assert(HasFix());

  return fix;
}

//
// Latitude
//
void GPSData::SetLatitude(double latitude,
                          double latitudeDev)
{
  latitudeLastData.SetToNow();
  this->latitude=latitude;
  this->latitudeDev=latitudeDev;
}

bool GPSData::HasLatitude() const
{
  Lum::Base::SystemTime now;

  return now-latitudeLastData<obsoleteDataAge;
}

double GPSData::GetLatitude() const
{
  assert(HasLatitude());

  return latitude;
}

double GPSData::GetLatitudeDev() const
{
  assert(HasLatitude());

  return latitudeDev;
}

//
// Longitude
//
void GPSData::SetLongitude(double longitude,
                           double longitudeDev)
{
  longitudeLastData.SetToNow();
  this->longitude=longitude;
  this->longitudeDev=longitudeDev;
}

bool GPSData::HasLongitude() const
{
  Lum::Base::SystemTime now;

  return now-longitudeLastData<obsoleteDataAge;
}

double GPSData::GetLongitude() const
{
  assert(HasLongitude());

  return longitude;
}

double GPSData::GetLongitudeDev() const
{
  assert(HasLongitude());

  return longitudeDev;
}

//
// Altitude
//
void GPSData::SetAltitude(double altitude,
                          double altitudeDev)
{
  altitudeLastData.SetToNow();
  this->altitude=altitude;
  this->altitudeDev=altitudeDev;
}

bool GPSData::HasAltitude() const
{
  Lum::Base::SystemTime now;

  return now-altitudeLastData<obsoleteDataAge;
}

double GPSData::GetAltitude() const
{
  assert(HasAltitude());

  return altitude;
}

double GPSData::GetAltitudeDev() const
{
  assert(HasAltitude());

  return altitudeDev;
}

//
// Speed
//
void GPSData::SetSpeed(double speed,
                       double speedDev)
{
  speedLastData.SetToNow();
  this->speed=speed;
  this->speedDev=speedDev;
}

bool GPSData::HasSpeed() const
{
  Lum::Base::SystemTime now;

  return now-speedLastData<obsoleteDataAge;
}

double GPSData::GetSpeed() const
{
  assert(HasSpeed());

  return speed;
}

double GPSData::GetSpeedDev() const
{
  assert(HasSpeed());

  return speedDev;
}

//
// Track
//
void GPSData::SetTrack(double track,
                       double trackDev)
{
  trackLastData.SetToNow();
  this->track=track;
  this->trackDev=trackDev;
}

bool GPSData::HasTrack() const
{
  Lum::Base::SystemTime now;

  return now-trackLastData<obsoleteDataAge;
}

double GPSData::GetTrack() const
{
  assert(HasTrack());

  return track;
}

double GPSData::GetTrackDev() const
{
  assert(HasTrack());

  return trackDev;
}

//
// Climb
//
void GPSData::SetClimb(double climb,
                       double climbDev)
{
  climbLastData.SetToNow();
  this->climb=climb;
  this->climbDev=climbDev;
}

bool GPSData::HasClimb() const
{
  Lum::Base::SystemTime now;

  return now-climbLastData<obsoleteDataAge;
}

double GPSData::GetClimb() const
{
  assert(HasClimb());

  return climb;
}

double GPSData::GetClimbDev() const
{
  assert(HasClimb());

  return climbDev;
}

//
// Time
//
void GPSData::SetTime(Lum::Base::SystemTime& time,
                      double timeDev)
{
  timeLastData.SetToNow();
  this->time=time;
  this->timeDev=timeDev;
}

bool GPSData::HasTime() const
{
  Lum::Base::SystemTime now;

  return now-timeLastData<obsoleteDataAge;
}

Lum::Base::SystemTime GPSData::GetTime() const
{
  assert(HasTime());

  return time;
}

double GPSData::GetTimeDev() const
{
  assert(HasTime());

  return timeDev;
}

//
// Satellites
//
void GPSData::SetSatellites(const std::vector<Satellite>& satellites)
{
  satellitesLastData.SetToNow();
  this->satellites=satellites;
}

bool GPSData::HasSatellites() const
{
  Lum::Base::SystemTime now;

  return now-satellitesLastData<obsoleteDataAge;
}

const std::vector<GPSData::Satellite>& GPSData::GetSatellites() const
{
  assert(HasSatellites());

  return satellites;
}

