/*
  GPSJinni - show raw data from the GPS subsystem.
  Copyright (C) 2009  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <limits>

#include "config.h"

#include <signal.h>

#include <glib.h>

#include <cerrno>
#include <cmath>
#include <iomanip>
#include <iostream>

#if defined (HAVE_LIB_GPS)
  #include <gps.h>
#endif

#if defined(HAVE_LIB_LOCATION)
extern "C" {
  #include <location/location-gps-device.h>
  #include <location/location-gpsd-control.h>
}
#endif

#include <netinet/in.h>

#include <fstream>

#include <Lum/Def/Menu.h>
#include <Lum/Def/MultiView.h>

#include <Lum/Images/Image.h>
#include <Lum/Images/Loader.h>

#include <Lum/Manager/Behaviour.h>
#include <Lum/Manager/Display.h>
#include <Lum/Manager/FileSystem.h>

#include <Lum/OS/Probe.h>
#include <Lum/OS/Thread.h>

#include <Lum/Base/DateTime.h>
#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/File.h>
#include <Lum/Dlg/Msg.h>
#include <Lum/Dlg/Properties.h>
#include <Lum/Dlg/Value.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Boolean.h>
#include <Lum/Model/DataStream.h>
#include <Lum/Model/Header.h>
#include <Lum/Model/String.h>

#include <Lum/Application.h>
#include <Lum/Button.h>
#include <Lum/Dialog.h>
#include <Lum/Label.h>
#include <Lum/LED.h>
#include <Lum/Panel.h>
#include <Lum/PercentBar.h>
#include <Lum/Tab.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>
#include <Lum/View.h>

#include "Configuration.h"

#include "GPSData.h"
#include "Tracks.h"
#include "Compass.h"
#include "Speed.h"
#include "TimeLine.h"
#include "Util.h"

static long reconnectTime   = 10; // Period of time after which we check for disconnection and
                                  // trigger reconnection
static long refreshTime     = 10; // Period of time after which we update the stream views

typedef Lum::Model::StdTable<GPSData::Satellite> SatellitesModel;
typedef Lum::Base::Reference<SatellitesModel>    SatellitesModelRef;

double GetRelativeQualityFromDbm(double quality)
{
  if (quality<0) {
    return 0;
  }
  if (quality>=0 && quality<40) {
    return 5*quality/2;
  }
  else {
    return 100;
  }
}

class SatellitesComparator : public SatellitesModel::Comparator
{
public:
  bool operator()(const GPSData::Satellite& a,
                  const GPSData::Satellite& b,
                  size_t column,
                  bool ascending) const
  {
    assert(column==1);

    if (ascending) {
      return a.snr<b.snr;
    }
    else {
      return a.snr>b.snr;
    }
  }
};

class SatellitesModelPainter : public Lum::TableCellPainter
{
private:
  Lum::OS::ColorRef  green;
  Lum::OS::ColorRef  grey;

public:
  SatellitesModelPainter()
    : green(0,1,0,Lum::OS::display->GetColor(Lum::OS::Display::tableTextColor)),
      grey(0.7,0.7,0.7,Lum::OS::display->GetColor(Lum::OS::Display::tableTextColor))
  {
    // no code
  }

  size_t GetProposedHeight() const
  {
    return Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
           GetFont()->height+
           Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder);
  }

  void Draw(Lum::OS::DrawInfo* draw,
            int x, int y, size_t width, size_t height) const
  {
    const GPSData::Satellite& satellite=dynamic_cast<const SatellitesModel*>(GetModel())->GetEntry(GetRow());

    //
    // Signal strength bar
    //

    double percent=GetRelativeQualityFromDbm(satellite.snr)/100.0;

    if (satellite.used) {
      draw->PushForeground(green);
    }
    else {
      draw->PushForeground(grey);
    }

    draw->FillRectangle(x,
                        y+
                        Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder),
                        (size_t)(width*percent),height-2);


    draw->PopForeground();

    //
    // Name of Satellite
    //

    draw->PushFont(GetFont());
    draw->PushForeground(GetTextColor(draw));
    draw->DrawString(x,
                     y+GetFont()->ascent,
                     Lum::Base::NumberToWString(satellite.id));
    draw->PopForeground();
    draw->PopFont();

    //
    //
    // Signal strength label

    std::wstring        label=Lum::Base::NumberToWString(lround(percent*100))+L"%";
    Lum::OS::FontExtent extent;

    GetFont()->StringExtent(label,extent);

    draw->PushForeground(Lum::OS::Display::textColor);
    draw->PushFont(GetFont(),Lum::OS::Font::normal);
    draw->DrawString(x+(width-extent.width+extent.left+extent.right)/2,
                     y+
                     Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
                     GetFont()->ascent,
                     label);
    draw->PopFont();
    draw->PopForeground();
  }
};

class MyWindow;

static MyWindow*         window=NULL;
static Lum::Def::AppInfo info;

#if defined(HAVE_LIB_GPS)
  #if GPSD_API_MAJOR_VERSION>=4
static void cb(struct gps_data_t * data,
               char * buf,
               size_t len);
  #else
static void cb(struct gps_data_t * data,
               char * buf,
               size_t len,
               int level);
  #endif
#elif defined(HAVE_LIB_LOCATION)
static void gps_location_changed(LocationGPSDevice *device, gpointer userdata);
static void gps_location_connected(LocationGPSDevice *new_device, gpointer userdata);
static void gps_location_disconnected(LocationGPSDevice *new_device, gpointer userdata);
static void gps_location_started(LocationGPSDControl *control, gpointer userdata);
static void gps_location_stopped(LocationGPSDControl *control, gpointer userdata);
static void gps_location_error(LocationGPSDControl *control, gpointer userdata);
#endif

class TracksModelPainter : public Lum::TableCellPainter
{
public:
  void Draw(Lum::OS::DrawInfo* draw,
            int x, int y, size_t width, size_t height) const
  {
    const Track& track=dynamic_cast<const TracksModel*>(GetModel())->GetEntry(GetRow());

    //
    // Name of Track
    //
    DrawStringLeftAligned(draw,
                           x,y,width,height,
                           track.GetName());

    //
    //
    // Time of track
    DrawStringRightAligned(draw,
                           x,y,width,height,
                           track.GetTimeString());
  }
};

class TrackComparator : public TracksModel::Comparator
{
public:
  bool operator()(const Track& a, const Track& b, size_t column, bool ascending) const
  {
    assert(column==1);

    if (ascending) {
      return a.GetName()<b.GetName();
    }
    else {
      return a.GetName()>b.GetName();
    }
  }
};

class MyWindow : public Lum::Dialog
{
private:
  Lum::Model::BooleanRef   connected;
  Lum::Model::BooleanRef   online;
  Lum::Model::BooleanRef   fix;
  Lum::Model::StringRef    time;
  Lum::Model::StringRef    latitude;
  Lum::Model::StringRef    longitude;
  Lum::Model::StringRef    altitude;
  Lum::Model::DoubleRef    speedValue;
  Lum::Model::StringRef    speed;
  Lum::Model::StringRef    track;
  Lum::Model::DoubleRef    direction;
  Lum::Model::StringRef    climb;
  SatellitesModelRef       satellites;
  Lum::Model::DoubleDataStreamRef speedStream;
  Lum::Model::DoubleDataStreamRef altitudeStream;
  TracksModelRef           tracks;
  Lum::Model::SingleLineSelectionRef trackSelection;

  Lum::Model::ActionRef    startTrackingAction;
  Lum::Model::ActionRef    stopTrackingAction;
  Lum::Model::ActionRef    deleteTrackAction;
  Lum::Model::ActionRef    exportTrackKMLAction;
  Lum::Model::ActionRef    exportTrackGPXAction;
  Lum::Model::ActionRef    settingsAction;
  Lum::Model::ActionRef    aboutAction;

  Lum::Model::ActionRef    dataChangedAction;
  Lum::Model::ActionRef    refreshTimer;
  Lum::Model::ActionRef    reconnectCheckTimer;

  Lum::Model::BooleanRef   tracking;
  Lum::Model::StringRef    trackingStartDateTime;
  Lum::Model::StringRef    trackingDataPoints;
  Lum::Model::StringRef    trackingFileSize;

  std::ofstream            trackFile;
  unsigned long            trackingCount;
  unsigned long            trackingSize;

#if defined(HAVE_LIB_LOCATION)
  LocationGPSDevice        *device;
  LocationGPSDControl      *control;
#endif
#if defined(HAVE_LIB_GPS)
  struct gps_data_t        *gpsData;
  pthread_t                gpsThread;
#elif defined(HAVE_LIB_LOCATION)
    guint                  idd_changed;
    guint                  idd_con;
    guint                  idd_disc;

    guint                  idc_run;
    guint                  idc_stop;
    guint                  idc_error;
#endif

public:
  Lum::OS::RWMutex         dataMutex;
  GPSData                  data;

private:
  Lum::TextValue* CreateTextValue(Lum::Model::StringRef& model, size_t size)
  {
    Lum::TextValue *textValue;

    textValue=Lum::TextValue::Create(model,Lum::TextValue::left,true,false);
    textValue->SetWidth(Lum::Base::Size::stdCharWidth,size);

    return textValue;
  }

  void InitializeGPS()
  {
      std::cout << "Initializing GPS..." << std::endl;
#if defined(HAVE_LIB_LOCATION)
    std::cout << "Initialize LocationGPSDcontrol..." << std::endl;
    control=location_gpsd_control_get_default();

    if (control!=NULL) {
#if defined(LOCATION_METHOD_USER_SELECTED)
      std::cout << "Initializing LocationGPSDevice..." << std::endl;
      device=(LocationGPSDevice*)g_object_new(LOCATION_TYPE_GPS_DEVICE, NULL);

      if (device!=NULL) {
        std::cout << "Connecting callbacks..." << std::endl;

        idd_changed=g_signal_connect(device,"changed",G_CALLBACK(gps_location_changed),NULL);
        idd_con=g_signal_connect(device,"connected",G_CALLBACK(gps_location_connected),NULL);
        idd_disc=g_signal_connect(device,"disconnected",G_CALLBACK(gps_location_disconnected),NULL);

        idc_run=g_signal_connect(control,"gpsd_running",G_CALLBACK(gps_location_started),NULL);
        idc_stop=g_signal_connect(control,"gpsd_stopped",G_CALLBACK(gps_location_stopped),NULL);
        idc_error=g_signal_connect(control,"error",G_CALLBACK(gps_location_error),NULL);
      }
      else {
        std::cerr << "GPS NOT initialized!" << std::endl;
      }

      g_object_set(G_OBJECT(control),
                   "preferred-method",
                   LOCATION_METHOD_USER_SELECTED,
                   NULL);
      g_object_set(G_OBJECT(control),
                   "preferred-interval",
                   LOCATION_INTERVAL_1S,
                   NULL);
#endif
      std::cout << "Starting GPS daemon..." << std::endl;
      location_gpsd_control_start(control);
    }
    else {
      std::cerr << "Cannot start GPS daemon!" << std::endl;
    }
#endif

#if defined(HAVE_LIB_GPS)
    std::cout << "Initializing GPS daemon..." << std::endl;
    // This fixed a bug in libgps, using locale aware functions to parse non-locale-aware float values
    setlocale(LC_NUMERIC,"C");
    gpsData=::gps_open("127.0.0.1","2947");

    if (gpsData!=NULL) {
#if GPSD_API_MAJOR_VERSION>=4
      gps_set_raw_hook(gpsData,cb);
      gps_stream(gpsData,WATCH_ENABLE,&gpsThread);
#else
      gps_query(gpsData,"j=0w+x\n");
      gps_set_callback(gpsData,cb,&gpsThread);
#endif
      std::cout << "GPS initialized!" << std::endl;
    }
    else {
      std::cerr << "GPS NOT initialized: " << errno << std::endl;
    }
#elif !defined(HAVE_LIB_LOCATION)
    std::cerr << "No GPS initialization possible!" << std::endl;
#endif
  }

  void DeinitializeGPS()
  {
    std::cout << "Deinitialize GPS..." << std::endl;

#if defined(HAVE_LIB_GPS)
    if (gpsData!=NULL) {
      ::gps_close(gpsData);
      gpsData=NULL;
    }
#elif defined(HAVE_LIB_LOCATION)
    if (device!=NULL) {
      g_signal_handler_disconnect(device,idd_changed);
      g_signal_handler_disconnect(device,idd_con);
      g_signal_handler_disconnect(device,idd_disc);

      g_signal_handler_disconnect(control,idc_run);
      g_signal_handler_disconnect(control,idc_stop);
      g_signal_handler_disconnect(control,idc_error);

      std::cout << "Freeing LocationGPSDevice..." << std::endl;
      g_object_unref(device);
    }
#endif

#if defined(HAVE_LIB_LOCATION)
    if (control!=NULL) {
      std::cout << "Freeing LocationGPSDcontrol..." << std::endl;

      location_gpsd_control_stop(control);

      /*Device control cleanup */
      g_object_unref(control);
      control=NULL;
    }
#endif
    std::cout << "GPS deinitialized!" << std::endl;
  }

  void ReconnectIfDisconnected()
  {
    if (data.HasConnected() && data.GetConnected()) {
      return;
    }

#if defined(HAVE_LIB_GPS)
    if (gpsData!=NULL) {
      ::gps_close(gpsData);
      gpsData=NULL;
    }

    gpsData=::gps_open("127.0.0.1","2947");

    if (gpsData!=NULL) {
#if GPSD_API_MAJOR_VERSION>=4
      gps_set_raw_hook(gpsData,cb);
      gps_stream(gpsData,WATCH_ENABLE,&gpsThread);
#else
      gps_query(gpsData,"j=0w+x\n");
      gps_set_callback(gpsData,cb,&gpsThread);
#endif
      data.SetConnected(true);
      std::cout << "GPS reinitialized!" << std::endl;
    }
    else {
      data.SetConnected(false);
      std::cout << "GPS NOT initialized: " << errno << std::endl;
    }
#endif
  }

public:
  MyWindow()
  : connected(new Lum::Model::Boolean(false)),
    online(new Lum::Model::Boolean(false)),
    fix(new Lum::Model::Boolean(false)),
    time(new Lum::Model::String(L"")),
    latitude(new Lum::Model::String(L"")),
    longitude(new Lum::Model::String(L"")),
    altitude(new Lum::Model::String(L"")),
    speedValue(new Lum::Model::Double()),
    speed(new Lum::Model::String(L"")),
    track(new Lum::Model::String(L"")),
    direction(new Lum::Model::Double()),
    climb(new Lum::Model::String(L"")),
    satellites(new SatellitesModel(new SatellitesComparator())),
    speedStream(new Lum::Model::DoubleDataStream()),
    altitudeStream(new Lum::Model::DoubleDataStream()),
    tracks(new TracksModel(new TrackComparator())),
    trackSelection(new Lum::Model::SingleLineSelection()),
    startTrackingAction(new Lum::Model::Action()),
    stopTrackingAction(new Lum::Model::Action()),
    deleteTrackAction(new Lum::Model::Action()),
    exportTrackKMLAction(new Lum::Model::Action()),
    exportTrackGPXAction(new Lum::Model::Action()),
    settingsAction(new Lum::Model::Action()),
    aboutAction(new Lum::Model::Action()),
    dataChangedAction(new Lum::Model::Action()),
    refreshTimer(new Lum::Model::Action()),
    reconnectCheckTimer(new Lum::Model::Action()),
    tracking(new Lum::Model::Boolean(false)),
    trackingStartDateTime(new Lum::Model::String()),
    trackingDataPoints(new Lum::Model::String()),
    trackingFileSize(new Lum::Model::String())
#if defined(HAVE_LIB_LOCATION)
    ,device(NULL)
    ,control(NULL)
#endif
#if defined(HAVE_LIB_GPS)
    ,gpsData(NULL)
#elif defined(HAVE_LIB_LOCATION)
    ,idd_changed(0)
    ,idd_con(0)
    ,idd_disc(0)
    ,idc_run(0)
    ,idc_stop(0)
    ,idc_error(0)
#endif
  {
    Observe(GetOpenedAction());
    Observe(GetClosedAction());
    Observe(startTrackingAction);
    Observe(stopTrackingAction);
    Observe(deleteTrackAction);
    Observe(exportTrackKMLAction);
    Observe(exportTrackGPXAction);
    Observe(settingsAction);
    Observe(aboutAction);

    Observe(dataChangedAction);
    Observe(refreshTimer);
    Observe(reconnectCheckTimer);
    Observe(trackSelection);

    Observe(unitMode);
    Observe(compassShowRawGPSDirection);
    Observe(prohibitScreenBlanking);

    ::window=this;

    startTrackingAction->Enable();
    stopTrackingAction->Disable();

    satellites->SetEmptyText(L"(no satellites found)");
    tracks->SetEmptyText(L"(no tracks found)");

    // For testsing purposes
    /*
    for (int i=0; i<20; i++) {
      GPSData::Satellite s;

      s.id=i;
      s.used=i%2!=0;
      s.snr=20;

      satellites->Append(s);
    }*/
  }

  ~MyWindow()
  {
    if (tracking->Get()) {
      StopTracking();
    }
  }

  void PreInit()
  {
    Lum::Model::HeaderRef headerModel;
    Lum::Label            *label;
    Lum::Panel            *hPanel;
    Lum::Table            *table;

    Lum::Def::MultiView multiView(Lum::Def::Desc(L"GPSJinni"));

    hPanel=Lum::HPanel::Create(true,true);

    label=Lum::Label::Create(true,true);

    label->AddLabel(L"Conn./Online/Fix:",
                    Lum::HPanel::Create()
                    ->Add(new Lum::LED(connected))
                    ->AddSpace()
                    ->Add(new Lum::LED(online))
                    ->AddSpace()
                    ->Add(new Lum::LED(fix)));
    label->AddLabel(L"Time:",CreateTextValue(time,24)); // TODO: Measure by example
    label->AddLabel(L"Latitude:",CreateTextValue(latitude,10));
    label->AddLabel(L"Longitude:",CreateTextValue(longitude,10));
    label->AddLabel(L"Altitude:",CreateTextValue(altitude,10));
    label->AddLabel(L"Speed:",CreateTextValue(speed,10));
    label->AddLabel(L"Direction:",CreateTextValue(track,10));
    label->AddLabel(L"Climb:",CreateTextValue(climb,10));
    hPanel->Add(label);

    multiView.AddView(Lum::Def::MultiView::View(0,
                                                Lum::Def::Desc(L"General"),
                                                hPanel));

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Satellite",Lum::Base::Size::stdCharWidth,25,true);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetShowHeader(false);
    table->SetHeaderModel(headerModel);
    table->SetModel(satellites);
    table->SetPainter(new SatellitesModelPainter());
    table->GetTableView()->SetAutoFitColumns(true);

    multiView.AddView(Lum::Def::MultiView::View(1,
                                                Lum::Def::Desc(L"Satellites"),
                                                table));

    multiView.AddView(Lum::Def::MultiView::View(2,
                                                Lum::Def::Desc(L"Speed"),
                                                Speed::Create(speedValue,true,true)));

    multiView.AddView(Lum::Def::MultiView::View(3,
                                                Lum::Def::Desc(L"Compass"),
                                                Compass::Create(direction,true,true,compassShowCourseDirection)));

    multiView.AddView(Lum::Def::MultiView::View(4,
                                                Lum::Def::Desc(L"Diagrams"),
                                                Lum::VPanel::Create(true,true)
                                                ->Add(Lum::Text::Create(L"Speed (m/s)"))
                                                ->AddSpace()
                                                ->Add(Lum::View::Create(TimeLine::Create(speedStream,true,true),true,true))
                                                ->AddSpace()
                                                ->Add(Lum::Text::Create(L"Altitude (m)"))
                                                ->AddSpace()
                                                ->Add(Lum::View::Create(TimeLine::Create(altitudeStream,true,true),true,true))));

    multiView.AddView(Lum::Def::MultiView::View(5,
                                                Lum::Def::Desc(L"Recording"),
                                                Lum::VPanel::Create(true,true)
                                                ->Add(Lum::HPanel::Create(true,false)
                                                      ->Add(Lum::Manager::Behaviour::Instance()->GetActionControl(Lum::Def::Action(Lum::Def::Desc(L"_Record"),
                                                                                                                                   startTrackingAction)
                                                                                                                  .SetImage(L"Record.png")))
                                                      ->AddSpace(true)
                                                      ->Add(Lum::Manager::Behaviour::Instance()->GetActionControl(Lum::Def::Action(Lum::Def::Desc(L"_Stop"),
                                                                                                                                   stopTrackingAction)
                                                                                                                  .SetImage(L"Stop.png"))))
                                                ->AddSpace(false)
                                                ->Add(Lum::Label::Create(true,false)
                                                      ->AddLabel(L"StartTime:",Lum::TextValue::Create(trackingStartDateTime,true))
                                                      ->AddLabel(L"Data points:",Lum::TextValue::Create(trackingDataPoints,true))
                                                      ->AddLabel(L"File size:",Lum::TextValue::Create(trackingFileSize,true))
                                                      )
                                                ->AddSpace(true)
                                               ));

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Track",Lum::Base::Size::stdCharWidth,40,true);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetShowHeader(false);
    table->SetHeaderModel(headerModel);
    table->SetModel(tracks);
    table->SetSelection(trackSelection);
    table->SetPainter(new TracksModelPainter());
    table->GetTableView()->SetAutoFitColumns(true);

    multiView.AddView(Lum::Def::MultiView::View(6,
                                                Lum::Def::Desc(L"Tracks"),
                                                Lum::VPanel::Create(true,true)
                                                ->Add(Lum::HPanel::Create(true,false)
                                                      ->Add(Lum::Manager::Behaviour::Instance()->GetActionControl(Lum::Def::Action(Lum::Def::Desc(L"_Delete"),
                                                                                                                                   deleteTrackAction)
                                                                                                                  .SetImage(L"Delete.png")))
                                                      ->AddSpace(true)
                                                      ->Add(Lum::Manager::Behaviour::Instance()->GetActionControl(Lum::Def::Action(Lum::Def::Desc(L"_Export KML"),
                                                                                                                                   exportTrackKMLAction)
                                                                                                                  .SetImage(L"Save_KML.png")))
                                                      ->Add(Lum::Manager::Behaviour::Instance()->GetActionControl(Lum::Def::Action(Lum::Def::Desc(L"_Export GPX"),
                                                                                                                                   exportTrackGPXAction)
                                                                                                                  .SetImage(L"Save_GPX.png"))))
                                                ->AddSpace()
                                                ->Add(table)));

    Lum::Def::Menu *preMenu=Lum::Def::Menu::Create();

    preMenu
      ->GroupProject()
        ->ActionQuit(GetClosedAction())
      ->End()
      ->GroupEdit()
        ->ActionSettings(settingsAction)
      ->End();

    Lum::Def::Menu *postMenu=Lum::Def::Menu::Create();

    postMenu
      ->GroupHelp()
        //->ActionHelp()
        ->ActionAbout(aboutAction)
      ->End();

    multiView.SetMenu(preMenu,postMenu);

    Lum::Manager::Behaviour::Instance()->ApplyMultiViewDlg(this,multiView);

    Dialog::PreInit();
  }

  void GetGPSDataCopy(GPSData& data)
  {
    Lum::OS::ReadGuard<Lum::OS::RWMutex> guard(dataMutex);

    data=this->data;
  }

  void StartTracking()
  {
    if (::StartTracking(GetWindow(),trackFile)) {
      Lum::Base::SystemTime now;

      trackingCount=0;
      trackingSize=0;
      trackingStartDateTime->Set(now.GetLocalLocaleDateTime());
      trackingDataPoints->Set(L"0");
      trackingFileSize->Set(Lum::Base::NumberToWString(GetFileSize(trackFile)));
      startTrackingAction->Disable();
      stopTrackingAction->Enable();
      tracking->Set(true);
    }
  }

  void AddTrackingRecord(const GPSData& data)
  {
    if (!tracking->Get()) {
      return;
    }

    if (::AddTrackingRecord(trackFile,data)) {
      trackingCount++;
      trackingDataPoints->Set(Lum::Base::NumberToWString(trackingCount));
      trackingFileSize->Set(Lum::Base::NumberToWString(GetFileSize(trackFile)));
    }
  }


  void StopTracking()
  {
    trackFile.close();

    startTrackingAction->Enable();
    stopTrackingAction->Disable();
    tracking->Set(false);

    trackingStartDateTime->SetNull();
    trackingDataPoints->SetNull();
    trackingFileSize->SetNull();

    RefreshTracks(tracks);
  }

  void SetCompassDirectionRaw(double newDirection, double newDeviation)
  {
    direction->Set(newDirection); // set to whatever it is.
    // TODO: eliminate hack to pass deviation to compass:
    if (finite(newDeviation)) {
      direction->SetRange(0, 360 + newDeviation);
    }
  }

  void SetCompassDirectionWeighted(double newDirection, double newDeviation)
  {
    // Check if old and new direction values are available and valid

    double oldDirection;

    if (isnan(newDirection) ||
        isnan(newDeviation) ||
        direction->IsNull() ||
        isnan(oldDirection = direction->GetDouble())) {
      SetCompassDirectionRaw(newDirection,newDeviation);
      return;
    }

    if (newDeviation<0) {     // just in case...
      newDeviation=-newDeviation;
    }

    // Ignore new direction if deviation is too high; leave old direction valid.

    if (newDeviation>180) {
      return;
    }

    // Weighten the new direction by its accuracy (that is inverse deviation)
    // in comparison to the old direction is a first experiment...
    // TODO: find better methods to find and eliminate wild shot values.
    newDirection=CalculateWeightedDirection(oldDirection,
                                            newDirection,
                                            1-(newDeviation/360), // normalized weight in range 0..1.,
                                            360);
    SetCompassDirectionRaw(newDirection, newDeviation);
  }

  void ShowChange(const GPSData& data)
  {
    if (data.HasConnected()) {
      connected->Set(data.GetConnected());
    }
    else {
      connected->Set(false);
    }

    if (data.HasOnline()) {
      online->Set(data.GetOnline());
    }
    else {
      online->Set(false);
    }

    if (data.HasFix()) {
      fix->Set(data.GetFix());
    }
    else {
      fix->Set(false);
    }

    if (data.HasTime()) {
      std::wstring t=data.GetTime().GetLocalLocaleDateTime();

      if (data.GetTimeDev()!=0 &&
        finite(data.GetTimeDev())) {
        t+=L"\u00b1"+DoubleToString(data.GetTimeDev());
      }

      time->Set(t);
    }
    else {
      time->SetNull();
    }

    if (data.HasLatitude()) {
      std::wstring l;
      l=GetAngleAsDegreeMinuteSecondString(data.GetLatitude(), L"S ", L"N ", 3);

      l+=L" = ";

      l+=DoubleToString(data.GetLatitude(), 5);

      l+=L"\u00b0";

      if (data.GetLatitudeDev()!=0 &&
          finite(data.GetLatitudeDev())) {
        l+=std::wstring(L"\u00b1")+GetMetricAndUnitAsString(data.GetLatitudeDev());
      }

      latitude->Set(l);
    }
    else {
      latitude->SetNull();
    }

    if (data.HasLongitude()) {
      std::wstring l;
      l=GetAngleAsDegreeMinuteSecondString(data.GetLongitude(), L"W ", L"E ", 3);

      l+=L" = ";

      l+=DoubleToString(data.GetLongitude(), 5);

      l+=+L"\u00b0";

      if (data.GetLongitudeDev()!=0 &&
          finite(data.GetLongitudeDev())) {
        l+=std::wstring(L"\u00b1")+GetMetricAndUnitAsString(data.GetLongitudeDev());
      }

      longitude->Set(l);
    }
    else {
      longitude->SetNull();
    }

    if (data.HasAltitude()) {
      altitude->Set(GetMetricAndUnitAsString(data.GetAltitude(),
                                             data.GetAltitudeDev()));
    }
    else {
      altitude->SetNull();
    }

    if (data.HasSpeed()) {
      double       speed=data.GetSpeed();
      double       speedDev=data.GetSpeedDev();
      std::wstring s;

      // We expect m/s here
      s=GetSpeedAndUnitAsString(speed,speedDev);

      speedValue->Set(speed);
      this->speed->Set(s);
    }
    else {
      speedValue->SetNull();
      speed->SetNull();
    }

    if (data.HasTrack()) {
      std::wstring t=DoubleToString(data.GetTrack());

      if (data.GetTrackDev()!=0 &&
          finite(data.GetTrackDev())) {
        t+=L"\u00b1"+DoubleToString(data.GetTrackDev());
      }

      t+=L"\u00b0";

      track->Set(t);

      if (compassShowRawGPSDirection->Get()) {
        SetCompassDirectionRaw(data.GetTrack(), data.GetTrackDev());
      }
      else {
        SetCompassDirectionWeighted(data.GetTrack(), data.GetTrackDev());
      }
    }
    else {
      track->SetNull();
      direction->SetNull();
    }

    if (data.HasClimb()) {
      climb->Set(GetMetricAndUnitAsString(data.GetClimb(),
                                          data.GetClimbDev()));
    }
    else {
      climb->SetNull();
    }

    if (data.HasSatellites()) {
      // Delete everything in model that is not in current list
      size_t i=1;
      while (i<=satellites->GetRows()) {
        bool found=false;

        for (size_t j=0; j<data.GetSatellites().size(); j++) {
          if (satellites->GetEntry(i).id==data.GetSatellites()[j].id) {
            found=true;
            break;
          }
        }

        if (found) {
          i++;
          continue;
        }
        else {
          satellites->Delete(i);
        }
      }

      // Now add everything to model that is not already in
      for (size_t i=0; i<data.GetSatellites().size(); i++) {
        bool found=false;

        for (size_t j=1; j<=satellites->GetRows(); j++) {
          if (satellites->GetEntry(j).id==data.GetSatellites()[i].id) {
            found=true;

            if (satellites->GetEntry(j).snr!=data.GetSatellites()[i].snr ||
                satellites->GetEntry(j).used!=data.GetSatellites()[i].used) {
              satellites->GetEntry(j)=data.GetSatellites()[i];
              satellites->RedrawRow(j);
            }
            break;
          }
        }

        if (!found) {
          satellites->Append(data.GetSatellites()[i]);
        }
      }
    }
    else {
      satellites->Clear();
    }
  }

  void UpdateStreams(const GPSData& data)
  {
    if (!data.HasFix() || !data.GetFix() ||
        !data.HasSpeed()) {
      speedStream->SetNull();
    }
    else {
      speedStream->Set(0,data.GetSpeed());
    }

    if (!data.HasFix() || !data.GetFix() ||
        !data.HasAltitude()) {
      altitudeStream->SetNull();
    }
    else {
      altitudeStream->Set(0,data.GetAltitude());
    }
  }

  void ShowSettings()
  {
    Lum::Def::PropGroup *props=new Lum::Def::PropGroup();

    Lum::Def::OneOfN    unitModeDef(Lum::Def::Desc(L"Unit mode"),
                                   ::unitMode);
    unitModeDef.AddChoice(Lum::Def::OneOfN::Choice(Lum::Def::Desc(L"meter/kilometer")));
    unitModeDef.AddChoice(Lum::Def::OneOfN::Choice(Lum::Def::Desc(L"feet/miles")));

    props->Boolean(Lum::Def::Boolean(Lum::Def::Desc(L"Prohibit screen blanking"),
                                     ::prohibitScreenBlanking));
    props->Boolean(Lum::Def::Boolean(Lum::Def::Desc(L"Compass shows course"),
                                     ::compassShowCourseDirection));
    props->Boolean(Lum::Def::Boolean(Lum::Def::Desc(L"Compass shows raw GPS data"),
                                     ::compassShowRawGPSDirection));
    props->OneOfN(unitModeDef);

    Lum::Dlg::Properties::Show(this,props);

    delete props;
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() &&
        GetOpenedAction()->IsFinished()) {
      InitializeGPS();
      Lum::OS::display->AddTimer(refreshTime,0,refreshTimer);
      Lum::OS::display->AddTimer(reconnectTime,0,reconnectCheckTimer);
      RefreshTracks(tracks);
      LoadConfig();
    }
    else if (model==GetClosedAction() &&
        GetClosedAction()->IsFinished()) {
      DeinitializeGPS();
      SaveConfig();
    }
    else if (model==aboutAction &&
             aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==dataChangedAction &&
             dataChangedAction->IsFinished()) {
      GPSData dataCopy;

      GetGPSDataCopy(dataCopy);
      AddTrackingRecord(dataCopy);
      ShowChange(dataCopy);
    }
    else if (model==refreshTimer &&
             refreshTimer->IsFinished()) {
      GPSData dataCopy;

      GetGPSDataCopy(dataCopy);
      ShowChange(dataCopy);
      AddTrackingRecord(dataCopy);
      UpdateStreams(dataCopy);

      Lum::OS::display->AddTimer(refreshTime,0,refreshTimer);
    }
    else if (model==reconnectCheckTimer &&
             reconnectCheckTimer->IsFinished()) {
      ReconnectIfDisconnected();

      Lum::OS::display->AddTimer(reconnectTime,0,reconnectCheckTimer);
    }
    else if (model==startTrackingAction &&
             startTrackingAction->IsFinished() &&
             startTrackingAction->IsEnabled()) {
      StartTracking();
    }
    else if (model==stopTrackingAction &&
             stopTrackingAction->IsFinished() &&
             stopTrackingAction->IsEnabled()) {
      StopTracking();
    }
    else if (model==deleteTrackAction &&
             deleteTrackAction->IsFinished() &&
             deleteTrackAction->IsEnabled()) {
      if (DeleteTrack(GetWindow(),
                      tracks->GetEntry(trackSelection->GetLine()).GetName())) {
        RefreshTracks(tracks);
      }
    }
    else if (model==exportTrackKMLAction &&
             exportTrackKMLAction->IsFinished() &&
             exportTrackKMLAction->IsEnabled()) {
      ExportTrackToKML(GetWindow(),
                       tracks->GetEntry(trackSelection->GetLine()).GetName());
    }
    else if (model==exportTrackGPXAction &&
             exportTrackGPXAction->IsFinished() &&
             exportTrackGPXAction->IsEnabled()) {
      ExportTrackToGPX(GetWindow(),
                       tracks->GetEntry(trackSelection->GetLine()).GetName());
    }
    else if (model==trackSelection) {
      if (trackSelection->HasSelection()) {
        deleteTrackAction->Enable();
        exportTrackKMLAction->Enable();
        exportTrackGPXAction->Enable();
      }
      else {
        deleteTrackAction->Disable();
        exportTrackKMLAction->Disable();
        exportTrackGPXAction->Disable();
      }
    }
    else if (model==settingsAction &&
             settingsAction->IsFinished()) {
      ShowSettings();
    }
    else if (model==prohibitScreenBlanking) {
      Lum::Manager::Display *displayManager=Lum::Manager::Display::Instance();

      std::cout << "prohibitScreenBlanking changed! displayManager "
                << (displayManager ? "!= NULL" : "== NULL") << std::endl;

      if (displayManager!=NULL) {
        displayManager->AllowScreenBlanking(!prohibitScreenBlanking->Get());
      }
    }
    else if (model==unitMode) {
      std::cout << "unitMode changed!" << std::endl;
      GPSData dataCopy;

      GetGPSDataCopy(dataCopy);
      ShowChange(dataCopy);
      //UpdateStreams(dataCopy);
    }
    else if (model==compassShowRawGPSDirection) {
      std::cout << "compassShowRawGPSDirection changed!" << std::endl;
      GPSData dataCopy;

      GetGPSDataCopy(dataCopy);
      ShowChange(dataCopy);
    }

    Dialog::Resync(model,msg);
  }

  void HandleChange()
  {
    Lum::OS::display->QueueActionForAsyncNotification(dataChangedAction);
  }
};

#if defined(HAVE_LIB_GPS)
  #if GPSD_API_MAJOR_VERSION>=4
static void cb(struct gps_data_t* data, char * buf, size_t len)
  #else
static void cb(struct gps_data_t* data, char * buf, size_t len, int level)
  #endif
{
  assert(window!=NULL);

  Lum::OS::WriteGuard<Lum::OS::RWMutex> guard(window->dataMutex);

  GPSData                               &gpsData=window->data;

  gpsData.SetConnected(true);

  if (data->set & ONLINE_SET) {
    gpsData.SetOnline(data->online);
  }

  if (data->set & STATUS_SET) {
    gpsData.SetFix(data->status!=STATUS_NO_FIX);
  }

  if ((data->set & LATLON_SET)  &&
      !isnan(data->fix.latitude)) {
    gpsData.SetLatitude(data->fix.latitude,data->fix.epy);
  }

  if ((data->set & LATLON_SET)  &&
      !isnan(data->fix.longitude)) {
    gpsData.SetLongitude(data->fix.longitude,data->fix.epx);
  }

  if ((data->set & ALTITUDE_SET) &&
      !isnan(data->fix.altitude)) {
    gpsData.SetAltitude(data->fix.altitude,data->fix.epv);
  }

  if ((data->set & SPEED_SET) &&
       !isnan(data->fix.speed)) {
    gpsData.SetSpeed(data->fix.speed,data->fix.eps); // m/s
  }

  if ((data->set & TRACK_SET) &&
      !isnan(data->fix.track)) {
    gpsData.SetTrack(data->fix.track,data->fix.epd);
  }

  if ((data->set & CLIMB_SET) &&
      !isnan(data->fix.climb)) {
    gpsData.SetClimb(data->fix.climb,data->fix.epc);
  }

  if (data->set & TIME_SET) {
    Lum::Base::SystemTime time(lround(data->fix.time));
    gpsData.SetTime(time,data->fix.ept);
  }

  if (data->set & SATELLITE_SET) {
    std::vector<GPSData::Satellite> satellites;

    for (int i=0; i<data->satellites_used; i++) {
      GPSData::Satellite s;

      s.id=data->PRN[i];
      s.used=data->used[i];
      s.snr=data->ss[i];

      satellites.push_back(s);
    }

    gpsData.SetSatellites(satellites);
  }

  window->HandleChange();
}
#elif defined(HAVE_LIB_LOCATION)
static void gps_location_changed(LocationGPSDevice *device, gpointer userdata)
{
  std::cout << "gps_location_changed!" << std::endl;
  assert(window!=NULL);

  Lum::OS::WriteGuard<Lum::OS::RWMutex> guard(window->dataMutex);

  GPSData                               &gpsData=window->data;

  gpsData.SetConnected(true);
  gpsData.SetOnline(device->online);
  gpsData.SetFix(device->status!=LOCATION_GPS_DEVICE_STATUS_NO_FIX);

  /* Fix information */
  LocationGPSDeviceFix *fix=device->fix;

  if (fix->fields & LOCATION_GPS_DEVICE_TIME_SET) {
    Lum::Base::SystemTime time(lround(fix->time));
    gpsData.SetTime(time,fix->ept);
  }
  if (fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET && !isnan(fix->latitude)) {
    gpsData.SetLatitude(fix->latitude,fix->eph/100);
  }
  if (fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET && !isnan(fix->longitude)) {
    gpsData.SetLongitude(fix->longitude,fix->eph/100);
  }
  if (fix->fields & LOCATION_GPS_DEVICE_ALTITUDE_SET && !isnan(fix->altitude)) {
    gpsData.SetAltitude(fix->altitude,fix->epv);
  }
  if (fix->fields & LOCATION_GPS_DEVICE_SPEED_SET && !isnan(fix->speed)) {
    gpsData.SetSpeed(fix->speed*1000.0/3600,
                     fix->eps*1000.0/3600); // km/h => m/s
  }
  if (fix->fields & LOCATION_GPS_DEVICE_TRACK_SET && !isnan(fix->track)) {
    gpsData.SetTrack(fix->track,fix->epd);
  }
  if (fix->fields & LOCATION_GPS_DEVICE_CLIMB_SET && !isnan(fix->climb)) {
    gpsData.SetClimb(fix->climb,fix->epc);
  }

  std::vector<GPSData::Satellite> satellites;

  if (device->satellites!=NULL) {
    for (size_t i=0; i<device->satellites->len; i++) {
      LocationGPSDeviceSatellite *sat=(LocationGPSDeviceSatellite*)g_ptr_array_index(device->satellites,i);
      GPSData::Satellite         s;

      s.id=sat->prn;
      s.used=sat->in_use;
      s.snr=sat->signal_strength;

      satellites.push_back(s);
    }
  }

  gpsData.SetSatellites(satellites);

  window->HandleChange();
}

static void gps_location_connected(LocationGPSDevice *new_device, gpointer userdata) {
  std::cout << "Connected..." << std::endl;
}

static void gps_location_disconnected(LocationGPSDevice *new_device, gpointer userdata){
  std::cout << "disconnected..." << std::endl;
}

static void gps_location_started(LocationGPSDControl *control, gpointer userdata) {
  std::cout << "GPSD started!" << std::endl;
}

static void gps_location_stopped(LocationGPSDControl *control, gpointer userdata) {
  std::cout << "GPSD stoped!" << std::endl;
}

static void gps_location_error(LocationGPSDControl *control, gpointer userdata) {
  std::cout << "GPSD error!" << std::endl;
}
#endif

class Main : public Lum::GUIApplication<MyWindow>
{
public:
  bool Initialize()
  {
#if defined(APP_DATADIR)
    Lum::Manager::FileSystem::Instance()->SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(L"Where the #*~@# am I?");
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2009, Tim Teulings");
    info.SetLicense(L"GNU Public License");

    std::wstring appDataDir;

    if (Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::appDataDir,
                                                       appDataDir)) {

      Lum::Base::Path path;

      path.SetNativeDir(appDataDir);
      path.AppendDir(L"icons");

      Lum::Images::loader->AddApplicationSearchPath(path);
    }

    Lum::Base::Path path;

    path.SetDir(L"data");
    path.AppendDir(L"icons");
    Lum::Images::loader->AddApplicationSearchPath(path);

    return Lum::GUIApplication<MyWindow>::Initialize();
  }
};

LUM_MAIN(Main,Lum::Base::StringToWString(PACKAGE_NAME))

