/*
  GPSJinni - show raw data from the GPS subsystem.
  Copyright (C) 2009  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <limits>

#include "config.h"

#include <signal.h>

#include <glib.h>

#include <cerrno>
#include <cmath>

#if defined (HAVE_LIB_GPS)
  #include <gps.h>
#endif

#if defined(HAVE_LIB_LOCATION)
extern "C" {
  #include <location/location-gps-device.h>
  #include <location/location-gpsd-control.h>
}
#endif

#include <netinet/in.h>

#include <fstream>

#include <Lum/Def/Menu.h>
#include <Lum/Def/MultiView.h>

#include <Lum/Images/Image.h>
#include <Lum/Images/Loader.h>

#include <Lum/Manager/Behaviour.h>
#include <Lum/Manager/Display.h>
#include <Lum/Manager/FileSystem.h>

#include <Lum/OS/Probe.h>
#include <Lum/OS/Thread.h>

#include <Lum/Base/DateTime.h>
#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/File.h>
#include <Lum/Dlg/Msg.h>
#include <Lum/Dlg/Properties.h>
#include <Lum/Dlg/Value.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Boolean.h>
#include <Lum/Model/DataStream.h>
#include <Lum/Model/Header.h>
#include <Lum/Model/String.h>

#include <Lum/Application.h>
#include <Lum/Button.h>
#include <Lum/Dialog.h>
#include <Lum/Label.h>
#include <Lum/LED.h>
#include <Lum/Panel.h>
#include <Lum/PercentBar.h>
#include <Lum/Tab.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>
#include <Lum/View.h>

#include "Configuration.h"

#include "GPSData.h"
#include "Compass.h"
#include "Speed.h"
#include "TimeLine.h"
#include "Util.h"

#include <iostream>

static long reconnectTime   = 10; // Period of time after which we check for disconnection and
                                  // trigger reconnection
static long refreshTime     = 10; // Period of time after which we update the stream views

std::wstring SystemTimeToISO8601(const Lum::Base::SystemTime& time)
{
  Lum::Base::DateTime t;
  std::wstring        res;

  res.reserve(10+1+8+1);

  time.GetUTCDateTime(t);

  // Date
  res.append(1,(wchar_t)(L'0'+(t.year / 1000) % 10));
  res.append(1,(wchar_t)(L'0'+(t.year /  100) % 10));
  res.append(1,(wchar_t)(L'0'+(t.year /   10) % 10));
  res.append(1,(wchar_t)(L'0'+(t.year /    1) % 10));
  res.append(L"-");
  res.append(1,(wchar_t)(L'0'+(t.month / 10) % 10));
  res.append(1,(wchar_t)(L'0'+(t.month /  1) % 10));
  res.append(L"-");
  res.append(1,(wchar_t)(L'0'+(t.dayOfMonth / 10) % 10));
  res.append(1,(wchar_t)(L'0'+(t.dayOfMonth /  1) % 10));

  res.append(1,'T');

  // Time
  res.append(1,(wchar_t)(L'0'+(t.hour /   10) % 10));
  res.append(1,(wchar_t)(L'0'+(t.hour /    1) % 10));
  res.append(L":");
  res.append(1,(wchar_t)(L'0'+(t.minute / 10) % 10));
  res.append(1,(wchar_t)(L'0'+(t.minute /  1) % 10));
  res.append(L":");
  res.append(1,(wchar_t)(L'0'+(t.second / 10) % 10));
  res.append(1,(wchar_t)(L'0'+(t.second /  1) % 10));

  res.append(1,'Z');

  return res;
}

typedef Lum::Model::StdTable<GPSData::Satellite> SatellitesModel;
typedef Lum::Base::Reference<SatellitesModel>    SatellitesModelRef;

double GetRelativeQualityFromDbm(double quality)
{
  if (quality<0) {
    return 0;
  }
  if (quality>=0 && quality<40) {
    return 5*quality/2;
  }
  else {
    return 100;
  }
}

class SatellitesModelPainter : public Lum::TableCellPainter
{
private:
  Lum::OS::ColorRef  green;
  Lum::OS::ColorRef  grey;

public:
  SatellitesModelPainter()
    : green(0,1,0,Lum::OS::display->GetColor(Lum::OS::Display::tableTextColor)),
      grey(0.7,0.7,0.7,Lum::OS::display->GetColor(Lum::OS::Display::tableTextColor))
  {
    // no code
  }

  size_t GetProposedHeight() const
  {
    return Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
           GetFont()->height+
           Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder);
  }

  void Draw(Lum::OS::DrawInfo* draw,
            int x, int y, size_t width, size_t height) const
  {
    const GPSData::Satellite& satellite=dynamic_cast<const SatellitesModel*>(GetModel())->GetEntry(GetRow());

    //
    // Signal strength bar
    //

    double percent=GetRelativeQualityFromDbm(satellite.snr)/100.0;

    if (satellite.used) {
      draw->PushForeground(green);
    }
    else {
      draw->PushForeground(grey);
    }

    draw->FillRectangle(x,
                        y+
                        Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder),
                        (size_t)(width*percent),height-2);


    draw->PopForeground();

    //
    // Name of Satellite
    //

    draw->PushFont(GetFont());
    draw->PushForeground(GetTextColor(draw));
    draw->DrawString(x,
                     y+GetFont()->ascent,
                     Lum::Base::NumberToWString(satellite.id));
    draw->PopForeground();
    draw->PopFont();

    //
    //
    // Signal strength label

    std::wstring        label=Lum::Base::NumberToWString(lround(percent*100))+L"%";
    Lum::OS::FontExtent extent;

    GetFont()->StringExtent(label,extent);

    draw->PushForeground(Lum::OS::Display::textColor);
    draw->PushFont(GetFont(),Lum::OS::Font::normal);
    draw->DrawString(x+(width-extent.width+extent.left+extent.right)/2,
                     y+
                     Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
                     GetFont()->ascent,
                     label);
    draw->PopFont();
    draw->PopForeground();
  }
};

class MyWindow;
static MyWindow*         window=NULL;
static Lum::Def::AppInfo info;
static unsigned long     trackingFileId='G'*256*256*256+'J'*256*256+'T'*256+'\0';
static unsigned long     trackingFileVersion=2;

#if defined(HAVE_LIB_GPS)
  #if GPSD_API_MAJOR_VERSION>=4
static void cb(struct gps_data_t * data,
               char * buf,
               size_t len);
  #else
static void cb(struct gps_data_t * data,
               char * buf,
               size_t len,
               int level);
  #endif
#elif defined(HAVE_LIB_LOCATION)
static void gps_location_changed(LocationGPSDevice *device, gpointer userdata);
static void gps_location_connected(LocationGPSDevice *new_device, gpointer userdata);
static void gps_location_disconnected(LocationGPSDevice *new_device, gpointer userdata);
static void gps_location_started(LocationGPSDControl *control, gpointer userdata);
static void gps_location_stopped(LocationGPSDControl *control, gpointer userdata);
static void gps_location_error(LocationGPSDControl *control, gpointer userdata);
#endif

class Track
{
private:
  std::wstring          name;
  Lum::Base::SystemTime time;

public:
  Track()
  {
    // no code
  }

  Track(const std::wstring& name, const Lum::Base::SystemTime& time)
  : name(name),
    time(time)
  {
    // no code
  }

  std::wstring GetName() const
  {
    return name;
  }

  std::wstring GetTimeString() const
  {
    return time.GetLocalLocaleDate()+L" "+time.GetLocalLocaleTime();
  }
};

typedef Lum::Model::StdTable<Track> TracksModel;
typedef Lum::Base::Reference<TracksModel> TracksModelRef;

class TracksModelPainter : public Lum::TableCellPainter
{
public:
  void Draw(Lum::OS::DrawInfo* draw,
            int x, int y, size_t width, size_t height) const
  {
    const Track& track=dynamic_cast<const TracksModel*>(GetModel())->GetEntry(GetRow());

    //
    // Name of Track
    //


    DrawStringLeftAligned(draw,
                           x,y,width,height,
                           track.GetName());

    //
    //
    // Time of track

    DrawStringRightAligned(draw,
                           x,y,width,height,
                           track.GetTimeString());
  }
};

class TrackComparator : public TracksModel::Comparator
{
public:
  bool operator()(const Track& a, const Track& b, size_t column, bool down) const
  {
    assert(column==1);

    if (down) {
      return a.GetName()<b.GetName();
    }
    else {
      return a.GetName()>b.GetName();
    }
  }
};

class MyWindow : public Lum::Dialog
{
private:
  Lum::Model::BooleanRef   connected;
  Lum::Model::BooleanRef   online;
  Lum::Model::BooleanRef   fix;
  Lum::Model::StringRef    time;
  Lum::Model::StringRef    latitude;
  Lum::Model::StringRef    longitude;
  Lum::Model::StringRef    altitude;
  Lum::Model::DoubleRef    speedValue;
  Lum::Model::StringRef    speed;
  Lum::Model::StringRef    track;
  Lum::Model::DoubleRef    direction;
  Lum::Model::StringRef    climb;
  SatellitesModelRef       satellites;
  Lum::Model::DoubleDataStreamRef speedStream;
  Lum::Model::DoubleDataStreamRef altitudeStream;
  TracksModelRef           tracks;
  Lum::Model::SingleLineSelectionRef trackSelection;

  Lum::Model::ActionRef    startTrackingAction;
  Lum::Model::ActionRef    stopTrackingAction;
  Lum::Model::ActionRef    deleteTrackAction;
  Lum::Model::ActionRef    exportTrackAction;
  Lum::Model::ActionRef    settingsAction;
  Lum::Model::ActionRef    aboutAction;

  Lum::Model::ActionRef    dataChangedAction;
  Lum::Model::ActionRef    refreshTimer;
  Lum::Model::ActionRef    reconnectCheckTimer;

  Lum::Model::BooleanRef   tracking;

  std::ofstream            trackFile;

#if defined(HAVE_LIB_LOCATION)
  LocationGPSDevice        *device;
  LocationGPSDControl      *control;
#endif
#if defined(HAVE_LIB_GPS)
  struct gps_data_t        *gpsData;
  pthread_t                gpsThread;
#elif defined(HAVE_LIB_LOCATION)
    guint                  idd_changed;
    guint                  idd_con;
    guint                  idd_disc;

    guint                  idc_run;
    guint                  idc_stop;
    guint                  idc_error;
#endif

public:
  Lum::OS::RWMutex         dataMutex;
  GPSData                  data;

private:
  Lum::TextValue* CreateTextValue(Lum::Model::StringRef& model, size_t size)
  {
    Lum::TextValue *textValue;

    textValue=Lum::TextValue::Create(model,Lum::TextValue::left,true,false);
    textValue->SetWidth(Lum::Base::Size::stdCharWidth,size);

    return textValue;
  }

  void InitializeGPS()
  {
      std::cout << "Initializing GPS..." << std::endl;
#if defined(HAVE_LIB_LOCATION)
    std::cout << "Initialize LocationGPSDcontrol..." << std::endl;
    control=location_gpsd_control_get_default();

    if (control!=NULL) {
#if defined(LOCATION_METHOD_USER_SELECTED)
      std::cout << "Initializing LocationGPSDevice..." << std::endl;
      device=(LocationGPSDevice*)g_object_new(LOCATION_TYPE_GPS_DEVICE, NULL);

      if (device!=NULL) {
        std::cout << "Connecting callbacks..." << std::endl;

        idd_changed=g_signal_connect(device,"changed",G_CALLBACK(gps_location_changed),NULL);
        idd_con=g_signal_connect(device,"connected",G_CALLBACK(gps_location_connected),NULL);
        idd_disc=g_signal_connect(device,"disconnected",G_CALLBACK(gps_location_disconnected),NULL);

        idc_run=g_signal_connect(control,"gpsd_running",G_CALLBACK(gps_location_started),NULL);
        idc_stop=g_signal_connect(control,"gpsd_stopped",G_CALLBACK(gps_location_stopped),NULL);
        idc_error=g_signal_connect(control,"error",G_CALLBACK(gps_location_error),NULL);
      }
      else {
        std::cerr << "GPS NOT initialized!" << std::endl;
      }

      g_object_set(G_OBJECT(control),
                   "preferred-method",
                   LOCATION_METHOD_USER_SELECTED,
                   NULL);
      g_object_set(G_OBJECT(control),
                   "preferred-interval",
                   LOCATION_INTERVAL_1S,
                   NULL);
#endif
      std::cout << "Starting GPS daemon..." << std::endl;
      location_gpsd_control_start(control);
    }
    else {
      std::cerr << "Cannot start GPS daemon!" << std::endl;
    }
#endif

#if defined(HAVE_LIB_GPS)
    std::cout << "Initializing GPS daemon..." << std::endl;
    // This fixed a bug in libgps, using locale aware functions to parse non-locale-aware float values
    setlocale(LC_NUMERIC,"C");
    gpsData=::gps_open("127.0.0.1","2947");

    if (gpsData!=NULL) {
#if GPSD_API_MAJOR_VERSION>=4
      gps_set_raw_hook(gpsData,cb);
      gps_stream(gpsData,WATCH_ENABLE,&gpsThread);
#else
      gps_query(gpsData,"j=0w+x\n");
      gps_set_callback(gpsData,cb,&gpsThread);
#endif
      std::cout << "GPS initialized!" << std::endl;
    }
    else {
      std::cerr << "GPS NOT initialized: " << errno << std::endl;
    }
#elif !defined(HAVE_LIB_LOCATION)
    std::cerr << "No GPS initialization possible!" << std::endl;
#endif
  }

  void DeinitializeGPS()
  {
    std::cout << "Deinitialize GPS..." << std::endl;

#if defined(HAVE_LIB_GPS)
    if (gpsData!=NULL) {
      ::gps_close(gpsData);
      gpsData=NULL;
    }
#elif defined(HAVE_LIB_LOCATION)
    if (device!=NULL) {
      g_signal_handler_disconnect(device,idd_changed);
      g_signal_handler_disconnect(device,idd_con);
      g_signal_handler_disconnect(device,idd_disc);

      g_signal_handler_disconnect(control,idc_run);
      g_signal_handler_disconnect(control,idc_stop);
      g_signal_handler_disconnect(control,idc_error);

      std::cout << "Freeing LocationGPSDevice..." << std::endl;
      g_object_unref(device);
    }
#endif

#if defined(HAVE_LIB_LOCATION)
    if (control!=NULL) {
      std::cout << "Freeing LocationGPSDcontrol..." << std::endl;

      location_gpsd_control_stop(control);

      /*Device control cleanup */
      g_object_unref(control);
      control=NULL;
    }
#endif
    std::cout << "GPS deinitialized!" << std::endl;
  }

  void ReconnectIfDisconnected()
  {
    if (data.HasConnected() && data.GetConnected()) {
      return;
    }

#if defined(HAVE_LIB_GPS)
    if (gpsData!=NULL) {
      ::gps_close(gpsData);
      gpsData=NULL;
    }

    gpsData=::gps_open("127.0.0.1","2947");

    if (gpsData!=NULL) {
#if GPSD_API_MAJOR_VERSION>=4
      gps_set_raw_hook(gpsData,cb);
      gps_stream(gpsData,WATCH_ENABLE,&gpsThread);
#else
      gps_query(gpsData,"j=0w+x\n");
      gps_set_callback(gpsData,cb,&gpsThread);
#endif
      data.SetConnected(true);
      std::cout << "GPS initialized!" << std::endl;
    }
    else {
      data.SetConnected(false);
      std::cout << "GPS NOT initialized: " << errno << std::endl;
    }
#endif
  }

public:
  MyWindow()
  : connected(new Lum::Model::Boolean(false)),
    online(new Lum::Model::Boolean(false)),
    fix(new Lum::Model::Boolean(false)),
    time(new Lum::Model::String(L"")),
    latitude(new Lum::Model::String(L"")),
    longitude(new Lum::Model::String(L"")),
    altitude(new Lum::Model::String(L"")),
    speedValue(new Lum::Model::Double()),
    speed(new Lum::Model::String(L"")),
    track(new Lum::Model::String(L"")),
    direction(new Lum::Model::Double()),
    climb(new Lum::Model::String(L"")),
    satellites(new SatellitesModel()),
    speedStream(new Lum::Model::DoubleDataStream()),
    altitudeStream(new Lum::Model::DoubleDataStream()),
    tracks(new TracksModel(new TrackComparator())),
    trackSelection(new Lum::Model::SingleLineSelection()),
    startTrackingAction(new Lum::Model::Action()),
    stopTrackingAction(new Lum::Model::Action()),
    deleteTrackAction(new Lum::Model::Action()),
    exportTrackAction(new Lum::Model::Action()),
    settingsAction(new Lum::Model::Action()),
    aboutAction(new Lum::Model::Action()),
    dataChangedAction(new Lum::Model::Action()),
    refreshTimer(new Lum::Model::Action()),
    reconnectCheckTimer(new Lum::Model::Action()),
    tracking(new Lum::Model::Boolean(false))
#if defined(HAVE_LIB_LOCATION)
    ,device(NULL)
    ,control(NULL)
#endif
#if defined(HAVE_LIB_GPS)
    ,gpsData(NULL)
#elif defined(HAVE_LIB_LOCATION)
    ,idd_changed(0)
    ,idd_con(0)
    ,idd_disc(0)
    ,idc_run(0)
    ,idc_stop(0)
    ,idc_error(0)
#endif
  {
    Observe(GetOpenedAction());
    Observe(GetClosedAction());
    Observe(startTrackingAction);
    Observe(stopTrackingAction);
    Observe(deleteTrackAction);
    Observe(exportTrackAction);
    Observe(settingsAction);
    Observe(aboutAction);

    Observe(dataChangedAction);
    Observe(refreshTimer);
    Observe(reconnectCheckTimer);
    Observe(trackSelection);

    Observe(unitMode);
    Observe(compassShowRawGPSDirection);
    Observe(prohibitScreenBlanking);

    ::window=this;

    startTrackingAction->Enable();
    stopTrackingAction->Disable();

    satellites->SetEmptyText(L"(no satellites found)");
    tracks->SetEmptyText(L"(no tracks found)");

    // For testsing purposes
    /*
    for (int i=0; i<20; i++) {
      GPSData::Satellite s;

      s.id=i;
      s.used=i%2!=0;
      s.snr=20;

      satellites->Append(s);
    }*/
  }

  ~MyWindow()
  {
    if (tracking->Get()) {
      StopTracking();
    }
  }

  void PreInit()
  {
    Lum::Model::HeaderRef headerModel;
    Lum::Label            *label;
    Lum::Panel            *hPanel;
    Lum::Table            *table;

    Lum::Def::MultiView multiView(Lum::Def::Desc(L"GPSJinni"));

    hPanel=Lum::HPanel::Create(true,true);

    label=Lum::Label::Create(true,true);

    label->AddLabel(L"Conn./Online/Fix:",
                    Lum::HPanel::Create()
                    ->Add(new Lum::LED(connected))
                    ->AddSpace()
                    ->Add(new Lum::LED(online))
                    ->AddSpace()
                    ->Add(new Lum::LED(fix)));
    label->AddLabel(L"Time:",CreateTextValue(time,24)); // TODO: Measure by example
    label->AddLabel(L"Latitude:",CreateTextValue(latitude,10));
    label->AddLabel(L"Longitude:",CreateTextValue(longitude,10));
    label->AddLabel(L"Altitude:",CreateTextValue(altitude,10));
    label->AddLabel(L"Speed:",CreateTextValue(speed,10));
    label->AddLabel(L"Direction:",CreateTextValue(track,10));
    label->AddLabel(L"Climb:",CreateTextValue(climb,10));
    hPanel->Add(label);

    multiView.AddView(Lum::Def::MultiView::View(0,
                                                Lum::Def::Desc(L"General"),
                                                hPanel));

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Satellite",Lum::Base::Size::stdCharWidth,25,true);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetShowHeader(false);
    table->SetHeaderModel(headerModel);
    table->SetModel(satellites);
    table->SetPainter(new SatellitesModelPainter());
    table->GetTableView()->SetAutoFitColumns(true);

    multiView.AddView(Lum::Def::MultiView::View(1,
                                                Lum::Def::Desc(L"Satellites"),
                                                table));

    multiView.AddView(Lum::Def::MultiView::View(2,
                                                Lum::Def::Desc(L"Speed"),
                                                Speed::Create(speedValue,true,true)));

    multiView.AddView(Lum::Def::MultiView::View(3,
                                                Lum::Def::Desc(L"Compass"),
                                                Compass::Create(direction,true,true,compassShowCourseDirection)));

    multiView.AddView(Lum::Def::MultiView::View(4,
                                                Lum::Def::Desc(L"Diagrams"),
                                                Lum::VPanel::Create(true,true)
                                                ->Add(Lum::Text::Create(L"Speed (m/s)"))
                                                ->AddSpace()
                                                ->Add(Lum::View::Create(TimeLine::Create(speedStream,true,true),true,true))
                                                ->AddSpace()
                                                ->Add(Lum::Text::Create(L"Altitude (m)"))
                                                ->AddSpace()
                                                ->Add(Lum::View::Create(TimeLine::Create(altitudeStream,true,true),true,true))));

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Track",Lum::Base::Size::stdCharWidth,40,true);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetShowHeader(false);
    table->SetHeaderModel(headerModel);
    table->SetModel(tracks);
    table->SetSelection(trackSelection);
    table->SetPainter(new TracksModelPainter());
    table->GetTableView()->SetAutoFitColumns(true);

    multiView.AddView(Lum::Def::MultiView::View(5,
                                                Lum::Def::Desc(L"Tracking"),
                                                Lum::VPanel::Create(true,true)
                                                ->Add(Lum::HPanel::Create(true,false)
                                                      ->Add(Lum::Manager::Behaviour::Instance()->GetActionControl(Lum::Def::Action(Lum::Def::Desc(L"_Record"),
                                                                                                                                   startTrackingAction)
                                                                                                                  .SetImage(L"Record.png")))
                                                      ->Add(Lum::Manager::Behaviour::Instance()->GetActionControl(Lum::Def::Action(Lum::Def::Desc(L"_Stop"),
                                                                                                                                   stopTrackingAction)
                                                                                                                  .SetImage(L"Stop.png")))
                                                      ->AddSpace(true)
                                                       ->Add(new Lum::LED(tracking))
                                                       ->AddSpace(true)
                                                      ->Add(Lum::Manager::Behaviour::Instance()->GetActionControl(Lum::Def::Action(Lum::Def::Desc(L"_Delete"),
                                                                                                                                   deleteTrackAction)
                                                                                                                  .SetImage(L"Delete.png")))
                                                      ->Add(Lum::Manager::Behaviour::Instance()->GetActionControl(Lum::Def::Action(Lum::Def::Desc(L"_Export"),
                                                                                                                                   exportTrackAction)
                                                                                                                  .SetImage(L"Save.png"))))
                                                ->AddSpace()
                                                ->Add(table)));

    Lum::Def::Menu *preMenu=Lum::Def::Menu::Create();

    preMenu
      ->GroupProject()
        ->ActionQuit(GetClosedAction())
      ->End()
      ->GroupEdit()
        ->ActionSettings(settingsAction)
      ->End();

    Lum::Def::Menu *postMenu=Lum::Def::Menu::Create();

    postMenu
      ->GroupHelp()
        //->ActionHelp()
        ->ActionAbout(aboutAction)
      ->End();

    multiView.SetMenu(preMenu,postMenu);

    Lum::Manager::Behaviour::Instance()->ApplyMultiViewDlg(this,multiView);

    Dialog::PreInit();
  }

  void GetGPSDataCopy(GPSData& data)
  {
    Lum::OS::ReadGuard<Lum::OS::RWMutex> guard(dataMutex);

    data=this->data;
  }

  void StartTracking()
  {
    Lum::Model::StringRef trackName=new Lum::Model::String(L"");
    std::wstring          homeDir;
    Lum::Base::Path       fileName;

    if (!Lum::Dlg::TextInput::GetText(GetWindow(),
                                      L"Enter track name!",
                                      L"Please enter a name for the to be recorded track.",
                                      trackName) || trackName->Empty()) {
      return;
    }

    if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDir,
        homeDir)) {
      return;
    }

    fileName.SetNativeDir(homeDir);
    fileName.AppendDir(L".gpsjinni");

    Lum::Base::Path::CreateDir(fileName.GetPath());

    fileName.SetBaseName(trackName->Get()+L".gjt");

    if (Lum::Base::Path::Exists(fileName.GetPath())) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),
                            L"Error while opening file!",
                            std::wstring(L"An error occured while opening file:\n")+
                            L"'"+fileName.GetPath()+L"'\n"+
                            L"Error:\n"+
                            L"'File already exists'");
      return;
    }

    trackFile.open(Lum::Base::WStringToString(fileName.GetPath()).c_str(),
                   std::ios::out|std::ios::trunc|std::ios::binary);

    if (!trackFile) {
      Lum::Base::Status status;

      status.SetToCurrentErrno();
      Lum::Dlg::Msg::ShowOk(GetWindow(),
                            L"Error while opening file!",
                            std::wstring(L"An error occured while opening file:\n")+
                            L"'"+fileName.GetPath()+L"'\n"+
                            L"Error:\n"+
                            L"'"+status.GetDescription()+L"'");

      return;
    }

    unsigned long tfid=htonl(trackingFileId);
    unsigned long tfv=htonl(trackingFileVersion);
    unsigned long tft=htonl(Lum::Base::SystemTime().GetTime());

    trackFile.write((const char*)&tfid,sizeof(tfid));
    trackFile.write((const char*)&tfv,sizeof(tfv));
    trackFile.write((const char*)&tft,sizeof(tft));

    startTrackingAction->Disable();
    stopTrackingAction->Enable();
    tracking->Set(true);
  }

  void AddTrackingRecord(const GPSData& data)
  {
    if (!tracking->Get()) {
      return;
    }

    if (!data.HasFix()) {
      return;
    }

    double                lon=0.0;
    double                lat=0.0;
    double                alt=0.0;
    Lum::Base::SystemTime tim;

    if (data.HasLongitude()) {
      lon=data.GetLongitude();
    }

    if (data.HasLatitude()) {
      lat=data.GetLatitude();
    }

    if (data.HasAltitude()) {
      alt=data.GetAltitude();
    }

    if (data.HasTime()) {
      tim=data.GetTime();
    }

    unsigned long flon=htonl((unsigned long)rint((lon+180.0)*10000000.0));
    unsigned long flat=htonl((unsigned long)rint((lat+90.0)*10000000.0));
    unsigned long falt=htonl((unsigned long)rint(alt*100));
    unsigned long ftim=htonl(tim.GetTime());

    trackFile.write((const char*)&flon,sizeof(flon));
    trackFile.write((const char*)&flat,sizeof(flat));
    trackFile.write((const char*)&falt,sizeof(falt));
    trackFile.write((const char*)&ftim,sizeof(ftim));
  }

  void StopTracking()
  {
    trackFile.close();

    startTrackingAction->Enable();
    stopTrackingAction->Disable();
    tracking->Set(false);

    RefreshTracks();
  }

  void RefreshTracks()
  {
    Lum::Base::Path              dirName,fileName;
    Lum::Base::DirScannerRef     scanner;
    Lum::Base::DirScanner::Entry entry;
    Lum::Base::Status            status;
    std::wstring                 homeDir;

    tracks->Clear();
    tracks->Off();

    if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDir,
        homeDir)) {
      return;
    }

    dirName.SetNativeDir(homeDir);
    dirName.AppendDir(L".gpsjinni");

    scanner=Lum::Base::DirScanner::Create(dirName);

    while (scanner->GetNext(entry,status)) {
      if (entry.isDir) {
        continue;
      }

      std::cout << "Found possible track '" << Lum::Base::WStringToString(entry.name) << "'" << std::endl;

      if (entry.name.length()<4 ||
          entry.name.substr(entry.name.length()-4)!=L".gjt") {
        std::cerr << "wrong suffix!" << std::endl;
        continue;
      }

      std::ifstream file;

      fileName=dirName;
      fileName.SetBaseName(entry.name);

      file.open(Lum::Base::WStringToString(fileName.GetPath()).c_str(),
                std::ios::in|std::ios::binary);

      if (!file) {
        std::cerr << "Cannot open file!" << std::endl;
        continue;
      }

      unsigned long tfid,tfv,tft;

      file.read((char*)&tfid,sizeof(tfid));
      file.read((char*)&tfv,sizeof(tfv));
      file.read((char*)&tft,sizeof(tft));

      if (!file) {
        std::cerr << "Cannot read file header!" << std::endl;
        continue;
      }

      tfid=ntohl(tfid);
      tfv=ntohl(tfv);
      tft=ntohl(tft);

      if (tfid!=trackingFileId) {
        std::cerr << "Wrong magic!" << std::endl;
        continue;
      }

      /* We support the current version and version 1 */
      if (tfv!=trackingFileVersion && tfv!=1) {
        std::cerr << "Wrong file version!" << std::endl;
        continue;
      }

      tracks->Append(Track(entry.name.substr(0,entry.name.length()-4),
                           Lum::Base::SystemTime(tft)));
    }

    tracks->Sort(1);

    tracks->On();
  }

  struct DataPoint
  {
    Lum::Base::SystemTime time;
    double                lat;
    double                lon;
    double                alt;
  };

  void ExportTrack(const std::wstring& name)
  {
    Lum::Base::Path                path;
    std::wstring                   file;
    Lum::Dlg::File::Options        *options=new Lum::Dlg::File::Options();
    Lum::Model::Dir::PatternFilter *filter;
    std::ifstream                  in;
    std::ofstream                  out;
    std::wstring                   homeDir;

    if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDir,
        homeDir)) {
      return;
    }

    path.SetNativeDir(homeDir);
    path.AppendDir(L".gpsjinni");
    path.SetBaseName(name+L".gjt");

    in.open(Lum::Base::WStringToString(path.GetPath()).c_str(),
            std::ios::in|std::ios::binary);

    if (!in) {
      Lum::Base::Status status;

      status.SetToCurrentErrno();
      Lum::Dlg::Msg::ShowOk(GetWindow(),
                            L"Error while opening file!",
                            std::wstring(L"An error occured while opening file:\n")+
                            L"'"+file+L"'\n"+
                            L"Error:\n"+
                            L"'"+status.GetDescription()+L"'");

      return;
    }

    unsigned long tfid,tfv,tft;

    in.read((char*)&tfid,sizeof(tfid));
    in.read((char*)&tfv,sizeof(tfv));
    in.read((char*)&tft,sizeof(tft));

    if (!in) {
      std::cerr << "Cannot read file header!" << std::endl;
      return;
    }

    tfid=ntohl(tfid);
    tfv=ntohl(tfv);
    tft=ntohl(tft);

    if (tfid!=trackingFileId) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),
                            L"Error while opening file!",
                            std::wstring(L"An error occured while opening file:\n")+
                            L"'"+file+L"'\n"+
                            L"Error:\n"+
                            L"'File is not a stored track'");
      return;
    }

    /* We support the current version and version 1 */
    if (tfv!=trackingFileVersion && tfv!=1) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),
                            L"Error while opening file!",
                            std::wstring(L"An error occured while opening file:\n")+
                            L"'"+file+L"'\n"+
                            L"Error:\n"+
                            L"'File format version not supported'");
      return;
    }

    std::cout << "Track data: " << std::hex << tfid << std::dec << " " << tfv << " " << tft << std::endl;

    std::wstring docsDir;

    if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDocumentDir,
                                                        docsDir)) {
      if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDir,
                                                          docsDir)) {
        return;
      }
    }

    path.SetNativeDirAndFile(docsDir,name+L".kml");

    file=path.GetPath();

    options->SetMode(Lum::Dlg::File::Options::modeFile);
    options->SetExistingOnly(false);

    filter=new Lum::Model::Dir::PatternFilter(L"KML Files");
    filter->AddPattern(L"*.kml");
    options->AddFilter(filter);

    if (!Lum::Dlg::File::SaveFile(GetWindow(),
                                  L"Exporting track...",
                                  file,
                                  options)) {
      return;
    }

    out.open(Lum::Base::WStringToString(file).c_str(),
             std::ios::out|std::ios::trunc);

    if (!out) {
      Lum::Base::Status status;

      status.SetToCurrentErrno();
      Lum::Dlg::Msg::ShowOk(GetWindow(),
                            L"Error while opening file!",
                            std::wstring(L"An error occured while opening file:\n")+
                            L"'"+file+L"'\n"+
                            L"Error:\n"+
                            L"'"+status.GetDescription()+L"'");

      return;
    }

    std::list<DataPoint> data;

    while (in) {
      unsigned long flon;
      unsigned long flat;
      unsigned long falt;
      unsigned long ftim;

      in.read((char*)&flon,sizeof(flon));
      in.read((char*)&flat,sizeof(flat));
      in.read((char*)&falt,sizeof(falt));
      in.read((char*)&ftim,sizeof(ftim));

      if (in) {
        DataPoint dataPoint;

        flon=ntohl(flon);
        flat=ntohl(flat);
        falt=ntohl(falt);
        ftim=ntohl(ftim);

        //std::cout << flon/10000000.0 << " " << flat/10000000.0 << " " << falt/100.0 << " " << ftim << std::endl;

        if (tfv==1) {
          dataPoint.lon=flon/10000000.0-90.0;
          dataPoint.lat=flat/10000000.0-180.0;
        }
        else {
          dataPoint.lon=flon/10000000.0-180.0;
          dataPoint.lat=flat/10000000.0-90.0;
        }
        dataPoint.alt=falt/100.0;
        dataPoint.time.SetTime(ftim);

        data.push_back(dataPoint);
      }
    }

    out << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << std::endl;
    out << "<kml xmlns=\"http://www.opengis.net/kml/2.2\">" << std::endl;
    out << "  <Document>" << std::endl;
    out << "    <name>" << Lum::Base::WStringToUTF8(name) << ".kml" << "</name>" << std::endl;
    out << "    <open>1</open>" << std::endl;
    out << std::endl;
    out << "    <Style id=\"startpointStyle\">" << std::endl;
    out << "      <IconStyle>" << std::endl;
    out << "        <Icon>" << std::endl;
    out << "          <href>http://maps.google.com/mapfiles/kml/paddle/A.png</href>" << std::endl;
    out << "        </Icon>" << std::endl;
    out << "      </IconStyle>" << std::endl;
    out << "    </Style>" << std::endl;
    out << std::endl;
    out << "    <Style id=\"waypointStyle\">" << std::endl;
    out << "      <IconStyle>" << std::endl;
    out << "        <Icon>" << std::endl;
    out << "          <href>http://maps.google.com/mapfiles/kml/pushpin/ylw-pushpin.png</href>" << std::endl;
    out << "        </Icon>" << std::endl;
    out << "      </IconStyle>" << std::endl;
    out << "    </Style>" << std::endl;
    out << std::endl;
    out << "    <Style id=\"endpointStyle\">" << std::endl;
    out << "      <IconStyle>" << std::endl;
    out << "        <Icon>" << std::endl;
    out << "          <href>http://maps.google.com/mapfiles/kml/paddle/B.png</href>" << std::endl;
    out << "        </Icon>" << std::endl;
    out << "      </IconStyle>" << std::endl;
    out << "    </Style>" << std::endl;
    out << std::endl;

    out << "    <Folder>" << std::endl;
    out << "      <name>" << Lum::Base::WStringToUTF8(name) << " waypoints with timestamps</name>" << std::endl;
    out << "      <open>1</open>" << std::endl;
    out << std::endl;
    out << "      <Style>" << std::endl;
    out << "        <ListStyle>" << std::endl;
    out << "          <listItemType>checkHideChildren</listItemType>" << std::endl;
    out << "        </ListStyle>" << std::endl;
    out << "      </Style>" << std::endl;

    for (std::list<DataPoint>::const_iterator dataPoint=data.begin();
         dataPoint!=data.end();
         ++dataPoint) {
      out << std::endl;
      out << "      <Placemark>" << std::endl;
      out << "        <TimeStamp>" << std::endl;
      out << "          <when>" << Lum::Base::WStringToString(SystemTimeToISO8601(dataPoint->time)) << "</when>" << std::endl;
      out << "        </TimeStamp>" << std::endl;
      out << "        <styleUrl>#waypointStyle</styleUrl>" << std::endl;
      out << "        <Point>" << std::endl;
      out << "          <coordinates>" << dataPoint->lon << "," << dataPoint->lat << "," << dataPoint->alt << "</coordinates>" << std::endl;
      out << "        </Point>" << std::endl;
      out << "      </Placemark>" << std::endl;
    }
    out << "    </Folder>" << std::endl;

    out << std::endl;
    out << std::endl;

    out << "    <Folder>" << std::endl;
    out << "      <name>" << Lum::Base::WStringToUTF8(name) << " path waypoints</name>" << std::endl;
    out << "      <open>1</open>" << std::endl;
    out << std::endl;
    out << "      <Style>" << std::endl;
    out << "        <ListStyle>" << std::endl;
    out << "          <listItemType>checkHideChildren</listItemType>" << std::endl;
    out << "        </ListStyle>" << std::endl;
    out << "      </Style>" << std::endl;

    for (std::list<DataPoint>::const_iterator dataPoint=data.begin();
         dataPoint!=data.end();
         ++dataPoint) {
      out << std::endl;
      out << "      <Placemark>" << std::endl;
      out << "        <styleUrl>#waypointStyle</styleUrl>" << std::endl;
      out << "        <Point>" << std::endl;
      out << "          <coordinates>" << dataPoint->lon << "," << dataPoint->lat << "," << dataPoint->alt << "</coordinates>" << std::endl;
      out << "        </Point>" << std::endl;
      out << "      </Placemark>" << std::endl;
    }
    out << "    </Folder>" << std::endl;

    out << std::endl;
    out << std::endl;

    if (data.size()>0) {
      out << "    <Placemark>" << std::endl;
      out << "      <styleUrl>#startpointStyle</styleUrl>" << std::endl;
      out << "      <name>Start</name>" << std::endl;
      out << "      <Point>" << std::endl;
      out << "        <coordinates>" << data.begin()->lon << "," << data.begin()->lat << "," << data.begin()->alt << "</coordinates>" << std::endl;
      out << "      </Point>" << std::endl;
      out << "    </Placemark>" << std::endl;
      out << std::endl;
    }

    if (data.size()>0) {
      out << "    <Placemark>" << std::endl;
      out << "      <styleUrl>#endpointStyle</styleUrl>" << std::endl;
      out << "      <name>End</name>" << std::endl;
      out << "      <Point>" << std::endl;
      out << "        <coordinates>" << data.rbegin()->lon << "," << data.rbegin()->lat << "," << data.rbegin()->alt << "</coordinates>" << std::endl;
      out << "      </Point>" << std::endl;
      out << "    </Placemark>" << std::endl;
      out << std::endl;
    }

    out << "    <Placemark>" << std::endl;
    out << "      <name>" << Lum::Base::WStringToUTF8(name) << " path</name>" << std::endl;
    out << std::endl;
    out << "      <Style>" << std::endl;
    out << "        <LineStyle>" << std::endl;
    out << "          <color>ff0000ff</color>" << std::endl;
    out << "          <width>2</width>" << std::endl;
    out << "        </LineStyle>" << std::endl;
    out << "      </Style>" << std::endl;
    out << std::endl;
    out << "      <LineString>" << std::endl;
    //out << "        <extrude>0</extrude>" << std::endl;
    out << "        <tessellate>1</tessellate>" << std::endl;
    out << "        <altitudeMode>clampToGround</altitudeMode>" << std::endl;
    out << "        <coordinates>" << std::endl;

    double lastLat=1000;
    double lastLon=1000;
    double lastAlt=1000;

    for (std::list<DataPoint>::const_iterator dataPoint=data.begin();
         dataPoint!=data.end();
         ++dataPoint) {
      if (dataPoint->lon!=lastLon ||
          dataPoint->lat!=lastLat ||
          dataPoint->alt!=lastAlt) {
        out << "          " << dataPoint->lon << "," << dataPoint->lat << "," << dataPoint->alt << std::endl;
        lastLon=dataPoint->lon;
        lastLat=dataPoint->lat;
        lastAlt=dataPoint->alt;
      }
    }

    out << "        </coordinates>" << std::endl;
    out << "      </LineString>" << std::endl;
    out << "    </Placemark>" << std::endl;
    out << "  </Document>" << std::endl;
    out << "</kml>" << std::endl;

    out.close();
    in.close();
  }

  void DeleteTrack(const std::wstring& name)
  {
    Lum::Base::Path   path;
    Lum::Base::Status status;
    std::wstring      homeDir;

    if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDir,
                                                        homeDir)) {
      return;
    }

    path.SetNativeDir(homeDir);
    path.AppendDir(L".gpsjinni");
    path.SetBaseName(name+L".gjt");

    status=Lum::Base::Path::RemoveFile(path.GetPath());

    if (status) {
      RefreshTracks();
    }
    else {
      Lum::Dlg::Msg::ShowOk(GetWindow(),
                            L"Error while deleting file!",
                            std::wstring(L"An error occured while deletion of file:\n")+
                            L"'"+path.GetPath()+L"'\n"+
                            L"Error:\n"+
                            L"'"+status.GetDescription()+L"'");
    }
  }

  void SetCompassDirectionRaw(double newDirection, double newDeviation)
  {
    direction->Set(newDirection); // set to whatever it is.
    // TODO: eliminate hack to pass deviation to compass:
    if (finite(newDeviation)) {
      direction->SetRange(0, 360 + newDeviation);
    }
  }

  void SetCompassDirectionWeighted(double newDirection, double newDeviation)
  {
    // Check if old and new direction values are available and valid

    double oldDirection;

    if (isnan(newDirection) ||
        isnan(newDeviation) ||
        direction->IsNull() ||
        isnan(oldDirection = direction->Get())
        ) {
      SetCompassDirectionRaw(newDirection, newDeviation);
      return;
    }

    if (newDeviation < 0) {     // just in case...
      newDeviation = -newDeviation;
    }

    // Ignore new direction if deviation is too high; leave old direction valid.

    if (newDeviation > 180) {
      return;
    }

    // Weighten the new direction by its accuracy (that is inverse deviation)
    // in comparison to the old direction is a first experiment...
    // TODO: find better methods to find and eliminate wild shot values.
    newDirection = CalculateWeightedDirection(
                      oldDirection,
                      newDirection,
                      1 - (newDeviation / 360),    // normalized weight in range 0..1.,
                      360);
    SetCompassDirectionRaw(newDirection, newDeviation);
  }

  void ShowChange(const GPSData& data)
  {
    if (data.HasConnected()) {
      connected->Set(data.GetConnected());
    }
    else {
      connected->Set(false);
    }

    if (data.HasOnline()) {
      online->Set(data.GetOnline());
    }
    else {
      online->Set(false);
    }

    if (data.HasFix()) {
      fix->Set(data.GetFix());
    }
    else {
      fix->Set(false);
    }

    if (data.HasTime()) {
      std::wstring t=data.GetTime().GetLocalLocaleDateTime();

      if (data.GetTimeDev()!=0 &&
        finite(data.GetTimeDev())) {
        t+=L"\u00b1"+DoubleToString(data.GetTimeDev());
      }

      time->Set(t);
    }
    else {
      time->SetNull();
    }

    if (data.HasLatitude()) {
      std::wstring l;
      l=GetAngleAsDegreeMinuteSecondString(data.GetLatitude(), L"S ", L"N ", 3);

      l+=L" = ";

      l+=DoubleToString(data.GetLatitude(), 5);

      l+=L"\u00b0";

      if (data.GetLatitudeDev()!=0 &&
          finite(data.GetLatitudeDev())) {
        l+=std::wstring(L"\u00b1")+GetMetricAndUnitAsString(data.GetLatitudeDev());
      }

      latitude->Set(l);
    }
    else {
      latitude->SetNull();
    }

    if (data.HasLongitude()) {
      std::wstring l;
      l=GetAngleAsDegreeMinuteSecondString(data.GetLongitude(), L"W ", L"E ", 3);

      l+=L" = ";

      l+=DoubleToString(data.GetLongitude(), 5);

      l+=+L"\u00b0";

      if (data.GetLongitudeDev()!=0 &&
          finite(data.GetLongitudeDev())) {
        l+=std::wstring(L"\u00b1")+GetMetricAndUnitAsString(data.GetLongitudeDev());
      }

      longitude->Set(l);
    }
    else {
      longitude->SetNull();
    }

    if (data.HasAltitude()) {
      altitude->Set(GetMetricAndUnitAsString(data.GetAltitude(),
                                             data.GetAltitudeDev()));
    }
    else {
      altitude->SetNull();
    }

    if (data.HasSpeed()) {
      double       speed=data.GetSpeed();
      double       speedDev=data.GetSpeedDev();
      std::wstring s;

      // We expect m/s here
      s=GetSpeedAndUnitAsString(speed,speedDev);

      speedValue->Set(speed);
      this->speed->Set(s);
    }
    else {
      speedValue->SetNull();
      speed->SetNull();
    }

    if (data.HasTrack()) {
      std::wstring t=DoubleToString(data.GetTrack());

      if (data.GetTrackDev()!=0 &&
          finite(data.GetTrackDev())) {
        t+=L"\u00b1"+DoubleToString(data.GetTrackDev());
      }

      t+=L"\u00b0";

      track->Set(t);

      if (compassShowRawGPSDirection->Get()) {
        SetCompassDirectionRaw(data.GetTrack(), data.GetTrackDev());
      }
      else {
        SetCompassDirectionWeighted(data.GetTrack(), data.GetTrackDev());
      }
    }
    else {
      track->SetNull();
      direction->SetNull();
    }

    if (data.HasClimb()) {
      climb->Set(GetMetricAndUnitAsString(data.GetClimb(),
                                          data.GetClimbDev()));
    }
    else {
      climb->SetNull();
    }

    if (data.HasSatellites()) {
      // Delete everything in model that is not in current list
      size_t i=1;
      while (i<=satellites->GetRows()) {
        bool found=false;

        for (size_t j=0; j<data.GetSatellites().size(); j++) {
          if (satellites->GetEntry(i).id==data.GetSatellites()[j].id) {
            found=true;
            break;
          }
        }

        if (found) {
          i++;
          continue;
        }
        else {
          satellites->Delete(i);
        }
      }

      // Now add everything to model that is not already in
      for (size_t i=0; i<data.GetSatellites().size(); i++) {
        bool found=false;

        for (size_t j=1; j<=satellites->GetRows(); j++) {
          if (satellites->GetEntry(j).id==data.GetSatellites()[i].id) {
            found=true;

            if (satellites->GetEntry(j).snr!=data.GetSatellites()[i].snr ||
                satellites->GetEntry(j).used!=data.GetSatellites()[i].used) {
              satellites->GetEntry(j)=data.GetSatellites()[i];
              satellites->RedrawRow(j);
            }
            break;
          }
        }

        if (!found) {
          satellites->Append(data.GetSatellites()[i]);
        }
      }
    }
    else {
      satellites->Clear();
    }
  }

  void UpdateStreams(const GPSData& data)
  {
    if (!data.HasFix() || !data.GetFix() ||
        !data.HasSpeed()) {
      speedStream->SetNull();
    }
    else {
      speedStream->Set(0,data.GetSpeed());
    }

    if (!data.HasFix() || !data.GetFix() ||
        !data.HasAltitude()) {
      altitudeStream->SetNull();
    }
    else {
      altitudeStream->Set(0,data.GetAltitude());
    }
  }

  void ShowSettings()
  {
    Lum::Def::PropGroup *props=new Lum::Def::PropGroup();

    Lum::Def::OneOfN    unitModeDef(Lum::Def::Desc(L"Unit mode"),
                                   ::unitMode);
    unitModeDef.AddChoice(Lum::Def::OneOfN::Choice(Lum::Def::Desc(L"meter/kilometer")));
    unitModeDef.AddChoice(Lum::Def::OneOfN::Choice(Lum::Def::Desc(L"feet/miles")));

    props->Boolean(Lum::Def::Boolean(Lum::Def::Desc(L"Prohibit screen blanking"),
                                     ::prohibitScreenBlanking));
    props->Boolean(Lum::Def::Boolean(Lum::Def::Desc(L"Compass shows course"),
                                     ::compassShowCourseDirection));
    props->Boolean(Lum::Def::Boolean(Lum::Def::Desc(L"Compass shows raw GPS data"),
                                     ::compassShowRawGPSDirection));
    props->OneOfN(unitModeDef);

    Lum::Dlg::Properties::Show(this,props);

    delete props;
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() &&
        GetOpenedAction()->IsFinished()) {
      InitializeGPS();
      Lum::OS::display->AddTimer(refreshTime,0,refreshTimer);
      Lum::OS::display->AddTimer(reconnectTime,0,reconnectCheckTimer);
      RefreshTracks();
      LoadConfig();
    }
    else if (model==GetClosedAction() &&
        GetClosedAction()->IsFinished()) {
      DeinitializeGPS();
      SaveConfig();
    }
    else if (model==aboutAction &&
             aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==dataChangedAction &&
             dataChangedAction->IsFinished()) {
      GPSData dataCopy;

      GetGPSDataCopy(dataCopy);
      AddTrackingRecord(dataCopy);
      ShowChange(dataCopy);
    }
    else if (model==refreshTimer &&
             refreshTimer->IsFinished()) {
      GPSData dataCopy;

      GetGPSDataCopy(dataCopy);
      ShowChange(dataCopy);
      AddTrackingRecord(dataCopy);
      UpdateStreams(dataCopy);

      Lum::OS::display->AddTimer(refreshTime,0,refreshTimer);
    }
    else if (model==reconnectCheckTimer &&
             reconnectCheckTimer->IsFinished()) {
      ReconnectIfDisconnected();

      Lum::OS::display->AddTimer(reconnectTime,0,reconnectCheckTimer);
    }
    else if (model==startTrackingAction &&
             startTrackingAction->IsFinished() &&
             startTrackingAction->IsEnabled()) {
      StartTracking();
    }
    else if (model==stopTrackingAction &&
             stopTrackingAction->IsFinished() &&
             stopTrackingAction->IsEnabled()) {
      StopTracking();
    }
    else if (model==deleteTrackAction &&
             deleteTrackAction->IsFinished() &&
             deleteTrackAction->IsEnabled()) {
      DeleteTrack(tracks->GetEntry(trackSelection->GetLine()).GetName());
    }
    else if (model==exportTrackAction &&
             exportTrackAction->IsFinished() &&
             exportTrackAction->IsEnabled()) {
      ExportTrack(tracks->GetEntry(trackSelection->GetLine()).GetName());
    }
    else if (model==trackSelection) {
      if (trackSelection->HasSelection()) {
        deleteTrackAction->Enable();
        exportTrackAction->Enable();
      }
      else {
        deleteTrackAction->Disable();
        exportTrackAction->Disable();
      }
    }
    else if (model==settingsAction &&
             settingsAction->IsFinished()) {
      ShowSettings();
    }
    else if (model==prohibitScreenBlanking) {
      Lum::Manager::Display *displayManager=Lum::Manager::Display::Instance();

      std::cout << "prohibitScreenBlanking changed! displayManager "
                << (displayManager ? "!= NULL" : "== NULL") << std::endl;

      if (displayManager!=NULL) {
        displayManager->AllowScreenBlanking(!prohibitScreenBlanking->Get());
      }
    }
    else if (model==unitMode) {
      std::cout << "unitMode changed!" << std::endl;
      GPSData dataCopy;

      GetGPSDataCopy(dataCopy);
      ShowChange(dataCopy);
      //UpdateStreams(dataCopy);
    }
    else if (model==compassShowRawGPSDirection) {
      std::cout << "compassShowRawGPSDirection changed!" << std::endl;
      GPSData dataCopy;

      GetGPSDataCopy(dataCopy);
      ShowChange(dataCopy);
    }

    Dialog::Resync(model,msg);
  }

  void HandleChange()
  {
    Lum::OS::display->QueueActionForAsyncNotification(dataChangedAction);
  }
};

#if defined(HAVE_LIB_GPS)
  #if GPSD_API_MAJOR_VERSION>=4
static void cb(struct gps_data_t* data, char * buf, size_t len)
  #else
static void cb(struct gps_data_t* data, char * buf, size_t len, int level)
  #endif
{
  assert(window!=NULL);

  Lum::OS::WriteGuard<Lum::OS::RWMutex> guard(window->dataMutex);

  GPSData                               &gpsData=window->data;

  gpsData.SetConnected(true);

  if (data->set & ONLINE_SET) {
    gpsData.SetOnline(data->online);
  }

  if (data->set & STATUS_SET) {
    gpsData.SetFix(data->status!=STATUS_NO_FIX);
  }

  if ((data->set & LATLON_SET)  && !isnan(data->fix.latitude)) {
    gpsData.SetLatitude(data->fix.latitude,data->fix.epy);
  }

  if ((data->set & LATLON_SET)  && !isnan(data->fix.longitude)) {
    gpsData.SetLongitude(data->fix.longitude,data->fix.epx);
  }

  if ((data->set & ALTITUDE_SET) && !isnan(data->fix.altitude)) {
    gpsData.SetAltitude(data->fix.altitude,data->fix.epv);
  }

  if ((data->set & SPEED_SET) && !isnan(data->fix.speed)) {
    gpsData.SetSpeed(data->fix.speed,data->fix.eps); // m/s
  }

  if ((data->set & TRACK_SET) && !isnan(data->fix.track)) {
    gpsData.SetTrack(data->fix.track,data->fix.epd);
  }

  if ((data->set & CLIMB_SET) && !isnan(data->fix.climb)) {
    gpsData.SetClimb(data->fix.climb,data->fix.epc);
  }

  if (data->set & TIME_SET) {
    Lum::Base::SystemTime time(lround(data->fix.time));
    gpsData.SetTime(time,data->fix.ept);
  }

  if (data->set & SATELLITE_SET) {
    std::vector<GPSData::Satellite> satellites;

    for (int i=0; i<data->satellites_used; i++) {
      GPSData::Satellite s;

      s.id=data->PRN[i];
      s.used=data->used[i];
      s.snr=data->ss[i];

      satellites.push_back(s);
    }

    gpsData.SetSatellites(satellites);
  }

  window->HandleChange();
}
#elif defined(HAVE_LIB_LOCATION)
static void gps_location_changed(LocationGPSDevice *device, gpointer userdata)
{
  std::cout << "gps_location_changed!" << std::endl;
  assert(window!=NULL);

  Lum::OS::WriteGuard<Lum::OS::RWMutex> guard(window->dataMutex);

  GPSData                               &gpsData=window->data;

  gpsData.SetConnected(true);
  gpsData.SetOnline(device->online);
  gpsData.SetFix(device->status!=LOCATION_GPS_DEVICE_STATUS_NO_FIX);

  /* Fix information */
  LocationGPSDeviceFix *fix=device->fix;

  if (fix->fields & LOCATION_GPS_DEVICE_TIME_SET) {
    Lum::Base::SystemTime time(lround(fix->time));
    gpsData.SetTime(time,fix->ept);
  }
  if (fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET && !isnan(fix->latitude)) {
    gpsData.SetLatitude(fix->latitude,fix->eph/100);
  }
  if (fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET && !isnan(fix->longitude)) {
    gpsData.SetLongitude(fix->longitude,fix->eph/100);
  }
  if (fix->fields & LOCATION_GPS_DEVICE_ALTITUDE_SET && !isnan(fix->altitude)) {
    gpsData.SetAltitude(fix->altitude,fix->epv);
  }
  if (fix->fields & LOCATION_GPS_DEVICE_SPEED_SET && !isnan(fix->speed)) {
    gpsData.SetSpeed(fix->speed*1000.0/3600,
                     fix->eps*1000.0/3600); // km/h => m/s
  }
  if (fix->fields & LOCATION_GPS_DEVICE_TRACK_SET && !isnan(fix->track)) {
    gpsData.SetTrack(fix->track,fix->epd);
  }
  if (fix->fields & LOCATION_GPS_DEVICE_CLIMB_SET && !isnan(fix->climb)) {
    gpsData.SetClimb(fix->climb,fix->epc);
  }

  std::vector<GPSData::Satellite> satellites;

  if (device->satellites!=NULL) {
    for (size_t i=0; i<device->satellites->len; i++) {
      LocationGPSDeviceSatellite *sat=(LocationGPSDeviceSatellite*)g_ptr_array_index(device->satellites,i);
      GPSData::Satellite         s;

      s.id=sat->prn;
      s.used=sat->in_use;
      s.snr=sat->signal_strength;

      satellites.push_back(s);
    }
  }

  gpsData.SetSatellites(satellites);

  window->HandleChange();
}

static void gps_location_connected(LocationGPSDevice *new_device, gpointer userdata) {
  std::cout << "Connected..." << std::endl;
}

static void gps_location_disconnected(LocationGPSDevice *new_device, gpointer userdata){
  std::cout << "disconnected..." << std::endl;
}

static void gps_location_started(LocationGPSDControl *control, gpointer userdata) {
  std::cout << "GPSD started!" << std::endl;
}

static void gps_location_stopped(LocationGPSDControl *control, gpointer userdata) {
  std::cout << "GPSD stoped!" << std::endl;
}

static void gps_location_error(LocationGPSDControl *control, gpointer userdata) {
  std::cout << "GPSD error!" << std::endl;
}
#endif

class Main : public Lum::GUIApplication<MyWindow>
{
public:
  bool Initialize()
  {
    std::wstring appDataDir;

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(L"Where the #*~@# am I?");
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2009, Tim Teulings");
    info.SetLicense(L"GNU Public License");

#if defined(APP_DATADIR)
    Lum::Manager::FileSystem::Instance()->SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    if (Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::appDataDir,
                                                       appDataDir)) {

      Lum::Base::Path path;

      path.SetNativeDir(appDataDir);
      path.AppendDir(L"icons");

      Lum::Images::loader->AddApplicationSearchPath(path);
    }

    Lum::Images::loader->SetNormalImagePostfix(L"");

    Lum::Base::Path path;

    path.SetDir(L"data");
    path.AppendDir(L"icons");
    Lum::Images::loader->AddApplicationSearchPath(path);

    return Lum::GUIApplication<MyWindow>::Initialize();
  }
};

LUM_MAIN(Main,Lum::Base::StringToWString(PACKAGE_NAME))

