/*
  GPSJinni - show raw data from the GPS subsystem.
  Copyright (C) 2010  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Util.h"

#include <cmath>
#include <sstream>
#include <iomanip>

#include <Lum/Base/String.h>

#include "Configuration.h"

std::wstring DoubleToString(double value)
{
  if (!isnan(value)) {
    std::stringstream buffer;

    buffer.imbue(std::locale(""));
    buffer << std::fixed;
    buffer << std::showpoint;
    buffer.precision(2);

    buffer << value;

    return Lum::Base::StringToWString(buffer.str());
  }
  else {
    return L"";
  }
}

std::wstring GetSpeedNoDataUnit()
{
  if (unitMode->Get()==1) {
    return L"m/s";
  }
  else {
    return L"ft/s";
  }
}

std::wstring GetSpeedAsString(double speed)
{
  if (unitMode->Get()==1) {
    if (speed>=1/3.6) {
      // km/h
      return DoubleToString(speed*3.6);
    }
    else {
      // m/s
      return DoubleToString(speed);
    }
  }
  else {
    if (speed>=1/2.236936) {
      // mi/h
      return DoubleToString(speed*2.236936);
    }
    else {
      //ft/s
      return DoubleToString(speed*3.280840);
    }
  }
}

std::wstring GetSpeedUnitAsString(double speed)
{
  if (unitMode->Get()==1) {
    if (speed>=1/3.6) {
      return L"km/h";
    }
    else {
      return L"m/s";
    }
  }
  else {
    if (speed>=1/2.236936) {
      return L"mi/h";
    }
    else {
      return L"ft/s";
    }
  }
}

std::wstring GetSpeedAndUnitAsString(double speed)
{
  if (unitMode->Get()==1) {
    if (speed>=1/3.6) {
      return DoubleToString(speed*3.6)+L" km/h";
    }
    else {
      return DoubleToString(speed)+L" m/s";
    }
  }
  else {
    if (speed>=1/2.236936) {
      return DoubleToString(speed*2.236936)+L" mi/h";
    }
    else {
      return DoubleToString(speed*3.280840)+L" ft/s";
    }
  }
}

std::wstring GetSpeedAndUnitAsString(double speed, double dev)
{
  if (unitMode->Get()==1) {
    if (speed>=1/3.6) {
      std::wstring result=DoubleToString(speed*3.6);
      
      if (dev!=0 && finite(dev)) {
        result+=L"\u00b1"+DoubleToString(dev*3.6);
      }

      result+=L"km/h";
      
      return result;
    }
    else {
      std::wstring result=DoubleToString(speed);
      
      if (dev!=0 && finite(dev)) {
        result+=L"\u00b1"+DoubleToString(dev);
      }

      result+=L"m/s";
      
      return result;
    }
  }
  else {
    if (speed>=1/2.236936) {
      std::wstring result=DoubleToString(speed*2.236936);
      
      if (dev!=0 && finite(dev)) {
        result+=L"\u00b1"+DoubleToString(dev*2.236936);
      }
      
      result+=L"mi/h";
      
      return result;
    }
    else {
      std::wstring result=DoubleToString(speed*3.280840);
      
      if (dev!=0 && finite(dev)) {
        result+=L"\u00b1"+DoubleToString(dev*3.280840);
      }
      
      result+=L"ft/s";
      
      return result;
    }
  }
}

std::wstring GetMetricAndUnitAsString(double metric)
{
  if (unitMode->Get()==1) {
    if (metric>=1000) {
      return DoubleToString(metric/1000)+L"km";
    }
    else {
      return DoubleToString(metric)+L"m";
    }
  }
  else {
    if (metric>1609.344) {
      return DoubleToString(metric/1609.344)+L"mi";
    }
    else {
      return DoubleToString(metric/0.30480)+L"ft";
    }
  }
}
std::wstring GetMetricAndUnitAsString(double metric, double dev)
{
  if (unitMode->Get()==1) {
    if (metric>=1000) {
      std::wstring result=DoubleToString(metric/1000);
      
      if (dev!=0 && finite(dev)) {
        result+=L"\u00b1"+DoubleToString(dev/1000);
      }

      result+=L"km";
      
      return result;
    }
    else {
      std::wstring result=DoubleToString(metric);
      
      if (dev!=0 && finite(dev)) {
        result+=L"\u00b1"+DoubleToString(dev);
      }

      result+=L"m";
      
      return result;
    }
  }
  else {
    if (metric>1609.344) {
      std::wstring result=DoubleToString(metric/1609.344);
      
      if (dev!=0 && finite(dev)) {
        result+=L"\u00b1"+DoubleToString(dev/1609.344);
      }
      
      result+=L"mi";
      
      return result;
    }
    else {
      std::wstring result=DoubleToString(metric/0.30480);
      
      if (dev!=0 && finite(dev)) {
        result+=L"\u00b1"+DoubleToString(dev/0.30480);
      }
      
      result+=L"ft";
      
      return result;
    }
  }
}

