/*
  GPSJinni - show raw data from the GPS subsystem.
  Copyright (C) 2009  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Compass.h"

#include <cmath>

#include <Lum/OS/Font.h>

#include <iostream>
Compass::Compass()
 : arrowColor(0.0,0.0,0.0,Lum::OS::Display::blackColor),
   arrowShadowColor(0.6,0.6,0.6,Lum::OS::Display::whiteColor)
{
  // no code
}

bool Compass::SetModel(Lum::Base::Model* model)
{
  this->model=dynamic_cast<Lum::Model::Double*>(model);

  Lum::Control::SetModel(this->model);

  return this->model.Valid();
}

void Compass::CalcSize()
{
  width=25;
  height=25;

  minWidth=width;
  minHeight=height;

  Lum::Control::CalcSize();
}

void Compass::Draw(int x, int, size_t w, size_t h)
{
  Lum::Control::Draw(x,y,w,h);

  if (!Intersect(x,y,w,h)) {
    return;
  }

  /* --- */


  Lum::OS::DrawInfo         *draw=GetDrawInfo();
  Lum::OS::FontRef          font=Lum::OS::display->GetFont();
  size_t                    directionBox=0;
  size_t                    area;
  size_t                    radius=0;
  std::vector<std::wstring> directions;

  directions.push_back(L"N");
  directions.push_back(L"E");
  directions.push_back(L"S");
  directions.push_back(L"W");

  for (size_t i=0; i<directions.size(); i++) {
    Lum::OS::FontExtent extent;

    font->StringExtent(directions[i],extent,Lum::OS::Font::bold);

    directionBox=std::max(directionBox,extent.width);
    directionBox=std::max(directionBox,extent.height);
  }

  area=std::min(width,height);

  radius=(area-
          2*std::max(Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceIntraObject),
                     Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceIntraObject))-
          2*directionBox)/2-1;

  int ax=this->x+(width-area)/2;
  int ay=this->y+(height-area)/2;
  int rx=ax+(area/2);
  int ry=ay+(area/2);

  /*
  draw->PushForeground(Lum::OS::Display::blackColor);
  draw->DrawRectangle(ax,ay,area,area);
  draw->PopForeground();*/

  // scale background
  draw->PushForeground(Lum::OS::Display::whiteColor);
  draw->FillArc(ax+(area-radius*2+1)/2,ay+(area-radius*2+1)/2,
                radius*2+1,radius*2+1,
                0,360*64);
  draw->PopForeground();

  // scale frame
  draw->PushForeground(Lum::OS::Display::blackColor);
  draw->DrawArc(ax+(area-radius*2+1)/2,ay+(area-radius*2+1)/2,
                radius*2+1,radius*2+1,
                0,360*64);
  draw->PopForeground();

  // directions
  draw->PushForeground(Lum::OS::Display::textColor);
  draw->PushFont(font,Lum::OS::Font::bold);

  Lum::OS::FontExtent extent;

  // north
  font->StringExtent(directions[0],extent,Lum::OS::Font::bold);
  draw->DrawString(ax+(area-directionBox)/2+(directionBox-extent.width)/2-extent.left,
                   ay+(directionBox-extent.height)/2+extent.ascent,
                   directions[0]);

  // east
  font->StringExtent(directions[1],extent,Lum::OS::Font::bold);
  draw->DrawString(ax+area-directionBox+(directionBox-extent.width)/2-extent.left,
                   ay+(area-directionBox)/2+(directionBox-extent.height)/2+extent.ascent,
                   directions[1]);

  // south
  font->StringExtent(directions[2],extent,Lum::OS::Font::bold);
  draw->DrawString(ax+(area-directionBox)/2+(directionBox-extent.width)/2-extent.left,
                   ay+height-directionBox+(directionBox-extent.height)/2+extent.ascent,
                   directions[2]);

  // west
  font->StringExtent(directions[3],extent,Lum::OS::Font::bold);
  draw->DrawString(ax+(directionBox-extent.width)/2-extent.left,
                   ay+(area-directionBox)/2+(directionBox-extent.height)/2+extent.ascent,
                   directions[3]);

  draw->PopFont();
  draw->PopForeground();

  if (!model.Valid() || model->IsNull() || isnan(model->Get())) {
    return;
  }

  Lum::OS::DrawInfo::Point points[5];
  double                   direction=*model;

  points[0].x=rx+lround(radius*90/100*sin(((direction+  0)*M_PI)/180));
  points[0].y=ry-lround(radius*90/100*cos(((direction+  0)*M_PI)/180));
  points[1].x=rx+lround(radius*90/100*sin(((direction+160)*M_PI)/180));
  points[1].y=ry-lround(radius*90/100*cos(((direction+160)*M_PI)/180));
  points[2].x=rx+lround(radius*50/100*sin(((direction+180)*M_PI)/180));
  points[2].y=ry-lround(radius*50/100*cos(((direction+180)*M_PI)/180));
  points[3].x=rx+lround(radius*90/100*sin(((direction-160)*M_PI)/180));
  points[3].y=ry-lround(radius*90/100*cos(((direction-160)*M_PI)/180));
  points[4]=points[0];

  for (size_t i=0; i<5; i++) {
    points[i].x+=3;
    points[i].y+=3;
  }

  draw->PushForeground(arrowShadowColor);
  draw->FillPolygon(points,5);
  draw->PopForeground();

  for (size_t i=0; i<5; i++) {
    points[i].x-=3;
    points[i].y-=3;
  }

  draw->PushForeground(arrowColor);
  draw->FillPolygon(points,5);
  draw->PopForeground();

}

void Compass::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==this->model) {
    Redraw();
  }

  Control::Resync(model,msg);
}

