/*
  GPSJinni - show raw data from the GPS subsystem.
  Copyright (C) 2009  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config.h"

#include <signal.h>

#include <glib.h>

#include <cerrno>
#include <cmath>

#if defined (HAVE_LIB_GPS)
  #include <gps.h>
#endif

#if defined(HAVE_LIB_LOCATION)
extern "C" {
  #include <location/location-gps-device.h>
  #include <location/location-gpsd-control.h>
}
#endif

#ifndef SIMPLE_CODE

#include <netinet/in.h>

#include <fstream>
#include <iomanip>
#include <sstream>

#include <Lum/Def/Menu.h>
#include <Lum/Def/MultiView.h>

#include <Lum/OS/Probe.h>
#include <Lum/OS/Thread.h>

#include <Lum/Base/DateTime.h>
#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/File.h>
#include <Lum/Dlg/Msg.h>
#include <Lum/Dlg/Value.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Boolean.h>
#include <Lum/Model/DataStream.h>
#include <Lum/Model/Header.h>
#include <Lum/Model/String.h>

#include <Lum/OS/Main.h>

#include <Lum/Button.h>
#include <Lum/Dialog.h>
#include <Lum/Label.h>
#include <Lum/LED.h>
#include <Lum/Panel.h>
#include <Lum/PercentBar.h>
#include <Lum/Tab.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>
#include <Lum/View.h>

#include "Compass.h"
#include "TimeLine.h"
#endif
#include <iostream>

#ifdef SIMPLE_CODE
GMainContext *context;
GMainLoop    *loop = NULL;
#else

static long obsoleteDataAge = 30; // maximum ge of data in seconds until value is invalid
static long reconnectTime   = 10; // Period of time after which we check for disconnection and
                                  // trigger reconnection
static long refreshTime     = 10; // Period of time after which we update the stream views

class GPSData
{
public:
  struct Satellite
  {
    int  id;
    bool used;
    int  snr;
  };

private:
  bool                   connected;
  Lum::Base::SystemTime  connectedLastData;

  bool                   online;
  Lum::Base::SystemTime  onlineLastData;

  bool                   fix;
  Lum::Base::SystemTime  fixLastData;

  double                 latitude;
  Lum::Base::SystemTime  latitudeLastData;

  double                 longitude;
  Lum::Base::SystemTime  longitudeLastData;

  double                 altitude;
  Lum::Base::SystemTime  altitudeLastData;

  double                 speed;
  Lum::Base::SystemTime  speedLastData;

  double                 track;
  Lum::Base::SystemTime  trackLastData;

  double                 climb;
  Lum::Base::SystemTime  climbLastData;

  Lum::Base::SystemTime  time;
  Lum::Base::SystemTime  timeLastData;

  std::vector<Satellite> satellites;
  Lum::Base::SystemTime  satellitesLastData;

public:
  GPSData()
   : connectedLastData(0),
     onlineLastData(0),
     fixLastData(0),
     latitudeLastData(0),
     longitudeLastData(0),
     altitudeLastData(0),
     speedLastData(0),
     trackLastData(0),
     climbLastData(0),
     timeLastData(0),
     satellitesLastData(0)
  {
    // no code
  }

  //
  // Connected
  //
  void SetConnected(bool connected)
  {
    connectedLastData.SetToNow();
    this->connected=connected;
  }

  bool HasConnected() const
  {
    Lum::Base::SystemTime now;

    return now-connectedLastData<obsoleteDataAge;
  }

  bool GetConnected() const
  {
    assert(HasConnected());

    return connected;
  }

  //
  // Online
  //
  void SetOnline(bool online)
  {
    onlineLastData.SetToNow();
    this->online=online;
  }

  bool HasOnline() const
  {
    Lum::Base::SystemTime now;

    return now-onlineLastData<obsoleteDataAge;
  }

  bool GetOnline() const
  {
    assert(HasOnline());

    return online;
  }

  //
  // Fix
  //
  void SetFix(bool fix)
  {
    fixLastData.SetToNow();
    this->fix=fix;
  }

  bool HasFix() const
  {
    Lum::Base::SystemTime now;

    return now-fixLastData<obsoleteDataAge;
  }

  bool GetFix() const
  {
    assert(HasFix());

    return fix;
  }

  //
  // Latitude
  //
  void SetLatitude(double latitude)
  {
    latitudeLastData.SetToNow();
    this->latitude=latitude;
  }

  bool HasLatitude() const
  {
    Lum::Base::SystemTime now;

    return now-latitudeLastData<obsoleteDataAge;
  }

  double GetLatitude() const
  {
    assert(HasLatitude());

    return latitude;
  }

  //
  // Longitude
  //
  void SetLongitude(double longitude)
  {
    longitudeLastData.SetToNow();
    this->longitude=longitude;
  }

  bool HasLongitude() const
  {
    Lum::Base::SystemTime now;

    return now-longitudeLastData<obsoleteDataAge;
  }

  double GetLongitude() const
  {
    assert(HasLongitude());

    return longitude;
  }

  //
  // Altitude
  //
  void SetAltitude(double altitude)
  {
    altitudeLastData.SetToNow();
    this->altitude=altitude;
  }

  bool HasAltitude() const
  {
    Lum::Base::SystemTime now;

    return now-altitudeLastData<obsoleteDataAge;
  }

  double GetAltitude() const
  {
    assert(HasAltitude());

    return altitude;
  }

  //
  // Speed
  //
  void SetSpeed(double speed)
  {
    speedLastData.SetToNow();
    this->speed=speed;
  }

  bool HasSpeed() const
  {
    Lum::Base::SystemTime now;

    return now-speedLastData<obsoleteDataAge;
  }

  double GetSpeed() const
  {
    assert(HasSpeed());

    return speed;
  }

  //
  // Track
  //
  void SetTrack(double track)
  {
    trackLastData.SetToNow();
    this->track=track;
  }

  bool HasTrack() const
  {
    Lum::Base::SystemTime now;

    return now-trackLastData<obsoleteDataAge;
  }

  double GetTrack() const
  {
    assert(HasTrack());

    return track;
  }

  //
  // Climb
  //
  void SetClimb(double climb)
  {
    climbLastData.SetToNow();
    this->climb=climb;
  }

  bool HasClimb() const
  {
    Lum::Base::SystemTime now;

    return now-climbLastData<obsoleteDataAge;
  }

  double GetClimb() const
  {
    assert(HasClimb());

    return climb;
  }

  //
  // Time
  //
  void SetTime(Lum::Base::SystemTime& time)
  {
    timeLastData.SetToNow();
    this->time=time;
  }

  bool HasTime() const
  {
    Lum::Base::SystemTime now;

    return now-timeLastData<obsoleteDataAge;
  }

  Lum::Base::SystemTime GetTime() const
  {
    assert(HasTime());

    return time;
  }

  //
  // Satellites
  //
  void SetSatellites(const std::vector<Satellite>& satellites)
  {
    satellitesLastData.SetToNow();
    this->satellites=satellites;
  }

  bool HasSatellites() const
  {
    Lum::Base::SystemTime now;

    return now-satellitesLastData<obsoleteDataAge;
  }

  const std::vector<Satellite>& GetSatellites() const
  {
    assert(HasSatellites());

    return satellites;
  }
};

typedef Lum::Model::StdTable<GPSData::Satellite> SatellitesModel;
typedef Lum::Base::Reference<SatellitesModel>    SatellitesModelRef;

double GetRelativeQualityFromDbm(double quality)
{
  if (quality<0) {
    return 0;
  }
  if (quality>=0 && quality<40) {
    return 5*quality/2;
  }
  else {
    return 100;
  }
}

class SatellitesDataProvider : public SatellitesModel::DataProvider
{
private:
  Lum::PercentBar                 *bar;
  Lum::Model::DoubleDataStreamRef signal;

public:
  SatellitesDataProvider()
  : bar(new Lum::PercentBar()),
    signal(new Lum::Model::DoubleDataStream())
  {
    bar->SetFlex(true,true);
    bar->SetModel(signal);

    signal->SetNotificationMode(Lum::Model::DoubleDataStream::notifyExplicit);
    signal->SetChannels(1);
  }

  ~SatellitesDataProvider()
  {
    delete bar;
  }

  std::wstring GetString(const SatellitesModel::Iterator& iter, size_t column) const
  {
    switch (column) {
    case 1:
      return Lum::Base::NumberToWString(iter->id);
    case 2:
      return iter->used ? L"X" : L"";
    default:
      return L"";
    }
  }

  Lum::Object* GetObject(const SatellitesModel::Iterator& iter, size_t column) const
  {
    switch (column) {
    case 3:
      signal->Set(0,GetRelativeQualityFromDbm(iter->snr)/100.0);
      return bar;
    default:
      return NULL;
    }
  }
};

class MyWindow;
static MyWindow*         window=NULL;
static Lum::Def::AppInfo info;
static unsigned long     trackingFileId='G'*256*256*256+'J'*256*256+'T'*256+'\0';
static unsigned long     trackingFileVersion=1;
#endif

#if defined(HAVE_LIB_GPS)
static void cb( struct gps_data_t * data,
               char * buf,
               size_t len,
               int level );
#endif

#ifndef SIMPLE_CODE

std::wstring DoubleToString(double value)
{
  if (!isnan(value)) {
    std::stringstream buffer;

    buffer.imbue(std::locale(""));
    buffer << std::fixed;
    buffer << std::showpoint;
    buffer.precision(2);

    buffer << value;

    return Lum::Base::StringToWString(buffer.str());
  }
  else {
    return L"";
  }
}

class Track
{
private:
  std::wstring          name;
  Lum::Base::SystemTime time;

public:
  Track(const std::wstring& name, const Lum::Base::SystemTime& time)
  : name(name),
    time(time)
  {
    // no code
  }

  std::wstring GetName() const
  {
    return name;
  }

  std::wstring GetTimeString() const
  {
    return time.GetLocalLocaleDate()+L" "+time.GetLocalLocaleTime();
  }
};


typedef Lum::Model::StdTable<Track> TracksModel;
typedef Lum::Base::Reference<TracksModel> TracksModelRef;

class TracksDataProvider : public TracksModel::DataProvider
{
public:
  std::wstring GetString(const TracksModel::Iterator& iter, size_t column) const
  {
    switch (column) {
    case 1:
      return iter->GetName();
    case 2:
      return iter->GetTimeString();
    default:
      assert(false);
    }
  }
};

class TrackComparator : public TracksModel::Comparator
{
public:
  bool operator()(const Track& a, const Track& b, size_t column, bool down) const
  {
    assert(column==1);

    if (down) {
      return a.GetName()<b.GetName();
    }
    else {
      return a.GetName()>b.GetName();
    }
  }
};

class MyWindow : public Lum::Dialog
{
private:
  Lum::Model::BooleanRef   connected;
  Lum::Model::BooleanRef   online;
  Lum::Model::BooleanRef   fix;
  Lum::Model::StringRef    time;
  Lum::Model::StringRef    latitude;
  Lum::Model::StringRef    longitude;
  Lum::Model::StringRef    altitude;
  Lum::Model::StringRef    speed;
  Lum::Model::StringRef    track;
  Lum::Model::DoubleRef    direction;
  Lum::Model::StringRef    climb;
  SatellitesModelRef       satellites;
  Lum::Model::DoubleDataStreamRef speedStream;
  Lum::Model::DoubleDataStreamRef altitudeStream;
  TracksModelRef           tracks;
  Lum::Model::SingleLineSelectionRef trackSelection;

  Lum::Model::ActionRef    startTrackingAction;
  Lum::Model::ActionRef    stopTrackingAction;
  Lum::Model::ActionRef    deleteTrackAction;
  Lum::Model::ActionRef    exportTrackAction;
  Lum::Model::ActionRef    aboutAction;

  Lum::Model::ActionRef    dataChangedAction;
  Lum::Model::ActionRef    refreshTimer;
  Lum::Model::ActionRef    reconnectCheckTimer;

  Lum::Model::BooleanRef   tracking;

  std::ofstream            trackFile;

#if defined(HAVE_LIB_LOCATION)
  LocationGPSDControl      *control;
#endif
#if defined(HAVE_LIB_GPS)
  struct gps_data_t        *gpsData;
  pthread_t                gpsThread;
#endif

public:
  Lum::OS::RWMutex         dataMutex;
  GPSData                  data;

private:
  Lum::TextValue* CreateTextValue(Lum::Model::StringRef& model, size_t size)
  {
    Lum::TextValue *textValue;

    textValue=Lum::TextValue::Create(model,Lum::TextValue::left,true,false);
    textValue->SetWidth(Lum::Base::Size::stdCharWidth,size);

    return textValue;
  }

  void InitializeGPS()
  {
    std::cout << "Initialize GPS..." << std::endl;
#if defined(HAVE_LIB_LOCATION)
    control=location_gpsd_control_get_default();

    location_gpsd_control_start(control);
#endif
#if defined(HAVE_LIB_GPS)
    // This fixed a bug in libgps, using locale aware functions to parse non-locale-aware float values
    setlocale(LC_NUMERIC,"C");
    gpsData=::gps_open("127.0.0.1","2947");

    if (gpsData!=NULL) {
      gps_query(gpsData,"j=0w+x\n");
      gps_set_callback(gpsData,cb,&gpsThread);
      std::cout << "GPS initialized!" << std::endl;
    }
    else {
      std::cout << "GPS NOT initialized: " << errno << std::endl;
    }
#endif
  }

  void DeinitializeGPS()
  {
    std::cout << "Deinitialize GPS..." << std::endl;

#if defined(HAVE_LIB_GPS)
    if (gpsData!=NULL) {
      ::gps_close(gpsData);
      gpsData=NULL;
    }
#endif

#if defined(HAVE_LIB_LOCATION)
    location_gpsd_control_stop(control);

    /*Device control cleanup */
    g_object_unref(control);
    control=NULL;
#endif
    std::cout << "GPS deinitialized!" << std::endl;
  }

  void ReconnectIfDisconnected()
  {
    if (data.HasConnected() && data.GetConnected()) {
      return;
    }

#if defined(HAVE_LIB_GPS)
    if (gpsData!=NULL) {
      ::gps_close(gpsData);
      gpsData=NULL;
    }

    gpsData=::gps_open("127.0.0.1","2947");

    if (gpsData!=NULL) {
      gps_query(gpsData,"j=0w+x\n");
      gps_set_callback(gpsData,cb,&gpsThread);
      data.SetConnected(true);
      std::cout << "GPS initialized!" << std::endl;
    }
    else {
      data.SetConnected(false);
      std::cout << "GPS NOT initialized: " << errno << std::endl;
    }
#endif
  }

public:
  MyWindow()
  : connected(new Lum::Model::Boolean(false)),
    online(new Lum::Model::Boolean(false)),
    fix(new Lum::Model::Boolean(false)),
    time(new Lum::Model::String(L"")),
    latitude(new Lum::Model::String(L"")),
    longitude(new Lum::Model::String(L"")),
    altitude(new Lum::Model::String(L"")),
    speed(new Lum::Model::String(L"")),
    track(new Lum::Model::String(L"")),
    direction(new Lum::Model::Double()),
    climb(new Lum::Model::String(L"")),
    satellites(new SatellitesModel(new SatellitesDataProvider())),
    speedStream(new Lum::Model::DoubleDataStream()),
    altitudeStream(new Lum::Model::DoubleDataStream()),
    tracks(new TracksModel(new TracksDataProvider(),new TrackComparator())),
    trackSelection(new Lum::Model::SingleLineSelection()),
    startTrackingAction(new Lum::Model::Action()),
    stopTrackingAction(new Lum::Model::Action()),
    deleteTrackAction(new Lum::Model::Action()),
    exportTrackAction(new Lum::Model::Action()),
    aboutAction(new Lum::Model::Action()),
    dataChangedAction(new Lum::Model::Action()),
    refreshTimer(new Lum::Model::Action()),
    reconnectCheckTimer(new Lum::Model::Action()),
    tracking(new Lum::Model::Boolean(false))
#if defined(HAVE_LIB_LOCATION)
    ,control(NULL)
#endif
#if defined(HAVE_LIB_GPS)
    ,gpsData(NULL)
#endif
  {
    Observe(GetOpenedAction());
    Observe(startTrackingAction);
    Observe(stopTrackingAction);
    Observe(deleteTrackAction);
    Observe(exportTrackAction);
    Observe(aboutAction);

    Observe(dataChangedAction);
    Observe(refreshTimer);
    Observe(reconnectCheckTimer);
    Observe(trackSelection);

    ::window=this;

    startTrackingAction->Enable();
    stopTrackingAction->Disable();
  }

  ~MyWindow()
  {
    if (tracking->Get()) {
      StopTracking();
    }
  }

  void PreInit()
  {
    Lum::Model::HeaderRef headerModel;
    Lum::Label            *label;
    Lum::Panel            *hPanel;
    Lum::Table            *table;
    Compass               *compass;

    Lum::Def::MultiView multiView(Lum::Def::Desc(L"GPSJinni"));

    hPanel=Lum::HPanel::Create(true,true);

    label=Lum::Label::Create(true,true);

    label->AddLabel(L"Conn./Online/Fix:",
                    Lum::HPanel::Create()
                    ->Add(new Lum::LED(connected))
                    ->AddSpace()
                    ->Add(new Lum::LED(online))
                    ->AddSpace()
                    ->Add(new Lum::LED(fix)));
    label->AddLabel(L"Time:",CreateTextValue(time,24)); // TODO: Measure by example
    label->AddLabel(L"Latitude:",CreateTextValue(latitude,10));
    label->AddLabel(L"Longitude:",CreateTextValue(longitude,10));
    label->AddLabel(L"Altitude:",CreateTextValue(altitude,10));
    label->AddLabel(L"Speed:",CreateTextValue(speed,10));
    label->AddLabel(L"Direction:",CreateTextValue(track,10));
    label->AddLabel(L"Climb:",CreateTextValue(climb,10));
    hPanel->Add(label);

    multiView.AddView(Lum::Def::MultiView::View(0,
                                                Lum::Def::Desc(L"General"),
                                                hPanel));

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Id",Lum::Base::Size::stdCharWidth,3);
    headerModel->AddColumn(L"Used",Lum::Base::Size::stdCharWidth,1);
    headerModel->AddColumn(L"Signal",Lum::Base::Size::stdCharWidth,25,true);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetShowHeader(true);
    table->SetHeaderModel(headerModel);
    table->SetModel(satellites);
    table->GetTableView()->SetAutoFitColumns(true);

    multiView.AddView(Lum::Def::MultiView::View(1,
                                                Lum::Def::Desc(L"Satelites"),
                                                table));

    compass=new Compass();
    compass->SetFlex(true,true);
    compass->SetModel(direction);

    multiView.AddView(Lum::Def::MultiView::View(2,
                                                Lum::Def::Desc(L"Compass"),
                                                compass));

    multiView.AddView(Lum::Def::MultiView::View(3,
                                                Lum::Def::Desc(L"Diagrams"),
                                                Lum::VPanel::Create(true,true)
                                                ->Add(Lum::Text::Create(L"Speed (m/s)"))
                                                ->AddSpace()
                                                ->Add(Lum::View::Create(TimeLine::Create(speedStream,true,true),true,true))
                                                ->AddSpace()
                                                ->Add(Lum::Text::Create(L"Altitude (m)"))
                                                ->AddSpace()
                                                ->Add(Lum::View::Create(TimeLine::Create(altitudeStream,true,true),true,true))));

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Track",Lum::Base::Size::stdCharWidth,10,true);
    headerModel->AddColumn(L"Time",Lum::Base::Size::stdCharWidth,19); // TODO

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetShowHeader(true);
    table->SetHeaderModel(headerModel);
    table->SetModel(tracks);
    table->SetSelection(trackSelection);
    table->GetTableView()->SetAutoFitColumns(true);

    multiView.AddView(Lum::Def::MultiView::View(4,
                                                Lum::Def::Desc(L"Tracking"),
                                                Lum::VPanel::Create(true,true)
                                                ->Add(Lum::HPanel::Create(true,false)
                                                      ->Add(Lum::Button::Create(L"Record",
                                                                                Lum::OS::Theme::imageMediaRecord,
                                                                                startTrackingAction,
                                                                                Lum::Button::typeIcon))
                                                      ->Add(Lum::Button::Create(L"Stop",
                                                                                Lum::OS::Theme::imageMediaStop,
                                                                                stopTrackingAction,
                                                                                Lum::Button::typeIcon))
                                                      ->AddSpace(true)
                                                       ->Add(new Lum::LED(tracking))
                                                       ->AddSpace(true)
                                                      ->Add(Lum::Button::Create(L"_Delete",
                                                                                Lum::OS::Theme::imageRemove,
                                                                                deleteTrackAction,
                                                                                Lum::Button::typeIcon))
                                                      ->Add(Lum::Button::Create(L"_Export",
                                                                                Lum::OS::Theme::imageSave,
                                                                                exportTrackAction,
                                                                                Lum::Button::typeIcon)))
                                                ->AddSpace()
                                                ->Add(table)));

    Lum::Def::Menu *preMenu=Lum::Def::Menu::Create();

    preMenu
      ->GroupProject()
        ->ActionQuit(GetClosedAction())
      ->End();

    Lum::Def::Menu *postMenu=Lum::Def::Menu::Create();

    postMenu
      ->GroupHelp()
        //->ActionHelp()
        ->ActionAbout(aboutAction)
      ->End();

    multiView.SetMenu(preMenu,postMenu);

    Lum::OS::display->GetBehaviour()->ApplyMultiViewDlg(this,multiView);

    Dialog::PreInit();
  }

  void GetGPSDataCopy(GPSData& data)
  {
    Lum::OS::ReadGuard<Lum::OS::RWMutex> guard(dataMutex);

    data=this->data;
  }

  void StartTracking()
  {
    Lum::Model::StringRef trackName=new Lum::Model::String(L"");
    Lum::Base::Path       fileName;

    if (!Lum::Dlg::TextInput::GetText(GetWindow(),
                                      L"Enter track name!",
                                      L"Please enter a name for the to be recorded track.",
                                      trackName) || trackName->Empty()) {
      return;
    }

    fileName.SetNativeDir(Lum::Base::Path::GetHome());
    fileName.AppendDir(L".gpsjinni");

    Lum::Base::Path::CreateDir(fileName.GetPath());

    fileName.SetBaseName(trackName->Get()+L".gjt");

    if (Lum::Base::Path::Exists(fileName.GetPath())) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),
                            L"Error while opening file!",
                            std::wstring(L"An error occured while opening file:\n")+
                            L"'"+fileName.GetPath()+L"'\n"+
                            L"Error:\n"+
                            L"'File already exists'");
      return;
    }

    trackFile.open(Lum::Base::WStringToString(fileName.GetPath()).c_str(),
                   std::ios::out|std::ios::trunc|std::ios::binary);

    if (!trackFile) {
      Lum::Base::Status status;

      status.SetToCurrentErrno();
      Lum::Dlg::Msg::ShowOk(GetWindow(),
                            L"Error while opening file!",
                            std::wstring(L"An error occured while opening file:\n")+
                            L"'"+fileName.GetPath()+L"'\n"+
                            L"Error:\n"+
                            L"'"+status.GetDescription()+L"'");

      return;
    }

    unsigned long tfid=htonl(trackingFileId);
    unsigned long tfv=htonl(trackingFileVersion);
    unsigned long tft=htonl(Lum::Base::SystemTime().GetTime());

    trackFile.write((const char*)&tfid,sizeof(tfid));
    trackFile.write((const char*)&tfv,sizeof(tfv));
    trackFile.write((const char*)&tft,sizeof(tft));

    startTrackingAction->Disable();
    stopTrackingAction->Enable();
    tracking->Set(true);
  }

  void AddTrackingRecord(const GPSData& data)
  {
    if (!tracking->Get()) {
      return;
    }

    double                lon=0.0;
    double                lat=0.0;
    double                alt=0.0;
    Lum::Base::SystemTime tim;

    if (data.HasLongitude()) {
      lon=data.GetLongitude();
    }

    if (data.HasLatitude()) {
      lat=data.GetLatitude();
    }

    if (data.HasAltitude()) {
      alt=data.GetAltitude();
    }

    if (data.HasTime()) {
      tim=data.GetTime();
    }

    unsigned long flon=htonl(round((lon+90.0)*10000000.0));
    unsigned long flat=htonl(round((lat+180.0)*10000000.0));
    unsigned long falt=htonl(round(alt*100));
    unsigned long ftim=htonl(tim.GetTime());

    //std::cout << "TrackRecord: " << std::hex << flon << " " << flat << " " << falt << " " << ftim << std::endl;

    trackFile.write((const char*)&flon,sizeof(flon));
    trackFile.write((const char*)&flat,sizeof(flat));
    trackFile.write((const char*)&falt,sizeof(falt));
    trackFile.write((const char*)&ftim,sizeof(ftim));
  }

  void StopTracking()
  {
    trackFile.close();

    startTrackingAction->Enable();
    stopTrackingAction->Disable();
    tracking->Set(false);

    RefreshTracks();
  }

  void RefreshTracks()
  {
    Lum::Base::Path              dirName,fileName;
    Lum::Base::DirScannerRef     scanner;
    Lum::Base::DirScanner::Entry entry;
    Lum::Base::Status            status;

    tracks->Clear();
    tracks->Off();

    dirName.SetNativeDir(Lum::Base::Path::GetHome());
    dirName.AppendDir(L".gpsjinni");

    scanner=Lum::Base::DirScanner::Create(dirName);

    while (scanner->GetNext(entry,status)) {
      if (entry.isDir) {
        continue;
      }

      std::cout << "Found possible track '" << Lum::Base::WStringToString(entry.name) << "'" << std::endl;

      if (entry.name.length()<4 ||
          entry.name.substr(entry.name.length()-4)!=L".gjt") {
        std::cerr << "wrong suffix!" << std::endl;
        continue;
      }

      std::ifstream file;

      fileName=dirName;
      fileName.SetBaseName(entry.name);

      file.open(Lum::Base::WStringToString(fileName.GetPath()).c_str(),
                std::ios::in|std::ios::binary);

      if (!file) {
        std::cerr << "Cannot open file!" << std::endl;
        continue;
      }

      unsigned long tfid,tfv,tft;

      file.read((char*)&tfid,sizeof(tfid));
      file.read((char*)&tfv,sizeof(tfv));
      file.read((char*)&tft,sizeof(tft));

      if (!file) {
        std::cerr << "Cannot read file header!" << std::endl;
        continue;
      }

      tfid=ntohl(tfid);
      tfv=ntohl(tfv);
      tft=ntohl(tft);

      if (tfid!=trackingFileId) {
        std::cerr << "Wrong magic!" << std::endl;
        continue;
      }

      if (tfv!=trackingFileVersion) {
        std::cerr << "Wrong file version!" << std::endl;
        continue;
      }

      tracks->Append(Track(entry.name.substr(0,entry.name.length()-4),
                           Lum::Base::SystemTime(tft)));
    }

    tracks->Sort(1);

    tracks->On();
  }

  void ExportTrack(const std::wstring& name)
  {
    Lum::Base::Path                path;
    std::wstring                   file;
    Lum::Dlg::File::Options        *options=new Lum::Dlg::File::Options();
    Lum::Model::Dir::PatternFilter *filter;
    std::ifstream                  in;
    std::ofstream                  out;

    path.SetNativeDir(Lum::Base::Path::GetHome());
    path.AppendDir(L".gpsjinni");
    path.SetBaseName(name+L".gjt");

    in.open(Lum::Base::WStringToString(path.GetPath()).c_str(),
            std::ios::in|std::ios::binary);

    if (!in) {
      Lum::Base::Status status;

      status.SetToCurrentErrno();
      Lum::Dlg::Msg::ShowOk(GetWindow(),
                            L"Error while opening file!",
                            std::wstring(L"An error occured while opening file:\n")+
                            L"'"+file+L"'\n"+
                            L"Error:\n"+
                            L"'"+status.GetDescription()+L"'");

      return;
    }

    unsigned long tfid,tfv,tft;

    in.read((char*)&tfid,sizeof(tfid));
    in.read((char*)&tfv,sizeof(tfv));
    in.read((char*)&tft,sizeof(tft));

    if (!in) {
      std::cerr << "Cannot read file header!" << std::endl;
      return;
    }

    tfid=ntohl(tfid);
    tfv=ntohl(tfv);
    tft=ntohl(tft);

    if (tfid!=trackingFileId) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),
                            L"Error while opening file!",
                            std::wstring(L"An error occured while opening file:\n")+
                            L"'"+file+L"'\n"+
                            L"Error:\n"+
                            L"'File is not a stored track'");
      return;
    }

    if (tfv!=trackingFileVersion) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),
                            L"Error while opening file!",
                            std::wstring(L"An error occured while opening file:\n")+
                            L"'"+file+L"'\n"+
                            L"Error:\n"+
                            L"'File format version not supported'");
      return;
    }

    path.SetNativeDir(Lum::Base::Path::GetHome());
    path.SetBaseName(name+L".kml");

    file=path.GetPath();

    options->SetMode(Lum::Dlg::File::Options::modeFile);
    options->SetExistingOnly(false);

    filter=new Lum::Model::Dir::PatternFilter(L"KML Files");
    filter->AddPattern(L"*.kml");
    options->AddFilter(filter);

    if (!Lum::Dlg::File::SaveFile(GetWindow(),
                                  L"Exporting track...",
                                  file,
                                  options)) {
      return;
    }

    std::cout << "Saving to '" << Lum::Base::WStringToString(file) << "'" << std::endl;

    out.open(Lum::Base::WStringToString(file).c_str(),
             std::ios::out|std::ios::trunc);

    if (!out) {
      Lum::Base::Status status;

      status.SetToCurrentErrno();
      Lum::Dlg::Msg::ShowOk(GetWindow(),
                            L"Error while opening file!",
                            std::wstring(L"An error occured while opening file:\n")+
                            L"'"+file+L"'\n"+
                            L"Error:\n"+
                            L"'"+status.GetDescription()+L"'");

      return;
    }

    out << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << std::endl;
    out << "<kml xmlns=\"http://www.opengis.net/kml/2.2\">" << std::endl;
    out << "  <Document>" << std::endl;
    out << "    <name>" << Lum::Base::WStringToUTF8(name) << ".kml" << "</name>" << std::endl;
    out << "    <open>1</open>" << std::endl;
    out << "    <Style id=\"lineStyle\">" << std::endl;
    out << "      <LineStyle>" << std::endl;
    out << "        <color>7f0000ff</color>" << std::endl;
    out << "        <width>4</width>" << std::endl;
    out << "      </LineStyle>" << std::endl;
    out << "    </Style>" << std::endl;

    out << "    <Placemark>" << std::endl;
    out << "      <name>" << Lum::Base::WStringToUTF8(name) << "</name>" << std::endl;
    out << "      <styleUrl>#lineStyle</styleUrl>" << std::endl;
    out << "      <LineString>" << std::endl;
    out << "        <extrude>0</extrude>" << std::endl;
//    out << "        <tessellate>1</tessellate>" << std::endl;
    out << "        <altitudeMode>absolute</altitudeMode>" << std::endl;
    out << "        <coordinates>" << std::endl;

    while (in) {
      unsigned long flon;
      unsigned long flat;
      unsigned long falt;
      unsigned long ftim;

      in.read((char*)&flon,sizeof(flon));
      in.read((char*)&flat,sizeof(flat));
      in.read((char*)&falt,sizeof(falt));
      in.read((char*)&ftim,sizeof(ftim));

      if (in) {
        double                lon;
        double                lat;
        double                alt;
        Lum::Base::SystemTime time;

        flon=ntohl(flon);
        flat=ntohl(flat);
        falt=ntohl(falt);
        ftim=ntohl(ftim);

        lon=flon/10000000.0-90.0;
        lat=flat/10000000.0-180.0;
        alt=falt/100;
        time.SetTime(ftim);

        out << "          " << lon << "," << lat << "," << alt << std::endl;
      }
    }

    out << "        </coordinates>" << std::endl;
    out << "      </LineString>" << std::endl;
    out << "    </Placemark>" << std::endl;
    out << "  </Document>" << std::endl;
    out << "</kml>" << std::endl;

    out.close();
    in.close();
  }

  void DeleteTrack(const std::wstring& name)
  {
    Lum::Base::Path   path;
    Lum::Base::Status status;

    path.SetNativeDir(Lum::Base::Path::GetHome());
    path.AppendDir(L".gpsjinni");
    path.SetBaseName(name+L".gjt");

    status=Lum::Base::Path::RemoveFile(path.GetPath());

    if (status) {
      RefreshTracks();
    }
    else {
      Lum::Dlg::Msg::ShowOk(GetWindow(),
                            L"Error while deleting file!",
                            std::wstring(L"An error occured while deletion of file:\n")+
                            L"'"+path.GetPath()+L"'\n"+
                            L"Error:\n"+
                            L"'"+status.GetDescription()+L"'");
    }
  }

  void ShowChange(const GPSData& data)
  {
    if (data.HasConnected()) {
      connected->Set(data.GetConnected());
    }
    else {
      connected->Set(false);
    }

    if (data.HasOnline()) {
      online->Set(data.GetOnline());
    }
    else {
      online->Set(false);
    }

    if (data.HasFix()) {
      fix->Set(data.GetFix());
    }
    else {
      fix->Set(false);
    }

    if (data.HasTime()) {
      time->Set(data.GetTime().GetLocalLocaleDateTime());
    }
    else {
      time->SetNull();
    }

    if (data.HasLatitude()) {
      latitude->Set(DoubleToString(data.GetLatitude())+L"\u00b0");
    }
    else {
      latitude->SetNull();
    }

    if (data.HasLongitude()) {
      longitude->Set(DoubleToString(data.GetLongitude())+L"\u00b0");
    }
    else {
      longitude->SetNull();
    }

    if (data.HasAltitude()) {
      altitude->Set(DoubleToString(data.GetAltitude())+L"m");
    }
    else {
      altitude->SetNull();
    }

    if (data.HasSpeed()) {
      if (data.GetSpeed()>=1/3.6) {
        speed->Set(DoubleToString(data.GetSpeed()*3.6)+L"km/h");
      }
      else {
        speed->Set(DoubleToString(data.GetSpeed())+L"m/s");
      }
    }
    else {
      speed->SetNull();
    }

    if (data.HasTrack()) {
      track->Set(DoubleToString(data.GetTrack())+L"\u00b0");
      direction->Set(data.GetTrack());
    }
    else {
      track->SetNull();
      direction->SetNull();
    }

    if (data.HasClimb()) {
      climb->Set(DoubleToString(data.GetClimb())+L"m");
    }
    else {
      climb->SetNull();
    }

    if (data.HasSatellites()) {
      // Delete everything in model that is not in current list
      size_t i=1;
      while (i<=satellites->GetRows()) {
        bool found=false;

        for (size_t j=0; j<data.GetSatellites().size(); j++) {
          if (satellites->GetEntry(i).id==data.GetSatellites()[j].id) {
            found=true;
            break;
          }
        }

        if (found) {
          i++;
          continue;
        }
        else {
          satellites->Delete(i);
        }
      }

      // Now add everything to model that is not already in
      for (size_t i=0; i<data.GetSatellites().size(); i++) {
        bool found=false;

        for (size_t j=1; j<=satellites->GetRows(); j++) {
          if (satellites->GetEntry(j).id==data.GetSatellites()[i].id) {
            found=true;

            if (satellites->GetEntry(j).snr!=data.GetSatellites()[i].snr ||
                satellites->GetEntry(j).used!=data.GetSatellites()[i].used) {
              satellites->GetEntry(j)=data.GetSatellites()[i];
              satellites->RedrawRow(j);
            }
            break;
          }
        }

        if (!found) {
          satellites->Append(data.GetSatellites()[i]);
        }
      }
    }
    else {
      satellites->Clear();
    }
  }

  void UpdateStreams(const GPSData& data)
  {
    if (!data.HasFix() || !data.GetFix() ||
        !data.HasSpeed()) {
      speedStream->SetNull();
    }
    else {
      speedStream->Set(0,data.GetSpeed());
    }

    if (!data.HasFix() || !data.GetFix() ||
        !data.HasAltitude()) {
      altitudeStream->SetNull();
    }
    else {
      altitudeStream->Set(0,data.GetAltitude());
    }
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      InitializeGPS();
      Lum::OS::display->AddTimer(refreshTime,0,refreshTimer);
      Lum::OS::display->AddTimer(reconnectTime,0,reconnectCheckTimer);
      RefreshTracks();
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      DeinitializeGPS();
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==dataChangedAction && dataChangedAction->IsFinished()) {
      GPSData dataCopy;

      GetGPSDataCopy(dataCopy);
      AddTrackingRecord(dataCopy);
      ShowChange(dataCopy);
    }
    else if (model==refreshTimer && refreshTimer->IsFinished()) {
      GPSData dataCopy;

      GetGPSDataCopy(dataCopy);
      ShowChange(dataCopy);
      AddTrackingRecord(dataCopy);
      UpdateStreams(dataCopy);

      Lum::OS::display->AddTimer(refreshTime,0,refreshTimer);
    }
    else if (model==reconnectCheckTimer && reconnectCheckTimer->IsFinished()) {
      ReconnectIfDisconnected();

      Lum::OS::display->AddTimer(reconnectTime,0,reconnectCheckTimer);
    }
    else if (model==startTrackingAction &&
             startTrackingAction->IsFinished() &&
             startTrackingAction->IsEnabled()) {
      StartTracking();
    }
    else if (model==stopTrackingAction &&
             stopTrackingAction->IsFinished() &&
             stopTrackingAction->IsEnabled()) {
      StopTracking();
    }
    else if (model==deleteTrackAction &&
             deleteTrackAction->IsFinished() &&
             deleteTrackAction->IsEnabled()) {
      DeleteTrack(tracks->GetEntry(trackSelection->GetLine()).GetName());
    }
    else if (model==exportTrackAction &&
             exportTrackAction->IsFinished() &&
             exportTrackAction->IsEnabled()) {
      ExportTrack(tracks->GetEntry(trackSelection->GetLine()).GetName());
    }
    else if (model==trackSelection) {
      if (trackSelection->HasSelection()) {
        deleteTrackAction->Enable();
        exportTrackAction->Enable();
      }
      else {
        deleteTrackAction->Disable();
        exportTrackAction->Disable();
      }
    }

    Dialog::Resync(model,msg);
  }

  void HandleChange()
  {
    Lum::OS::display->QueueActionForAsyncNotification(dataChangedAction);
  }
};
#endif

#ifdef SIMPLE_CODE
static void gps_location_changed (LocationGPSDevice *device, gpointer userdata);
static void gps_location_connected (LocationGPSDevice *new_device, gpointer userdata);
static void gps_location_disconnected (LocationGPSDevice *new_device, gpointer userdata);
static void gps_location_started (LocationGPSDControl *control, gpointer userdata);
static void gps_location_stopped (LocationGPSDControl *control, gpointer userdata);
static void gps_location_error (LocationGPSDControl *control, gpointer userdata);

static void gps_location_changed(LocationGPSDevice *device, gpointer userdata)
{
  std::cout << "gps_location_changed" << std::endl;

    std::cout << "-----------------" << std::endl;
    std::cout << "Status: " << (device->online? "true" : "false") << std::endl;
    std::cout << "Detailed status: ";
    switch (device->status) {
    case LOCATION_GPS_DEVICE_STATUS_NO_FIX:
      std::cout << "No fix";
      break;
      case LOCATION_GPS_DEVICE_STATUS_FIX:
      std::cout << "Fix";
      break;
    case LOCATION_GPS_DEVICE_STATUS_DGPS_FIX:
      std::cout << "DGPS fix";
      break;
    }

    std::cout << std::endl;

    std::cout << device->satellites_in_view << " satellites in view, " << device->satellites_in_use << " used!" << std::endl;

    /* Fix information */
    LocationGPSDeviceFix *fix;
    fix = device->fix;

    std::cout << "Fix mode: ";
    switch(fix->mode) {
      case LOCATION_GPS_DEVICE_MODE_NOT_SEEN:
      std::cout << "No satelite";
      break;
    case LOCATION_GPS_DEVICE_MODE_NO_FIX:
      std::cout << "No fix";
      break;
    case LOCATION_GPS_DEVICE_MODE_2D:
      std::cout << "2D information";
      break;
    case LOCATION_GPS_DEVICE_MODE_3D:
      std::cout << "3D information";
      break;
    }
    std::cout << std::endl;

    if (fix->fields & LOCATION_GPS_DEVICE_TIME_SET) {
      std::cout << "Time: " << fix->time << " uncertainty: " << fix->ept << std::endl;
    }
    if (fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET) {
      std::cout << "Latitude: " << fix->latitude << " longitude: " << fix->longitude << " uncertainty: " <<  fix->eph << std::endl;
    }
    if (fix->fields & LOCATION_GPS_DEVICE_ALTITUDE_SET) {
      std::cout << "Altitude: " << fix->altitude << " uncertainty: " << fix->epv << std::endl;
    }
    if (fix->fields & LOCATION_GPS_DEVICE_SPEED_SET) {
      std::cout << "Speed: " << fix->speed << " uncertainty: " << fix->eps << std::endl;
    }
    if (fix->fields & LOCATION_GPS_DEVICE_TRACK_SET) {
      std::cout << "Track: " << fix->track << " uncertainty: " << fix->epd << std::endl;
    }
    if (fix->fields & LOCATION_GPS_DEVICE_CLIMB_SET) {
      std::cout << "Climb: " << fix->climb << " uncertainty: " << fix->epc << std::endl;
    }

    fix = NULL;

    if (device->satellites!=NULL) {
      for (int i =0; i<device->satellites_in_view; i++) {
        LocationGPSDeviceSatellite* satellite=(LocationGPSDeviceSatellite*)g_ptr_array_index(device->satellites,i);

        std::cout << "Satellite id:" << satellite->prn << " S/N: " << satellite->signal_strength << " elevation: " << satellite->elevation << " azimuth: " << satellite->azimuth << " " << (satellite->in_use ? "used" : "unused") << std::endl;
      }
    }
}

static void gps_location_connected(LocationGPSDevice *new_device, gpointer userdata) {
  std::cout << "Connected..." << std::endl;
}

static void gps_location_disconnected(LocationGPSDevice *new_device, gpointer userdata){
  std::cout << "disconnected..." << std::endl;
}

static void gps_location_started(LocationGPSDControl *control, gpointer userdata) {
  std::cout << "GPSD started!" << std::endl;
}

static void gps_location_stopped(LocationGPSDControl *control, gpointer userdata) {
  std::cout << "GPSD stoped!" << std::endl;
}

static void gps_location_error(LocationGPSDControl *control, gpointer userdata) {
  std::cout << "GPSD error!" << std::endl;
}

static void gps_location_quit(int signal) {
  printf("[Info] SIGINT caught, exiting\n");
  g_main_loop_quit(loop);
}
#endif

#if defined(HAVE_LIB_GPS)
static void cb(struct gps_data_t* data, char * buf, size_t len, int level)
{
  assert(window!=NULL);

  Lum::OS::WriteGuard<Lum::OS::RWMutex> guard(window->dataMutex);

  GPSData                               &gpsData=window->data;

  gpsData.SetConnected(true);

  if (data->set & ONLINE_SET) {
    gpsData.SetOnline(data->online);
  }

  if (data->set & STATUS_SET) {
    gpsData.SetFix(data->status!=STATUS_NO_FIX);
  }

  if ((data->set & LATLON_SET)  && !isnan(data->fix.latitude)) {
    gpsData.SetLatitude(data->fix.latitude);
  }

  if ((data->set & LATLON_SET)  && !isnan(data->fix.longitude)) {
    gpsData.SetLongitude(data->fix.longitude);
  }

  if ((data->set & ALTITUDE_SET) && !isnan(data->fix.altitude)) {
    gpsData.SetAltitude(data->fix.altitude);
  }

  if ((data->set & SPEED_SET) && !isnan(data->fix.speed)) {
    gpsData.SetSpeed(data->fix.speed);
  }

  if ((data->set & TRACK_SET) && !isnan(data->fix.track)) {
    gpsData.SetTrack(data->fix.track);
  }

  if ((data->set & CLIMB_SET) && !isnan(data->fix.climb)) {
    gpsData.SetClimb(data->fix.climb);
  }

  if (data->set & TIME_SET) {
    Lum::Base::SystemTime time(lround(data->fix.time));
    gpsData.SetTime(time);
  }

  if (data->set & SATELLITE_SET) {
    std::vector<GPSData::Satellite> satellites;

    for (int i=0; i<data->satellites; i++) {
      GPSData::Satellite s;

      s.id=data->PRN[i];
      s.used=data->used[i];
      s.snr=data->ss[i];

      satellites.push_back(s);
    }

    gpsData.SetSatellites(satellites);
  }

  window->HandleChange();
}
#endif

#ifndef SIMPLE_CODE
class Main : public Lum::OS::MainDialog<MyWindow>
{
public:
  Main()
  {
    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(L"Where the #*~@# am I?");
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2009, Tim Teulings");
    info.SetLicense(L"GNU Public License");
  }
};

LUM_MAIN(Main,Lum::Base::StringToWString(PACKAGE_NAME))

#else

int main(int argc, char *argv[]) {

	/* Inicialitzacio de la llibreria glib */
	g_type_init();
	g_thread_init(NULL);
	printf("[Info] Glib started\n");
	
	
	/* Location Control setup*/
	LocationGPSDControl *control = NULL;
	control = location_gpsd_control_get_default();
	if(control->can_control)
		location_gpsd_control_start (control);
	
	/* Recupera informacio actual */
	LocationGPSDevice *device = NULL;
	device = (LocationGPSDevice*)g_object_new(LOCATION_TYPE_GPS_DEVICE, NULL);
	location_gps_device_reset_last_known (device);
	printf("[Debug] Device pointer adress: %x\n", (unsigned int)device);
	printf("[Info] Device object obtained\n");
	
	/* Install signal handlers */
	signal(SIGINT, gps_location_quit);
	guint idd_changed = g_signal_connect (device, "changed", G_CALLBACK (gps_location_changed), NULL);
	guint idd_con = g_signal_connect (device, "connected", G_CALLBACK (gps_location_connected), NULL);
	guint idd_disc = g_signal_connect (device, "disconnected", G_CALLBACK (gps_location_disconnected), NULL);
	
	guint idc_run = g_signal_connect (control, "gpsd_running", G_CALLBACK (gps_location_started), NULL);
	guint idc_stop = g_signal_connect (control, "gpsd_stopped", G_CALLBACK (gps_location_stopped), NULL);
	guint idc_error = g_signal_connect (control, "error", G_CALLBACK (gps_location_error), NULL);
	
	printf("[Info] Entering the main loop event\n");
        context=g_main_context_new();
  	loop = g_main_loop_new(context, FALSE);
	/* g_timeout_add(1000, (GSourceFunc)callback, NULL); */
 	g_main_loop_run(loop);
 	
 	/* Exiting */
 	g_main_loop_unref(loop);
 	g_main_context_unref(context);
	
	/* Disconnect signal */
	if(control->can_control)
		location_gpsd_control_stop(control);
		
	g_signal_handler_disconnect(device, idd_changed);
	g_signal_handler_disconnect(device, idd_con);
	g_signal_handler_disconnect(device, idd_disc);
	
	g_signal_handler_disconnect(control, idc_run);
	g_signal_handler_disconnect(control, idc_stop);
	g_signal_handler_disconnect(control, idc_error);

	/*	Device control cleanup */
	g_object_unref(control);	
	g_object_unref(device);

	return 0;
}
#endif
