/*
 * This file is part of gps-data-logger-widget
 *
 * Copyright (C) 2009-2011 Marcell Lengyel <marcell@maemo.org>
 *
 */


#include <hildon/hildon.h>
#include <hildon-fm-2/hildon/hildon-file-chooser-dialog.h>
#include <locale.h>
#include <glib/gi18n-lib.h> /* Declares _(). */

#include <glib/gprintf.h> /* g_sprintf */

#include "settings-dialog.h"

/* -------------------------------------------------------------------- */
/* Settings
 */

void
apply_settings (GpsDataLoggerWidget * self)
{
    LocationGPSDControlMethod my_method = 0;
    gint my_interval;
    
    if (self->settings->method_agnss)
            my_method = my_method | LOCATION_METHOD_AGNSS;
    if (self->settings->method_gnss)
        my_method = my_method | LOCATION_METHOD_GNSS;
    if (self->settings->method_acwp)
        my_method = my_method | LOCATION_METHOD_ACWP;
    if (self->settings->method_cwp)
        my_method = my_method | LOCATION_METHOD_CWP;
    if (self->settings->interval == 0)
        my_interval = LOCATION_INTERVAL_1S;
    else if (self->settings->interval == 1)
        my_interval = LOCATION_INTERVAL_2S;
    else if (self->settings->interval == 2)
        my_interval = LOCATION_INTERVAL_5S;
    else if (self->settings->interval == 3)
        my_interval = LOCATION_INTERVAL_10S;
    else if (self->settings->interval == 4)
        my_interval = LOCATION_INTERVAL_20S;
    else if (self->settings->interval == 5)
        my_interval = LOCATION_INTERVAL_30S;
    else if (self->settings->interval == 6)
        my_interval = LOCATION_INTERVAL_60S;
    else if (self->settings->interval == 7)
        my_interval = LOCATION_INTERVAL_120S;

    g_object_set(G_OBJECT (self->control),
                 "preferred-method", my_method,
                 "preferred-interval", my_interval,
                 NULL);
}

void
on_btn_path (GtkWidget * widget, gpointer user_data)
{   
    GtkWidget *chooser; 
    GtkDialog *dialog = GTK_DIALOG(user_data);

    chooser = hildon_file_chooser_dialog_new(GTK_WINDOW(dialog),
                    GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);
    /* Not in GTK 2.14 :(
    gtk_file_chooser_set_create_folders(GTK_FILE_CHOOSER (chooser),
                    TRUE); */
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER (chooser),
                    "/home/user/MyDocs");
    if (gtk_dialog_run (GTK_DIALOG (chooser)) == GTK_RESPONSE_OK) {
        hildon_button_set_value(HILDON_BUTTON(widget),
                 gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooser)));
        gtk_widget_destroy (chooser);
    } else {
        gtk_widget_destroy (chooser);
    }
}

static void
on_settings_dialog_response (GtkDialog *dialog,
  gint response_id, gpointer data)
{   
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);
    gboolean    new_value, save_needed = FALSE;
    gint        new_int;

    g_assert (self->settings_dialog);
    g_assert (self->settings_dialog == GTK_WIDGET (dialog));

    if (response_id == GTK_RESPONSE_OK) {
        /* save settings */
        new_value = hildon_check_button_get_active(
                    HILDON_CHECK_BUTTON(self->chk_method_agnss));
        if (self->settings->method_agnss != new_value) {
            save_needed = TRUE;
            self->settings->method_agnss = new_value;
        }
        new_value = hildon_check_button_get_active(
                    HILDON_CHECK_BUTTON(self->chk_method_gnss));
        if (self->settings->method_gnss != new_value) {
            save_needed = TRUE;
            self->settings->method_gnss = new_value;
        }
        new_value = hildon_check_button_get_active(
                    HILDON_CHECK_BUTTON(self->chk_method_acwp));
        if (self->settings->method_acwp != new_value) {
            save_needed = TRUE;
            self->settings->method_acwp = new_value;
        }
        new_value = hildon_check_button_get_active(
                    HILDON_CHECK_BUTTON(self->chk_method_cwp));
        if (self->settings->method_cwp != new_value) {
            save_needed = TRUE;
            self->settings->method_cwp = new_value;
        }
        new_int = hildon_picker_button_get_active(
                  HILDON_PICKER_BUTTON(self->pkr_interval));
        if (self->settings->interval != new_int) {
            save_needed = TRUE;
            self->settings->interval = new_int;
        }
        new_value = hildon_check_button_get_active(
                    HILDON_CHECK_BUTTON(self->chb_auto_pause));
        if (self->settings->auto_pause != new_value) {
            save_needed = TRUE;
            self->settings->auto_pause = new_value;
        }
        new_int = hildon_picker_button_get_active(
                  HILDON_PICKER_BUTTON(self->pkr_pause_speed_threshold));
        if (self->settings->pause_speed_threshold != new_int) {
            save_needed = TRUE;
            self->settings->pause_speed_threshold = new_int;
        }
        new_int = hildon_picker_button_get_active(
                  HILDON_PICKER_BUTTON(self->pkr_pause_time_threshold));
        if (self->settings->pause_time_threshold != new_int) {
            save_needed = TRUE;
            self->settings->pause_time_threshold = new_int;
        }

        if (g_ascii_strcasecmp(self->settings->log_folder,
                               hildon_button_get_value(
                               HILDON_BUTTON(self->btn_path))) != 0) {
            g_stpcpy(self->settings->log_folder,
                     hildon_button_get_value(
                     HILDON_BUTTON(self->btn_path)));
            save_needed = TRUE;
        }
        if (save_needed) {
            settings_save(self->settings);
            /* apply settings */
            apply_settings(self);
        }
    }
    gtk_widget_destroy (self->settings_dialog);
    self->settings_dialog = NULL;
}


GtkDialog *
settings_dialog_new (GpsDataLoggerWidget * self)
{
    GtkWidget *dialog;
    GtkWidget *vbox;
    GtkWidget *selector, *speed_selector, *time_selector;
    GtkWidget *panarea;
    gchar *text;
    int i;

    /* dialog title */
    dialog = gtk_dialog_new_with_buttons (_("GPS Data Logger Widget Settings"),
                                          NULL, /* parent */
                                          GTK_DIALOG_DESTROY_WITH_PARENT |
                                          GTK_DIALOG_NO_SEPARATOR,
                                          GTK_STOCK_SAVE,
                                          GTK_RESPONSE_OK, NULL);
    gtk_widget_set_size_request (dialog, 800, 350);
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox),
                        gtk_label_new (_("Location Methods")),
                        TRUE, TRUE, 4);
    /* AGNSS */
    self->chk_method_agnss = hildon_check_button_new (HILDON_SIZE_FINGER_HEIGHT);
    hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->chk_method_agnss),
                                self->settings->method_agnss);
    gtk_button_set_label (GTK_BUTTON (self->chk_method_agnss),
                          _("Assisted GPS"));
    gtk_button_set_alignment (GTK_BUTTON (self->chk_method_agnss),
                              0.0, 0.5);
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->chk_method_agnss, TRUE, TRUE, 4);
    /* GNSS */
    self->chk_method_gnss = hildon_check_button_new (HILDON_SIZE_FINGER_HEIGHT);
    hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->chk_method_gnss),
                                self->settings->method_gnss);
    gtk_button_set_label (GTK_BUTTON (self->chk_method_gnss),
                          _("GPS"));
    gtk_button_set_alignment (GTK_BUTTON (self->chk_method_gnss),
                              0.0, 0.5);
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->chk_method_gnss, TRUE, TRUE, 4);
    /* ACWP */
    self->chk_method_acwp = hildon_check_button_new (HILDON_SIZE_FINGER_HEIGHT);
    hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->chk_method_acwp),
                                self->settings->method_acwp);
    gtk_button_set_label (GTK_BUTTON (self->chk_method_acwp),
                          _("Assisted, using cellular base stations"));
    gtk_button_set_alignment (GTK_BUTTON (self->chk_method_acwp),
                              0.0, 0.5);
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->chk_method_acwp, TRUE, TRUE, 4);

    /* CWP */
    self->chk_method_cwp = hildon_check_button_new (HILDON_SIZE_FINGER_HEIGHT);
    hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->chk_method_cwp),
                                self->settings->method_cwp);
    gtk_button_set_label (GTK_BUTTON (self->chk_method_cwp),
                          _("Center of the current country"));
    gtk_button_set_alignment (GTK_BUTTON (self->chk_method_cwp),
                              0.0, 0.5);
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->chk_method_cwp, TRUE, TRUE, 4);
    /* Interval */
    gtk_box_pack_start (GTK_BOX (vbox),
                        gtk_label_new (_("Interval")),
                        TRUE, TRUE, 4);
    /* Interval picker */
    self->pkr_interval = hildon_picker_button_new
            (HILDON_SIZE_FINGER_HEIGHT,
             HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    gtk_button_set_alignment (GTK_BUTTON (self->pkr_interval), 0.0, 0.5);
    hildon_button_set_title (HILDON_BUTTON (self->pkr_interval),
                             _("Logging interval"));
    selector = hildon_touch_selector_new_text ();
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      _(" 1 second"));
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      _(" 2 seconds"));
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      _(" 5 seconds"));
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      _("10 seconds"));
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      _("20 seconds"));
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      _("30 seconds"));
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      _("60 seconds"));
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      _("120 seconds"));
    hildon_picker_button_set_selector(HILDON_PICKER_BUTTON
             (self->pkr_interval), HILDON_TOUCH_SELECTOR (selector));
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->pkr_interval, TRUE, TRUE, 5); 
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (selector),
                                      0, self->settings->interval);

    /* Auto pause */
    self->chb_auto_pause = hildon_check_button_new (HILDON_SIZE_FINGER_HEIGHT);
    gtk_button_set_label (GTK_BUTTON (self->chb_auto_pause), _("Auto pause enabled"));
    hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->chb_auto_pause),
                                self->settings->auto_pause);
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->chb_auto_pause, TRUE, TRUE, 5);
    /* speed threshold */
    self->pkr_pause_speed_threshold = hildon_picker_button_new
                       (HILDON_SIZE_FINGER_HEIGHT,
                        HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    gtk_button_set_alignment (GTK_BUTTON (self->pkr_pause_speed_threshold),
                                0.0, 0.5);
    hildon_button_set_title (HILDON_BUTTON (self->pkr_pause_speed_threshold),
                               _("Pause if speed is less than"));
    speed_selector = hildon_touch_selector_new_text ();
    text = g_malloc0(10);
    for (i = 1; i <= 30; i++ ) {
        g_sprintf(text, "%d km/h", i);
        hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (speed_selector),
                        text);
    }
    g_free(text);
    hildon_picker_button_set_selector(HILDON_PICKER_BUTTON
             (self->pkr_pause_speed_threshold),
             HILDON_TOUCH_SELECTOR (speed_selector));
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->pkr_pause_speed_threshold, TRUE, TRUE, 5);
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (speed_selector),
                                      0, self->settings->pause_speed_threshold);
    /* interval threshold */
    self->pkr_pause_time_threshold = hildon_picker_button_new
                       (HILDON_SIZE_FINGER_HEIGHT,
                        HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    gtk_button_set_alignment (GTK_BUTTON (self->pkr_pause_time_threshold),
                                0.0, 0.5);
    hildon_button_set_title (HILDON_BUTTON (self->pkr_pause_time_threshold),
                               _("For at least"));
    time_selector = hildon_touch_selector_new_text ();
    text = g_malloc0(25);
    for (i = 1; i <= 25; i++ ) {
        g_sprintf(text, _("%d seconds"), i * AUTO_PAUSE_TIME_MULTIPLIER);
        hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (time_selector),
                        text);
    }
    g_free(text);
    hildon_picker_button_set_selector(HILDON_PICKER_BUTTON
             (self->pkr_pause_time_threshold),
             HILDON_TOUCH_SELECTOR (time_selector));
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->pkr_pause_time_threshold, TRUE, TRUE, 5);
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (time_selector),
                                      0, self->settings->pause_time_threshold);


    /* Log */
    gtk_box_pack_start (GTK_BOX (vbox),
                        gtk_label_new (_("Log")),
                        TRUE, TRUE, 4);
    /* Log path */
    self->btn_path = hildon_button_new (HILDON_SIZE_FINGER_HEIGHT,
                                  HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    gtk_button_set_alignment (GTK_BUTTON (self->btn_path), 0.0, 0.5);
    hildon_button_set_title(HILDON_BUTTON(self->btn_path),
                            _("Folder to store the logs"));
    hildon_button_set_value(HILDON_BUTTON(self->btn_path),
                            self->settings->log_folder);
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->btn_path, TRUE, TRUE, 5);
    g_signal_connect(self->btn_path, "clicked",
                     G_CALLBACK (on_btn_path), dialog);
    /* */
    panarea = hildon_pannable_area_new ();
    g_object_set (panarea, "initial-hint", TRUE, NULL);
    hildon_pannable_area_add_with_viewport
            (HILDON_PANNABLE_AREA (panarea), GTK_WIDGET (vbox));
    gtk_box_pack_start (GTK_BOX
                        (GTK_DIALOG (dialog)->vbox), panarea,
                        TRUE, TRUE, 5);
    gtk_widget_show_all (dialog);
    hildon_pannable_area_scroll_to (HILDON_PANNABLE_AREA (panarea), -1,
                                    0);
    return GTK_DIALOG (dialog);
}

void
on_settings_dialog_show (GtkWidget *widget,
                         gpointer data)
{
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);
    if (!self->settings_dialog) {
        self->settings_dialog = GTK_WIDGET (settings_dialog_new (self));
        g_signal_connect (self->settings_dialog, "response",
                          G_CALLBACK (on_settings_dialog_response),
                          self);
    } else {
        gtk_widget_show (self->settings_dialog);
    }
}

