
/*
 * This file is part of gps-data-logger-widget
 *
 * Copyright (C) 2009 Marcell Lengyel <marcell@maemo.org>
 *
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>	/*  getenv */
#include <time.h>
#include <locale.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <glib/gprintf.h>

#include <glib/gi18n-lib.h> /* Declares _(). */

#include <hildon/hildon.h>
#include <libhildondesktop/libhildondesktop.h>
#include <hildon-fm-2/hildon/hildon-file-chooser-dialog.h>
#include <hildon/hildon-banner.h>

#include "gps-data-logger-widget.h"

#define PADDING 10
#define THEME_DIR "/etc/hildon/theme/images/"
#define BUTTON_BG THEME_DIR "ApplicationShortcutApplet.png"
#define BUTTON_PRESSED_BG THEME_DIR "ApplicationShortcutAppletPressed.png"
#define REC_ICON "camera_video_recording"
#define STOP_ICON "camera_video_stop"
#define PAUSE_ICON "camera_video_pause"
#define GPS_REFRESH_ICON "general_refresh"
#define GPS_ON_ICON "gps_location"
#define GPS_OFF_ICON "gps_not_connected"

#define OSSO_GDL_NAME    "gps_data_logger"
#define OSSO_GDL_SERVICE "org.maemo."OSSO_GDL_NAME
#define OSSO_GDL_OBJECT  "/org/maemo/"OSSO_GDL_NAME
#define OSSO_GDL_IFACE   "org.maemo."OSSO_GDL_NAME


HD_DEFINE_PLUGIN_MODULE (GpsDataLoggerWidget, gps_data_logger_widget, HD_TYPE_HOME_PLUGIN_ITEM)

static void
update_status_and_buttons (GpsDataLoggerWidget * self);

/* -------------------------------------------------------------------- */
/* Settings
 */

static void
apply_settings (GpsDataLoggerWidget * self)
{
    LocationGPSDControlMethod my_method = 0;
    gint my_interval;

    if (self->settings->method_agnss)
        my_method = my_method | LOCATION_METHOD_AGNSS;
    if (self->settings->method_gnss)
        my_method = my_method | LOCATION_METHOD_GNSS;
    if (self->settings->method_acwp)
        my_method = my_method | LOCATION_METHOD_ACWP;
    if (self->settings->method_cwp)
        my_method = my_method | LOCATION_METHOD_CWP;
    if (self->settings->interval == 0)
        my_interval = LOCATION_INTERVAL_1S;
    else if (self->settings->interval == 1)
        my_interval = LOCATION_INTERVAL_2S;
    else if (self->settings->interval == 2)
        my_interval = LOCATION_INTERVAL_5S;
    else if (self->settings->interval == 3)
        my_interval = LOCATION_INTERVAL_10S;
    else if (self->settings->interval == 4)
        my_interval = LOCATION_INTERVAL_20S;
    else if (self->settings->interval == 5)
        my_interval = LOCATION_INTERVAL_30S;
    else if (self->settings->interval == 6)
        my_interval = LOCATION_INTERVAL_60S;
    else if (self->settings->interval == 7)
        my_interval = LOCATION_INTERVAL_120S;

    g_object_set(G_OBJECT (self->control),
                 "preferred-method", my_method,
                 "preferred-interval", my_interval,
                 NULL);
}

void
on_btn_path (GtkWidget * widget, gpointer user_data)
{
    GtkWidget *chooser;
    GtkDialog *dialog = GTK_DIALOG(user_data);

    chooser = hildon_file_chooser_dialog_new(GTK_WINDOW(dialog),
		    GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);
    /* Not in GTK 2.14 :(
    gtk_file_chooser_set_create_folders(GTK_FILE_CHOOSER (chooser),
		    TRUE); */
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER (chooser),
		    "/home/user/MyDocs");
    if (gtk_dialog_run (GTK_DIALOG (chooser)) == GTK_RESPONSE_OK) {
        hildon_button_set_value(HILDON_BUTTON(widget),
		 gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooser)));
	gtk_widget_destroy (chooser);
    } else {
	gtk_widget_destroy (chooser);
    }
}

static void
on_settings_dialog_response (GtkDialog *dialog,
  gint response_id, gpointer data)
{
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);
    gboolean	new_value, save_needed = FALSE;
    gint 	new_int;
    
    g_assert (self->settings_dialog);
    g_assert (self->settings_dialog == GTK_WIDGET (dialog));

    if (response_id == GTK_RESPONSE_OK) {
        /* save settings */
	new_value = hildon_check_button_get_active(
	            HILDON_CHECK_BUTTON(self->chk_method_agnss));
        if (self->settings->method_agnss != new_value) {
	    save_needed = TRUE;
	    self->settings->method_agnss = new_value;
	}
	new_value = hildon_check_button_get_active(
		    HILDON_CHECK_BUTTON(self->chk_method_gnss));
        if (self->settings->method_gnss != new_value) {
	    save_needed = TRUE;
	    self->settings->method_gnss = new_value;
	}
        new_value = hildon_check_button_get_active(
		    HILDON_CHECK_BUTTON(self->chk_method_acwp));
        if (self->settings->method_acwp != new_value) {
	    save_needed = TRUE;
	    self->settings->method_acwp = new_value;
	}
        new_value = hildon_check_button_get_active(
		    HILDON_CHECK_BUTTON(self->chk_method_cwp));
        if (self->settings->method_cwp != new_value) {
	    save_needed = TRUE;
	    self->settings->method_cwp = new_value;
	}
        new_int = hildon_picker_button_get_active(
		  HILDON_PICKER_BUTTON(self->pkr_interval));
	if (self->settings->interval != new_int) {
            save_needed = TRUE;
	    self->settings->interval = new_int;
	}
	if (g_ascii_strcasecmp(self->settings->log_folder,
                               hildon_button_get_value(
			       HILDON_BUTTON(self->btn_path))) != 0) {
	    g_stpcpy(self->settings->log_folder,
		     hildon_button_get_value(
		     HILDON_BUTTON(self->btn_path)));
	    save_needed = TRUE;
	}
	if (save_needed) {
	    settings_save(self->settings);
	    /* apply settings */
	    apply_settings(self);
	}
    } 
    gtk_widget_destroy (self->settings_dialog);
    self->settings_dialog = NULL;
}
	      
	
GtkDialog *
settings_dialog_new (GpsDataLoggerWidget * self)
{
    GtkWidget *dialog;
    GtkWidget *vbox;
    GtkWidget *selector;
    GtkWidget *panarea;

    dialog = gtk_dialog_new_with_buttons (_("GPS Data Logger Widget Settings"),
					  NULL, /* parent */
					  GTK_DIALOG_DESTROY_WITH_PARENT |
					  GTK_DIALOG_NO_SEPARATOR,
					  GTK_STOCK_SAVE, 
					  GTK_RESPONSE_OK, NULL);
    gtk_widget_set_size_request (dialog, 800, 350);
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox),
                        gtk_label_new (_("Location Methods")),
                        TRUE, TRUE, 4);
    /* AGNSS */
    self->chk_method_agnss = hildon_check_button_new (HILDON_SIZE_FINGER_HEIGHT);
    hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->chk_method_agnss),
		    		self->settings->method_agnss);
    gtk_button_set_label (GTK_BUTTON (self->chk_method_agnss),
                          _("Assisted GPS"));
    gtk_button_set_alignment (GTK_BUTTON (self->chk_method_agnss),
                              0.0, 0.5);
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->chk_method_agnss, TRUE, TRUE, 4);
    /* GNSS */
    self->chk_method_gnss = hildon_check_button_new (HILDON_SIZE_FINGER_HEIGHT);
    hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->chk_method_gnss),
		    		self->settings->method_gnss);
    gtk_button_set_label (GTK_BUTTON (self->chk_method_gnss),
                          _("GPS"));
    gtk_button_set_alignment (GTK_BUTTON (self->chk_method_gnss),
                              0.0, 0.5);
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->chk_method_gnss, TRUE, TRUE, 4);
    /* ACWP */
    self->chk_method_acwp = hildon_check_button_new (HILDON_SIZE_FINGER_HEIGHT);
    hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->chk_method_acwp),
		    		self->settings->method_acwp);
    gtk_button_set_label (GTK_BUTTON (self->chk_method_acwp),
                          _("Assisted, using cellular base stations"));
    gtk_button_set_alignment (GTK_BUTTON (self->chk_method_acwp),
                              0.0, 0.5);
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->chk_method_acwp, TRUE, TRUE, 4);

    /* CWP */
    self->chk_method_cwp = hildon_check_button_new (HILDON_SIZE_FINGER_HEIGHT);
    hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->chk_method_cwp),
		    		self->settings->method_cwp);
    gtk_button_set_label (GTK_BUTTON (self->chk_method_cwp),
                          _("Center of the current country"));
    gtk_button_set_alignment (GTK_BUTTON (self->chk_method_cwp),
                              0.0, 0.5);
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->chk_method_cwp, TRUE, TRUE, 4);
    /* Interval */
    gtk_box_pack_start (GTK_BOX (vbox),
                        gtk_label_new (_("Interval")),
                        TRUE, TRUE, 4);
    /* Interval picker */
    self->pkr_interval = hildon_picker_button_new
            (HILDON_SIZE_FINGER_HEIGHT,
             HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    gtk_button_set_alignment (GTK_BUTTON (self->pkr_interval), 0.0, 0.5);
    hildon_button_set_title (HILDON_BUTTON (self->pkr_interval),
                             _("Logging interval"));
    selector = hildon_touch_selector_new_text ();
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      _(" 1 second"));
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      _(" 2 seconds"));
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      _(" 5 seconds"));
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      _("10 seconds"));
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      _("20 seconds"));
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      _("30 seconds"));
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      _("60 seconds"));
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      _("120 seconds"));
    hildon_picker_button_set_selector(HILDON_PICKER_BUTTON
             (self->pkr_interval), HILDON_TOUCH_SELECTOR (selector));
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->pkr_interval, TRUE, TRUE, 5);
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (selector),
                                      0, self->settings->interval);
    /* Log */
    gtk_box_pack_start (GTK_BOX (vbox),
                        gtk_label_new (_("Log")),
                        TRUE, TRUE, 4);
    /* Log path */
    self->btn_path = hildon_button_new (HILDON_SIZE_FINGER_HEIGHT,
		                  HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    gtk_button_set_alignment (GTK_BUTTON (self->btn_path), 0.0, 0.5);
    hildon_button_set_title(HILDON_BUTTON(self->btn_path),
		            _("Folder to store the logs"));
    hildon_button_set_value(HILDON_BUTTON(self->btn_path),
		            self->settings->log_folder);
    gtk_box_pack_start (GTK_BOX (vbox),
		        self->btn_path, TRUE, TRUE, 5);
    g_signal_connect(self->btn_path, "clicked",
		     G_CALLBACK (on_btn_path), dialog);    
    /* */
    panarea = hildon_pannable_area_new ();
    g_object_set (panarea, "initial-hint", TRUE, NULL);
    hildon_pannable_area_add_with_viewport
            (HILDON_PANNABLE_AREA (panarea), GTK_WIDGET (vbox));
    gtk_box_pack_start (GTK_BOX
                        (GTK_DIALOG (dialog)->vbox), panarea,
                        TRUE, TRUE, 5);
    gtk_widget_show_all (dialog);
    hildon_pannable_area_scroll_to (HILDON_PANNABLE_AREA (panarea), -1,
                                    0);
    return GTK_DIALOG (dialog);
}

void
on_settings_dialog_show (GtkWidget *widget,
			 gpointer data)
{
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);
    if (!self->settings_dialog) {
    	self->settings_dialog = GTK_WIDGET (settings_dialog_new (self));
        g_signal_connect (self->settings_dialog, "response",
                          G_CALLBACK (on_settings_dialog_response),
                          self);
    } else {
	gtk_widget_show (self->settings_dialog);
    }
}

/* -------------------------------------------------------------------- */
/* Pixbufs and icons
 */

static void
free_pixbuf(GdkPixbuf **pixbuf)
{
    if (*pixbuf)
    {
        g_object_unref(*pixbuf);
        *pixbuf = NULL;
    }
}

static void
free_theme_pixbufs(GpsDataLoggerWidget * self)
{
    free_pixbuf(&self->button_bg);
    free_pixbuf(&self->button_pressed_bg);
}

static void
load_theme_pixbufs(GpsDataLoggerWidget * self)
{
    self->button_bg = gdk_pixbuf_new_from_file(BUTTON_BG, NULL);
    self->button_pressed_bg = gdk_pixbuf_new_from_file(BUTTON_PRESSED_BG, NULL);
}

static gboolean
theme_pixbufs_loaded(GpsDataLoggerWidget * self)
{
    if (self->button_bg && self->button_pressed_bg)
        return TRUE;

    return FALSE;
}

static gboolean
on_style_set(GtkWidget *widget, GtkStyle *previous_style, gpointer data)
{
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);

    /* if pixmaps already loaded, free them */
    if (theme_pixbufs_loaded(self))
            free_theme_pixbufs(self);

    load_theme_pixbufs(self);

    if (!theme_pixbufs_loaded(self))
            free_theme_pixbufs(self);

    return FALSE;
}

static void
free_icons(GpsDataLoggerWidget * self)
{
        free_pixbuf(&self->rec_icon);
        free_pixbuf(&self->stop_icon);
        free_pixbuf(&self->pause_icon);
	free_pixbuf(&self->gps_refresh_icon);
	free_pixbuf(&self->gps_on_icon);
	free_pixbuf(&self->gps_off_icon);
}

static void
load_icons(GpsDataLoggerWidget * self)
{
        GtkIconTheme *theme;

        theme = gtk_icon_theme_get_default();

        /* Load icons */
        self->rec_icon = gtk_icon_theme_load_icon(theme,
                        REC_ICON, 48, 0, NULL);
        self->stop_icon = gtk_icon_theme_load_icon(theme,
                        STOP_ICON, 48, 0, NULL);
        self->pause_icon = gtk_icon_theme_load_icon(theme,
                        PAUSE_ICON, 48, 0, NULL);
        self->gps_refresh_icon = gtk_icon_theme_load_icon(theme,
                        GPS_REFRESH_ICON, 48, 0, NULL);
        self->gps_on_icon = gtk_icon_theme_load_icon(theme,
                        GPS_ON_ICON, 48, 0, NULL);
        self->gps_off_icon = gtk_icon_theme_load_icon(theme,
                        GPS_OFF_ICON, 48, 0, NULL);
}

/* -------------------------------------------------------------------- */

static gboolean
on_widget_leave(GtkWidget* widget, GdkEventCrossing *event, gpointer data)
{
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);

    self->rec_button_down = FALSE;
    self->stop_button_down = FALSE;
    gtk_widget_queue_draw(widget);

    return FALSE;
}


#ifdef DEBUG
static void
debuglog_close(GpsDataLoggerWidget * self)
{
    if (self->logfile) {
	fclose(self->logfile);
	self->logfile = NULL;
    }
}


static void
debuglog_start(GpsDataLoggerWidget * self)
{
    gchar *path;
    time_t current;
    struct tm *tp;


    /* If the logfile is already open (should not) then close it */
    if (self->logfile)
        debuglog_close(self);
    /* Name the log file using the current time */
    current = time(NULL);
    tp = localtime(&current);
    path = g_strdup_printf("%s/%d%02d%02d_%02d%02d%02d.log", 
		           self->settings->log_folder,
		    	   tp->tm_year + 1900,
		   	   tp->tm_mon + 1,
		   	   tp->tm_mday,
		   	   tp->tm_hour,
		   	   tp->tm_min,
		   	   tp->tm_sec );
    self->logfile = fopen(path, "w");
    g_free(path);
    
}
#endif

static void
logfile_close(GpsDataLoggerWidget * self)
{
    if (self->file) {
        g_fprintf(self->file, "</trkseg>\n</trk>\n</gpx>\n");
        fclose(self->file);
        self->file = NULL;
    }
}

static void
logfile_start(GpsDataLoggerWidget * self)
{
    gchar *path;
    time_t current;
    struct tm *tp;
    char string_buf[5];

    /* logfile_start can only be called in APP_STATE_STARTING state */
    if (self->app_state != APP_STATE_STARTING)
	return;
    /* Test locale if it has a comma for the decimal point */
    g_snprintf(string_buf, 5, "%f", 3.14);
    if (g_strrstr(string_buf,",") != NULL) {
        self->has_comma_as_decimal_point = TRUE;
    } else {
        self->has_comma_as_decimal_point = FALSE;
    }
    /* If the logfile is already open (should not) then close it */
    if (self->file)
        logfile_close(self);
    /* Name the log file using the current time */
    current = time(NULL);
    tp = localtime(&current);
    path = g_strdup_printf("%s/%d%02d%02d_%02d%02d%02d.gpx", 
		           self->settings->log_folder,
		    	   tp->tm_year + 1900,
		   	   tp->tm_mon + 1,
		   	   tp->tm_mday,
		   	   tp->tm_hour,
		   	   tp->tm_min,
		   	   tp->tm_sec );
    self->file = fopen(path, "w");
    if (! (self->file)) {
        hildon_banner_show_informationf(NULL,
                        NULL,
                        _("Could not open \"%s\" for writing!"),
			path);
#ifndef __arm__
	g_print(_("Could not open logfile for writing, going to IDLE\n"));
#endif
        self->app_state = APP_STATE_IDLE;
        location_gpsd_control_stop(self->control);
        update_status_and_buttons(self);
    } else {
        g_fprintf(self->file, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
        g_fprintf(self->file, "<gpx version=\"1.0\"\n");
        g_fprintf(self->file, "xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n");
        g_fprintf(self->file, "creator=\"gps-data-logger on Maemo\"\n");
        g_fprintf(self->file, "xmlns=\"http://www.topografix.com/GPX/1/0\"\n");
        g_fprintf(self->file, "xsi:schemaLocation=\"http://www.topografix.com/GPX/1/0 http://www.topografix.com/GPX/1/0/gpx.xsd\">\n");
        g_fprintf(self->file, "<trk>\n");
        g_fprintf(self->file, "<trkseg>\n");
    }
    g_free(path);
}

static void
update_status_and_buttons (GpsDataLoggerWidget * self)
{
#ifdef DEBUG
    if (self->logfile) {
        g_fprintf(self->logfile, _("%s -- enter (state: %d)\n"), __func__, self->app_state);
	fflush(self->logfile);
    }
#endif

    switch(self->app_state) {
	case APP_STATE_STARTING:
		gtk_image_set_from_pixbuf (GTK_IMAGE(self->gps_status), 
				           self->gps_refresh_icon);
		self->rec_button_enabled = FALSE;
		self->stop_button_enabled = TRUE;
#ifndef __arm__
		g_print(_(" starting: loading new icon, disabling rec\n"));
#endif
		break;
	case APP_STATE_RECORDING:
		gtk_image_set_from_pixbuf (GTK_IMAGE(self->gps_status), 
				           self->gps_on_icon);
		self->pause_button_visible = TRUE;
		self->rec_button_enabled = TRUE;
		self->stop_button_enabled = TRUE;
		break;
	case APP_STATE_PAUSED:
		self->pause_button_visible = FALSE;
		break;
	case APP_STATE_STOPPING:
		gtk_image_set_from_pixbuf (GTK_IMAGE(self->gps_status), 
				           self->gps_off_icon);
		self->pause_button_visible = FALSE;
		self->rec_button_enabled = FALSE;
		self->stop_button_enabled = FALSE;
		break;
	case APP_STATE_IDLE:
	default:
		gtk_image_set_from_pixbuf (GTK_IMAGE(self->gps_status), 
				           self->gps_off_icon);
		self->pause_button_visible = FALSE;
		self->rec_button_enabled = TRUE;
		self->stop_button_enabled = FALSE;
#ifndef __arm__
		g_print(_(" idle or default\n"));
#endif
		break;
    }
    gtk_widget_queue_draw(GTK_WIDGET(self));
#ifdef DEBUG
    if (self->logfile) {
        g_fprintf(self->logfile, _("%s -- leave\n"), __func__);
	fflush(self->logfile);
    }
#endif
}

static void
handle_button_press (GtkWidget *button,
		     gpointer data)
{
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);
#ifdef DEBUG
    if (self->logfile) {
        g_fprintf(self->logfile, _("%s -- enter (state: %d)\n"), __func__, self->app_state);
	fflush(self->logfile);
    }
#endif

    if (button == self->button_rec) {
        if (self->app_state == APP_STATE_IDLE) {
            self->app_state = APP_STATE_STARTING;
#ifndef __arm__
	    g_print(_(" new state: starting\n"));
            g_print(_("handle_button_press : calling update_status_and_buttons\n"));
#endif
            update_status_and_buttons(self);
	    /* open a new log file */
	    logfile_start(self);
#ifdef DEBUG
    if (self->logfile) {
        g_fprintf(self->logfile, _("%s -- calling location_gpsd_control_start\n"), __func__);
	fflush(self->logfile);
    }
#endif
            location_gpsd_control_start(self->control);
	    g_usleep(500);
	    location_gps_device_reset_last_known(self->device);
	}
	else if (self->app_state == APP_STATE_RECORDING) {
            self->app_state = APP_STATE_PAUSED;
#ifndef __arm__
	    g_print(_(" new state: paused\n"));
            g_print(_("handle_button_press : calling update_status_and_buttons\n"));
#endif
            update_status_and_buttons(self);
            location_gpsd_control_stop(self->control);
	}
	else if (self->app_state == APP_STATE_PAUSED) {
            self->app_state = APP_STATE_STARTING;
#ifndef __arm__
	    g_print(_(" new state: starting, after pause\n"));
            g_print(_("handle_button_press : calling update_status_and_buttons\n"));
#endif
            update_status_and_buttons(self);
	    /* close tarck segment and start a new one,
	     * if there were points recorded in the previous
	     * segment */
	    if (self->points_recorded_in_current_segment)
	        g_fprintf(self->file, "</trkseg>\n<trkseg>\n");
	    location_gps_device_reset_last_known(self->device);
            location_gpsd_control_start(self->control);
	}
    } else if (button == self->button_stop) {
	self->app_state = APP_STATE_STOPPING;
#ifndef __arm__
	g_print(_(" new state: stoping\n"));
        g_print(_("handle_button_press : calling update_status_and_buttons\n"));
#endif
        update_status_and_buttons(self);
        location_gpsd_control_stop(self->control);
	/* close log file */
	logfile_close(self);
    }
#ifdef DEBUG
    if (self->logfile) {
        g_fprintf(self->logfile, _("%s -- leave (new state: %d)\n"), __func__, self->app_state);
	fflush(self->logfile);
    }
#endif
}

static gboolean
on_button_release(GtkWidget* widget, GdkEventButton *button_event, gpointer data)
{
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);
    gboolean retval = FALSE;

#ifdef DEBUG
    if (self->logfile) {
        g_fprintf(self->logfile, _("%s -- enter "), __func__);
        if (widget == GTK_WIDGET(self->background)) {
	    g_fprintf(self->logfile, _(" widget: background\n"));
	} else if (widget == GTK_WIDGET(self->button_rec)) {
	    g_fprintf(self->logfile, _(" widget: button_rec\n"));
	} else if (widget == GTK_WIDGET(self->button_stop)) {
	    g_fprintf(self->logfile, _(" widget: button_stop\n"));
	}
	fflush(self->logfile);
    }
#endif
    self->rec_button_down = FALSE;
    self->stop_button_down = FALSE;
    gtk_widget_queue_draw(GTK_WIDGET(self));
    g_usleep(500);

    if (widget == GTK_WIDGET(self->background)) {
        if (button_event->x >= self->button_rec->allocation.x &&
            button_event->y >= self->button_rec->allocation.y &&
            button_event->x <= self->button_rec->allocation.x +
                           self->button_rec->allocation.width &&
            button_event->y <= self->button_rec->allocation.y +
                           self->button_rec->allocation.height &&
            self->rec_button_enabled) {
#ifdef DEBUG
                if (self->logfile) {
                    g_fprintf(self->logfile, _("  calling handle_button_press on background for button_rec\n"));
	            fflush(self->logfile);
		}
#endif
                handle_button_press(self->button_rec, self);
        } else if (button_event->x >= self->button_stop->allocation.x &&
            button_event->y >= self->button_stop->allocation.y &&
            button_event->x <= self->button_stop->allocation.x +
                           self->button_stop->allocation.width &&
            button_event->y <= self->button_stop->allocation.y +
                           self->button_stop->allocation.height &&
            self->stop_button_enabled) {
#ifdef DEBUG
                if (self->logfile) {
                    g_fprintf(self->logfile, _("  calling handle_button_press on background for button_stop\n"));
	            fflush(self->logfile);
		}
#endif
                handle_button_press(self->button_stop, self);
        }	
    } else if ((widget == GTK_WIDGET(self->button_rec)) && 
		    self->rec_button_enabled) {
#ifdef DEBUG
        if (self->logfile) {
            g_fprintf(self->logfile, _("  calling handle_button_press on button_rec for button_rec\n"));
	    fflush(self->logfile);
	}
#endif
        handle_button_press(self->button_rec, self);
    } else if ((widget == GTK_WIDGET(self->button_stop)) && 
		    self->stop_button_enabled) {
#ifdef DEBUG
        if (self->logfile) {
             g_fprintf(self->logfile, _("  calling handle_button_press on button_stop for button_stop\n"));
	     fflush(self->logfile);
	}
#endif
        handle_button_press(self->button_stop, self);
    }
    
#ifdef DEBUG
    if (self->logfile) {
        g_fprintf(self->logfile, _("%s -- leave\n"), __func__);
	fflush(self->logfile);
    }
#endif
    return retval;
}

static gboolean
on_button_press(GtkWidget* widget, GdkEventButton *button_event, gpointer data)
{
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);
    gboolean retval = FALSE;

#ifdef DEBUG
    if (self->logfile) {
        g_fprintf(self->logfile, _("%s -- enter "), __func__);
        if (widget == GTK_WIDGET(self->background)) {
	    g_fprintf(self->logfile, _(" widget: background\n"));
	} else if (widget == GTK_WIDGET(self->button_rec)) {
	    g_fprintf(self->logfile, _(" widget: button_rec\n"));
	} else if (widget == GTK_WIDGET(self->button_stop)) {
	    g_fprintf(self->logfile, _(" widget: button_stop\n"));
	}
	fflush(self->logfile);
    }
#endif
    if (widget == GTK_WIDGET(self->background)) {
        if (button_event->x >= self->button_rec->allocation.x &&
           button_event->y >= self->button_rec->allocation.y &&
            button_event->x <= self->button_rec->allocation.x +
                           self->button_rec->allocation.width &&
            button_event->y <= self->button_rec->allocation.y +
                           self->button_rec->allocation.height &&
            self->rec_button_enabled)
        {
            self->rec_button_down = TRUE;
            gtk_widget_queue_draw(widget);
        } else if (button_event->x >= self->button_stop->allocation.x &&
            button_event->y >= self->button_stop->allocation.y &&
            button_event->x <= self->button_stop->allocation.x +
                           self->button_stop->allocation.width &&
            button_event->y <= self->button_stop->allocation.y +
                           self->button_stop->allocation.height &&
            self->stop_button_enabled)
        {
            self->stop_button_down = TRUE;
            gtk_widget_queue_draw(widget);
        }
    } else if ((widget == GTK_WIDGET(self->button_rec)) &&
		    self->rec_button_enabled) {
        self->rec_button_down = TRUE;
        gtk_widget_queue_draw(widget);
    } else if ((widget == GTK_WIDGET(self->button_stop)) &&
		    self->stop_button_enabled) {
        self->stop_button_down = TRUE;
        gtk_widget_queue_draw(widget);
    }

#ifdef DEBUG
    if (self->logfile) {
        g_fprintf(self->logfile, _("%s -- leave\n"), __func__);
	fflush(self->logfile);
    }
#endif
    return retval;
}

static void
on_error(LocationGPSDControl *control, gpointer data)
{
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);
#ifndef __arm__
    g_printf("%s\n", __func__);
#endif
#ifdef DEBUG
    if (self->logfile) {
        g_fprintf(self->logfile, _("%s -- enter\n"), __func__);
	fflush(self->logfile);
    }
#endif
    if (self->app_state != APP_STATE_IDLE) {
	hildon_banner_show_information(NULL,
			NULL,
			_("on_error, going to IDLE state"));
	self->app_state = APP_STATE_IDLE;
	location_gpsd_control_stop(self->control);
#ifndef __arm__
        g_print(_("on_error : calling update_status_and_buttons\n"));
#endif
	update_status_and_buttons(self);
	logfile_close(self);
    }
}

static void
on_start(LocationGPSDControl *control, gpointer data)
{
#ifndef __arm__
        g_printf("%s\n", __func__);
#endif
#ifdef DEBUG
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);
    if (self->logfile) {
        g_fprintf(self->logfile, _("%s -- was here\n"), __func__);
	fflush(self->logfile);
    }
#endif
}

static void
on_stop(LocationGPSDControl *control, gpointer data)
{
	GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);
#ifndef __arm__
        g_printf("%s\n", __func__);
#endif
#ifdef DEBUG
    if (self->logfile) {
        g_fprintf(self->logfile, _("%s -- enter\n"), __func__);
	fflush(self->logfile);
    }
#endif
	if (self->app_state == APP_STATE_STOPPING) {
        	hildon_banner_show_information(NULL,
			                       NULL,
		        	               _("on_stop, going to IDLE state"));
		self->app_state = APP_STATE_IDLE;
#ifndef __arm__
                g_print(_("on_stop : calling update_status_and_buttons\n"));
#endif
#ifdef DEBUG
            if (self->logfile) {
                g_fprintf(self->logfile, _("%s -- going to IDLE, calling update_status_and_buttons\n"), __func__);
	        fflush(self->logfile);
            }
#endif
		update_status_and_buttons(self);
	}
}

static void
on_changed(LocationGPSDevice *device, gpointer data)
{
	GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);
	time_t tme;
	struct tm *tp;
	char string_buf[256];
	int i;
	
        if (!device)
                return;

        if (device->fix) {
		if (((self->app_state == APP_STATE_STARTING) ||
		     (self->app_state == APP_STATE_RECORDING)) &&
		    (device->fix->fields & LOCATION_GPS_DEVICE_TIME_SET) &&
		    (device->fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET)) {
			if (self->app_state == APP_STATE_STARTING) {
			    hildon_banner_show_information(NULL,
                                NULL,
                                _("on_changed, going to RECORDING state"));
                  
                            self->app_state = APP_STATE_RECORDING;
#ifndef __arm__
                            g_print(_("on_change : calling update_status_and_buttons\n"));
#endif
                            update_status_and_buttons(self);
	                    self->points_recorded_in_current_segment = 0;
			}
			tme = device->fix->time;
			tp = gmtime(&tme);
			g_snprintf(string_buf, 256, "<trkpt lat=\"%f\" lon=\"%f\">\n",
					device->fix->latitude,
					device->fix->longitude);
			if (self->has_comma_as_decimal_point) {
			    for (i = 0; i < sizeof(string_buf); i++) {
				if (string_buf[i] == ',')
					string_buf[i] = '.';
			    }
			}
			g_fprintf(self->file, string_buf);
			if (device->fix->fields & LOCATION_GPS_DEVICE_ALTITUDE_SET) {
			    g_snprintf(string_buf, 256, "  <ele>%.1f</ele>\n",
						device->fix->altitude);
			    if (self->has_comma_as_decimal_point) {
				for (i=0; i<sizeof(string_buf); i++) {
					if (string_buf[i] == ',')
						string_buf[i] = '.';
				}
			    }
			    g_fprintf(self->file, string_buf);
			}
			g_fprintf(self->file, "  <time>%d-%02d-%02dT%02d:%02d:%02dZ</time>\n",
					tp->tm_year + 1900,
					tp->tm_mon + 1,
					tp->tm_mday,
					tp->tm_hour,
					tp->tm_min,
					tp->tm_sec);
			g_fprintf(self->file, "  <sat>%d</sat>\n",
					device->satellites_in_use);
			g_fprintf(self->file, "</trkpt>\n");
			self->points_recorded_in_current_segment += 1;
		}
#ifndef __arm__
                if (device->fix->fields & LOCATION_GPS_DEVICE_TIME_SET)
                        g_printf("%s:%s: time = %f\n",
                                        __FILE__, __func__,
                                        device->fix->time);

                if (device->fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET)
                        g_printf("%s:%s: lat = %f, long = %f\n",
                                        __FILE__,
                                        __func__,
                                        device->fix->latitude,
                                        device->fix->longitude);

                if (device->fix->fields & LOCATION_GPS_DEVICE_ALTITUDE_SET)
                        g_printf("%s:%s: alt = %f\n",
                                        __FILE__, __func__,
                                        device->fix->altitude);

                if (device->fix->fields & LOCATION_GPS_DEVICE_SPEED_SET)
                        g_printf("%s:%s: speed = %f\n",
                                        __FILE__, __func__,
                                        device->fix->speed);

                if (device->fix->fields & LOCATION_GPS_DEVICE_TRACK_SET)
                        g_printf("%s:%s: track = %f\n",
                                        __FILE__, __func__,
                                        device->fix->track);
                if (device->fix->fields & LOCATION_GPS_DEVICE_CLIMB_SET)
                        g_printf( "%s:%s: climb = %f\n",
                                        __FILE__, __func__,
                                        device->fix->climb);

                g_printf("%s:%s Accuracy values:\n", __FILE__, __func__);
                g_printf("\tept = %e, eph = %e, epv = %e, epd = %e, "
                                "eps = %e, epc = %e\n",
                                device->fix->ept,
                                device->fix->eph,
                                device->fix->epv,
                                device->fix->epd,
                                device->fix->eps,
                                device->fix->epc);
        }

        g_printf("%s:%s: Satellites in view: %d\n",
                        __FILE__, __func__,
                        device->satellites_in_view);

        g_printf("%s:%s: Satellites in use: %d\n", __FILE__, __func__,
                        device->satellites_in_use);

        if (device && device->cell_info) {
                if (device->cell_info->flags & LOCATION_CELL_INFO_GSM_CELL_INFO_SET)
                        g_printf( "Mobile Coutry Code GSM: %d\n",
                                        device->cell_info->gsm_cell_info.mcc);

                if (device->cell_info->flags & LOCATION_CELL_INFO_WCDMA_CELL_INFO_SET)
                        g_printf( "Mobile Coutry Code WCDMA: %d\n",
                                        device->cell_info->wcdma_cell_info.mcc);
#endif
        }
}

static void
draw_pixbuf(cairo_t *cr, GdkPixbuf *pbuf, int x, int y, gboolean dimmed)
{
        int pbuf_width, pbuf_height;

        pbuf_width = gdk_pixbuf_get_width(pbuf);
        pbuf_height = gdk_pixbuf_get_height(pbuf);

        gdk_cairo_set_source_pixbuf(cr, pbuf, 
			            x - (pbuf_width / 2),
                                    y - (pbuf_height / 2));
	if (dimmed)
	    cairo_paint_with_alpha(cr, 0.3);
	else
            cairo_paint(cr);
}

static void
draw_rounded_rectangle(cairo_t *cr, double x, double y,
                double width, double height, double rad)
{
        cairo_move_to(cr, x+rad, y);
        cairo_line_to(cr, x+width-rad, y);
        cairo_curve_to(cr, x+width, y, x+width, y, x+width, y+rad);
        cairo_line_to(cr, x+width, y+height-rad);
        cairo_curve_to(cr, x+width, y+height, x+width, y+height, x+width-rad, y+height);
        cairo_line_to(cr, x+rad, y+height);
        cairo_curve_to(cr, x, y+height, x, y+height, x, y+height-rad);
        cairo_line_to(cr, x, y+rad);
        cairo_curve_to(cr, x, y, x, y, x+rad, y);
        cairo_close_path(cr);
}

static void
draw_background(cairo_t *cr, GdkRectangle *area, gint border, gdouble transparency)
{
        cairo_pattern_t *grad;

	gdk_cairo_rectangle(cr, area);
        cairo_clip(cr);

        cairo_set_operator(cr, CAIRO_OPERATOR_OVER);

        /* Gradient background */
        grad = cairo_pattern_create_linear(area->x+border,
                        area->y+border, area->x+border,
                        area->y+area->height-(2*border));
        cairo_pattern_add_color_stop_rgba(grad, 0, 0.4, 0.4, 0.4, transparency);
        cairo_pattern_add_color_stop_rgba(grad, 1, 0.05, 0.05, 0.05, transparency);

        /* Draw border */
        cairo_set_line_width(cr, 2);
        draw_rounded_rectangle(cr, area->x+border,
                        area->y+border, area->width-(2*border),
                        area->height-(2*border), 10.0);
        cairo_set_source_rgba(cr, 0.4, 0.4, 0.4, transparency);
        cairo_stroke_preserve(cr);

        /* Fill background with gradient */
        cairo_set_source(cr, grad);
        cairo_fill(cr);

        cairo_pattern_destroy(grad);
}


static gboolean
on_background_expose(GtkWidget *widget, GdkEventExpose *expose_event, gpointer data)
{
    /* GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data); */
    cairo_t *cr = NULL;

    cr = gdk_cairo_create(widget->window);

    if (!cr) 
        return FALSE;

    cairo_set_operator(cr, CAIRO_OPERATOR_CLEAR);
    gdk_cairo_region(cr, expose_event->region);
    cairo_fill(cr);

    draw_background(cr, &expose_event->area, 3, 0.6);

    cairo_destroy(cr);

    return FALSE;
}

static gboolean
on_button_expose(GtkWidget *widget, GdkEventExpose *expose_event, gpointer data)
{
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);

    cairo_t *cr = NULL;
    cr = gdk_cairo_create(widget->window);

    if (!cr) return FALSE;

    /* button background */
    /* If theme images are not loaded draw with cairo */
    if (!theme_pixbufs_loaded(self))
    {
	if (((widget == self->button_rec) && (self->rec_button_down)) ||
	   ((widget == self->button_stop) && (self->stop_button_down)))
                draw_background(cr, &expose_event->area, 0, 0.6);
            else
                draw_background(cr, &expose_event->area, 0, 0.8);
    } else {
        if (((widget == self->button_rec) && (self->rec_button_down)) ||
            ((widget == self->button_stop) && (self->stop_button_down))) 
                draw_pixbuf(cr, self->button_pressed_bg,
			    widget->allocation.x + widget->allocation.width/2,
			    widget->allocation.y + widget->allocation.height/2,
			    FALSE);
        else if (((widget == self->button_rec) && (! self->rec_button_enabled)) ||
                ((widget == self->button_stop) && (! self->stop_button_enabled))) 
                draw_pixbuf(cr, self->button_bg,
			    widget->allocation.x + widget->allocation.width/2,
			    widget->allocation.y + widget->allocation.height/2,
			    TRUE);
        else
                draw_pixbuf(cr, self->button_bg,
			    widget->allocation.x + widget->allocation.width/2,
			    widget->allocation.y + widget->allocation.height/2,
			    FALSE);
    }
    /* button icons */
    if (widget == self->button_stop)
            draw_pixbuf(cr, self->stop_icon,
			    widget->allocation.x + widget->allocation.width/2,
			    widget->allocation.y + widget->allocation.height/2,
			    ! self->stop_button_enabled);
    else if ((widget == self->button_rec) && (self->pause_button_visible))
            draw_pixbuf(cr, self->pause_icon,
			    widget->allocation.x + widget->allocation.width/2,
			    widget->allocation.y + widget->allocation.height/2,
			    ! self->rec_button_enabled);
    else if ((widget == self->button_rec) && (! self->pause_button_visible))
            draw_pixbuf(cr, self->rec_icon,
			    widget->allocation.x + widget->allocation.width/2,
			    widget->allocation.y + widget->allocation.height/2,
			    ! self->rec_button_enabled);

    cairo_destroy(cr);

    return FALSE;
}

/* Callback for hardware D-BUS events */
void hw_event_handler(osso_hw_state_t *state, gpointer data)
{
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);
    gboolean should_stop_recording = FALSE;
#ifdef DEBUG
    if (self->logfile) {
        g_fprintf(self->logfile, "%s -- enter\n", __func__);
        g_fprintf(self->logfile, "  hw event received - ");
        fflush(self->logfile);
    }
#endif

    if (state->shutdown_ind)
    {
        hildon_banner_show_information(NULL, NULL, 
                                       _("Shutdown event!"));
        should_stop_recording = TRUE;
#ifdef DEBUG
        if (self->logfile) {
            g_fprintf(self->logfile, _("  Shutdown event!\n"));
            fflush(self->logfile);
        }
#endif
    }
    if (state->save_unsaved_data_ind)
    {
        hildon_banner_show_information(NULL, NULL, 
                                       _("Must save unsaved data event!"));
        should_stop_recording = TRUE;
#ifdef DEBUG
        if (self->logfile) {
            g_fprintf(self->logfile, _("  Must save unsaved data event!\n"));
            fflush(self->logfile);
        }
#endif
    }
/*    if (state->memory_low_ind)
    {
        hildon_banner_show_information(NULL, NULL, 
                                       "Memory low event!");
#ifdef DEBUG
        if (self->logfile) {
            g_fprintf(self->logfile, "  Memory low event!\n");
            fflush(self->logfile);
        }
#endif
    }
    if (state->system_inactivity_ind)
    {
        hildon_banner_show_information(NULL, NULL, 
                                       "Minimize application inactivity event!");
#ifdef DEBUG
        if (self->logfile) {
            g_fprintf(self->logfile, "  Minimize application inactivity event!\n");
            fflush(self->logfile);
        }
#endif
    } */
    /* shall we stop recording and flush the log? */
    if ((self->app_state != APP_STATE_IDLE) && should_stop_recording) {
        /* let's prettend that the user pressed stop */
        handle_button_press(self->button_stop, self);
    }
}


static void
gps_data_logger_widget_init (GpsDataLoggerWidget * self)
{
    GtkWidget *vbox, *hbox_status, *hbox_buttons;
    GtkWidget *status_text;
    GdkScreen *screen;
    GdkColormap *colormap;
    osso_return_t result;
    GString *version = NULL ;
    GString *tmp = NULL;

    version = g_string_new(getenv("OSSO_PRODUCT_RELEASE_VERSION"));
#ifndef __arm__
    g_print("SW version: %s\n", version->str);
#endif
    self->osso_context = osso_initialize(OSSO_GDL_SERVICE, "0.0.1", TRUE, NULL);
    /* Check that initialization was ok */
    if (self->osso_context == NULL) {
#ifndef __arm__
        g_print("init : osso_initialize failed.\n");
#endif
        hildon_banner_show_information(NULL,
			NULL,
			_("osso_initialize failed"));
    }
    self->file = NULL;
#ifdef DEBUG
    self->logfile = NULL;
#endif
    self->app_state = APP_STATE_IDLE;
    /* Preload all icon pixmaps */
    load_icons(self);
    load_theme_pixbufs(self);

    /* Needed for transparency */
    screen = gtk_widget_get_screen(GTK_WIDGET(self));
    colormap = gdk_screen_get_rgba_colormap(screen);
    gtk_widget_set_colormap(GTK_WIDGET(self), colormap);

    self->background = gtk_event_box_new();
    gtk_widget_set_app_paintable(self->background, TRUE);
    gtk_container_add(GTK_CONTAINER(self), self->background);

    vbox = gtk_vbox_new (FALSE, HILDON_MARGIN_HALF);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), HILDON_MARGIN_DEFAULT);
    gtk_container_add(GTK_CONTAINER(self->background), vbox);

    self->rec_button_down = FALSE;
    self->rec_button_enabled = TRUE;
    self->button_rec = gtk_event_box_new();
    if (theme_pixbufs_loaded(self))
        gtk_widget_set_size_request(self->button_rec, 
		        gdk_pixbuf_get_width(self->button_bg),
                        gdk_pixbuf_get_height(self->button_bg));
    else
        gtk_widget_set_size_request(self->button_rec,96,96); 
    gtk_widget_set_app_paintable(self->button_rec, TRUE);
    gtk_event_box_set_visible_window(GTK_EVENT_BOX(self->button_rec), FALSE);
    
    self->stop_button_down = FALSE;
    self->stop_button_enabled = FALSE;
    self->button_stop = gtk_event_box_new();
    if (theme_pixbufs_loaded(self))
        gtk_widget_set_size_request(self->button_stop,
		        gdk_pixbuf_get_width(self->button_bg),
                        gdk_pixbuf_get_height(self->button_bg));
    else
        gtk_widget_set_size_request(self->button_stop,96,96);
    gtk_widget_set_app_paintable(self->button_stop, TRUE);
    gtk_event_box_set_visible_window(GTK_EVENT_BOX(self->button_stop), FALSE);

    self->gps_status = gtk_image_new();    
    status_text = gtk_label_new(_("Status: "));

    hbox_status = gtk_hbox_new (FALSE, 2);
    hbox_buttons = gtk_hbox_new (FALSE, 2);
    gtk_box_pack_start (GTK_BOX (hbox_status), status_text, FALSE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hbox_status), self->gps_status, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox_buttons), self->button_rec, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox_buttons), self->button_stop, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), hbox_status, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), hbox_buttons, FALSE, FALSE, 0);
     
#ifndef __arm__
    g_print(_("init : calling update_status_and_buttons\n"));
#endif
    update_status_and_buttons(self);

    gtk_widget_show_all (self->background);
    
    /* Set the resizing behavior */
    /*
    hd_home_widget_item_set_resize_type (HD_HOME_PLUGIN_ITEM (home_widget),
					 HD_HOME_PLUGIN_ITEM_RESIZE_BOTH);
    */

    self->control = location_gpsd_control_get_default();

    self->settings = settings_load();
    apply_settings(self);

    self->device  = g_object_new(LOCATION_TYPE_GPS_DEVICE, NULL);
    
    /* location signals */
    g_signal_connect(self->control, "error", G_CALLBACK(on_error), self);
    g_signal_connect(self->control, "gpsd-running", G_CALLBACK(on_start),
                                                                   self);
    g_signal_connect(self->control, "gpsd-stopped", G_CALLBACK(on_stop),
                                                                   self);
    g_signal_connect(self->device,  "changed", G_CALLBACK(on_changed),
    		    self);

    /* widget settings */
    hd_home_plugin_item_set_settings (HD_HOME_PLUGIN_ITEM (self), TRUE);
    g_signal_connect (self, "show-settings", G_CALLBACK (on_settings_dialog_show),
		    self);

    tmp = g_string_new("1.");
    g_string_truncate(version, 2);
    if (g_string_equal(version, tmp)) {
        /* On the first release of Fremantle all the button events are
         * going to the background, so let's register those too */
        g_signal_connect(G_OBJECT(self->background), "button-release-event",
                         G_CALLBACK(on_button_release), self);
        g_signal_connect(G_OBJECT(self->background), "button-press-event",
                         G_CALLBACK(on_button_press), self);
    } else {
        /* button signals */
        g_signal_connect(G_OBJECT(self->button_rec), "button-release-event",
		         G_CALLBACK(on_button_release), self);
        g_signal_connect(G_OBJECT(self->button_rec), "button-press-event",
		         G_CALLBACK(on_button_press), self);
        g_signal_connect(G_OBJECT(self->button_stop), "button-release-event",
		         G_CALLBACK(on_button_release), self);
        g_signal_connect(G_OBJECT(self->button_stop), "button-press-event",
		         G_CALLBACK(on_button_press), self);
    }
    
    g_signal_connect(G_OBJECT(self->background), "leave-notify-event",
                    G_CALLBACK(on_widget_leave), self);
    /* expose events */
    g_signal_connect(self->background, "expose-event", 
		     G_CALLBACK(on_background_expose),
		     (gpointer)self);
    g_signal_connect(self->button_rec, "expose-event", 
                     G_CALLBACK(on_button_expose),
                     (gpointer)self);
    g_signal_connect(self->button_stop, "expose-event", 
                     G_CALLBACK(on_button_expose),
                     (gpointer)self);
    g_signal_connect(G_OBJECT(self->background), "style-set",
                     G_CALLBACK(on_style_set), (gpointer)self);
#ifdef DEBUG
    debuglog_start(self);
#endif
    if (self->osso_context) {
        /* Add handler for hardware D-BUS messages */
        result = osso_hw_set_event_cb( self->osso_context,
            NULL, hw_event_handler, (gpointer)self );

        if (result != OSSO_OK)
        {
#ifndef __arm__
            g_print(_("Error setting HW state callback (%d)\n"), result);
#endif
        }
    }
    g_string_free(version, TRUE);
    g_string_free(tmp, TRUE);

}

static void
gps_data_logger_widget_finalize (GObject *object)
{
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (object);
    /* Close the logfile if it is open */
    logfile_close(self);
#ifdef DEBUG
    debuglog_close(self);
#endif
    g_object_unref(self->device);
    g_object_unref(self->control);
    if (self->settings_dialog) {
        gtk_widget_destroy (self->settings_dialog);
        self->settings_dialog = NULL;
    }
    settings_free(self->settings);
    g_free(self->settings);
    free_icons(self);
    free_theme_pixbufs(self);
    /* Deinitialize OSSO */
    if (self->osso_context) {
        osso_deinitialize(self->osso_context);
        self->osso_context = NULL;
    }
		    
    /* Call the base class's implementation: */
    G_OBJECT_CLASS(gps_data_logger_widget_parent_class)->finalize (object);
}

static void
gps_data_logger_widget_class_init (GpsDataLoggerWidgetClass * klass)
{
    GObjectClass         *object_class;
    object_class = G_OBJECT_CLASS (klass);
    object_class->finalize = gps_data_logger_widget_finalize;
}

static void
gps_data_logger_widget_class_finalize (GpsDataLoggerWidgetClass * klass)
{
}

