
/*
 * This file is part of gps-data-logger-widget
 *
 * Copyright (C) 2009 Marcell Lengyel <marcell@maemo.org>
 *
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <time.h>
#include <locale.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <glib/gprintf.h>

#include <hildon/hildon.h>
#include <libhildondesktop/libhildondesktop.h>
#include <hildon-fm-2/hildon/hildon-file-chooser-dialog.h>
#include <hildon/hildon-banner.h>

#include "gps-data-logger-widget.h"

#define PADDING 10

HD_DEFINE_PLUGIN_MODULE (GpsDataLoggerWidget, gps_data_logger_widget, HD_TYPE_HOME_PLUGIN_ITEM)

static void
update_status_and_buttons (GpsDataLoggerWidget * self);

static void
apply_settings (GpsDataLoggerWidget * self)
{
    LocationGPSDControlMethod my_method = 0;
    gint my_interval;

    if (self->settings->method_agnss)
        my_method = my_method | LOCATION_METHOD_AGNSS;
    if (self->settings->method_gnss)
        my_method = my_method | LOCATION_METHOD_GNSS;
    if (self->settings->method_acwp)
        my_method = my_method | LOCATION_METHOD_ACWP;
    if (self->settings->method_cwp)
        my_method = my_method | LOCATION_METHOD_CWP;
    if (self->settings->interval == 0)
        my_interval = LOCATION_INTERVAL_1S;
    else if (self->settings->interval == 1)
        my_interval = LOCATION_INTERVAL_2S;
    else if (self->settings->interval == 2)
        my_interval = LOCATION_INTERVAL_5S;
    else if (self->settings->interval == 3)
        my_interval = LOCATION_INTERVAL_10S;
    else if (self->settings->interval == 4)
        my_interval = LOCATION_INTERVAL_20S;
    else if (self->settings->interval == 5)
        my_interval = LOCATION_INTERVAL_30S;

    g_object_set(G_OBJECT (self->control),
                 "preferred-method", my_method,
                 "preferred-interval", my_interval,
                 NULL);
}

void
on_btn_path (GtkWidget * widget, gpointer user_data)
{
    GtkWidget *chooser;
    GtkDialog *dialog = GTK_DIALOG(user_data);

    chooser = hildon_file_chooser_dialog_new(GTK_WINDOW(dialog),
		    GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);
    /* Not in GTK 2.14 :(
    gtk_file_chooser_set_create_folders(GTK_FILE_CHOOSER (chooser),
		    TRUE); */
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER (chooser),
		    "/home/user/MyDocs");
    if (gtk_dialog_run (GTK_DIALOG (chooser)) == GTK_RESPONSE_OK) {
        hildon_button_set_value(HILDON_BUTTON(widget),
		 gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooser)));
	gtk_widget_destroy (chooser);
    } else {
	gtk_widget_destroy (chooser);
    }
}

static void
on_settings_dialog_response (GtkDialog *dialog,
  gint response_id, gpointer data)
{
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);
    gboolean	new_value, save_needed = FALSE;
    gint 	new_int;
    
    g_assert (self->settings_dialog);
    g_assert (self->settings_dialog == GTK_WIDGET (dialog));

    if (response_id == GTK_RESPONSE_OK) {
        /* save settings */
	new_value = hildon_check_button_get_active(
	            HILDON_CHECK_BUTTON(self->chk_method_agnss));
        if (self->settings->method_agnss != new_value) {
	    save_needed = TRUE;
	    self->settings->method_agnss = new_value;
	}
	new_value = hildon_check_button_get_active(
		    HILDON_CHECK_BUTTON(self->chk_method_gnss));
        if (self->settings->method_gnss != new_value) {
	    save_needed = TRUE;
	    self->settings->method_gnss = new_value;
	}
        new_value = hildon_check_button_get_active(
		    HILDON_CHECK_BUTTON(self->chk_method_acwp));
        if (self->settings->method_acwp != new_value) {
	    save_needed = TRUE;
	    self->settings->method_acwp = new_value;
	}
        new_value = hildon_check_button_get_active(
		    HILDON_CHECK_BUTTON(self->chk_method_cwp));
        if (self->settings->method_cwp != new_value) {
	    save_needed = TRUE;
	    self->settings->method_cwp = new_value;
	}
        new_int = hildon_picker_button_get_active(
		  HILDON_PICKER_BUTTON(self->pkr_interval));
	if (self->settings->interval != new_int) {
            save_needed = TRUE;
	    self->settings->interval = new_int;
	}
	if (g_ascii_strcasecmp(self->settings->log_folder,
                               hildon_button_get_value(
			       HILDON_BUTTON(self->btn_path))) != 0) {
	    g_stpcpy(self->settings->log_folder,
		     hildon_button_get_value(
		     HILDON_BUTTON(self->btn_path)));
	    save_needed = TRUE;
	}
	if (save_needed) {
	    settings_save(self->settings);
	    /* apply settings */
	    apply_settings(self);
	}
    } 
    gtk_widget_destroy (self->settings_dialog);
    self->settings_dialog = NULL;
}
	      
	
GtkDialog *
settings_dialog_new (GpsDataLoggerWidget * self)
{
    GtkWidget *dialog;
    GtkWidget *vbox;
    GtkWidget *selector;
    GtkWidget *panarea;

    dialog = gtk_dialog_new_with_buttons ("GPS Data Logger Widget Settings",
					  NULL, /* parent */
					  GTK_DIALOG_DESTROY_WITH_PARENT |
					  GTK_DIALOG_NO_SEPARATOR,
					  GTK_STOCK_SAVE, 
					  GTK_RESPONSE_OK, NULL);
    gtk_widget_set_size_request (dialog, 800, 350);
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox),
                        gtk_label_new ("Location Methods"),
                        TRUE, TRUE, 4);
    /* AGNSS */
    self->chk_method_agnss = hildon_check_button_new (HILDON_SIZE_FINGER_HEIGHT);
    hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->chk_method_agnss),
		    		self->settings->method_agnss);
    gtk_button_set_label (GTK_BUTTON (self->chk_method_agnss),
                          "Assisted GPS");
    gtk_button_set_alignment (GTK_BUTTON (self->chk_method_agnss),
                              0.0, 0.5);
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->chk_method_agnss, TRUE, TRUE, 4);
    /* GNSS */
    self->chk_method_gnss = hildon_check_button_new (HILDON_SIZE_FINGER_HEIGHT);
    hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->chk_method_gnss),
		    		self->settings->method_gnss);
    gtk_button_set_label (GTK_BUTTON (self->chk_method_gnss),
                          "GPS");
    gtk_button_set_alignment (GTK_BUTTON (self->chk_method_gnss),
                              0.0, 0.5);
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->chk_method_gnss, TRUE, TRUE, 4);
    /* ACWP */
    self->chk_method_acwp = hildon_check_button_new (HILDON_SIZE_FINGER_HEIGHT);
    hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->chk_method_acwp),
		    		self->settings->method_acwp);
    gtk_button_set_label (GTK_BUTTON (self->chk_method_acwp),
                          "Assisted, using cellular base stations");
    gtk_button_set_alignment (GTK_BUTTON (self->chk_method_acwp),
                              0.0, 0.5);
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->chk_method_acwp, TRUE, TRUE, 4);

    /* CWP */
    self->chk_method_cwp = hildon_check_button_new (HILDON_SIZE_FINGER_HEIGHT);
    hildon_check_button_set_active(HILDON_CHECK_BUTTON(self->chk_method_cwp),
		    		self->settings->method_cwp);
    gtk_button_set_label (GTK_BUTTON (self->chk_method_cwp),
                          "Center of the current country");
    gtk_button_set_alignment (GTK_BUTTON (self->chk_method_cwp),
                              0.0, 0.5);
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->chk_method_cwp, TRUE, TRUE, 4);
    /* Interval */
    gtk_box_pack_start (GTK_BOX (vbox),
                        gtk_label_new ("Interval"),
                        TRUE, TRUE, 4);
    /* Interval picker */
    self->pkr_interval = hildon_picker_button_new
            (HILDON_SIZE_FINGER_HEIGHT,
             HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    gtk_button_set_alignment (GTK_BUTTON (self->pkr_interval), 0.0, 0.5);
    hildon_button_set_title (HILDON_BUTTON (self->pkr_interval),
                             "Logging interval");
    selector = hildon_touch_selector_new_text ();
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      " 1 second");
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      " 2 seconds");
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      " 5 seconds");
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      "10 seconds");
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      "20 seconds");
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (selector),
                                      "30 seconds");
    hildon_picker_button_set_selector(HILDON_PICKER_BUTTON
             (self->pkr_interval), HILDON_TOUCH_SELECTOR (selector));
    gtk_box_pack_start (GTK_BOX (vbox),
                        self->pkr_interval, TRUE, TRUE, 5);
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (selector),
                                      0, self->settings->interval);
    /* Log */
    gtk_box_pack_start (GTK_BOX (vbox),
                        gtk_label_new ("Log"),
                        TRUE, TRUE, 4);
    /* Log path */
    self->btn_path = hildon_button_new (HILDON_SIZE_FINGER_HEIGHT,
		                  HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    gtk_button_set_alignment (GTK_BUTTON (self->btn_path), 0.0, 0.5);
    hildon_button_set_title(HILDON_BUTTON(self->btn_path),
		            "Folder to store the logs");
    hildon_button_set_value(HILDON_BUTTON(self->btn_path),
		            self->settings->log_folder);
    gtk_box_pack_start (GTK_BOX (vbox),
		        self->btn_path, TRUE, TRUE, 5);
    g_signal_connect(self->btn_path, "clicked",
		     G_CALLBACK (on_btn_path), dialog);    
    /* */
    panarea = hildon_pannable_area_new ();
    g_object_set (panarea, "initial-hint", TRUE, NULL);
    hildon_pannable_area_add_with_viewport
            (HILDON_PANNABLE_AREA (panarea), GTK_WIDGET (vbox));
    gtk_box_pack_start (GTK_BOX
                        (GTK_DIALOG (dialog)->vbox), panarea,
                        TRUE, TRUE, 5);
    gtk_widget_show_all (dialog);
    hildon_pannable_area_scroll_to (HILDON_PANNABLE_AREA (panarea), -1,
                                    0);
    return GTK_DIALOG (dialog);
}



void
settings_dialog_show (GtkWidget *widget,
			 gpointer data)
{
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);
    if (!self->settings_dialog) {
    	self->settings_dialog = GTK_WIDGET (settings_dialog_new (self));
        g_signal_connect (self->settings_dialog, "response",
                          G_CALLBACK (on_settings_dialog_response),
                          self);
    } else {
	gtk_widget_show (self->settings_dialog);
    }
    /*
    gtk_dialog_run (GTK_DIALOG (self->settings_dialog));
    gtk_widget_destroy (self->settings_dialog);
    */
}


static void
logfile_close(GpsDataLoggerWidget * self)
{
    if (self->file) {
        g_fprintf(self->file, "</trkseg>\n</trk>\n</gpx>\n");
        fclose(self->file);
        self->file = NULL;
    }
}

static void
logfile_start(GpsDataLoggerWidget * self)
{
    gchar *path;
    time_t current;
    struct tm *tp;
    char string_buf[5];

    /* logfile_start can only be called in APP_STATE_STARTING state */
    if (self->app_state != APP_STATE_STARTING)
	return;
    /* Test locale if it has a comma for the decimal point */
    g_snprintf(string_buf, 5, "%f", 3.14);
    if (g_strrstr(string_buf,",") != NULL) {
        self->has_comma_as_decimal_point = TRUE;
    } else {
        self->has_comma_as_decimal_point = FALSE;
    }
    /* If the logfile is already open (should not) then close it */
    if (self->file)
        logfile_close(self);
    /* Name the log file using the current time */
    current = time(NULL);
    tp = localtime(&current);
    path = g_strdup_printf("%s/%d%02d%02d_%02d%02d%02d.gpx", 
		           self->settings->log_folder,
		    	   tp->tm_year + 1900,
		   	   tp->tm_mon + 1,
		   	   tp->tm_mday,
		   	   tp->tm_hour,
		   	   tp->tm_min,
		   	   tp->tm_sec );
    self->file = fopen(path, "w");
    if (! (self->file)) {
        hildon_banner_show_informationf(NULL,
                        NULL,
                        "Could not open \"%s\" for writing!",
			path);
        self->app_state = APP_STATE_IDLE;
        location_gpsd_control_stop(self->control);
        update_status_and_buttons(self);
    } else {
        g_fprintf(self->file, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
        g_fprintf(self->file, "<gpx version=\"1.0\"\n");
        g_fprintf(self->file, "xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n");
        g_fprintf(self->file, "creator=\"gps-data-logger on Maemo\"\n");
        g_fprintf(self->file, "xmlns=\"http://www.topografix.com/GPX/1/0\"\n");
        g_fprintf(self->file, "xsi:schemaLocation=\"http://www.topografix.com/GPX/1/0 http://www.topografix.com/GPX/1/0/gpx.xsd\">\n");
        g_fprintf(self->file, "<trk>\n");
        g_fprintf(self->file, "<trkseg>\n");
    }
    g_free(path);
}

static void
update_status_and_buttons (GpsDataLoggerWidget * self)
{
    GtkIconTheme *icon_theme;
    GdkPixbuf *status_icon, *button_rec_icon;
    GtkWidget *button_rec_image;
    gboolean update_status_icon = FALSE;
    gboolean update_button_image = FALSE;

    icon_theme = gtk_icon_theme_get_default ();
    switch(self->app_state) {
	case APP_STATE_STARTING:
		status_icon = gtk_icon_theme_load_icon (icon_theme, "general_refresh", 48, 0, NULL);
		gtk_widget_set_sensitive(self->button_rec, FALSE);
		gtk_widget_set_sensitive(self->button_stop, TRUE);
		update_status_icon = TRUE;
#ifndef __arm__
		g_print(" starting: loading new icon, disabling rec\n");
#endif
		break;
	case APP_STATE_RECORDING:
		status_icon = gtk_icon_theme_load_icon (icon_theme, "gps_location", 48, 0, NULL);
		button_rec_icon = gtk_icon_theme_load_icon (icon_theme, "camera_video_pause", 48, 0, NULL);
		update_status_icon = TRUE;
		update_button_image = TRUE;
		gtk_widget_set_sensitive(self->button_rec, TRUE);
		gtk_widget_set_sensitive(self->button_stop, TRUE);
		break;
	case APP_STATE_PAUSED:
		button_rec_icon = gtk_icon_theme_load_icon (icon_theme, "camera_video_recording", 48, 0, NULL);
		update_button_image = TRUE;
		break;
	case APP_STATE_STOPPING:
		status_icon = gtk_icon_theme_load_icon (icon_theme, "gps_not_connected", 48, 0, NULL);
		button_rec_icon = gtk_icon_theme_load_icon (icon_theme, "camera_video_recording", 48, 0, NULL);
		update_status_icon = TRUE;
		update_button_image = TRUE;
		gtk_widget_set_sensitive(self->button_rec, FALSE);
		gtk_widget_set_sensitive(self->button_stop, FALSE);
		break;
	case APP_STATE_IDLE:
	default:
		status_icon = gtk_icon_theme_load_icon (icon_theme, "gps_not_connected", 48, 0, NULL);
		button_rec_icon = gtk_icon_theme_load_icon (icon_theme, "camera_video_recording", 48, 0, NULL);
		update_status_icon = TRUE;
		update_button_image = TRUE;
		gtk_widget_set_sensitive(self->button_stop, FALSE);
		gtk_widget_set_sensitive(self->button_rec, TRUE);
#ifndef __arm__
		g_print(" vagy idle vagy default volt\n");
#endif
    }
    if (update_status_icon) {
        if (status_icon == NULL)
            status_icon = gtk_icon_theme_load_icon (icon_theme,
	                                     "qgn_list_gene_default_app",
	                                     48, 0, NULL);
        gtk_image_set_from_pixbuf (GTK_IMAGE(self->gps_status), status_icon);
        gtk_widget_show_all (self->gps_status);
    }
    if (update_button_image) {
        if (button_rec_icon == NULL)
            button_rec_icon = gtk_icon_theme_load_icon (icon_theme,
	                                     "qgn_list_gene_default_app",
	                                     48, 0, NULL);
        button_rec_image = gtk_image_new_from_pixbuf (button_rec_icon);
        gtk_misc_set_padding (GTK_MISC (button_rec_image), PADDING, PADDING);
        g_object_unref (G_OBJECT (button_rec_icon));
        gtk_button_set_image (GTK_BUTTON (self->button_rec), button_rec_image);
        gtk_widget_show_all (self->button_rec);
    }
}

void
handle_button_press (GtkWidget *button,
		     gpointer data)
{
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);

    if (button == self->button_rec) {
        if (self->app_state == APP_STATE_IDLE) {
            self->app_state = APP_STATE_STARTING;
#ifndef __arm__
	    g_print(" new state: starting\n");
#endif
            location_gpsd_control_start(self->control);
	    /* open a new log file */
	    logfile_start(self);
	}
	else if (self->app_state == APP_STATE_RECORDING) {
            self->app_state = APP_STATE_PAUSED;
#ifndef __arm__
	    g_print(" new state: paused\n");
#endif
            location_gpsd_control_stop(self->control);
	}
	else if (self->app_state == APP_STATE_PAUSED) {
            self->app_state = APP_STATE_STARTING;
#ifndef __arm__
	    g_print(" new state: starting, after pause\n");
#endif
	    /* close tarck segment and start a new one,
	     * if there were points recorded in the previous
	     * segment */
	    if (self->points_recorded_in_current_segment)
	        g_fprintf(self->file, "</trkseg>\n<trkseg>\n");
            location_gpsd_control_start(self->control);
	}
    } else if (button == self->button_stop) {
	self->app_state = APP_STATE_STOPPING;
#ifndef __arm__
	g_print(" new state: stoping\n");
#endif
        location_gpsd_control_stop(self->control);
	/* close log file */
	logfile_close(self);
    }
    
    update_status_and_buttons(self);
}


static void on_error(LocationGPSDControl *control, gpointer data)
{
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);
    g_printf("%s\n", __func__);
    if (self->app_state != APP_STATE_IDLE) {
	hildon_banner_show_information(NULL,
			NULL,
			"on_error, going to IDLE state");
	self->app_state = APP_STATE_IDLE;
	location_gpsd_control_stop(self->control);
	update_status_and_buttons(self);
	logfile_close(self);
    }
}

static void on_start(LocationGPSDControl *control, gpointer data)
{
        g_printf("%s\n", __func__);
}

static void on_stop(LocationGPSDControl *control, gpointer data)
{
	GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);
        g_printf("%s\n", __func__);
	if (self->app_state == APP_STATE_STOPPING) {
        	hildon_banner_show_information(NULL,
			                       NULL,
		        	               "on_stop, going to IDLE state");
		self->app_state = APP_STATE_IDLE;
		update_status_and_buttons(self);
	}
}

static void on_changed(LocationGPSDevice *device, gpointer data)
{
	GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (data);
	time_t tme;
	struct tm *tp;
	char string_buf[256];
	int i;
	
        if (!device)
                return;

        if (device->fix) {
		if (((self->app_state == APP_STATE_STARTING) ||
		     (self->app_state == APP_STATE_RECORDING)) &&
		    (device->fix->fields & LOCATION_GPS_DEVICE_TIME_SET) &&
		    (device->fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET)) {
			if (self->app_state == APP_STATE_STARTING) {
			    hildon_banner_show_information(NULL,
                                NULL,
                                "on_changed, going to RECORDING state");
                  
                            self->app_state = APP_STATE_RECORDING;
                            update_status_and_buttons(self);
	                    self->points_recorded_in_current_segment = 0;
			}
			tme = device->fix->time;
			tp = gmtime(&tme);
			g_snprintf(string_buf, 256, "<trkpt lat=\"%f\" lon=\"%f\">\n",
					device->fix->latitude,
					device->fix->longitude);
			if (self->has_comma_as_decimal_point) {
			    for (i = 0; i < sizeof(string_buf); i++) {
				if (string_buf[i] == ',')
					string_buf[i] = '.';
			    }
			}
			g_fprintf(self->file, string_buf);
			g_fprintf(self->file, "  <time>%d-%02d-%02dT%02d:%02d:%02dZ</time>\n",
					tp->tm_year + 1900,
					tp->tm_mon + 1,
					tp->tm_mday,
					tp->tm_hour,
					tp->tm_min,
					tp->tm_sec);
			if (device->fix->fields & LOCATION_GPS_DEVICE_ALTITUDE_SET) {
			    g_snprintf(string_buf, 256, "  <ele>%.1f</ele>\n",
						device->fix->altitude);
			    if (self->has_comma_as_decimal_point) {
				for (i=0; i<sizeof(string_buf); i++) {
					if (string_buf[i] == ',')
						string_buf[i] = '.';
				}
			    }
			    g_fprintf(self->file, string_buf);
			}
			g_fprintf(self->file, "  <sat>%d</sat>\n",
					device->satellites_in_use);
			g_fprintf(self->file, "</trkpt>\n");
			self->points_recorded_in_current_segment += 1;
		}
#ifndef __arm__
                if (device->fix->fields & LOCATION_GPS_DEVICE_TIME_SET)
                        g_printf("%s:%s: time = %f\n",
                                        __FILE__, __func__,
                                        device->fix->time);

                if (device->fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET)
                        g_printf("%s:%s: lat = %f, long = %f\n",
                                        __FILE__,
                                        __func__,
                                        device->fix->latitude,
                                        device->fix->longitude);

                if (device->fix->fields & LOCATION_GPS_DEVICE_ALTITUDE_SET)
                        g_printf("%s:%s: alt = %f\n",
                                        __FILE__, __func__,
                                        device->fix->altitude);

                if (device->fix->fields & LOCATION_GPS_DEVICE_SPEED_SET)
                        g_printf("%s:%s: speed = %f\n",
                                        __FILE__, __func__,
                                        device->fix->speed);

                if (device->fix->fields & LOCATION_GPS_DEVICE_TRACK_SET)
                        g_printf("%s:%s: track = %f\n",
                                        __FILE__, __func__,
                                        device->fix->track);
                if (device->fix->fields & LOCATION_GPS_DEVICE_CLIMB_SET)
                        g_printf( "%s:%s: climb = %f\n",
                                        __FILE__, __func__,
                                        device->fix->climb);

                g_printf("%s:%s Accuracy values:\n", __FILE__, __func__);
                g_printf("\tept = %e, eph = %e, epv = %e, epd = %e, "
                                "eps = %e, epc = %e\n",
                                device->fix->ept,
                                device->fix->eph,
                                device->fix->epv,
                                device->fix->epd,
                                device->fix->eps,
                                device->fix->epc);
        }

        g_printf("%s:%s: Satellites in view: %d\n",
                        __FILE__, __func__,
                        device->satellites_in_view);

        g_printf("%s:%s: Satellites in use: %d\n", __FILE__, __func__,
                        device->satellites_in_use);

        if (device && device->cell_info) {
                if (device->cell_info->flags & LOCATION_CELL_INFO_GSM_CELL_INFO_SET)
                        g_printf( "Mobile Coutry Code GSM: %d\n",
                                        device->cell_info->gsm_cell_info.mcc);

                if (device->cell_info->flags & LOCATION_CELL_INFO_WCDMA_CELL_INFO_SET)
                        g_printf( "Mobile Coutry Code WCDMA: %d\n",
                                        device->cell_info->wcdma_cell_info.mcc);
#endif
        }
}


GtkWidget *
custom_button_new (int padding, char * icon_name)
{
    GtkIconTheme *icon_theme;
    GdkPixbuf *icon;
    GtkWidget *icon_image, *button;

    icon_theme = gtk_icon_theme_get_default ();
    icon = gtk_icon_theme_load_icon (icon_theme, icon_name, 48, 0, NULL);
    if (icon == NULL)
	icon = gtk_icon_theme_load_icon (icon_theme,
					 "qgn_list_gene_default_app",
					 48, 0, NULL);

    icon_image = gtk_image_new_from_pixbuf (icon);
    gtk_misc_set_padding (GTK_MISC (icon_image), padding, padding);
    g_object_unref (G_OBJECT (icon));
    button = gtk_button_new ();
    gtk_container_add (GTK_CONTAINER (button), icon_image);

    gtk_widget_show_all (button);

    return button;
}

GtkWidget *
custom_image_new (char * icon_name)
{
    GtkIconTheme *icon_theme;
    GdkPixbuf *icon;
    GtkWidget *icon_image;

    icon_theme = gtk_icon_theme_get_default ();
    icon = gtk_icon_theme_load_icon (icon_theme, icon_name, 48, 0, NULL);
    if (icon == NULL)
	icon = gtk_icon_theme_load_icon (icon_theme,
					 "qgn_list_gene_default_app",
					 48, 0, NULL);

    icon_image = gtk_image_new_from_pixbuf (icon);

    return icon_image;
}

static void
gps_data_logger_widget_init (GpsDataLoggerWidget * self)
{
    GtkWidget *vbox, *hbox_status, *hbox_buttons;
    GtkWidget *button_rec, *button_stop, *status_icon, *status_text;

    self->file = NULL;
    
    button_rec = gtk_button_new();
    button_stop = custom_button_new (PADDING, "camera_video_stop");
    status_icon = gtk_image_new();
    
    status_text = gtk_label_new("Status: ");

    vbox = gtk_vbox_new (FALSE, 0);
    hbox_status = gtk_hbox_new (FALSE, 2);
    hbox_buttons = gtk_hbox_new (FALSE, 2);
    gtk_box_pack_start (GTK_BOX (hbox_status), status_text, FALSE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hbox_status), status_icon, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox_buttons), button_rec, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox_buttons), button_stop, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), hbox_status, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), hbox_buttons, FALSE, FALSE, 0);
    
    self->gps_status = status_icon;
    self->button_rec = button_rec;
    self->button_stop = button_stop;
    self->app_state = APP_STATE_IDLE;
    
    update_status_and_buttons(self);
    
    g_signal_connect (self->button_rec, "clicked",
		      G_CALLBACK (handle_button_press), self);
    g_signal_connect (self->button_stop, "clicked",
		      G_CALLBACK (handle_button_press), self);

    gtk_widget_show_all (vbox);

    /* Set the resizing behavior */
    /*
    hd_home_widget_item_set_resize_type (HD_HOME_PLUGIN_ITEM (home_widget),
					 HD_HOME_PLUGIN_ITEM_RESIZE_BOTH);
    */
    gtk_container_add (GTK_CONTAINER (self), vbox);

    self->control = location_gpsd_control_get_default();

    self->settings = settings_load();
    apply_settings(self);

    self->device  = g_object_new(LOCATION_TYPE_GPS_DEVICE, NULL);

    g_signal_connect(self->control, "error", G_CALLBACK(on_error), self);
    g_signal_connect(self->control, "gpsd-running", G_CALLBACK(on_start),
                                                                   self);
    g_signal_connect(self->control, "gpsd-stopped", G_CALLBACK(on_stop),
                                                                   self);
    g_signal_connect(self->device,  "changed", G_CALLBACK(on_changed),
    		    self);

    /* widget settings */
    hd_home_plugin_item_set_settings (HD_HOME_PLUGIN_ITEM (self), TRUE);
    g_signal_connect (self, "show-settings", G_CALLBACK (settings_dialog_show),
		    self);

}

static void
gps_data_logger_widget_finalize (GObject *object)
{
    GpsDataLoggerWidget *self = GPS_DATA_LOGGER_WIDGET (object);
    g_object_unref(self->device);
    g_object_unref(self->control);
    if (self->settings_dialog) {
        gtk_widget_destroy (self->settings_dialog);
        self->settings_dialog = NULL;
    }
    settings_free(self->settings);
    g_free(self->settings);
		    
    /* Call the base class's implementation: */
    G_OBJECT_CLASS(gps_data_logger_widget_parent_class)->finalize (object);
}

static void
gps_data_logger_widget_class_init (GpsDataLoggerWidgetClass * klass)
{
    GObjectClass         *object_class;
    object_class = G_OBJECT_CLASS (klass);
    object_class->finalize = gps_data_logger_widget_finalize;
}

static void
gps_data_logger_widget_class_finalize (GpsDataLoggerWidgetClass * klass)
{
}

