/*
 * Copyright (C) 2005 Luca De Cicco <ldecicco@gmail.com>
 * Hildonized by Luca De Cicco
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "sudoku_worksheet.h"
#include "callbacks.h"
#include "main_window.h"
#include "actions.h"
#include "guesses.h"
#include "save_game.h"

#define FIXED_COLOR "red"
#define GUESS_COLOR "black"

#define GLADE_HOOKUP_OBJECT(component,widget,name) \
  g_object_set_data_full (G_OBJECT (component), name, \
    gtk_widget_ref (widget), (GDestroyNotify) gtk_widget_unref)

#define GLADE_HOOKUP_OBJECT_NO_REF(component,widget,name) \
  g_object_set_data (G_OBJECT (component), name, widget)

void 
draw_all_squares();
void 
draw_number(struct sudoku_big_square *bs, gint i , guint value, gboolean fixed);
void 
draw_guess(struct sudoku_big_square *bs, gint i, gchar *str);

	
GtkWidget *main_table;
GtkToolItem *nums[10];
GtkToolItem *guess_toggle;
GtkWidget *save_dlg;
GdkGC *white_gc , *black_gc ;

GtkWidget *window1;

struct sudoku_worksheet *ws;

/* Save point - number */
guint state = 0;


#ifdef IS_HILDON
static HildonProgram *app;
static HildonWindow *main_appview;
static GtkWidget *fullscreen_control;
#endif 


void
change_state(GtkButton *button, gpointer user_data)
{
	state++;
#ifdef IS_HILDON
	hildon_banner_show_information (GTK_WIDGET (main_appview) ,NULL, _("Check point saved."));
#endif 
}

void
undo(GtkButton *button, gpointer user_data)
{
	struct action *a;

	a = action_get_last(&ws->actions);
	
	if(a){
		if(a->n != state){
			state--;
		}
	
		//printf("Last action. cell: %d. old: %d new: %d\n",a->cell,a->old,a->new);
		ws->s_cells[a->cell]->cell = a->old;
		draw_all_squares();	
	}else{
		//printf("Nothing to undo");
	}
	g_free(a);
	ws->changed_since_saved = TRUE;
}

gchar *
interface_file_chooser (GtkFileChooserAction action)
{
	GtkWidget *dialog;
	gchar *filename = NULL;
	gchar *ts_str = timestamp_to_string();
	gchar *default_filename = g_strdup_printf("sudoku_%s.xml",ts_str);

#ifdef IS_HILDON	
  	dialog = hildon_file_chooser_dialog_new (GTK_WINDOW (window1), action);
 #else
        dialog = gtk_file_chooser_dialog_new (_("Open"), GTK_WINDOW (window1), 
                 action, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OK, GTK_RESPONSE_OK, NULL);
 #endif 	
	gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog),default_filename);
	g_free(ts_str);
	g_free(default_filename);
	gtk_widget_show_all (GTK_WIDGET (dialog));
	
	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK) {
		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
	}
	gtk_widget_destroy(dialog);
	return filename;
}

void 
generate_puzzle (GtkButton *button, gpointer user_data)
{
	if( decide_on_changes (ws) == FALSE)
	{
		return;
	}
	
	fill_worksheet_with_sensei(&ws);
	draw_all_squares();
#ifdef IS_HILDON
	hildon_banner_show_information (GTK_WIDGET (main_appview) ,NULL, _("Puzzle Generated"));
#else
       gpe_popup_infoprint (GDK_DISPLAY(), _("Puzzle Generated"));
#endif
	
	ws->changed_since_saved = TRUE;
	
}

void
savegame (GtkButton *button, gpointer user_data)
{

	GtkWidget *d;
	gchar *xml = worksheet_to_xml (ws);
	gchar *filename = interface_file_chooser(GTK_FILE_CHOOSER_ACTION_SAVE);
	if(!filename)
		return;
	
	FILE *fp = fopen(filename,"w");
	
	fprintf(fp, "%s",xml);
	g_free(xml);
	fclose(fp);
#ifdef IS_HILDON
	hildon_banner_show_information (GTK_WIDGET (main_appview) ,NULL, _("Game saved."));
#endif	
	ws->changed_since_saved = FALSE;
}


gboolean 
decide_on_changes(struct sudoku_worksheet *ws)
{
	if(ws->changed_since_saved)
	{
		const gchar question[]="There are unsaved changes. Are you sure you want to proceed?";
#ifdef IS_HILDON
		GtkWidget *note;
		guint i;
		note = hildon_note_new_confirmation (window1, _(question));

		i = gtk_dialog_run (GTK_DIALOG (note));
		gtk_widget_destroy (GTK_WIDGET (note));

		if (i == GTK_RESPONSE_OK)
		     return TRUE;
		else
		     return FALSE;


#else
		return gpe_question_ask (_(question), _("Question"), "question", "!gtk-no", NULL, "!gtk-yes", NULL, NULL);
#endif
	}		
}
void
loadgame (GtkButton *button, gpointer user_data)
{
	gchar *filename = NULL;
	if( decide_on_changes (ws) == FALSE)
	{
		return;
	}
		
	filename = interface_file_chooser (GTK_FILE_CHOOSER_ACTION_OPEN);

	if (filename == NULL) {
		return ;
	}
	fill_worksheet_from_xmlfile(filename, ws);
#ifdef IS_HILDON
	hildon_banner_show_information (GTK_WIDGET (main_appview) ,NULL, _("Game loaded."));
#else
        gpe_popup_infoprint (GDK_DISPLAY(), _("Game loaded."));
#endif
	
	ws->changed_since_saved = FALSE;
	
}
void 
goto_last_save_point(GtkButton *button, gpointer user_data)
{
	GSList *iter;
	gboolean undo_something=FALSE;
	do{
		struct action *a,last;
		
		iter = g_slist_last(ws->actions);
		if(!iter){
			break;
		}
		a = (struct action *)iter->data;
		
		if(a->n == state){
			
			ws->s_cells[a->cell]->cell = a->old;
			ws->actions = g_slist_delete_link(ws->actions,g_slist_last(ws->actions));
			undo_something = TRUE;
		}else{
			break;
		}
	}while(iter);
	
	actions_dump(ws->actions);
	draw_all_squares();	
	if(undo_something == TRUE && state != 0){
		state--;	
	}
#ifdef IS_HILDON
	hildon_banner_show_information (GTK_WIDGET (main_appview) ,NULL, _("Loaded last check point."));
#else
	gpe_popup_infoprint (GDK_DISPLAY(), _("Loaded last check point."));
#endif
		
	ws->changed_since_saved = TRUE;
}
	
gboolean
cell_pressed(gint selected)
{
	int i=0;
	
	gint sel_num;
	gchar *num;
	gint error_fl=0;
	gchar *error_msg;
	struct action a;
	
	error_msg = (gchar *)g_malloc(90);
	
	sel_num = get_selected_num();

	if(ws->s_cells[selected]->cell > 0 && sel_num != 0){
		error_msg = g_strdup(_("Cell not empty"));
		error_fl = 1;
	}else{	
	
		if(check_row(selected,sel_num) || 
		   check_column(selected,sel_num) ||
		   check_square(selected,sel_num))
		{
		   error_msg = g_strdup(_("This number does not fit here"));
		   error_fl = 1;
		}else {
		   if(ws->s_cells[selected]->fixed){
		      printf("This cell is fixed\n");
		      error_msg = g_strdup(_("This cell is fixed"));
		      error_fl = 1;
		   }
		}
	}
	 	
	if(error_fl){
#ifndef IS_HILDON
		gpe_popup_infoprint(GDK_DISPLAY(), error_msg);
#else
		hildon_banner_show_information (GTK_WIDGET (main_appview) ,NULL, error_msg);
#endif
		g_free (error_msg);
		return FALSE;	
	}
	
	/* No error */
	a.cell = selected;
	a.new = sel_num;
	a.old = ws->s_cells[selected]->cell;
	
	if(sel_num >0){
		gboolean del_flag = FALSE;
		ws->s_cells[selected]->cell = sel_num;
		del_flag |= del_guess_column(selected, sel_num);	
		del_flag |= del_guess_row(selected, sel_num);
		del_flag |= del_guess_square(selected, sel_num);
		ws->s_cells[selected]->guesses = NULL;
		if(del_flag){
			g_debug("I deleted something. Need to redraw all squares\n");
			draw_all_squares();
		}
		
	}else{
		ws->s_cells[selected]->cell = 0;
		ws->s_cells[selected]->guesses = NULL;
	}
	
	action_add(&(ws->actions),a);
	if (error_msg)
		g_free (error_msg);
	
	ws->changed_since_saved = TRUE;
	return TRUE;
}

/* Returns selected numer used to add new numbers to cell */
gint 
get_selected_num(void)
{
	int i;
	
	for(i = 0 ; i <= 9 ; i++)
	{
		if(gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(nums[i])))
		{
			g_debug("Selected %d button",i);	
			return(i);
		}
	}
	return 0;
}

guint 
cell_number_from_event(guint x, guint y, struct sudoku_big_square *bs)
{
	guint row,col,big_row, big_col,abs_row,abs_col,index;
	
	index = bs->index;
	
	/* Evaluate the cell number */
	row = y/(bs->height/3);
	col = x/(bs->width/3);
	
	big_row = index/3;
	big_col = index%3;
	
	abs_row = row + big_row * 3;
	abs_col = col + big_col * 3;
	
	return abs_row * 9 + abs_col;
}


void 
draw_big_square(struct sudoku_big_square *bs)
{
	
	GdkDrawable *d;
	gint i,index,big_row, big_col, abs_col,abs_row,row, col;
	
	d = bs->drawing->window;
  	index = bs->index;
	
	black_gc = bs->drawing->style->black_gc;
	white_gc = bs->drawing->style->white_gc;
	
	/* Background */
	gdk_draw_rectangle(d, white_gc, TRUE, 0, 0, bs->width, bs->height);
	/* Border */
	gdk_draw_rectangle(d, black_gc, FALSE, 0, 0, bs->width, bs->height);
	gdk_draw_rectangle(d, black_gc, FALSE, 1, 1, bs->width-2, bs->height-2);
	
	for(i = 0 ; i < 9 ; i++){
		GdkRectangle r = bs->squares[i];
		gchar *str;
		gdk_draw_rectangle(d, black_gc, FALSE, r.x, r.y, r.width, r.height);
		
		row = i/3;
		col = i%3;
		
		big_row = index/3;
		big_col = index%3;
	
		abs_row = row + big_row * 3;
		abs_col = col + big_col * 3;
		
		draw_number(bs,i,ws->s_cells[abs_row * 9 + abs_col]->cell, ws->s_cells[abs_row * 9 + abs_col]->fixed);
		str = guess_to_str(ws->s_cells[abs_row * 9 + abs_col]->guesses);
		
		draw_guess(bs, i , str);	
		
	}
}

void 
draw_guess(struct sudoku_big_square *bs, gint i, gchar *str)
{
	PangoLayout *pl;
	PangoRectangle pr;
	GtkWidget *widget ;
	GdkRectangle gr;
	gchar *buffer, *buf;
	widget = bs->drawing;
	
	buf = g_malloc(60);
	
	buffer = g_malloc(60);
	
	pl = gtk_widget_create_pango_layout (widget, NULL);
	if(i > 81){
		g_error("Something is going wrong");
	}
	gr = bs->squares[i];
	
	if(str){
		sprintf(buf,"<span size='%d'>%s</span>",300*gr.height,str);
	}else{
		sprintf(buf," ");
	}
	
	buffer = g_locale_to_utf8 (buf, -1, NULL, NULL, NULL);
	pango_layout_set_markup(pl, buffer, strlen (buffer));
	
	pango_layout_set_width(pl, gr.width*PANGO_SCALE);
	pango_layout_set_alignment(pl,PANGO_ALIGN_RIGHT);
	
	pango_layout_get_pixel_extents (pl, &pr, NULL);
	
    
    gtk_paint_layout (widget->style,
                widget->window,
                GTK_WIDGET_STATE (widget),
                FALSE,
                &gr,
                widget,
                "label",
                gr.x,gr.y,
                pl);

	g_free(buffer);
	g_free(buf);
	
}



void 
draw_number(struct sudoku_big_square *bs, gint i , guint value, gboolean fixed )
{
	PangoLayout *pl;
	PangoRectangle pr;
	GtkWidget *widget ;
	GdkRectangle gr;
	gchar *buffer, *buf;
	widget = bs->drawing;
	
	buf = g_malloc(128);
	buffer = g_malloc(128);
	
	pl = gtk_widget_create_pango_layout (widget, NULL);
	if(i > 81){
		g_error("Something is going wrong");
	}
	gr = bs->squares[i];
	
	if(value){
		sprintf(buf,"<span size='%d' foreground='%s'>%d</span>",680*gr.height, (fixed ? FIXED_COLOR : GUESS_COLOR), value);
	}else{
		sprintf(buf," ");
	}
	
	buffer = g_locale_to_utf8 (buf, -1, NULL, NULL, NULL);
	pango_layout_set_markup(pl, buffer, strlen (buffer));
	
	pango_layout_set_width(pl, gr.width*PANGO_SCALE);
	pango_layout_set_alignment(pl,PANGO_ALIGN_CENTER);
	
	pango_layout_get_pixel_extents (pl, &pr, NULL);
	
    
    gtk_paint_layout (widget->style,
                widget->window,
                GTK_WIDGET_STATE (widget),
                FALSE,
                &gr,
                widget,
                "label",
                gr.x,gr.y,
                pl);

	g_free(buffer);
	g_free(buf);
	
}


void 
draw_all_squares()
{
	int i;
	for(i = 0 ; i < 9 ; i++){ 
	 	draw_big_square(ws->bs[i]);
	}	
}

#ifdef IS_HILDON
static void
toggle_fullscreen (GtkCheckMenuItem *menuitem, gpointer user_data)
{
	GtkWindow *window = user_data;
	
	if (gtk_check_menu_item_get_active (menuitem))
		gtk_window_fullscreen (window);
	else
		gtk_window_unfullscreen (window);
}

static void
create_app_menu(HildonWindow *appview)
{
	GtkMenu *main_menu;
	GtkWidget *item_save, *item_undo, *item_restore, *item_fullscreen;
	GtkWidget *item_savegame , *item_loadgame, *item_gen, *item_close;
	
	main_menu = GTK_MENU( gtk_menu_new () );
	hildon_window_set_menu(appview, main_menu); 

	item_loadgame = gtk_menu_item_new_with_label(_("Load Game"));
	g_signal_connect(G_OBJECT(item_loadgame), "activate", loadgame, NULL);
	gtk_menu_append(main_menu, item_loadgame);

	item_savegame = gtk_menu_item_new_with_label(_("Save Game"));
	g_signal_connect(G_OBJECT(item_savegame), "activate", savegame, NULL);
	gtk_menu_append(main_menu, item_savegame);

	item_undo = gtk_menu_item_new_with_label(_("Undo"));
	g_signal_connect(G_OBJECT(item_undo), "activate", G_CALLBACK(undo), NULL);
	gtk_menu_append(main_menu, item_undo);
	
	item_save = gtk_menu_item_new_with_label(_("New check point"));
	g_signal_connect(G_OBJECT(item_save), "activate", G_CALLBACK(change_state), NULL);
	gtk_menu_append(main_menu, item_save);
	
	item_restore = gtk_menu_item_new_with_label(_("Goto last check point"));
	g_signal_connect(G_OBJECT(item_restore), "activate", G_CALLBACK(goto_last_save_point), NULL);
	gtk_menu_append(main_menu, item_restore);
	
	item_gen 	= gtk_menu_item_new_with_label(_("Generate Puzzle"));
	g_signal_connect(G_OBJECT(item_gen), "activate", G_CALLBACK(generate_puzzle), NULL);
	gtk_menu_append(main_menu, item_gen);
		
    item_fullscreen = gtk_check_menu_item_new_with_label(_("Fullscreen"));
	g_signal_connect(G_OBJECT(item_fullscreen), "activate", G_CALLBACK(toggle_fullscreen), appview);
	gtk_menu_append(main_menu, item_fullscreen);
	
    item_close = gtk_menu_item_new_with_label(_("Close"));
	g_signal_connect(G_OBJECT(item_close), "activate", G_CALLBACK(on_close), NULL);
	gtk_menu_append(main_menu, item_close);

	fullscreen_control = item_fullscreen;
	gtk_widget_show_all(main_menu);
}

static gboolean
window_key_press_event (GtkWidget *window, GdkEventKey *k, GtkWidget *data)
{
	switch (k->keyval)
	{
		case GDK_F6:
			gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (fullscreen_control), 
				!gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (fullscreen_control)));
		break;
	}
}

#endif

/* handle window resizing */
gboolean 
window_configure(GtkWindow *window, GdkEventConfigure *event, gpointer user_data)
{
  gint size;
  GtkWidget *toolbar = user_data;
  GtkRequisition req;
  gint free_height;

  if (event->type == GDK_CONFIGURE)
  {
	gtk_widget_size_request (toolbar, &req);
	free_height = event->height - req.height;
	size = (event->width > free_height) ? free_height : event->width;
#ifdef IS_HILDON
	size -= 26; /* for decoration */
#else
	size -= 1;
#endif
	sudoku_worksheet_resize (ws, size, size);
  }
  return FALSE;
}


GtkWidget*
create_window1 (void)
{

#ifdef IS_HILDON
 	osso_context_t *osso_context;
#endif
  	GtkWidget *vbox1;
  	GtkWidget *toolbar2;
  	GtkIconSize tmp_toolbar_icon_size;
  	GtkToolItem *button_save;
	GtkToolItem *last_save_point;
	GtkToolItem *button_undo;
	GtkToolItem *new_puzzle;
	GtkToolItem *separator;	
	GtkWidget *clear_image, *undo_image;
	
  	gint i;
	gint w, h, size;
  	GdkRectangle main_square;
   	
	
#ifdef IS_HILDON
	app = HILDON_PROGRAM ( hildon_program_get_instance());
	g_set_application_name ( _("Sudoku") );
	main_appview = HILDON_WINDOW (hildon_window_new());
	
	hildon_program_add_window( app, main_appview );


	window1 = main_appview;
	gtk_widget_show_all(window1);
	g_signal_connect (G_OBJECT (window1), "key-press-event", 
	                  G_CALLBACK (window_key_press_event), NULL);

#else
  	window1 = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  	gtk_window_set_default_size (GTK_WINDOW (window1), 240, 320);	
  	gtk_window_set_title (GTK_WINDOW (window1), _("Sudoku"));
#endif 
	gtk_widget_show (window1); 

#ifdef IS_HILDON	
	/* Initialize maemo application */
	osso_context = osso_initialize("Sudoku", VERSION, FALSE, NULL);
	
	/* Check that initialization was ok */
	if (osso_context == NULL)
	{
		return OSSO_ERROR;
	}
#endif
	    
	gtk_signal_connect (GTK_OBJECT (window1), "delete_event",
                        GTK_SIGNAL_FUNC(on_close),
                        NULL);


	black_gc = window1->style->black_gc ;
	white_gc = window1->style->white_gc ;
  	
	vbox1 = gtk_vbox_new (FALSE, 0);
  	gtk_widget_show (vbox1);
  	gtk_container_add (GTK_CONTAINER (window1), vbox1);

  	toolbar2 = gtk_toolbar_new ();
#ifdef IS_HILDON
	hildon_window_add_toolbar(main_appview, GTK_TOOLBAR(toolbar2));
#else
	gtk_box_pack_start (GTK_BOX (vbox1), toolbar2, FALSE, FALSE, 0);
#endif    
  	gtk_widget_show (toolbar2);
  	tmp_toolbar_icon_size = gtk_toolbar_get_icon_size (GTK_TOOLBAR (toolbar2));

  	for (i = 1 ; i<=9 ; i++) {
  	   gchar num_c[3];
	   sprintf(num_c,"%d",i);
	
           nums[i] = gtk_toggle_tool_button_new ();
           gtk_tool_button_set_label (GTK_TOOL_BUTTON (nums[i]), num_c);
           gtk_toolbar_insert (GTK_TOOLBAR (toolbar2), nums[i],-1);
           g_signal_connect(G_OBJECT(nums[i]), "clicked", G_CALLBACK(select_num), NULL);
           gtk_tool_item_set_homogeneous (nums[i], FALSE);
	}
	
	separator = gtk_separator_tool_item_new ();
	gtk_separator_tool_item_set_draw (separator, TRUE);
	gtk_toolbar_insert (GTK_TOOLBAR (toolbar2), separator, -1);
	
	nums[0] = gtk_toggle_tool_button_new_from_stock (GTK_STOCK_CLEAR);
	
	gtk_toolbar_insert (GTK_TOOLBAR(toolbar2), nums[0], -1);
	g_signal_connect(G_OBJECT(nums[0]), "clicked", G_CALLBACK(select_num), NULL);
        gtk_tool_item_set_homogeneous (nums[0], FALSE);
	
        guess_toggle = gtk_toggle_tool_button_new ();
        gtk_tool_button_set_label (GTK_TOOL_BUTTON (guess_toggle), "G");
        gtk_toolbar_insert(GTK_TOOLBAR(toolbar2), guess_toggle, -1);
        gtk_tool_item_set_homogeneous (guess_toggle, FALSE);
        
        new_puzzle = gtk_tool_button_new_from_stock(GTK_STOCK_NEW);
        gtk_toolbar_insert (GTK_TOOLBAR (toolbar2), new_puzzle, -1);
        g_signal_connect(G_OBJECT(new_puzzle), "clicked", G_CALLBACK(generate_puzzle), NULL);
       
#ifdef IS_HILDON
        create_app_menu(main_appview);
#else
        button_save = gtk_tool_button_new_from_stock (GTK_STOCK_SAVE);
        gtk_toolbar_insert (GTK_TOOLBAR (toolbar2), button_save, -1);
    
	g_signal_connect (G_OBJECT(button_save), "clicked",
				  G_CALLBACK (change_state),NULL);

        button_undo = gtk_tool_button_new_from_stock (GTK_STOCK_UNDO);
        gtk_toolbar_insert (GTK_TOOLBAR (toolbar2), button_undo, -1);

	g_signal_connect (GTK_OBJECT(button_undo), "clicked",
				  G_CALLBACK (undo), NULL);
  
        last_save_point = gtk_tool_button_new_from_stock (GTK_STOCK_GO_BACK);
        gtk_toolbar_insert (GTK_TOOLBAR (toolbar2), last_save_point, -1);

	g_signal_connect (GTK_OBJECT(last_save_point), "clicked",
				  G_CALLBACK (goto_last_save_point),NULL);
#endif
	  
	
/*
	g_signal_connect ( GTK_OBJECT(guess_toggle), "clicked",
				G_CALLBACK (toggle_guess), NULL);
*/	
#ifdef IS_HILDON
       button_undo = gtk_tool_button_new_from_stock (GTK_STOCK_UNDO);
       gtk_widget_show (button_undo);
       gtk_toolbar_insert (GTK_TOOLBAR (toolbar2), button_undo, -1);
       g_signal_connect (GTK_OBJECT(button_undo), "clicked",
                                G_CALLBACK (undo), NULL);
#endif 
  	main_table = gtk_table_new (3, 3, FALSE);
  	gtk_widget_show (main_table);
  	gtk_box_pack_start (GTK_BOX (vbox1), main_table, TRUE, TRUE, 0);

#ifdef IS_HILDON
	size = 340;
#else
	gtk_window_get_size (GTK_WINDOW (window1), &w, &h);
	
	size = MIN (w,h) - 10;	
#endif
        ws = sudoku_worksheet_new (size, size, main_table);
 
  	draw_all_squares();
	generate_puzzle(NULL,NULL);	
        g_signal_connect (G_OBJECT (window1), "configure-event", 
                          G_CALLBACK (window_configure), toolbar2);
        gtk_widget_show_all (window1);
       
	return window1;
}
