/*
 * This file is part of goban770
 *
 * Copyright (C) 2006,2007 Jarmo Ahosola.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <stdio.h>
#include <gtk/gtk.h>
#include <ui/interface.h>
#include <utils/log.h>
#include <utils/sgf.h>
#include <game/goban.h>
#include <game/gametree.h>
#include <game/notes.h>
#include <stdlib.h>

static gboolean sgf_h_requestPL = FALSE; /* Does player in turn need to be determined for previous position */
static gboolean sgf_h_needToSaveTrash = FALSE;
static GtkTextBuffer * sgf_h_Buffer = NULL;
static GtkTextIter sgf_h_Iter;
static gchar * sgf_h_in;
static gchar * sgf_h_stuckAt = NULL;
static gchar * sgf_h_eof = NULL;
static SgfTerminate sgf_h_eject = SGFTERM_OK;
static gboolean sgf_h_root_semicolon_read = FALSE;

static gchar * supportedSGF[] =
{
  "B", /* Black play */
  "W", /* White play */
  "GM", /* Game */
  "FF", /* File format */
  "ST", /* Style */
  "SZ", /* Size */
  "KM", /* Komi */
  "C", /* Comment */
  "CR", /* Circle */
  "TR", /* Triangle */
  "SQ", /* Square */
  "MA", /* Cross */
  "M", /* Old SGF mark */
  "HA", /* Handicap */
  "AB", /* Add black stones */
  "AW", /* Add white stones */
  "L", /* Backwards compatbility, label */
  "LB", /* Label */
  "GN", /* Game name */
  "BR", /* Black rank */
  "EV", /* Event */
  "PB", /* Black player name */
  "PC", /* Place where game was played */
  "PW", /* White player name */
  "RE", /* Result of the game */
  "WR", /* White rank */
  "TB", /* Black territory */
  "TW", /* White territory */
  "V",  /* Estimated score */
  "PL",  /* Player in turn */
  "MULTIGOBM", /* MultiGo bookmark, used as home */
  "DI", /* Goproblems.com problem difficulty */
  "GE", /* Goproblems.com problem classification */
  "AE" /* Add empty at setup position */
};
#define SUPPORTED_SGF_SYMBOLS 34
#define MINIMAL_SGF_SYMBOLS 7
gboolean HasProblem()
{
  static int tryCount = 0;
  
  if(sgf_h_in >= sgf_h_eof) {
    sgf_h_eject = SGFTERM_EXCEED_EOF;
    return TRUE;
  }
  if(sgf_h_stuckAt != sgf_h_in) {
    sgf_h_stuckAt = sgf_h_in;
    tryCount = 0;
  }
  tryCount++;
  if(tryCount >= 10000) {
    sgf_h_eject = SGFTERM_INFINITE_LOOP;
    return TRUE;
  }
  return FALSE;
}

/* SGF file reading */

gchar GetNextChar() /* Get next sgf character skipping all invalid characters including white spaces */
{
  gint peek = 0;
  gchar result;
  do {
    if(sgf_h_in[0] >= 'a' && sgf_h_in[0] <= 'z') {
      break;
    }
    if(sgf_h_in[0] >= 'A' && sgf_h_in[0] <= 'Z') {
      break;
    }
    if(sgf_h_in[0] >= '0' && sgf_h_in[0] <= '9') {
      break;
    }
    if(sgf_h_in[0] == '(') {
      break;
    }
    if(sgf_h_in[0] == ')') {
      break;
    }
    if(sgf_h_in[0] == ':') {
      break;
    }
    if(sgf_h_in[0] == ']') {
      break;
    }
    if(sgf_h_in[0] == '-') {
      break;
    }
    if(sgf_h_in[0] == '+') {
      break;
    }
    if(sgf_h_in[0] == ',') {
      break;
    }
    if(sgf_h_in[0] == '.') {
      break;
    }
    if(sgf_h_in[0] == ';') {
      break;
    }
    if(sgf_h_in[0] == '[') {
      if(sgf_h_in[1] == ']') {
        peek = 2;
        while((sgf_h_in[peek] != '[') && ((sgf_h_in[peek] < 'A') || (sgf_h_in[peek] > 'Z')) && (&sgf_h_in[peek] < sgf_h_eof)) {
          peek++;
        }
        if(sgf_h_in[peek] == '[') {
          sgf_h_in += peek;
          sgf_h_in--;
        }
        else {
          break;
        }
      }
      else {
        break;
      }
    }
    sgf_h_in++;
  } while (sgf_h_in < sgf_h_eof);
  result = sgf_h_in[0];
  sgf_h_in++;
  return result;
}

/* Check that sqf is valid and measures some statistics from it */
gboolean IsValidSgf(gchar * readIn, gchar * theEnd, gint * nodeCount, gint * paramCount, gint * branchCount, gint * charCount)
{
  gint paranthesisCount = 0;
  gboolean semicolonFound = TRUE;
  gboolean insideParameterBlock = FALSE;
  nodeCount[0] = 0;
  paramCount[0] = 0;
  branchCount[0] = 0;
  charCount[0] = 0;
  while(theEnd > readIn) {
    if(readIn[0] == ')') {
	   if(paranthesisCount <= 0) {
#ifndef PRODUCTION_VERSION  
        msg2log("sgf log error: Unexpected )", LOGLEVEL_ERROR, LOGSCOPE_FORCED);
#endif
	     return FALSE;
	   }
	   if(semicolonFound == FALSE) {
#ifndef PRODUCTION_VERSION  
        msg2log("sgf log error: Missing ; between ( and )", LOGLEVEL_ERROR, LOGSCOPE_FORCED);
#endif
	     return FALSE;
	   }
	   paranthesisCount--;
	 }
	 if(readIn[0] == '(') {
	   branchCount[0]++;
	   if(semicolonFound == FALSE) {
#ifndef PRODUCTION_VERSION  
        msg2log("sgf log error: Missing ; between ( and (", LOGLEVEL_ERROR, LOGSCOPE_FORCED);
#endif
	     return FALSE;
	   }
	   paranthesisCount++;
	   semicolonFound = FALSE;
	 }
	 if(readIn[0] == '[') {
	   paramCount[0]++;
	   insideParameterBlock = TRUE;
	   while(theEnd > readIn) {
	     readIn++;
        charCount[0]++;
		  while(readIn[0] == '\\') {
		    readIn += 2;
          charCount[0] += 2;
		  }
		  if(readIn[0] == ']') {
		    insideParameterBlock = FALSE;
		    break;
		  }
	   }
	 }
	 if(readIn[0] == ';') {
	   semicolonFound = TRUE;
	   nodeCount[0]++;
	 }
    readIn++;
    charCount[0]++;
  }
  if(insideParameterBlock == TRUE) {
#ifndef PRODUCTION_VERSION  
    msg2log("sgf log error: Missing ]", LOGLEVEL_ERROR, LOGSCOPE_FORCED);
#endif
    return FALSE;
  }
  if(paranthesisCount > 0) {
#ifndef PRODUCTION_VERSION  
    msg2log("sgf log error: Missing )", LOGLEVEL_ERROR, LOGSCOPE_FORCED);
#endif
    return FALSE;
  }
#ifndef PRODUCTION_VERSION  
  msg2log("Sgf syntax check passed", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
  return TRUE;
}

/* Skip sgf token and it's parameters and returns number of characters skipped */
gint SkipSgfToken()
{
  gint result = 0;
  gboolean nextTokenFound = FALSE;
  if(HasProblem() == TRUE) return  result; /* loop preventation */
  while(sgf_h_in[result] != 0 && nextTokenFound == FALSE){ /* Until we find end of file or start of next token */
    while(sgf_h_in[result] == '\\'){ /* we skip escaped charatcers */
	   result+=2;
#ifndef PRODUCTION_VERSION  
        msg2log("Read escaped character", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
      if(HasProblem() == TRUE) return result; /* loop preventation */
	}
	if((sgf_h_in[result] == ']') && (sgf_h_in[result+1] != '[')) { /* If token parameter end mark is found and it is not followed by token parameter start mark, then next character starts new token */
	  nextTokenFound = TRUE;
	}
    result++;
    if(HasProblem() == TRUE) return result; /* loop preventation */
  }
  sgf_h_in = &sgf_h_in[result]; /* Skip result amount of characters */
  return result;
}

gint ReadGobanSize(gchar * where, gchar * upto)
{
  gint result = 0;
  gint semicolons = 0;
  while((semicolons < 2) && (where < upto)) {
    if(where[0] == ';') {
      semicolons++;
    }
    if(where[0] == 'S' && where[1] == 'Z' && where[2] == '[') {
      result = where[3] - '0';
      if(where[4] != ']') {
        result *= 10;
        result += where[4] - '0';
      }
      return result;
    }
    where++;
  }
  return 19;
}

SgfToken ReadSgfToken(Node * trashcanNode, MainView * main)
{
  GtkTextIter trashIter;
  gint copyAmount = 0, c = 0, s = 0, supportedSymbolsLimit = 0;
  gchar * copyFrom = NULL;
  /* gchar inRead; */
  if(HasProblem() == TRUE) return ; /* loop preventation */
  while(TRUE){ /* Search until something is identified */
    /* Skip formatting symbols */
    while(sgf_h_in[0] == '\n' || 
          sgf_h_in[0] == '\r' || 
          sgf_h_in[0] == '\t' || 
          sgf_h_in[0] == '\b' || 
          sgf_h_in[0] == ' ') {
      sgf_h_in++;
      if(HasProblem() == TRUE) return; /* loop preventation */
    }
    /* Try to identify language separators and terminators */
    /* inRead = GetNextChar(); */
    switch (sgf_h_in[0])
    {
      case ';':
        sgf_h_in++;
#ifndef PRODUCTION_VERSION  
        msg2log("Read ;", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
        if(sgf_h_root_semicolon_read == FALSE) { /* If semicolon is the first one */
          sgf_h_root_semicolon_read = TRUE; /* mark it found */
          return ReadSgfToken(trashcanNode, main); /* and ignore it */
        }
        sgf_h_needToSaveTrash = TRUE; /* After new node up to branch end or new branch all syntax has to be saved */
	     return SGF_SEMICOLON;
      case '(':
        sgf_h_in++;
#ifndef PRODUCTION_VERSION  
        msg2log("Read (", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
        sgf_h_needToSaveTrash = FALSE; /* Between ( and first ; is no valid sgf syntax */
	     return SGF_OPEN_BRACKET;
      case ')':
        sgf_h_in++;
#ifndef PRODUCTION_VERSION  
        msg2log("Read )", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
        sgf_h_needToSaveTrash = FALSE; /* After ) but before either ; or ( or ) is no valid sgf syntax */
	     return SGF_CLOSE_BRACKET;
      case 0:
        sgf_h_in++;
#ifndef PRODUCTION_VERSION  
        msg2log("Read EOF", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     return SGF_EOF;
	   case '[':
	     sgf_h_in++;
#ifndef PRODUCTION_VERSION  
        msg2log("Read unexpected [", LOGLEVEL_WARNING, LOGSCOPE_SGF);
#endif
	     return SGF_OPEN_PARAMETER;
    }
	 /* Try to identify language symbols */
	 if(main->minimalload == TRUE) {
	   supportedSymbolsLimit = MINIMAL_SGF_SYMBOLS;
	 }
	 else {
	   supportedSymbolsLimit = SUPPORTED_SGF_SYMBOLS;
	 }
	 if(main->tryLoad > 0) {
	   supportedSymbolsLimit = main->tryLoad;
	 }
	 for(s = 0; s < supportedSymbolsLimit; s++){
	   c = 0;
	   while(sgf_h_in[c] != '[' && sgf_h_in[c] != 0 && sgf_h_in[c] == supportedSGF[s][c]){
	     c++;
        if(HasProblem() == TRUE) return; /* loop preventation */
	   }
	   if(sgf_h_in[c] == '[') { /* Supported symbol found */
	     sgf_h_in = &sgf_h_in[c]; /* Skip the symbol */
#ifndef PRODUCTION_VERSION  
        msg2logGint("Read language symbol %d", LOGLEVEL_INFO, LOGSCOPE_SGF, s);
#endif
		  return s; /* Returns the index of supported symbol */
	   }
	   if(sgf_h_in[c] == 0) { /* End of file found */
#ifndef PRODUCTION_VERSION  
        msg2log("Read EOF", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     return SGF_EOF;
	   }
      if(HasProblem() == TRUE) return; /* loop preventation */
	 }
	 /* If symbol is not identified, it is not supported */
	 copyFrom = sgf_h_in; /* Were from we need to take unsupported sgf content */
	 copyAmount = SkipSgfToken(); /* How many charatcers we need to take */
	 if(trashcanNode != NULL) { /* If we have a node where to store unidentified syntax */
	   if(trashcanNode->unsupportedSgf == NULL) { /* If we do not yet have trashcan space reserved, we reserve it */
	     trashcanNode->unsupportedSgf = gtk_text_buffer_new(NULL);
	   }
	   if(trashcanNode->unsupportedSgf != NULL) { /* In case we have trashcan space, */ 
	     gtk_text_buffer_get_end_iter(trashcanNode->unsupportedSgf,&trashIter); /* we append */
	     gtk_text_buffer_insert(trashcanNode->unsupportedSgf,&trashIter,copyFrom,copyAmount); /* the unsupported syntax there */
	   }
	 }
	 else { /* If we do not have the node, we need to store the trash at root */
	   if(main->unsupportedSgf == NULL) { /* If we do not yet have trashcan space reserved, we reserve it */
	     main->unsupportedSgf = gtk_text_buffer_new(NULL);
	   }
	   if(main->unsupportedSgf != NULL) { /* In case we have trashcan space, */ 
	     gtk_text_buffer_get_end_iter(main->unsupportedSgf,&trashIter); /* we append */
	     gtk_text_buffer_insert(main->unsupportedSgf,&trashIter,copyFrom,copyAmount); /* the unsupported syntax there */
	   }
	 }
    if(HasProblem() == TRUE) return; /* loop preventation */
  }
}

void ParseSgfGame(MainView * main, gchar * unparsed, gchar * endOfFile)
{
  gint i = 0, any = 0, leafs = 0;
  
  sgf_h_needToSaveTrash = FALSE;
  sgf_h_stuckAt = unparsed;
  sgf_h_eof = endOfFile;
  sgf_h_in = unparsed;
  sgf_h_eject = SGFTERM_OK;
  main->isGoProblem = FALSE;
  sgf_h_requestPL = FALSE;
  if(main->tryLoad > 0) {
    msg2log("Supported sgf-syntax:", LOGLEVEL_INFO, LOGSCOPE_SGF);
    for(i=0; i<main->tryLoad; i++) {
      msg2log(supportedSGF[i], LOGLEVEL_INFO, LOGSCOPE_SGF);
    }
    msg2log("Next unupported sgf-syntax:", LOGLEVEL_ERROR, LOGSCOPE_SGF);
    msg2log(supportedSGF[main->tryLoad], LOGLEVEL_ERROR, LOGSCOPE_SGF);    
  }
  while(ReadSgfToken(NULL, main) != SGF_OPEN_BRACKET); /* Eat starting ( of a game to ensure that several games of a collection are not mixed with each other */
  sgf_h_root_semicolon_read = FALSE;
  /* while(ReadSgfToken(NULL, main) != SGF_SEMICOLON); Eat root node ; symbol to ensure that node is not reserved for the root */
  ParseSgf(main, NULL);
#ifndef PRODUCTION_VERSION  
  switch(sgf_h_eject) {
    case SGFTERM_OK:
      msg2log("Parsing completed successfully", LOGLEVEL_INFO, LOGSCOPE_SGF);
      break;
    case SGFTERM_INFINITE_LOOP:
      msg2log("Parsing terminated to infinite loop", LOGLEVEL_ERROR, LOGSCOPE_SGF);
      break;
    case SGFTERM_EXCEED_EOF:
      msg2log("Parsing terminated at unexpected file end", LOGLEVEL_ERROR, LOGSCOPE_SGF);
      break;
    case SGFTERM_OUT_OF_MEMORY:
      msg2log("Parsing terminated to memory problems", LOGLEVEL_ERROR, LOGSCOPE_SGF);
      break;
  }
#endif
  MakeTreeLayout(main->child);
  GetTreeMeasures(&any, &any, &any, &leafs);
  if(leafs > BIG_TREE_LIMIT) {
    main->isBigTree = TRUE;
    main->gridRestore = main->gridSize;
    main->gridSize = BIGTREE_SCALE;
  }
  else {
    main->isBigTree = FALSE;
  }
  if(main->isGoProblem == TRUE) {
    if(main->rndOrientation == TRUE) {
      /* MakeTreeLayout(main->child); This is needed as RotateGoban expects the tree measures to be in place */
      RotateGoban(main, rand()%8);
    }
    if(main->sgfPL == STONE_NONE) {
      main->currentNode = main->child; /* Open problem at the position where player is expected to make the move */
      while(main->currentNode != NULL) {
        if(main->currentNode->child == NULL) {
          break;
        }
        if(main->currentNode->child->at != AT_IS_SETUP) {        
          break;
        }
        main->currentNode = main->currentNode->child;
      }
      main->compareNode = main->currentNode; /* Set home to there as well, so that user can jump at the begining of the problem */
    }
    if(main->problemTraining == TRUE) {
      gtk_menu_item_activate(GTK_MENU_ITEM(main->training_item));
      /* main->CurrentMouseMode = main->training_item; */
    }
    if(main->problem2step == TRUE) {
      main->step_2 = TRUE;
      gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(main->step_2_option_tb), TRUE);
      main->currentNode = main->compareNode;
    }
    if(main->sgfC != NULL) {
      UserMessage(GTK_WIDGET(main->goban_view), NULL, main->sgfC);
    }
    if(main->compareNode != NULL) {
      if(main->compareNode->sgfC != NULL) {
        UserMessage(GTK_WIDGET(main->goban_view), NULL, main->compareNode->sgfC);
      }
    }
  }
}

gint ReadReal2DoubleGint()
{
  gint sign = 1;
  gint result = 0;
  sgf_h_in++;
  if(sgf_h_in[0] == '-') {
    sign = -1;
    sgf_h_in++;
  }
  if(sgf_h_in[0] == '+') {
    sign = 1;
    sgf_h_in++;
  }
  while (sgf_h_in[0] != ']' && sgf_h_in[0] != 0 && sgf_h_in[0] != '.' && sgf_h_in[0] != ',') {
    result = result * 10;
    result += sgf_h_in[0] - '0';
    sgf_h_in++;
  }
  result *= 2;
  if(sgf_h_in[0] == '.' || sgf_h_in[0] == ',') {
    sgf_h_in++;
    while (sgf_h_in[0] != ']' && sgf_h_in[0] != 0) {
      if((sgf_h_in[0] > '0') && (sgf_h_in[0] <= '9')) {
        result |= 1;
      }
      sgf_h_in++;
    }
  }
  result *= sign;
  sgf_h_in++; /* Skip ] after real */
#ifndef PRODUCTION_VERSION  
      if((result*sign)%2 == 1) {
        msg2logGint("Read value %d.5", LOGLEVEL_INFO, LOGSCOPE_SGF, result/2);
      }
      else {
        msg2logGint("Read value %d.0", LOGLEVEL_INFO, LOGSCOPE_SGF, result/2);
      }
#endif
  return result;
}

void ParseSgf(MainView * main, Node * node)
{
  SgfToken token = SGF_EOF;
  Node * bigSister = NULL;
  Node * sister = NULL;

  if(HasProblem() == TRUE) return; /* loop preventation */
  token = ReadSgfToken(node, main);
  while(token != SGF_EOF) {
#ifndef PRODUCTION_VERSION  
      if(node == NULL) {
        msg2log("Node is NULL", LOGLEVEL_WARNING, LOGSCOPE_SGF);
      }
#endif
    switch (token)
    {
	   case SGF_OPEN_BRACKET: /* Start of branch */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing (", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     ParseSgf(main, node); /* Remember at stack the current node while parsing content of brackets */
	     break;
	   case SGF_CLOSE_BRACKET: /* End of branch */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing )", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
		  sgf_h_requestPL = FALSE; /* Cancel request to correct player in turn */
	     return; /* Take from stack the node stored there with open bracket */
	   case SGF_SEMICOLON: /* Start of node and end of previous node */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing ;", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
	       if(main->child == NULL) {
		      node = NewNode(NULL,STONE_NONE,NULL,AT_IS_UNDEFINED,-1); /* parent, big sister */
		      main->child = node;
		    }
		    else {
		      bigSister = main->child;
		      while(bigSister->sibling != main->child) {
		        bigSister = bigSister->sibling;
		      }
		      node = NewNode(NULL,STONE_NONE,bigSister,-1,-1); /* parent, big sister */
		    }
		    if(main->sgfPL == STONE_NONE && main->sgfAW != NULL) { /* If setup position does not have player in turn defined */
		      sgf_h_requestPL = TRUE; /* request correcting player in turn */
		    }
		  }
		  else {
		    if(node->at == AT_IS_SETUP && node->stone == STONE_NONE) { /* If setup position does not have player in turn defined */
		      sgf_h_requestPL = TRUE; /* request correcting player in turn */
		    }
		    if(node->at == AT_IS_UNDEFINED) { /* If node had no plays or set up stones */
		      if(main->omitEmptyNodes == TRUE) { /* and such nodes are wanted to be omitted */
		        break; /* then we keep using the node that exists instead of creating new one */
		      }
		    }
		    node = NewNode(node,STONE_NONE,get_bigsister(node->child),AT_IS_UNDEFINED,-1); /* parent, big sister */
		  }
		  break;
	   case SGF_B: /* Black play */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing B", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) { /* Sgf syntax is missing first semicolon. */
          sgf_h_root_semicolon_read = TRUE; /* Let's assume it was there */
          if(main->child == NULL) {
		      node = NewNode(NULL,STONE_NONE,NULL,AT_IS_SETUP,-1); /* parent, big sister */
		      main->child = node;
		    }
		    else {
		      bigSister = main->child;
		      while(bigSister->sibling != main->child) {
		        bigSister = bigSister->sibling;
		      }
		      node = NewNode(NULL,STONE_NONE,bigSister,-1,-1); /* parent, big sister */
		    }
		    if(node == NULL) {
		      sgf_h_eject = SGFTERM_OUT_OF_MEMORY;
		      return;
		    }
		  }
	     node->stone = STONE_BLACK;
		  node->at = ReadLocation();
		  sister = node->sibling; /* Get first sister */
		  while(sister != node) { /* and as long as there is any sister left */
		    if(sister->at == node->at) { /* if it is played at same position than this stone */
		      node_destroy(node); /* destroy this node */
		      node = sister; /* and use that sister instead */
		    }
		    else {
		      sister = sister->sibling; /* same for next sister */
		    }
		  }
		  if(sgf_h_requestPL == TRUE) { /* If player in turn for previous position is missing */
		    if(node->parent != NULL) { /* and it is not root */
		      if(node->parent->at == AT_IS_SETUP) { /* and it was setup position */
		        node->parent->stone = STONE_WHITE; /* set stone color played at that turn */
		        sgf_h_requestPL = FALSE; /* and clear the request for correcting PL */
		      }
		    }
		    else { /* if it was root */
		      if(main->sgfAW != NULL) { /* and it was setup position */
              main->sgfPL = STONE_BLACK; /* set player in turn */
		        sgf_h_requestPL = FALSE; /* and clear the request for correcting PL */
		      }
		    }
		  }
		  break;
	   case SGF_W: /* White play */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing W", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) { /* Sgf syntax is missing first semicolon. */
          sgf_h_root_semicolon_read = TRUE; /* Let's assume it was there */
          if(main->child == NULL) {
		      node = NewNode(NULL,STONE_NONE,NULL,AT_IS_SETUP,-1); /* parent, big sister */
		      main->child = node;
		    }
		    else {
		      bigSister = main->child;
		      while(bigSister->sibling != main->child) {
		        bigSister = bigSister->sibling;
		      }
		      node = NewNode(NULL,STONE_NONE,bigSister,-1,-1); /* parent, big sister */
		    }
		    if(node == NULL) {
		      sgf_h_eject = SGFTERM_OUT_OF_MEMORY;
		      return;
		    }
		  }
	     node->stone = STONE_WHITE;
		  node->at = ReadLocation();
		  sister = node->sibling; /* Get first sister */
		  while(sister != node) { /* and as long as there is any sister left */
		    if(sister->at == node->at) { /* if it is played at same position than this stone */
		      node_destroy(node); /* destroy this node */
		      node = sister; /* and use that sister instead */
		    }
		    else {
		      sister = sister->sibling; /* same for next sister */
		    }
		  }
		  if(sgf_h_requestPL == TRUE) { /* If player in turn for previous position is missing */
		    if(node->parent != NULL) { /* and it is not root */
		      if(node->parent->at == AT_IS_SETUP) { /* and it was setup position */
		        node->parent->stone = STONE_BLACK; /* set stone color played at that turn */
		        sgf_h_requestPL = FALSE; /* and clear the request for correcting PL */
		      }
		    }
		    else { /* if it was root */
		      if(main->sgfAW != NULL) { /* and it was setup position */
              main->sgfPL = STONE_WHITE; /* set player in turn */
		        sgf_h_requestPL = FALSE; /* and clear the request for correcting PL */
		      }
		    }
		  }
		  break;
	   case SGF_C: /* Comment. Sgf4 spec says that only one note may exist for each node. To simplyfy issues each note replaces any notes that existed in the same node. */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing C", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
		    if(main->sgfC != NULL) {
		      DestroyNote(main->sgfC);
		    }
		    main->sgfC = ReadNote();
		  }
		  else {
		    if(node->sgfC != NULL) {
	   	   DestroyNote(node->sgfC);
		    }
		    node->sgfC = ReadNote();
		  }
	     break;
	   case SGF_CR: /* Circle */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing CR", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
		    main->sgfCR = ReadPointlist(main->sgfCR);
		  }
		  else {
		    node->sgfCR = ReadPointlist(node->sgfCR);
		  }
		  break;
	   case SGF_TR: /* Triangle */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing TR", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
          main->sgfTR = ReadPointlist(main->sgfTR);
		  }
		  else {
          node->sgfTR = ReadPointlist(node->sgfTR);
		  }
		  break;
	   case SGF_SQ: /* Square */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing SQ", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
          main->sgfSQ = ReadPointlist(main->sgfSQ);
		  }
		  else {
          node->sgfSQ = ReadPointlist(node->sgfSQ);
		  }
		  break;
	   case SGF_MA: /* Cross */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing MA", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
          main->sgfMA = ReadPointlist(main->sgfMA);
		  }
		  else {
          node->sgfMA = ReadPointlist(node->sgfMA);
		  }
		  break;
	   case SGF_TB: /* Black territory */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing TB", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) { /* Incorrect syntax */
		    ReadPointlist(NULL); /* Need to be parsed to prevent problems */
		    break; /* ignored */
		  }
		  node->sgfTB = ReadPointlist(node->sgfTB);
		  break;
	   case SGF_TW: /* White territory */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing TW", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) { /* Incorrect syntax */
		    ReadPointlist(NULL); /* Need to be parsed to prevent problems */
		    break; /* ignored */
		  }
		  node->sgfTW = ReadPointlist(node->sgfTW);
		  break;
	   case SGF_M: /* Old sgf mark symbol */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing M", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
          main->sgfMA = ReadPointlist(main->sgfMA); /* Old mark is replaced by new sgf mark */
		  }
		  else {
          node->sgfMA = ReadPointlist(node->sgfMA); /* Old mark is replaced by new sgf mark */
		  }
		  break;
	   case SGF_GM: /* Game */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing GM", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(ReadGint() != 1) { /* Check that the file is about go */
#ifndef PRODUCTION_VERSION  
        msg2log("Game in sgf file is not go. Only go files supported.", LOGLEVEL_ERROR, LOGSCOPE_SGF);
#endif
		    return;
		  }
		  break;
	   case SGF_FF: /* File format */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing FF", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(ReadGint() > 4) { /* Check file format */
#ifndef PRODUCTION_VERSION  
        msg2log("File format is newer than supported.", LOGLEVEL_WARNING, LOGSCOPE_FORCED);
#endif
		  }
		  break;
	   case SGF_ST: /* View style */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing ST", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
        main->sgfST = ReadGint();
		  break;
	   case SGF_SZ: /* Goban size */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing SZ", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
        main->sgfSZ = ReadGint();
        if(main->sgfSZ > MAX_GOBANSIZE) {
#ifndef PRODUCTION_VERSION  
        msg2log("Goban size used in file is not supported.", LOGLEVEL_ERROR, LOGSCOPE_FORCED);
#endif
          main->sgfSZ = MAX_GOBANSIZE;
          return;
        }
		  break;
	   case SGF_HA: /* Handicap */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing HA", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
        main->sgfHA = ReadGint();
		  break;
	   case SGF_AB: /* Add black */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing AB", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
        if(node == NULL) {
          main->sgfAB = ReadPointlist(main->sgfAB);
        }
        else {
          node->sgfAB = ReadPointlist(node->sgfAB);
          if(node->at == AT_IS_UNDEFINED) {
            node->at = AT_IS_SETUP;
          }
        }
		  break;
	   case SGF_AW: /* Add white */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing AW", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
        if(node == NULL) {
          main->isGoProblem = TRUE;
          main->sgfAW = ReadPointlist(main->sgfAW);
        }
        else {
          if(node->nodeTurnNumber <= 2) {
            main->isGoProblem = TRUE;
          }
          node->sgfAW = ReadPointlist(node->sgfAW);
          if(node->at == AT_IS_UNDEFINED) {
            node->at = AT_IS_SETUP;
          }
        }
		  break;
	   case SGF_AE: /* Add empty */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing AE", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
        if(node == NULL) {
          ReadPointlist(NULL); /* Empty setup stones at startup position are meaningless, but such oddity need to be parsed */
        }
        else {
          node->sgfAE = ReadPointlist(node->sgfAE);
          if(node->at == AT_IS_UNDEFINED) {
            node->at = AT_IS_SETUP;
          }
        }
		  break;
	   case SGF_L: /* Label, backwards compatibility */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing L", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
          main->sgfLB = ReadOldStyleLabellist(main->sgfLB);
		  }
		  else {
          node->sgfLB = ReadOldStyleLabellist(node->sgfLB);
		  }
		  break;
	   case SGF_LB: /* Label */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing LB", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
          main->sgfLB = ReadLabellist(main->sgfLB);
		  }
		  else {
          node->sgfLB = ReadLabellist(node->sgfLB);
		  }
		  break;
	   case SGF_GN:
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing GN", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
		    if(main->sgfGN != NULL) {
		      DestroyNote(main->sgfGN);
		    }
		    main->sgfGN = ReadNote();
		  }
	     break;
	   case SGF_BR:
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing BR", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
		    if(main->sgfBR != NULL) {
		      DestroyNote(main->sgfBR);
		    }
		    main->sgfBR = ReadNote();
		  }
	     break;
	   case SGF_EV:
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing EV", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
		    if(main->sgfEV != NULL) {
		      DestroyNote(main->sgfEV);
		    }
		    main->sgfEV = ReadNote();
		  }
	     break;
	   case SGF_PB:
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing PB", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
		    if(main->sgfPB != NULL) {
		      DestroyNote(main->sgfPB);
		    }
		    main->sgfPB = ReadNote();
		  }
	     break;
	   case SGF_PC:
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing PC", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
		    if(main->sgfPC != NULL) {
		      DestroyNote(main->sgfPC);
		    }
		    main->sgfPC = ReadNote();
		  }
	     break;
	   case SGF_PW:
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing PW", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
		    if(main->sgfPW != NULL) {
		      DestroyNote(main->sgfPW);
		    }
		    main->sgfPW = ReadNote();
		  }
	     break;
	   case SGF_RE:
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing RE", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     /* if(node == NULL) { Let's assume one result for now and accept it both at the end and begining of the game */
		    if(main->sgfRE != NULL) {
		      DestroyNote(main->sgfRE);
		    }
		    main->sgfRE = ReadNote();
		  /* } */
	     break;
	   case SGF_WR:
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing WR", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
		    if(main->sgfWR != NULL) {
		      DestroyNote(main->sgfWR);
		    }
		    main->sgfWR = ReadNote();
		  }
	     break;
	   case SGF_KM:
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing KM", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
		    main->sgfKMx2 = ReadReal2DoubleGint();
		  }
	     break;
	   case SGF_V:
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing V", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node != NULL) {
		    node->sgfVx2 = ReadReal2DoubleGint();
		  }
	     break;
	   case SGF_PL: /* Set player in turn */
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing PL", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
	     if(node == NULL) {
		    main->sgfPL = ReadColor();
		  }
		  else {
		    if(node->stone == STONE_NONE) { /* If there is no color */
            node->stone = OPPONENT(ReadColor()); /* read setup color */
            node->at = AT_IS_SETUP; /* and indicate that position is setup */
          }
          else {
            ReadColor(); /* if we have color, let's ignore the PL tag instead */
          }
		  }
		  break;
	   case SGF_HOME:
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing MULTIGOBM", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
        main->compareNode = node; /* Set home position */
        main->currentNode = node; /* Open file at home position */
	     ReadNote(); /* Just skip [] to avoid problems */
	     break;
	   case SGF_DI:
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing DI", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
		  main->sgfDI = ReadNote();
	     break;
	   case SGF_GE:
#ifndef PRODUCTION_VERSION  
        msg2log("Parsing GE", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
		  main->sgfGE = ReadNote();
	     break;
		default:
		 break;
    }
    token = ReadSgfToken(node, main);
    if(HasProblem() == TRUE) return; /* loop preventation */
  }
}

/* Skip all symbols that may not be included in the note. Returns TRUE if there is still note content to show, FALSE if entire note has been processed */
gboolean FindPlainNote()
{
  if(HasProblem() == TRUE) return FALSE; /* loop preventation */
  if(sgf_h_in[0] == 0) {
    return FALSE; /* Unexpected end of file */
  }
  while(sgf_h_in[0] == ']' || sgf_h_in[0] == '\\') {
    if(sgf_h_in[0] == ']') {
	   sgf_h_in++;
	   return FALSE; /* We want to terminate at note ending ] */
	 }
    if(sgf_h_in[1] == ']' || sgf_h_in[1] == '\\') {
	   sgf_h_in++;
	   return TRUE; /* We want to show escaped ] and escaped \ */
	 }
	 sgf_h_in = &sgf_h_in[2]; /* Other escaped symbols we prefer to skip. Valid sgf4 includes only line end as such */
	 /* Line end may be composed from several symbols which of all need to be skipped. Here we assume that line ending \ as line continuation symbol is not used before several line feeds, so such are omitted */
    while(sgf_h_in[0] == '\n' || 
          sgf_h_in[0] == '\r' || 
          sgf_h_in[0] == '\t' || 
          sgf_h_in[0] == '\b') {
      sgf_h_in++;
      if(HasProblem() == TRUE) return FALSE; /* loop preventation */
    }
    if(HasProblem() == TRUE) return FALSE; /* loop preventation */
  }
  return TRUE;
}

/* Returns length of text to be included in the note and moves point of reading at next character of interest */
gint SkipPlainNote()
{
  gint textLength = 0;
  if(HasProblem() == TRUE) return textLength; /* loop preventation */
  do {
    textLength++;
	 sgf_h_in++;
    if(HasProblem() == TRUE) return textLength; /* loop preventation */
  } while (sgf_h_in[0] != 0 && sgf_h_in[0] != '\\' && sgf_h_in[0] != ']');
  return textLength;
}



gint ReadGint()
{
  gint result = 0;
  sgf_h_in++; /* Skip [ before gint */
  while (sgf_h_in[0] != ']' && sgf_h_in[0] != 0) {
    result = result * 10;
    result += sgf_h_in[0] - '0';
    sgf_h_in++;
  }
  sgf_h_in++; /* Skip ] after gint */
#ifndef PRODUCTION_VERSION  
        msg2logGint("Read value %d", LOGLEVEL_INFO, LOGSCOPE_SGF, result);
#endif
  return result;
}

gchar * ReadNote()
{
  gchar * note = NULL;
  
  sgf_h_in++; /* Eat [ before note */
  note = ExtractNoteFromSgf(sgf_h_in, sgf_h_eof);
  while((sgf_h_in[0] != ']' || sgf_h_in[1] == '[') && sgf_h_in < sgf_h_eof) { /* Find end of sequence of notes. This will skip illegal extra notes after legal note. */
    if(sgf_h_in[0] == '\\') {
      sgf_h_in++;
    }
    sgf_h_in++;
    if(HasProblem() == TRUE) return note; /* loop preventation */
  }
  sgf_h_in++;
  return note;
}

gint ReadLocation()
{
  gint result = 0;
  gint x = 0;
  gint y = 0;
  if(HasProblem() == TRUE) return result; /* loop preventation */
  x = sgf_h_in[1] - 'a';
  y = sgf_h_in[2] - 'a';
  if(sgf_h_in[1] == ']') {
    sgf_h_in = &sgf_h_in[2]; /* Skip [] */
    return AT_IS_PASS;
  }
  if((x == 't') && (y == 't')) {
    result = AT_IS_PASS;
  }
  else {
    result = XY2Location(x,y);
  }
  sgf_h_in = &sgf_h_in[4]; /* Skip [, x-coordinate, y-coordinate and ] */
#ifndef PRODUCTION_VERSION  
  msg2logGint("Read point %d", LOGLEVEL_INFO, LOGSCOPE_SGF, result);
#endif
  return result;
}

Stone ReadColor()
{
  Stone color = STONE_NONE;
  static gchar text[49] = "Color preset to no color, symbol ? interpreted\0"; 
#ifndef PRODUCTION_VERSION  
    text[33] = sgf_h_in[1];
    msg2log(text, LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
  if((sgf_h_in[1] == 'W') || (sgf_h_in[1] == 'w')) {
    color = STONE_WHITE;
#ifndef PRODUCTION_VERSION  
    msg2log("Read color white", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
  }
  if((sgf_h_in[1] == 'B') || (sgf_h_in[1] == 'b')) {
    color = STONE_BLACK;
#ifndef PRODUCTION_VERSION  
    msg2log("Read color black", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
  }
  sgf_h_in = &sgf_h_in[3]; /* Skip [, color and ] */
  return color;
}

Labellist * ReadOldStyleLabellist(Labellist * oldList)
{
  /* gchar label[2] = "\0\0"; */
  Labellist * goThrough = NULL;
  Labellist * result = NULL;
  Labellist * previous = oldList;
  gint at = -1;
  if(HasProblem() == TRUE) return result; /* loop preventation */
  result = oldList; 
  while(sgf_h_in[0] == '[') {
    at = ReadLocation();
    if(IsLabelInList(at, result) == TRUE) { /* If there is some label for the location already */
#ifndef PRODUCTION_VERSION  
      msg2log("Label ignored, double definition", LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
      continue; /* ignore any additional label definition */
    }
    result = NewLabel(at);
    result->next = previous;
    goThrough = previous;
    result->label[0] = 'A';
    result->label[1] = '\0';
    while(goThrough != NULL) {
      if(goThrough->label[0] >= result->label[0]) {
        result->label[0] = goThrough->label[0] + 1;
      }
      goThrough = goThrough->next;
    }
#ifndef PRODUCTION_VERSION  
    msg2log("Label is set to value:", LOGLEVEL_INFO, LOGSCOPE_SGF);
    msg2log(result->label, LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
    previous = result;
    if(HasProblem() == TRUE) return result; /* loop preventation */
  }
  return result;
}

Labellist * ReadLabellist(Labellist * oldList)
{
  gint index = 0;
  Labellist * result = NULL;
  Labellist * previous = oldList;
  if(HasProblem() == TRUE) return result; /* loop preventation */
  while(sgf_h_in[0] == '[') {
    result = NewLabel(ReadLocation());
    result->next = previous;
    index = 0;
    while(sgf_h_in[0] != ']') {
      if(index < MAX_LABEL_LENGTH) {
        result->label[index] = sgf_h_in[0];
      }
      index++;
      sgf_h_in = &sgf_h_in[1];
    if(HasProblem() == TRUE) return result; /* loop preventation */
    }
#ifndef PRODUCTION_VERSION  
    msg2log("Label is set to value:", LOGLEVEL_INFO, LOGSCOPE_SGF);
    msg2log(result->label, LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
    sgf_h_in = &sgf_h_in[1];
    previous = result;
    if(HasProblem() == TRUE) return result; /* loop preventation */
  }
  return result;
}

Pointlist * ReadPointlist(Pointlist * oldList)
{
  Pointlist * point = NULL;
  Pointlist * previous = NULL;
  Pointlist * firstPoint = NULL;
  gint fx = 0, fy = 0, tx = 0, ty = 0, swaptmp = 0, x = 0, y = 0;
  if(HasProblem() == TRUE) return firstPoint; /* loop preventation */
  if(sgf_h_in[0] != '[') {
    return oldList;
  }
  if(sgf_h_in[3] == ']') {
    point = NewPoint(ReadLocation());
	 firstPoint = point;
  }
  if(sgf_h_in[3] == ':') { /* Shortened sgf for area of points */
    fx = sgf_h_in[1] - 'a';
    fy = sgf_h_in[2] - 'a';
    tx = sgf_h_in[4] - 'a';
    ty = sgf_h_in[5] - 'a';
	 if(tx < fx) { /* Make sure fx and tx are in right order */
	   swaptmp = tx;
	   tx = fx;
	   fx = swaptmp;
	 }
	 if(ty < fy) { /* Make sure fy and ty are in right order */
	   swaptmp = ty;
	   ty = fy;
	   fy = swaptmp;
	 }
	 for(x = fx; x <= tx; x++) {
	   for(y = fy; y <= ty; y++) {
        point = NewPoint(XY2Location(x,y));
	     if(firstPoint == NULL) {
		    firstPoint = point;
		  }
		  else {
		    previous->next = point;
		  }
		  previous = point;
        if(HasProblem() == TRUE) return firstPoint; /* loop preventation */
	   }
      if(HasProblem() == TRUE) return firstPoint; /* loop preventation */
	 }
  }
  point->next = ReadPointlist(oldList);
  return firstPoint;
}

/* SGF file writing */

GtkTextBuffer * InitSgfHandling()
{
  if(sgf_h_Buffer == NULL) {
    sgf_h_Buffer = gtk_text_buffer_new(NULL); /* Memory release for this is missing. Don't know how it should be done */
  }
  gtk_text_buffer_set_text(sgf_h_Buffer,"",-1);
  gtk_text_buffer_get_end_iter(sgf_h_Buffer,&sgf_h_Iter);
  return sgf_h_Buffer;
}

void Append2File(gchar * text)
{
  gtk_text_buffer_insert(sgf_h_Buffer,&sgf_h_Iter,text,-1);
}

void Labellist2File(Labellist * llist)
{
  static gchar text[5] = "[  :\0"; 
  while(llist != NULL) {
    text[1] = 'a' + Location2X(llist->at);
    text[2] = 'a' + Location2Y(llist->at);
	 Append2File(text);
	 Append2File(llist->label);
	 Append2File("]\0");
    llist = llist->next;
  }
}

void Pointlist2File(Pointlist * point)
{
  static gchar text[5] = "[  ]\0"; 
  while(point != NULL) {
    text[1] = 'a' + Location2X(point->at);
    text[2] = 'a' + Location2Y(point->at);
	 Append2File(text);
    point = point->next;
  }
}

void Point2File(gint at)
{
  static gchar text[5] = "[  ]\0"; 
  text[1] = 'a' + Location2X(at);
  text[2] = 'a' + Location2Y(at);
  Append2File(text);
}

void Gint2File(gint value)
{
  static gchar text[10] = "";
  sprintf(text,"%d",value);
  Append2File(text);
}

void CreateSgfFile(MainView * main)
{
  gchar * note = NULL;

  Append2File("(;GM[1]FF[4]CA[UTF-8]AP[goban770:02]"); /* Game = go, file format = 4, character set = UTF-8, application = goban770, application version = 02 */
  if(main->sgfC != NULL) {
    Append2File("C[");
    note = AddSgfEscapes(main->sgfC);
    Append2File(note);
    DestroyNote(note);
    Append2File("]");
  }
  if(main->sgfGN != NULL) {
    Append2File("GN[");
    note = AddSgfEscapes(main->sgfGN);
    Append2File(note);
    DestroyNote(note);
    Append2File("]");
  }
  if(main->sgfBR != NULL) {
    Append2File("BR[");
    note = AddSgfEscapes(main->sgfBR);
    Append2File(note);
    DestroyNote(note);
    Append2File("]");
  }
  if(main->sgfEV != NULL) {
    Append2File("EV[");
    note = AddSgfEscapes(main->sgfEV);
    Append2File(note);
    DestroyNote(note);
    Append2File("]");
  }
  if(main->sgfPB != NULL) {
    Append2File("PB[");
    note = AddSgfEscapes(main->sgfPB);
    Append2File(note);
    DestroyNote(note);
    Append2File("]");
  }
  if(main->sgfPC != NULL) {
    Append2File("PC[");
    note = AddSgfEscapes(main->sgfPC);
    Append2File(note);
    DestroyNote(note);
    Append2File("]");
  }
  if(main->sgfPW != NULL) {
    Append2File("PW[");
    note = AddSgfEscapes(main->sgfPW);
    Append2File(note);
    DestroyNote(note);
    Append2File("]");
  }
  if(main->sgfRE != NULL) {
    Append2File("RE[");
    note = AddSgfEscapes(main->sgfRE);
    Append2File(note);
    DestroyNote(note);
    Append2File("]");
  }
  if(main->sgfWR != NULL) {
    Append2File("WR[");
    note = AddSgfEscapes(main->sgfWR);
    Append2File(note);
    DestroyNote(note);
    Append2File("]");
  }
  if(main->sgfGE != NULL) {
    Append2File("GE[");
    note = AddSgfEscapes(main->sgfGE);
    Append2File(note);
    DestroyNote(note);
    Append2File("]");
  }
  if(main->sgfDI != NULL) {
    Append2File("DI[");
    note = AddSgfEscapes(main->sgfDI);
    Append2File(note);
    DestroyNote(note);
    Append2File("]");
  }
  if(main->sgfHA > 1) {
    Append2File("HA["); /* Handicap amount */
    Gint2File(main->sgfHA);
    Append2File("]");
  }
  if(main->sgfAB != NULL) {
    Append2File("AB"); /* Handicap stones */
    Pointlist2File(main->sgfAB);
  }
  if(main->sgfAW != NULL) {
    Append2File("AW"); /* White setup stones */
    Pointlist2File(main->sgfAW);
  }
  if(main->sgfMA != NULL) {
    Append2File("MA"); /* Crosses */
    Pointlist2File(main->sgfMA);
  }
  if(main->sgfTR != NULL) {
    Append2File("TR"); /* Triangles */
    Pointlist2File(main->sgfTR);
  }
  if(main->sgfSQ != NULL) {
    Append2File("SQ"); /* Squares */
    Pointlist2File(main->sgfSQ);
  }
  if(main->sgfCR != NULL) {
    Append2File("CR"); /* Circles */
    Pointlist2File(main->sgfCR);
  }
  if(main->sgfLB != NULL) {
    Append2File("LB"); /* Labels */
    Labellist2File(main->sgfLB);
  }
  if(main->sgfKMx2 != 0) {
    Append2File("KM["); /* Komi */
    Gint2File(main->sgfKMx2/2);
    Append2File(".");
    Gint2File((main->sgfKMx2%2)*5);
    Append2File("]");
  }
  if(main->sgfPL == STONE_WHITE) {
    Append2File("PL[W]");
  }
  if(main->sgfPL == STONE_BLACK) {
    Append2File("PL[B]");
  }
  Append2File("ST["); /* Viewing style */
  Gint2File(main->sgfST);
  Append2File("]SZ["); /* Goban size */
  Gint2File(main->sgfSZ);
  Append2File("]");
  Node2Sgf(main->child,main->compareNode); /* Nodes */
  Append2File(")");
}

void Node2Sgf(Node * node, Node * home)
{
  Node * firstChild = node;
  gchar * note = NULL;
  if(node == NULL) {
    return;
  }
  do{
    if(node->sibling != node) {
      Append2File("(");
    }
    Append2File("\n;");
    if(node->sgfC != NULL) {
      Append2File("C[");
      note = AddSgfEscapes(node->sgfC);
      Append2File(note);
      DestroyNote(note);
      Append2File("]");
    }
	 if(node->stone == STONE_BLACK) {
	   switch (node->at) {
	     case AT_IS_PASS:
          Append2File("B[]"); /* Black pass */
	       break;
	     case AT_IS_SETUP:
          Append2File("PL[W]"); /* White to play */
	       break;
	     default:
          Append2File("B"); /* Black play */
          Point2File(node->at);
	   }
	 }
	 if(node->stone == STONE_WHITE) {
	   switch (node->at) {
	     case AT_IS_PASS:
          Append2File("W[]"); /* White pass */
	       break;
	     case AT_IS_SETUP:
          Append2File("PL[B]"); /* Black to play */
	       break;
	     default:
          Append2File("W"); /* White play */
          Point2File(node->at);
	   }
	 }
    if(node->sgfVx2 != 0) {
      Append2File("V["); /* Estimated score */
      Gint2File(node->sgfVx2/2);
      Append2File(".");
      if(node->sgfVx2 > 0) {
        Gint2File((node->sgfVx2%2)*5);
      }
      else {
        Gint2File(((0-node->sgfVx2)%2)*5);
      }
      Append2File("]");
    }
	 if(node->sgfMA != NULL) {
      Append2File("MA"); /* Cross */
      Pointlist2File(node->sgfMA);
 	 }
	 if(node->sgfSQ != NULL) {
      Append2File("SQ"); /* Square */
      Pointlist2File(node->sgfSQ);
	 }
	 if(node->sgfTR != NULL) {
      Append2File("TR"); /* Triangle */
      Pointlist2File(node->sgfTR);
	 }
	 if(node->sgfCR != NULL) {
      Append2File("CR"); /* Circle */
      Pointlist2File(node->sgfCR);
 	 }
	 if(node->sgfTB != NULL) {
      Append2File("TB"); /* Black territory */
      Pointlist2File(node->sgfTB);
	 }
	 if(node->sgfTW != NULL) {
      Append2File("TW"); /* White territory */
      Pointlist2File(node->sgfTW);
	 }
	 if(node->sgfAE != NULL) {
      Append2File("AE"); /* Add empty */
      Pointlist2File(node->sgfAE);
	 }
	 if(node->sgfAB != NULL) {
      Append2File("AB"); /* Add black stones */
      Pointlist2File(node->sgfAB);
	 }
	 if(node->sgfAW != NULL) {
      Append2File("AW"); /* Add white stones */
      Pointlist2File(node->sgfAW);
	 }
	 if(node->sgfLB != NULL) {
      Append2File("LB"); /* Label */
      Labellist2File(node->sgfLB);
	 }
	 if(node == home) {
      Append2File("MULTIGOBM[]"); /* Bookmark for home */
	 }
	 if(node->child != NULL) {
	   Node2Sgf(node->child,home);
	 }
    if(node->sibling != node) {
      Append2File(")");
    }
	 node = node->sibling;
  } while(node != firstChild);
}
