/*
 * This file is part of goban770
 *
 * Copyright (C) 2006,2007 Jarmo Ahosola.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <stdio.h>
#include <string.h>
#include <utils/log.h>

/* Gnome VFS for file i/o */
#include <libgnomevfs/gnome-vfs.h>

#include <hildon/hildon-banner.h>

static GtkTextBuffer * log_h_logBuffer;
static LogMethod log_h_logMethod; /* Was LogLevel and worked, propably was error that had no effect */
static LogLevel log_h_logLevel;
static LogScope log_h_logScope;
/* static MainView * log_h_main; */
static GtkWidget * log_h_window;
static gint log_h_memory_usage;
static char log_h_tmp_text[100];
static LogLevel log_h_severestMsgAtLog;
static gboolean log_h_log2file;
static gchar log_h_logfile_at[100];

/* Training statistics */
static gint log_h_wrongGuess; 
static gint log_h_rightGuess; 
static gint log_h_peekNext; 
static gint log_h_takeHint; 
static gint log_h_currentWrong; 
static gint log_h_worstWrong; 
static gboolean log_h_hintTaken;
static gint log_h_firstWrong;
static gint log_h_firstHint;
static gint log_h_firstPeek;

/* Problem statistics */
static gint log_h_problemsSeen = 0;
static gint log_h_problemsSolved = 0;
static gint log_h_problemsSkipped = 0;
static gboolean log_h_problemBeenAtEnd = FALSE;
static gboolean log_h_problemIsAtEnd = FALSE;
static gboolean log_h_problemHasCheated = FALSE;
static gboolean log_h_problemHasBrowsed = FALSE;

void Browsing()
{
  log_h_problemHasBrowsed = TRUE;
  msg2log("Browsing!!!", LOGLEVEL_WARNING, LOGSCOPE_USER);
}

void GuessWrong()
{
  log_h_wrongGuess++;
  log_h_currentWrong++;
  if(log_h_firstWrong == -1) {
    log_h_firstWrong = log_h_rightGuess;
  }
}

void GuessRight()
{
  log_h_rightGuess++;
  if(log_h_currentWrong > log_h_worstWrong) {
    log_h_worstWrong = log_h_currentWrong;
  }
  log_h_currentWrong = 0;
  log_h_hintTaken = FALSE;
}

void TakeMoveBack()
{
  log_h_rightGuess--;
  log_h_problemHasCheated = TRUE;
}

void TakeHint()
{
  if(log_h_hintTaken == FALSE) {
    log_h_takeHint++;
    log_h_hintTaken = TRUE;
  }
  if(log_h_firstHint == -1) {
    log_h_firstHint = log_h_rightGuess;
  }
}

void PeekNext()
{
  log_h_peekNext++;
  if(log_h_currentWrong > log_h_worstWrong) {
    log_h_worstWrong = log_h_currentWrong;
  }
  log_h_currentWrong = 0;
  log_h_hintTaken = FALSE;
  if(log_h_firstPeek == -1) {
    log_h_firstPeek = log_h_rightGuess;
  }
  log_h_problemHasCheated = TRUE;
}

void LastNodeEnter()
{
  log_h_problemIsAtEnd = TRUE;
}

void LastNodeExit()
{
  log_h_problemBeenAtEnd = TRUE;
  log_h_problemIsAtEnd = FALSE;
}

void EnterFile()
{
  log_h_problemBeenAtEnd = FALSE;
  log_h_problemIsAtEnd = FALSE;
  log_h_problemHasBrowsed = FALSE;
  log_h_problemHasCheated = FALSE;
}

void PrintTrainingStatistics(gboolean isProblem)
{
  gint correctness = 0;
  
  if((log_h_rightGuess + log_h_peekNext) == 0) {
    return;
  }
  /* Correctness calculation justification
    1) If user peek every move, correctness should be 0%
    2) If user knows correctly every move without any assistance, correctness should be 100%
    3) If user take hint and using that randomly guess next 5 plays, the result should in average be equal to peeking those 5 moves
    4) If user optimize use of hint, quess known the hint and peek, the result should in average be same as simply peeking
    Values for each action has been calculated directly from listed 4 fact above. Calculation assumes that hint gives next 5 plays without telling their order.  
    Additionally, taking redundant hints is counted as one hint only, so that user can take same hint again after wrong guess without additional cost.
  */
  correctness = (1000 - (250 * log_h_wrongGuess + 1000 * log_h_peekNext + 2500 * log_h_takeHint) / (log_h_rightGuess + log_h_peekNext))/10;
  if(isProblem == TRUE) {
    if(log_h_problemsSeen > 0) {
      msg2log("Problem statistics", LOGLEVEL_STATISTICS, LOGSCOPE_USER);
      msg2logGint("Solved at first try %d problems", LOGLEVEL_STATISTICS, LOGSCOPE_USER, log_h_problemsSolved);
      msg2logGint("Failed at first try %d problems", LOGLEVEL_STATISTICS, LOGSCOPE_USER, log_h_problemsSeen - log_h_problemsSolved - log_h_problemsSkipped);
      msg2logGint("Skipped %d problems", LOGLEVEL_STATISTICS, LOGSCOPE_USER, log_h_problemsSkipped);
      msg2logGint("Giving up percentage %d%%", LOGLEVEL_STATISTICS, LOGSCOPE_USER, (log_h_problemsSkipped * 100)/log_h_problemsSeen);
      msg2logGint("Solving percentage %d%%\n", LOGLEVEL_STATISTICS, LOGSCOPE_USER, (log_h_problemsSolved * 100)/log_h_problemsSeen);
    }
  }
  else {
    msg2log("Training statistics", LOGLEVEL_STATISTICS, LOGSCOPE_USER);
    msg2logGint("Total moves %d", LOGLEVEL_STATISTICS, LOGSCOPE_USER, log_h_rightGuess + log_h_peekNext);
    if(log_h_firstWrong != -1) {
      msg2logGint("Wrong guess %d times", LOGLEVEL_STATISTICS, LOGSCOPE_USER, log_h_wrongGuess);
      msg2logGint("First wrong guess at move %d", LOGLEVEL_STATISTICS, LOGSCOPE_USER, log_h_firstWrong);
    }
    if(log_h_firstPeek != -1) {
      msg2logGint("Cheated %d times", LOGLEVEL_STATISTICS, LOGSCOPE_USER, log_h_peekNext);
      msg2logGint("First cheating at move %d", LOGLEVEL_STATISTICS, LOGSCOPE_USER, log_h_firstPeek);
    }
    if(log_h_firstHint != -1) {
      msg2logGint("Hint used at %d positions", LOGLEVEL_STATISTICS, LOGSCOPE_USER, log_h_takeHint);
      msg2logGint("First hint used at move %d", LOGLEVEL_STATISTICS, LOGSCOPE_USER, log_h_firstHint);
    }
    msg2logGint("Hardest move %d wrong guess", LOGLEVEL_STATISTICS, LOGSCOPE_USER, log_h_worstWrong);
    msg2logGint("Correctness %d%%", LOGLEVEL_STATISTICS, LOGSCOPE_USER, correctness);
  }
}  

void ResetTrainingStatistics(gboolean fileChanges, gboolean isProblem)
{
  if(fileChanges == TRUE) {
    if(isProblem == TRUE) { /* Problem statistics are reseted only if file changes and the file was problem */
      log_h_problemsSeen++; /* Count amount of seen problems */
      if((log_h_problemBeenAtEnd == FALSE) && (log_h_wrongGuess == 0) && (log_h_problemHasCheated == FALSE) && (log_h_problemHasBrowsed == FALSE)) { /* If user has not been at end of branch before, has not made wrong quesses and has not cheated */
        if(log_h_problemIsAtEnd == FALSE) { /* but is not currently at the end of the line */
          log_h_problemsSkipped++; /* count problem as skipped without trying */
        }
        else { /* and is currently at the end of the line */
          log_h_problemsSolved++;/* count problem as solved */
        }
      }
      msg2logBol("Has been at end", "Hasn't been at end", LOGLEVEL_INFO, LOGSCOPE_FORCED, log_h_problemBeenAtEnd);
      msg2logBol("Is at end", "Isn't at end", LOGLEVEL_INFO, LOGSCOPE_FORCED, log_h_problemIsAtEnd);
      msg2logBol("No wrong guesses", "Wrong guesses", LOGLEVEL_INFO, LOGSCOPE_FORCED, (log_h_wrongGuess == 0));
      msg2logBol("Has cheated", "Hasn't cheated", LOGLEVEL_INFO, LOGSCOPE_FORCED, log_h_problemHasCheated);
      msg2logGint("Solved %d problems", LOGLEVEL_STATISTICS, LOGSCOPE_USER, log_h_problemsSolved);
      msg2logGint("Skipped %d problems", LOGLEVEL_STATISTICS, LOGSCOPE_USER, log_h_problemsSkipped);
    }
  }
  log_h_wrongGuess = 0; 
  log_h_rightGuess = 0; 
  log_h_peekNext = 0; 
  log_h_takeHint = 0; 
  log_h_currentWrong = 0; 
  log_h_worstWrong = 0; 
  log_h_hintTaken = FALSE;
  log_h_firstPeek = -1;
  log_h_firstHint = -1;
  log_h_firstWrong = -1;
}

/* Privates: */

/* Public functions */
void init_log_service(GtkTextBuffer * logBuffer, GtkWidget * scrollWindow) {
  log_h_logBuffer = logBuffer;
  log_h_window = scrollWindow;
  log_h_memory_usage = 0;
  log_h_severestMsgAtLog = LOGLEVEL_NOT_USED;
  log_h_log2file = FALSE;
  strcpy(log_h_logfile_at,"goban770.log"); /* Tell me if you find WHERE that goes */
}

void ResetSeverestMsg() {
  log_h_severestMsgAtLog = LOGLEVEL_NOT_USED;
}

LogLevel GetSeverestMsg() {
  return log_h_severestMsgAtLog;
}

LogMethod GetLogMethod() {
  return log_h_logMethod;
}

void set_log2file(gboolean newValue, gchar * logAt) {
  gchar * charAt = NULL;
  log_h_log2file = newValue;
  if((log_h_log2file == TRUE) && (logAt != NULL)) {
    if(strlen(logAt) < 80) {
      strcpy(log_h_logfile_at,logAt);
      charAt = log_h_logfile_at;
      while(charAt[0] != '\0') {
        charAt++;
      }
      while(charAt[0] != '/' && charAt[0] != '\\' && charAt != log_h_logfile_at) {
        charAt--;
      }
      charAt++;
      strcpy(charAt,"goban770.log");
    }
  }
  msg2logBol("Logging to the file set on", "Logging to the file set off", LOGLEVEL_INFO, LOGSCOPE_FORCED, log_h_log2file);
  msg2log("Logfile is written at:", LOGLEVEL_INFO, LOGSCOPE_FORCED); 
  msg2log(log_h_logfile_at, LOGLEVEL_INFO, LOGSCOPE_FORCED); 
}

void set_log_method(LogMethod method) {
  log_h_logMethod = method;
#ifndef PRODUCTION_VERSION  
/*  msg2logGint("New logging method set to %d.", LOGLEVEL_INFO, LOGSCOPE_NONE, method); */
#endif  
}

void set_log_level(LogLevel level) {
  log_h_logLevel = level;
#ifndef PRODUCTION_VERSION  
/*  msg2logGint("New logging level set to %d.", level, LOGSCOPE_NONE, level); */
#endif  
}

void set_log_scope(LogScope scope) {
  log_h_logScope = scope;
#ifndef PRODUCTION_VERSION  
/*  msg2log("New logging scope set", LOGLEVEL_INFO, LOGSCOPE_NONE); */
#endif  
}

void LogReserveMemory(const char * what)
{
  log_h_memory_usage++;
  sprintf(log_h_tmp_text,"Reserved %s",what);
  msg2log(log_h_tmp_text,LOGLEVEL_INFO,LOGSCOPE_MEMORY);
  msg2logGint("Currently %d datastructures reserved",LOGLEVEL_INFO,LOGSCOPE_MEMORY,log_h_memory_usage);
}

void LogReleaseMemory(const char * what)
{
  log_h_memory_usage--;
  sprintf(log_h_tmp_text,"Released %s",what);
  msg2log(log_h_tmp_text,LOGLEVEL_INFO,LOGSCOPE_MEMORY);
  msg2logGint("Currently %d datastructures reserved",LOGLEVEL_INFO,LOGSCOPE_MEMORY,log_h_memory_usage);
}

void UserMessage(GtkWidget * widget, Node * node, const char * msg)
{
  hildon_banner_show_information(widget, NULL, msg);
  msg2log(msg, LOGLEVEL_INFO, LOGSCOPE_USER);
}

/* Write log message with number in it. Calling message should have %d at place where number is wanted at */
void msg2logGint(const char * msg, LogLevel level, LogScope scope, gint number) {
  sprintf(log_h_tmp_text,msg,number);
  msg2log(log_h_tmp_text, level, scope);
}

/* Write log message with number in it. Calling message should have %d at place where number is wanted at */
void msg2logBol(const char * yes, const char * no, LogLevel level, LogScope scope, gboolean bol) {
  if(bol == TRUE) {
    msg2log(yes, level, scope);
  }
  else {
    msg2log(no, level, scope);
  }
}

gboolean WouldTheseParametersLog(LogLevel level, LogScope scope)
{
  if((log_h_logMethod == LOGMETHOD_PAUSED) && (log_h_logScope != LOGSCOPE_FORCED)) {
    return FALSE; /* Log messages are currently paused and requested messge is not forced */
  }
  if(log_h_logLevel < level) {
    return FALSE; /* Requested log message is not severe enough */
  }
  if((scope & log_h_logScope) != log_h_logScope) {
    return FALSE; /* Requested log message is out of the selected scope */
  }
  return TRUE;
}

/* Write log mesage in case it is wanted. Use OR to combine values of LogScope. */
void msg2log(const char * msg, LogLevel level, LogScope scope) 
{
  GtkTextTag *tag = NULL;
  GtkTextIter bufferend;
  GtkAdjustment * verticalPosition;
  
  if(WouldTheseParametersLog(level, scope) == FALSE) {
    return;
  }
  if(level < log_h_severestMsgAtLog) {
    log_h_severestMsgAtLog = level;
  }
  switch (level) {
    case LOGLEVEL_NONE:
      tag = gtk_text_buffer_create_tag (log_h_logBuffer, NULL, "foreground", "white", NULL); /* No message color */
      break;
    case LOGLEVEL_ERROR:
      tag = gtk_text_buffer_create_tag (log_h_logBuffer, NULL, "foreground", "red", NULL); /* Error color */
      break;
    case LOGLEVEL_WARNING:
      tag = gtk_text_buffer_create_tag (log_h_logBuffer, NULL, "foreground", "blue", NULL); /* Warning color */ 
      break;
    case LOGLEVEL_STATISTICS:
      tag = gtk_text_buffer_create_tag (log_h_logBuffer, NULL, "foreground", "black", NULL); /* Statistics color */
      break;
    case LOGLEVEL_INFO:
      tag = gtk_text_buffer_create_tag (log_h_logBuffer, NULL, "foreground", "black", NULL); /* Info color */
      break;
	default:
	  break;
  }
  gtk_text_buffer_get_iter_at_offset (log_h_logBuffer, &bufferend, -1); /* Find where log end was before log message */
  gtk_text_buffer_insert_with_tags(log_h_logBuffer,&bufferend,msg,-1,tag,NULL);
  gtk_text_buffer_get_iter_at_offset (log_h_logBuffer, &bufferend, -1); /* Find where log end is after log message */
  gtk_text_buffer_insert_with_tags(log_h_logBuffer,&bufferend,"\n",-1,tag,NULL);
  if(log_h_logMethod == LOGMETHOD_APPEND) {
    gtk_text_buffer_get_iter_at_offset (log_h_logBuffer, &bufferend, -1); /* Find where log end is after linefeed */
    gtk_text_buffer_place_cursor (log_h_logBuffer,&bufferend);
    verticalPosition = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW (log_h_window)); /* Get vertical position control */
    if(verticalPosition != NULL) { /* In case there is something to scroll */
      gtk_adjustment_set_value (verticalPosition,verticalPosition->upper); /* Set vertical position to the bottom */
    }
  }
  if(log_h_log2file == TRUE) {
    CopyLog2File();
  }
}

void CopyLog2File()
{
  GnomeVFSResult vfs_result;
  GnomeVFSHandle *handle = NULL;
  GnomeVFSFileSize out_bytes;
  gchar *temp_buffer = NULL;
  GtkTextIter start, end;

  /* try to create handle to file */
  vfs_result = gnome_vfs_create(&handle, log_h_logfile_at, GNOME_VFS_OPEN_WRITE, 0, 0600);
  if ( vfs_result != GNOME_VFS_OK ) {
    return;
  }

  /* find start and end of text */
  gtk_text_buffer_get_bounds( GTK_TEXT_BUFFER (log_h_logBuffer), &start, &end);

  /* copy all text to temp_buffer */
  temp_buffer = gtk_text_buffer_get_slice( GTK_TEXT_BUFFER (log_h_logBuffer), &start, &end, TRUE);

  /* write text to file */
  gnome_vfs_write(handle, temp_buffer, strlen(temp_buffer), &out_bytes);

  /* free temp, close file and return */
  g_free(temp_buffer);
  gnome_vfs_close(handle);
}
