/*
 * This file is part of goban770
 *
 * Copyright (C) 2006,2007 Jarmo Ahosola.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <gtk/gtk.h>
#include <ui/interface.h>
#include <utils/log.h>
#include <utils/draw.h>
#include <game/goban.h>
#include <game/gametree.h>

/* privates */
static gchar draw_h_text[100];
static gint draw_h_treescale = 1;
static gint draw_h_vshift = 0;
static gint draw_h_hshift = 0;
#define HEIGHT2Y(h) (((h)+(draw_h_vshift)) * (draw_h_treescale))
#define TURN2X(t) (((t)+(draw_h_hshift)) * (draw_h_treescale))
#define Y2HEIGHT(y) (((y)/(draw_h_treescale)) - (draw_h_vshift))
#define X2TURN(x) (((x)/(draw_h_treescale)) - (draw_h_hshift))

gint Y2Height(gint y) {
  return Y2HEIGHT(y);
}

gint X2Turn(gint x) {
  return X2TURN(x);
}

gchar * X2ColumnTitle(gint x)
{
  draw_h_text[0] = 'A' + x;
  if(draw_h_text[0] >= 'I') {
    draw_h_text[0]++;
  }
  draw_h_text[1] = 0;
  return draw_h_text;
} 

gchar * AddGintInString(gchar * in, gint value)
{
  sprintf(draw_h_text,in,value);
  return draw_h_text;
}

gchar * Add2GintInString(gchar * in, gint value1, gint value2)
{
  sprintf(draw_h_text,in,value1,value2);
  return draw_h_text;
}

gchar * ProblemText(gchar * classification, gchar * difficulty)
{
  if(classification == NULL) {
    if(difficulty != NULL) {
      sprintf(draw_h_text,"(%s)",difficulty);
    }
  }
  else {
    if(difficulty == NULL) {
      sprintf(draw_h_text,"%s",classification);
    }
    else {
      sprintf(draw_h_text,"%s (%s)",classification,difficulty);
    }
  }
  return draw_h_text;
}

gchar * Gint2String(gint value)
{
  sprintf(draw_h_text,"%d",value);
  return draw_h_text;
}

gchar * At2String(gint at, gint gobanSize)
{
  gint x, y;
  static gchar col[2] = "\0\0";
  
  if(at == AT_IS_PASS) {
    return "PAS";
  }
  if(at == AT_IS_SETUP) {
    return "SET";
  }
  if(at == AT_IS_UNDEFINED) {
    return "UDF";
  }
  x = Location2X(at);
  y = Location2Y(at);
  col[0] = 'A' + x;
  if(col[0] >= 'I') {
    col[0]++;
  }
  sprintf(draw_h_text,"%s%d",col,gobanSize - y);
  return draw_h_text;
}

gchar * Y2RowTitle(gint y, gint gobanSize)
{
  return Gint2String(gobanSize - y);
}

gchar * MoveRange2String(gint from, gint to)
{
  sprintf(draw_h_text,"(%d-%d)",from,to);
  return draw_h_text;
}

gchar * WherePlayed2String(gint number, gint at, gint gobanSize)
{
  gint i = 0, x = 0, y = 0;
  x = Location2X(at);
  y = Location2Y(at);
  if(x >= ('I' - 'A')) {
    x++;
  }
  y = gobanSize - y;
  for(i=0; i<8; i++) {
    draw_h_text[i] = ' ';
  }
  i = 2;
  while(number > 0) {
    draw_h_text[i] = number%10 + '0';
    number = number / 10;
    i--;
  }
  /* draw_h_text[3] = 8594; / * unicode '->' character */
  draw_h_text[3] = '-';
  draw_h_text[4] = '>';
  draw_h_text[5] = x + 'A';
  if(y >= 10) {
    draw_h_text[6] = (y/10) + '0';
    draw_h_text[7] = (y%10) + '0';
  }
  else {
    draw_h_text[6] = y + '0';
  }
  draw_h_text[8] = '\0';
  return draw_h_text;
}

void WriteText( GtkWidget * widget, GdkPixmap * pixmap, gchar * text, gint centerX, gint centerY, gint width, gint height, GdkGC * gc)
{
  PangoLayout * pangoLayout = NULL;
  /* PangoFontDescription * pangoFontDescription; */
  int xsize = 0, ysize = 0;

  if((centerX + (width/2)) < 0) {
    return; /* text is left of drawing area */
  }
  if((centerY + (height/2)) < 0) {
    return; /* text is above drawing area */
  }
  if((centerX - (width/2)) > widget->allocation.width) {
    return; /* text is right of drawing area */
  }
  if((centerY - (height/2)) > widget->allocation.height) {
    return; /* text is below drawing area */
  }

  pangoLayout = gtk_widget_create_pango_layout (widget,text);
  if(pangoLayout == NULL) {
#ifndef PRODUCTION_VERSION  
    msg2log("pangoLayout == NULL", LOGLEVEL_ERROR, LOGSCOPE_MEMORY);
#endif
    return;
  }
  pango_layout_get_pixel_size (pangoLayout,&xsize,&ysize);
  while((xsize >= width) && (text[0] <= '9') && (text[0] >= '0')) {
    text = &text[1];
    if(pangoLayout != NULL) {
      g_object_unref(G_OBJECT( pangoLayout)); /* If crash remove this line */
    }
    pangoLayout = gtk_widget_create_pango_layout (widget,text);
    if(pangoLayout == NULL) {
#ifndef PRODUCTION_VERSION  
      msg2log("pangoLayout == NULL in while", LOGLEVEL_ERROR, LOGSCOPE_MEMORY);
#endif
      return;
    }
    pango_layout_get_pixel_size (pangoLayout,&xsize,&ysize);
  }
  if((xsize < width) && (ysize < height)) {
    if(gc == NULL) {
      gdk_draw_layout (pixmap,widget->style->black_gc,centerX-(xsize/2),centerY-(ysize/2),pangoLayout);
    }
    else {
      gdk_draw_layout (pixmap,gc,centerX-(xsize/2),centerY-(ysize/2),pangoLayout);
    }
  }
  if(pangoLayout != NULL) {
    g_object_unref(G_OBJECT( pangoLayout)); /* If crash remove this line */
  }
}

gint GetHoshiInterval(gint gobanSize)
{
  gint result = 0;
  
#ifndef PRODUCTION_VERSION  
    msg2logGint("GetHoshiInterval called with %d", LOGLEVEL_INFO, LOGSCOPE_PATH, gobanSize);
#endif
  result = (gobanSize - 7)/2;
  if(result < 2) {
    result = (gobanSize + 1)/4;
  }
#ifndef PRODUCTION_VERSION  
    msg2logGint("GetHoshiInterval returns %d", LOGLEVEL_INFO, LOGSCOPE_PATH, result);
#endif
  return result;
}

void DrawHoshiPoints(MainView * main)
{
  gint interval = 0;
  gint x = 0, y = 0;

  interval = GetHoshiInterval(main->sgfSZ);
  if(main->sgfSZ > 14) { /* 9 hoshi points */
    for(x=0-interval;x<=interval;x+=interval) {
      for(y=0-interval;y<=interval;y+=interval) {
#ifndef PRODUCTION_VERSION  
    msg2log("9 hoshi points loop", LOGLEVEL_INFO, LOGSCOPE_LOOP);
#endif
        DrawHoshiAt(main,main->sgfSZ/2+x,main->sgfSZ/2+y);
      }
    }
  }
  if(main->sgfSZ < 14 && main->sgfSZ > 4) { /* Corner hoshi points */
    for(x=0-interval;x<=interval;x+=interval+interval) {
      for(y=0-interval;y<=interval;y+=interval+interval) {
#ifndef PRODUCTION_VERSION  
    msg2log("4 hoshi points loop", LOGLEVEL_INFO, LOGSCOPE_LOOP);
#endif
        DrawHoshiAt(main,main->sgfSZ/2+x,main->sgfSZ/2+y);
      }
    }
  }
  if(main->sgfSZ > 10 && main->sgfSZ < 14) { /* Tengen hoshi point */
    DrawHoshiAt(main,main->sgfSZ/2,main->sgfSZ/2);
  }
}

void DrawHoshiAt(MainView * main, gint x, gint y)
{
  /* Draw filled black circle with diameter about 10th of a stone */
  gint xcenter = 0, ycenter = 0, radius = 0, left = 0, top = 0;

#ifndef PRODUCTION_VERSION  
    msg2logGint("Hoshi x =  %d", LOGLEVEL_INFO, LOGSCOPE_PATH, x);
    msg2logGint("Hoshi y =  %d", LOGLEVEL_INFO, LOGSCOPE_PATH, y);
#endif
  xcenter = gobanx2viewx(x, main );
  ycenter = gobany2viewy(y, main );
  radius = (2 * main->stoneCenter) / 5;
  left = xcenter - radius;
  top = ycenter - radius;
  gdk_draw_arc (main->gobanPixmap,
                main->goban_view->style->black_gc,
                TRUE,
                left, top, radius + radius, radius + radius,
                0, 64 * 360);
}

void DrawThumb(MainView * main, gint top, gint left, gint size, ThumbList * thumb)
{
  gint stoneSize;
  gint i, x, y, shift;
  if(thumb == NULL) {
    return;
  }
  if(thumb->sgfSZ < 1) {
    return;
  }
  stoneSize = size / thumb->sgfSZ;
  shift = (size - (stoneSize * thumb->sgfSZ)) / 2;
  size = stoneSize * thumb->sgfSZ;
  top += shift;
  left += shift;
  gdk_draw_rectangle (main->gobanPixmap,
                      main->gobanGC,
                      TRUE,
                      left, 
                      top, 
                      size, 
                      size);
  for(i=0; i<thumb->sgfSZ; i++) {
    gdk_draw_line (main->gobanPixmap,
               main->goban_view->style->black_gc,
               left + stoneSize*i + stoneSize/2,
               top + stoneSize/2,
               left + stoneSize*i + stoneSize/2,
			      top + stoneSize*(thumb->sgfSZ-1) + stoneSize/2);
    gdk_draw_line (main->gobanPixmap,
               main->goban_view->style->black_gc,
               left + stoneSize/2,
               top + stoneSize*i + stoneSize/2,
               left + stoneSize*(thumb->sgfSZ-1) + stoneSize/2,
               top + stoneSize*i + stoneSize/2);
  }
  if(thumb->thumb == NULL) {
    return;
  }
  for(x=0; x<thumb->sgfSZ; x++) {
    for(y=0; y<thumb->sgfSZ; y++) {
      switch(thumb->thumb[x+y*thumb->sgfSZ]) {
        case 'b':
          gdk_draw_arc (main->gobanPixmap,
                main->goban_view->style->black_gc,
                TRUE,
                left+x*stoneSize, top+y*stoneSize, stoneSize, stoneSize,
                0, 64 * 360);
          gdk_draw_arc (main->gobanPixmap,
                main->goban_view->style->black_gc,
                FALSE,
                left+x*stoneSize, top+y*stoneSize, stoneSize, stoneSize,
                0, 64 * 360);
          break;
        case 'w':
          gdk_draw_arc (main->gobanPixmap,
                main->goban_view->style->white_gc,
                TRUE,
                left+x*stoneSize, top+y*stoneSize, stoneSize, stoneSize,
                0, 64 * 360);
          gdk_draw_arc (main->gobanPixmap,
                main->goban_view->style->black_gc,
                FALSE,
                left+x*stoneSize, top+y*stoneSize, stoneSize, stoneSize,
                0, 64 * 360);
          break;
        case 'B':
          gdk_draw_arc (main->gobanPixmap,
                main->indicateGC,
                TRUE,
                left+x*stoneSize, top+y*stoneSize, stoneSize, stoneSize,
                0, 64 * 360);
          gdk_draw_arc (main->gobanPixmap,
                main->goban_view->style->black_gc,
                TRUE,
                left+x*stoneSize+1, top+y*stoneSize+1, stoneSize-2, stoneSize-2,
                0, 64 * 360);
          break;
        case 'W':
          gdk_draw_arc (main->gobanPixmap,
                main->indicateGC,
                TRUE,
                left+x*stoneSize, top+y*stoneSize, stoneSize, stoneSize,
                0, 64 * 360);
          gdk_draw_arc (main->gobanPixmap,
                main->goban_view->style->white_gc,
                TRUE,
                left+x*stoneSize+1, top+y*stoneSize+1, stoneSize-2, stoneSize-2,
                0, 64 * 360);
          break;
      }
    }
  }
}

void DrawGoban(MainView * main)
{
  Pointlist * points;
  Labellist * labels;
  gchar text[2];
  Node * node = NULL;
  Node * firstNode = NULL;
  Node * numberingBase = NULL;
  gint underStones = 0;
  gboolean gobanShifted = FALSE, topColumnSpaceUsed = FALSE, bottomColumnSpaceUsed = FALSE;
  gint numberingOffset = 0;
  gint scorex2 = 0;
  gint haveEstimate = FALSE;
  gint x = 0, y = 0, at = 0, sizeWithCoordinates = 0, i = 0, top = 0, bottom = 0, left = 0, right = 0, titletop = 0, titlebottom = 0, titleleft = 0, titleright = 0;
  ThumbList * thumb = NULL;

  underStones = setCurrentPosition(main);
  numberingBase = findStartOfNumbering( main );
  if (main->gobanPixmap)
    g_object_unref (main->gobanPixmap);

  main->gobanPixmap = gdk_pixmap_new (main->goban_view->window,
			   main->goban_view->allocation.width,
			   main->goban_view->allocation.height,
			   -1);
  gdk_draw_rectangle (main->gobanPixmap,
		      main->goban_view->style->white_gc,
		      TRUE,
		      0, 0,
		      main->goban_view->allocation.width,
		      main->goban_view->allocation.height);
		      
  if(main->gobanGC != NULL) {
    g_object_unref(G_OBJECT( main->gobanGC)); /* If crash remove this line */
  }
  main->gobanGC = gdk_gc_new(main->gobanPixmap);
  gdk_gc_set_foreground(main->gobanGC,&(main->gobanColor)); /* */
  if(main->indicateGC != NULL) {
    g_object_unref(G_OBJECT( main->indicateGC)); /* If crash remove this line */
  }
  main->indicateGC = gdk_gc_new(main->gobanPixmap);
  gdk_gc_set_foreground(main->indicateGC,&(main->indicationColor)); /* */
  if(main->temporal_mode == TEMPMODE_FILE_THUMBS) {
    thumb = main->thumbnails;
    for(x=THUMB_LEFT_MARGIN; (x+THUMB_HORIZONTAL_DELTA)<SCREEN_MAX_X; x+=THUMB_HORIZONTAL_DELTA) {
      for(y=THUMB_TOP_MARGIN; (y+THUMB_VERTICAL_DELTA)<SCREEN_MAX_Y; y+=THUMB_VERTICAL_DELTA) {
        if(thumb != NULL) {
          DrawThumb(main, y, x, THUMB_MAX_SIZE, thumb);
          WriteText( main->goban_view, main->gobanPixmap, thumb->title, x+THUMB_TITLE_CENTER_X, y+THUMB_TITLE_CENTER_Y, THUMB_TITLE_SPACE_X, THUMB_TITLE_SPACE_Y, main->goban_view->style->black_gc);
          if(main->thumbedPlaylist == NULL) {
            continue;
          }
          if(thumb->indexAtPlaylist == main->thumbedPlaylist->current) { /* If thumbnail is of the currently open file */
            gdk_draw_rectangle (main->gobanPixmap,
		        main->goban_view->style->black_gc,
		        FALSE,
		        x - THUMB_FRAME_MARGIN, y - THUMB_FRAME_MARGIN,
		        THUMB_FRAME_MARGIN + THUMB_MAX_SIZE + THUMB_FRAME_MARGIN,
		        THUMB_VERTICAL_DELTA);
          }
          thumb = thumb->next;
        }
      }
    }
    return;
  }
  /* Let's divide drawing area to slices */
  sizeWithCoordinates = main->sgfSZ + 2;
  if((sizeWithCoordinates%2) == 0) { /* Math below is designed to work with odd sizes only, so if size is even */
    sizeWithCoordinates++; /* to avoid usability problems, just grant extra space for margin */
  }
  main->rowHeight = main->goban_view->allocation.height / sizeWithCoordinates;
  main->colWidth = main->goban_view->allocation.width / sizeWithCoordinates;
  /* GOBAN_ASPECTRATIO */
  if(main->rowHeight * GOBAN_ASPECTRATIO > main->colWidth) {
    main->rowHeight = main->colWidth / GOBAN_ASPECTRATIO;
  }
  else {
    main->colWidth = main->rowHeight * GOBAN_ASPECTRATIO;
  }
  main->stoneSize = main->rowHeight;
  if(main->stoneSize > main->colWidth) {
    main->stoneSize = main->colWidth;
  }
  main->stoneSize += (main->stoneSize%2 - 1); 
  main->centerMostStone = main->sgfSZ / 2;
  main->stoneCenter = main->stoneSize / 2;
  main->centerX = (main->colWidth * sizeWithCoordinates) / 2;
  if(main->CurrentMouseMode == main->diagram_item) {
    gobanShifted = TRUE;
  }
  /* To center goban remove this line
  else {
    main->centerX = main->goban_view->allocation.width / 2;
  }
  To center goban remove this line */
  main->centerY = main->goban_view->allocation.height / 2;

  titletop = gobany2viewy(-1,main);
  titlebottom = gobany2viewy(main->sgfSZ,main);
  titleleft = gobanx2viewx(-1,main);
  titleright = gobanx2viewx(main->sgfSZ,main);
  top = gobany2viewy(0,main);
  bottom = gobany2viewy(main->sgfSZ - 1,main);
  left = gobanx2viewx(0,main);
  right = gobanx2viewx(main->sgfSZ - 1,main);
  gdk_draw_rectangle (main->gobanPixmap,
                      main->gobanGC, /* gc, / * main->goban_view->style->black_gc, */
                      TRUE,
                      (left+titleleft)/2 - GOBAN_BORDER, 
                      (top+titletop)/2 - GOBAN_BORDER, 
                      main->colWidth * main->sgfSZ + GOBAN_BORDER + GOBAN_BORDER, 
                      main->rowHeight * main->sgfSZ + GOBAN_BORDER + GOBAN_BORDER);
  if(main->currentNode != NULL) {
    scorex2 = main->currentNode->sgfVx2;
    at = main->currentNode->at;
  }
  else {
    if(main->sgfPL != STONE_NONE) {
      scorex2 = -1; /* Very dirty code */
    }
  }
  for(i=0;i<main->sgfSZ;i++) {
    gdk_draw_line (main->gobanPixmap,
               main->goban_view->style->black_gc,
               gobanx2viewx(i,main),
               top,
			      gobanx2viewx(i,main),
			      bottom);
    gdk_draw_line (main->gobanPixmap,
               main->goban_view->style->black_gc,
               left,
               gobany2viewy(i,main),
			      right,
			      gobany2viewy(i,main));
    WriteText( main->goban_view, main->gobanPixmap, Y2RowTitle(i,main->sgfSZ), titleleft, gobany2viewy(i,main), main->colWidth, main->rowHeight,NULL);
    WriteText( main->goban_view, main->gobanPixmap, Y2RowTitle(i,main->sgfSZ), titleright, gobany2viewy(i,main), main->colWidth, main->rowHeight,NULL);
  } 
  DrawHoshiPoints(main);
  /* setCurrentPosition(main); */

  /* Show mouse mode */
  switch(main->temporal_mode) {
    case TEMPMODE_CHANGE_ORIENTATION:
      break;
    case TEMPMODE_EDIT_AREA:
      for(x=main->select_left; x<=main->select_right; x++) {
        for(y=main->select_top; y<=main->select_bottom; y++) {
          switch(GetStoneXY(x,y)) {
            case STONE_BLACK:
              SetStoneAt(XY2Location(x,y), STONE_TRANSPARENT_BLACK);
              break;
            case STONE_WHITE:
              SetStoneAt(XY2Location(x,y), STONE_TRANSPARENT_WHITE);
              break;
			default:
			  break;
          }
        }
      }
      break;
    case TEMPMODE_NONE:
      if(main->CurrentMouseMode == main->play_item) { /* Mouse mode is play mode */
        DrawStone(main, -1, -1, main->playerInTurn); /* Show whose turn it is */
      }
      if(main->CurrentMouseMode == main->training_item) { /* Mouse mode is training mode */
        DrawStone(main, -1, -1, main->playerInTurn); /* Show whose turn it is */
        if(main->playerInTurn == STONE_WHITE) {
          WriteText( main->goban_view, main->gobanPixmap, "?", gobanx2viewx( -1, main ), gobany2viewy( -1, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
        }
        else {
          WriteText( main->goban_view, main->gobanPixmap, "?", gobanx2viewx( -1, main ), gobany2viewy( -1, main ), main->colWidth, main->rowHeight,main->goban_view->style->white_gc);
        }
      }
      if(main->CurrentMouseMode == main->diagram_item) { /* Mouse mode is diagram view */
        DrawStone(main, -1, -1, main->playerInTurn); /* Show whose turn it is */
        if(main->playerInTurn == STONE_WHITE) {
          if(main->currentNode == NULL) {
            WriteText( main->goban_view, main->gobanPixmap, "1", gobanx2viewx( -1, main ), gobany2viewy( -1, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
          }
          else {
            WriteText( main->goban_view, main->gobanPixmap, Gint2String(main->currentNode->nodeTurnNumber + 1), gobanx2viewx( -1, main ), gobany2viewy( -1, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
          }
        }
        else {
          if(main->currentNode == NULL) {
            WriteText( main->goban_view, main->gobanPixmap, "1", gobanx2viewx( -1, main ), gobany2viewy( -1, main ), main->colWidth, main->rowHeight,main->goban_view->style->white_gc);
          }
          else {
            WriteText( main->goban_view, main->gobanPixmap, Gint2String(main->currentNode->nodeTurnNumber + 1), gobanx2viewx( -1, main ), gobany2viewy( -1, main ), main->colWidth, main->rowHeight,main->goban_view->style->white_gc);
          }
        }
      }
      if(main->CurrentMouseMode == main->addblack_item) { /* Mouse mode to add black stones */
        DrawStone(main, -1, -1, STONE_BLACK);
        DrawStone(main, main->sgfSZ + 1, 0, STONE_BLACK);
        DrawStone(main, main->sgfSZ + 1, 1, STONE_WHITE);
        MarkSelected(main, main->sgfSZ + 1, 0);
        WriteText( main->goban_view, main->gobanPixmap, "+", gobanx2viewx( -1, main ), gobany2viewy( -1, main ), main->colWidth, main->rowHeight,main->goban_view->style->white_gc);
      }
      if(main->CurrentMouseMode == main->addwhite_item) { /* Mouse mode to add white stones */
        DrawStone(main, -1, -1, STONE_WHITE);
        DrawStone(main, main->sgfSZ + 1, 0, STONE_BLACK);
        DrawStone(main, main->sgfSZ + 1, 1, STONE_WHITE);
        MarkSelected(main, main->sgfSZ + 1, 1);
        WriteText( main->goban_view, main->gobanPixmap, "+", gobanx2viewx( -1, main ), gobany2viewy( -1, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
      }
      if(main->CurrentMouseMode == main->triangle_item) { /* Mouse mode to toggle triangles */
        DrawTriangle(main, -1, -1, STONE_WHITE);
        DrawTriangle(main, main->sgfSZ + 1, 0, STONE_WHITE);
        DrawSquare(main, main->sgfSZ + 1, 1, STONE_WHITE);
        DrawCircle(main, main->sgfSZ + 1, 2, STONE_WHITE);
        DrawCross(main, main->sgfSZ + 1, 3, STONE_WHITE);
        text[0] = 'A' + GetUsedLabelCount(main, TRUE);
        text[1] = '\0';
        WriteText( main->goban_view, main->gobanPixmap, text, gobanx2viewx( main->sgfSZ + 1, main ), gobany2viewy( 4, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
        WriteText( main->goban_view, main->gobanPixmap, Gint2String(GetUsedLabelCount(main, FALSE) + 1), gobanx2viewx( main->sgfSZ + 1, main ), gobany2viewy( 5, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
        MarkSelected(main, main->sgfSZ + 1, 0);
      }
      if(main->CurrentMouseMode == main->square_item) { /* Mouse mode to toggle squares */
        DrawSquare(main, -1, -1, STONE_WHITE);
        DrawTriangle(main, main->sgfSZ + 1, 0, STONE_WHITE);
        DrawSquare(main, main->sgfSZ + 1, 1, STONE_WHITE);
        DrawCircle(main, main->sgfSZ + 1, 2, STONE_WHITE);
        DrawCross(main, main->sgfSZ + 1, 3, STONE_WHITE);
        text[0] = 'A' + GetUsedLabelCount(main, TRUE);
        text[1] = '\0';
        WriteText( main->goban_view, main->gobanPixmap, text, gobanx2viewx( main->sgfSZ + 1, main ), gobany2viewy( 4, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
        WriteText( main->goban_view, main->gobanPixmap, Gint2String(GetUsedLabelCount(main, FALSE) + 1), gobanx2viewx( main->sgfSZ + 1, main ), gobany2viewy( 5, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
        MarkSelected(main, main->sgfSZ + 1, 1);
      }
      if(main->CurrentMouseMode == main->circle_item) { /* Mouse mode to toggle circles */
        DrawCircle(main, -1, -1, STONE_WHITE);
        DrawTriangle(main, main->sgfSZ + 1, 0, STONE_WHITE);
        DrawSquare(main, main->sgfSZ + 1, 1, STONE_WHITE);
        DrawCircle(main, main->sgfSZ + 1, 2, STONE_WHITE);
        DrawCross(main, main->sgfSZ + 1, 3, STONE_WHITE);
        text[0] = 'A' + GetUsedLabelCount(main, TRUE);
        text[1] = '\0';
        WriteText( main->goban_view, main->gobanPixmap, text, gobanx2viewx( main->sgfSZ + 1, main ), gobany2viewy( 4, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
        WriteText( main->goban_view, main->gobanPixmap, Gint2String(GetUsedLabelCount(main, FALSE) + 1), gobanx2viewx( main->sgfSZ + 1, main ), gobany2viewy( 5, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
        MarkSelected(main, main->sgfSZ + 1, 2);
      }
      if(main->CurrentMouseMode == main->cross_item) { /* Mouse mode to toggle crosses */
        DrawCross(main, -1, -1, STONE_WHITE);
        DrawTriangle(main, main->sgfSZ + 1, 0, STONE_WHITE);
        DrawSquare(main, main->sgfSZ + 1, 1, STONE_WHITE);
        DrawCircle(main, main->sgfSZ + 1, 2, STONE_WHITE);
        DrawCross(main, main->sgfSZ + 1, 3, STONE_WHITE);
        text[0] = 'A' + GetUsedLabelCount(main, TRUE);
        text[1] = '\0';
        WriteText( main->goban_view, main->gobanPixmap, text, gobanx2viewx( main->sgfSZ + 1, main ), gobany2viewy( 4, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
        WriteText( main->goban_view, main->gobanPixmap, Gint2String(GetUsedLabelCount(main, FALSE) + 1), gobanx2viewx( main->sgfSZ + 1, main ), gobany2viewy( 5, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
        MarkSelected(main, main->sgfSZ + 1, 3);
      }
      if(main->CurrentMouseMode == main->labeling_item) { /* Mouse mode to set labels A,B,C,... */
        text[0] = 'A' + GetUsedLabelCount(main, TRUE);
        text[1] = '\0';
        WriteText( main->goban_view, main->gobanPixmap, text, gobanx2viewx( -1, main ), gobany2viewy( -1, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
        DrawTriangle(main, main->sgfSZ + 1, 0, STONE_WHITE);
        DrawSquare(main, main->sgfSZ + 1, 1, STONE_WHITE);
        DrawCircle(main, main->sgfSZ + 1, 2, STONE_WHITE);
        DrawCross(main, main->sgfSZ + 1, 3, STONE_WHITE);
        text[0] = 'A' + GetUsedLabelCount(main, TRUE);
        text[1] = '\0';
        WriteText( main->goban_view, main->gobanPixmap, text, gobanx2viewx( main->sgfSZ + 1, main ), gobany2viewy( 4, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
        WriteText( main->goban_view, main->gobanPixmap, Gint2String(GetUsedLabelCount(main, FALSE) + 1), gobanx2viewx( main->sgfSZ + 1, main ), gobany2viewy( 5, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
        MarkSelected(main, main->sgfSZ + 1, 4);
      }
      if(main->CurrentMouseMode == main->numbering_item) { /* Mouse mode to set labels 1,2,3,... */
        WriteText( main->goban_view, main->gobanPixmap, Gint2String(GetUsedLabelCount(main, FALSE) + 1), gobanx2viewx( -1, main ), gobany2viewy( -1, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
        DrawTriangle(main, main->sgfSZ + 1, 0, STONE_WHITE);
        DrawSquare(main, main->sgfSZ + 1, 1, STONE_WHITE);
        DrawCircle(main, main->sgfSZ + 1, 2, STONE_WHITE);
        DrawCross(main, main->sgfSZ + 1, 3, STONE_WHITE);
        text[0] = 'A' + GetUsedLabelCount(main, TRUE);
        text[1] = '\0';
        WriteText( main->goban_view, main->gobanPixmap, text, gobanx2viewx( main->sgfSZ + 1, main ), gobany2viewy( 4, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
        WriteText( main->goban_view, main->gobanPixmap, Gint2String(GetUsedLabelCount(main, FALSE) + 1), gobanx2viewx( main->sgfSZ + 1, main ), gobany2viewy( 5, main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
        MarkSelected(main, main->sgfSZ + 1, 5);
      }
      if(main->CurrentMouseMode == main->search_item) { /* Outcome view mode */
        DrawStone(main, -1, -1, main->playerInTurn); /* Show whose turn it is */
        DrawBranchMark(main, -1, -1);
        /* DrawSearchIcon(main); */
      }
      break;
	default:
	  break;
  }
  
  for(x=0;x<main->sgfSZ;x++) {
    for(y=0;y<main->sgfSZ;y++) {
      if(GetStoneXY(x,y) != STONE_NONE) {
        DrawStone(main, x, y, GetStoneXY(x,y));
      }
    }
  }
  
  if(main->currentNode != NULL) {
    points = main->currentNode->sgfMA;
    while(points != NULL) {
      DrawCross(main, Location2X(points->at), Location2Y(points->at), GetStoneAt(points->at));
      points = points->next;
    }
    points = main->currentNode->sgfTR;
    while(points != NULL) {
      DrawTriangle(main, Location2X(points->at), Location2Y(points->at), GetStoneAt(points->at));
      points = points->next;
    }
    points = main->currentNode->sgfCR;
    while(points != NULL) {
      DrawCircle(main, Location2X(points->at), Location2Y(points->at), GetStoneAt(points->at));
      points = points->next;
    }
    points = main->currentNode->sgfSQ;
    while(points != NULL) {
      DrawSquare(main, Location2X(points->at), Location2Y(points->at), GetStoneAt(points->at));
      points = points->next;
    }
    labels = main->currentNode->sgfLB;
    while(labels != NULL) {
      switch(GetStoneAt(labels->at)) {
        case STONE_BLACK:
          WriteText( main->goban_view, main->gobanPixmap, labels->label, gobanx2viewx( Location2X(labels->at), main ), gobany2viewy( Location2Y(labels->at), main ), main->colWidth, main->rowHeight,main->goban_view->style->white_gc);
          break;
        case STONE_WHITE:
          WriteText( main->goban_view, main->gobanPixmap, labels->label, gobanx2viewx( Location2X(labels->at), main ), gobany2viewy( Location2Y(labels->at), main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
          break;
        case STONE_NONE:
          DrawStone( main, Location2X(labels->at), Location2Y(labels->at), STONE_NONE);
          WriteText( main->goban_view, main->gobanPixmap, labels->label, gobanx2viewx( Location2X(labels->at), main ), gobany2viewy( Location2Y(labels->at), main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
          break;
		default:
		  break;
      }
      labels = labels->next;
    }
  }
  else {
    points = main->sgfMA;
    while(points != NULL) {
      DrawCross(main, Location2X(points->at), Location2Y(points->at), GetStoneAt(points->at));
      points = points->next;
    }
    points = main->sgfTR;
    while(points != NULL) {
      DrawTriangle(main, Location2X(points->at), Location2Y(points->at), GetStoneAt(points->at));
      points = points->next;
    }
    points = main->sgfCR;
    while(points != NULL) {
      DrawCircle(main, Location2X(points->at), Location2Y(points->at), GetStoneAt(points->at));
      points = points->next;
    }
    points = main->sgfSQ;
    while(points != NULL) {
      DrawSquare(main, Location2X(points->at), Location2Y(points->at), GetStoneAt(points->at));
      points = points->next;
    }
    labels = main->sgfLB;
    while(labels != NULL) {
      switch(GetStoneAt(labels->at)) {
        case STONE_BLACK:
          WriteText( main->goban_view, main->gobanPixmap, labels->label, gobanx2viewx( Location2X(labels->at), main ), gobany2viewy( Location2Y(labels->at), main ), main->colWidth, main->rowHeight,main->goban_view->style->white_gc);
          break;
        case STONE_WHITE:
          WriteText( main->goban_view, main->gobanPixmap, labels->label, gobanx2viewx( Location2X(labels->at), main ), gobany2viewy( Location2Y(labels->at), main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
          break;
        case STONE_NONE:
          DrawStone( main, Location2X(labels->at), Location2Y(labels->at), STONE_NONE);
          WriteText( main->goban_view, main->gobanPixmap, labels->label, gobanx2viewx( Location2X(labels->at), main ), gobany2viewy( Location2Y(labels->at), main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
          break;
		default:
		  break;
      }
      labels = labels->next;
    }
  }
  if(((main->sgfST / 2) == 0) && (main->CurrentMouseMode != main->training_item)) { /* Do board mark up */
    if((main->sgfST % 2) == 0) { /* Children view mode */
      if(main->currentNode == NULL) {
        node = main->child;
      }
      else {
        node = main->currentNode->child;
      }
    }
    else { /* Sibling view mode */
      node = main->currentNode;
    }
    if(node != NULL) {
      text[0] = 'A';
      text[1] = '\0';
      if(node != node->sibling) {
        if(main->currentNode == NULL) {
          node = main->child;
          do {
            switch(GetStoneAt(node->at)) {
              case STONE_BLACK:
              WriteText( main->goban_view, main->gobanPixmap, text, gobanx2viewx( Location2X(node->at), main ), gobany2viewy( Location2Y(node->at), main ), main->colWidth, main->rowHeight,main->goban_view->style->white_gc);
              break;
            case STONE_WHITE:
              WriteText( main->goban_view, main->gobanPixmap, text, gobanx2viewx( Location2X(node->at), main ), gobany2viewy( Location2Y(node->at), main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
              break;
            case STONE_NONE:
              DrawStone( main, Location2X(node->at), Location2Y(node->at), STONE_NONE);
              WriteText( main->goban_view, main->gobanPixmap, text, gobanx2viewx( Location2X(node->at), main ), gobany2viewy( Location2Y(node->at), main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
              break;
			default:
			  break;
            }
            if(node->nodeHeight >= node->sibling->nodeHeight) {
              break;
            }
            text[0]++;
            node = node->sibling;
          } while(TRUE);
        }
        else {
          if(node->parent == NULL) {
            node = main->child;
          }
          else {
            node = node->parent->child;
          }
          do {
            if( (IsLabelInList(node->at, main->currentNode->sgfLB) == FALSE) &&
                (IsPointInList(node->at, main->currentNode->sgfSQ) == FALSE) &&
                (IsPointInList(node->at, main->currentNode->sgfTR) == FALSE) &&
                (IsPointInList(node->at, main->currentNode->sgfMA) == FALSE) &&
                (IsPointInList(node->at, main->currentNode->sgfCR) == FALSE)) {
              switch(GetStoneAt(node->at)) {
                case STONE_BLACK:
                WriteText( main->goban_view, main->gobanPixmap, text, gobanx2viewx( Location2X(node->at), main ), gobany2viewy( Location2Y(node->at), main ), main->colWidth, main->rowHeight,main->goban_view->style->white_gc);
                break;
              case STONE_WHITE:
                WriteText( main->goban_view, main->gobanPixmap, text, gobanx2viewx( Location2X(node->at), main ), gobany2viewy( Location2Y(node->at), main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
                break;
              case STONE_NONE:
                DrawStone( main, Location2X(node->at), Location2Y(node->at), STONE_NONE);
                WriteText( main->goban_view, main->gobanPixmap, text, gobanx2viewx( Location2X(node->at), main ), gobany2viewy( Location2Y(node->at), main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
                break;
			  default:
			    break;
			  }
            }
            if(node->nodeHeight >= node->sibling->nodeHeight) {
              break;
            }
            text[0]++;
            node = node->sibling;
          } while(TRUE);
        }
      }
    }
  }
  
  /* Show stone numbering */
  /* numberingBase = findStartOfNumbering( main ); */
  if(numberingBase != NULL) {
    if((main->currentNode != NULL) && (gobanShifted == TRUE)) {
      /* WriteText( main->goban_view, main->gobanPixmap, MoveRange2String(numberingBase->nodeTurnNumber,main->currentNode->nodeTurnNumber), gobanx2viewx( main->sgfSZ / 2, main ), gobany2viewy( -1 , main ), main->colWidth * 4, main->rowHeight,main->goban_view->style->black_gc); */
      WriteText( main->goban_view, main->gobanPixmap, MoveRange2String(numberingBase->nodeTurnNumber,main->currentNode->nodeTurnNumber), gobanx2viewx( main->sgfSZ + 3, main ), gobany2viewy( -1 , main ), main->colWidth * 4, main->rowHeight,main->goban_view->style->black_gc);
    }
    if(main->resetNumbering == TRUE) {
      numberingOffset = 1 - numberingBase->nodeTurnNumber;
    }
    else {
      numberingOffset = -100 * ((numberingBase->nodeTurnNumber - 1) / 100);
    }
    node = main->currentNode; /* Numbering ends at current position */
    if(main->CurrentMouseMode == main->search_item) { /* If outcome view */
      if(node == NULL) {
        node = get_active(main->child);
      }
      if(node != NULL) {
        while(node->child != NULL) { /* find end of */
          node = get_active(node->child); /* active branch where numbering ends instead */
        }
      }
    }
    underStones = 0;
    while(node != NULL) {
      if( (IsLabelInList(node->at, main->currentNode->sgfLB) == FALSE) &&
          (IsPointInList(node->at, main->currentNode->sgfSQ) == FALSE) &&
          (IsPointInList(node->at, main->currentNode->sgfTR) == FALSE) &&
          (IsPointInList(node->at, main->currentNode->sgfMA) == FALSE) &&
          (IsPointInList(node->at, main->currentNode->sgfCR) == FALSE) &&
          (node->underTheStones == FALSE)) {
        switch(GetStoneAt(node->at)) {
          case STONE_TRANSPARENT_BLACK:
            break;
          case STONE_BLACK:
            WriteText( main->goban_view, main->gobanPixmap, Gint2String(node->nodeTurnNumber + numberingOffset), gobanx2viewx( Location2X(node->at), main ), gobany2viewy( Location2Y(node->at), main ), main->colWidth, main->rowHeight,main->goban_view->style->white_gc);
            break;
          case STONE_TRANSPARENT_WHITE:
            break;
          case STONE_WHITE:
            WriteText( main->goban_view, main->gobanPixmap, Gint2String(node->nodeTurnNumber + numberingOffset), gobanx2viewx( Location2X(node->at), main ), gobany2viewy( Location2Y(node->at), main ), main->colWidth, main->rowHeight,main->goban_view->style->black_gc);
            break;
		  default:
			break;
        }
      }
      else {
        underStones++;
      }
      if(node == numberingBase) {
        break;
      }
      node = node->parent;
    }
    node = main->currentNode;
    if(gobanShifted == TRUE) {
      while(node != NULL) {
        if( (IsLabelInList(node->at, main->currentNode->sgfLB) == TRUE) ||
            (IsPointInList(node->at, main->currentNode->sgfSQ) == TRUE) ||
            (IsPointInList(node->at, main->currentNode->sgfTR) == TRUE) ||
            (IsPointInList(node->at, main->currentNode->sgfMA) == TRUE) ||
            (IsPointInList(node->at, main->currentNode->sgfCR) == TRUE) ||
            (node->underTheStones == TRUE)) {
          WriteText( main->goban_view, main->gobanPixmap, WherePlayed2String(node->nodeTurnNumber + numberingOffset, node->at, main->sgfSZ), gobanx2viewx( main->sgfSZ + 3, main ), gobany2viewy( underStones - 1, main ), main->colWidth * 4, main->rowHeight,main->goban_view->style->black_gc);
          underStones--;
        }
        node = node->parent;
      }
    }
  }
  
  if((main->shownext == TRUE) && (main->CurrentMouseMode != main->training_item)) {
    if(main->currentNode == NULL) {
      node = main->child;
    }
    else {
      node = main->currentNode->child;
    }
    if(node != NULL) {
      firstNode = node;
      do {
        if(node->at >= 0) { /* If not PASS or SETUP */
          DrawBranchmark(main, Location2X(node->at), Location2Y(node->at), node->activeBranch);
        }
        node = node->sibling;
      } while(node != firstNode);
    }
  }
  
  /* Show markings for goban size and handicap amount */
  if((main->child == NULL) && (main->sgfAB == NULL) && (main->sgfAW == NULL)) {
    for(i=1;i<(main->sgfSZ - 1);i++) {
      MarkSelected(main, -1, i);
    }
    for(i=2;i<=9;i++) {
      if(i > main->sgfSZ) {
        break;
      }
      MarkSelected(main, main->sgfSZ, main->sgfSZ - i);
    }
    WriteText( main->goban_view, main->gobanPixmap, "Goban size", gobanx2viewx( 0, main ), gobany2viewy( main->sgfSZ, main ), main->colWidth * 4, main->rowHeight,main->goban_view->style->black_gc);
    WriteText( main->goban_view, main->gobanPixmap, "Handicap", gobanx2viewx( main->sgfSZ - 1, main ), gobany2viewy( main->sgfSZ, main ), main->colWidth * 4, main->rowHeight,main->goban_view->style->black_gc);
    bottomColumnSpaceUsed = TRUE;
  }
  
  /* Show branch marks */
  if(main->CurrentMouseMode == main->search_item) { /* Outcome view mode */
    node = main->currentNode;
    if(node == NULL) {
      node = main->child;
    }
    while(node != NULL) {
      if(node->child == NULL) {
        break;
      }
      node = forward_in_tree( node, main->child );
    }
    if((node != NULL) && (node != main->currentNode)) {
      node = node->parent;
    }
    while((node != NULL) && (node != main->currentNode)) { /* Show marks only at future nodes */
      if(node->child != node->child->sibling) {
        DrawBranchMark(main, Location2X(node->at), Location2Y(node->at));
      }
      node = node->parent;
    }
    if(main->currentNode != NULL) {
      if(main->currentNode->child != NULL) {
        if(main->currentNode->child != main->currentNode->child->sibling) {
          DrawBranchMark(main, Location2X(main->currentNode->at), Location2Y(main->currentNode->at));
        }
      }
    }
  }
  
  /* Show score estimation */
  scorex2 = ((main->capturedWhite - main->capturedBlack)*2) - main->sgfKMx2;
  if(main->currentNode != NULL) {  
    points = main->currentNode->sgfTB;
    while(points != NULL) {
      if(GetStoneAt(points->at) == STONE_WHITE) {
        scorex2 += 4; /* Dead white stone */
      }
      else {
        scorex2 += 2; /* Black territory */
      }
      DrawTerritory(main, Location2X(points->at), Location2Y(points->at), STONE_BLACK);
      points = points->next;
    }
    points = main->currentNode->sgfTW;
    while(points != NULL) {
      if(GetStoneAt(points->at) == STONE_BLACK) {
        scorex2 -= 4; /* Dead black stone */
      }
      else {
        scorex2 -= 2; /* Whote territory */
      }
      DrawTerritory(main, Location2X(points->at), Location2Y(points->at), STONE_WHITE);
      points = points->next;
    }
    if((main->currentNode->sgfTW != NULL) || (main->currentNode->sgfTB != NULL)) {
      main->currentNode->sgfVx2 = scorex2;
      haveEstimate = TRUE;
    }
    else {
      scorex2 = main->currentNode->sgfVx2;
      haveEstimate = FALSE;
    }
  }
  if(main->CurrentMouseMode == main->search_item) {
    node = main->currentNode;
    if(node == NULL) {
      node = main->child;
    }
    while(node != NULL) {
      if(node->child == NULL) {
        break;
      }
      node = forward_in_tree( node, main->child );
    }
    if(node == NULL) {
      haveEstimate = FALSE;
    }
    else {
      scorex2 = node->sgfVx2;
      if((node->sgfTW != NULL) || (node->sgfTB != NULL)) {
        haveEstimate = TRUE;
      }
      else {
        haveEstimate = FALSE;
      }
    }
  }
  if(main->currentNode != NULL) {
    if(main->currentNode->at == AT_IS_PASS) {
      if(topColumnSpaceUsed == FALSE) {
        if(main->currentNode->stone == STONE_BLACK) {
          WriteText( main->goban_view, main->gobanPixmap, "Black pass", gobanx2viewx(main->sgfSZ/2,main), titletop, main->colWidth * main->sgfSZ, main->rowHeight,NULL);
        }
        if(main->currentNode->stone == STONE_WHITE) {
          WriteText( main->goban_view, main->gobanPixmap, "White pass", gobanx2viewx(main->sgfSZ/2,main), titletop, main->colWidth * main->sgfSZ, main->rowHeight,NULL);
        }
        topColumnSpaceUsed = TRUE;
      }
    }
  }
  if((main->sgfDI != NULL || main->sgfGE != NULL) && topColumnSpaceUsed == FALSE) {
    WriteText( main->goban_view, main->gobanPixmap, ProblemText(main->sgfGE, main->sgfDI), gobanx2viewx(main->sgfSZ/2,main), titletop, main->colWidth * main->sgfSZ, main->rowHeight,NULL);
    topColumnSpaceUsed = TRUE;
  }
  if(((main->CurrentMouseMode == main->territory_item) || (main->CurrentMouseMode == main->search_item) || (haveEstimate == TRUE))) {
    if(scorex2 > 0) {
      if(topColumnSpaceUsed == FALSE) {
        WriteText( main->goban_view, main->gobanPixmap, Add2GintInString("Black +%d.%d", scorex2/2, (scorex2%2)*5), gobanx2viewx(main->sgfSZ/2,main), titletop, main->colWidth * main->sgfSZ, main->rowHeight,NULL);
        topColumnSpaceUsed = TRUE;
      }
      if(main->CurrentMouseMode == main->territory_item) {
        DrawTerritory(main, -1, -1, STONE_BLACK);
      }
    }
    if(scorex2 < 0) {
      if(topColumnSpaceUsed == FALSE) {
        WriteText( main->goban_view, main->gobanPixmap, Add2GintInString("White +%d.%d", (0-scorex2)/2, ((0-scorex2)%2)*5), gobanx2viewx(main->sgfSZ/2,main), titletop, main->colWidth * main->sgfSZ, main->rowHeight,NULL);
        topColumnSpaceUsed = TRUE;
      }
      if(main->CurrentMouseMode == main->territory_item) {
        DrawTerritory(main, -1, -1, STONE_WHITE);
      }
    }
    if(scorex2 == 0) {
      if(topColumnSpaceUsed == FALSE) {
        if(haveEstimate == TRUE) {
          WriteText( main->goban_view, main->gobanPixmap, "Jigo", gobanx2viewx(main->sgfSZ/2,main), titletop, main->colWidth * main->sgfSZ, main->rowHeight,NULL);
        }
        else {
          WriteText( main->goban_view, main->gobanPixmap, "No estimate", gobanx2viewx(main->sgfSZ/2,main), titletop, main->colWidth * main->sgfSZ, main->rowHeight,NULL);
        }
        topColumnSpaceUsed = TRUE;
      }
      if(main->CurrentMouseMode == main->territory_item) {
        DrawTerritory(main, -1, -1, STONE_NONE);
      }
    }
  }

  /* Show current move */
  MarkCurrentMove(main);

  /* Show goban orientation alternatives */
  if(main->temporal_mode == TEMPMODE_CHANGE_ORIENTATION) {
    if(main->currentNode != NULL) {
      for(i=0; i<8; i++) {
        at = RotateLocation(main->currentNode->at, i);
        if(GetStoneAt(at) != STONE_MARKED_NONE) {
          MarkSelected(main, Location2X(at), Location2Y(at));
          SetStoneAt(at, STONE_MARKED_NONE);
        }
      }
    }
  }

  for(i=0;i<main->sgfSZ;i++) {
	 if(topColumnSpaceUsed == FALSE) {
      WriteText( main->goban_view, main->gobanPixmap, X2ColumnTitle(i), gobanx2viewx(i,main), titletop, main->colWidth, main->rowHeight,NULL);
    }
    if(bottomColumnSpaceUsed == FALSE) {
      WriteText( main->goban_view, main->gobanPixmap, X2ColumnTitle(i), gobanx2viewx(i,main), titlebottom, main->colWidth, main->rowHeight,NULL);
    }
  } 

  
#ifndef PRODUCTION_VERSION  
  msg2log("Exit configure_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
}

void MarkBranch(MainView * main, Node * node) {
  if(node == NULL) {
    MarkFamily(main,main->child);
  }
  else {
    MarkFamily(main,node->child);
  }
}

void MarkFamily(MainView * main, Node * node) {
  Node * at = NULL;
  if(node == NULL) {
    return;
  }
  at = node;
  do {
    if((main->isBigTree == FALSE) || (main->bigTreeVariations == TRUE) || (at->activeBranch == TRUE)) {
      MarkFamily(main,at->child);
    }
    MarkSelectedNode(main,at);
    at = at->sibling;
  } while(at != node);
}

void MarkSelectedNode(MainView * main, Node * node) {
  GdkRectangle update_rect;
  GdkGC * gc;
  gint left = 0, top = 0;
#ifndef PRODUCTION_VERSION  
  msg2log("MarkSelectedNode", LOGLEVEL_INFO, LOGSCOPE_PATH | LOGSCOPE_GAMETREE);
#endif
  if(node == NULL) {
#ifndef PRODUCTION_VERSION  
  msg2log("node == NULL", LOGLEVEL_WARNING, LOGSCOPE_PATH | LOGSCOPE_GAMETREE);
#endif
    return; /* Do not mark offtree */
  }
  left = TURN2X(node->nodeDrawX);
  top = HEIGHT2Y(node->nodeHeight);
#ifndef PRODUCTION_VERSION  
  msg2logGint("left = %d", LOGLEVEL_INFO, LOGSCOPE_PATH | LOGSCOPE_GAMETREE, left);
  msg2logGint("top = %d", LOGLEVEL_INFO, LOGSCOPE_PATH | LOGSCOPE_GAMETREE, top);
#endif
  update_rect.x = left;
  update_rect.y = top;
  update_rect.width = draw_h_treescale;
  update_rect.height = draw_h_treescale;
  gc = gdk_gc_new(main->gametreePixmap);
  gdk_gc_set_function (gc, GDK_INVERT);
  gdk_draw_rectangle (main->gametreePixmap, gc, TRUE, update_rect.x, update_rect.y, update_rect.width, update_rect.height);
  gtk_widget_draw (main->gametree_view, &update_rect);
  /* g_free(gc); Is this neccessary? */
}

void MarkSelected(MainView* mainview, gint x, gint y) 
{
  GdkRectangle update_rect;
  GdkGC * gc;
  gint xcenter = 0, ycenter = 0, radius = 0, left = 0, top = 0;
  /* if((x < 0) || (y < 0) || (x >= mainview->sgfSZ) || (y >= mainview->sgfSZ)) { */
  if((x < -1) || (y < -1) || (x > (mainview->sgfSZ + 1)) || (y > mainview->sgfSZ)) {
    return; /* Do not mark far offboard */
  }
  xcenter = gobanx2viewx(x, mainview );
  ycenter = gobany2viewy(y, mainview );
  radius = mainview->stoneCenter;
  left = xcenter - radius;
  top = ycenter - radius;
  update_rect.x = left;
  update_rect.y = top;
  update_rect.width = radius + radius + 1;
  update_rect.height = radius + radius + 1;
  gc = gdk_gc_new(mainview->gobanPixmap);
  gdk_gc_set_function (gc, GDK_INVERT);
  gdk_draw_rectangle (mainview->gobanPixmap, gc, TRUE, update_rect.x, update_rect.y, update_rect.width, update_rect.height);
  gtk_widget_draw (mainview->goban_view, &update_rect);
}

void MarkStone(MainView* mainview, gint x, gint y, Stone * stone)
{
  GdkRectangle update_rect;
  GdkGC * gc;
  gint xcenter = 0, ycenter = 0, radius = 0, left = 0, top = 0;
  /* if((x < 0) || (y < 0) || (x >= mainview->sgfSZ) || (y >= mainview->sgfSZ)) { */
  if((x < -1) || (y < -1) || (x > mainview->sgfSZ) || (y > mainview->sgfSZ)) {
    return; /* Do not mark far offboard */
  }
  xcenter = gobanx2viewx(x, mainview );
  ycenter = gobany2viewy(y, mainview );
  radius = mainview->stoneCenter;
  left = xcenter - radius;
  top = ycenter - radius;
  update_rect.x = left;
  update_rect.y = top;
  update_rect.width = radius + radius + 1;
  update_rect.height = radius + radius + 1;
  gc = gdk_gc_new(mainview->gobanPixmap);
  gdk_gc_set_function (gc, GDK_INVERT);
  /* Let's calculate first the area on which stone is drawn at */
  left = xcenter - mainview->stoneCenter;
  top = ycenter - mainview->stoneCenter;
  gdk_draw_arc (mainview->gobanPixmap,
                gc,
                TRUE,
                left, top, mainview->stoneSize - 1, mainview->stoneSize - 1,
                0, 64 * 360);
  if((*stone == STONE_WHITE) || (*stone == STONE_TRANSPARENT_WHITE)) {
    gdk_draw_arc (mainview->gobanPixmap,
                gc,
                TRUE,
                left+2, top+2, mainview->stoneSize - 5, mainview->stoneSize - 5,
                0, 64 * 360);
  }
  gtk_widget_draw (mainview->goban_view, &update_rect);
}

void MarkArea(MainView* mainview, gint x1, gint y1, gint x2, gint y2)
{
  GdkRectangle update_rect;
  GdkGC * gc;
  gint xcenter = 0, ycenter = 0, radius = 0, left = 0, top = 0, aid = 0;
  
  if(x1 > x2) {
    aid = x1;
    x1 = x2;
    x2 = aid;
  }
  if(y1 > y2) {
    aid = y1;
    y1 = y2;
    y2 = aid;
  }
  if((x1 < -1) || (y1 < -1) || (x2 > mainview->sgfSZ) || (y2 > mainview->sgfSZ)) {
    return; /* Do not mark far offboard */
  }
  xcenter = gobanx2viewx(x1, mainview );
  ycenter = gobany2viewy(y1, mainview );
  radius = mainview->stoneCenter;
  left = xcenter - radius;
  top = ycenter - radius;
  update_rect.x = left;
  update_rect.y = top;
  update_rect.width = mainview->colWidth * (x2 - x1 + 1);
  update_rect.height = mainview->rowHeight * (y2 - y1 + 1);
  gc = gdk_gc_new(mainview->gobanPixmap);
  gdk_gc_set_function (gc, GDK_INVERT);
  gdk_draw_rectangle (mainview->gobanPixmap, gc, TRUE, update_rect.x, update_rect.y, update_rect.width, update_rect.height);
  update_rect.x += 3;
  update_rect.y += 3;
  update_rect.width -= 6;
  update_rect.height -= 6;
  gdk_draw_rectangle (mainview->gobanPixmap, gc, TRUE, update_rect.x, update_rect.y, update_rect.width, update_rect.height);
  update_rect.x -= 3;
  update_rect.y -= 3;
  update_rect.width += 6;
  update_rect.height += 6;
  gtk_widget_draw (mainview->goban_view, &update_rect);
}

void DrawCross(MainView* mainview, gint x, gint y, Stone color)
{
  gint xcenter = gobanx2viewx(x, mainview );
  gint ycenter = gobany2viewy(y, mainview );
  gint radius = (2 * mainview->stoneCenter) / 3;
  gint left = xcenter - radius;
  gint right = xcenter + radius;
  gint top = ycenter - radius;
  gint bottom = ycenter + radius;
  if(color == STONE_BLACK) {
    gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->white_gc,
                left, top, right, bottom);
    gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->white_gc,
                right, top, left, bottom);
  }
  else {
    gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                left, top, right, bottom);
    gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                right, top, left, bottom);
  }
}

void DrawBranchmark(MainView* mainview, gint x, gint y, gboolean active)
{
  gint xcenter = 0, ycenter = 0, side = 0, left = 0, right = 0, top = 0, bottom = 0;

  xcenter = gobanx2viewx(x, mainview );
  ycenter = gobany2viewy(y, mainview );
  left = xcenter - mainview->stoneCenter;
  top = ycenter - mainview->stoneCenter;
  right = xcenter + mainview->stoneCenter;
  bottom = ycenter + mainview->stoneCenter;
  side = mainview->stoneCenter / 2;
  gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                left, top, left+side, top);
  gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                left, top, left, top+side);
  gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                right, top, right-side, top);
  gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                right, top, right, top+side);
  gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                left, bottom, left+side, bottom);
  gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                left, bottom, left, bottom-side);
  gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                right, bottom, right-side, bottom);
  gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                right, bottom, right, bottom-side);
  if(active == TRUE) {
    gdk_draw_line (mainview->gobanPixmap,
                  mainview->goban_view->style->black_gc,
                  right-side, bottom, right, bottom-side);
    gdk_draw_line (mainview->gobanPixmap,
                  mainview->goban_view->style->black_gc,
                  left+side, bottom, left, bottom-side);
    gdk_draw_line (mainview->gobanPixmap,
                  mainview->goban_view->style->black_gc,
                  right-side, top, right, top+side);
    gdk_draw_line (mainview->gobanPixmap,
                  mainview->goban_view->style->black_gc,
                  left+side, top, left, top+side);
  }
}

void DrawTriangle(MainView* mainview, gint x, gint y, Stone color)
{
  gint xcenter = 0, ycenter = 0, radius = 0, left = 0, right = 0, top = 0, bottom = 0;

  xcenter = gobanx2viewx(x, mainview );
  ycenter = gobany2viewy(y, mainview );
  radius = (2 * mainview->stoneCenter) / 3;
  left = xcenter - radius;
  right = xcenter + radius;
  top = ycenter - radius;
  bottom = ycenter + radius;
  if(color == STONE_BLACK) {
    gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->white_gc,
                xcenter, top, right, bottom);
    gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->white_gc,
                xcenter, top, left, bottom);
    gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->white_gc,
                right, bottom, left, bottom);
  }
  else {
    gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                xcenter, top, right, bottom);
    gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                xcenter, top, left, bottom);
    gdk_draw_line (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                right, bottom, left, bottom);
  }
}

void DrawBranchMark(MainView* mainview, gint x, gint y)
{
  gint xcenter = 0, ycenter = 0, radius = 0, left = 0, right = 0, top = 0, bottom = 0;

  xcenter = gobanx2viewx(x, mainview );
  ycenter = gobany2viewy(y, mainview );
  radius = (2 * mainview->stoneCenter) / 3;
  left = xcenter - radius;
  right = xcenter + radius;
  top = ycenter;
  bottom = ycenter + radius;
  gdk_draw_line (mainview->gobanPixmap,
                mainview->indicateGC,
                xcenter, top, right, bottom);
  gdk_draw_line (mainview->gobanPixmap,
                mainview->indicateGC,
                right, top, left, top);
}

void DrawSquare(MainView* mainview, gint x, gint y, Stone color)
{
  gint xcenter = 0, ycenter = 0, radius = 0, left = 0, top = 0;

  xcenter = gobanx2viewx(x, mainview );
  ycenter = gobany2viewy(y, mainview );
  radius = (2 * mainview->stoneCenter) / 3;
  left = xcenter - radius;
  top = ycenter - radius;
  if(color == STONE_BLACK) {
    gdk_draw_rectangle (mainview->gobanPixmap,
                mainview->goban_view->style->white_gc,
                FALSE,
                left, top, radius + radius, radius + radius);
  }
  else {
    gdk_draw_rectangle (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                FALSE,
                left, top, radius + radius, radius + radius);
  }
}

/* Draw circle at the location */
void DrawCircle(MainView* mainview, gint x, gint y, Stone color)
{
  gint xcenter = 0, ycenter = 0, radius = 0, left = 0, top = 0;

  xcenter = gobanx2viewx(x, mainview );
  ycenter = gobany2viewy(y, mainview );
  radius = (2 * mainview->stoneCenter) / 3;
  left = xcenter - radius;
  top = ycenter - radius;
  if(color == STONE_BLACK) {
    gdk_draw_arc (mainview->gobanPixmap,
                mainview->goban_view->style->white_gc,
                FALSE,
                left, top, radius + radius, radius + radius,
                0, 64 * 360);
  }
  else {
    gdk_draw_arc (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                FALSE,
                left, top, radius + radius, radius + radius,
                0, 64 * 360);
  }
}

void DrawTerritory(MainView* mainview, gint x, gint y, Stone stone)
{
  gint xcenter = 0, ycenter = 0, left = 0, right = 0, top = 0, bottom = 0, width = 0, height = 0; 
#ifndef PRODUCTION_VERSION  
    msg2log("Enter DrawTerritory", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
    /* Let's calculate first the area on which stone is drawn at */
  xcenter = gobanx2viewx(x, mainview );
  ycenter = gobany2viewy(y, mainview );
  left = xcenter - (mainview->stoneCenter/2);
  right = xcenter + (mainview->stoneCenter/2);
  top = ycenter - (mainview->stoneCenter/2);
  bottom = ycenter + (mainview->stoneCenter/2);
  width = right - left;
  height = bottom - top;
	 switch (stone) {
	 case STONE_TRANSPARENT_WHITE:
	   ;
	 case STONE_WHITE:
      gdk_draw_arc (mainview->gobanPixmap,
                mainview->goban_view->style->white_gc,
                TRUE,
                left, top, width, height,
                0, 64 * 360);
      gdk_draw_arc (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                FALSE,
                left, top, width, height,
                0, 64 * 360);
	 	break;
	 case STONE_TRANSPARENT_BLACK:
	   ;
	 case STONE_BLACK:
      gdk_draw_arc (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                TRUE,
                left, top, width, height,
                0, 64 * 360);
	 	break;
	 case STONE_NONE:
      gdk_draw_arc (mainview->gobanPixmap,
                mainview->goban_view->style->white_gc,
                TRUE,
                left, top, width, height,
                64 * 90, 64 * 180);
      gdk_draw_arc (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                FALSE,
                left, top, width, height,
                0, 64 * 360);
      gdk_draw_arc (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                TRUE,
                left, top, width, height,
                64 * 270, 64 * 180);
	 	break;
	  default:
		break;
    }

#ifndef PRODUCTION_VERSION  
    msg2log("Exit DrawTerritory", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
}

/* draw stone or place of stone */
void DrawStone(MainView* mainview, gint x, gint y, Stone stone)
{
  gint xcenter = 0, ycenter = 0, left = 0, right = 0, top = 0, bottom = 0; 
#ifndef PRODUCTION_VERSION  
    msg2log("Enter drawStone", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
    /* Let's calculate first the area on which stone is drawn at */
  xcenter = gobanx2viewx(x, mainview );
  ycenter = gobany2viewy(y, mainview );
  left = xcenter - mainview->stoneCenter;
  right = xcenter + mainview->stoneCenter;
  top = ycenter - mainview->stoneCenter;
  bottom = ycenter + mainview->stoneCenter;
	 switch (stone) {
	 case STONE_WHITE:
      gdk_draw_arc (mainview->gobanPixmap,
                mainview->goban_view->style->white_gc,
                TRUE,
                left, top, mainview->stoneSize - 1, mainview->stoneSize - 1,
                0, 64 * 360);
      gdk_draw_arc (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                FALSE,
                left, top, mainview->stoneSize - 1, mainview->stoneSize - 1,
                0, 64 * 360);
	 	break;
	 case STONE_BLACK:
      gdk_draw_arc (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                TRUE,
                left, top, mainview->stoneSize - 1, mainview->stoneSize - 1,
                /* left - 1, top - 1, mainview->stoneSize + 1, mainview->stoneSize + 1, */
                0, 64 * 360);
	 	break;
	 case STONE_NONE:
      gdk_draw_arc (mainview->gobanPixmap,
                mainview->gobanGC,
                TRUE,
                left, top, mainview->stoneSize - 1, mainview->stoneSize - 1,
                /* left - 1, top - 1, mainview->stoneSize + 1, mainview->stoneSize + 1, */
                0, 64 * 360);
	 	break;
	 case STONE_TRANSPARENT_WHITE:
      gdk_draw_arc (mainview->gobanPixmap,
                mainview->goban_view->style->white_gc,
                FALSE,
                left, top, mainview->stoneSize - 1, mainview->stoneSize - 1,
                0, 64 * 360);
	 	break;
	 case STONE_TRANSPARENT_BLACK:
      gdk_draw_arc (mainview->gobanPixmap,
                mainview->goban_view->style->black_gc,
                FALSE,
                left, top, mainview->stoneSize - 1, mainview->stoneSize - 1,
                0, 64 * 360);
	 	break;
	 default:
        break;
    }

#ifndef PRODUCTION_VERSION  
    msg2log("Exit drawStone", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
}

void MarkCurrentMove(MainView* mainview)
{
  gint xcenter = 0, ycenter = 0, left = 0, right = 0, top = 0, bottom = 0; 
#ifndef PRODUCTION_VERSION  
    msg2log("Enter MarkCurrentMove", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
  if(mainview->currentNode == NULL) {
    return;
  }
  if(mainview->currentNode->at < 0) {
    return;
  }
    /* Let's calculate first the area on which stone is drawn at */
  xcenter = gobanx2viewx(Location2X(mainview->currentNode->at), mainview );
  ycenter = gobany2viewy(Location2Y(mainview->currentNode->at), mainview );
  left = xcenter - mainview->stoneCenter;
  right = xcenter + mainview->stoneCenter;
  top = ycenter - mainview->stoneCenter;
  bottom = ycenter + mainview->stoneCenter;
  gdk_draw_arc (mainview->gobanPixmap,
                mainview->indicateGC,
                FALSE,
                left, top, mainview->stoneSize - 1, mainview->stoneSize - 1,
                0, 64 * 360);
  gdk_draw_arc (mainview->gobanPixmap,
                mainview->indicateGC,
                FALSE,
                left+1, top+1, mainview->stoneSize - 3, mainview->stoneSize - 3,
                0, 64 * 360);
  gdk_draw_arc (mainview->gobanPixmap,
                mainview->indicateGC,
                FALSE,
                left, top, mainview->stoneSize - 1, mainview->stoneSize - 2,
                0, 64 * 360);
  gdk_draw_arc (mainview->gobanPixmap,
                mainview->indicateGC,
                FALSE,
                left, top, mainview->stoneSize - 2, mainview->stoneSize - 1,
                0, 64 * 360);
  gdk_draw_arc (mainview->gobanPixmap,
                mainview->indicateGC,
                FALSE,
                left, top+1, mainview->stoneSize - 1, mainview->stoneSize - 2,
                0, 64 * 360);
  gdk_draw_arc (mainview->gobanPixmap,
                mainview->indicateGC,
                FALSE,
                left+1, top, mainview->stoneSize - 2, mainview->stoneSize - 1,
                0, 64 * 360);
#ifndef PRODUCTION_VERSION  
    msg2log("Exit MarkCurrentMove", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
}

void DrawGametree(MainView * main)
{
  /* gint centerX = main->gametree_view->allocation.width / 2; */
  /* gint centerY = main->gametree_view->allocation.height / 2; */
  /* gint stoneDiameter = 1; */
  gint tmpScale = 800; 
  gint treeWidth = 0;
  gint treeHeight = 0;
  gint fitWidth = 0;
  gint fitHeight = 0;
  /* gboolean drawEntireTree = TRUE; */
  
#ifndef PRODUCTION_VERSION  
  msg2log("--== DrawGametree ==--", LOGLEVEL_INFO, LOGSCOPE_GAMETREE | LOGSCOPE_PATH);
#endif
  treeWidth = GetTreeWidth();
#ifndef PRODUCTION_VERSION  
  msg2logGint("treeWidth = %d", LOGLEVEL_INFO, LOGSCOPE_GAMETREE, treeWidth);
#endif
  treeHeight = GetTreeHeight();
#ifndef PRODUCTION_VERSION  
  msg2logGint("treeHeight = %d", LOGLEVEL_INFO, LOGSCOPE_GAMETREE, treeHeight);
#endif
  if((main->CurrentMouseMode == main->diagram_item) && (main->gridSize < MINIMUM_DIAGRAM_SCALE)) {
    draw_h_treescale = MINIMUM_DIAGRAM_SCALE; /* Minimum scale that show move numbers */
  }
  else {
    draw_h_treescale = main->gridSize; /* User selected minimum scale */
  }
#ifndef PRODUCTION_VERSION  
  msg2logGint("draw_h_treescale = %d", LOGLEVEL_INFO, LOGSCOPE_GAMETREE, draw_h_treescale);
#endif
  if(treeWidth > 0) {
    draw_h_treescale = main->gametree_view->allocation.width / treeWidth; /* Scale that fits width of entire game tree */
#ifndef PRODUCTION_VERSION  
  msg2logGint("draw_h_treescale = %d", LOGLEVEL_INFO, LOGSCOPE_GAMETREE, draw_h_treescale);
#endif
  }
  else {
#ifndef PRODUCTION_VERSION  
    msg2logGint("treeWidth = %d", LOGLEVEL_WARNING, LOGSCOPE_GAMETREE, treeWidth);
#endif
    return;
  }
  if(treeHeight > 0) {
    tmpScale = main->gametree_view->allocation.height / treeHeight; /* Scale that fits height of entire game tree */
#ifndef PRODUCTION_VERSION  
  msg2logGint("tmpScale = %d", LOGLEVEL_INFO, LOGSCOPE_GAMETREE, tmpScale);
#endif
    if(tmpScale < draw_h_treescale) {
      draw_h_treescale = tmpScale; /* Scale that fits both width and height of entire game tree */
#ifndef PRODUCTION_VERSION  
  msg2logGint("draw_h_treescale = %d", LOGLEVEL_INFO, LOGSCOPE_GAMETREE, draw_h_treescale);
#endif
    }
  }
  else {
#ifndef PRODUCTION_VERSION  
    msg2logGint("treeHeight = %d", LOGLEVEL_WARNING, LOGSCOPE_GAMETREE, treeHeight);
#endif
    return;
  }
  if((MINIMUM_DIAGRAM_SCALE > draw_h_treescale) && (main->CurrentMouseMode == main->diagram_item)) {
    draw_h_treescale = MINIMUM_DIAGRAM_SCALE; /* Diagram view minimum scale */
  }
  if(main->gridSize > draw_h_treescale) {
    draw_h_treescale = main->gridSize; /* User selected minimum scale */
#ifndef PRODUCTION_VERSION  
  msg2logGint("draw_h_treescale = %d", LOGLEVEL_INFO, LOGSCOPE_GAMETREE, draw_h_treescale);
#endif
  }
  if(draw_h_treescale == 0) {
#ifndef PRODUCTION_VERSION  
  msg2logGint("draw_h_treescale = %d", LOGLEVEL_WARNING, LOGSCOPE_GAMETREE, draw_h_treescale);
#endif
    return;
  }
  while(treeHeight > (G_MAXINT / draw_h_treescale)) {
    draw_h_treescale--;
  }
  if(draw_h_treescale == 0) {
#ifndef PRODUCTION_VERSION  
  msg2logGint("draw_h_treescale = %d", LOGLEVEL_WARNING, LOGSCOPE_GAMETREE, draw_h_treescale);
#endif
    return;
  }
  fitWidth = main->gametree_view->allocation.width / draw_h_treescale; /* Amount of nodes in horizontal direction that fits on game tree with chosen scale */
#ifndef PRODUCTION_VERSION  
  msg2logGint("fitWidth = %d", LOGLEVEL_INFO, LOGSCOPE_GAMETREE, fitWidth);
#endif
  fitHeight = main->gametree_view->allocation.height / draw_h_treescale; /* Amount of nodes in horizontal direction that fits on game tree with chosen scale */
#ifndef PRODUCTION_VERSION  
  msg2logGint("fitHeight = %d", LOGLEVEL_INFO, LOGSCOPE_GAMETREE, fitHeight);
#endif

  draw_h_vshift = 0;
  draw_h_hshift = 0;
  if(main->currentNode != NULL) { /* In case we are not at root */
    if(fitHeight < treeHeight) { /* If tree does not fit vertically */
      if(main->currentNode->nodeHeight > (fitHeight / 2)) { /* and current node in below central line of fitting view */
        draw_h_vshift = (fitHeight / 2) - main->currentNode->nodeHeight; /* shift view to get current node at center line */
      }
      if(draw_h_vshift < (fitHeight - treeHeight)) { /* if that would waste space at bottom */
        draw_h_vshift = fitHeight - treeHeight; /* shift view to show bottom most node at the bottom row */
      }
    }
    if(fitWidth < treeWidth) { /* If tree does not fit horizontally */
      if(main->currentNode->nodeDrawX > (fitWidth / 2)) { /* and current node in right of central line of fitting view */
        draw_h_hshift = (fitWidth / 2) - main->currentNode->nodeDrawX; /* shift view to get current node at center line */
      }
      if(draw_h_hshift < (fitWidth - treeWidth)) { /* if that would waste space at right */
        draw_h_hshift = fitWidth - treeWidth; /* shift view to show right most node at the right side */
      }
    }
  }

  if (main->gametreePixmap)
    g_object_unref (main->gametreePixmap);

  main->gametreePixmap = gdk_pixmap_new (main->gametree_view->window,
			   main->gametree_view->allocation.width,
			   main->gametree_view->allocation.height,
			   -1);
  gdk_draw_rectangle (main->gametreePixmap,
		      main->gametree_view->style->white_gc,
		      TRUE,
		      0, 0,
		      main->gametree_view->allocation.width,
		      main->gametree_view->allocation.height);
  DrawCurrentNodeAtTree(main); /* Mark current node */
  DrawStoneAtTree(main, NULL, TRUE); /* Draw the root node */
  DrawNode(main, main->child, main->child, TRUE, 0); /* Draw rest of the game tree */
}

void DrawStoneAtTree(MainView * main, Node * node, gboolean active)
{
  gint stoneDiameter = 0;
  gint numberFitArea = 0;
  gint leftX = 0;
  gint topY = 0;
  gint noteMarkX = 0;
  gint noteMarkY = 0;
  gint noteMarkSize = 0;
  gboolean isNote = FALSE;
  
  noteMarkSize = draw_h_treescale / 7;
  if(node == NULL) {
    noteMarkX = TURN2X(1) - noteMarkSize;
    noteMarkY = HEIGHT2Y(0) + noteMarkSize;
    isNote = (main->sgfC != NULL);
  }
  else {
    noteMarkX = TURN2X(node->nodeDrawX + 1) - noteMarkSize;
    noteMarkY = HEIGHT2Y(node->nodeHeight) + noteMarkSize;
    isNote = (node->sgfC != NULL);
  }
  if(isNote) {
    gdk_draw_line (main->gametreePixmap,
               main->indicateGC,
			      noteMarkX,
               noteMarkY - noteMarkSize,
			      noteMarkX,
               noteMarkY + noteMarkSize);
    gdk_draw_line (main->gametreePixmap,
               main->indicateGC,
			      noteMarkX - noteMarkSize/2,
               noteMarkY - noteMarkSize/2,
			      noteMarkX + noteMarkSize/2,
               noteMarkY + noteMarkSize/2);
    gdk_draw_line (main->gametreePixmap,
               main->indicateGC,
			      noteMarkX + noteMarkSize/2,
               noteMarkY - noteMarkSize/2,
			      noteMarkX - noteMarkSize/2,
               noteMarkY + noteMarkSize/2);
  }
  if(node == NULL) {
    return;
  }
  if((active == TRUE) || ((main->isBigTree == TRUE) && (main->bigTreeVariations == FALSE))) {
    stoneDiameter = (9 * draw_h_treescale) / 20; 
  }
  else {
    stoneDiameter = (9 * draw_h_treescale) / 40; 
  }
  leftX = TURN2X(node->nodeDrawX) + (draw_h_treescale/2) - stoneDiameter;
  topY = HEIGHT2Y(node->nodeHeight) + (draw_h_treescale/2) - stoneDiameter;
  numberFitArea = (stoneDiameter * 3) / 2;
  if((active == FALSE) && (main->isBigTree == TRUE) && (main->bigTreeVariations == FALSE)) {
    gdk_draw_arc (main->gametreePixmap,
                main->gametree_view->style->white_gc,
                TRUE,
                leftX,
                topY,
                stoneDiameter<<1, stoneDiameter<<1,
                0, 64 * 360);
    if(stoneDiameter >= 8) { /* Try to draw text only if it will fit on stone. This is important for performance. */
      if((main->isBigTree == FALSE) || (main->bigTreeVariations == TRUE)) {
        WriteText( main->gametree_view, main->gametreePixmap, Gint2String(node->nodeTurnNumber), TURN2X(node->nodeDrawX) + (draw_h_treescale/2), HEIGHT2Y(node->nodeHeight) + (draw_h_treescale/2), numberFitArea, numberFitArea,main->gametree_view->style->black_gc);
      }
      else {
        WriteText( main->gametree_view, main->gametreePixmap, At2String(node->at,main->sgfSZ), TURN2X(node->nodeDrawX) + (draw_h_treescale/2), HEIGHT2Y(node->nodeHeight) + (draw_h_treescale/2), numberFitArea, numberFitArea,main->gametree_view->style->black_gc);
      }
    }
    return;
  }
  if(node->stone == STONE_WHITE) {
    gdk_draw_arc (main->gametreePixmap,
                main->gametree_view->style->white_gc,
                TRUE,
                leftX,
                topY,
                stoneDiameter<<1, stoneDiameter<<1,
                0, 64 * 360);
      gdk_draw_arc (main->gametreePixmap,
                main->gametree_view->style->black_gc,
                FALSE,
                leftX,
                topY,
                stoneDiameter<<1, stoneDiameter<<1,
                0, 64 * 360);
    if(stoneDiameter >= 8) { /* Try to draw text only if it will fit on stone. This is important for performance. */
      if((main->isBigTree == FALSE) || (main->bigTreeVariations == TRUE)) {
        WriteText( main->gametree_view, main->gametreePixmap, Gint2String(node->nodeTurnNumber), TURN2X(node->nodeDrawX) + (draw_h_treescale/2), HEIGHT2Y(node->nodeHeight) + (draw_h_treescale/2), numberFitArea, numberFitArea,main->gametree_view->style->black_gc);
      }
      else {
        WriteText( main->gametree_view, main->gametreePixmap, At2String(node->at,main->sgfSZ), TURN2X(node->nodeDrawX) + (draw_h_treescale/2), HEIGHT2Y(node->nodeHeight) + (draw_h_treescale/2), numberFitArea, numberFitArea,main->gametree_view->style->black_gc);
      }
    }
  }
  if(node->stone == STONE_BLACK) {
    gdk_draw_arc (main->gametreePixmap,
                main->gametree_view->style->black_gc,
                TRUE,
                leftX,
                topY,
                stoneDiameter<<1, stoneDiameter<<1,
                0, 64 * 360);
    if(stoneDiameter >= 8) { /* Try to draw text only if it will fit on stone. This is important for performance. */
      if((main->isBigTree == FALSE) || (main->bigTreeVariations == TRUE)) {
        WriteText( main->gametree_view, main->gametreePixmap, Gint2String(node->nodeTurnNumber), TURN2X(node->nodeDrawX) + (draw_h_treescale/2), HEIGHT2Y(node->nodeHeight) + (draw_h_treescale/2), numberFitArea, numberFitArea,main->gametree_view->style->white_gc);
      }
      else {
        WriteText( main->gametree_view, main->gametreePixmap, At2String(node->at,main->sgfSZ), TURN2X(node->nodeDrawX) + (draw_h_treescale/2), HEIGHT2Y(node->nodeHeight) + (draw_h_treescale/2), numberFitArea, numberFitArea,main->gametree_view->style->white_gc);
      }
    }
  }
}

void DrawLineAtTree(MainView * main,gint fx, gint fy, gint tx, gint ty)
{
  gdk_draw_line (main->gametreePixmap,
               main->gametree_view->style->black_gc,
			      TURN2X(fx) + (draw_h_treescale/2),
               HEIGHT2Y(fy) + (draw_h_treescale/2),
			      TURN2X(tx) + (draw_h_treescale/2),
               HEIGHT2Y(ty) + (draw_h_treescale/2));
}

void DrawCurrentNodeAtTree(MainView * main)
{
  gint atX = 0;
  gint atY = 0;
  gint top = 0, left = 0, right = 0, bottom = 0;
  
  if(main->currentNode != NULL) {
    atX = main->currentNode->nodeDrawX;
    atY = main->currentNode->nodeHeight;
  }
  top = HEIGHT2Y(atY);
  left = TURN2X(atX);
  bottom = HEIGHT2Y(atY+1);
  right = TURN2X(atX+1);
  gdk_draw_line (main->gametreePixmap,
               main->gametree_view->style->black_gc,
               left,
			      top,
               right,
			      top);
  gdk_draw_line (main->gametreePixmap,
               main->gametree_view->style->black_gc,
               right,
			      bottom,
               left,
			      bottom);
  gdk_draw_line (main->gametreePixmap,
               main->gametree_view->style->black_gc,
               left,
			      top,
               left,
			      bottom);
  gdk_draw_line (main->gametreePixmap,
               main->gametree_view->style->black_gc,
               right,
			      top,
               right,
			      bottom);
}

void DrawNode(MainView * main, Node * node, Node * firstChild, gboolean activeParent, gint parentHeight)
{
  gboolean thisActive = TRUE;
  if(node == NULL) {
#ifndef PRODUCTION_VERSION  
    msg2log("DrawNode node == NULL", LOGLEVEL_WARNING, LOGSCOPE_GAMETREE);
#endif
    return;
  }
  if(main == NULL) {
#ifndef PRODUCTION_VERSION  
    msg2log("DrawNode main == NULL", LOGLEVEL_WARNING, LOGSCOPE_GAMETREE);
#endif
    return;
  }
  if(main->gametreePixmap == NULL) {
#ifndef PRODUCTION_VERSION  
    msg2log("DrawNode main->gametreePixmap == NULL", LOGLEVEL_WARNING, LOGSCOPE_GAMETREE);
#endif
    return;
  }
  if(main->gametree_view == NULL) {
#ifndef PRODUCTION_VERSION  
    msg2log("DrawNode main->gametree_view == NULL", LOGLEVEL_WARNING, LOGSCOPE_GAMETREE);
#endif
    return;
  }
  thisActive = (activeParent && node->activeBranch); /* Node is active only if it is on a selected branch and it's parent is active. */
  if((parentHeight + 1) >= node->nodeHeight) {
    /* Draw line from center of the node to center of the parent node */
    DrawLineAtTree(main,node->nodeDrawX - 1, parentHeight, node->nodeDrawX, node->nodeHeight);
    if(node->parent != NULL) {
      DrawLineAtTree(main,node->nodeDrawX - 1, parentHeight, node->parent->nodeDrawX, parentHeight);
    }
  }
  else {
    /* Draw line from center of the node to center of height - 1 at parent level */
    DrawLineAtTree(main,node->nodeDrawX - 1, node->nodeHeight - 1, node->nodeDrawX, node->nodeHeight);
  }
  if(parentHeight < node->nodeHeight - 1) {
    /* Draw vertical line at parent node center to node height - 1 at parent level */
    DrawLineAtTree(main,node->nodeDrawX - 1, parentHeight, node->nodeDrawX - 1, node->nodeHeight - 1);
  }
  if((node->child != NULL) && ((main->bigTreeVariations == TRUE) || (main->isBigTree == FALSE) || (node->activeBranch))) {
    DrawNode(main, node->child,node->child,thisActive,node->nodeHeight); /* Draw children of the node */
  }
  if(node->sibling != firstChild) {
    DrawNode(main, node->sibling,firstChild,activeParent,parentHeight); /* Draw siblings of the node */
  }
  /* So far only the sceleton of the game tree of the node has been drawn. Now it is time to draw stones on top of the sceleton. 
     Drawing sceleton before children and siblings and stones after that ensures that children and siblings do not draw lines over the stone. */
  if(node->at != AT_IS_SETUP) { /* If the node is not setup node */
    DrawStoneAtTree(main, node, thisActive); /* draw color of the stone played at the node */
  }
  /* if(main->currentNode == node) {
    / * Mark this node as current node * /
  } */
}

void DrawAll(MainView * mainview)
{
  GdkRectangle update_rect;
  LogLevel severity;

  if(mainview->compareNode == NULL) {
    gtk_widget_hide(GTK_WIDGET(mainview->gotohome_tb));
    if(mainview->CurrentMouseMode == mainview->play_item) {
      gtk_widget_show(GTK_WIDGET(mainview->add_tb));
    }
    if(mainview->CurrentMouseMode == mainview->training_item) {
      gtk_widget_show(GTK_WIDGET(mainview->gotofirst_tb));
    }
  }
  else {
    if(mainview->CurrentMouseMode == mainview->play_item) {
      gtk_widget_hide(GTK_WIDGET(mainview->add_tb));
    }
    if(mainview->CurrentMouseMode == mainview->training_item) {
      gtk_widget_hide(GTK_WIDGET(mainview->gotofirst_tb));
    }
    if((mainview->CurrentMouseMode != mainview->territory_item) && 
       (mainview->CurrentMouseMode != mainview->search_item)) {
      gtk_widget_show(GTK_WIDGET(mainview->gotohome_tb));
    }
    else {
      gtk_widget_hide(GTK_WIDGET(mainview->gotohome_tb));
    }
  }

  set_active(mainview->currentNode);
  SetPlayerInTurn(mainview);
  DrawGoban(mainview); /* Let's draw the goban from the scratch */
  update_rect.x = 0;
  update_rect.y = 0;
  update_rect.width = mainview->goban_view->allocation.width;
  update_rect.height = mainview->goban_view->allocation.height;
  gtk_widget_draw (mainview->goban_view, &update_rect);
  if((mainview->isBigTree == TRUE) && (mainview->bigTreeVariations == FALSE)) {
    MakePrunedLayout(mainview->child, 1);
  }
  else {
    MakeTreeLayout(mainview->child);
  }
  DrawGametree(mainview);
  update_rect.x = 0;
  update_rect.y = 0;
  update_rect.width = mainview->gametree_view->allocation.width;
  update_rect.height = mainview->gametree_view->allocation.height;
  gtk_widget_draw (mainview->gametree_view, &update_rect);
  update_rect.x = 0;
  update_rect.y = 0;
  /* update_rect.width = mainview->note_scrolledwindow->allocation.width;
  update_rect.height = mainview->note_scrolledwindow->allocation.height; */
  update_rect.width = 800;
  update_rect.height = 480;
  gtk_widget_draw (mainview->note_scrolledwindow, &update_rect);
  if(mainview->sgfRE != NULL) {
    gtk_widget_show (GTK_WIDGET(mainview->game_result_element));
    gtk_label_set_text(GTK_LABEL(mainview->game_result_element),mainview->sgfRE);
  }
  else {
    gtk_widget_hide (GTK_WIDGET(mainview->game_result_element));
  }
  if(mainview->sgfHA == 0) {
    gtk_frame_set_label(GTK_FRAME(mainview->black_frame),"Black, even game");
  }
  else {
    gtk_frame_set_label(GTK_FRAME(mainview->black_frame),AddGintInString("Black, %d handicap", mainview->sgfHA));
  }
  if(mainview->sgfKMx2 == 0) {
    gtk_frame_set_label(GTK_FRAME(mainview->white_frame),"White, no komi");
  }
  else {
    gtk_frame_set_label(GTK_FRAME(mainview->white_frame),Add2GintInString("White, %d.%d komi", mainview->sgfKMx2/2, (mainview->sgfKMx2%2)*5));
  }
  if(mainview->capturedBlack == 0) {
    gtk_label_set_text(GTK_LABEL(mainview->white_prisoners_text),"No prisoners");
  }
  else {
    if(mainview->capturedBlack == 1) {
      gtk_label_set_text(GTK_LABEL(mainview->white_prisoners_text),"1 prisoner");
    }
    else {
      gtk_label_set_text(GTK_LABEL(mainview->white_prisoners_text),AddGintInString("%d prisoners", mainview->capturedBlack));
    }
  }
  if(mainview->capturedWhite == 0) {
    gtk_label_set_text(GTK_LABEL(mainview->black_prisoners_text),"No prisoners");
  }
  else {
    if(mainview->capturedWhite == 1) {
      gtk_label_set_text(GTK_LABEL(mainview->black_prisoners_text),"1 prisoner");
    }
    else {
      gtk_label_set_text(GTK_LABEL(mainview->black_prisoners_text),AddGintInString("%d prisoners", mainview->capturedWhite));
    }
  }
  if(mainview->currentNode == NULL) {
    gtk_label_set_text(GTK_LABEL(mainview->analysis_tab),"Ana");
  }
  else {
    gtk_label_set_text(GTK_LABEL(mainview->analysis_tab),Gint2String(mainview->currentNode->nodeTurnNumber));
  }
  gtk_entry_set_text(GTK_ENTRY(mainview->black_name),mainview->sgfPB);
  gtk_entry_set_text(GTK_ENTRY(mainview->white_name),mainview->sgfPW);
  severity = GetSeverestMsg();
  gtk_widget_hide (mainview->current_debug_tab);
  switch (severity) {
    case LOGLEVEL_NONE:
        mainview->current_debug_tab = mainview->debug_tab;
      break;
    case LOGLEVEL_ERROR:
        mainview->current_debug_tab = mainview->debug_error_tab;
      break;
    case LOGLEVEL_ABOUT:
        mainview->current_debug_tab = mainview->debug_about_tab;
      break;
    case LOGLEVEL_WARNING:
        mainview->current_debug_tab = mainview->debug_warn_tab;
      break;
    case LOGLEVEL_STATISTICS:
        mainview->current_debug_tab = mainview->debug_stat_tab;
      break;
    case LOGLEVEL_HELP:
        mainview->current_debug_tab = mainview->debug_help_tab;
      break;
    case LOGLEVEL_INFO:
        mainview->current_debug_tab = mainview->debug_info_tab;
      break;
    case LOGLEVEL_NOT_USED:
        mainview->current_debug_tab = mainview->debug_tab;
      break;
  }
  gtk_widget_show (mainview->current_debug_tab);
}

gint GetUsedLabelCount(MainView * main, gboolean countLetters) {
  Labellist * llist = NULL;
  gint count = 0;
  if(main->currentNode == NULL) {
    llist = main->sgfLB;
  }
  else {
    llist = main->currentNode->sgfLB;
  }
  while(llist != NULL) {
    if((llist->label[0] < '0') || (llist->label[0] > '9')) {
      if(countLetters == TRUE) {
        count++;
      }
    }
    else {
      if(countLetters == FALSE) {
        count++;
      }
    }
    llist = llist->next;
  }
  return count;
}
