/*
 * This file is part of goban770
 *
 * Copyright (C) 2005 Nokia Corporation.
 * Copyright (C) 2006,2007 Jarmo Ahosola.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <utils/log.h>
#include <ui/interface.h>
#include <ui/callbacks.h>
#include <game/gametree.h>
#include <gtk/gtk.h>
#include <libintl.h>

#include <hildon/hildon-file-chooser-dialog.h>
#include <hildon/hildon-note.h>
#include <hildon/hildon-font-selection-dialog.h>

#include <time.h>
#include <stdlib.h>

#include <game/goban.h>
/* Privates: */

void CreateLogNotePage( MainView *main );
void CreateAnalysisNotePage( MainView *main );
static void create_menu( MainView *main );
static void create_toolbar( MainView *main );
void create_textarea( MainView *main );
void create_layout( MainView *main );
void setup_game( MainView *main );

/* Creates and initialises a main_view */
MainView* interface_main_view_new( AppData *data )
{
    /* Zero memory with g_new0 */
    MainView* result = g_new0( MainView, 1 );
    /* Create GTK vbox which will be our text area */
    GtkWidget *main_vbox = gtk_vbox_new( FALSE, 0 );
    result->clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
    gtk_clipboard_set_can_store(result->clipboard, NULL, 0);

    /* Initialize interface data */
    result->file_edited = FALSE;
    /* Store handle to app's data to view's data */
    result->data = data;
    /* Store handle to hildon_window to app's data */
    result->data->window = HILDON_WINDOW(hildon_window_new());
    /* Set up game information */
    setup_game( result );
    /* Create all our necessary parts */
    create_layout( result );
    /* create_textarea( result ); */
    create_toolbar( result ); 
    create_menu( result );
    /* Put GTK vbox to app's view */
    gtk_container_add( GTK_CONTAINER( result->data->window ), main_vbox );
    /* Put scrolledwindow to GTK vbox and show it */
    gtk_box_pack_start( GTK_BOX( main_vbox ), result->toplevel, TRUE, TRUE, 0 );
    /* gtk_widget_set_size_request (result->aspectframe, result->toplevel->allocation.height, result->toplevel->allocation.height * GOBAN_ASPECTRATIO); / * Request initial goban size to be as big as there fits */ 
    gtk_widget_show( main_vbox );
    /* Focus on goban */    
    gtk_widget_grab_focus( GTK_WIDGET(result->goban_view ));
    /* Initialize our textview's file (no file opened yet) */
    result->file_edited=FALSE;
    result->file_name=NULL;
    /* Add top level call back functions */
    g_signal_connect(G_OBJECT(result->data->window), "key_press_event", G_CALLBACK(key_press_cb), result);
    /* Set default values for goban */
    result->sgfSZ = 19;
    result->unsupportedSgf = NULL;
    InitGoban(result->sgfSZ);
    result->koAt = -1;
    result->mouseAtGoban = FALSE;
    result->gridSize = DEFAULT_SCALE;
    result->temporal_mode = TEMPMODE_NONE;
    result->noteNode = NULL;
    /* result->filesToCycle = NULL; */
    /* result->directory = NULL; */
    result->playlist_radio = NULL;
    result->firstPlaylist = NULL;
    result->currentPlaylist = NULL;
    ResetGame(result->sgfSZ, result);
    /* Set up clock */
    result->clockStartTime = 0;
    result->clockStopTime = 0;   
    result->clockRunning = FALSE;
    result->clockInUse = FALSE;
    result->mainTime = 3900; /* 65 minutes. Well, that is my tournament maintime at 24.2.2007 :-) */
    /* gtk_widget_hide (result->stopClock_button); */
    /* Initialize random number generator */
    result->clockAtStartUp = time(NULL);
    srand(result->clockAtStartUp); /* randomize(); */
#ifndef PRODUCTION_VERSION  
    msg2log("Copyright (C) 2005 Nokia Corporation.\nCopyright (C) 2006,2007 Jarmo Ahosola.\n\nThanks to:\nLauri Paatero 3 dan, GoWrite author, for hints how to design sgf-editor.\nMirel Florescu 6 dan, Tsurukame go-school certified go-instructor, for instructions how to study go.\nTom Geiger, Ibon Igartua and Im, Hyo Jun, for beta testing and ideas.\nGraham Philips, of valuable pointers.\nAnd some others whose names I didn't find.", LOGLEVEL_ABOUT, LOGSCOPE_USER);
    msg2log("\n\nThis software is free software; you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License as published by the Free Software Foundation; either version 2.1 of the License, or (at your option) any later version.\n\nThis software is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details.\n\nYou should have received a copy of the GNU Lesser General Public License along with this software; if not, write to the Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA", LOGLEVEL_ABOUT, LOGSCOPE_USER);
#endif
    /* g_timeout_add_seconds(1,OnceEverySecond,result); DEFINITION IS MISSING. Where to get it? */
    /* Return with new mainview handle */
    
    /* Maximize goban by default */ 
    result->fullscreen = TRUE;
    gtk_window_fullscreen(GTK_WINDOW(result->data->window));
    gtk_widget_show(GTK_WIDGET(result->saveas_tb));
    /* Maximize goban end */
    return result;
}

/* clean up the allocated memory */
void interface_main_view_destroy( MainView *main )
{
	 kill_family( main->child ); /* destroy entire gametree */
    g_free( main );
}

/* File chooser */
gchar* interface_file_chooser(MainView * mainview, GtkFileChooserAction action)
{
    GtkWidget *dialog;
    gchar* filename = NULL;

    dialog = hildon_file_chooser_dialog_new( GTK_WINDOW(mainview->data->program), action);
    gtk_widget_show_all (GTK_WIDGET(dialog));
    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
        filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
    }
    gtk_widget_destroy (dialog);
    return filename;
}

/* Save changes note */
gint interface_save_changes_note(MainView * main)
{
    HildonNote *hn = NULL;
    gint response = FALSE;

    g_assert(main != NULL && main->data->program != NULL);

    hn = HILDON_NOTE(hildon_note_new_confirmation_add_buttons
                     (GTK_WINDOW(main->data->program),
                      _("maemopad_save_changes_made"),
                      _("maemopad_yes"), CONFRESP_YES,
                      _("maemopad_no"), CONFRESP_NO,
                      NULL, NULL));
    response = gtk_dialog_run(GTK_DIALOG(hn));
    gtk_widget_destroy(GTK_WIDGET(hn));
    return response;
}

/* Error note */
void interface_error( MaemopadError me, MainView *main )
{
    GtkWidget *dialog = NULL;
    gchar *msg = NULL;

    switch (me) {

    case MAEMOPAD_ERROR_INVALID_URI:
        msg = g_strdup( _("maemopad_invalid_uri") );
        break;

    case MAEMOPAD_ERROR_SAVE_FAILED:
        msg = g_strdup( _("maemopad_save_failed") );
        break;

    case MAEMOPAD_ERROR_OPEN_FAILED:
        msg = g_strdup( _("maemopad_open_failed") );
        break;

    case MAEMOPAD_NO_ERROR:
        return;

    default:
        g_assert_not_reached();
    }

    dialog = hildon_note_new_information( GTK_WINDOW( main->data->program ), msg );

    gtk_dialog_run( GTK_DIALOG( dialog ) );
    gtk_widget_destroy( dialog );
    g_free( msg );
}

/* Privates */

/* Create the menu items needed for the drop down menu */
static void create_menu( MainView *main )
{
    /* Create needed handles */
    GtkMenu *main_menu;
    GtkWidget *file_menu, *edit_menu, *game_edit_menu, *options_menu, *load_options_menu, *result_menu;
    GtkWidget *separator = NULL;
    GtkWidget *separator2 = NULL;
    GtkWidget *close = NULL;

    /* Create popup menu and it's content */
    main->mousemode_menu = gtk_menu_new ();
    main->play_item = gtk_radio_menu_item_new_with_label ( NULL, "Play mode" );
    main->mousemode_radio = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM (main->play_item));
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (main->play_item), TRUE);
    main->CurrentMouseMode = main->play_item;
    gtk_menu_append( main->mousemode_menu, main->play_item );
    main->training_item = gtk_radio_menu_item_new_with_label ( main->mousemode_radio, "Training mode" );
    main->mousemode_radio = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM (main->training_item));
    gtk_menu_append( main->mousemode_menu, main->training_item );
    main->territory_item = gtk_radio_menu_item_new_with_label ( main->mousemode_radio, "Territory counting" );
    main->mousemode_radio = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM (main->territory_item));
    gtk_menu_append( main->mousemode_menu, main->territory_item );
    main->search_item = gtk_radio_menu_item_new_with_label ( main->mousemode_radio, "Outcome view" );
    main->mousemode_radio = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM (main->search_item));
    gtk_menu_append( main->mousemode_menu, main->search_item );
    main->diagram_item = gtk_radio_menu_item_new_with_label ( main->mousemode_radio, "Diagram view" );
    main->mousemode_radio = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM (main->diagram_item));
    gtk_menu_append( main->mousemode_menu, main->diagram_item );
    main->addblack_item = gtk_radio_menu_item_new_with_label ( main->mousemode_radio, "Add setup stones" );
    main->mousemode_radio = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM (main->addblack_item));
    gtk_menu_append( main->mousemode_menu, main->addblack_item );
    main->addwhite_item = gtk_radio_menu_item_new_with_label ( main->mousemode_radio, "Add white stones" );
    /*
    main->mousemode_radio = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM (main->addwhite_item));
    gtk_menu_append( main->mousemode_menu, main->addwhite_item );
    */
    main->numbering_item = gtk_radio_menu_item_new_with_label ( main->mousemode_radio, "Numbering mode" );
    /*
    main->mousemode_radio = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM (main->numbering_item));
    gtk_menu_append( main->mousemode_menu, main->numbering_item );
    */
    main->labeling_item = gtk_radio_menu_item_new_with_label ( main->mousemode_radio, "Labeling mode" );
    /*
    main->mousemode_radio = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM (main->labeling_item));
    gtk_menu_append( main->mousemode_menu, main->labeling_item );
    */
    main->triangle_item = gtk_radio_menu_item_new_with_label ( main->mousemode_radio, "Edit stone decorations" );
    main->mousemode_radio = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM (main->triangle_item));
    gtk_menu_append( main->mousemode_menu, main->triangle_item );
    main->square_item = gtk_radio_menu_item_new_with_label ( main->mousemode_radio, "Toggle squares" );
    /*
    main->mousemode_radio = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM (main->square_item));
    gtk_menu_append( main->mousemode_menu, main->square_item );
    */
    main->circle_item = gtk_radio_menu_item_new_with_label ( main->mousemode_radio, "Toggle circles" );
    /*
    main->mousemode_radio = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM (main->circle_item));
    gtk_menu_append( main->mousemode_menu, main->circle_item );
    */
    main->cross_item = gtk_radio_menu_item_new_with_label ( main->mousemode_radio, "Toggle crosses" );
    /*
    main->mousemode_radio = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM (main->cross_item));
    gtk_menu_append( main->mousemode_menu, main->cross_item );
    */
    gtk_widget_show_all( main->mousemode_menu );
    g_signal_connect( G_OBJECT( main->play_item ), "activate", 
		      G_CALLBACK ( callback_mousemode), main );
    g_signal_connect( G_OBJECT( main->training_item ), "activate", 
		      G_CALLBACK ( callback_mousemode), main );
    g_signal_connect( G_OBJECT( main->territory_item ), "activate", 
		      G_CALLBACK ( callback_mousemode), main );
    g_signal_connect( G_OBJECT( main->search_item ), "activate", 
		      G_CALLBACK ( callback_mousemode), main );
    g_signal_connect( G_OBJECT( main->diagram_item ), "activate", 
		      G_CALLBACK ( callback_mousemode), main );
    g_signal_connect( G_OBJECT( main->addblack_item ), "activate", 
		      G_CALLBACK ( callback_mousemode), main );
	 /*
    g_signal_connect( G_OBJECT( main->addwhite_item ), "activate", 
		      G_CALLBACK ( callback_mousemode), main );
    g_signal_connect( G_OBJECT( main->numbering_item ), "activate", 
		      G_CALLBACK ( callback_mousemode), main );
    g_signal_connect( G_OBJECT( main->labeling_item ), "activate", 
		      G_CALLBACK ( callback_mousemode), main );
    */
    g_signal_connect( G_OBJECT( main->triangle_item ), "activate", 
		      G_CALLBACK ( callback_mousemode), main );
    /*
    g_signal_connect( G_OBJECT( main->square_item ), "activate", 
		      G_CALLBACK ( callback_mousemode), main );
    g_signal_connect( G_OBJECT( main->circle_item ), "activate", 
		      G_CALLBACK ( callback_mousemode), main );
    g_signal_connect( G_OBJECT( main->cross_item ), "activate", 
		      G_CALLBACK ( callback_mousemode), main );
    */

    /* Create main menu and new menus for submenus in our drop down menu */
    main_menu = GTK_MENU( gtk_menu_new () );
    file_menu = gtk_menu_new ();
    edit_menu = gtk_menu_new ();
    game_edit_menu = gtk_menu_new ();
    options_menu = gtk_menu_new ();
    load_options_menu = gtk_menu_new ();
    result_menu = gtk_menu_new ();
    main->playlists_menu = gtk_menu_new ();

    /* Create the menu items */
    main->file_item = gtk_menu_item_new_with_label ( _("File") );
    main->new_item = gtk_menu_item_new_with_label ( _("New") );
    main->open_item = gtk_menu_item_new_with_label ( _("Open") );
    main->save_item = gtk_menu_item_new_with_label ( _("Save") );
    main->saveas_item = gtk_menu_item_new_with_label ( _("Save As...") );
    main->edit_item = gtk_menu_item_new_with_label ( _("Edit") );
    main->cut_item = gtk_menu_item_new_with_label ( _("Cut") );
    main->copy_item = gtk_menu_item_new_with_label ( _("Copy") );
    main->paste_item = gtk_menu_item_new_with_label ( _("Paste") );
    /* main->font_item = gtk_menu_item_new_with_label ( _("Font") ); */
    main->fullscreen_item = gtk_menu_item_new_with_label ( _("Full Screen") );
    /* main->propose_item = gtk_menu_item_new_with_label ( "Find analysis mainline" ); */
    main->compare_item = gtk_menu_item_new_with_label ( "Mark position" );
    main->hint_item = gtk_menu_item_new_with_label ( "Give hint" );
    main->playlist_item = gtk_menu_item_new_with_label ( "Playlists" );
    main->refresh_playlist_item = gtk_menu_item_new_with_label ( "Refresh current playlist" );
    main->thumb_playlist_item = gtk_menu_item_new_with_label ( "Create thumbnails" );
    main->game_edit_item = gtk_menu_item_new_with_label ( "Edit game" );
    main->prune_item = gtk_menu_item_new_with_label ( "Prune variations" );
    main->rotate_item = gtk_menu_item_new_with_label ( "Change goban orientation" );
    main->circle_lp_item = gtk_menu_item_new_with_label ( "Circle last played" );
    main->remove_cm_item = gtk_menu_item_new_with_label ( "Clear markings of uncommented positions" );
    main->fill_item = gtk_menu_item_new_with_label ( "Fill goban" );
    main->create_problem_item = gtk_menu_item_new_with_label ( "Create problem" );
    main->set_result_item = gtk_menu_item_new_with_label ( "Set game result" );
    main->result_bwinr_item = gtk_menu_item_new_with_label ( "B+R" );
    main->result_bwint_item = gtk_menu_item_new_with_label ( "B+T" );
    main->result_bwinf_item = gtk_menu_item_new_with_label ( "B+Forfeit" );
    main->result_wwinr_item = gtk_menu_item_new_with_label ( "W+R" );
    main->result_wwint_item = gtk_menu_item_new_with_label ( "W+T" );
    main->result_wwinf_item = gtk_menu_item_new_with_label ( "W+Forfeit" );
    main->result_void_item = gtk_menu_item_new_with_label ( "Void" );
    main->result_unknown_item = gtk_menu_item_new_with_label ( "Unknown" );
    main->result_estimate_item = gtk_menu_item_new_with_label ( "Take result from score estimate" );
    main->options_item = gtk_menu_item_new_with_label ( "Options" );
    main->shownext_item = gtk_check_menu_item_new_with_label ( "Show next moves" );
    main->boardmarkup_item = gtk_check_menu_item_new_with_label ( "Board mark up" );
    main->childview_item = gtk_check_menu_item_new_with_label ( "Sibling variations" );
    main->diagramview_item = gtk_check_menu_item_new_with_label ( "Number stones" );
    main->resetnumbering_item = gtk_check_menu_item_new_with_label ( "Reset numbering at first number" );
    main->numberNPlays_item = gtk_check_menu_item_new_with_label ( "Number n last plays" );
    main->rndOrientation_item = gtk_check_menu_item_new_with_label ( "Randomize go-problems orientation" );
    main->autoRefute_item = gtk_check_menu_item_new_with_label ( "Analysis autorefute" );
    main->alphaTesting_item = gtk_check_menu_item_new_with_label ( "Alpha testing" );
    main->manualOnly_item = gtk_check_menu_item_new_with_label ( "Only manual tab control" );
    main->bigTreeVariations_item = gtk_check_menu_item_new_with_label ( "Show big tree variations" );
    main->load_options_item = gtk_menu_item_new_with_label ( "File open options" );
    main->merge_load_item = gtk_check_menu_item_new_with_label ( "Merge files" );
    main->minimal_load_item = gtk_check_menu_item_new_with_label ( "Read plays only" );
    main->omit_empty_nodes_item = gtk_check_menu_item_new_with_label ( "Omit nodes that have no plays or setup stones" );
    main->try_load_item = gtk_check_menu_item_new_with_label ( "Find problematic sgf-syntax" );
    main->problem2step_item = gtk_check_menu_item_new_with_label ( "Autoreply at go-problems" );
    main->problemTraining_item = gtk_check_menu_item_new_with_label ( "Switch to training if go-problem" );
    separator = gtk_separator_menu_item_new();
    separator2 = gtk_separator_menu_item_new();
/*    help = gtk_menu_item_new_with_label( _("Help") ); */
    close = gtk_menu_item_new_with_label( _("Close") );

    /* Add menu items to right menus */
    gtk_menu_append( main_menu, main->file_item );
    gtk_menu_append( file_menu, main->new_item );
    gtk_menu_append( file_menu, main->open_item );
    gtk_menu_append( file_menu, main->save_item );
    gtk_menu_append( file_menu, main->saveas_item );
    gtk_menu_append( main_menu, main->edit_item );
    gtk_menu_append( edit_menu, main->cut_item );
    gtk_menu_append( edit_menu, main->copy_item );
    gtk_menu_append( edit_menu, main->paste_item );
    /* gtk_menu_append( main_menu, main->font_item ); */
    gtk_menu_append( main_menu, main->fullscreen_item );
    /* gtk_menu_append( main_menu, main->propose_item ); */
    gtk_menu_append( main_menu, main->compare_item );
    gtk_menu_append( main_menu, main->hint_item );
    gtk_menu_append( main_menu, main->game_edit_item );
    gtk_menu_append( main_menu, main->playlist_item );
    gtk_menu_append( main->playlists_menu, main->refresh_playlist_item );
    gtk_menu_append( main->playlists_menu, main->thumb_playlist_item );
    gtk_menu_append( main->playlists_menu, separator2 );
    gtk_menu_append( game_edit_menu, main->prune_item );
    gtk_menu_append( game_edit_menu, main->rotate_item );
    gtk_menu_append( game_edit_menu, main->circle_lp_item );
    gtk_menu_append( game_edit_menu, main->remove_cm_item );
    gtk_menu_append( game_edit_menu, main->fill_item );
    gtk_menu_append( game_edit_menu, main->create_problem_item );
    gtk_menu_append( game_edit_menu, main->set_result_item );
    gtk_menu_append( result_menu, main->result_bwinr_item );
    gtk_menu_append( result_menu, main->result_bwint_item );
    gtk_menu_append( result_menu, main->result_bwinf_item );
    gtk_menu_append( result_menu, main->result_wwinr_item );
    gtk_menu_append( result_menu, main->result_wwint_item );
    gtk_menu_append( result_menu, main->result_wwinf_item );
    gtk_menu_append( result_menu, main->result_void_item );
    gtk_menu_append( result_menu, main->result_unknown_item );
    gtk_menu_append( result_menu, main->result_estimate_item );
    gtk_menu_append( main_menu, main->options_item );
    gtk_menu_append( options_menu, main->shownext_item );
    gtk_menu_append( options_menu, main->boardmarkup_item );
    gtk_menu_append( options_menu, main->childview_item );
    gtk_menu_append( options_menu, main->diagramview_item );
    gtk_menu_append( options_menu, main->resetnumbering_item );
    gtk_menu_append( options_menu, main->numberNPlays_item );
    gtk_menu_append( options_menu, main->autoRefute_item );
#ifndef PRODUCTION_VERSION  
    gtk_menu_append( options_menu, main->alphaTesting_item );
#endif
    gtk_menu_append( options_menu, main->manualOnly_item );
    gtk_menu_append( options_menu, main->bigTreeVariations_item );
    gtk_menu_append( options_menu, main->load_options_item );
    gtk_menu_append( load_options_menu, main->rndOrientation_item );
    gtk_menu_append( load_options_menu, main->problem2step_item );
    gtk_menu_append( load_options_menu, main->problemTraining_item );
    gtk_menu_append( load_options_menu, main->merge_load_item );
    gtk_menu_append( load_options_menu, main->minimal_load_item );
    gtk_menu_append( load_options_menu, main->omit_empty_nodes_item );
#ifndef PRODUCTION_VERSION  
    gtk_menu_append( load_options_menu, main->try_load_item );
#endif
    gtk_menu_append( main_menu, separator );
/*    gtk_menu_append( main_menu, help ); */
    gtk_menu_append( main_menu, close );

    /* Add submenus to the right items */
    gtk_menu_item_set_submenu( GTK_MENU_ITEM (main->file_item), file_menu );
    gtk_menu_item_set_submenu( GTK_MENU_ITEM (main->playlist_item), main->playlists_menu );
    gtk_menu_item_set_submenu( GTK_MENU_ITEM (main->edit_item), edit_menu );
    gtk_menu_item_set_submenu( GTK_MENU_ITEM (main->game_edit_item), game_edit_menu );
    gtk_menu_item_set_submenu( GTK_MENU_ITEM (main->options_item), options_menu );
    gtk_menu_item_set_submenu( GTK_MENU_ITEM (main->load_options_item), load_options_menu );
    gtk_menu_item_set_submenu( GTK_MENU_ITEM (main->set_result_item), result_menu );

    /* Set initial values */
    main->bigTreeVariations = FALSE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (main->bigTreeVariations_item), main->bigTreeVariations);
    main->tryLoad = 0;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (main->try_load_item), (main->tryLoad > 0));
    main->manualOnly = FALSE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (main->manualOnly_item), main->manualOnly);
    main->minimalload = FALSE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (main->minimal_load_item), main->minimalload);
    main->mergeload = FALSE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (main->merge_load_item), main->mergeload);
    main->omitEmptyNodes = TRUE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (main->omit_empty_nodes_item), main->omitEmptyNodes);
    main->shownext = TRUE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (main->shownext_item), main->shownext);
    SetStyle(main);
    main->resetNumbering = FALSE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (main->resetnumbering_item), main->resetNumbering);
    main->numberLastPlays = MAX_SHOWN_NUMBERS;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (main->numberNPlays_item), FALSE);
    main->diagramview = FALSE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (main->diagramview_item), main->diagramview);    
    main->rndOrientation = TRUE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (main->rndOrientation_item), main->rndOrientation);
    main->problem2step = TRUE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (main->problem2step_item), main->problem2step);
    main->problemTraining = TRUE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (main->problemTraining_item), main->problemTraining);
    main->analysisAutoRefute = TRUE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (main->autoRefute_item), main->analysisAutoRefute);
    main->alphaTesting = FALSE;
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (main->alphaTesting_item), main->alphaTesting);
    /* Attach the callback functions to the activate signal */
    g_signal_connect( G_OBJECT( main->new_item ), "activate", 
		      G_CALLBACK ( callback_file_new), main );
    g_signal_connect( G_OBJECT( main->open_item ), "activate", 
		      G_CALLBACK ( callback_file_open), main );
    g_signal_connect( G_OBJECT( main->save_item ), "activate", 
              G_CALLBACK ( callback_file_save), main );
    g_signal_connect( G_OBJECT( main->saveas_item ), "activate", 
		      G_CALLBACK ( callback_file_saveas), main );
    g_signal_connect( G_OBJECT( main->cut_item ), "activate", 
		      G_CALLBACK ( callback_edit_cut), main );
    g_signal_connect( G_OBJECT( main->copy_item ), "activate", 
		      G_CALLBACK( callback_edit_copy ), main );
    g_signal_connect( G_OBJECT( main->paste_item ), "activate", 
		      G_CALLBACK( callback_edit_paste ), main );
    /* g_signal_connect( G_OBJECT( main->font_item ), "activate", 
		      G_CALLBACK ( callback_font ), main ); */
    g_signal_connect( G_OBJECT( main->fullscreen_item ), "activate", 
		      G_CALLBACK ( callback_fullscreen ), main );
    /* g_signal_connect( G_OBJECT( main->propose_item ), "activate", 
		      G_CALLBACK ( callback_propose ), main ); */
    g_signal_connect( G_OBJECT( main->compare_item ), "activate", 
		      G_CALLBACK ( callback_compare ), main );
    g_signal_connect( G_OBJECT( main->hint_item ), "activate", 
		      G_CALLBACK ( callback_hint ), main );
    g_signal_connect( G_OBJECT( main->refresh_playlist_item ), "activate", 
		      G_CALLBACK ( callback_refresh_playlist ), main );
    g_signal_connect( G_OBJECT( main->thumb_playlist_item ), "activate", 
		      G_CALLBACK ( callback_thumb_playlist ), main );
    g_signal_connect( G_OBJECT( main->prune_item ), "activate", 
		      G_CALLBACK ( callback_prune ), main );
    g_signal_connect( G_OBJECT( main->rotate_item ), "activate", 
		      G_CALLBACK ( callback_rotate ), main );
    g_signal_connect( G_OBJECT( main->circle_lp_item ), "activate", 
		      G_CALLBACK ( callback_circle_lp ), main );
    g_signal_connect( G_OBJECT( main->remove_cm_item ), "activate", 
		      G_CALLBACK ( callback_remove_cm ), main );
    g_signal_connect( G_OBJECT( main->fill_item ), "activate", 
		      G_CALLBACK ( callback_fill ), main );
    g_signal_connect( G_OBJECT( main->create_problem_item ), "activate", 
		      G_CALLBACK ( callback_create_problem ), main );
    g_signal_connect( G_OBJECT( main->result_bwinr_item ), "activate", 
		      G_CALLBACK ( callback_set_bwinr ), main );
    g_signal_connect( G_OBJECT( main->result_bwint_item ), "activate", 
		      G_CALLBACK ( callback_set_bwint ), main );
    g_signal_connect( G_OBJECT( main->result_bwinf_item ), "activate", 
		      G_CALLBACK ( callback_set_bwinf ), main );
    g_signal_connect( G_OBJECT( main->result_wwinr_item ), "activate", 
		      G_CALLBACK ( callback_set_wwinr ), main );
    g_signal_connect( G_OBJECT( main->result_wwint_item ), "activate", 
		      G_CALLBACK ( callback_set_wwint ), main );
    g_signal_connect( G_OBJECT( main->result_wwinf_item ), "activate", 
		      G_CALLBACK ( callback_set_wwinf ), main );
    g_signal_connect( G_OBJECT( main->result_void_item ), "activate", 
		      G_CALLBACK ( callback_set_void ), main );
    g_signal_connect( G_OBJECT( main->result_unknown_item ), "activate", 
		      G_CALLBACK ( callback_set_unknown ), main );
    g_signal_connect( G_OBJECT( main->result_estimate_item ), "activate", 
		      G_CALLBACK ( callback_set_estimate ), main );
    g_signal_connect( G_OBJECT( main->shownext_item ), "toggled", 
		      G_CALLBACK ( callback_shownext ), main );
    g_signal_connect( G_OBJECT( main->boardmarkup_item ), "toggled", 
		      G_CALLBACK ( callback_boardmarkup ), main );
    g_signal_connect( G_OBJECT( main->childview_item ), "toggled", 
		      G_CALLBACK ( callback_childview ), main );
    g_signal_connect( G_OBJECT( main->diagramview_item ), "toggled", 
		      G_CALLBACK ( callback_diagram ), main );
    g_signal_connect( G_OBJECT( main->resetnumbering_item ), "toggled", 
		      G_CALLBACK ( callback_resetnumbering ), main );
    g_signal_connect( G_OBJECT( main->numberNPlays_item ), "toggled", 
		      G_CALLBACK ( callback_number_n_plays_menu ), main );
    g_signal_connect( G_OBJECT( main->rndOrientation_item ), "toggled", 
		      G_CALLBACK ( callback_rndOrientation ), main );
    g_signal_connect( G_OBJECT( main->problem2step_item ), "toggled", 
		      G_CALLBACK ( callback_problem2step ), main );
    g_signal_connect( G_OBJECT( main->problemTraining_item ), "toggled", 
		      G_CALLBACK ( callback_problemTraining ), main );
    g_signal_connect( G_OBJECT( main->autoRefute_item ), "toggled", 
		      G_CALLBACK ( callback_autoRefute ), main );
    g_signal_connect( G_OBJECT( main->alphaTesting_item ), "toggled", 
		      G_CALLBACK ( callback_alphaTesting ), main );
    g_signal_connect( G_OBJECT( main->merge_load_item ), "toggled", 
		      G_CALLBACK ( callback_mergeLoad ), main );
    g_signal_connect( G_OBJECT( main->minimal_load_item ), "toggled", 
		      G_CALLBACK ( callback_minimalLoad ), main );
    g_signal_connect( G_OBJECT( main->manualOnly_item ), "toggled", 
		      G_CALLBACK ( callback_manualOnly ), main );
    g_signal_connect( G_OBJECT( main->try_load_item ), "toggled", 
		      G_CALLBACK ( callback_tryLoad ), main );
    g_signal_connect( G_OBJECT( main->omit_empty_nodes_item ), "toggled", 
		      G_CALLBACK ( callback_omitEmptyNodes ), main );
    g_signal_connect( G_OBJECT( main->bigTreeVariations_item ), "toggled", 
		      G_CALLBACK ( callback_bigTreeVariations ), main );
    /* g_signal_connect( G_OBJECT( help ), "activate", 
		      G_CALLBACK ( callback_help ), main ); */
    g_signal_connect( G_OBJECT( close ), "activate", gtk_main_quit, NULL );

    /* Add menu to HildonWindow */
    hildon_window_set_menu(main->data->window, main_menu);

    /* We need to show menu items */
    gtk_widget_show_all( GTK_WIDGET( main_menu ) );
}


/* Create toolbar to mainview */
static void create_toolbar ( MainView *main )
{
    /* Create new GTK toolbar */
    main->toolbar = gtk_toolbar_new ();

 	/* Set toolbar properties */
    gtk_toolbar_set_orientation( GTK_TOOLBAR(main->toolbar), GTK_ORIENTATION_HORIZONTAL);
    gtk_toolbar_set_style( GTK_TOOLBAR(main->toolbar), GTK_TOOLBAR_BOTH_HORIZ);

    /* Make menus and buttons to toolbar: */
    /* Create toolitems using defined items from stock */
    main->alphatesting_tb = gtk_tool_button_new_from_stock(GTK_STOCK_MISSING_IMAGE);
    main->gotofirst_tb = gtk_tool_button_new_from_stock(GTK_STOCK_GOTO_FIRST);
    main->gotohome_tb = gtk_tool_button_new_from_stock(GTK_STOCK_HOME);
    main->gotomain_tb = gtk_tool_button_new_from_stock(GTK_STOCK_GOTO_TOP);
    main->gotovari_tb = gtk_tool_button_new_from_stock(GTK_STOCK_GO_DOWN);
    main->gotopbra_tb = gtk_tool_button_new_from_stock(GTK_STOCK_MEDIA_REWIND);
    main->gotoprev_tb = gtk_tool_button_new_from_stock(GTK_STOCK_GO_BACK);
    main->gotonext_tb = gtk_tool_button_new_from_stock(GTK_STOCK_GO_FORWARD);
    main->gotonbra_tb = gtk_tool_button_new_from_stock(GTK_STOCK_MEDIA_FORWARD);
    main->gotolast_tb = gtk_tool_button_new_from_stock(GTK_STOCK_GOTO_LAST);
    main->menu_tb = gtk_tool_button_new_from_stock(GTK_STOCK_PREFERENCES);
    main->new_tb = gtk_tool_button_new_from_stock(GTK_STOCK_NEW);
    main->open_tb = gtk_tool_button_new_from_stock(GTK_STOCK_OPEN);
    main->nextfile_tb = gtk_tool_button_new_from_stock(GTK_STOCK_MEDIA_NEXT);
    main->index_tb = gtk_tool_button_new_from_stock(GTK_STOCK_INDEX);
    main->save_tb = gtk_tool_button_new_from_stock(GTK_STOCK_SAVE);
    main->saveas_tb = gtk_tool_button_new_from_stock(GTK_STOCK_SAVE_AS);
    main->separator_tb1 = gtk_separator_tool_item_new();
    main->cut_tb = gtk_tool_button_new_from_stock(GTK_STOCK_CUT);
    main->copy_tb = gtk_tool_button_new_from_stock(GTK_STOCK_COPY);
    main->paste_tb = gtk_tool_button_new_from_stock(GTK_STOCK_PASTE);
    main->separator_tb2 = gtk_separator_tool_item_new();
    /* main->font_tb = gtk_tool_button_new_from_stock( GTK_STOCK_SELECT_FONT ); */
    main->fullscreen_tb = gtk_tool_button_new_from_stock( GTK_STOCK_ZOOM_FIT );
    main->findmainline_tb = gtk_tool_button_new_from_stock( GTK_STOCK_FIND );
    main->givehint_tb = gtk_tool_button_new_from_stock( GTK_STOCK_HELP );
    main->clear_tb = gtk_tool_button_new_from_stock( GTK_STOCK_CLEAR );
    main->add_tb = gtk_tool_button_new_from_stock( GTK_STOCK_ADD );
    main->delete_tb = gtk_tool_button_new_from_stock( GTK_STOCK_DELETE );
    main->mark_territory_tb = gtk_tool_button_new_from_stock( GTK_STOCK_APPLY );
    main->step_2_option_tb = gtk_toggle_tool_button_new();
    main->step_2_option_icon = gtk_label_new ("2");
    gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(main->step_2_option_tb),main->step_2_option_icon);
    main->set_mark_tb = gtk_tool_button_new(NULL,NULL); /* <-- CRASH ?? */
    main->set_mark_icon = gtk_label_new ("1.");
    gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(main->set_mark_tb),main->set_mark_icon);
    main->transparent_tb = gtk_toggle_tool_button_new();
    main->transparent_icon = gtk_label_new ("tp");
    gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(main->transparent_tb),main->transparent_icon);
    main->resetNumbering_tb = gtk_toggle_tool_button_new();
    main->resetNumbering_icon = gtk_label_new ("1.=1");
    gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(main->resetNumbering_tb),main->resetNumbering_icon);
    main->numberNlast_tb = gtk_toggle_tool_button_new();
    main->numberNlast_icon = gtk_label_new ("..n");
    gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(main->numberNlast_tb),main->numberNlast_icon);
    /* Insert items to toolbar */
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->alphatesting_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->gotofirst_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->gotohome_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->gotomain_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->gotovari_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->gotopbra_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->gotoprev_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->gotonext_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->gotonbra_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->gotolast_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->separator_tb1, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->menu_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->new_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->open_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->nextfile_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->index_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->save_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->saveas_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->separator_tb2, -1);
    /*
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->separator_tb1, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->cut_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->copy_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->paste_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->separator_tb2, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->font_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->fullscreen_tb, -1);
    */
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->findmainline_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->givehint_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->clear_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->add_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->delete_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->mark_territory_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->step_2_option_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->set_mark_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->transparent_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->resetNumbering_tb, -1);
    gtk_toolbar_insert ( GTK_TOOLBAR(main->toolbar), main->numberNlast_tb, -1);

    /* Connect signals to buttons */
    g_signal_connect(G_OBJECT(main->alphatesting_tb), "clicked",
                     G_CALLBACK(callback_test), main);
    g_signal_connect(G_OBJECT(main->gotofirst_tb), "clicked",
                     G_CALLBACK(callback_first), main);
    g_signal_connect(G_OBJECT(main->gotohome_tb), "clicked",
                     G_CALLBACK(callback_home), main);
    g_signal_connect(G_OBJECT(main->gotomain_tb), "clicked",
                     G_CALLBACK(callback_mainline), main);
    g_signal_connect(G_OBJECT(main->gotovari_tb), "clicked",
                     G_CALLBACK(callback_variation), main);
    g_signal_connect(G_OBJECT(main->gotopbra_tb), "clicked",
                     G_CALLBACK(callback_pbranch), main);
    g_signal_connect(G_OBJECT(main->gotoprev_tb), "clicked",
                     G_CALLBACK(callback_previous), main);
    g_signal_connect(G_OBJECT(main->gotonext_tb), "clicked",
                     G_CALLBACK(callback_next), main);
    g_signal_connect(G_OBJECT(main->gotonbra_tb), "clicked",
                     G_CALLBACK(callback_nbranch), main);
    g_signal_connect(G_OBJECT(main->gotolast_tb), "clicked",
                     G_CALLBACK(callback_last), main);
    g_signal_connect(G_OBJECT(main->menu_tb), "clicked",
                     G_CALLBACK(callback_open_menu), main);
    g_signal_connect(G_OBJECT(main->new_tb), "clicked",
                     G_CALLBACK(callback_file_new), main);
    g_signal_connect(G_OBJECT(main->open_tb), "clicked",
                     G_CALLBACK(callback_file_open), main);
    g_signal_connect(G_OBJECT(main->nextfile_tb), "clicked",
                     G_CALLBACK(callback_next_file), main);
    g_signal_connect(G_OBJECT(main->index_tb), "clicked",
                     G_CALLBACK(callback_index), main);
    g_signal_connect(G_OBJECT(main->save_tb), "clicked",
                     G_CALLBACK(callback_file_save), main);
    g_signal_connect(G_OBJECT(main->saveas_tb), "clicked",
                     G_CALLBACK(callback_file_saveas), main);
    g_signal_connect(G_OBJECT(main->cut_tb), "clicked",
                     G_CALLBACK(callback_edit_cut), main);
    g_signal_connect(G_OBJECT(main->copy_tb), "clicked",
                     G_CALLBACK(callback_edit_copy), main);
    g_signal_connect(G_OBJECT(main->paste_tb), "clicked",
                     G_CALLBACK(callback_edit_paste), main);
    g_signal_connect(G_OBJECT(main->fullscreen_tb), "clicked",
                     G_CALLBACK(callback_fullscreen), main);
    g_signal_connect(G_OBJECT(main->findmainline_tb), "clicked",
                     G_CALLBACK(callback_propose), main);
    g_signal_connect(G_OBJECT(main->givehint_tb), "clicked",
                     G_CALLBACK(callback_hint), main);
    g_signal_connect(G_OBJECT(main->clear_tb), "clicked",
                     G_CALLBACK(callback_clear_territory), main);
    g_signal_connect(G_OBJECT(main->add_tb), "clicked",
                     G_CALLBACK(callback_add_kikashi), main);
    g_signal_connect(G_OBJECT(main->delete_tb), "clicked",
                     G_CALLBACK(callback_delete), main);
    g_signal_connect(G_OBJECT(main->mark_territory_tb), "clicked",
                     G_CALLBACK(callback_mterri), main);
    g_signal_connect(G_OBJECT(main->step_2_option_tb), "clicked",
                     G_CALLBACK(callback_step_2_option), main);
    g_signal_connect(G_OBJECT(main->set_mark_tb), "clicked",
                     G_CALLBACK(callback_compare), main);
    g_signal_connect(G_OBJECT(main->transparent_tb), "clicked",
                     G_CALLBACK(callback_transparent), main);
    g_signal_connect(G_OBJECT(main->resetNumbering_tb), "clicked",
                     G_CALLBACK(callback_resetnumbering), main);
    g_signal_connect(G_OBJECT(main->numberNlast_tb), "clicked",
                     G_CALLBACK(callback_number_n_plays_tb), main);
	
    /* Add toolbar to the HildonWindow */
    hildon_window_add_toolbar(main->data->window, GTK_TOOLBAR(main->toolbar));

    /* Show toolbar */
    gtk_widget_show_all (GTK_WIDGET(main->toolbar));
    gtk_widget_show_all (GTK_WIDGET(main->data->window) );
    /* Hide elements visible only at full screen mode */
    gtk_widget_hide(GTK_WIDGET(main->saveas_tb));
    /* Hide elements visible only other modes than initial mode */
    gtk_widget_hide(GTK_WIDGET(main->givehint_tb));
    gtk_widget_hide(GTK_WIDGET(main->clear_tb));
    gtk_widget_hide(GTK_WIDGET(main->mark_territory_tb));
    gtk_widget_hide(GTK_WIDGET(main->gotohome_tb));
    gtk_widget_hide(GTK_WIDGET(main->step_2_option_tb));
    gtk_widget_hide(GTK_WIDGET(main->set_mark_tb));
    gtk_widget_hide(GTK_WIDGET(main->transparent_tb));
    gtk_widget_hide(GTK_WIDGET(main->resetNumbering_tb));
    gtk_widget_hide(GTK_WIDGET(main->numberNlast_tb));
    gtk_widget_hide(GTK_WIDGET(main->nextfile_tb));
    gtk_widget_hide(GTK_WIDGET(main->alphatesting_tb));
    gtk_widget_hide(GTK_WIDGET(main->findmainline_tb));
}

/* Create application layout */
void create_layout( MainView *main )
{
    PangoFontDescription * pangoFontDescription = NULL;
    gint i;
    GdkColormap * systemColormap = NULL;
    
    /* Reserving color for goban */
    systemColormap = gdk_colormap_get_system();
    /* if(gdk_color_parse("LightBrown",&(main->gobanColor))) { */
    if(gdk_color_parse("#daa520",&(main->gobanColor))) {
      gdk_colormap_alloc_color(systemColormap,&(main->gobanColor),FALSE,TRUE);
    }
    if(gdk_color_parse("#ff0000",&(main->indicationColor))) {
      gdk_colormap_alloc_color(systemColormap,&(main->indicationColor),FALSE,TRUE);
    }
    /*
    main->gobanGC = gdk_gc_new(main->gobanPixmap);
    gdk_gc_set_foreground(main->gobanGC,&(main->gobanColor));
    */
    
	 /* Top level of application layout */
    /* main->toplevel = gtk_hbox_new (FALSE, 0); */
    main->toplevel = gtk_hpaned_new ();
    gtk_widget_show(main->toplevel);
    
    /* Aspectframe for making goban correct shape */
    /* main->aspectframe = gtk_aspect_frame_new ("", 0.0, 0.5, GOBAN_ASPECTRATIO, FALSE); */
    main->gobanFrame = gtk_frame_new (NULL);
    /* gtk_box_pack_start (GTK_BOX (main->toplevel), main->aspectframe, TRUE, TRUE, 0); */
    gtk_paned_pack1 (GTK_PANED (main->toplevel), main->gobanFrame, TRUE, FALSE);
    gtk_widget_set_size_request (main->gobanFrame, 50, -1);
    gtk_widget_show(main->gobanFrame);
    
    /* Drawing area for the goban */
    /* main->goban_view = gtk_label_new ("Goban place holder"); */
    main->goban_view = gtk_drawing_area_new ();
    pangoFontDescription = pango_font_description_from_string ("Sans bold 8"); /* ("Courier,Narrow 8"); */
    gtk_widget_modify_font (main->goban_view, pangoFontDescription);
    if(pangoFontDescription != NULL) {
      pango_font_description_free(pangoFontDescription); /* In case of crash remove this line */
    }
  /* Nice idea, but creates infinite loop               
  PangoFontDescription * pangoFontDescription;
  pangoFontDescription = pango_font_description_from_string ("Courier,Medium 8");
  gtk_widget_modify_font (widget, pangoFontDescription); */
    /* gtk_widget_set_size_request (main->goban_view, 200, 200); */
    gtk_container_add (GTK_CONTAINER (main->gobanFrame), main->goban_view);
    gtk_widget_show(main->goban_view);
    g_signal_connect (G_OBJECT (main->goban_view), "expose_event", G_CALLBACK (expose_goban_event), main);
    g_signal_connect (G_OBJECT (main->goban_view),"configure_event", G_CALLBACK (configure_goban_event), main);
    g_signal_connect (G_OBJECT (main->goban_view), "motion_notify_event", G_CALLBACK (motion_notify_goban_event), main);
    g_signal_connect (G_OBJECT (main->goban_view),"button_press_event", G_CALLBACK (button_press_goban_event), main);
    g_signal_connect (G_OBJECT (main->goban_view),"button_release_event", G_CALLBACK (button_release_goban_event), main);
    gtk_widget_set_events (main->goban_view, GDK_EXPOSURE_MASK 
                                           | GDK_LEAVE_NOTIFY_MASK 
                                           | GDK_BUTTON_PRESS_MASK
                                           | GDK_BUTTON_RELEASE_MASK
                                           | GDK_POINTER_MOTION_MASK
                                           | GDK_POINTER_MOTION_HINT_MASK);
        
    /* Notebook for alternative parts of main view */
    main->notebook = gtk_notebook_new ();
    /* gtk_label_new ("Notebook place holder"); */
    /* gtk_box_pack_start (GTK_BOX (main->toplevel), main->notebook, FALSE, TRUE, 0); */
    /* gtk_box_pack_start (GTK_BOX (main->toplevel), main->notebook, FALSE, FALSE, 0); */
    gtk_paned_pack2 (GTK_PANED (main->toplevel), main->notebook, FALSE, FALSE);
    gtk_widget_set_size_request (main->notebook, TABS_WIDTH_NORMAL, -1);
    gtk_widget_show(main->notebook);
    
    /* Notebook page for program debuging */
    CreateLogNotePage( main );
    
    /* Notebook page for game recording */
    main->game_recording_page = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(main->game_recording_page),
                                   GTK_POLICY_NEVER,
                                   GTK_POLICY_AUTOMATIC);  /* GTK_POLICY_AUTOMATIC */
    main->gamedata_vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add(GTK_CONTAINER(main->game_recording_page), main->gamedata_vbox);
    gtk_widget_show (main->gamedata_vbox);
    main->game_title_element = gtk_label_new ("No title");
    gtk_widget_show (main->game_title_element);
    gtk_box_pack_start(GTK_BOX (main->gamedata_vbox),main->game_title_element,FALSE,FALSE,0);

    main->game_result_element = gtk_label_new ("Result not known");
    gtk_widget_hide (main->game_result_element);
    gtk_box_pack_start(GTK_BOX (main->gamedata_vbox),main->game_result_element,FALSE,FALSE,0);

    main->black_frame = gtk_frame_new("Black");
    gtk_widget_show (main->black_frame);
    main->black_rows_vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (main->black_rows_vbox);
    gtk_container_add (GTK_CONTAINER (main->black_frame), main->black_rows_vbox);
    gtk_box_pack_start(GTK_BOX (main->gamedata_vbox),main->black_frame,FALSE,FALSE,0);
    main->black_id_hbox = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (main->black_id_hbox);
    gtk_box_pack_start(GTK_BOX (main->black_rows_vbox),main->black_id_hbox,TRUE,TRUE,0);
    main->black_name = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(main->black_name),""); /* No black name */
    gtk_widget_show (main->black_name);
    gtk_box_pack_start(GTK_BOX (main->black_id_hbox),main->black_name,TRUE,TRUE,0);
    main->black_rank = gtk_combo_box_new_text();
    for(i=MAX_RANK;i>=MIN_RANK;i--) {
      gtk_combo_box_insert_text (GTK_COMBO_BOX (main->black_rank) , MAX_RANK-i, ranktext[i-MIN_RANK]);
    }
    gtk_combo_box_set_active (GTK_COMBO_BOX (main->black_rank),MAX_RANK-UNKNOWN_RANK);
    gtk_widget_show (main->black_rank);
    gtk_box_pack_start(GTK_BOX (main->black_id_hbox),main->black_rank,FALSE,FALSE,0);
    main->black_time_text = gtk_label_new ("No time set");
    gtk_misc_set_alignment(GTK_MISC(main->black_time_text),0.0,0.5);
    gtk_widget_show (main->black_time_text);
    gtk_box_pack_start(GTK_BOX (main->black_rows_vbox),main->black_time_text,TRUE,FALSE,0);
    main->black_prisoners_text = gtk_label_new ("Prisoners not counted");
    gtk_misc_set_alignment(GTK_MISC(main->black_prisoners_text),0.0,0.5);
    gtk_widget_show (main->black_prisoners_text);
    gtk_box_pack_start(GTK_BOX (main->black_rows_vbox),main->black_prisoners_text,TRUE,FALSE,0);
    
    main->white_frame = gtk_frame_new("White");
    gtk_widget_show (main->white_frame);
    main->white_rows_vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (main->white_rows_vbox);
    gtk_container_add (GTK_CONTAINER (main->white_frame), main->white_rows_vbox);
    gtk_box_pack_start(GTK_BOX (main->gamedata_vbox),main->white_frame,FALSE,FALSE,0);
    gtk_widget_show (main->white_rows_vbox);
    gtk_container_add (GTK_CONTAINER (main->white_frame), main->white_rows_vbox);
    gtk_box_pack_start(GTK_BOX (main->game_recording_page),main->white_frame,TRUE,TRUE,0);
    main->white_id_hbox = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (main->white_id_hbox);
    gtk_box_pack_start(GTK_BOX (main->white_rows_vbox),main->white_id_hbox,TRUE,TRUE,0);
    main->white_name = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(main->white_name),""); /* No white name */
    gtk_widget_show (main->white_name);
    gtk_box_pack_start(GTK_BOX (main->white_id_hbox),main->white_name,TRUE,TRUE,0);
    main->white_rank = gtk_combo_box_new_text();
    for(i=MAX_RANK;i>=MIN_RANK;i--) {
      gtk_combo_box_insert_text (GTK_COMBO_BOX (main->white_rank) , MAX_RANK-i, ranktext[i-MIN_RANK]);
    }
    gtk_combo_box_set_active (GTK_COMBO_BOX (main->white_rank),MAX_RANK-UNKNOWN_RANK);
    gtk_widget_show (main->white_rank);
    gtk_box_pack_start(GTK_BOX (main->white_id_hbox),main->white_rank,FALSE,FALSE,0);
    main->white_time_text = gtk_label_new ("No time set");
    gtk_misc_set_alignment(GTK_MISC(main->white_time_text),0.0,0.5);
    gtk_widget_show (main->white_time_text);
    gtk_box_pack_start(GTK_BOX (main->white_rows_vbox),main->white_time_text,TRUE,FALSE,0);
    main->white_prisoners_text = gtk_label_new ("Prisoners not counted");
    gtk_misc_set_alignment(GTK_MISC(main->white_prisoners_text),0.0,0.5);
    gtk_widget_show (main->white_prisoners_text);
    gtk_box_pack_start(GTK_BOX (main->white_rows_vbox),main->white_prisoners_text,TRUE,FALSE,0);
    
    main->game_buttons_hbox = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (main->game_buttons_hbox);
    gtk_box_pack_start(GTK_BOX (main->gamedata_vbox),main->game_buttons_hbox,FALSE,FALSE,0);
    main->startClock_button = gtk_button_new_with_label("Start clock");
    g_signal_connect (G_OBJECT (main->startClock_button), "clicked", G_CALLBACK (pass_event), main);
    gtk_widget_show (main->startClock_button);
    gtk_box_pack_start(GTK_BOX (main->game_buttons_hbox),main->startClock_button,TRUE,TRUE,0);
    main->stopClock_button = gtk_button_new_with_label("Stop clock");
    g_signal_connect (G_OBJECT (main->stopClock_button), "clicked", G_CALLBACK (pass_event), main);
    gtk_widget_hide (main->stopClock_button);
    gtk_box_pack_start(GTK_BOX (main->game_buttons_hbox),main->stopClock_button,TRUE,TRUE,0);
    main->pass_button = gtk_button_new_with_label("Pass");
    g_signal_connect (G_OBJECT (main->pass_button), "clicked", G_CALLBACK (pass_event), main);
    gtk_widget_show (main->pass_button);
    gtk_box_pack_start(GTK_BOX (main->game_buttons_hbox),main->pass_button,TRUE,TRUE,0);
    main->game_recording_tab = gtk_label_new ("Rec");
    gtk_notebook_append_page (GTK_NOTEBOOK (main->notebook), main->game_recording_page, main->game_recording_tab);
    gtk_widget_show (main->game_recording_tab);
    gtk_widget_show (main->game_recording_page);

    /* Notebook page for game analysis */
    CreateAnalysisNotePage( main );
    pangoFontDescription = pango_font_description_from_string ("Sans Bold 6"); /* ("Courier,Narrow 6"); */
    gtk_widget_modify_font (main->gametree_view, pangoFontDescription);
    if(pangoFontDescription != NULL) {
      pango_font_description_free(pangoFontDescription); /* In case of crash remove this line */
    }
    gtk_notebook_set_current_page(GTK_NOTEBOOK(main->notebook), 2); /* Set analysis notebook tab active */

}

/* Create notebook page for log */
void CreateAnalysisNotePage( MainView *main )
{
  main->analysis_page = gtk_vpaned_new (); /* gtk_label_new ("Analysis page place holder"); */
  main->analysis_tab = gtk_label_new ("Ana");
  gtk_notebook_append_page (GTK_NOTEBOOK (main->notebook), main->analysis_page, main->analysis_tab);
  gtk_widget_show (main->analysis_tab);
  gtk_widget_show (main->analysis_page);
  main->anal_note_frame = gtk_frame_new (NULL);
  main->anal_gametree_frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (main->anal_note_frame), GTK_SHADOW_IN);
  gtk_frame_set_shadow_type (GTK_FRAME (main->anal_gametree_frame), GTK_SHADOW_IN);
  gtk_paned_pack1 (GTK_PANED (main->analysis_page), main->anal_note_frame, FALSE, FALSE);
  gtk_widget_set_size_request (main->anal_note_frame, -1, 50);
  
  main->note_scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_show(main->note_scrolledwindow);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(main->note_scrolledwindow),
                                 GTK_POLICY_NEVER,
                                 GTK_POLICY_AUTOMATIC);
  /* Text view */
  main->note_textview = gtk_text_view_new ();

  /* Some text view settings */
  gtk_text_view_set_editable ( GTK_TEXT_VIEW (main->note_textview), TRUE );
  gtk_text_view_set_left_margin (GTK_TEXT_VIEW (main->note_textview), 10 );
  gtk_text_view_set_right_margin (GTK_TEXT_VIEW (main->note_textview), 10 );
  gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (main->note_textview), GTK_WRAP_WORD_CHAR);

  /* Get handle */
  main->noteBuffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (main->note_textview));

  /* Enable Rich Text Support */
  gtk_text_buffer_register_serialize_tagset(main->noteBuffer, "RTF");

  /* Put textview under scrolledwindow and show it*/
  gtk_container_add(GTK_CONTAINER(main->note_scrolledwindow), main->note_textview);
  gtk_widget_show(main->note_textview);

  /* Change default font throughout the widget */
  main->font_desc = pango_font_description_from_string ("Monospace Regular 10");
  gtk_widget_modify_font (main->note_textview, main->font_desc);

  /* Connect signals */
  g_signal_connect (G_OBJECT (main->noteBuffer), "modified-changed",
                    G_CALLBACK(callback_note_modified), main);
  g_signal_connect (G_OBJECT (main->noteBuffer), "changed",
                    G_CALLBACK(callback_note_modified), main);
  gtk_container_add (GTK_CONTAINER (main->anal_note_frame), main->note_scrolledwindow);
  gtk_widget_show (main->note_scrolledwindow);

  gtk_paned_pack2 (GTK_PANED (main->analysis_page), main->anal_gametree_frame, TRUE, FALSE);
  gtk_widget_set_size_request (main->anal_gametree_frame, -1, 50);
  gtk_widget_show (main->anal_note_frame);
  gtk_widget_show (main->anal_gametree_frame);
  main->gametree_view = gtk_drawing_area_new ();
  gtk_container_add (GTK_CONTAINER (main->anal_gametree_frame), main->gametree_view);
  gtk_widget_show (main->gametree_view);
  g_signal_connect (G_OBJECT (main->gametree_view), "expose_event", G_CALLBACK (expose_gametree_event), main);
  g_signal_connect (G_OBJECT (main->gametree_view),"configure_event", G_CALLBACK (configure_gametree_event), main);
  g_signal_connect (G_OBJECT (main->gametree_view), "motion_notify_event", G_CALLBACK (motion_notify_gametree_event), main);
  g_signal_connect (G_OBJECT (main->gametree_view),"button_press_event", G_CALLBACK (button_press_gametree_event), main);
  g_signal_connect (G_OBJECT (main->gametree_view),"button_release_event", G_CALLBACK (button_release_gametree_event), main);
  gtk_widget_set_events (main->gametree_view, GDK_EXPOSURE_MASK 
                                            | GDK_LEAVE_NOTIFY_MASK 
                                            | GDK_BUTTON_PRESS_MASK
                                            | GDK_BUTTON_RELEASE_MASK
                                            | GDK_POINTER_MOTION_MASK
                                            | GDK_POINTER_MOTION_HINT_MASK);
}

/* initialise game information */
void setup_game( MainView *main )
{
    main->currentNode = NULL;
    main->panicNode = NULL;
    main->compareNode = NULL;
    main->playerInTurn = STONE_BLACK;
}

/* Create notebook page for log */
void CreateLogNotePage( MainView *main )
{
    PangoFontDescription * pangoFontDescription = NULL;
    gint i;
    
    pangoFontDescription = pango_font_description_from_string ("Courier,Narrow 8");
    main->debug_page = gtk_vbox_new (FALSE, 0);
    main->debug_tab = gtk_label_new ("Log");
    main->debug_info_tab = gtk_image_new_from_stock (GTK_STOCK_DIALOG_INFO,GTK_ICON_SIZE_SMALL_TOOLBAR);
    main->debug_help_tab = gtk_image_new_from_stock (GTK_STOCK_HELP,GTK_ICON_SIZE_SMALL_TOOLBAR);
    main->debug_stat_tab = gtk_image_new_from_stock (GTK_STOCK_DIALOG_INFO,GTK_ICON_SIZE_SMALL_TOOLBAR);
    main->debug_about_tab = gtk_image_new_from_stock (GTK_STOCK_ABOUT,GTK_ICON_SIZE_SMALL_TOOLBAR);
    main->debug_warn_tab = gtk_image_new_from_stock (GTK_STOCK_DIALOG_WARNING,GTK_ICON_SIZE_SMALL_TOOLBAR);
    main->debug_error_tab = gtk_image_new_from_stock (GTK_STOCK_DIALOG_ERROR,GTK_ICON_SIZE_SMALL_TOOLBAR);
    main->current_debug_tab = main->debug_about_tab;
    main->debug_tab_all = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (main->debug_tab_all), main->debug_tab, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (main->debug_tab_all), main->debug_info_tab, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (main->debug_tab_all), main->debug_help_tab, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (main->debug_tab_all), main->debug_stat_tab, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (main->debug_tab_all), main->debug_about_tab, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (main->debug_tab_all), main->debug_warn_tab, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (main->debug_tab_all), main->debug_error_tab, TRUE, TRUE, 0);
    gtk_widget_show (main->current_debug_tab);
    gtk_notebook_append_page (GTK_NOTEBOOK (main->notebook), main->debug_page, main->debug_tab_all);
    gtk_widget_show (main->debug_tab_all);
    gtk_widget_show (main->debug_page);
    
    main->debug_edit = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (main->debug_page), main->debug_edit, FALSE, TRUE, 0);
    gtk_widget_show (main->debug_edit);
    
    main->debug_options = gtk_hbox_new (FALSE, 0);
    /* gtk_widget_set_size_request (main->debug_options, 10, 10); */
    gtk_box_pack_start (GTK_BOX (main->debug_page), main->debug_options, FALSE, TRUE, 0);
    gtk_widget_show (main->debug_options);
    
    main->debug_method_combo = gtk_combo_box_new_text();
    gtk_widget_modify_font (main->debug_method_combo, pangoFontDescription);
    gtk_widget_set_size_request (main->debug_method_combo, 120, -1);
    for(i=0;i<=MAX_LOGMETHOD;i++) {
      gtk_combo_box_insert_text (GTK_COMBO_BOX (main->debug_method_combo) , i, logmethodtext[i]);
    }
	 main->debugMethod = DEFAULT_LOGMETHOD;
	 set_log_method(main->debugMethod);
    gtk_combo_box_set_active (GTK_COMBO_BOX (main->debug_method_combo),main->debugMethod);
    gtk_box_pack_start (GTK_BOX (main->debug_options), main->debug_method_combo, FALSE, FALSE, 0);
    gtk_widget_show (main->debug_method_combo);
    g_signal_connect (G_OBJECT (main->debug_method_combo), "changed", G_CALLBACK (debug_method_combo_event), main);
    
    main->debug_scope_combo = gtk_combo_box_new_text();
    gtk_widget_modify_font (main->debug_scope_combo, pangoFontDescription);
    gtk_widget_set_size_request (main->debug_scope_combo, 120, -1);
    for(i=0;i<=MAX_LOGSCOPE;i++) {
      gtk_combo_box_insert_text (GTK_COMBO_BOX (main->debug_scope_combo) , i, logscopetext[i]);
    }
	 main->debugScope = DEFAULT_LOGSCOPE;
	 set_log_scope(logscopeconvert[main->debugScope]);
    gtk_combo_box_set_active (GTK_COMBO_BOX (main->debug_scope_combo),main->debugScope);
    gtk_box_pack_start (GTK_BOX (main->debug_options), main->debug_scope_combo, FALSE, FALSE, 0);
    gtk_widget_show (main->debug_scope_combo);
    g_signal_connect (G_OBJECT (main->debug_scope_combo), "changed", G_CALLBACK (debug_scope_combo_event), main);

    main->debug_level_combo = gtk_combo_box_new_text();
    gtk_widget_modify_font (main->debug_level_combo, pangoFontDescription);
    gtk_widget_set_size_request (main->debug_level_combo, 120, -1);
    for(i=0;i<=MAX_LOGLEVEL;i++) {
      gtk_combo_box_insert_text (GTK_COMBO_BOX (main->debug_level_combo) , i, logleveltext[i]);
    }
	 main->debugLevel = DEFAULT_LOGLEVEL;
    set_log_level(main->debugLevel);
    gtk_combo_box_set_active (GTK_COMBO_BOX (main->debug_level_combo),main->debugLevel);
    gtk_box_pack_start (GTK_BOX (main->debug_options), main->debug_level_combo, TRUE, TRUE, 0);
    gtk_widget_show (main->debug_level_combo);
    g_signal_connect (G_OBJECT (main->debug_level_combo), "changed", G_CALLBACK (debug_level_combo_event), main);

    main->faq_button = gtk_button_new_with_label("FAQ");
    g_signal_connect (G_OBJECT (main->faq_button), "clicked", G_CALLBACK (faq_event), main);
    gtk_box_pack_start (GTK_BOX (main->debug_edit), main->faq_button, FALSE, TRUE, 0);
    gtk_widget_show (main->faq_button);

    main->debug_clear_button = gtk_button_new ();
    main->debug_clearButton_image = gtk_image_new_from_stock (GTK_STOCK_DELETE,GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_container_add (GTK_CONTAINER (main->debug_clear_button), main->debug_clearButton_image);
    gtk_widget_show(main->debug_clearButton_image);
    g_signal_connect (G_OBJECT (main->debug_clear_button), "clicked", G_CALLBACK (delete_logs_event), main);
    gtk_box_pack_end (GTK_BOX (main->debug_edit), main->debug_clear_button, FALSE, TRUE, 0);
    gtk_widget_show (main->debug_clear_button);

    create_textarea( main );
    gtk_box_pack_start (GTK_BOX (main->debug_page), main->debug_scrolledwindow, TRUE, TRUE, 0);
    gtk_widget_show (main->debug_scrolledwindow);
    if(pangoFontDescription != NULL) {
      pango_font_description_free(pangoFontDescription); /* In case of crash remove this line */
    }
}

/* Create the text area */
void create_textarea( MainView *main )
{
    /* Scrolled window */
    main->debug_scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
    gtk_widget_show(main->debug_scrolledwindow);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(main->debug_scrolledwindow),
                                   GTK_POLICY_NEVER,  /* GTK_POLICY_AUTOMATIC */
                                   GTK_POLICY_AUTOMATIC);
    /* Text view */
    main->debug_textview = gtk_text_view_new ();

    /* Some text view settings */
    gtk_text_view_set_editable ( GTK_TEXT_VIEW (main->debug_textview), TRUE );
    gtk_text_view_set_left_margin (GTK_TEXT_VIEW (main->debug_textview), 10 );
    gtk_text_view_set_right_margin (GTK_TEXT_VIEW (main->debug_textview), 10 );
    gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (main->debug_textview), GTK_WRAP_WORD_CHAR);

    /* Get handle */
    main->debugBuffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (main->debug_textview));
    init_log_service(main->debugBuffer,main->debug_scrolledwindow);

    /* Enable Rich Text Support */
  	gtk_text_buffer_register_serialize_tagset(main->debugBuffer, "RTF");

    /* Put textview under scrolledwindow and show it*/
    gtk_container_add(GTK_CONTAINER(main->debug_scrolledwindow), main->debug_textview);
    gtk_widget_show(main->debug_textview);

    /* Change default font throughout the widget */
    main->font_desc = pango_font_description_from_string ("Monospace Regular 10");
    gtk_widget_modify_font (main->debug_textview, main->font_desc);

    /* Connect signals */
    g_signal_connect (G_OBJECT (main->debugBuffer), "modified-changed",
                      G_CALLBACK(callback_buffer_modified), main);
    g_signal_connect (G_OBJECT (main->debugBuffer), "changed",
                      G_CALLBACK(callback_buffer_modified), main);
}
