/*
 * This file is part of goban770
 *
 * Copyright (C) 2005 Nokia Corporation.
 * Copyright (C) 2006,2007 Jarmo Ahosola.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <utils/log.h>
#include <ui/callbacks.h>
#include <ui/interface.h>
#include <game/gametree.h>
#include <game/goban.h>
#include <game/notes.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <libintl.h>
#include <dirent.h>
#include <unistd.h>
#include <time.h>
/* strlen needed from string.h */
#include <string.h>
/* Gnome VFS for file i/o */
#include <libgnomevfs/gnome-vfs.h>

#include <utils/sgf.h>

/* #define VIEWX2GOBANX(x) () */
/* #define VIEWY2GOBANY(y) () */

/* Privates: */
void set_working_directory ( MainView* mainview );
void read_file_from_playlist(MainView * mainview);
void read_directory_to_list ( MainView* mainview, gboolean readOnlyNewLists );
/* void read_directory_to_list ( MainView* mainview ); */
void read_file_to_buffer ( MainView* mainview );
void write_buffer_to_file ( MainView* mainview );
void draw_brush (GtkWidget * widget, gdouble x,  gdouble y, GdkPixmap * pixmap);

void pass_event (GtkButton *button, gpointer data )
{
    Node * node = NULL;
    /* connect pointer to our MainView struct */
    MainView *mainview = NULL;
    mainview = ( MainView * ) data;
    g_assert(mainview != NULL && mainview->data != NULL );

    if((mainview->CurrentMouseMode == mainview->play_item) || (mainview->CurrentMouseMode == mainview->diagram_item)) { /* Mouse mode is play mode or diagram mode */
      Panicable(mainview);
      ExitNode(mainview);
      node = AddNonmove2GameTree(mainview->currentNode, mainview->child, AT_IS_PASS, mainview->playerInTurn); /* add it into game tree */
      mainview->file_edited = TRUE;
      if(mainview->child == NULL) { /* If game had no moves before */
        mainview->child = node; /* Set the node to be the first move of the game */
      }
      mainview->currentNode = node; /* Set the position after the move to be the current position */
      if(mainview->analysisAutoRefute == TRUE) {
        node = RefutePlay(mainview->currentNode);
        if(node != mainview->currentNode) {
#ifndef PRODUCTION_VERSION  
              msg2log("\nAnalysis autorefute\n\nWhen analysis autorefute option is on and you make a play for which analysis line exists, the analysis is searched for to find refutation for the move. If such is found, it is automatically made.\n\nThe idea of analysis autorefute is to help you to concentrate on positions where player in turn needs to find an alternative that has not yet been analysed.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
          mainview->currentNode = node;
        }
      }
      set_active(mainview->currentNode); /* Set the active branches from current position to root of the game */
      SetPlayerInTurn(mainview); /* Change color of player in turn */
      EnterNode(mainview);
    }
    if(mainview->CurrentMouseMode == mainview->training_item) { /* Mouse mode is training mode */
      if(mainview->currentNode == NULL) {
        node = mainview->child;
      }  
      else {
        node = mainview->currentNode->child;
      }
      if(node == NULL) {
        UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "End of variation, no moves to guess.");
      }
      else {
        if(node->at < 0) {
          UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Correct. Mainline.");
          GuessRight();
          Panicable(mainview);
          ExitNode(mainview);
          mainview->currentNode = node;
          if(mainview->step_2 == TRUE) {
            if(mainview->currentNode->child != NULL) {
              mainview->currentNode = RandomSibling(mainview->currentNode->child);
            }
          }
          EnterNode(mainview);
          PrintTrainingStatistics(mainview->isGoProblem);
#ifndef PRODUCTION_VERSION  
          msg2log("\nCorrectness determination\n\n- Guessing all moves correctly is 100%\n- Educated use of hint, cheat and quess without go-knowledge is in average 0%", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
          DrawAll(mainview);
          return;
        }
        while(node->nodeHeight < node->sibling->nodeHeight) {
          node = node->sibling;
          if(node->at < 0) {
            UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Entering variation line.");
            GuessRight();
            Panicable(mainview);
            ExitNode(mainview);
            mainview->currentNode = node;
            if(mainview->step_2 == TRUE) {
              if(mainview->currentNode->child != NULL) {
                mainview->currentNode = RandomSibling(mainview->currentNode->child);
              }
            }
            EnterNode(mainview);
            PrintTrainingStatistics(mainview->isGoProblem);
            DrawAll(mainview);
            return;
          }
        }
        UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Incorrect.");
        GuessWrong();
      }
    }
    DrawAll(mainview);
} 

void debug_method_combo_event (GtkComboBox *widget, gpointer data )
{
  gint selectedValue;

  selectedValue = gtk_combo_box_get_active (widget);
  if((selectedValue < 0) || (selectedValue > MAX_LOGMETHOD)) {
    return;
  }
  set_log_method(selectedValue);
}
 
void debug_scope_combo_event (GtkComboBox *widget, gpointer data )
{
  gint selectedValue;

  selectedValue = gtk_combo_box_get_active (widget);
  if((selectedValue < 0) || (selectedValue > MAX_LOGSCOPE)) {
    return;
  }
  set_log_scope(logscopeconvert[selectedValue]);
}
 
void debug_level_combo_event (GtkComboBox *widget, gpointer data )
{
  gint selectedValue;

  selectedValue = gtk_combo_box_get_active (widget);
  if((selectedValue < 0) || (selectedValue > MAX_LOGLEVEL)) {
    return;
  }
  set_log_level(selectedValue);
}

void faq_event (GtkButton *button, gpointer data )
{
    /* connect pointer to our MainView struct */
    MainView *mainview = NULL;
    mainview = ( MainView * ) data;
    g_assert(mainview != NULL && mainview->data != NULL );

    gtk_text_buffer_set_text (mainview->debugBuffer,"Q:I know I can do X with goban770 but I don't find how. Where is it?\nA:Goban770 has a lot of features that are available only at certain modes. You can select the mode from popup menu that opens when you click somewhere aside the goban. Features most useful for the mode are at the toolbar.\n\nQ:Why using X does nothing?\nA:Some features need certain preconditions to do anything. When they don't there should be help text available here. At beta version there is also few interface elements for which implementation has not been finished yet.\n\nQ:How help is used?\nA:Click userinterface element of functionality or area which of you want to get help. If help is available, help icon will appear at log tab title. The help text is shown here.\n\nQ:What is icon X for?\nA:Try it out. Explanation should be shown here. You may wish to open unimportant file before you do that to be safe.\n\nQ:Why save as icon is not visible?\nA:All icons do not fit on the toolbar when screen is not maximized. I have chosen save as icon to be the one that is shown only at the fullscreen mode.\n\nQ:Why home icon is not visible?\nA:Home icon is shown only if home has been defined by using Mark position feture.\n\nQ:What are the blue things on the goban?\nA:It depends. Blue on goban is negative of goban color. It is used for mark up purposes. If you release mouse at misclick region blue indicates that what you tried to do was not completed.\n\nQ:Goban770 way to do x is a bit funny. Why is it done that way?\nA:Goban770 has been primarly designed for analysis of go, training go and recording tournament games. Sgf-editors designed for different purposes has different goals. For example while recording tournament game running out of time should not end the game, making move in analysis that has already been refuted by earlier analysis should indicate that and so on.\n\nQ:This FAQ wasn't helpful. What would help me?\nA:Visit https://garage.maemo.org/projects/goban770/ pages. Open discussion has FAQ thread. At the pages you can also post bug reports and feature suggestions. You can also subscribe for RSS-feed to get project news and info about new available versions.",-1);
    ResetSeverestMsg();
    DrawAll(mainview);
} 

void delete_logs_event (GtkButton *button, gpointer data )
{
    /* connect pointer to our MainView struct */
    MainView *mainview = NULL;
    mainview = ( MainView * ) data;
    g_assert(mainview != NULL && mainview->data != NULL );

    gtk_text_buffer_set_text (mainview->debugBuffer,"",-1);
    ResetSeverestMsg();
    DrawAll(mainview);
} 

gboolean key_press_cb(GtkWidget * widget, GdkEventKey * event, gpointer data )
{
    static gint then = 0;
    static gint now = 0;
    /* connect pointer to our MainView struct */
    MainView *mainview = NULL;
    GdkRectangle update_rect;
    Node * node = NULL;

#ifndef PRODUCTION_VERSION  
    msg2log("Enter key_press_cb", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
    mainview = ( MainView * ) data;
    g_assert(mainview != NULL && mainview->data != NULL );
    if((mainview->goban_view->allocation.height <= GOBAN_MAX_HEIGHT_WITH_VIRTUAL_KEYBOARD) && /* If user is editing text */
       (event->keyval != GDK_F6) && /* and key is not fullscreen toggle */
       (event->keyval != GDK_F7) && /* and key is not increase */
       (event->keyval != GDK_F8) && /* and key is not decrease */
       (event->keyval != GDK_Escape)) { /* and key is not escape */
      return FALSE; /* then let the active object handle the key press itself */
    }
	 /* Callback for hardware keys */
	 switch (event->keyval) {
	 case GDK_Up:
#ifndef PRODUCTION_VERSION  
      msg2log("Navigation Key Up", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
      if(mainview->CurrentMouseMode == mainview->search_item) {
        ExitNode(mainview);
      }
      if(mainview->currentNode != NULL) {
        if(mainview->currentNode->child != NULL) {
          upper_branch_in_tree(mainview->currentNode->child);
        }
      }
      else {
        if(mainview->child != NULL) {
          upper_branch_in_tree(mainview->child);
        }
      }
      if(mainview->CurrentMouseMode == mainview->search_item) {
        EnterNode(mainview);
      }
      DrawAll(mainview);
/*      DrawGametree(mainview);
      update_rect.x = 0;
      update_rect.y = 0;
      update_rect.width = mainview->gametree_view->allocation.width;
      update_rect.height = mainview->gametree_view->allocation.height;
      gtk_widget_draw (mainview->gametree_view, &update_rect); */
#ifndef PRODUCTION_VERSION  
      msg2log("Exit key_press_cb", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
	 	return TRUE;
	 case GDK_Down:
#ifndef PRODUCTION_VERSION  
      msg2log("Navigation Key Down", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
      if(mainview->CurrentMouseMode == mainview->search_item) {
        ExitNode(mainview);
      }
      if(mainview->currentNode != NULL) {
        if(mainview->currentNode->child != NULL) {
          lower_branch_in_tree(mainview->currentNode->child);
        }
      }
      else {
        if(mainview->child != NULL) {
          lower_branch_in_tree(mainview->child);
        }
      }
      if(mainview->CurrentMouseMode == mainview->search_item) {
        EnterNode(mainview);
      }
      DrawAll(mainview);
/*      DrawGametree(mainview);
      update_rect.x = 0;
      update_rect.y = 0;
      update_rect.width = mainview->gametree_view->allocation.width;
      update_rect.height = mainview->gametree_view->allocation.height;
      gtk_widget_draw (mainview->gametree_view, &update_rect); */
#ifndef PRODUCTION_VERSION  
      msg2log("Exit key_press_cb", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
	 	return TRUE;
	 case GDK_Left:
#ifndef PRODUCTION_VERSION  
      msg2log("Navigation Key Left", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
      Browsing();
      Panicable(mainview);
      ExitNode(mainview);
      if(mainview->CurrentMouseMode == mainview->search_item) { /* In outcome view */
        while(mainview->currentNode != NULL) { /* until */
          if(mainview->currentNode != mainview->currentNode->sibling) { /* child of branch is found */
            break;
          }
          mainview->currentNode = back_in_tree( mainview->currentNode ); /* go backwards */
        }
      }
      mainview->currentNode = back_in_tree( mainview->currentNode ); /* Change position to the previous one */
      EnterNode(mainview);
      TakeMoveBack();
      DrawAll(mainview);
#ifndef PRODUCTION_VERSION  
      msg2log("Exit key_press_cb", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
	 	return TRUE;
	 case GDK_Right:
#ifndef PRODUCTION_VERSION  
      msg2log("Navigation Key Right", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
      Browsing();
      PeekNext();
      PrintTrainingStatistics(mainview->isGoProblem);
      Panicable(mainview);
      ExitNode(mainview);
      if(mainview->CurrentMouseMode == mainview->search_item) { /* In outcome view */
        if(mainview->currentNode == NULL) {
          if(mainview->child != NULL) {
            mainview->currentNode = forward_in_tree( mainview->currentNode, mainview->child ); /* Change position to the next one */
            if(mainview->currentNode->child != NULL) {
              if(mainview->currentNode->child != mainview->currentNode->child->sibling) {
                EnterNode(mainview);
                DrawAll(mainview);
                break;
              }
            }
          }
        }
        else {
          mainview->currentNode = forward_in_tree( mainview->currentNode, mainview->child ); /* Change position to the next one */
        }
        while(mainview->currentNode->child != NULL) { /* until */
          if(mainview->currentNode->child != mainview->currentNode->child->sibling) { /* child of branch is found */
            break;
          }
          mainview->currentNode = forward_in_tree( mainview->currentNode, mainview->child ); /* Change position to the next one */
        }
      }
      else {
        mainview->currentNode = forward_in_tree( mainview->currentNode, mainview->child ); /* Change position to the next one */
      }
      EnterNode(mainview);
      DrawAll(mainview);
#ifndef PRODUCTION_VERSION  
      msg2log("Exit key_press_cb", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
	 	return TRUE;
	 case GDK_Return:
#ifndef PRODUCTION_VERSION  
      msg2log("Navigation Key Select", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
#ifndef PRODUCTION_VERSION  
      msg2log("Exit key_press_cb", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
	 	return TRUE;
	 case GDK_F6:
      /* toggle fullscreen on<->off */
      if(then == 0) {
        then = time(NULL);
      }
      else {
        now = time(NULL);
        if((now - then) < 2) { /* If fullscreen button is kept pressed in */
          return TRUE; /* we ignore that */
        } 
        then = now;
      }
      mainview->fullscreen = !mainview->fullscreen;

      if (mainview->fullscreen) {
#ifndef PRODUCTION_VERSION  
        msg2log("Enter fullscreen", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
        gtk_window_fullscreen(GTK_WINDOW(mainview->data->window));
        gtk_widget_show(GTK_WIDGET(mainview->saveas_tb));
        if((mainview->file_name != NULL) && (mainview->CurrentMouseMode == mainview->play_item)) {
          gtk_widget_show(GTK_WIDGET(mainview->nextfile_tb));
        }
      } 
      else {
#ifndef PRODUCTION_VERSION  
        msg2log("Exit fullscreen", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
        gtk_widget_hide(GTK_WIDGET(mainview->saveas_tb));
        if(mainview->CurrentMouseMode == mainview->play_item) {
          gtk_widget_hide(GTK_WIDGET(mainview->nextfile_tb));
        }
        gtk_window_unfullscreen(GTK_WINDOW(mainview->data->window));
      }
      /* gtk_widget_set_size_request (mainview->aspectframe, mainview->toplevel->allocation.height, mainview->toplevel->allocation.height * GOBAN_ASPECTRATIO); / * Request goban size to be as big as there fits */ 
#ifndef PRODUCTION_VERSION  
      msg2log("Exit key_press_cb", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
	 	return TRUE;
	 case GDK_F7:
#ifndef PRODUCTION_VERSION  
      msg2log("Increase (zoom in)", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
#ifndef PRODUCTION_VERSION  
      msg2log("\nZoom in\n\nZoom in change the minimum scale used for drawing game tree. Effect of zoom in is not visible if bigger scale is used because game tree fits in the window entirely or diagram view demanding minimum require greater zoom.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
      mainview->gridSize++;
#ifndef PRODUCTION_VERSION  
      msg2logGint("Game tree minimum scale increased to %d", LOGLEVEL_INFO, LOGSCOPE_GAMETREE, mainview->gridSize);
#endif
      DrawGametree(mainview);
      update_rect.x = 0;
      update_rect.y = 0;
      update_rect.width = mainview->gametree_view->allocation.width;
      update_rect.height = mainview->gametree_view->allocation.height;
      gtk_widget_draw (mainview->gametree_view, &update_rect);
#ifndef PRODUCTION_VERSION  
      msg2log("Exit key_press_cb", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
	 	return TRUE;
	 case GDK_F8:
#ifndef PRODUCTION_VERSION  
      msg2log("Decrease (zoom out)", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
#ifndef PRODUCTION_VERSION  
      msg2log("\nZoom out\n\nZoom out change the minimum scale used for drawing game tree. Zoom out cannot be used to make game tree smaller than what fits in the game tree window. In diagram view mode minimum scale is set big enough to show move numbers on the stones. Zoom out cannot be used to make more stones to fit in view that case. Instead you may want to drag border between goban and game tree to make more stones visible.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
      if(mainview->gridSize > 1) {
        mainview->gridSize--;
#ifndef PRODUCTION_VERSION  
      msg2logGint("Game tree minimum scale decreased to %d", LOGLEVEL_INFO, LOGSCOPE_GAMETREE, mainview->gridSize);
#endif
        DrawGametree(mainview);
        update_rect.x = 0;
        update_rect.y = 0;
        update_rect.width = mainview->gametree_view->allocation.width;
        update_rect.height = mainview->gametree_view->allocation.height;
        gtk_widget_draw (mainview->gametree_view, &update_rect);
      }
#ifndef PRODUCTION_VERSION  
      msg2log("Exit key_press_cb", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
	 	return TRUE;
	 case GDK_Escape:
#ifndef PRODUCTION_VERSION  
      msg2log("Cancel/close", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
      Browsing();
      node = mainview->panicNode;
      Panicable(mainview);
      ExitNode(mainview);
      mainview->currentNode = node;
      EnterNode(mainview);
#ifndef PRODUCTION_VERSION  
      msg2log("\nPanic button\n\nPanic button revert back to previous situation if such is possible. This feature is ment to be used in case you accidentally do something you don't know what you did. In that case you usually get back to situaion before your misclick with panic button.\n\nPanic button can also be used for comparing 2 positions as instructed with mark position feature.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
      DrawAll(mainview);
#ifndef PRODUCTION_VERSION  
      msg2log("Exit key_press_cb", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
	 	return TRUE;
	 }
#ifndef PRODUCTION_VERSION  
    msg2log("Exit key_press_cb", LOGLEVEL_WARNING, LOGSCOPE_PATH);
#endif
	 return FALSE;
}

void Panicable(MainView * mainview)
{
  gint width = 0, height = 0, nodes = 0, leafs = 0;
  static gboolean sampling = FALSE;
  mainview->panicNode = mainview->currentNode;
  if(GetLogMethod() == LOGMETHOD_REPLACE) {
    gtk_text_buffer_set_text (mainview->debugBuffer,"",-1);
    ResetSeverestMsg();
  }
#ifndef PRODUCTION_VERSION  
  if(WouldTheseParametersLog(LOGLEVEL_STATISTICS, LOGSCOPE_GAMETREE) == TRUE) { /* Let's not be sluggish if no need to */
    GetTreeMeasures(&width, &height, &nodes, &leafs);
    msg2logGint("Tree depth is %d", LOGLEVEL_STATISTICS, LOGSCOPE_GAMETREE, width);
    msg2logGint("Tree max width is %d", LOGLEVEL_STATISTICS, LOGSCOPE_GAMETREE, height);
    msg2logGint("Tree has %d nodes", LOGLEVEL_STATISTICS, LOGSCOPE_GAMETREE, nodes);
    msg2logGint("Tree has %d leafs", LOGLEVEL_STATISTICS, LOGSCOPE_GAMETREE, leafs);
  }
#endif
  if(GetLogMethod() == LOGMETHOD_SAMPLE) {
    if(sampling == FALSE) {
      sampling = TRUE;
    }
    else {
      set_log_method(LOGMETHOD_PAUSED);
      gtk_combo_box_set_active (GTK_COMBO_BOX (mainview->debug_method_combo),LOGMETHOD_PAUSED);
      sampling = FALSE;
    }
  }
}

/* Create a new backing pixmap of the appropriate size */
gboolean configure_goban_event( GtkWidget         *widget,
                                 GdkEventConfigure *event, gpointer data  )
{
  PangoFontDescription * pangoFontDescription = NULL;
  static gint everySecondTime = 0; 
  gint fontSize = 0;
  MainView * main_view;
#ifndef PRODUCTION_VERSION  
  msg2log("Enter configure_goban_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
  main_view = (MainView *) data;
  g_assert(main_view != NULL && main_view->data != NULL );
#ifndef PRODUCTION_VERSION  
  msg2logGint("Goban configure height is set to %d", LOGLEVEL_INFO, LOGSCOPE_PATH, main_view->goban_view->allocation.height);
#endif
  if((everySecondTime++%2) == 1) {
    /* Calculate font size */
    fontSize = (main_view->goban_view->allocation.height * 100) / (GOBAN_FONT_SCALE_FACTOR * (2+main_view->sgfSZ));
#ifndef PRODUCTION_VERSION  
    msg2logGint("Calculated font size is %d", LOGLEVEL_INFO, LOGSCOPE_PATH, fontSize);
#endif
    pangoFontDescription = pango_font_description_from_string (AddGintInString("Sans Bold %d", fontSize));
    /* pangoFontDescription = pango_font_description_from_string ("Courier,Narrow 8"); */
    gtk_widget_modify_font (main_view->goban_view, pangoFontDescription);
  }
  DrawGoban(main_view);
  if(pangoFontDescription != NULL) {
    pango_font_description_free(pangoFontDescription); /* if crash, remove this line */
  }
#ifndef PRODUCTION_VERSION  
  msg2log("Exit configure_goban_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
  return TRUE;
}

gboolean configure_gametree_event( GtkWidget         *widget,
                                 GdkEventConfigure *event, gpointer data  )
{
  MainView * main_view;
#ifndef PRODUCTION_VERSION  
  msg2log("Enter configure_gametree_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
  main_view = (MainView *) data;
  g_assert(main_view != NULL && main_view->data != NULL );

  DrawGametree(main_view);
#ifndef PRODUCTION_VERSION  
  msg2log("Exit configure_gametree_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
  return TRUE;
}


/* Redraw the screen from the backing pixmap */
gboolean expose_goban_event( GtkWidget      *widget,
                              GdkEventExpose *event, gpointer data )
{
  MainView * main_view;

#ifndef PRODUCTION_VERSION  
  msg2log("Enter expose_goban_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
  main_view = (MainView *) data;
  g_assert(main_view != NULL && main_view->data != NULL );

  /* GdkPixmap * pixmap = NULL; */

  if(main_view->gobanPixmap == NULL) {
#ifndef PRODUCTION_VERSION  
    msg2log("Exit expose_goban_event (no pixmap)", LOGLEVEL_WARNING, LOGSCOPE_PATH);
#endif
    return FALSE;
  }

  gdk_draw_drawable (widget->window,
		     widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		     main_view->gobanPixmap,
		     event->area.x, event->area.y,
		     event->area.x, event->area.y,
		     event->area.width, event->area.height);

#ifndef PRODUCTION_VERSION  
  msg2log("Exit expose_goban_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
  return FALSE;
}

gboolean expose_gametree_event( GtkWidget      *widget,
                              GdkEventExpose *event, gpointer data )
{
  MainView * main_view;

#ifndef PRODUCTION_VERSION  
  msg2log("Enter expose_gametree_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
  main_view = (MainView *) data;
  g_assert(main_view != NULL && main_view->data != NULL );

  /* GdkPixmap * pixmap = NULL; */

  if(main_view->gametreePixmap == NULL) {
#ifndef PRODUCTION_VERSION  
    msg2log("Exit expose_gametree_event (no pixmap)", LOGLEVEL_WARNING, LOGSCOPE_PATH);
#endif
    return FALSE;
  }

  gdk_draw_drawable (widget->window,
		     widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		     main_view->gametreePixmap,
		     event->area.x, event->area.y,
		     event->area.x, event->area.y,
		     event->area.width, event->area.height);

#ifndef PRODUCTION_VERSION  
  msg2log("Exit expose_gametree_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
  return FALSE;
}

/* Pointer pointed at goban */
gboolean button_press_goban_event( GtkWidget * widget, GdkEventButton *event, gpointer data ) 
{
  /*GdkRectangle update_rect;*/
  Stone stone;
  Pointlist * point = NULL;
  gint clickedNumber = 0;
  gint at = 0, x = 0, y = 0;
  gint hoshiInterval = 0;
  MainView * main_view;

#ifndef PRODUCTION_VERSION  
  msg2log("Enter button_press_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
 
  main_view = (MainView *) data;
  g_assert(main_view != NULL && main_view->data != NULL );
  
  if(main_view->temporal_mode == TEMPMODE_FILE_THUMBS) {
    return TRUE;
  }
  if((main_view->CurrentMouseMode == main_view->addblack_item) ||
     (main_view->CurrentMouseMode == main_view->addwhite_item)) {
    clickedNumber = GetClickedNumber(event->x,event->y,main_view,2);
    switch(main_view->sgfSZ - clickedNumber) {
      case 0: /* Select add black mode */ 
        main_view->CurrentMouseMode = main_view->addblack_item;
        /* gtk_check_menu_item_set_active(main_view->addblack_item,TRUE); */
        main_view->mouseAtGoban = FALSE;
        DrawAll(main_view);
        return TRUE;
        break;
      case 1: /* Select add white mode */
        main_view->CurrentMouseMode = main_view->addwhite_item;
        /* gtk_check_menu_item_set_active(main_view->addwhite_item,TRUE); */
        main_view->mouseAtGoban = FALSE;
        DrawAll(main_view);
        return TRUE;
        break;
    }
  }
  if((main_view->CurrentMouseMode == main_view->triangle_item) ||
     (main_view->CurrentMouseMode == main_view->square_item) ||
     (main_view->CurrentMouseMode == main_view->circle_item) ||
     (main_view->CurrentMouseMode == main_view->cross_item) ||
     (main_view->CurrentMouseMode == main_view->labeling_item) ||
     (main_view->CurrentMouseMode == main_view->numbering_item)) {
    clickedNumber = GetClickedNumber(event->x,event->y,main_view,2);
    switch(main_view->sgfSZ - clickedNumber) {
      case 0: /* Select triangle mode */
        main_view->CurrentMouseMode = main_view->triangle_item;
        /* gtk_check_menu_item_set_active(main_view->triangle_item,TRUE); */
        main_view->mouseAtGoban = FALSE;
        DrawAll(main_view);
        return TRUE;
        break;
      case 1: /* Select square mode */
        main_view->CurrentMouseMode = main_view->square_item;
        /* gtk_check_menu_item_set_active(main_view->square_item,TRUE); */
        main_view->mouseAtGoban = FALSE;
        DrawAll(main_view);
        return TRUE;
        break;
      case 2: /* Select circle mode */
        main_view->CurrentMouseMode = main_view->circle_item;
        /* gtk_check_menu_item_set_active(main_view->circle_item,TRUE); */
        main_view->mouseAtGoban = FALSE;
        DrawAll(main_view);
        return TRUE;
        break;
      case 3: /* Select cross mode */
        main_view->CurrentMouseMode = main_view->cross_item;
        /* gtk_check_menu_item_set_active(main_view->cross_item,TRUE); */
        main_view->mouseAtGoban = FALSE;
        DrawAll(main_view);
        return TRUE;
        break;
      case 4: /* Select label mode */
        main_view->CurrentMouseMode = main_view->labeling_item;
        /* gtk_check_menu_item_set_active(main_view->labeling_item,TRUE); */
        main_view->mouseAtGoban = FALSE;
        DrawAll(main_view);
        return TRUE;
        break;
      case 5: /* Select number mode */
        main_view->CurrentMouseMode = main_view->numbering_item;
        /* gtk_check_menu_item_set_active(main_view->numbering_item,TRUE); */
        main_view->mouseAtGoban = FALSE;
        DrawAll(main_view);
        return TRUE;
        break;
    }
  }
  if((main_view->child == NULL) && (main_view->sgfAB == NULL) && (main_view->sgfAW == NULL)) {
    clickedNumber = GetClickedNumber(event->x,event->y,main_view,0);
    if(clickedNumber > 1) {
      main_view->sgfSZ = clickedNumber;
      InitGoban(main_view->sgfSZ);
      DrawAll(main_view);
      return TRUE;
    }
    clickedNumber = GetClickedNumber(event->x,event->y,main_view,1);
    if((clickedNumber > 1) && (clickedNumber < 10)) {
      hoshiInterval = GetHoshiInterval(main_view->sgfSZ);
      main_view->sgfHA = clickedNumber;
      switch(clickedNumber) {
        case 8:
          at = XY2Location(main_view->sgfSZ/2,main_view->sgfSZ/2 - hoshiInterval);
          point = NewPoint(at);
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
          at = XY2Location(main_view->sgfSZ/2,main_view->sgfSZ/2 + hoshiInterval);
          point = NewPoint(at);
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
          /* No break */
        case 6:
          at = XY2Location(main_view->sgfSZ/2 + hoshiInterval,main_view->sgfSZ/2);
          point = NewPoint(at);
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
          at = XY2Location(main_view->sgfSZ/2 - hoshiInterval,main_view->sgfSZ/2);
          point = NewPoint(at);
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
          at = XY2Location(main_view->sgfSZ/2 + hoshiInterval,main_view->sgfSZ/2 + hoshiInterval);
          point = NewPoint(at);
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
          at = XY2Location(main_view->sgfSZ/2 + hoshiInterval,main_view->sgfSZ/2 - hoshiInterval);
          point = NewPoint(at);
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
          at = XY2Location(main_view->sgfSZ/2 - hoshiInterval,main_view->sgfSZ/2 + hoshiInterval);
          point = NewPoint(at);
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
          at = XY2Location(main_view->sgfSZ/2 - hoshiInterval,main_view->sgfSZ/2 - hoshiInterval);
          point = NewPoint(at);
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
          break;
        case 9:
          at = XY2Location(main_view->sgfSZ/2,main_view->sgfSZ/2 - hoshiInterval);
          point = NewPoint(at);
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
          at = XY2Location(main_view->sgfSZ/2,main_view->sgfSZ/2 + hoshiInterval);
          point = NewPoint(at);
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
          /* No break */
        case 7:
          at = XY2Location(main_view->sgfSZ/2 + hoshiInterval,main_view->sgfSZ/2);
          point = NewPoint(at);
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
          at = XY2Location(main_view->sgfSZ/2 - hoshiInterval,main_view->sgfSZ/2);
          point = NewPoint(at);
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
          /* No break */
        case 5:
          at = XY2Location(main_view->sgfSZ/2,main_view->sgfSZ/2);
          point = NewPoint(at);
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
          /* No break */
        case 4:
          at = XY2Location(main_view->sgfSZ/2 - hoshiInterval,main_view->sgfSZ/2 - hoshiInterval);
          point = NewPoint(at);
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
          /* No break */
        case 3:
          at = XY2Location(main_view->sgfSZ/2 + hoshiInterval,main_view->sgfSZ/2 + hoshiInterval);
          point = NewPoint(at);
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
          /* No break */
        case 2:
          at = XY2Location(main_view->sgfSZ/2 + hoshiInterval,main_view->sgfSZ/2 - hoshiInterval);
          point = NewPoint(at);
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
          at = XY2Location(main_view->sgfSZ/2 - hoshiInterval,main_view->sgfSZ/2 + hoshiInterval);
          point = NewPoint(at);
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
          break;
      } 
      DrawAll(main_view);
      return TRUE;
    }
  }
  if (IsOutofGoban(event->x,event->y,main_view) == TRUE) {
    gtk_menu_popup(GTK_MENU(main_view->mousemode_menu),NULL,NULL,NULL,NULL,1,gtk_get_current_event_time());
    return TRUE;
  }
  if ((IsValidClick(event->x,event->y,main_view) == FALSE) && (main_view->temporal_mode != TEMPMODE_EDIT_AREA)) {
    return TRUE;
  }
  if(event->button == 1 && main_view->gobanPixmap != NULL) {
    main_view->mouseGobanClickX = viewx2gobanx(event->x, main_view );
    main_view->mouseGobanClickY = viewy2gobany(event->y, main_view );
    main_view->mouseGobanCurrentX = main_view->mouseGobanClickX;
    main_view->mouseGobanCurrentY = main_view->mouseGobanClickY;
    main_view->mouseAtGoban = TRUE;
    if(main_view->temporal_mode == TEMPMODE_EDIT_AREA) {
      for(x = main_view->select_left; x <= main_view->select_right; x++) {
        for(y = main_view->select_top; y <= main_view->select_bottom; y++) {
          stone = GetStoneXY(x,y);
          switch(stone) {
            case STONE_TRANSPARENT_BLACK: ;
            case STONE_BLACK: ;
            case STONE_WHITE: ;
            case STONE_TRANSPARENT_WHITE:
              MarkStone(main_view, x, y, &stone);
              break;
			default:
			  break;
          }
        }
      }
    }
    else {
      MarkSelected(main_view, main_view->mouseGobanClickX,main_view->mouseGobanClickY);
    }
  }
#ifndef PRODUCTION_VERSION  
  msg2log("Exit button_press_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
  return TRUE;
}

gboolean button_press_gametree_event( GtkWidget * widget, GdkEventButton *event, gpointer data ) 
{
  gint x = 0, y = 0;
  MainView * main_view;
#ifndef PRODUCTION_VERSION  
  msg2log("Enter button_press_gametree_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
  msg2logGint("event->button = %d", LOGLEVEL_INFO, LOGSCOPE_GAMETREE, event->button);
  msg2logGint("event->x = %d", LOGLEVEL_INFO, LOGSCOPE_GAMETREE, event->x);
  msg2logGint("event->y = %d", LOGLEVEL_INFO, LOGSCOPE_GAMETREE, event->y);
#endif
  main_view = (MainView *) data;
  g_assert(main_view != NULL && main_view->data != NULL );
  
  if(event->button == 1 && main_view->gametreePixmap != NULL) {
    y = event->y;
    x = event->x;
    main_view->mouseGtreeClick = GetNodeAt(main_view->child, Y2Height(y), X2Turn(x), ((main_view->isBigTree == FALSE) || (main_view->bigTreeVariations == TRUE)));
    main_view->mouseGtreeCurrent = main_view->mouseGtreeClick;
    main_view->mouseAtGtree = TRUE;
    MarkSelectedNode(main_view, main_view->mouseGtreeClick);
  }
#ifndef PRODUCTION_VERSION  
  msg2log("Exit button_press_gametree_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
  return TRUE;
}

/* Pointer lifted at goban */
gboolean button_release_goban_event( GtkWidget * widget, GdkEventButton *event, gpointer data ) 
{
  Pointlist * point = NULL;
  Labellist * llist = NULL;
  gint cx = 0, cy = 0;
  Node * node;
  Stone color;
  ThumbList * thumb = NULL;
  gint at = -1, digits = 100, index = 2, number = 0, i = 0;
  MainView * main_view;

#ifndef PRODUCTION_VERSION  
  msg2log("Enter button_release_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
 
  main_view = (MainView *) data;
  g_assert(main_view != NULL && main_view->data != NULL );
  
  cx = viewx2gobanx(event->x, main_view );
  cy = viewy2gobany(event->y, main_view );
  if(main_view->temporal_mode == TEMPMODE_FILE_THUMBS) {
    gtk_widget_show(main_view->notebook);
    gtk_widget_show(main_view->toolbar);
    if(main_view->fullscreen == FALSE) {
      gtk_window_unfullscreen(GTK_WINDOW(main_view->data->window));
    }
    main_view->temporal_mode = TEMPMODE_NONE;
    cx = (event->x - THUMB_LEFT_MARGIN) / THUMB_HORIZONTAL_DELTA;
    cy = (event->y - THUMB_TOP_MARGIN) / THUMB_VERTICAL_DELTA;
    if(cx < 0) {
      cx = 0;
    }
    if(cy < 0) {
      cy = 0;
    }
    i = cy + cx * THUMBS_MAX_AT_COLUMN;
    thumb = main_view->thumbnails;
    while(i>0 && thumb != NULL) {
      i--;
      thumb = thumb->next;
    }
    if(thumb != NULL) {
      main_view->currentPlaylist->current = (thumb->indexAtPlaylist - 1 + main_view->currentPlaylist->size) % main_view->currentPlaylist->size;
      callback_next_file (NULL , data);
    }
    DrawAll(main_view);
    return TRUE;
  }
  if (IsValidClick(event->x,event->y,main_view) == FALSE) {
    if(main_view->temporal_mode != TEMPMODE_EDIT_AREA) {
      if((IS_EMPTY(GetStoneXY(main_view->mouseGobanClickX,main_view->mouseGobanClickY)) == FALSE) ||
         (
           (cx - main_view->mouseGobanClickX < 2) && 
           (cy - main_view->mouseGobanClickY < 2) && 
           (cx - main_view->mouseGobanClickX > -2) && 
           (cy - main_view->mouseGobanClickY > -2) 
         )) {
        main_view->mouseAtGoban = FALSE;
        return TRUE;
      }
    }
    else {
      main_view->mouseAtGoban = FALSE;
      return TRUE;
    }
  }
  if(main_view->mouseAtGoban == FALSE) {
    return TRUE;
  }
  if(event->button == 1 && main_view->gobanPixmap != NULL) {
    main_view->mouseGobanCurrentX = cx;
    main_view->mouseGobanCurrentY = cy;
    at = XY2Location(main_view->mouseGobanCurrentX,main_view->mouseGobanCurrentY);
    main_view->mouseAtGoban = FALSE;
    color = main_view->playerInTurn;
    if(main_view->temporal_mode == TEMPMODE_CHANGE_ORIENTATION) {
      if(main_view->currentNode != NULL) {
        for(i=0; i<8; i++) {
          if(at == RotateLocation(main_view->currentNode->at, i)) {
            RotateGoban(main_view, i);
            main_view->temporal_mode = TEMPMODE_NONE;
            break;
          }
        }
      }
      DrawAll(main_view);
      return TRUE;
    }
    if(main_view->temporal_mode == TEMPMODE_EDIT_AREA) {
      if( (main_view->mouseGobanCurrentX != main_view->mouseGobanClickX) || /* If mouse is released at different position than where it clicked */
          (main_view->mouseGobanCurrentY != main_view->mouseGobanClickY)) {
        DragArea(main_view);
      }
      main_view->temporal_mode = TEMPMODE_NONE;
      DrawAll(main_view);
      return TRUE;
    }
    if(main_view->CurrentMouseMode == main_view->play_item) { /* Mouse mode is play mode */
      if( (main_view->mouseGobanCurrentX == main_view->mouseGobanClickX) && /* If mouse is released where it was clicked */
          (main_view->mouseGobanCurrentY == main_view->mouseGobanClickY) &&
          (IS_EMPTY(GetStoneXY(main_view->mouseGobanClickX,main_view->mouseGobanClickY)) == FALSE)) { /* and it was not at vacant position when clicked */
        Panicable(main_view);
        ExitNode(main_view);
        main_view->currentNode = FindAncestorAtXY(main_view->currentNode,main_view->mouseGobanClickX,main_view->mouseGobanClickY);
        EnterNode(main_view);
      }
      if( (IS_EMPTY(GetStoneXY(main_view->mouseGobanCurrentX,main_view->mouseGobanCurrentY)) == TRUE) && /* If mouse is at vacant position */
          (IS_EMPTY(GetStoneXY(main_view->mouseGobanClickX,main_view->mouseGobanClickY)) == FALSE)) { /* and it was not at vacant position when clicked */
        Panicable(main_view);
        ExitNode(main_view);
        CreateWhatIf( main_view->mouseGobanClickX, main_view->mouseGobanClickY, main_view->mouseGobanCurrentX, main_view->mouseGobanCurrentY, main_view); /* create what if branch for case if the stone were at where it is dragged at */
#ifndef PRODUCTION_VERSION  
        msg2log("\nCreate what if\n\nCreate what if feature creates copy of currently selected line of play with the difference that the stone user dragged where played at where it is dragged at. This could make moves later in game in theory illegal and thus you should check the created what if line.\n\nAfter using create what if feature you could use prune variations feature to make the created line of play only line of the play. This is usefull in case create what if is used for correcting earlier mistake in recording tournament game. There is no need to use prune variations feature immediately after create what if. If you did not want what happened, click panic button below cursor keys.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
        EnterNode(main_view);
        main_view->file_edited = TRUE;
        DrawAll(main_view);
        return TRUE;
      }
      /* if( (IS_EMPTY(GetStoneXY(main_view->mouseGobanCurrentX,main_view->mouseGobanCurrentY)) == TRUE) && / * If mouse is at vacant position * /
          (IS_EMPTY(GetStoneXY(main_view->mouseGobanClickX,main_view->mouseGobanClickY)) == TRUE)) { / * and it was at vacant position when clicked */
      if( (IS_EMPTY(GetStoneXY(main_view->mouseGobanClickX,main_view->mouseGobanClickY)) == TRUE)) { /* If mouse was at vacant position when clicked */
        if((main_view->mouseGobanCurrentX != main_view->mouseGobanClickX) || (main_view->mouseGobanCurrentY != main_view->mouseGobanClickY)) { /* if release position is not the same as click position */
          main_view->temporal_mode = TEMPMODE_EDIT_AREA;
          if(main_view->mouseGobanCurrentX > main_view->mouseGobanClickX) {
            main_view->select_left = main_view->mouseGobanClickX;
            main_view->select_right = main_view->mouseGobanCurrentX;
          }
          else {
            main_view->select_right = main_view->mouseGobanClickX;
            main_view->select_left = main_view->mouseGobanCurrentX;
          }
          if(main_view->mouseGobanCurrentY > main_view->mouseGobanClickY) {
            main_view->select_top = main_view->mouseGobanClickY;
            main_view->select_bottom = main_view->mouseGobanCurrentY;
          }
          else {
            main_view->select_bottom = main_view->mouseGobanClickY;
            main_view->select_top = main_view->mouseGobanCurrentY;
          }
#ifndef PRODUCTION_VERSION  
          msg2log("\nDrag area\n\nIn play mode you can select area by dragging from empty corner of the area to the diagonally opposite corner. After that stones that has been selected become transparent. You can drag the selected stones at new location. Be sure to release stylus near intersection of goban lines to get expected drag.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
          DrawAll(main_view);
          return TRUE;
        }
        if(MakeMove(main_view->mouseGobanCurrentX,main_view->mouseGobanCurrentY,color,main_view,FALSE) == TRUE) { /* and if move to current position is legal */
          Panicable(main_view);
          ExitNode(main_view);
          node = AddMove2GameTree(main_view->currentNode, main_view->child, main_view->mouseGobanCurrentX, main_view->mouseGobanCurrentY, color, main_view->koAt); /* add it into game tree */
          main_view->file_edited = TRUE;
          if(main_view->child == NULL) { /* If game had no moves before */
            main_view->child = node; /* Set the node to be the first move of the game */
          }
          main_view->currentNode = node; /* Set the position after the move to be the current position */
          if(main_view->analysisAutoRefute == TRUE) {
            node = RefutePlay(main_view->currentNode);
            if(node != main_view->currentNode) {
#ifndef PRODUCTION_VERSION  
              msg2log("\nAnalysis autorefute\n\nWhen analysis autorefute option is on and you make a play for which analysis line exists, the analysis is searched for to find refutation for the move. If such is found, it is automatically made.\n\nThe idea of analysis autorefute is to help you to concentrate on positions where player in turn needs to find an alternative that has not yet been analysed.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
              main_view->currentNode = node;
            }
          }
          set_active(main_view->currentNode); /* Set the active branches from current position to root of the game */
          SetPlayerInTurn(main_view); /* Change color of player in turn */
          EnterNode(main_view);
        }
      }
    }
    if(main_view->CurrentMouseMode == main_view->training_item) { /* Mouse mode is training mode */
      if( (IS_EMPTY(GetStoneXY(main_view->mouseGobanCurrentX,main_view->mouseGobanCurrentY)) == TRUE) && /* If mouse is at vacant position */
          (IS_EMPTY(GetStoneXY(main_view->mouseGobanClickX,main_view->mouseGobanClickY)) == TRUE)) { /* and it was at vacant position when clicked */
        if(main_view->currentNode == NULL) {
          node = main_view->child;
        }  
        else {
          node = main_view->currentNode->child;
        }
        if(node == NULL) {
          UserMessage(GTK_WIDGET(main_view->goban_view), NULL, "End of variation.");
        }
        else {
          if(node->at == at) {
            if(main_view->isGoProblem == FALSE) {
              UserMessage(GTK_WIDGET(main_view->goban_view), NULL, "Correct. Mainline.");
            }
            GuessRight();
            Panicable(main_view);
            ExitNode(main_view);
            main_view->currentNode = node;
            if(main_view->step_2 == TRUE) {
              if(main_view->currentNode->child != NULL) {
                main_view->currentNode = RandomSibling(main_view->currentNode->child);
              }
            }
            EnterNode(main_view);
            PrintTrainingStatistics(main_view->isGoProblem);
            if(main_view->isGoProblem == FALSE) {
#ifndef PRODUCTION_VERSION  
              msg2log("\nCorrectness determination\n\nCorrectness calculation bases on following assumptations\n- Guessing all moves correctly is 100%\n- Educated use of hint, cheat and quess without go-knowledge is in average 0%", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
            }
            else { /* If it is go-problem */
              if(main_view->currentNode != NULL) { 
                if(main_view->currentNode->sgfC != NULL) { /* and node has message */
                  UserMessage(GTK_WIDGET(main_view->goban_view), NULL, main_view->currentNode->sgfC); /* show message there is */
                }
              }
            }
            DrawAll(main_view);
            return TRUE;
          }
          while(node->nodeHeight < node->sibling->nodeHeight) {
            node = node->sibling;
            if(node->at == at) {
              if(main_view->isGoProblem == FALSE) {
                UserMessage(GTK_WIDGET(main_view->goban_view), NULL, "Entering variation line.");
              }
              GuessRight();
              Panicable(main_view);
              ExitNode(main_view);
              main_view->currentNode = node;
              if(main_view->step_2 == TRUE) {
                if(main_view->currentNode->child != NULL) {
                  main_view->currentNode = RandomSibling(main_view->currentNode->child);
                }
              }
              EnterNode(main_view);
              PrintTrainingStatistics(main_view->isGoProblem);
              if(main_view->isGoProblem == TRUE) { /* If it is go-problem */
                if(main_view->currentNode != NULL) { 
                  if(main_view->currentNode->sgfC != NULL) { /* and node has message */
                    UserMessage(GTK_WIDGET(main_view->goban_view), NULL, main_view->currentNode->sgfC); /* show message there is */
                  }
                }
              }
              DrawAll(main_view);
              return TRUE;
            }
          }
          UserMessage(GTK_WIDGET(main_view->goban_view), NULL, "Incorrect.");
          GuessWrong();
          return TRUE;
        }
      }
    }
    
    if(main_view->CurrentMouseMode == main_view->territory_item) { /* Mouse mode is territory counting */
      if(main_view->currentNode != NULL) {
        if(IS_EMPTY(GetStoneXY(main_view->mouseGobanCurrentX,main_view->mouseGobanCurrentY)) == TRUE) { /* If mouse is at vacant position */
          if(IsPointInList(at, main_view->currentNode->sgfTB) == TRUE) {
            main_view->currentNode->sgfTB = RemovePoint(main_view->currentNode->sgfTB, at);
          }
          else {
            if(IsPointInList(at, main_view->currentNode->sgfTW) == TRUE) {
              main_view->currentNode->sgfTW = RemovePoint(main_view->currentNode->sgfTW, at);
            }
            else {
              if(IsNeighbour(at, main_view->currentNode->sgfTW, STONE_WHITE, main_view->currentNode->sgfTB) == TRUE) {
                main_view->currentNode->sgfTW = AddPoint2List(at, main_view->currentNode->sgfTW);
              }
              else {
                if(IsNeighbour(at, main_view->currentNode->sgfTB, STONE_BLACK, main_view->currentNode->sgfTW) == TRUE) {
                  main_view->currentNode->sgfTB = AddPoint2List(at, main_view->currentNode->sgfTB);
                }
              }
            }
          }
        }
        else { /* If mouse is at stone */
          SwapGroupStatus(main_view, at);
        }
      }
    }
    
    if(main_view->CurrentMouseMode == main_view->search_item) { /* Mouse mode is search mode */
      if( (main_view->mouseGobanCurrentX == main_view->mouseGobanClickX) && /* If mouse is released where it was clicked */
          (main_view->mouseGobanCurrentY == main_view->mouseGobanClickY)) {
        node = main_view->currentNode;
        do {
          node = forward_in_tree( node, main_view->child ); /* Change position to the next one */
          if(node == NULL) {
            break;
          }
        } while(node->child != NULL);
        node = FindAncestorAtXY(node,main_view->mouseGobanClickX,main_view->mouseGobanClickY);
        if((node != NULL) && (node != main_view->currentNode)) {
          Panicable(main_view);
          ExitNode(main_view);
          main_view->currentNode = node;
          EnterNode(main_view);
        }
      }
    }

    if(main_view->CurrentMouseMode == main_view->diagram_item) { /* Mouse mode is diagram view */
      if( (main_view->mouseGobanCurrentX == main_view->mouseGobanClickX) && /* If mouse is released where it was clicked */
          (main_view->mouseGobanCurrentY == main_view->mouseGobanClickY)) { /* regardless what there is */
        Panicable(main_view);
        ExitNode(main_view);
        node = AddMove2GameTree(main_view->currentNode, main_view->child, main_view->mouseGobanCurrentX, main_view->mouseGobanCurrentY, color, main_view->koAt); /* add it into game tree */
        main_view->file_edited = TRUE;
        if(main_view->child == NULL) { /* If game had no moves before */
          main_view->child = node; /* Set the node to be the first move of the game */
        }
        main_view->currentNode = node; /* Set the position after the move to be the current position */
        set_active(main_view->currentNode); /* Set the active branches from current position to root of the game */
        SetPlayerInTurn(main_view); /* Change color of player in turn */
        EnterNode(main_view);
      }
      else {
        if( (IS_EMPTY(GetStoneXY(main_view->mouseGobanClickX,main_view->mouseGobanClickY)) == FALSE)) { /* If mouse was not at vacant position when clicked */
          Panicable(main_view);
          ExitNode(main_view);
          /*
          node = findStartOfNumbering(main_view);
          if(node == NULL) {
            node = main_view->currentNode;
            if(node != NULL) {
              while(node->parent != NULL) {
                node = node->parent;
              }
            }
          }
          */
          if(MoveStoneInHistory(main_view->currentNode,main_view->mouseGobanClickX,main_view->mouseGobanClickY,main_view->mouseGobanCurrentX,main_view->mouseGobanCurrentY) == FALSE) { /* Change location where the stone was played */ 
            CreateWhatIf( main_view->mouseGobanClickX, main_view->mouseGobanClickY, main_view->mouseGobanCurrentX, main_view->mouseGobanCurrentY, main_view); /* or if that fails, create what if branch for case if the stone were at where it is dragged at */
          }
          EnterNode(main_view);
#ifndef PRODUCTION_VERSION  
          msg2log("\nMove stone in history\n\nMove stone in history feature can be used to re-position played stone. Unlike create what if feature this feature does not change the game tree. Feature always moves the stone that is visible currently. So, if you move a stone away ko, the moved stone depends on current numbering options as they affect on which of the stones played at the location is shown on the goban.\n\nThis feature is efficient with fill goban feature to construct games from book diagrams presenting entire game in a single diagram.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
          main_view->file_edited = TRUE;
        }
      }    
    }

    if(main_view->CurrentMouseMode == main_view->addblack_item) { /* Mouse mode is add black stones */
      if(main_view->currentNode == NULL) { /* if we are at the root */
        main_view->sgfAW = RemovePoint(main_view->sgfAW, at); /* remove white setup stone at the position if there is any */
        if(IsPointInList(at, main_view->sgfAB) == FALSE) { /* if there isn't black setup stone at the position */
          point = NewPoint(at); /* add such */
          point->next = main_view->sgfAB;
          main_view->sgfAB = point;
        }
        else {
          main_view->sgfAB = RemovePoint(main_view->sgfAB, at); /* else remove such */
        }
        point = main_view->sgfAB;
        main_view->sgfHA = 0; /* Reset amount of handicap stones */
        while((point != NULL) && (main_view->sgfAW == NULL)) { /* while there are black stones, but no white stones */
          main_view->sgfHA++; /* count black stones as handicap stones */
          point = point->next;
        }
        main_view->sgfPL = STONE_WHITE; /* Set position white to play */
      }
      else { /* If we are outside of the root position */
        if(main_view->currentNode->at != AT_IS_SETUP) { /* if position is not setup position */
		    main_view->currentNode = NewNode(main_view->currentNode,STONE_NONE,get_bigsister(main_view->currentNode->child),AT_IS_SETUP,-1); /* create such */
        }
		  main_view->currentNode->stone = STONE_BLACK; /* Set position white to play */
        main_view->currentNode->sgfAW = RemovePoint(main_view->currentNode->sgfAW, at); /* clear white setup stone if any */
        main_view->currentNode->sgfAE = RemovePoint(main_view->currentNode->sgfAE, at); /* clear empty setup stone if any */
        if(GetStoneAt(at) != STONE_BLACK) { /* if there is no black stone at the location */
          point = NewPoint(at); /* add there black setup stone */
          point->next = main_view->currentNode->sgfAB;
          main_view->currentNode->sgfAB = point;
        }
        else { /* but if there is */
          if(IsPointInList(at, main_view->currentNode->sgfAB) == TRUE) { /* and it is black setup stone */
            main_view->currentNode->sgfAB = RemovePoint(main_view->currentNode->sgfAB, at); /* remove it */
          }
          else { /* and if it is not setup stone */
            point = NewPoint(at); /* add empty setup stone */
            point->next = main_view->currentNode->sgfAE;
            main_view->currentNode->sgfAE = point;
          }
        }
      }
    }
    if(main_view->CurrentMouseMode == main_view->addwhite_item) { /* Mouse mode is add white stones */
      if(main_view->currentNode == NULL) { /* If we are at the root */
        main_view->sgfAB = RemovePoint(main_view->sgfAB, at); /* remove black setup stone if any */
        if(IsPointInList(at, main_view->sgfAW) == FALSE) { /* and if the isn't white setup stone */
          point = NewPoint(at); /* add such */
          point->next = main_view->sgfAW;
          main_view->sgfAW = point;
        }
        else { /* and if there is */
          main_view->sgfAW = RemovePoint(main_view->sgfAW, at); /* remove it */
        }
        point = main_view->sgfAB;
        main_view->sgfHA = 0; /* Reset amount of handicap stones */
        while((point != NULL) && (main_view->sgfAW == NULL)) { /* while there are black stones, but no white stones */
          main_view->sgfHA++; /* count black stones as handicap stones */
          point = point->next;
        }
        main_view->sgfPL = STONE_BLACK; /* Set position black to play */
      }
      else { /* If we are outside of the root position */
        if(main_view->currentNode->at != AT_IS_SETUP) { /* if position is not setup position */
		    main_view->currentNode = NewNode(main_view->currentNode,STONE_NONE,get_bigsister(main_view->currentNode->child),AT_IS_SETUP,-1); /* create such */
        }
		  main_view->currentNode->stone = STONE_WHITE; /* Set position black to play */
        main_view->currentNode->sgfAB = RemovePoint(main_view->currentNode->sgfAB, at); /* clear black setup stone if any */
        main_view->currentNode->sgfAE = RemovePoint(main_view->currentNode->sgfAE, at); /* clear empty setup stone if any */
        if(GetStoneAt(at) != STONE_WHITE) { /* if there is no white stone at the location */
          point = NewPoint(at); /* add there white setup stone */
          point->next = main_view->currentNode->sgfAW;
          main_view->currentNode->sgfAW = point;
        }
        else { /* but if there is */
          if(IsPointInList(at, main_view->currentNode->sgfAW) == TRUE) { /* and it is white setup stone */
            main_view->currentNode->sgfAW = RemovePoint(main_view->currentNode->sgfAW, at); /* remove it */
          }
          else { /* and if it is not setup stone */
            point = NewPoint(at); /* add empty setup stone */
            point->next = main_view->currentNode->sgfAE;
            main_view->currentNode->sgfAE = point;
          }
        }
      }
    }
    if(main_view->CurrentMouseMode == main_view->triangle_item) { /* Mouse mode to toggle triangles */
      if(main_view->currentNode == NULL) {
        main_view->sgfMA = RemovePoint(main_view->sgfMA, at);
        main_view->sgfSQ = RemovePoint(main_view->sgfSQ, at);
        main_view->sgfCR = RemovePoint(main_view->sgfCR, at);
        if(IsPointInList(at, main_view->sgfTR) == FALSE) {
          point = NewPoint(at);
          point->next = main_view->sgfTR;
          main_view->sgfTR = point;
        }
        else {
          main_view->sgfTR = RemovePoint(main_view->sgfTR, at);
        }
        if(IsLabelInList(at, main_view->sgfLB) == TRUE) {
          DestroyLabellist(main_view->sgfLB);
          main_view->sgfLB = NULL;
        }
      }
      else {
        main_view->currentNode->sgfMA = RemovePoint(main_view->currentNode->sgfMA, at);
        main_view->currentNode->sgfSQ = RemovePoint(main_view->currentNode->sgfSQ, at);
        main_view->currentNode->sgfCR = RemovePoint(main_view->currentNode->sgfCR, at);
        if(IsPointInList(at, main_view->currentNode->sgfTR) == FALSE) {
          point = NewPoint(at);
          point->next = main_view->currentNode->sgfTR;
          main_view->currentNode->sgfTR = point;
        }
        else {
          main_view->currentNode->sgfTR = RemovePoint(main_view->currentNode->sgfTR, at);
        }
        if(IsLabelInList(at, main_view->currentNode->sgfLB) == TRUE) {
          DestroyLabellist(main_view->currentNode->sgfLB);
          main_view->currentNode->sgfLB = NULL;
        }
      }
    }  
    if(main_view->CurrentMouseMode == main_view->square_item) { /* Mouse mode to toggle squares */
      if(main_view->currentNode == NULL) {
        main_view->sgfMA = RemovePoint(main_view->sgfMA, at);
        main_view->sgfTR = RemovePoint(main_view->sgfTR, at);
        main_view->sgfCR = RemovePoint(main_view->sgfCR, at);
        if(IsPointInList(at, main_view->sgfSQ) == FALSE) {
          point = NewPoint(at);
          point->next = main_view->sgfSQ;
          main_view->sgfSQ = point;
        }
        else {
          main_view->sgfSQ = RemovePoint(main_view->sgfSQ, at);
        }
        if(IsLabelInList(at, main_view->sgfLB) == TRUE) {
          DestroyLabellist(main_view->sgfLB);
          main_view->sgfLB = NULL;
        }
      }
      else {
        main_view->currentNode->sgfMA = RemovePoint(main_view->currentNode->sgfMA, at);
        main_view->currentNode->sgfTR = RemovePoint(main_view->currentNode->sgfTR, at);
        main_view->currentNode->sgfCR = RemovePoint(main_view->currentNode->sgfCR, at);
        if(IsPointInList(at, main_view->currentNode->sgfSQ) == FALSE) {
          point = NewPoint(at);
          point->next = main_view->currentNode->sgfSQ;
          main_view->currentNode->sgfSQ = point;
        }
        else {
          main_view->currentNode->sgfSQ = RemovePoint(main_view->currentNode->sgfSQ, at);
        }
        if(IsLabelInList(at, main_view->currentNode->sgfLB) == TRUE) {
          DestroyLabellist(main_view->currentNode->sgfLB);
          main_view->currentNode->sgfLB = NULL;
        }
      }
    }  
    if(main_view->CurrentMouseMode == main_view->circle_item) { /* Mouse mode to toggle circles */
      if(main_view->currentNode == NULL) {
        main_view->sgfMA = RemovePoint(main_view->sgfMA, at);
        main_view->sgfSQ = RemovePoint(main_view->sgfSQ, at);
        main_view->sgfTR = RemovePoint(main_view->sgfTR, at);
        if(IsPointInList(at, main_view->sgfCR) == FALSE) {
          point = NewPoint(at);
          point->next = main_view->sgfCR;
          main_view->sgfCR = point;
        }
        else {
          main_view->sgfCR = RemovePoint(main_view->sgfCR, at);
        }
        if(IsLabelInList(at, main_view->sgfLB) == TRUE) {
          DestroyLabellist(main_view->sgfLB);
          main_view->sgfLB = NULL;
        }
      }
      else {
        main_view->currentNode->sgfMA = RemovePoint(main_view->currentNode->sgfMA, at);
        main_view->currentNode->sgfSQ = RemovePoint(main_view->currentNode->sgfSQ, at);
        main_view->currentNode->sgfTR = RemovePoint(main_view->currentNode->sgfTR, at);
        if(IsPointInList(at, main_view->currentNode->sgfCR) == FALSE) {
          point = NewPoint(at);
          point->next = main_view->currentNode->sgfCR;
          main_view->currentNode->sgfCR = point;
        }
        else {
          main_view->currentNode->sgfCR = RemovePoint(main_view->currentNode->sgfCR, at);
        }
        if(IsLabelInList(at, main_view->currentNode->sgfLB) == TRUE) {
          DestroyLabellist(main_view->currentNode->sgfLB);
          main_view->currentNode->sgfLB = NULL;
        }
      }
    }  
    if(main_view->CurrentMouseMode == main_view->cross_item) { /* Mouse mode to toggle crosses */
      if(main_view->currentNode == NULL) {
        main_view->sgfTR = RemovePoint(main_view->sgfTR, at);
        main_view->sgfSQ = RemovePoint(main_view->sgfSQ, at);
        main_view->sgfCR = RemovePoint(main_view->sgfCR, at);
        if(IsPointInList(at, main_view->sgfMA) == FALSE) {
          point = NewPoint(at);
          point->next = main_view->sgfMA;
          main_view->sgfMA = point;
        }
        else {
          main_view->sgfMA = RemovePoint(main_view->sgfMA, at);
        }
        if(IsLabelInList(at, main_view->sgfLB) == TRUE) {
          DestroyLabellist(main_view->sgfLB);
          main_view->sgfLB = NULL;
        }
      }
      else {
        main_view->currentNode->sgfTR = RemovePoint(main_view->currentNode->sgfTR, at);
        main_view->currentNode->sgfSQ = RemovePoint(main_view->currentNode->sgfSQ, at);
        main_view->currentNode->sgfCR = RemovePoint(main_view->currentNode->sgfCR, at);
        if(IsPointInList(at, main_view->currentNode->sgfMA) == FALSE) {
          point = NewPoint(at);
          point->next = main_view->currentNode->sgfMA;
          main_view->currentNode->sgfMA = point;
        }
        else {
          main_view->currentNode->sgfMA = RemovePoint(main_view->currentNode->sgfMA, at);
        }
        if(IsLabelInList(at, main_view->currentNode->sgfLB) == TRUE) {
          DestroyLabellist(main_view->currentNode->sgfLB);
          main_view->currentNode->sgfLB = NULL;
        }
      }
    }  
    if(main_view->CurrentMouseMode == main_view->labeling_item) { /* Mouse mode to set labels A,B,C,... */
      if(main_view->currentNode == NULL) {
        if(IsLabelInList(at, main_view->sgfLB) == FALSE) {
          main_view->sgfTR = RemovePoint(main_view->sgfTR, at);
          main_view->sgfSQ = RemovePoint(main_view->sgfSQ, at);
          main_view->sgfCR = RemovePoint(main_view->sgfCR, at);
          main_view->sgfMA = RemovePoint(main_view->sgfMA, at);
          llist = NewLabel(at);
          llist->label[0] = 'A' + GetUsedLabelCount(main_view,TRUE);
          llist->next = main_view->sgfLB;
          main_view->sgfLB = llist;
        }
        else {
          DestroyLabellist(main_view->sgfLB);
          main_view->sgfLB = NULL;
        }
      }
      else {
        if(IsLabelInList(at, main_view->currentNode->sgfLB) == FALSE) {
          main_view->currentNode->sgfTR = RemovePoint(main_view->currentNode->sgfTR, at);
          main_view->currentNode->sgfSQ = RemovePoint(main_view->currentNode->sgfSQ, at);
          main_view->currentNode->sgfCR = RemovePoint(main_view->currentNode->sgfCR, at);
          main_view->currentNode->sgfMA = RemovePoint(main_view->currentNode->sgfMA, at);
          llist = NewLabel(at);
          llist->label[0] = 'A' + GetUsedLabelCount(main_view,TRUE);
          llist->next = main_view->currentNode->sgfLB;
          main_view->currentNode->sgfLB = llist;
        }
        else {
          DestroyLabellist(main_view->currentNode->sgfLB);
          main_view->currentNode->sgfLB = NULL;
        }
      }
    }  
    if(main_view->CurrentMouseMode == main_view->numbering_item) { /* Mouse mode to set labels 1,2,3,... */
      if(main_view->currentNode == NULL) {
        if(IsLabelInList(at, main_view->sgfLB) == FALSE) {
          main_view->sgfTR = RemovePoint(main_view->sgfTR, at);
          main_view->sgfSQ = RemovePoint(main_view->sgfSQ, at);
          main_view->sgfCR = RemovePoint(main_view->sgfCR, at);
          main_view->sgfMA = RemovePoint(main_view->sgfMA, at);
          llist = NewLabel(at);
          digits = 100;
          index = 2;
          number = GetUsedLabelCount(main_view,FALSE) + 1;
          while(number > 0) { /* Eliminate spaces in front of the number */
            if(number >= digits) {
              break;
            }
            digits /= 10;
            index--;
          }
          while(number > 0) {
            llist->label[index] = '0' + number%10;
            number /= 10;
            index--;
          }
          llist->next = main_view->sgfLB;
          main_view->sgfLB = llist;
        }
        else {
          DestroyLabellist(main_view->sgfLB);
          main_view->sgfLB = NULL;
        }
      }
      else {
        if(IsLabelInList(at, main_view->currentNode->sgfLB) == FALSE) {
          main_view->currentNode->sgfTR = RemovePoint(main_view->currentNode->sgfTR, at);
          main_view->currentNode->sgfSQ = RemovePoint(main_view->currentNode->sgfSQ, at);
          main_view->currentNode->sgfCR = RemovePoint(main_view->currentNode->sgfCR, at);
          main_view->currentNode->sgfMA = RemovePoint(main_view->currentNode->sgfMA, at);
          llist = NewLabel(at);
          digits = 100;
          index = 2;
          number = GetUsedLabelCount(main_view,FALSE) + 1;
          while(number > 0) { /* Eliminate spaces in front of the number */
            if(number >= digits) {
              break;
            }
            digits /= 10;
            index--;
          }
          while(number > 0) {
            llist->label[index] = '0' + number%10;
            number /= 10;
            index--;
          }
          llist->next = main_view->currentNode->sgfLB;
          main_view->currentNode->sgfLB = llist;
        }
        else {
          DestroyLabellist(main_view->currentNode->sgfLB);
          main_view->currentNode->sgfLB = NULL;
        }
      }
    }  
    DrawAll(main_view);
  }
#ifndef PRODUCTION_VERSION  
  msg2log("Exit button_release_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
  return TRUE;
}

gboolean button_release_gametree_event( GtkWidget * widget, GdkEventButton *event, gpointer data ) 
{
  gint x = 0, y = 0;
  Node * node;
  MainView * main_view;
  main_view = (MainView *) data;
#ifndef PRODUCTION_VERSION  
  msg2log("Enter button_release_gametree_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
 
  g_assert(main_view != NULL && main_view->data != NULL );
  
/*  return TRUE; */
  if(event->button == 1 && main_view->gametreePixmap != NULL) {
    y = event->y;
    x = event->x;
    node = GetNodeAt(main_view->child, Y2Height(y), X2Turn(x), ((main_view->isBigTree == FALSE) || (main_view->bigTreeVariations == TRUE)));
    if(node == main_view->mouseGtreeClick) { /* If clicked at stone */
      if((node != NULL) || ((Y2Height(y) == 0) && (X2Turn(x) == 0))) {
        Browsing();
        Panicable(main_view);
        ExitNode(main_view);
        main_view->currentNode = node; /* Jump to position of the clicked stone */
        set_active(main_view->currentNode); /* Set the active branches from current position to root of the game */
        SetPlayerInTurn(main_view);
        EnterNode(main_view);
      }
    }
    else {
      if(main_view->mouseGtreeClick != NULL) {
        if(node != NULL) {
          if(node->parent == main_view->mouseGtreeClick->parent) { /* If dragged under same parent */
            MoveBranchTo(main_view->mouseGtreeClick,node); /* Re-order branches under the parent */
            main_view->mouseAtGtree = FALSE;
            main_view->file_edited = TRUE;
            DrawAll(main_view);
			/* TODO: Is this return right? */
            return TRUE;
          }
        }
        if(Y2Height(y) < main_view->mouseGtreeClick->nodeHeight != 0) { /* If dragged upwards */
          node = LiftBranchUpwards(main_view->mouseGtreeClick,Y2Height(y)); /* Re-order branches to lift clicked node as first child upto draged height */
          if(node != NULL) {
            main_view->child = node;
          }
          main_view->file_edited = TRUE;
        }
      }
    }
    main_view->mouseAtGtree = FALSE;
#ifndef PRODUCTION_VERSION  
    msg2log("\nGame tree window\n\nVia game tree window you can\n- Jump to a position\n- Move branch upwards\n- Re-order children nodes of a position\n\nYou can jump to a position by clicking on a stone. If you drag stone on top of another stone that has same parent, the dragged stone is positioned at it's place on the different side from which the stone was dragged. If you drag stone upwards, it is tried to move upwards as high as possible, but not higher than to where the stone was dragged. This second type of dragging does not neccessary preserve respective order of ancestors siblings.\n\nYou cannot move or copy subtrees from place to another. That means that you cannot accidentally mess the game up via game tree window.\n\nYou may change the zooming of game tree with zoom in and zoom out buttons most of the time.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
    DrawAll(main_view);
  }
#ifndef PRODUCTION_VERSION  
  msg2log("Exit button_release_gametree_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
  return TRUE;
}

/* Pointer drag on goban */
gboolean motion_notify_goban_event( GtkWidget * widget, GdkEventMotion *event, gpointer data )
{
  Stone stone;
  MainView * main_view;
  int x, y;
  gint gobanX, gobanY;
  GdkModifierType state;

#ifndef PRODUCTION_VERSION  
  msg2log("Enter motion_notify_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
  main_view = (MainView *) data;
  g_assert(main_view != NULL && main_view->data != NULL );
  
  if(main_view->mouseAtGoban == FALSE) {
    return FALSE;
  } 
  
    if (event->is_hint) {
    gdk_window_get_pointer (event->window, &x, &y, &state);
  }
  else {
    x = event->x;
    y = event->y;
    state = event->state;
  }
  if (IsValidClick(x,y,main_view) == FALSE) {
    return TRUE;
  }
  if (state & GDK_BUTTON1_MASK && main_view->gobanPixmap != NULL) {
    gobanX = viewx2gobanx(x, main_view );
    gobanY = viewy2gobany(y, main_view );
    if((gobanX != main_view->mouseGobanCurrentX) || (gobanY != main_view->mouseGobanCurrentY)) {
      if(main_view->temporal_mode == TEMPMODE_EDIT_AREA) {
        for(x = main_view->select_left; x <= main_view->select_right; x++) {
          for(y = main_view->select_top; y <= main_view->select_bottom; y++) {
            stone = GetStoneXY(x,y);
            switch(stone) {
              case STONE_TRANSPARENT_BLACK: ;
              case STONE_BLACK: ;
              case STONE_WHITE: ;
              case STONE_TRANSPARENT_WHITE:
                MarkStone(main_view, x + main_view->mouseGobanCurrentX - main_view->mouseGobanClickX, y + main_view->mouseGobanCurrentY - main_view->mouseGobanClickY, &stone);
                break;
			  default:
				break;
            }
          }
        }
        main_view->mouseGobanCurrentX = gobanX;
        main_view->mouseGobanCurrentY = gobanY;
        for(x = main_view->select_left; x <= main_view->select_right; x++) {
          for(y = main_view->select_top; y <= main_view->select_bottom; y++) {
            stone = GetStoneXY(x,y);
            switch(stone) {
              case STONE_TRANSPARENT_BLACK: ;
              case STONE_BLACK: ;
              case STONE_WHITE: ;
              case STONE_TRANSPARENT_WHITE:
                MarkStone(main_view, x + main_view->mouseGobanCurrentX - main_view->mouseGobanClickX, y + main_view->mouseGobanCurrentY - main_view->mouseGobanClickY, &stone);
                break;
			  default:
                break;
            }
          }
        }
      }
      else {
        if((main_view->CurrentMouseMode == main_view->play_item) && (IS_EMPTY(GetStoneXY(main_view->mouseGobanClickX,main_view->mouseGobanClickY)) == TRUE)) {
          MarkArea(main_view,main_view->mouseGobanClickX,main_view->mouseGobanClickY,main_view->mouseGobanCurrentX,main_view->mouseGobanCurrentY);
          main_view->mouseGobanCurrentX = gobanX;
          main_view->mouseGobanCurrentY = gobanY;
          MarkArea(main_view,main_view->mouseGobanClickX,main_view->mouseGobanClickY,main_view->mouseGobanCurrentX,main_view->mouseGobanCurrentY);
        }
        else {
          MarkSelected(main_view, main_view->mouseGobanCurrentX,main_view->mouseGobanCurrentY);
          main_view->mouseGobanCurrentX = gobanX;
          main_view->mouseGobanCurrentY = gobanY;
          MarkSelected(main_view, main_view->mouseGobanCurrentX,main_view->mouseGobanCurrentY);
        }
      }
    }
  }
#ifndef PRODUCTION_VERSION  
  msg2log("Exit motion_notify_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
  return TRUE;
}

gboolean motion_notify_gametree_event( GtkWidget * widget, GdkEventMotion *event, gpointer data )
{
  Node * node;
  MainView * main_view;
  int x, y;
  gint gtreeX, gtreeY;
  GdkModifierType state;

 #ifndef PRODUCTION_VERSION  
  msg2log("Enter motion_notify_gametree_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
 #endif
  main_view= (MainView *) data;
  g_assert(main_view != NULL && main_view->data != NULL );
  
  if(main_view->mouseAtGtree == FALSE) {
    return FALSE;
  } 
  
 
  if (event->is_hint) {
    gdk_window_get_pointer (event->window, &x, &y, &state);
  }
  else {
    x = event->x;
    y = event->y;
    state = event->state;
  }
  if (state & GDK_BUTTON1_MASK && main_view->gametreePixmap != NULL) {
    gtreeY = y;
    gtreeX = x;
    node = GetNodeAt(main_view->child, Y2Height(gtreeY), X2Turn(gtreeX), ((main_view->isBigTree == FALSE) || (main_view->bigTreeVariations == TRUE)));
    if(main_view->mouseGtreeCurrent != node) {
      if((main_view->mouseGtreeCurrent == main_view->mouseGtreeClick) || (node == main_view->mouseGtreeClick)) {
        MarkBranch(main_view, main_view->mouseGtreeClick);
      }
      main_view->mouseGtreeCurrent = node;
    }
  }
#ifndef PRODUCTION_VERSION  
  msg2log("Exit motion_notify_gametree_event", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
  return TRUE;
}

/* cut */
void callback_edit_cut( GtkAction * action, gpointer data )
{
    /* connect pointer to our MainView struct */
    MainView *mainview = NULL;
    mainview = ( MainView * ) data;
    g_assert(mainview != NULL && mainview->data != NULL );

    /* do cut */
    gtk_text_buffer_cut_clipboard(GTK_TEXT_BUFFER(mainview->debugBuffer), mainview->clipboard, TRUE);
}

/* copy */
void callback_edit_copy( GtkAction * action, gpointer data )
{
    /* connect pointer to our MainView struct */
    MainView *mainview = NULL;
    mainview = ( MainView * ) data;
    g_assert(mainview != NULL && mainview->data != NULL );

    /* do copy */
    gtk_text_buffer_copy_clipboard (GTK_TEXT_BUFFER(mainview->debugBuffer), mainview->clipboard);
}

/* paste */
void callback_edit_paste( GtkAction * action, gpointer data )
{
    /* connect pointer to our MainView struct */
    MainView *mainview = NULL;
    mainview = ( MainView * ) data;
    g_assert(mainview != NULL && mainview->data != NULL );

    /* do paste */
    gtk_text_buffer_paste_clipboard (GTK_TEXT_BUFFER (mainview->debugBuffer), mainview->clipboard, NULL, TRUE);
}

/* Open menu option */
void callback_open_menu(GtkAction *action, gpointer data) 
{
    MainView *mainview = (MainView*) data;
    g_assert(mainview != NULL && mainview->data != NULL); 
    GtkMenu *main_menu = hildon_window_get_menu(mainview->data->window);
    g_assert(main_menu != NULL);
    gtk_menu_popup(main_menu,NULL,NULL,NULL,NULL,0,gtk_get_current_event_time());
}

/* new */
void callback_file_new(GtkAction * action, gpointer data)
{
  gint answer;
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  /* save changes note if file is edited */
  if( mainview->file_edited ) {
    answer = interface_save_changes_note( mainview );
    if( answer == CONFRESP_YES ) {
      if( mainview->file_name == NULL ) {
        mainview->file_name = interface_file_chooser ( mainview, GTK_FILE_CHOOSER_ACTION_SAVE );
      }
      write_buffer_to_file ( mainview );
    }
  }
  /* clear buffer, filename and free buffer text */
  gtk_text_buffer_set_text ( GTK_TEXT_BUFFER (mainview->debugBuffer), "", -1 );
  mainview->file_name = NULL;
  mainview->file_edited = FALSE;
  /* Reset settings for new game */
  ResetGame(19, mainview);
#ifndef PRODUCTION_VERSION  
  msg2log("\nSelecting goban size and handicap\n\nWhen game has no moves in it at left of the goban some row numbers are marked. By clicking marked numbers you can request for smaller goban size. After you are satisfied with the goban size, you can click the marked numbers on the right of the goban to select handicap used. Once handicap has been chosen, goban size can no longer be changed. To get to initial situation before selecing goban size, click new game icon at the toolbar. After setting handicap you can reposition the handicap stones by dragging them to correct places. If you need more handicap stones than 9 or you want to place handicap stones in non-standard pattern, you can use add black stones mode for that purpose. You can switch the mode at popup that opens when you click aside goban at non-marked position.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  if(mainview->manualOnly == FALSE) {
    gtk_notebook_set_current_page(GTK_NOTEBOOK(mainview->notebook),GAME_PAGE);
  }
  EnterFile();
  DrawAll(mainview);
}

/* Rotate goban 90 degrees */
void callback_rotate(GtkAction * action, gpointer data)
{
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->temporal_mode = TEMPMODE_CHANGE_ORIENTATION;
  /* RotateGoban(mainview, 1); */
  UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Click where last played stone should be.");
#ifndef PRODUCTION_VERSION  
  msg2log("\nGoban orientation editing\n\nGoban orientation editing can be used when game is not at the root position. Preferable the last played stone should be unsymmetrical stone. Feature shows on the goban all locations where the last played stone could be as result of rotating and mirroring goban in different ways. Click the location where you want the last played stone be. As result all moves of the game are rotated and mirrored so that game remains basicly same.\n\nThis feature can be used for turning tables after tournament game recording if you want to change point of view from one player to another or to bystander perspective. Also, if you have similar fuseki in several games you can rotate and turn them so that they are easier to compare.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

/* Circle last played stone on all positions of the game */
void callback_circle_lp(GtkAction * action, gpointer data)
{
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  CircleLastPlayed(mainview->child);
#ifndef PRODUCTION_VERSION  
  msg2log("\nCircle last played\n\nCircle last played feature is ment for marking last played stones in the sgf file like Cgoban2 did. This may be convinient if file is ment to be viewed with editor that does not mark the last played move.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

/* Remove markings from nodes where is no comments */
void callback_remove_cm ( GtkAction * action, gpointer data )
{
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  RemoveCustomMarkings	(mainview->child);
#ifndef PRODUCTION_VERSION  
  msg2log("\nRemove custom markings\n\nRemove custom markings feature is ment for stripping away sgf syntax that interfare some sgf editors capabilities for searching positions. This feature assumes that if the node has something written in the node comment, then the markings in the node may have meaning relating to the comment and thus markings from such nodes are not cleaned.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

/* Fill goban with legal plays */
void callback_fill ( GtkAction * action, gpointer data )
{
  gint x = 0;
  gint y = 0;
  gboolean numbering = FALSE;
  Node * node = NULL;
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  Panicable(mainview);
  ExitNode(mainview);
  UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Click first unneccessary stone, then delete and switch to diagram mode to drag stones in correct places. You may want to change goban orientation to get unplaced stones at suitable side.");
  for(y=0; y<mainview->sgfSZ; y++) {
    for(x=0; x<mainview->sgfSZ; x++) {
      if(MakeMove(x,y,mainview->playerInTurn,mainview,FALSE) == TRUE) { /* If move to position is legal */
        node = AddMove2GameTree(mainview->currentNode, mainview->child, x, y, mainview->playerInTurn, -1); /* add it into game tree */
        mainview->file_edited = TRUE;
        if(mainview->child == NULL) { /* If game had no moves before */
          mainview->child = node; /* Set the node to be the first move of the game */
        }
        mainview->currentNode = node; /* Set the position after the move to be the current position */
        set_active(mainview->currentNode); /* Set the active branches from current position to root of the game */
        SetPlayerInTurn(mainview); /* Change color of player in turn */
      }
    }
  }
  numbering = mainview->diagramview;
  mainview->diagramview = TRUE;
  EnterNode(mainview);
#ifndef PRODUCTION_VERSION  
  msg2log("\nFill goban\n\nFill goban feature is ment for assisting recording games from book diagrams that contains the game in one or a couple diagrams. The idea is that you fill goban with stones, then delete stones that are not needed for the diagram and after that drag the stones in the correct place. This makes it possible that you construct the game in the visual order of the diagram instead of the move order of the game. Because of this you would not need to find where the next play is in the diagram.\n\nYou may wish to use goban orientation editing to rotate the stones at the side of the goban at which the diagram in the book has least played stones.\n\nIf game is in several diagrams, make first diagram as described above, then fill goban again and delete unneccessary stones. After that place numbering at the begining of the diagram and drag the number stones at correct places like with first diagram.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
  mainview->diagramview = numbering;
}

void callback_create_problem ( GtkAction * action, gpointer data )
{
  gint x = 0, y = 0, at = 0;
  Pointlist * listswap = NULL;
  Labellist * labelswap = NULL;
  gchar * noteswap = NULL;
  Node * node;
  Node * sister;
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  if((mainview->temporal_mode == TEMPMODE_EDIT_AREA) && (mainview->currentNode != NULL)) {
    ExitNode(mainview);
    for(x=0; x<mainview->sgfSZ; x++) {
      for(y=0; y<mainview->sgfSZ; y++) {
        if((x < mainview->select_left) || (x > mainview->select_right) || (y < mainview->select_top) || (y > mainview->select_bottom)) {
          SetStoneAt(XY2Location(x, y), STONE_NONE); /* Clear stones outside the selected area */
        } 
      }
    }
    /* Clear set up stones */
    DestroyPointlist(mainview->sgfAB); /* release black stones */
    DestroyPointlist(mainview->sgfAW); /* release white stones */
    mainview->sgfAB = NULL; /* forget pointer to released black stones */
    mainview->sgfAW = NULL; /* forget pointer to released white stones */
    for(x = mainview->select_left; x <= mainview->select_right; x++) { /* For all locations */
      for(y = mainview->select_top; y <= mainview->select_bottom; y++) { /* at the selected area */
        at = XY2Location(x, y); /* get the point */
        switch(GetStoneAt(at)) { /* and stone at the point */
          case STONE_TRANSPARENT_BLACK: /* if it is black */
            mainview->sgfAB = AddPoint2List(at, mainview->sgfAB); /* add it to the list of black setup stones */
            break;
          case STONE_TRANSPARENT_WHITE: /* if it is white */
            mainview->sgfAW = AddPoint2List(at, mainview->sgfAW); /* add it to the list of white setup stones */
            break;
		  default:
		    break;
        }
      }
    }
    mainview->sgfPL = OPPONENT(mainview->currentNode->stone); /* Set player in turn */
    node = mainview->currentNode->child; /* Get pointer to the subtree after current position */
    mainview->currentNode->child = NULL; /* Remove that subtree out of the game tree as child */
    if(node != NULL) {
      sister = node;
      while(sister->parent != NULL) {
        sister->parent = NULL; /* and remove each branch of that subtree out of the game tree as parent */
        sister = sister->sibling;
      }
    }
    listswap = mainview->sgfMA; mainview->sgfMA = mainview->currentNode->sgfMA; mainview->currentNode->sgfMA = listswap; /* Swap crosses between root and current position */
    listswap = mainview->sgfTR; mainview->sgfTR = mainview->currentNode->sgfTR; mainview->currentNode->sgfTR = listswap; /* Swap triangles between root and current position */
    listswap = mainview->sgfCR; mainview->sgfCR = mainview->currentNode->sgfCR; mainview->currentNode->sgfCR = listswap; /* Swap circles between root and current position */
    listswap = mainview->sgfSQ; mainview->sgfSQ = mainview->currentNode->sgfSQ; mainview->currentNode->sgfSQ = listswap; /* Swap squares between root and current position */
    labelswap = mainview->sgfLB; mainview->sgfLB = mainview->currentNode->sgfLB; mainview->currentNode->sgfLB = labelswap; /* Swap squares between root and current position */
    noteswap = mainview->sgfC; mainview->sgfC = mainview->currentNode->sgfC; mainview->currentNode->sgfC = noteswap; /* Swap notes between root and current position */
    /* this way releasing memory of the game tree will release memories we no longer want to have and keep what we wish to have */
    node_destroy( mainview->child ); /* Get rid of the current game */
    mainview->child = node; /* except the continuation after problem position */
    ReNumberMovesOfTree(mainview->child,1); /* Update movenumber of moves after setu position */
    mainview->currentNode = NULL; /* Set current move to point the setup position */
    mainview->panicNode = NULL; /* Sorry, no undo */
    mainview->compareNode = NULL; /* No home, just to be on the safe side */
    EnterNode(mainview);
  }
#ifndef PRODUCTION_VERSION  
  msg2log("\nCreate problem\n\nThis feature creates problem from game.\n\nStep 1:\nGo to the position where situation of the problem to be created occured. If you want to change player in turn, tenuki.\n\nStep 2:\nSelect the area where stones included to the problem are. You can selecte area by dragging stylus from empty intersection to the opposite corner of selected rectangle.\n\nStep 3:\nChose Create problem from Edit game menu. This will clip away everything outside of the selected area and make stones at the selected area to be a setup position. Continuations after current position will remain.\n\nStep 4:\nDrag the problem where you would want it to be or click on the goban to accept current position. Continuations if any, will not be moved. So you can either have continuations or option to reposition the problem.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_propose ( GtkAction * action, gpointer data )
{
  Node * node = NULL;
  Node * bestLineOfPlay = NULL;
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  if(mainview->currentNode == NULL) {
    node = mainview->child;
  }
  else {
    node = mainview->currentNode->child;
  }
  if(node == NULL) {
    return;
  }
  bestLineOfPlay = SelectBestLineOfPlay(node,0);
#ifndef PRODUCTION_VERSION  
  msg2log("\nFind analysis mainline\n\nFind analysis mainline feature tries to figure out what is the best line of play for both players. Before using this feature you should be at the position from which onwards you want to know the continuation. This feature makes a lot of assumptations of the game tree structure and thus result is not reliable. To help this feature work, you should always make as last move of any branch move as the player who has better position. This is intuitively true with exception of some sente sequences that require extra play to show how the sente should be used.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
  if(bestLineOfPlay != NULL) {
    if(bestLineOfPlay->stone == STONE_WHITE) {
      UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Black needs improvement on selected line of analysis");
    }
    if(bestLineOfPlay->stone == STONE_BLACK) {
      UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "White needs improvement on selected line of analysis");
    }
  }
}

/* Mark position for comparison */
void callback_compare ( GtkAction * action, gpointer data )
{
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  if(mainview->compareNode == NULL) {
    mainview->compareNode = mainview->currentNode;
    if(mainview->CurrentMouseMode != mainview->diagram_item) {
      UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Current position marked. Mark another position for comparison.");
    }
#ifndef PRODUCTION_VERSION  
    msg2log("\nMark position\n\nMark position creates single mark in the game at the present position. This marking can be used for several ways.\n- Home position of analysis\n- Start position of numbering\n- Setting how many moves to number\n- Selecting position for paired comparison\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
    DrawAll(mainview);
    return;
  }
  if(mainview->compareNode == mainview->currentNode) {
    if(mainview->CurrentMouseMode != mainview->diagram_item) {
      UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Current position is already marked.");
    }
    return;
  }
  mainview->panicNode = mainview->compareNode;
  mainview->compareNode = mainview->currentNode;
  if(mainview->CurrentMouseMode != mainview->diagram_item) {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Current position marked. Use esc-key to compare previously marked position.");
  }
  DrawAll(mainview);
}

/* Mark variations and 5 next moves of the mainline */
void callback_hint(GtkAction * action, gpointer data)
{
  gint i = 0;
  Node * node = NULL;
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

#ifndef PRODUCTION_VERSION  
  msg2log("\nGive hint\n\nGive hint takes about 5 plays from the game analysis after current position and marks them on the goban. How those 5 plays are selected depends on the selected mode, availability of branches at the present position and possible ko-fight in the near future.\n\nIn training mode give hint show all continuations of the present position and if that does not give 5 plays, then adds next not yet chosen plays from the mainline of the game.\n\nIn play mode give hint searches from the mainline of analysis next 5 plays that are not analysed alternatives in the current position.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  TakeHint();
  if(mainview->currentNode == NULL) {
    node = mainview->child;
  }
  else {
    node = mainview->currentNode->child;
  }
  if(node == NULL) {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "No analysis to get hints from.");
    return;
  }
  i=0;
  while(node != NULL) { /* For each continuation of the present position */
    SetStoneAt(node->at,STONE_MARKED_NONE); /* indicate that markings of mainline does not toggle them */
    if(mainview->CurrentMouseMode == mainview->training_item) { /* in training mode they are wanted to be included, in analysis not */
      MarkSelected(mainview, Location2X(node->at), Location2Y(node->at));
      i++; /* count home many has been marked allready so that training mode does not get too many hints */
    }
    if(node->nodeHeight >= node->sibling->nodeHeight) {
      break;
    }
    node = node->sibling;
  }
  node = node->sibling;
  while(node != NULL) { /* Mark next few moves of the mainline */
    if(GetStoneAt(node->at) == STONE_NONE) {
      SetStoneAt(node->at,STONE_MARKED_NONE);
      MarkSelected(mainview, Location2X(node->at), Location2Y(node->at));
      i++; /* No more than five hints. */
    }
    node = node->child;
    if(i >= 5) {
      break;
    }
  }
  /* Below markings are cleaned. If this is not done, clicking on markings jumps to root */
  if(mainview->currentNode == NULL) {
    node = mainview->child;
  }
  else {
    node = mainview->currentNode->child;
  }
  if(node == NULL) {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Please report bug: Mark up mess up current node.");
    return;
  }
  while(node != NULL) {
    SetStoneAt(node->at,STONE_NONE);
    if(node->nodeHeight >= node->sibling->nodeHeight) {
      break;
    }
    node = node->sibling;
  }
  node = node->sibling;
  while(node != NULL) {
    if(GetStoneAt(node->at) == STONE_MARKED_NONE) {
      SetStoneAt(node->at,STONE_NONE);
    }
    node = node->child;
  }
}

void callback_clear_territory(GtkAction * action, gpointer data)
{
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  if(mainview->currentNode != NULL) {
    DestroyPointlist(mainview->currentNode->sgfTW);
    DestroyPointlist(mainview->currentNode->sgfTB);
    mainview->currentNode->sgfTW = NULL;
    mainview->currentNode->sgfTB = NULL;
  }
#ifndef PRODUCTION_VERSION  
  msg2log("\nClear territory markings\n\nClear territory markings toolbar icon clear all territory markings of the present position. The estimated score is not cleared and will be saved in the sgf file.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

/* Destroy all siblings and siblings of ancestors and their children leaving only nodes that are 
   either directly ascending or directly descending from the node */ 
void callback_prune(GtkAction * action, gpointer data)
{
  Node * node = NULL;
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  node = mainview->currentNode;
  while(node != NULL) {
    while(node->sibling != node) {
      node_destroy(node->sibling);
    }
    node = node->parent;
  }
#ifndef PRODUCTION_VERSION  
  msg2log("\nPrune variations\n\nPrune variations feature delete all siblings of all ancestor positions. This feature is ment for fast cleaning of game file used for analysing ongoing game. Go to the position that is current position in the ongoing game and by using this feature you can remove all analysis that are not relevant any longer leaving all relevant analysis still in the file.\n\nFeature cannot detect local analysis, so analysis that is related to other part of the goban will be removed even though last moves have not changed that local situation in anyway.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void SetStyle(MainView * mainview)
{
  if((mainview->sgfST / 2) == 0) {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (mainview->boardmarkup_item), TRUE);
  }
  else {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (mainview->boardmarkup_item), FALSE);
  }
  if((mainview->sgfST % 2) == 0) {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (mainview->childview_item), FALSE);
  }
  else {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM (mainview->childview_item), TRUE);
  }
}

void ResetGame(gint newSize, MainView * mainview)
{
  PangoFontDescription * pangoFontDescription = NULL;

  ExitNode(mainview);
  mainview->isGoProblem = FALSE;
  if(mainview->isBigTree == TRUE) {
    mainview->gridSize = mainview->gridRestore;
  }
  mainview->isBigTree = FALSE;
  mainview->currentNode = NULL;
  mainview->compareNode = NULL;
  kill_family(mainview->child);
  mainview->child = NULL;
  mainview->sgfSZ = newSize;
  DestroyPointlist(mainview->sgfAB);
  DestroyPointlist(mainview->sgfAW);
  DestroyPointlist(mainview->sgfTR);
  DestroyPointlist(mainview->sgfMA);
  DestroyPointlist(mainview->sgfCR);
  DestroyPointlist(mainview->sgfSQ);
  DestroyLabellist(mainview->sgfLB);
  mainview->sgfAB = NULL;
  mainview->sgfAW = NULL;
  mainview->sgfTR = NULL;
  mainview->sgfMA = NULL;
  mainview->sgfCR = NULL;
  mainview->sgfSQ = NULL;
  mainview->sgfLB = NULL;
  mainview->sgfHA = 0;
  mainview->sgfKMx2 = 0;
  mainview->sgfPL = STONE_NONE;
  DestroyNote(mainview->sgfC);
  mainview->sgfC = NULL;
  DestroyNote(mainview->sgfGN);
  mainview->sgfGN = NULL;
  DestroyNote(mainview->sgfBR);
  mainview->sgfBR = NULL;
  DestroyNote(mainview->sgfEV);
  mainview->sgfEV = NULL;
  DestroyNote(mainview->sgfPB);
  mainview->sgfPB = NULL;
  DestroyNote(mainview->sgfPC);
  mainview->sgfPC = NULL;
  DestroyNote(mainview->sgfPW);
  mainview->sgfPW = NULL;
  DestroyNote(mainview->sgfRE);
  mainview->sgfRE = NULL;
  DestroyNote(mainview->sgfWR);
  mainview->sgfWR = NULL;
  DestroyNote(mainview->sgfDI);
  mainview->sgfDI = NULL;
  DestroyNote(mainview->sgfGE);
  mainview->sgfGE = NULL;
  InitGoban(mainview->sgfSZ);
  pangoFontDescription = pango_font_description_from_string ("Courier,Narrow 8");
  gtk_widget_modify_font (mainview->goban_view, pangoFontDescription);
  if(pangoFontDescription != NULL) {
    pango_font_description_free(pangoFontDescription); /* In case of crash remove this line */
  }
  SetPlayerInTurn(mainview);
  Panicable(mainview);
  EnterNode(mainview);
}

/* open */
void callback_file_open(GtkAction * action, gpointer data)
{
    gint answer;
    gchar* filename = NULL;
    MainView *mainview = NULL;
    mainview = ( MainView * ) data;
    g_assert(mainview != NULL && mainview->data != NULL );

    /* show training statistics if at training mode and reset training statistics in any way */
    ResetTrainingStatistics(TRUE, mainview->isGoProblem);
    /* save changes note if file is edited */
    if( mainview->file_edited ) {
        answer = interface_save_changes_note( mainview );
        if( answer == CONFRESP_YES ) {
            /* check is we had a new file */
            if( mainview->file_name == NULL ) {
                mainview->file_name = interface_file_chooser ( mainview, GTK_FILE_CHOOSER_ACTION_SAVE );
            }
            write_buffer_to_file ( mainview );
        }
    }

    /* open new file */
    filename = interface_file_chooser ( mainview, GTK_FILE_CHOOSER_ACTION_OPEN );

    /* if we got a file name from chooser -> open file */
    if( filename != NULL ) {
        mainview->file_name = filename;
        read_file_to_buffer ( mainview );
        mainview->file_edited = FALSE;
        if((mainview->CurrentMouseMode == mainview->training_item) || 
           (mainview->CurrentMouseMode == mainview->search_item) ||
           ((mainview->fullscreen) && (mainview->CurrentMouseMode == mainview->play_item))) {
          gtk_widget_show(GTK_WIDGET(mainview->nextfile_tb));
        }
        set_working_directory ( mainview );
        read_directory_to_list ( mainview , TRUE );
    }
    Panicable(mainview);
}

void callback_next_file ( GtkAction * action, gpointer data )
{
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  ResetTrainingStatistics(TRUE, mainview->isGoProblem);
  /* read_directory_to_list ( mainview , TRUE ); */
  if(mainview->currentPlaylist == NULL) {
    return;
  }
  if(mainview->currentPlaylist->size <= 0) {
#ifndef PRODUCTION_VERSION  
    msg2logGint("Invalid playlist size (%d)", LOGLEVEL_INFO, LOGSCOPE_SGF, mainview->currentPlaylist->size);
#endif
    return;
  }
  if(mainview->tryLoad == 0) {
    mainview->currentPlaylist->current = (mainview->currentPlaylist->current + 1) % mainview->currentPlaylist->size;
  }
  else {
    mainview->tryLoad++;
  }
  read_file_from_playlist ( mainview );
#ifndef PRODUCTION_VERSION  
  msg2log("\nNext file\n\nOpen next file in current playlist. You can select current playlist from menu playlists. New playlist is created every time you open file from or save file to directory of which playlist has not been made of. Playlist is alphabetically sorted looped list of all sgf-files in the directory at time of playlist creation. File opened or saved when playlist is cretaed will become the current file of that playlist.\n\nExample 1: Searching file\n\nOpen first file in the directory you think the file you want could be. If it is not the file you wanted go to outcome mode, then click the next file icon until you see on the goban the correct file.\n\nExample 2: Solving go-problems\n\nOpen first problem you havn't solved at the directory of go-problems. Once you have solved the problem click next file icon to get next problem.", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

/* save */
void callback_file_save(GtkAction * action, gpointer data)
{
    gchar* filename = NULL;
    MainView *mainview = NULL;
    mainview = ( MainView * ) data;
    g_assert(mainview != NULL && mainview->data != NULL );

    /* check is we had a new file */
    if( mainview->file_name != NULL ) {
        write_buffer_to_file ( mainview );
    } else { 
        filename = interface_file_chooser ( mainview, GTK_FILE_CHOOSER_ACTION_SAVE );
        /* if we got a file name from chooser -> save file */
        if( filename != NULL ) {
            mainview->file_name = filename;
            write_buffer_to_file ( mainview );
            mainview->file_edited = FALSE;
            set_working_directory ( mainview );
            read_directory_to_list ( mainview , TRUE );
        }
    }
}

/* save as... */
void callback_file_saveas(GtkAction * action, gpointer data)
{
    gchar* filename = NULL;
    MainView *mainview = NULL;
    mainview = ( MainView * ) data;
    g_assert(mainview != NULL && mainview->data != NULL );

    filename = interface_file_chooser ( mainview, GTK_FILE_CHOOSER_ACTION_SAVE );

    /* if we got a file name from chooser -> save file */
    if( filename != NULL ) {
        mainview->file_name = filename;
        write_buffer_to_file ( mainview );
        mainview->file_edited = FALSE;
        set_working_directory ( mainview );
        read_directory_to_list ( mainview , TRUE );
    }
}

/* fullscreen */
void callback_fullscreen( GtkAction * action, gpointer data )
{
    /* connect pointer to our MainView struct */
    MainView *mainview = NULL;
    mainview = ( MainView * ) data;
    g_assert(mainview != NULL && mainview->data != NULL );

    /* toggle fullscreen on<->off */
    mainview->fullscreen = !mainview->fullscreen;

    if (mainview->fullscreen) {
        gtk_window_fullscreen(GTK_WINDOW(mainview->data->window));
    } else {
        gtk_window_unfullscreen(GTK_WINDOW(mainview->data->window));
    }
}

/* delete */
void callback_delete( GtkAction * action, gpointer data )
{
  Node * newCurrent = NULL;
  gboolean root = FALSE;
  Node * replacer = NULL;
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  /* Delete current move */
  if(mainview->child == NULL && mainview->sgfAB == NULL) {
    return; /* Nothing to delete */
  }
  mainview->file_edited = TRUE;
  if(mainview->currentNode == NULL) { /* If at root */
    ResetGame(mainview->sgfSZ, mainview);
  }
  else {
    ExitNode(mainview);
    if(mainview->child == mainview->currentNode) {
      root = TRUE;
    }
    newCurrent = mainview->currentNode->parent;
    replacer = node_destroy(mainview->currentNode);
    mainview->currentNode = newCurrent;
    if(root == TRUE) {
      mainview->child = replacer;
    }
    EnterNode(mainview);
  }
#ifndef PRODUCTION_VERSION  
  if(mainview->child == NULL) {
    msg2log("mainview->child == NULL", LOGLEVEL_WARNING, LOGSCOPE_GAMETREE);
  }
  if(mainview->currentNode == NULL) {
    msg2log("mainview->currentNode == NULL", LOGLEVEL_WARNING, LOGSCOPE_GAMETREE);
  }
#endif
  DrawAll(mainview);
  Panicable(mainview);
}

void callback_shownext( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->shownext = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mainview->shownext_item));
#ifndef PRODUCTION_VERSION  
  if(mainview->shownext == TRUE) {
    msg2log("shownext set to TRUE", LOGLEVEL_INFO, LOGSCOPE_FORCED);
  }
  else {
    msg2log("shownext set to FALSE", LOGLEVEL_INFO, LOGSCOPE_FORCED);
  }
#endif
#ifndef PRODUCTION_VERSION  
  msg2log("\nShow next move\n\nShow next move option affect weather or not place of next move is shown at the goban.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_boardmarkup( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  if(gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mainview->boardmarkup_item)) == TRUE) {
    mainview->sgfST &= 1; 
  }
  else {
    mainview->sgfST |= 2; 
  }
#ifndef PRODUCTION_VERSION  
  msg2log("\nBoard markup\n\nBoard mark up option indicates weather or not letters are shown on the goban to indicate alternative next moves in case there are several alternatives.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_childview( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  if(gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mainview->childview_item)) == TRUE) {
    mainview->sgfST |= 1; 
  }
  else {
    mainview->sgfST &= 2; 
  }
#ifndef PRODUCTION_VERSION  
  msg2log("\nSibling view\n\nSibling view indicate weather alternatives shown by board mark up option are shown at the position where the alternatives are available or at the position for which alternatives are available.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_diagram( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->diagramview = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mainview->diagramview_item));
  if(mainview->compareNode == NULL) {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Mark the position from which numbering starts.");
  }
#ifndef PRODUCTION_VERSION  
  msg2log("\nStone numbering\n\nStone numbering show numbers on non-captured stones. To see stone numbering also on captured stones, use diagram mode instead.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_resetnumbering( GtkWidget * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  if(action == mainview->resetnumbering_item) {
    mainview->resetNumbering = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mainview->resetnumbering_item));
  }
  if(action == GTK_WIDGET(mainview->resetNumbering_tb)) {
    mainview->resetNumbering = gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(mainview->resetNumbering_tb));
  }
  /*
  if(mainview->resetNumbering == TRUE) {
    mainview->resetNumbering = FALSE;
  }
  else {
    mainview->resetNumbering = FALSE;
  }
  gtk_check_menu_item_set_active(mainview->resetnumbering_item, mainview->resetNumbering);
  gtk_toggle_tool_button_set_active(mainview->resetNumbering_tb, mainview->resetNumbering);
  */
  /* mainview->resetNumbering = gtk_check_menu_item_get_active(mainview->resetnumbering_item); */
  /* mainview->resetNumbering = gtk_toggle_tool_button_get_active(mainview->resetNumbering_tb); */
  /*
  if(mainview->diagramview == FALSE) {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Set diagram view to see numbering.");
  }
  */
#ifndef PRODUCTION_VERSION  
  msg2log("\nReset numbering\n\nReset numbering set number 1 for the first stone that gets number. If option is not in use, real move numbers are used instead.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_number_n_plays_tb(GtkToggleToolButton * tb, gpointer data )
{
  number_n_plays( data, FALSE );
}

void callback_number_n_plays_menu(GtkCheckMenuItem * menu, gpointer data )
{
  number_n_plays( data, TRUE );
}

void number_n_plays( gpointer data , gboolean setTb)
{
  Node * node = NULL;
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  if(mainview->numberLastPlays == MAX_SHOWN_NUMBERS) {
    node = findStartOfNumbering( mainview );
    if(node != NULL && mainview->currentNode != NULL) {
      mainview->numberLastPlays = mainview->currentNode->nodeTurnNumber - node->nodeTurnNumber + 1;
      mainview->compareNode = NULL;
      if(setTb == TRUE) {
        gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mainview->numberNPlays_item), TRUE);
      }
      if(mainview->CurrentMouseMode == mainview->diagram_item) {
        gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(mainview->numberNlast_tb), TRUE);
      }
    }
    else {
      /* gtk_check_menu_item_set_active(mainview->numberNPlays_item, FALSE); */
      UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Before setting this option wanted amount of numbers must be visible. Try other options first.");
      if(setTb == TRUE) {
        gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mainview->numberNPlays_item), FALSE);
      }
      if(mainview->CurrentMouseMode == mainview->diagram_item) {
        gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(mainview->numberNlast_tb), FALSE);
      }
    }
  }
  else {
    mainview->numberLastPlays = MAX_SHOWN_NUMBERS;
    if(setTb == TRUE) {
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mainview->numberNPlays_item), FALSE);
    }
    if(mainview->CurrentMouseMode == mainview->diagram_item) {
      gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(mainview->numberNlast_tb), FALSE);
    }
  }
  /* if(gtk_check_menu_item_get_active(mainview->numberNPlays_item) == TRUE) { */
  /* if(gtk_toggle_tool_button_get_active(mainview->numberNlast_tb) == TRUE) {
    node = findStartOfNumbering( mainview );
    if(node != NULL && mainview->currentNode != NULL) {
      mainview->numberLastPlays = mainview->currentNode->nodeTurnNumber - node->nodeTurnNumber + 1;
      mainview->compareNode = NULL;
    }
    else {
      / * gtk_check_menu_item_set_active(mainview->numberNPlays_item, FALSE); * /
      gtk_toggle_tool_button_set_active(mainview->numberNlast_tb,FALSE);
      UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Before setting this option wanted amount of numbers must be visible. Try other options first.");
    }
  }
  else {
    mainview->numberLastPlays = MAX_SHOWN_NUMBERS;
  }
  / *
  if(mainview->diagramview == FALSE) {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Set diagram view to see numbering.");
  }
  */
#ifndef PRODUCTION_VERSION  
  msg2log("\nNumber n last plays\n\nNumber n last plays fixes the current amount of shown numbers and shows that many latest moves while the option is on.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_rndOrientation( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->rndOrientation = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mainview->rndOrientation_item));
  if(mainview->rndOrientation == TRUE) {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "When opened file is a go-problem the orientation of the goban will be chosen randomly.");
  }
}

void callback_problem2step( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->problem2step = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mainview->problem2step_item));
  if(mainview->problem2step == TRUE) {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "When opened file is a go-problem the 2-step option will be set.");
  }
}

void callback_problemTraining( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->problemTraining = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mainview->problemTraining_item));
  if(mainview->problemTraining == TRUE) {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "When opened file is a go-problem the training mode will be set.");
  }
}

void callback_autoRefute( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->analysisAutoRefute = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mainview->autoRefute_item));
}

void callback_mousemode ( GtkRadioMenuItem * item, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  if(GTK_WIDGET(item) == mainview->CurrentMouseMode) { /* Exit mode */
    return; /* Ignore message about deselecting old radio selection */
  }
  if(mainview->CurrentMouseMode == mainview->play_item) {
    gtk_widget_hide(GTK_WIDGET(mainview->add_tb));
    gtk_widget_hide(GTK_WIDGET(mainview->delete_tb));
    gtk_widget_hide(GTK_WIDGET(mainview->nextfile_tb));
  }
  if(mainview->CurrentMouseMode == mainview->training_item) {
    gtk_widget_hide(GTK_WIDGET(mainview->givehint_tb));
    gtk_widget_hide(GTK_WIDGET(mainview->step_2_option_tb));
    gtk_widget_hide(GTK_WIDGET(mainview->nextfile_tb));
    gtk_widget_show(GTK_WIDGET(mainview->save_tb));
    if(mainview->compareNode != NULL) {
      gtk_widget_show(GTK_WIDGET(mainview->gotofirst_tb));
    }
  }
  if(mainview->CurrentMouseMode == mainview->territory_item) {
    gtk_widget_hide(GTK_WIDGET(mainview->clear_tb));
    gtk_widget_hide(GTK_WIDGET(mainview->mark_territory_tb));
  }
  if(mainview->CurrentMouseMode == mainview->diagram_item) {
    gtk_widget_hide(GTK_WIDGET(mainview->add_tb));
    gtk_widget_hide(GTK_WIDGET(mainview->set_mark_tb));
    gtk_widget_hide(GTK_WIDGET(mainview->transparent_tb));
    gtk_widget_hide(GTK_WIDGET(mainview->resetNumbering_tb));
    gtk_widget_hide(GTK_WIDGET(mainview->numberNlast_tb));
    gtk_widget_show(GTK_WIDGET(mainview->gotopbra_tb));
    gtk_widget_show(GTK_WIDGET(mainview->gotonbra_tb));
    gtk_widget_show(GTK_WIDGET(mainview->gotofirst_tb));
    gtk_widget_show(GTK_WIDGET(mainview->gotovari_tb));
    gtk_widget_show(GTK_WIDGET(mainview->new_tb));
    gtk_widget_show(GTK_WIDGET(mainview->open_tb));
    gtk_widget_show(GTK_WIDGET(mainview->gotomain_tb));
    gtk_widget_set_size_request (mainview->notebook, TABS_WIDTH_NORMAL, -1);
  }
  if(mainview->CurrentMouseMode == mainview->search_item) {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Normal key-mode activated");
    gtk_widget_hide(GTK_WIDGET(mainview->nextfile_tb));
    gtk_widget_hide(GTK_WIDGET(mainview->findmainline_tb));
  }
  ExitNode(mainview);
  mainview->CurrentMouseMode = GTK_WIDGET(item);
  EnterNode(mainview);
  if(mainview->CurrentMouseMode == mainview->play_item) {
    if(mainview->manualOnly == FALSE) {
      gtk_notebook_set_current_page(GTK_NOTEBOOK(mainview->notebook),ANALYSIS_PAGE);
    }
    gtk_widget_show(GTK_WIDGET(mainview->add_tb));
    gtk_widget_show(GTK_WIDGET(mainview->delete_tb));
    if((mainview->file_name != NULL) && (mainview->fullscreen)) {
      gtk_widget_show(GTK_WIDGET(mainview->nextfile_tb));
    }
#ifndef PRODUCTION_VERSION  
    msg2log("\nPlay mode\n\nPlay mode is ment for game analysis and game recording.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  }
  if(mainview->CurrentMouseMode == mainview->training_item) {
    if(mainview->manualOnly == FALSE) {
      gtk_notebook_set_current_page(GTK_NOTEBOOK(mainview->notebook),LOG_PAGE);
    }
    gtk_widget_hide(GTK_WIDGET(mainview->save_tb));
    if(mainview->compareNode != NULL) {
      gtk_widget_hide(GTK_WIDGET(mainview->gotofirst_tb));
    }
    gtk_widget_show(GTK_WIDGET(mainview->givehint_tb));
    gtk_widget_show(GTK_WIDGET(mainview->step_2_option_tb));
    if(mainview->file_name != NULL) {
      gtk_widget_show(GTK_WIDGET(mainview->nextfile_tb));
    }
    ResetTrainingStatistics(FALSE, mainview->isGoProblem);
#ifndef PRODUCTION_VERSION  
    msg2log("\nTraining mode\n\nTraining mode is ment for self study of go. Following ways to study has been taken into consideration:\n- Replay pro games\n- Memorize own games\n- Solve tsume go problems\n- Study joseki\n- Study fuseki\n\nUse help icon at the toolbar to get hint if you cannot figure out the next move.\n\nUse two step option 2 at the toolbar to let goban770 play moves of one color.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  }
  if(mainview->CurrentMouseMode == mainview->territory_item) {
    gtk_widget_show(GTK_WIDGET(mainview->clear_tb));
    gtk_widget_show(GTK_WIDGET(mainview->mark_territory_tb));
    if(mainview->currentNode != NULL) {
      if(mainview->currentNode->child == NULL) {
        UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "To inherit dead groups estimating territory of common ancestor positions before terminal positions");
      }
    }
#ifndef PRODUCTION_VERSION  
    msg2log("\nTerritory counting mode\n\nTerritory counting mode can be used for final result counting or for midgame estimate of the score. Estimate bases on the assumptation that nearest alive stone color gets the territory.\n- Start from common ancestor of analysed positions\n- Correct positions where groups die\n- Use apply to inherit dead groups\n- Use clear to remove territory markings\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  }
  if(mainview->CurrentMouseMode == mainview->diagram_item) {
    if(mainview->manualOnly == FALSE) {
      gtk_notebook_set_current_page(GTK_NOTEBOOK(mainview->notebook),ANALYSIS_PAGE);
    }
    gtk_widget_show(GTK_WIDGET(mainview->add_tb));
    gtk_widget_show(GTK_WIDGET(mainview->set_mark_tb));
    gtk_widget_show(GTK_WIDGET(mainview->transparent_tb)); 
    gtk_widget_show(GTK_WIDGET(mainview->resetNumbering_tb));
    gtk_widget_show(GTK_WIDGET(mainview->numberNlast_tb));
    gtk_widget_hide(GTK_WIDGET(mainview->gotopbra_tb));
    gtk_widget_hide(GTK_WIDGET(mainview->gotonbra_tb));
    gtk_widget_hide(GTK_WIDGET(mainview->gotofirst_tb));
    gtk_widget_hide(GTK_WIDGET(mainview->gotovari_tb));
    gtk_widget_hide(GTK_WIDGET(mainview->new_tb));
    gtk_widget_hide(GTK_WIDGET(mainview->open_tb));
    gtk_widget_hide(GTK_WIDGET(mainview->gotomain_tb));
    gtk_widget_set_size_request (mainview->notebook, TABS_WIDTH_DIAGRAM, -1);
#ifndef PRODUCTION_VERSION  
    msg2log("\nDiagram mode\n\nDiagram mode is meant for recording game from book diagram. You can either record the game as move by move in the order in which it is played or you could use fill goban feature before diagram view to get stones to drag at correct places.\n- 1. Sets current position to have first number\n- 1.=1 Sets first number to be 1\n- ..n fixes amount of last numbers shown\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  }
  if(mainview->CurrentMouseMode == mainview->search_item) {
    if(mainview->manualOnly == FALSE) {
      gtk_notebook_set_current_page(GTK_NOTEBOOK(mainview->notebook),ANALYSIS_PAGE);
    }
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Branch jump key-mode activated");
    if(mainview->file_name != NULL) {
      gtk_widget_show(GTK_WIDGET(mainview->nextfile_tb));
    }
    gtk_widget_show(GTK_WIDGET(mainview->findmainline_tb));
#ifndef PRODUCTION_VERSION  
    msg2log("\nOutcome mode\n\nOutcome mode is ment for comparison of possible outcome as result of the move. At each node marked on the goban chose the outcome best for the player in turn. If you work from the end of the variations towards point of decision, you get to see correct outcomes of different choises at the point of decision by using up and down arrows.\n\nIf you have autorefute option on, it would take care of selecting correct branches while you do the analysis. At the end of the branch you could use territory counting mode to get estimate. The estimates calculated are shown at outcome mode.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  }
  DrawAll(mainview);
}

void callback_mterri ( GtkAction * action, gpointer data )
{
  Node * node = NULL;
  Node * ancestor = NULL;
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  if(mainview->currentNode != NULL) {
    node = mainview->currentNode->parent;
    while(node != NULL) { /* Find ancestor with defined territories */
      if((node->sgfTB != NULL) || (node->sgfTW != NULL)) {
        break;
      }
      node = node->parent;
    }
    DestroyPointlist(mainview->currentNode->sgfTB);
    mainview->currentNode->sgfTB = NULL;
    DestroyPointlist(mainview->currentNode->sgfTW);
    mainview->currentNode->sgfTW = NULL;
    if(node != NULL) {
      /* Mark as dead stones that are on enemy territory at ancestor position */
      MarkDeadListedGroupsOfColor(mainview, node->sgfTW, STONE_BLACK);
      MarkDeadListedGroupsOfColor(mainview, node->sgfTB, STONE_WHITE);
      /* Mark as alive stones that have been played after the ancetor position */
      ancestor = node;
      node = mainview->currentNode;
      while((node != ancestor) && (node != NULL)) {
        if(node->stone == GetStoneAt(node->at)) {
          MarkGroupAlive(mainview, node->at);
        }
        node = node->parent;
      }
    }
    SetTerritoryAtGroups(mainview);
    PessimisticExpand(mainview->currentNode);
    /*
    if(mainview->currentNode->stone == STONE_BLACK) {
      ExpandTerritories(mainview->currentNode->sgfTW, mainview->currentNode->sgfTB);
    }
    else {
      ExpandTerritories(mainview->currentNode->sgfTB, mainview->currentNode->sgfTW);
    } */
    ClearTerritoryAtGroups(mainview);
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Mark dead groups before correcting territories.");
  }
  DrawAll(mainview);
}

void callback_step_2_option ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->step_2 = gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(mainview->step_2_option_tb));
  if(mainview->step_2 == TRUE) {
    if(mainview->currentNode == NULL) {
      if(mainview->child == NULL) {
        return;
      }
      Panicable(mainview);
      ExitNode(mainview);
      mainview->currentNode = RandomSibling(mainview->child);
      EnterNode(mainview);
    }
    else {
      if(mainview->currentNode->child != NULL) {
        Panicable(mainview);
        ExitNode(mainview);
        mainview->currentNode = RandomSibling(mainview->currentNode->child);
        EnterNode(mainview);
      }
    }
#ifndef PRODUCTION_VERSION  
    msg2log("\nTwo step option\n\nWhen two step option is activated one of the available continuations is randomly selected and that move is made. When two step option is on, available continuation is randomly selected after each made move.\n\nIn practice two step option let you play against fuseki library, joseki library or as one of the players in game. This can also be used while solving go-problems. Before activating this option, make sure that it is wanted opponent color move. When move is selected randomly, possible continations are weightened on amount of available leafs of game tree\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
    DrawAll(mainview);
  }
}

void callback_transparent ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->transparentCaptured = gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(mainview->transparent_tb));
#ifndef PRODUCTION_VERSION  
    msg2log("\nTransparent captured option\n\nWhen transparent captured option is on, only outline of captured stones is shown. The purpose is that you would easier locate ko and under the stone plays referred by the book from which diagram is recorded.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

/* help */
void callback_help( GtkAction * action, gpointer data )
{
    osso_return_t retval;

    /* connect pointer to our MainView struct */
    MainView *mainview = NULL;
    mainview = ( MainView * ) data;
    g_assert(mainview != NULL && mainview->data != NULL );

    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Help is not available");
}

/* buffer edited */
void callback_buffer_modified ( GtkAction * action, gpointer data )
{
    /* connect pointer to our MainView struct */
    MainView *mainview = NULL;
    mainview = ( MainView * ) data;
    g_assert(mainview != NULL && mainview->data != NULL );

    /* change state */
    mainview->file_edited = TRUE;
}

void callback_note_modified ( GtkAction * action, gpointer data )
{
    /* connect pointer to our MainView struct */
    MainView *mainview = NULL;
    mainview = ( MainView * ) data;
    g_assert(mainview != NULL && mainview->data != NULL );

return; /* function not in use yet */

    if(mainview->noteNode == NULL) {
      mainview->sgfC = GetNewNoteFromBuffer(mainview->sgfC, mainview->noteBuffer);
    }
    else {
      mainview->noteNode->sgfC = GetNewNoteFromBuffer(mainview->noteNode->sgfC, mainview->noteBuffer);
    }
}

/* Privates */

static int IsSgfFile(struct dirent * candidate)
{
  gchar * here;
  here = candidate->d_name;
  while(here[0] != '\0') {
    here++;
  }
  here-=4;
  if((here[0] != '.') || (here[1] != 's') || (here[2] != 'g') || (here[3] != 'f')) {
    return 0; /* Reject the file */
  }
  return 1; /* Accept the file */
}

static int MySort(struct dirent ** a, struct dirent ** b )
{
  gint position = 0;
  while((*a)->d_name[position] == (*b)->d_name[position]) {
    position++;
  }
  return (*a)->d_name[position] - (*b)->d_name[position];
}

void set_working_directory ( MainView* mainview )
{
  gchar * workingDirectory = NULL;
  gchar * here = NULL;
  gint dirLength = 0;
  gint i = 0;
  int success = 0;
  if(mainview->file_name == NULL) {
    return;
  }
  /* Extract the directoryname from the file name */
  here = mainview->file_name;
  while(here[0] != '\0') {
    here++;
    dirLength++;
  }
  while((here[0] != '\\') && (here[0] != '/')) {
    here--;
    dirLength--;
  }
  dirLength++;
  workingDirectory = g_malloc(dirLength + 1);
  workingDirectory[dirLength] = 0;
  for(i=0; i<dirLength; i++) {
    workingDirectory[i] = mainview->file_name[i];
  }
  success = chdir (workingDirectory);
  g_free(workingDirectory);
  if(success != 0) {
#ifndef PRODUCTION_VERSION  
    msg2logGint("Changing working directory return code is %d", LOGLEVEL_ERROR, LOGSCOPE_FORCED, success);
#endif
  }
}

/* Open file from playlist */
void read_file_from_playlist(MainView * mainview)
{
  gint nameLength = 0;
  gint i = 0;
  gint from = 0;
  /* Old file name has to be released */
  g_free(mainview->file_name);
  /* Space of new file name need to be reserved */
  while(mainview->currentPlaylist->directory[nameLength] != '\0') {
    nameLength++; /* Measure directory length */
  }
  while(mainview->currentPlaylist->sgfplaylist[mainview->currentPlaylist->current]->d_name[i] != '\0') {
    i++; /* Measure name length */
  }
  nameLength += i;
  mainview->file_name = g_malloc(nameLength + 1);
  memset(mainview->file_name, 0, nameLength + 1);
  /* directory + filename has to be copied to that space */
  i = 0;
  while(mainview->currentPlaylist->directory[i] != '\0') {
    mainview->file_name[i] = mainview->currentPlaylist->directory[i];
    i++;
  }
  while(mainview->currentPlaylist->sgfplaylist[mainview->currentPlaylist->current]->d_name[from] != '\0') {
    mainview->file_name[i] = mainview->currentPlaylist->sgfplaylist[mainview->currentPlaylist->current]->d_name[from];
    i++;
    from++;
  }
  mainview->file_name[i] = '\0';
  /* the file has to be then opened */
  read_file_to_buffer ( mainview );
  PrintTrainingStatistics(mainview->isGoProblem);
#ifndef PRODUCTION_VERSION  
  if(mainview->file_name != NULL) {
    msg2log("Current file is:", LOGLEVEL_INFO, LOGSCOPE_SGF);
    msg2log(mainview->file_name, LOGLEVEL_INFO, LOGSCOPE_SGF);
  }
  if(mainview->currentPlaylist->directory != NULL) {
    msg2log("Current directory is:", LOGLEVEL_INFO, LOGSCOPE_SGF);
    msg2log(mainview->currentPlaylist->directory, LOGLEVEL_INFO, LOGSCOPE_SGF);
  }
  else {
    msg2log("Current directory is undefined", LOGLEVEL_INFO, LOGSCOPE_SGF);
  }
#endif
}

/* read directory content */
void read_directory_to_list ( MainView* mainview, gboolean readOnlyNewLists )
{
  int cnt;
  gchar * here = NULL;
  gint dirLength = 0;
  gint i = 0;
  gchar * name_only = NULL;
  PlayList * plist = NULL;
  PlayList * plist2 = NULL;

  if(mainview->file_name == NULL) { /* If there is no file open */
    return; /* we cannot determine directory of that file */
  }
  /* Extract the directoryname from the file name */
  here = mainview->file_name;
  while(here[0] != '\0') {
    here++;
    dirLength++;
  }
  while((here[0] != '\\') && (here[0] != '/')) {
    here--;
    dirLength--;
  }
  name_only = here;
  name_only++;
  dirLength++;
  /* Let's try to find existing playlist */
  plist = mainview->firstPlaylist;
  while(plist != NULL) {
    i=0;
    while((mainview->file_name[i] == plist->directory[i]) && (plist->directory[i] != '\0')) {
      i++;
    }
    if(((mainview->file_name[i-1] == '\\') || (mainview->file_name[i-1] == '/')) && (plist->directory[i] == '\0')) { /* If we found existing playlist */
      while((mainview->file_name[i] != '\\') && (mainview->file_name[i] != '/') && (mainview->file_name[i] != '\0')) { /* let's check if there are subdirectories left in the path */
        i++;
      }
      if(mainview->file_name[i] == '\0') { /* If no subdirectories in the path */
        if(readOnlyNewLists == TRUE) { /* and we want only new lists */
          mainview->currentPlaylist = plist;
          gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(plist->playlist_item),TRUE);
          return; /* we do nothing */
        }
        break; /* else we stop searching */
      }
    }
    plist = plist->next;
  }
  if(plist == NULL) { /* If such list does not exist yet */
    plist = g_malloc(sizeof(PlayList)); /* reserve such and clear the content */
    plist->next = NULL;
    plist->directory = NULL;
    plist->sgfplaylist = NULL;
    plist->size = 0;
    plist->current = 0;
    plist->playlist_item = NULL;
    plist->title = NULL;
    if(mainview->firstPlaylist == NULL) { /* If we had no playlists before */
      mainview->firstPlaylist = plist; /* make this one the first */
    }
    else { /* but if we had playlists before */
      plist2 = mainview->firstPlaylist;
      while(plist2->next != NULL) { /* find the last one */
        plist2 = plist2->next;
      }
      plist2->next = plist; /* and add the new list after it */
    }
  }
  mainview->currentPlaylist = plist;
  
  if(plist->directory != NULL) {
    g_free(plist->directory);
  }
  plist->directory = g_malloc(dirLength + 1);
  plist->directory[dirLength] = 0;
  for(i=0; i<dirLength; i++) {
    plist->directory[i] = mainview->file_name[i];
  }
  plist->title = plist->directory; /* Get title for end of the directory */
  while(plist->title[0] != '\0') { /* find the end */
    plist->title++;
  }
  plist->title -= 2; /* Skip last directory slash */
  while((plist->title[0] != '\\') && (plist->title[0] != '/') && (plist->title != plist->directory)) { /* find start of last subdirectory name */
    plist->title--;
  }
  if((plist->title[0] == '\\') || (plist->title[0] == '/')) { /* remove directory character from title */
    plist->title++;
  }
  /* Get directory information */
  plist->size = scandir(plist->directory, &(plist->sgfplaylist), IsSgfFile, MySort); /* alphasort is not recognized as defined */
  plist->current = 0;
  if(plist->size >= 0) {
    for(cnt = 0; cnt < plist->size; ++cnt) {
#ifndef PRODUCTION_VERSION  
      msg2log(plist->sgfplaylist[cnt]->d_name, LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
      /*
      i = 0;
      while((name_only[i] != '\0') && (name_only[i] == plist->sgfplaylist[cnt]->d_name[i])) {
        i++;
      }
      if(plist->sgfplaylist[cnt]->d_name[i] == '\0') { */
      if(strcmp(name_only,plist->sgfplaylist[cnt]->d_name) == 0) {
        plist->current = cnt;
#ifndef PRODUCTION_VERSION  
        msg2logGint("Found current file at index %d", LOGLEVEL_INFO, LOGSCOPE_SGF,cnt);
#endif
      }
    }
  }
  /* Create menu entry */
  if(plist->playlist_item == NULL) {
#ifndef PRODUCTION_VERSION  
        msg2log("Adding new playlist menu", LOGLEVEL_STATISTICS, LOGSCOPE_SGF);
        msg2log(plist->title, LOGLEVEL_STATISTICS, LOGSCOPE_SGF);
        msg2log("List title is above this line", LOGLEVEL_STATISTICS, LOGSCOPE_SGF);
#endif
    plist->playlist_item = gtk_radio_menu_item_new_with_label ( mainview->playlist_radio, plist->title );
    mainview->playlist_radio = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM (plist->playlist_item));
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (plist->playlist_item), TRUE);
    gtk_menu_append( mainview->playlists_menu, plist->playlist_item );
    gtk_widget_show( plist->playlist_item );
    g_signal_connect( G_OBJECT( plist->playlist_item ), "activate", 
		      G_CALLBACK ( callback_playlist), mainview );
  }
  else {
#ifndef PRODUCTION_VERSION  
        msg2log("No need to add new playlist menu", LOGLEVEL_STATISTICS, LOGSCOPE_SGF);
#endif
  }
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(plist->playlist_item),TRUE);
}

/* read file */
void read_file_to_buffer ( MainView* mainview )
{
  gint nodeCount = 0, paramCount = 0, branchCount = 0, charCount = 0;
    GnomeVFSResult vfs_result;
    GnomeVFSHandle *handle = NULL;
    GnomeVFSFileSize in_bytes;
    GnomeVFSFileInfo finfo;
    gchar *temp_buffer = NULL;
    g_assert( mainview != NULL );

#ifndef PRODUCTION_VERSION  
    msg2log(mainview->file_name, LOGLEVEL_INFO, LOGSCOPE_SGF);
#endif
    /* try to get file info */
    vfs_result = gnome_vfs_get_file_info(mainview->file_name, &finfo, GNOME_VFS_FILE_INFO_DEFAULT);
    if ( vfs_result != GNOME_VFS_OK ) {
        interface_error( MAEMOPAD_ERROR_OPEN_FAILED, mainview );
        return;
    }
    /* try to create handle to file */
    vfs_result = gnome_vfs_open(&handle, mainview->file_name, GNOME_VFS_OPEN_READ);
    if ( vfs_result != GNOME_VFS_OK ) {
        interface_error( MAEMOPAD_ERROR_OPEN_FAILED, mainview );
        return;
    }

    /* allocate memory for temp_buffer */
    temp_buffer = g_malloc(finfo.size + 1);
    memset(temp_buffer, 0, finfo.size + 1);
    
    /* read from file to buffer */
    gnome_vfs_read(handle, temp_buffer, finfo.size, &in_bytes);

    /* set text to screen */
    /* gtk_text_buffer_set_text( GTK_TEXT_BUFFER (mainview->debugBuffer), temp_buffer, -1); */
#ifndef PRODUCTION_VERSION  
    msg2log("==== SGF FILE IN START ====", LOGLEVEL_INFO, LOGSCOPE_SGF);
    msg2log(temp_buffer, LOGLEVEL_INFO, LOGSCOPE_SGF);
    msg2log("==== SGF FILE IN END ====", LOGLEVEL_INFO, LOGSCOPE_SGF);
    /* DrawAll(mainview); */
#endif
	
	 /* Check validity of the sgf-file */
    if(IsValidSgf(temp_buffer, &temp_buffer[finfo.size], &nodeCount, &paramCount, &branchCount, &charCount) == FALSE) {
#ifndef PRODUCTION_VERSION  
      msg2logGint("%d successfully checked nodes", LOGLEVEL_STATISTICS, LOGSCOPE_SGF, nodeCount);
      msg2logGint("%d successfully checked parameters", LOGLEVEL_STATISTICS, LOGSCOPE_SGF, paramCount);
      msg2logGint("%d successfully checked branches", LOGLEVEL_STATISTICS, LOGSCOPE_SGF, branchCount);
      msg2logGint("%d successfully checked characters", LOGLEVEL_STATISTICS, LOGSCOPE_SGF, charCount);
#endif
      g_free(temp_buffer);
      gnome_vfs_close(handle);
      return;
    }
    else {
#ifndef PRODUCTION_VERSION  
      msg2logGint("%d nodes in gametree of the file", LOGLEVEL_STATISTICS, LOGSCOPE_SGF, nodeCount);
      msg2logGint("%d parameters in gametree of the file", LOGLEVEL_STATISTICS, LOGSCOPE_SGF, paramCount);
      msg2logGint("%d branches in gametree of the file", LOGLEVEL_STATISTICS, LOGSCOPE_SGF, branchCount);
      msg2logGint("%d characters in gametree of the file", LOGLEVEL_STATISTICS, LOGSCOPE_SGF, charCount);
#endif
    }
    if(nodeCount < 1) {
#ifndef PRODUCTION_VERSION  
      msg2log("File has no sgf content", LOGLEVEL_WARNING, LOGSCOPE_SGF);
#endif
      g_free(temp_buffer);
      gnome_vfs_close(handle);
      return;
    }

    if(mainview->mergeload == FALSE) {
      /* Reset settings for new game */
      ResetGame(ReadGobanSize(temp_buffer, &temp_buffer[finfo.size + 1]), mainview);
    }

    /* Parse sgf file */
    ParseSgfGame(mainview, temp_buffer, &temp_buffer[finfo.size + 1]);
    SetPlayerInTurn(mainview);
    SetStyle(mainview);
    if(mainview->sgfGN == NULL) {
      /* gtk_label_set_text(mainview->game_title_element,mainview->file_name); */
      gtk_label_set_text(GTK_LABEL(mainview->game_title_element),SkipSlashes(mainview->file_name));
    }
    else {
      gtk_label_set_text(GTK_LABEL(mainview->game_title_element),mainview->sgfGN);
    }
    /* gtk_notebook_set_current_page(mainview->notebook, 2); / * Set analysis notebook tab active */
    EnterNode(mainview);
    DrawAll(mainview);

    /* free temp, close file and return */
    g_free(temp_buffer);
    gnome_vfs_close(handle);
    EnterFile();
}



/* write to file */
void write_buffer_to_file ( MainView* mainview )
{
    GnomeVFSResult vfs_result;
    GnomeVFSHandle *handle = NULL;
    GnomeVFSFileSize out_bytes;
    gchar *temp_buffer = NULL;
    GtkTextIter start, end;
    g_assert( mainview != NULL );

    ExitNode(mainview);
    /* Create content of the file buffer */
    mainview->fileBuffer = InitSgfHandling();
    CreateSgfFile(mainview);
    EnterNode(mainview);
    /* try to create handle to file */
    vfs_result = gnome_vfs_create(&handle, mainview->file_name, GNOME_VFS_OPEN_WRITE, 0, 0600);
    if ( vfs_result != GNOME_VFS_OK ) {
        interface_error( MAEMOPAD_ERROR_SAVE_FAILED, mainview );
        return;
    }

    /* find start and end of text */
    gtk_text_buffer_get_bounds( GTK_TEXT_BUFFER (mainview->fileBuffer), &start, &end);

    /* copy all text to temp_buffer */
    temp_buffer = gtk_text_buffer_get_slice( GTK_TEXT_BUFFER (mainview->fileBuffer), &start, &end, TRUE);

    /* write text to file */
    gnome_vfs_write(handle, temp_buffer, strlen(temp_buffer), &out_bytes);

    /* free temp, close file and return */
    g_free(temp_buffer);
    gnome_vfs_close(handle);
}

void callback_first ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  Browsing();
  ResetTrainingStatistics(FALSE, mainview->isGoProblem);
  Panicable(mainview);
  ExitNode(mainview);
  mainview->currentNode = NULL;
  EnterNode(mainview);
  DrawAll(mainview);
}

void callback_home ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  Browsing();
  ResetTrainingStatistics(FALSE, mainview->isGoProblem);
  Panicable(mainview);
  ExitNode(mainview);
  mainview->currentNode = mainview->compareNode;
  EnterNode(mainview);
#ifndef PRODUCTION_VERSION  
    msg2log("\nHome toolbar icon\n\nGoes to position marked by toolbar icon 1. or mark position menu selection.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_previous ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  Browsing();
  Panicable(mainview);
  ExitNode(mainview);
  mainview->currentNode = back_in_tree( mainview->currentNode ); /* Change position to the previous one */
  EnterNode(mainview);
  TakeMoveBack();
  DrawAll(mainview);
}

void callback_next ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  Browsing();
  PeekNext();
  PrintTrainingStatistics(mainview->isGoProblem);
  Panicable(mainview);
  ExitNode(mainview);
  mainview->currentNode = forward_in_tree( mainview->currentNode, mainview->child ); /* Change position to the next one */
  EnterNode(mainview);
  DrawAll(mainview);
}

void callback_last ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  Browsing();
  Panicable(mainview);
  ExitNode(mainview);
  do {
    mainview->currentNode = forward_in_tree( mainview->currentNode, mainview->child ); /* Change position to the next one */
    if(mainview->currentNode == NULL) {
      break;
    }
  } while(mainview->currentNode->child != NULL);
  EnterNode(mainview);
#ifndef PRODUCTION_VERSION  
    msg2log("\nLast toolbar icon\n\nGoes to the end of currently selected line of analysis.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_mainline ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  Browsing();
  Panicable(mainview);
  ExitNode(mainview);
  if(mainview->currentNode != NULL) { /* If not at root */
    while(mainview->currentNode->nodeHeight > 0) { /* go back until main line found */
      mainview->currentNode = mainview->currentNode->parent;
      if(mainview->currentNode == NULL) { /* or end to be at root */
        break;
      }
    }
  }
  ResetActive(mainview->child);
  EnterNode(mainview);
#ifndef PRODUCTION_VERSION  
    msg2log("\nGoto mainline\n\nGoes backwards until mainline is reached. In addition this feature resets all branch selections to the first choise.\n\nThis feature has been designed for going through analysed game. You can get back to the main game from variation with this feature. Resetting all branches supports use of next variation toolbar icon.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_variation ( GtkAction * action, gpointer data )
{
  Node * node;
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  Browsing();
  Panicable(mainview);
  ExitNode(mainview);
  if(mainview->currentNode != NULL) {
    if(mainview->currentNode->child != NULL) {
      mainview->currentNode = mainview->currentNode->child;
    }
  }
  while(mainview->currentNode != NULL) { /* Towards root find a branch */
    if((mainview->currentNode->sibling->nodeHeight > mainview->currentNode->nodeHeight) || /* where we are not the bottom most branch */
       (mainview->currentNode->sibling->nodeHeight == 0 && mainview->currentNode->nodeHeight > 0)) { /* unless main branch is reached */
      break;
    }
    mainview->currentNode = mainview->currentNode->parent;
  }
  if(mainview->currentNode != NULL) {
    node = get_active(mainview->currentNode);
    if(node != NULL) {
      set_active(node->sibling);
    }
    mainview->currentNode = mainview->currentNode->parent;
  }
  else {
    node = get_active(mainview->child);
    if(node != NULL) {
      set_active(node->sibling);
    }
  }
  EnterNode(mainview);
#ifndef PRODUCTION_VERSION  
    msg2log("\nGoto variation\n\nGoto variation is not the same functionality that downwards arrow has. When there are several choises for next play at the current position, this feature does the same as downwards arrow. When there is no continuation or only one continuation, this feature goes backwards until mainline is reached or branch where current line of analysis is not the last alternative is found. If such branching point is found before mainline, then next branch at there is selected.\n\nYou can use this feature for going through variations of interest. First click goto mainline icon and goto beginning in which ever order. Then use fastforward toolbar icon to jump forward in game until you reach a position where are variations of interest. Use next variation toolbar icon to select the first variation of interest and then proceed with fastforward until end of variation possibly via other positions where are variations. Finally, when variation no longer is interesting, clicking next variation toolbar icon gets you to the next variation you have not yet rejected or if you have already looked all interesting subvariations you go back to the mainline. You can ofcause go lines step by step, but above describtion is the fasted way to look through the variations of analysed game.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_nbranch ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  Browsing();
  Panicable(mainview);
  ExitNode(mainview);
  do {
    mainview->currentNode = forward_in_tree( mainview->currentNode, mainview->child ); /* Change position to the next one */
    if(mainview->currentNode == NULL) {
      break;
    }
    if(mainview->currentNode->child != NULL) {
      if(mainview->currentNode->child != mainview->currentNode->child->sibling) {
        break;
      }
    }
  } while(mainview->currentNode->child != NULL && 
          mainview->currentNode->sgfC == NULL);
  EnterNode(mainview);
#ifndef PRODUCTION_VERSION  
    msg2log("\nFast forward\n\nFast forward moves to next position that has either comment or several continuation alternatives. If no such, goes to the end of variation.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_pbranch ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  Browsing();
  Panicable(mainview);
  ExitNode(mainview);
  do {
    mainview->currentNode = back_in_tree( mainview->currentNode ); /* Change position to the previous one */
    if(mainview->currentNode == NULL) {
      break;
    }
    if(mainview->currentNode->child != NULL) {
      if(mainview->currentNode->child != mainview->currentNode->child->sibling) {
        break;
      }
    }
  } while(mainview->currentNode->sgfC == NULL);
  EnterNode(mainview);
#ifndef PRODUCTION_VERSION  
    msg2log("\nFast backwards\n\nFast backwards goes to the previous position that has either comment or several alternatives for continuation. If no such, goes to the beginning of the game.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_add_kikashi ( GtkAction * action, gpointer data )
{
  Node * what = NULL;
  Node * node = NULL;
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  Panicable(mainview);
  ExitNode(mainview);
  what = mainview->currentNode; /* Starting from current node */
  while(what != NULL) { /* up to root */
    if(what->parent != NULL) {
      if(what->parent->child != what) { /* search for branching point where current branch is not the first child */
        node = what;
        what = what->parent->child;
        while(what->sibling != node) { /* In case of several corrections at same node */
          what = what->sibling; /* we want to add the kikashi to variation containing latest correction */
        }
        break;
      }
    }
    what = what->parent; /* going towards root */
  }
  if(what == NULL) { /* If such is not found */
#ifndef PRODUCTION_VERSION  
    msg2log("\nAdd kikashi\n\nAdd kikashi can not be used at the mainline. To get more help, try add kikashi outside the mainline.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
    return; /* do nothing */
  }
  mainview->file_edited = TRUE;
  PlayMoves( what, mainview, FALSE); /* In practical situation it is more likely that user accidentally gives too many moves in place of missing moves than any other reason to create illegal sequence, thus omitting illegal moves only is good quess of what user intended */
  EnterNode(mainview);
#ifndef PRODUCTION_VERSION  
    msg2log("\nAdd kikashi\n\nAdd kikashi play all possible moves starting from begining of variation following variation line above current variation via the route last visitted at that variation line.\n\nAdd kikashi feature is ment for later adding plays that has been forgot to record when they are played. When you realise that you have forgot to play some moves, go to the position where those moves should have been played and then play the missing moves. After that, click add kikashi toolbar icon. It does not matter if you play extra moves after the moves that has been forgot.\n\nExample 1: Add one missing move\nGo to the position where the move is missing. Play the missing move. Click add kikashi toolbar icon. As result plays after the move that has been forgot are played, but their color is reversed.\n\nExample 2: Move one kikashi earlier in the game.\nGo to the position where the kikashi should have been played. Play both moves of the kikashi. Click add kikashi toolbar icon. As result the kikashi moves are played after position following rest of the game and then omitted from the place where they were.\n\nExample 3: Swap order of two consequent plays\nGo to the position before the plays in the wrong order. Play the move that should have been played first. You may play the other move, but need not to. Click add kikashi toolbar icon. As result order and color of those plays has been changed.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void OnceEverySecond ( gpointer data )
{
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  return;
}

void callback_alphaTesting ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->alphaTesting = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mainview->alphaTesting_item));
  set_log2file(mainview->alphaTesting,mainview->file_name);
  if(mainview->alphaTesting == TRUE) {
    gtk_widget_show(GTK_WIDGET(mainview->alphatesting_tb));
#ifndef PRODUCTION_VERSION  
    msg2log("Alpha testing option activates features that for some reason are difficult to test in the development environment. This option is ment for application developpers only. If you are not application developper I strongly suggest that you switch off this option before doing anything else.", LOGLEVEL_WARNING, LOGSCOPE_USER);
#endif
  }
  else {
    gtk_widget_hide(GTK_WIDGET(mainview->alphatesting_tb));
#ifndef PRODUCTION_VERSION  
    msg2log("Thank you for your co-operation.", LOGLEVEL_INFO, LOGSCOPE_USER);
#endif
  }
  DrawAll(mainview);
}

void callback_test ( GtkAction * action, gpointer data )
{
  gint clock1;
  gint clock2;
  gint i;
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

#ifndef PRODUCTION_VERSION  
  msg2log("Test started, resetting timer.", LOGLEVEL_INFO, LOGSCOPE_FORCED);
#endif
  clock1 = time(NULL);
  /* Insert call for your test call after this */
  for(i=0; i<mainview->playlistsize; i++) {
    callback_next_file (action , data);
  }
  /* Insert call for your test call before this */
  clock2 = time(NULL);
#ifndef PRODUCTION_VERSION  
  msg2logGint("Test completed in %d seconds", LOGLEVEL_INFO, LOGSCOPE_FORCED,clock2 - clock1);
#endif
}

void callback_mergeLoad ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->mergeload = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mainview->merge_load_item));
  if(mainview->mergeload == TRUE) {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Opened games will be combined with previously opened game.");
  }
  else {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Opened games will replace previously opened game.");
  }
#ifndef PRODUCTION_VERSION  
  msg2log("\nMerge load\n\nWhen this option is selected, the file opened to Goban770 is not cleared before loading new file. This will effectively combine the 2 files into the same game tree.\n\nExample: Constructing fuseki dictionary\nDownload set of games that are all of opening of your interest. Cut each game to the length you like, by deleting the end of the file and edit orientation of the files so that first stones are played at same locations. Save all such files in a single directory with no other sgf-content. Open first of those files. Set merge load option on. Click next file icon until game tree no longer changes. Save the created file as your new fuseki dictionary. Switch merge load option off.", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_minimalLoad ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->minimalload = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mainview->minimal_load_item));
  if(mainview->minimalload == TRUE) {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Only plays and game tree structure will be opened.");
  }
  else {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "All supported sgf is read in.");
  }
#ifndef PRODUCTION_VERSION  
  msg2log("\nRead plays only option\n\nWhen this option is selected, only played stones and the game tree structure is read in. All comments, board markup symbols, setup stones etc. are ignored. This makes the file as small as possible and will remove any custom markings.", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_manualOnly ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->manualOnly = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mainview->manualOnly_item));
  if(mainview->manualOnly == TRUE) {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Tabs will be changed only manually.");
  }
  else {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Tabs will be changed automatically based on situation.");
  }
#ifndef PRODUCTION_VERSION  
  msg2log("\nOnly manual tab control\n\nWhen this option is selected tabs will be changed only when manually selected.\n\nWhen this option is not selected, other actions user does may change the tabs based on assumptations on what user wants.", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_tryLoad ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );
  
  if(gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mainview->try_load_item)) == TRUE) {
    mainview->tryLoad = 1;
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Click next file to reload with limitted SGF support.");
  }
  else {
    mainview->tryLoad = 0;
  }
#ifndef PRODUCTION_VERSION  
  msg2log("\nTry load\n\nThis is SGF-syntax parsing debugging option. Switch this option off unless you are developper of Goban770.", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_bigTreeVariations ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->bigTreeVariations = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mainview->bigTreeVariations_item));
  if(mainview->bigTreeVariations == TRUE) {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "All variations of the gametree will be shown regardless of the gametree size.");
  }
  else {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "If gametree is big, only start of the non-active variations are shown.");
  }
#ifndef PRODUCTION_VERSION  
  msg2log("\nShow big tree variations\n\nWhen this option is selected gametree is shown complately.\n\nWhen this option is not selected, only start of the variations is shown if game tree is big.", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_omitEmptyNodes ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->omitEmptyNodes = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mainview->omit_empty_nodes_item));
  if(mainview->omitEmptyNodes == TRUE) {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Nodes without stones will be combined with next node if possible.");
  }
  else {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Nodes without stones are loaded as is.");
  }
#ifndef PRODUCTION_VERSION  
  msg2log("\nOmit nodes\n\nIn some rare cases sgf-file has content that is not ment as part of the game tree but comments in between. Such comments could have been placed at nodes that do not have played stones or set up position. This option need to be switched off to see such files as intended. How ever, Goban770 is designed to work with game trees where each node as something sensible to show on the goban and many features work better if such nodes are not in the game tree.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void callback_set_bwinr ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->sgfRE = SetNote(mainview->sgfRE, "B+R");
  DrawAll(mainview);
}

void callback_set_bwint ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->sgfRE = SetNote(mainview->sgfRE, "B+T");
  DrawAll(mainview);
}

void callback_set_bwinf ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->sgfRE = SetNote(mainview->sgfRE, "B+Forfeit");
  DrawAll(mainview);
}

void callback_set_wwinr ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->sgfRE = SetNote(mainview->sgfRE, "W+R");
  DrawAll(mainview);
}

void callback_set_wwint ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->sgfRE = SetNote(mainview->sgfRE, "W+T");
  DrawAll(mainview);
}

void callback_set_wwinf ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->sgfRE = SetNote(mainview->sgfRE, "W+Forfeit");
  DrawAll(mainview);
}

void callback_set_void ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->sgfRE = SetNote(mainview->sgfRE, "Void");
  DrawAll(mainview);
}

void callback_set_unknown ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->sgfRE = SetNote(mainview->sgfRE, "?");
  DrawAll(mainview);
}

void callback_set_estimate ( GtkAction * action, gpointer data )
{
  gint scorex2 = 0;
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  if(mainview->currentNode == NULL) {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Move to the end position of the game first.");
    DrawAll(mainview);
    return;
  }
  scorex2 = mainview->currentNode->sgfVx2;
  if(scorex2 == 0) {
    if(mainview->currentNode->sgfTW == NULL) {
      UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "Count the territories first. Click aside goban to switch to territory counting mode.");
      DrawAll(mainview);
      return;
    }
    mainview->sgfRE = SetNote(mainview->sgfRE, "Draw");
    DrawAll(mainview);
    return;
  }
  if(scorex2 > 0) {
    if((scorex2%2) == 0) {
      mainview->sgfRE = SetNote(mainview->sgfRE, AddGintInString("B+%d", scorex2/2));
    }
    else {
      mainview->sgfRE = SetNote(mainview->sgfRE, Add2GintInString("B+%d.%d", scorex2/2, (scorex2%2)*5));
    }
  }
  else {
    if(((0-scorex2)%2) == 0) {
      mainview->sgfRE = SetNote(mainview->sgfRE, AddGintInString("W+%d", (0-scorex2)/2));
    }
    else {
      mainview->sgfRE = SetNote(mainview->sgfRE, Add2GintInString("W+%d.%d", (0-scorex2)/2, ((0-scorex2)%2)*5));
    }
  }
  DrawAll(mainview);
}

void callback_playlist ( GtkRadioMenuItem * item, gpointer data )
{
  PlayList * plist = NULL;
  
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  if(gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(item)) == FALSE) {
    return; /* We are not interested in deselecting previous playlist */ 
  }
  plist = mainview->firstPlaylist;
  while(plist != NULL) {
    if(plist->playlist_item == GTK_WIDGET(item)) {
      break;
    }
    plist = plist->next;
  }
  if(mainview->currentPlaylist != plist) {
    mainview->currentPlaylist = plist;
    read_file_from_playlist(mainview);
  }
  DrawAll(mainview);
}

void callback_refresh_playlist ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  read_directory_to_list ( mainview, FALSE );
#ifndef PRODUCTION_VERSION  
  msg2log("\nRefresh current playlist\n\nThis feature reconstructs current playlist from the scratch.\n\nPlaylist of directory is constructed at the first time when file is opened from or saved at the directory. If you open another file from that directory and use next file feature, you continue at the previous point of playlist where you were, not next to the opened file. Also, if you have changed the directory content, the deleted files are at the playlist and added files are not. Renamed files are considered as deleted and added. Using this feature avoid those problems.\n", LOGLEVEL_HELP, LOGSCOPE_USER);
#endif
  DrawAll(mainview);
}

void AnnihilateThumbs (ThumbList * thumb)
{
  if(thumb == NULL) { /* If nothing to annihilate */
    return; /* we are at end of list and can start to go upwards */
  }
  AnnihilateThumbs(thumb->next); /* Process child before parent */
  if(thumb->title != NULL) { /* If title exists */
    g_free(thumb->title); /* release title */
  }
  if(thumb->sgfPW != NULL) { /* If white name exists exists */
    g_free(thumb->sgfPW); /* release it */
  }
  if(thumb->sgfPB != NULL) { /* If black name exists */
    g_free(thumb->sgfPB); /* release it */
  }
  if(thumb->thumb != NULL) { /* If thumb exists */
    g_free(thumb->thumb); /* release thumb */
  }
  g_free(thumb); /* Release list item holding thumb information */
}

void callback_thumb_playlist ( GtkAction * action, gpointer data )
{
  gint i, at;
  gint firstItem;
  gint whiteCommon = 0;
  gint blackCommon = 0;
  gint cutPoint = 0;
  ThumbList * thumb = NULL;
  ThumbList * thumb2 = NULL;
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  if(mainview->currentPlaylist == NULL) {
    UserMessage(GTK_WIDGET(mainview->goban_view), NULL, "No playlist to create thumbnails for.");
    DrawAll(mainview);
    return;
  }
  if(mainview->thumbedPlaylist != NULL) { /* If thumbs existed they are released */
    AnnihilateThumbs(mainview->thumbnails);
    mainview->thumbnails = NULL;
  }
  firstItem = mainview->currentPlaylist->current;
  mainview->currentPlaylist->current = (mainview->currentPlaylist->current - 1 + mainview->currentPlaylist->size) % mainview->currentPlaylist->size;
  for(i=0; ((i<MAX_AMOUNT_OF_THUMBNAILS) && (i<mainview->currentPlaylist->size)); i++) {
    callback_next_file (action , data);
    if(mainview->currentNode == NULL) {
      mainview->currentNode = mainview->child;
      if(mainview->currentNode != NULL) {
        while(mainview->currentNode->child != NULL) {
          mainview->currentNode = mainview->currentNode->child;
        }
      }
    }
    DrawAll(mainview);
    thumb = g_new0( ThumbList, 1 );
    thumb->next = mainview->thumbnails;
    thumb->thumb = g_new0( gchar, mainview->sgfSZ * mainview->sgfSZ );
    for(at=0; at<(mainview->sgfSZ * mainview->sgfSZ); at++) {
      switch(GetStoneAt(at)) {
        case STONE_BLACK:
          thumb->thumb[at] = 'b';
          break;
        case STONE_WHITE:
          thumb->thumb[at] = 'w';
          break;
        default :
          thumb->thumb[at] = ' ';
      }
    }
    if(mainview->currentNode != NULL) {
      if(mainview->currentNode->at >= 0) {
        if(mainview->currentNode->stone == STONE_BLACK) {
          thumb->thumb[mainview->currentNode->at] = 'B';
        }
        if(mainview->currentNode->stone == STONE_WHITE) {
          thumb->thumb[mainview->currentNode->at] = 'W';
        }
      }
    }
    if(mainview->sgfPW != NULL) {
      thumb->sgfPW = g_new0( gchar, strlen(mainview->sgfPW) + 1 );
      strcpy(thumb->sgfPW,mainview->sgfPW);
    }
    else {
      thumb->sgfPW = NULL;
    }
    if(mainview->sgfPB != NULL) {
      thumb->sgfPB = g_new0( gchar, strlen(mainview->sgfPB) + 1 );
      strcpy(thumb->sgfPB,mainview->sgfPB);
    }
    else {
      thumb->sgfPB = NULL;
    }
    thumb->title = NULL;
    thumb->sgfSZ = mainview->sgfSZ;
    thumb->indexAtPlaylist = mainview->currentPlaylist->current;
    mainview->thumbnails = thumb;
  }
  thumb = mainview->thumbnails;
  while(thumb != NULL) { /* Let's figure out what would be informative titles for the thumbnails */
    whiteCommon = 0;
    blackCommon = 0;
    if((thumb->sgfPW != NULL) && (thumb->sgfPB != NULL)) { /* If there are player names */
      /* let's figure out how common they are */
      thumb2 = mainview->thumbnails;
      while(thumb2 != NULL) {
        if(thumb2->sgfPB != NULL) {
          if(strcmp(thumb2->sgfPB,thumb->sgfPB) == 0) {
            blackCommon++;
          }
          if(strcmp(thumb2->sgfPB,thumb->sgfPW) == 0) {
            whiteCommon++;
          }
        }
        if(thumb2->sgfPW != NULL) {
          if(strcmp(thumb2->sgfPW,thumb->sgfPB) == 0) {
            blackCommon++;
          }
          if(strcmp(thumb2->sgfPW,thumb->sgfPW) == 0) {
            whiteCommon++;
          }
        }
        thumb2 =thumb2->next;
      }
    }
    if(whiteCommon == blackCommon) { /* If neither player name is common */
      /* form thumb title from the file name */
      thumb->title = g_new0( gchar, strlen(mainview->currentPlaylist->sgfplaylist[thumb->indexAtPlaylist]->d_name) + 1 );
      strcpy(thumb->title,mainview->currentPlaylist->sgfplaylist[thumb->indexAtPlaylist]->d_name);
    }
    if(whiteCommon > blackCommon) { /* If white player name is more common */
      /* form thumb title from the black player name */
      thumb->title = thumb->sgfPB;
      thumb->sgfPB = NULL;
    }
    if(whiteCommon < blackCommon) { /* If black player name is more common */
      /* form thumb title from the white player name */
      thumb->title = thumb->sgfPW;
      thumb->sgfPW = NULL;
    }
    for(cutPoint=0; cutPoint<THUMB_TITLE_MAX_LENGTH; cutPoint++) {
      if((thumb->title[cutPoint] == '\0') || (thumb->title[cutPoint] == '.') || (thumb->title[cutPoint] == '(')) {
        break;
      }
    }
    thumb->title[cutPoint] = '\0';
    thumb = thumb->next;
  }
  mainview->thumbedPlaylist = mainview->currentPlaylist;
  DrawAll(mainview);
}

void callback_index ( GtkAction * action, gpointer data )
{
  /* connect pointer to our MainView struct */
  MainView *mainview = NULL;
  mainview = ( MainView * ) data;
  g_assert(mainview != NULL && mainview->data != NULL );

  mainview->temporal_mode = TEMPMODE_FILE_THUMBS;
  gtk_widget_hide(mainview->notebook);
  gtk_widget_hide(mainview->toolbar);
  if(mainview->fullscreen == FALSE) {
    gtk_window_fullscreen(GTK_WINDOW(mainview->data->window));
  }
  DrawAll(mainview);
}

/* ------------------------- test code --------------------------- */

void draw_brush (GtkWidget * widget, gdouble x,  gdouble y, GdkPixmap * pixmap)
{
  GdkRectangle update_rect;
  GdkGC * gc;
  
  update_rect.x = x - 5;
  update_rect.y = y - 5;
  update_rect.width = 10;
  update_rect.height = 10;
  gc = gdk_gc_new(pixmap);
  gdk_gc_set_function (gc, GDK_INVERT);
  gdk_draw_rectangle (pixmap, gc, TRUE, update_rect.x, update_rect.y, update_rect.width, update_rect.height);
/*  gdk_draw_rectangle (pixmap, widget->style->black_gc, TRUE, update_rect.x, update_rect.y, update_rect.width, update_rect.height); */
  gtk_widget_draw (widget, &update_rect);
}

