/*
 * This file is part of goban770
 *
 * Copyright (C) 2006,2007 Jarmo Ahosola.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <gtk/gtk.h>
#include <utils/log.h>
#include <game/notes.h>
#include <utils/draw.h>
#include <string.h>

/* Set text into a note */
gchar * SetNote(gchar * oldNote, gchar * newText)
{
  gchar * newNote = NULL;
  
  DestroyNote(oldNote);
  if(newText == NULL) {
    return NULL;
  }
  newNote = ReserveNote(strlen(newText));
  strcpy(newNote,newText);
  return newNote;
}

/* Reserve memory for note */
gchar * ReserveNote(gint sizeOfNote)
{
  gchar * note = NULL;

  note = g_malloc(sizeof(gchar) * (sizeOfNote + 1));
  memset(note, 0, sizeof(gchar) * (sizeOfNote + 1));
  return note;
}

/* Release memory reserved for note */
void DestroyNote(gchar * note)
{
  if(note == NULL) {
    return;
  }
  g_free(note);
}

/* Replace note content form content of text buffer */
gchar * GetNewNoteFromBuffer(gchar * note, GtkTextBuffer * textBuffer)
{
  GtkTextIter start;
  GtkTextIter end;
  gchar * newNote = NULL;

  DestroyNote(note);
  if(textBuffer == NULL) {
    return NULL;
  }
  if(gtk_text_buffer_get_char_count(textBuffer) == 0) {
    return NULL;
  }
  gtk_text_buffer_get_start_iter(textBuffer,&start);
  gtk_text_buffer_get_end_iter(textBuffer,&end);
  newNote = gtk_text_buffer_get_text(textBuffer,&start,&end,FALSE);
  return newNote;
}

/* Replace text buffer content with note content */
void SetNoteAtBuffer(gchar * note, GtkTextBuffer * textBuffer)
{
  if(note == NULL) {
    gtk_text_buffer_set_text(textBuffer,"",-1);
  }
  else {
    gtk_text_buffer_set_text(textBuffer,note,-1);
  }
}

/* Create new string that has \ and ] properly escaped according sgf syntax. */
gchar * AddSgfEscapes(gchar * note)
{
  gint newLength = 0;
  gint position = 0;
  gchar * at = NULL;
  gchar * newNote = NULL;
  
  at = note;
  while(at[0] != '\0') {
    if(at[0] == '\\') {
      newLength++;
    }
    if(at[0] == ']') {
      newLength++;
    }
    newLength++;
    at++;
  }
  newNote = ReserveNote(newLength);
  at = note;
  while(at[0] != '\0') {
    if(at[0] == '\\') {
      newNote[position++] = '\\';
    }
    if(at[0] == ']') {
      newNote[position++] = '\\';
    }
    newNote[position++] = at[0];
    at++;
  }
  return newNote;
}

/* Remove escapes from \\ and \] converted according sgf syntax. */
void RemoveSgfEscapes(gchar * note)
{
  gchar * newValue = NULL;
  gboolean escaped = FALSE;
  
  newValue = note;
  while(newValue[0] != '\0') {
    note[0] = newValue[0];
    if((escaped == TRUE) || (newValue[0] != '\\')) {
      note++;
      escaped = FALSE;
    }
    else {
      escaped = TRUE;
    }
    newValue++;
  }
  note[0] = newValue[0];
}

gchar * ExtractNoteFromSgf(gchar * sgfPointer, gchar * sgfEnd)
{
  gchar * note = NULL;
  gchar * at = NULL;
  gint index = 0;
  
  at = sgfPointer;
  while(at[0] != ']' && at < sgfEnd) {
    if(at[0] == '\\') {
      at++;
    }
    at++;
  }
  if(at == sgfPointer) {
    return NULL;
  }
  note = ReserveNote(at - sgfPointer);
  while(sgfPointer < at) {
    note[index] = sgfPointer[0];
    index++;
    sgfPointer++;
  }
  RemoveSgfEscapes(note);
  return note;
}

/* Preparations done when game position is entered */
void EnterNode(MainView * mainview)
{
  GtkTextTag *bluetag = NULL;
  GtkTextTag *blacktag = NULL;
  GtkTextIter bufferend;
  Node * here = NULL;
#ifndef PRODUCTION_VERSION  
  msg2log("EnterNode", LOGLEVEL_WARNING, LOGSCOPE_PATH);
#endif
  if(mainview->CurrentMouseMode == mainview->search_item) {
    gtk_text_buffer_set_text(mainview->noteBuffer,"",-1);
    here = mainview->currentNode;
    bluetag = gtk_text_buffer_create_tag (mainview->noteBuffer, NULL, "foreground", "blue", NULL); /* Move title color */
    blacktag = gtk_text_buffer_create_tag (mainview->noteBuffer, NULL, "foreground", "black", NULL); /* Text color */
    if(here == NULL) {
      if(mainview->sgfC != NULL) {
        gtk_text_buffer_get_iter_at_offset (mainview->noteBuffer, &bufferend, -1);
        gtk_text_buffer_insert_with_tags(mainview->noteBuffer,&bufferend,"Game comment:\n",-1,bluetag,NULL);
        gtk_text_buffer_get_iter_at_offset (mainview->noteBuffer, &bufferend, -1);
        gtk_text_buffer_insert_with_tags(mainview->noteBuffer,&bufferend,mainview->sgfC,-1,blacktag,NULL);
        gtk_text_buffer_get_iter_at_offset (mainview->noteBuffer, &bufferend, -1);
        gtk_text_buffer_insert_with_tags(mainview->noteBuffer,&bufferend,"\n\n",-1,blacktag,NULL);
      }
      here = mainview->child;
    }
    while(here != NULL) {
      here = get_active(here);
      if(here->sgfC != NULL) {
        gtk_text_buffer_get_iter_at_offset (mainview->noteBuffer, &bufferend, -1);
        if(here->stone == STONE_BLACK) {
          gtk_text_buffer_insert_with_tags(GTK_TEXT_BUFFER(mainview->noteBuffer),&bufferend,AddGintInString("%d: Black ", here->nodeTurnNumber),-1,bluetag,NULL);
        }
        if(here->stone == STONE_WHITE) {
          gtk_text_buffer_insert_with_tags(GTK_TEXT_BUFFER(mainview->noteBuffer),&bufferend,AddGintInString("%d: White ", here->nodeTurnNumber),-1,bluetag,NULL);
        }
        gtk_text_buffer_get_iter_at_offset (mainview->noteBuffer, &bufferend, -1);
        if(here->at == AT_IS_PASS) {
          gtk_text_buffer_insert_with_tags(mainview->noteBuffer,&bufferend,"pass",-1,bluetag,NULL);
        }
        if(here->at >= 0) {
          gtk_text_buffer_insert_with_tags(mainview->noteBuffer,&bufferend,At2String(here->at, mainview->sgfSZ),-1,bluetag,NULL);
        }
        gtk_text_buffer_get_iter_at_offset (mainview->noteBuffer, &bufferend, -1);
        gtk_text_buffer_insert_with_tags(mainview->noteBuffer,&bufferend,"\n",-1,blacktag,NULL);
        gtk_text_buffer_get_iter_at_offset (mainview->noteBuffer, &bufferend, -1);
        gtk_text_buffer_insert_with_tags(mainview->noteBuffer,&bufferend,here->sgfC,-1,blacktag,NULL);
        gtk_text_buffer_get_iter_at_offset (mainview->noteBuffer, &bufferend, -1);
        gtk_text_buffer_insert_with_tags(mainview->noteBuffer,&bufferend,"\n\n",-1,blacktag,NULL);
      }
      here = here->child;
    }
  }
  else {
    if(mainview->currentNode == NULL) {
      SetNoteAtBuffer(mainview->sgfC, mainview->noteBuffer);
    }
    else {
      SetNoteAtBuffer(mainview->currentNode->sgfC, mainview->noteBuffer);
      if(mainview->currentNode->child == NULL) {
        LastNodeEnter();
      }
    }
  }
}

/* Preparations done when game position is exited */
void ExitNode(MainView * mainview)
{
#ifndef PRODUCTION_VERSION  
  msg2log("ExitNode", LOGLEVEL_WARNING, LOGSCOPE_PATH);
#endif
  if(mainview->CurrentMouseMode == mainview->search_item) {
  }
  else {
    if(mainview->currentNode == NULL) {
      mainview->sgfC = GetNewNoteFromBuffer(mainview->sgfC, mainview->noteBuffer);
    }
    else {
      mainview->currentNode->sgfC = GetNewNoteFromBuffer(mainview->currentNode->sgfC, mainview->noteBuffer);
      if(mainview->currentNode->child == NULL) {
        LastNodeExit();
      }
    }
    SetNoteAtBuffer(NULL, mainview->noteBuffer);
  }
}

gchar * SkipSlashes(gchar * directory)
{
  gchar * filename = NULL;
  
  filename = directory;
  while(filename[0] != '\0') {
    filename++;
  }
  while((filename != directory) && (filename[0] != '\\') && (filename[0] != '/')) {
    filename--;
  } 
  if((filename[0] == '\\') || (filename[0] == '/')) {
    filename++;
  }
  return filename;
}
