/*
 * This file is part of goban770
 *
 * Copyright (C) 2006,2007 Jarmo Ahosola.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <gtk/gtk.h>
#include <utils/log.h>
#include <game/goban.h>
#include <string.h>

/* privates */
gboolean FindLiberty(gint around);

static Stone oneDgoban[MAX_GOBAN_SPACE];
static gint oneDhelp[MAX_GOBAN_SPACE];
static gint goban_h_goban_size;

Stone GetStoneAt(gint at)
{
  return oneDgoban[at];
}

void SetStoneAt(gint at, Stone stone)
{
  oneDgoban[at] = stone;
}

Stone GetStoneXY(gint x,gint y)
{
  return oneDgoban[XY2LOCATION(x,y)];
}

void InitGoban(gint size)
{
  goban_h_goban_size = size;
}

gboolean IsOutofGoban(gint x, gint y, MainView* main)
{
  gint nedge = 0, sedge = 0, wedge = 0, eedge = 0;
  gint xerr = 0, yerr = 0;
  xerr = ((MISSCLICKLIMIT) * main->colWidth) / 100;
  yerr = ((MISSCLICKLIMIT) * main->rowHeight) / 100;
  wedge = main->centerX - (main->centerMostStone * main->colWidth) - (main->colWidth/2) - xerr;
  eedge = main->centerX + ((main->centerMostStone - 1 +(main->sgfSZ%2)) * main->colWidth) + (main->colWidth/2) + xerr;
  nedge = main->centerY - (main->centerMostStone * main->rowHeight) - (main->rowHeight/2) - yerr;
  sedge = main->centerY + ((main->centerMostStone - 1 +(main->sgfSZ%2)) * main->rowHeight) + (main->rowHeight/2) + yerr;
  if(x<wedge) return TRUE;
  if(x>eedge) return TRUE;
  if(y<nedge) return TRUE;
  if(y>sedge) return TRUE;
  return FALSE;
}

gboolean IsValidClick(gint x, gint y, MainView* main)
{
  gint ex = 0, ey = 0, xerr = 0, yerr = 0;
  xerr = (MISSCLICKLIMIT * main->colWidth) / 100;
  yerr = (MISSCLICKLIMIT * main->rowHeight) / 100;
  ex = gobanx2viewx(viewx2gobanx(x, main), main );
  ey = gobany2viewy(viewy2gobany(y, main), main );
  if(xerr + x < ex) return FALSE;
  if(xerr + ex < x) return FALSE;
  if(yerr + y < ey) return FALSE;
  if(yerr + ey < y) return FALSE;
  return TRUE;
}

gint GetClickedNumber(gint x, gint y, MainView* main, gint column)
{
  gint wedge = 0, eedge = 0, fedge = 0, clickY = 0;
  gint xerr = 0, yerr = 0;
  xerr = ((MISSCLICKLIMIT) * main->colWidth) / 200;
  yerr = ((MISSCLICKLIMIT) * main->rowHeight) / 100;
  wedge = main->centerX - (main->centerMostStone * main->colWidth) - (main->colWidth/2) - xerr;
  eedge = main->centerX + (((main->centerMostStone - 1 +(main->sgfSZ%2)) - 1 + column) * main->colWidth) + (main->colWidth/2) + xerr;
  fedge = main->centerX + (((main->centerMostStone - 1 +(main->sgfSZ%2)) + column) * main->colWidth) + (main->colWidth/2) - xerr;
  if((column <= 0) && (wedge < x)) {
    return 0;
  }
  if((column > 0) && ((eedge > x) || (fedge < x))) {
    return 0;
  }
  clickY = viewy2gobany(y, main);
  if((clickY < 0) || (clickY >= main->sgfSZ)) {
    return 0;
  }
  return main->sgfSZ - clickY;
}

gint gobanx2viewx(gint x, MainView* main )
{
/* #define GOBANX2VIEWX(x) ((main->centerX) + (((x) - (main->centerMostStone)) * (main->colWidth))) */
  return main->centerX - (main->centerMostStone * main->colWidth) + (x * main->colWidth);
}

gint gobany2viewy(gint y, MainView* main )
{
/* #define GOBANY2VIEWY(y) ((main->centerY) + (((x) - (main->centerMostStone)) * (main->colWidth))) */
  return main->centerY - (main->centerMostStone * main->rowHeight) + (y * main->rowHeight);
}

gint viewx2gobanx(gint x, MainView* main )
{
  return ( ((main->colWidth / 2) +x - main->centerX + (main->colWidth * main->centerMostStone)) / main->colWidth );
}

gint viewy2gobany(gint y, MainView* main )
{
  return ( ((main->rowHeight / 2) + y + (main->rowHeight * main->centerMostStone) - main->centerY) / main->rowHeight);
}

gint Location2X(gint at)
{
  return LOCATION2X(at);
}

gint Location2Y(gint at)
{
  return LOCATION2Y(at);
}

gint XY2Location(gint x,gint y)
{
  return XY2LOCATION(x,y);
}

Node * findStartOfNumbering( MainView* main )
{
  gint i;
  Node * node;

  if((main->diagramview == TRUE) || (main->CurrentMouseMode == main->diagram_item)) {
    node = main->currentNode;
    i = 0;
    while(node != NULL) { /* Numbering starts at */
      if(node->parent == NULL) { /* first node or */
        return node;
      }
      if(node == main->compareNode) { /* marked node or */
        return node;
      }
      i++;
      if(i >= main->numberLastPlays) { /* fixed amount of moves before current position, which ever is closest to current position */
        return node;
      }
      node = node->parent;
    }
  }
  return NULL;
}

/* set current position on the game map from the scratch */
gint setCurrentPosition( MainView* main )
{
    gint i;
    gint underStones = 0;
    Pointlist * hstone = main->sgfAB;
    gboolean removeCapturedStones = TRUE;
    Node * node;
    Node * numberingBase = NULL;
#ifndef PRODUCTION_VERSION  
    msg2log("Enter setCurrentPosition", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
    
    /* Set goban empty to start with */
    for(i=0;i<(goban_h_goban_size * goban_h_goban_size);i++) {
      oneDgoban[i] = STONE_NONE;
    }
    SetPlayerInTurn(main); /* Determine player in turn at current position */
#ifndef PRODUCTION_VERSION  
    msg2logGint("Cleared %d goban locations from stones", LOGLEVEL_INFO, LOGSCOPE_GO_RULES,i);
#endif
    /* Put in place the handicap stones */
    while(hstone != NULL) {
      oneDgoban[hstone->at] = STONE_BLACK;
      hstone = hstone->next;
    }
    /* Put in place white setup stones */
    hstone = main->sgfAW;
    while(hstone != NULL) {
      oneDgoban[hstone->at] = STONE_WHITE;
      hstone = hstone->next;
    }
    /* Reset capture stones */
    main->capturedBlack = 0;
    main->capturedWhite = 0;
    /* Play rest of the moves */
#ifndef PRODUCTION_VERSION  
    msg2log("Play rest of the moves", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
    if(main->child == NULL) {
#ifndef PRODUCTION_VERSION  
      msg2log("Exit setCurrentPosition (no game)", LOGLEVEL_WARNING, LOGSCOPE_PATH);
#endif
      return 0; /* No game */
    }
    if((main->currentNode == NULL) && (main->CurrentMouseMode != main->search_item)) {
#ifndef PRODUCTION_VERSION  
      msg2log("Exit setCurrentPosition (at root)", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
      return 0; /* At root node */
    }
    numberingBase = findStartOfNumbering( main );
    node = main->child; /* Go to top level */
#ifndef PRODUCTION_VERSION  
    msg2log("Go to top level", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
    while(node != NULL) {
#ifndef PRODUCTION_VERSION  
      msg2log("Get the active branch", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
      node = get_active(node); /* At current level get the active branch */
      if(node == NULL) {
#ifndef PRODUCTION_VERSION  
        msg2log("Active branch was NULL", LOGLEVEL_ERROR, LOGSCOPE_NONE);
#endif
#ifndef PRODUCTION_VERSION  
        msg2log("Exit setCurrentPosition", LOGLEVEL_WARNING, LOGSCOPE_PATH);
#endif
        return underStones;
      }
#ifndef PRODUCTION_VERSION  
      msg2log("Place stone on goban", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
#ifndef PRODUCTION_VERSION  
      msg2logGint("Put stone at %d", LOGLEVEL_INFO, LOGSCOPE_NONE,node->at);
#endif
		if(removeCapturedStones == TRUE) {
        MakeMove(LOCATION2X(node->at),LOCATION2Y(node->at),node->stone,main,FALSE);
        node->underTheStones = FALSE;
        if((node == numberingBase) && (main->CurrentMouseMode == main->diagram_item)) {
          removeCapturedStones = FALSE;
        }
      }
      else {
        if(main->transparentCaptured == TRUE) {
          if((oneDgoban[node->at] == STONE_TRANSPARENT_BLACK) || (oneDgoban[node->at] == STONE_TRANSPARENT_WHITE)) {
            node->underTheStones = TRUE;
          }
          else {
            node->underTheStones = FALSE;
          }
          MakeMove(LOCATION2X(node->at),LOCATION2Y(node->at),node->stone,main,TRUE);
        }
        else {
          if(IS_EMPTY(oneDgoban[node->at]) == TRUE) {
            oneDgoban[node->at] = node->stone;
            node->underTheStones = FALSE;
          }
          else {
            underStones++;
            node->underTheStones = TRUE;
          }
        }
      }
      /* Put in place empty setup stones */
      hstone = node->sgfAE;
      while(hstone != NULL) {
        oneDgoban[hstone->at] = STONE_NONE;
        hstone = hstone->next;
      }
      /* Put in place black setup stones */
      hstone = node->sgfAB;
      while(hstone != NULL) {
        oneDgoban[hstone->at] = STONE_BLACK;
        hstone = hstone->next;
      }
      /* Put in place white setup stones */
      hstone = node->sgfAW;
      while(hstone != NULL) {
        oneDgoban[hstone->at] = STONE_WHITE;
        hstone = hstone->next;
      }
      
      if((node == main->currentNode) && (main->CurrentMouseMode != main->search_item)) {
#ifndef PRODUCTION_VERSION  
        msg2log("Current position reached", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
#ifndef PRODUCTION_VERSION  
        msg2log("Exit setCurrentPosition", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
        return underStones; /* Stop producing moves if current position is reached */
      }
#ifndef PRODUCTION_VERSION  
      msg2log("Next level in the game tree", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
      node = node->child; /* Go to next level in the game tree */
    }
#ifndef PRODUCTION_VERSION  
        msg2log("Exit setCurrentPosition", LOGLEVEL_WARNING, LOGSCOPE_PATH);
#endif
    return underStones;
}

/* Tells if group has liberty. Use this instead of FindLiberty function */
gboolean HasLiberty(gint groupAt)
{
  memset(oneDhelp, 0, sizeof(gint) * MAX_GOBAN_SPACE); /* Clear the help space */
  return FindLiberty(groupAt);
}

/* Find if there is a liberty for the group. Do not call this directly */
gboolean FindLiberty(gint around)
{
  /* Look towards north */
  if(AT_NORTH_EDGE(around) == FALSE) { /* If we are not at the edge */
    if(oneDhelp[NORTH_OF(around)] == 0) { /* and the direction has not been searched yet */
	  if(IS_EMPTY(oneDgoban[NORTH_OF(around)]) == TRUE) { /* If there is liberty */
	    return TRUE; /* the group has liberty */
	  }
	  oneDhelp[NORTH_OF(around)] = 1; /* As we tested liberty already, we do not want to search here again */
	  if(oneDgoban[around] == oneDgoban[NORTH_OF(around)]) { /* If the group extends to the direction */
	    if(FindLiberty(NORTH_OF(around)) == TRUE) { /* check if somewhere in there is liberty */
	      return TRUE; /* Once we find a liberty we quit searching for more */
		}
	  }
	}
  }
  /* Look towards west */
  if(AT_WEST_EDGE(around) == FALSE) { /* If we are not at the edge */
    if(oneDhelp[WEST_OF(around)] == 0) { /* and the direction has not been searched yet */
	  if(IS_EMPTY(oneDgoban[WEST_OF(around)]) == TRUE) { /* If there is liberty */
	    return TRUE; /* the group has liberty */
	  }
	  oneDhelp[WEST_OF(around)] = 1; /* As we tested liberty already, we do not want to search here again */
	  if(oneDgoban[around] == oneDgoban[WEST_OF(around)]) { /* If the group extends to the direction */
	    if(FindLiberty(WEST_OF(around)) == TRUE) { /* check if somewhere in there is liberty */
	      return TRUE; /* Once we find a liberty we quit searching for more */
		}
	  }
	}
  }
  /* Look towards south */
  if(AT_SOUTH_EDGE(around) == FALSE) { /* If we are not at the edge */
    if(oneDhelp[SOUTH_OF(around)] == 0) { /* and the direction has not been searched yet */
	  if(IS_EMPTY(oneDgoban[SOUTH_OF(around)]) == TRUE) { /* If there is liberty */
	    return TRUE; /* the group has liberty */
	  }
	  oneDhelp[SOUTH_OF(around)] = 1; /* As we tested liberty already, we do not want to search here again */
	  if(oneDgoban[around] == oneDgoban[SOUTH_OF(around)]) { /* If the group extends to the direction */
	    if(FindLiberty(SOUTH_OF(around)) == TRUE) { /* check if somewhere in there is liberty */
	      return TRUE; /* Once we find a liberty we quit searching for more */
		}
	  }
	}
  }
  /* Look towards east */
  if(AT_EAST_EDGE(around) == FALSE) { /* If we are not at the edge */
    if(oneDhelp[EAST_OF(around)] == 0) { /* and the direction has not been searched yet */
	  if(IS_EMPTY(oneDgoban[EAST_OF(around)]) == TRUE) { /* If there is liberty */
	    return TRUE; /* the group has liberty */
	  }
	  oneDhelp[EAST_OF(around)] = 1; /* As we tested liberty already, we do not want to search here again */
	  if(oneDgoban[around] == oneDgoban[EAST_OF(around)]) { /* If the group extends to the direction */
	    if(FindLiberty(EAST_OF(around)) == TRUE) { /* check if somewhere in there is liberty */
	      return TRUE; /* Once we find a liberty we quit searching for more */
		}
	  }
	}
  }
  /* No liberty around here */
  return FALSE;
}

/* Removes solidly connected group of same color stones from the goban */
gint RemoveGroup(gint at, gboolean transparentCaptured)
{
  Stone color;
  gint sizeOfRemovedGroup = 1; /* Count here the stone itself */
  color = oneDgoban[at]; /* Remember which color group we are removing */
  if(IS_EMPTY(color) == TRUE) {
    return 0; /* There is no group to remove */
  }
  if(transparentCaptured == FALSE) { /* Normal stone capturing */
    oneDgoban[at] = STONE_NONE; /* Remove stone at position from the goban */
  }
  else { /* Transparent stone capturing */
    if(oneDgoban[at] == STONE_BLACK) {
      oneDgoban[at] = STONE_TRANSPARENT_BLACK;
    }
    if(oneDgoban[at] == STONE_WHITE) {
      oneDgoban[at] = STONE_TRANSPARENT_WHITE;
    }
  }
  /* Look towards north */
  if(AT_NORTH_EDGE(at) == FALSE) { /* If we are not at the edge */
    if(oneDgoban[NORTH_OF(at)] == color) { /* and group continues to the direction */
	  sizeOfRemovedGroup += RemoveGroup(NORTH_OF(at), transparentCaptured); /* it has to be removed also */
	}
  }
  /* Look towards west */
  if(AT_WEST_EDGE(at) == FALSE) { /* If we are not at the edge */
    if(oneDgoban[WEST_OF(at)] == color) { /* and group continues to the direction */
	  sizeOfRemovedGroup += RemoveGroup(WEST_OF(at), transparentCaptured); /* it has to be removed also */
	}
  }
  /* Look towards south */
  if(AT_SOUTH_EDGE(at) == FALSE) { /* If we are not at the edge */
    if(oneDgoban[SOUTH_OF(at)] == color) { /* and group continues to the direction */
	  sizeOfRemovedGroup += RemoveGroup(SOUTH_OF(at), transparentCaptured); /* it has to be removed also */
	}
  }
  /* Look towards east */
  if(AT_EAST_EDGE(at) == FALSE) { /* If we are not at the edge */
    if(oneDgoban[EAST_OF(at)] == color) { /* and group continues to the direction */
	  sizeOfRemovedGroup += RemoveGroup(EAST_OF(at), transparentCaptured); /* it has to be removed also */
	}
  }
#ifndef PRODUCTION_VERSION  
  msg2logGint("Removed group of size %d stones", LOGLEVEL_INFO, LOGSCOPE_GO_RULES,sizeOfRemovedGroup);
#endif
  return sizeOfRemovedGroup; /* Returns amount of removed stones */
}

/* Capture stones the played stone captures */
gint CaptureNeighbours(gint at, gboolean transparentCaptured)
{
  gint prisoners = 0;
  /* Look towards north */
  if(AT_NORTH_EDGE(at) == FALSE) { /* If we are not at the edge */
    if(oneDgoban[NORTH_OF(at)] == OPPONENT(oneDgoban[at])) { /* and there is opponent group at the direction */
	   if(HasLiberty(NORTH_OF(at)) == FALSE) { /* that has no liberties */
#ifndef PRODUCTION_VERSION  
      msg2log("Group at north has no liberties", LOGLEVEL_INFO, LOGSCOPE_GO_RULES);
#endif
	     prisoners += RemoveGroup(NORTH_OF(at), transparentCaptured); /* we remove it and count the removed stones */
	   }
	 }
  }
  /* Look towards west */
  if(AT_WEST_EDGE(at) == FALSE) { /* If we are not at the edge */
    if(oneDgoban[WEST_OF(at)] == OPPONENT(oneDgoban[at])) { /* and there is opponent group at the direction */
	   if(HasLiberty(WEST_OF(at)) == FALSE) { /* that has no liberties */
#ifndef PRODUCTION_VERSION  
      msg2log("Group at west has no liberties", LOGLEVEL_INFO, LOGSCOPE_GO_RULES);
#endif
	     prisoners += RemoveGroup(WEST_OF(at), transparentCaptured); /* we remove it and count the removed stones */
	   }
	 }
  }
  /* Look towards south */
  if(AT_SOUTH_EDGE(at) == FALSE) { /* If we are not at the edge */
    if(oneDgoban[SOUTH_OF(at)] == OPPONENT(oneDgoban[at])) { /* and there is opponent group at the direction */
	   if(HasLiberty(SOUTH_OF(at)) == FALSE) { /* that has no liberties */
#ifndef PRODUCTION_VERSION  
      msg2log("Group at south has no liberties", LOGLEVEL_INFO, LOGSCOPE_GO_RULES);
#endif
	     prisoners += RemoveGroup(SOUTH_OF(at), transparentCaptured); /* we remove it and count the removed stones */
	   }
    }
  }
  /* Look towards east */
  if(AT_EAST_EDGE(at) == FALSE) { /* If we are not at the edge */
    if(oneDgoban[EAST_OF(at)] == OPPONENT(oneDgoban[at])) { /* and there is opponent group at the direction */
	   if(HasLiberty(EAST_OF(at)) == FALSE) { /* that has no liberties */
#ifndef PRODUCTION_VERSION  
      msg2log("Group at east has no liberties", LOGLEVEL_INFO, LOGSCOPE_GO_RULES);
#endif
	     prisoners += RemoveGroup(EAST_OF(at), transparentCaptured); /* we remove it and count the removed stones */
	   }
	 }
  }
  return prisoners; /* Returns amount of removed stones */
}

/* Find ko if stone at parameter captured such */
gint FindKo(gint at)
{
  gint libertyCount = 0; /* Number of known liberties */
  gint koAt = -1;
  /* Look towards north */
  if(AT_NORTH_EDGE(at) == FALSE) { /* If we are not at the edge */
    if(oneDgoban[NORTH_OF(at)] == oneDgoban[at]) { /* and aside is another color of same color */
#ifndef PRODUCTION_VERSION  
     msg2log("Group continues to north -> not a ko", LOGLEVEL_INFO, LOGSCOPE_GO_RULES);
#endif
	  return -1; /* then this is not a ko */
	}
	if(IS_EMPTY(oneDgoban[NORTH_OF(at)]) == TRUE) { /* if there is a liberty */
	  libertyCount++; /* we count it */
	  koAt = NORTH_OF(at);
	}
  }
  /* Look towards west */
  if(AT_WEST_EDGE(at) == FALSE) { /* If we are not at the edge */
    if(oneDgoban[WEST_OF(at)] == oneDgoban[at]) { /* and aside is another color of same color */
#ifndef PRODUCTION_VERSION  
     msg2log("Group continues to west -> not a ko", LOGLEVEL_INFO, LOGSCOPE_GO_RULES);
#endif
	  return -1; /* then this is not a ko */
	}
	if(IS_EMPTY(oneDgoban[WEST_OF(at)]) == TRUE) { /* if there is a liberty */
	  libertyCount++; /* we count it */
	  koAt = WEST_OF(at);
	}
  }
  /* Look towards south */
  if(AT_SOUTH_EDGE(at) == FALSE) { /* If we are not at the edge */
    if(oneDgoban[SOUTH_OF(at)] == oneDgoban[at]) { /* and aside is another color of same color */
#ifndef PRODUCTION_VERSION  
     msg2log("Group continues to south -> not a ko", LOGLEVEL_INFO, LOGSCOPE_GO_RULES);
#endif
	  return -1; /* then this is not a ko */
	}
	if(IS_EMPTY(oneDgoban[SOUTH_OF(at)]) == TRUE) { /* if there is a liberty */
	  libertyCount++; /* we count it */
	  koAt = SOUTH_OF(at);
	}
  }
  /* Look towards east */
  if(AT_EAST_EDGE(at) == FALSE) { /* If we are not at the edge */
    if(oneDgoban[EAST_OF(at)] == oneDgoban[at]) { /* and aside is another color of same color */
#ifndef PRODUCTION_VERSION  
     msg2log("Group continues to east -> not a ko", LOGLEVEL_INFO, LOGSCOPE_GO_RULES);
#endif
	  return -1; /* then this is not a ko */
	}
	if(IS_EMPTY(oneDgoban[EAST_OF(at)]) == TRUE) { /* if there is a liberty */
	  libertyCount++; /* we count it */
	  koAt = EAST_OF(at);
	}
  }
  if(libertyCount == 1) { /* If there is exactly one liberty */
#ifndef PRODUCTION_VERSION  
    msg2log("Group has one liberty -> it is a ko", LOGLEVEL_INFO, LOGSCOPE_GO_RULES);
#endif
    return koAt; /* then it is a ko */
  }
#ifndef PRODUCTION_VERSION  
  msg2log("Group has more than one liberty -> not a ko", LOGLEVEL_INFO, LOGSCOPE_GO_RULES);
#endif
  return -1; /* else, there is no ko */
}

/* Makes move of the player in turn at the given position if the move is legal. Returns FALSE if move was not legal */
gboolean MakeMove(gint x, gint y, Stone color, MainView *main, gboolean transparentCaptured)
{
  gint prisoners = 0;
  /* Stone color = main->playerInTurn; */
  gint at = XY2LOCATION(x,y);
#ifndef PRODUCTION_VERSION  
  msg2logGint("MakeMove at %d", LOGLEVEL_INFO, LOGSCOPE_GO_RULES+LOGSCOPE_PATH,at);
#endif
  if((x<0) || (y<0) || (x>=goban_h_goban_size) || (y>=goban_h_goban_size)){
#ifndef PRODUCTION_VERSION  
    msg2log("Stone is not at goban", LOGLEVEL_INFO, LOGSCOPE_GO_RULES);
#endif
    return FALSE; /* Stone is not at goban */
  }
  if(IS_EMPTY(oneDgoban[at]) == FALSE) {
#ifndef PRODUCTION_VERSION  
    msg2log("The location is not empty", LOGLEVEL_INFO, LOGSCOPE_GO_RULES);
#endif
    return FALSE; /* The location is not empty */
  }
  if(main->koAt == at) {
#ifndef PRODUCTION_VERSION  
    msg2log("Ko cannot be retaken immediately", LOGLEVEL_INFO, LOGSCOPE_GO_RULES);
#endif
    return FALSE; /* Ko cannot be retaken immediately */
  }
  oneDgoban[at] = color; /* Play the stone for now */
  prisoners = CaptureNeighbours(at, transparentCaptured); /* Count and capture prisoners */
  if(prisoners == 0) { /* If stone capture nothing */
    if(HasLiberty(at) == FALSE) { /* and has no liberties */
	  /* If suicide is allowed:
	  opponentPrisoners = RemoveGroup(at);
	  SetPlayerInTurn(main);
	  return ?
	  */
	  oneDgoban[at] = STONE_NONE; /* Unplay the stone that would be suicide */
#ifndef PRODUCTION_VERSION  
     msg2log("Move is suicide", LOGLEVEL_INFO, LOGSCOPE_GO_RULES);
#endif
	  return FALSE; /* Move is suicide */
	}
  }
  if(prisoners == 1) { /* If exactly one stone was captured */
    main->koAt = FindKo(at); /* let's try to find if there is a ko */
  }
  else {
    main->koAt = -1; /* Reset information about ko */
  }
  if(prisoners > 0) {
    if(color == STONE_BLACK) { /* Black plays ... */
      main->capturedWhite += prisoners; /* ... capture white stones */
    }
    if(color == STONE_WHITE) { /* White plays ... */
      main->capturedBlack += prisoners; /* ... capture black stones */
    }
  }
  return TRUE; /* The play was legal */
}

/* Determine based on current position who is going to play next stone */
void SetPlayerInTurn(MainView* main)
{
  if(main->currentNode == NULL) {
    if(main->sgfPL != STONE_NONE) {
      main->playerInTurn = main->sgfPL;
      return;
    }
    if(main->sgfHA > 0) {
      main->playerInTurn = STONE_WHITE;
      return;
    }
    main->playerInTurn = STONE_BLACK;
    return;
  }
  main->playerInTurn = OPPONENT(main->currentNode->stone);
  return;
}

/* Try to play moves starting from parameter from under current position */
void PlayMoves( Node * from, MainView * mainview, gboolean stopAtIllegal)
{
  Node * node = NULL;
  Stone color;
  while(from != NULL) {
    color = OPPONENT(mainview->currentNode->stone);
    if(MakeMove(LOCATION2X(from->at),LOCATION2Y(from->at),color,mainview,FALSE) == TRUE) { /* Try to play the copied move */
      node = AddMove2GameTree(mainview->currentNode, mainview->child, LOCATION2X(from->at), LOCATION2Y(from->at), color, mainview->koAt); /* add it into game tree */
      mainview->currentNode = node; /* Set the position after the move to be the current position */
      set_active(mainview->currentNode); /* Set the active branches from current position to root of the game */
      SetPlayerInTurn(mainview); /* Change color of player in turn */
    }
    else { /* Move is illegal ... */
      if(stopAtIllegal == TRUE) { /* If we want to stop at illegal move */
        break; /* we don't do anything more */
      } /* else we just ignore the illegal move and try the next one */
    }
    from = get_active(from->child);
  }
}

/* Copy active line of play to a variation where given play has been moved to another location */
void CreateWhatIf( gint isX, gint isY, gint wereX, gint wereY, MainView * mainview)
{
  Pointlist * point = NULL;
  Stone color;
  gint isAt = 0, turn = 0;
  Node * node = NULL;
  Node * moved = NULL;
  isAt = XY2LOCATION(isX,isY);
  moved = mainview->currentNode; /* From current node */
  while(moved != NULL) { /* try to find the what if stone */
    if(moved->at == isAt) {
      break;
    }
    moved = moved->parent; /* towards root */
  }
  if(moved == NULL) { /* If such is not found */
    point = mainview->sgfAB; /* let's look the handicap stones */
    while(point != NULL) {
      if(point->at == isAt) {
        point->at = XY2LOCATION(wereX,wereY);
        return; /* Handicap stone has been moved */
      }
      point = point->next;
    }
    point = mainview->sgfAW; /* let's look the white setup stones */
    while(point != NULL) {
      if(point->at == isAt) {
        point->at = XY2LOCATION(wereX,wereY);
        return; /* White setup stone has been moved */
      }
      point = point->next;
    }
    return; /* do nothing */
  }
  turn = mainview->currentNode->nodeTurnNumber;
  color = moved->stone; /* After moving the stone to what if position the color does not change */
  mainview->currentNode = moved->parent; /* The what if stone is played at parent of the original stone */
  setCurrentPosition( mainview ); /* The position where to stones are played must be constructed from scratch */
  if(MakeMove(wereX,wereY,color,mainview,FALSE) == TRUE) { /* Try to play the what if move */
    node = AddMove2GameTree(mainview->currentNode, mainview->child, wereX, wereY, color, mainview->koAt); /* add it into game tree */
    mainview->currentNode = node; /* Set the position after the move to be the current position */
    set_active(mainview->currentNode); /* Set the active branches from current position to root of the game */
    SetPlayerInTurn(mainview); /* Change color of player in turn */
  }
  else {
    return; /* The what if move is not legal */
  }
  node = get_active(moved->child);
  PlayMoves( node, mainview, TRUE); /* Play all moves after the what if move */
  while(mainview->currentNode != NULL && mainview->currentNode->nodeTurnNumber != turn) { /* Find the same turn at what plaer was befor what if */
    mainview->currentNode = mainview->currentNode->parent;
  }
}

gint RotateLocation(gint at, gint orientation)
{
  /* 0 = no change in orientation, 1-3 = rotate, 4-7 = rotate and mirror */
  gint x = 0, y = 0, swap = 0;
  if((at == AT_IS_PASS) || (at == AT_IS_SETUP)) {
    return at; /* We do not want to rotate neither pass nor set up indicator */
  }
  x = LOCATION2X(at);
  y = LOCATION2Y(at);
  if((orientation&1) == 1) { /* 1,3,5,7 */
    x = goban_h_goban_size - x - 1;
  }
  if((orientation&2) == 2) { /* 2,3,6,7 */
    y = goban_h_goban_size - y - 1;
  }
  if((orientation == 1) || (orientation == 2) || (orientation == 4) || (orientation == 7)) { /* 1,2,4,7 */
    swap = x;
    x = y;
    y = swap;
  }
  return XY2LOCATION(x,y);
  /*
  x = LOCATION2Y(at);
  y = goban_h_goban_size - LOCATION2X(at) - 1;
  return XY2LOCATION(x,y); */
}

void RotateGoban(MainView * main, gint orientation)
{
  RotateList(main->sgfAB, orientation); /* Rotate handicap stones */
  RotateList(main->sgfAW, orientation);
  RotateList(main->sgfMA, orientation); /* Rotate markup */
  RotateList(main->sgfSQ, orientation);
  RotateList(main->sgfTR, orientation);
  RotateList(main->sgfCR, orientation);
  RotateLabels(main->sgfLB, orientation);
  RotateNode(main->child, orientation);
}

void RotateList(Pointlist * plist, gint orientation)
{
  while(plist != NULL) {
    plist->at = RotateLocation(plist->at, orientation);
    plist = plist->next;
  }
}

void RotateLabels(Labellist * llist, gint orientation)
{
  while(llist != NULL) {
    llist->at = RotateLocation(llist->at, orientation);
    llist = llist->next;
  }
}

void RotateNode(Node * node, gint orientation)
{
  if(node == NULL) {
    return;
  }
  RotateList(node->sgfMA, orientation);
  RotateList(node->sgfSQ, orientation);
  RotateList(node->sgfTR, orientation);
  RotateList(node->sgfCR, orientation);
  RotateList(node->sgfTB, orientation);
  RotateList(node->sgfTW, orientation);
  RotateList(node->sgfAB, orientation);
  RotateList(node->sgfAW, orientation);
  RotateList(node->sgfAE, orientation);
  RotateLabels(node->sgfLB, orientation);
  node->at = RotateLocation(node->at, orientation);
  RotateNode(node->child, orientation);
  if(node->sibling->nodeHeight > node->nodeHeight) {
    RotateNode(node->sibling, orientation);
  }
}

/* Clear territory markers out of stones and set own territory marker on stones unless there is opponent territory marker */
void SetTerritoryAtGroups(MainView * main)
{
  gint at = 0;

  if(main->currentNode == NULL) {
    return;
  }
  for(at = 0; at < goban_h_goban_size * goban_h_goban_size; at++) {
    switch(oneDgoban[at]) {
      case STONE_WHITE:
        if(IsPointInList(at, main->currentNode->sgfTB) == FALSE) {
          main->currentNode->sgfTW = AddPoint2List(at, main->currentNode->sgfTW);
        }
        break;
      case STONE_BLACK:
        if(IsPointInList(at, main->currentNode->sgfTW) == FALSE) {
          main->currentNode->sgfTB = AddPoint2List(at, main->currentNode->sgfTB);
        }
        break;
      default:
        main->currentNode->sgfTW = RemovePoint(main->currentNode->sgfTW, at);
        main->currentNode->sgfTB = RemovePoint(main->currentNode->sgfTB, at);
    }
  }
}

/* Clear territory markers at own stones */
void ClearTerritoryAtGroups(MainView * main)
{
  gint at = 0;

  if(main->currentNode == NULL) {
    return;
  }
  for(at = 0; at < goban_h_goban_size * goban_h_goban_size; at++) {
    switch(oneDgoban[at]) {
      case STONE_WHITE:
        main->currentNode->sgfTW = RemovePoint(main->currentNode->sgfTW, at);
        break;
      case STONE_BLACK:
        main->currentNode->sgfTB = RemovePoint(main->currentNode->sgfTB, at);
        break;
      default:
        break;
    }
  }
}

/* Returns TRUE if any point in the list is adjacent location to at */
gboolean IsNeighbour(gint at, Pointlist * list, Stone stone, Pointlist * deny)
{
  gint try = 0;
  
  if(AT_NORTH_EDGE(at) == FALSE) {
    try = NORTH_OF(at);
    if(oneDgoban[try] == stone) {
      if(IsPointInList(try,deny) == FALSE) {
        return TRUE;
      }
    }
    if(IsPointInList(try,list) == TRUE) {
      return TRUE;
    }
  }
  if(AT_SOUTH_EDGE(at) == FALSE) {
    try = SOUTH_OF(at);
    if(oneDgoban[try] == stone) {
      if(IsPointInList(try,deny) == FALSE) {
        return TRUE;
      }
    }
    if(IsPointInList(try,list) == TRUE) {
      return TRUE;
    }
  }
  if(AT_WEST_EDGE(at) == FALSE) {
    if(oneDgoban[WEST_OF(at)] == stone) {
      if(IsPointInList(WEST_OF(at),deny) == FALSE) {
        return TRUE;
      }
    }
    if(IsPointInList(WEST_OF(at),list) == TRUE) {
      return TRUE;
    }
  }
  if(AT_EAST_EDGE(at) == FALSE) {
    if(oneDgoban[EAST_OF(at)] == stone) {
      if(IsPointInList(EAST_OF(at),deny) == FALSE) {
        return TRUE;
      }
    }
    if(IsPointInList(EAST_OF(at),list) == TRUE) {
      return TRUE;
    }
  }
  return FALSE;
}

/* Returns TRUE if any point in the list is within 3x3 area centered by the at */
/* gboolean IsNear(gint at, Pointlist * list) replaced by: */
/* returns distance to nearest match within 3x3 area or 3 if no match as follows:
212
101
212
*/
gint HowNear(gint at, Pointlist * list)
{
  gint try = 0;
  
  if(IsPointInList(at,list) == TRUE) {
    return 0;
  }
  if(AT_WEST_EDGE(at) == FALSE) {
    if(IsPointInList(WEST_OF(at),list) == TRUE) {
      return 1;
    }
  }
  if(AT_EAST_EDGE(at) == FALSE) {
    if(IsPointInList(EAST_OF(at),list) == TRUE) {
      return 1;
    }
  }
  if(AT_SOUTH_EDGE(at) == FALSE) {
    try = SOUTH_OF(at);
    if(IsPointInList(try,list) == TRUE) {
      return 1;
    }
  }
  if(AT_NORTH_EDGE(at) == FALSE) {
    try = NORTH_OF(at);
    if(IsPointInList(try,list) == TRUE) {
      return 1;
    }
    if(AT_WEST_EDGE(try) == FALSE) {
      if(IsPointInList(WEST_OF(try),list) == TRUE) {
        return 2;
      }
    }
    if(AT_EAST_EDGE(try) == FALSE) {
      if(IsPointInList(EAST_OF(try),list) == TRUE) {
        return 2;
      }
    }
  }
  if(AT_SOUTH_EDGE(at) == FALSE) {
    try = SOUTH_OF(at);
    if(AT_WEST_EDGE(try) == FALSE) {
      if(IsPointInList(WEST_OF(try),list) == TRUE) {
        return 2;
      }
    }
    if(AT_EAST_EDGE(try) == FALSE) {
      if(IsPointInList(EAST_OF(try),list) == TRUE) {
        return 2;
      }
    }
  }
  return 3;
}

/* Expand territory of expanded list so that it would not overlap my own territory and does not come near to opponent territory */
Pointlist * ExpandTerritory(Pointlist * expandable, Pointlist * mine, Pointlist * opponent)
{
  Pointlist * result = NULL;
  gboolean candidate = FALSE;
  
  while(expandable != NULL) {
    candidate = FALSE;
    if(AT_NORTH_EDGE(expandable->at) == FALSE) {
      if(IsPointInList(NORTH_OF(expandable->at), mine) == FALSE) {
        if(HowNear(NORTH_OF(expandable->at), opponent) > 1) {
          result = AddPoint2List(NORTH_OF(expandable->at), result);
        }
      }
    }
    if(AT_SOUTH_EDGE(expandable->at) == FALSE) {
      if(IsPointInList(SOUTH_OF(expandable->at), mine) == FALSE) {
        if(HowNear(SOUTH_OF(expandable->at), opponent) > 1) {
          result = AddPoint2List(SOUTH_OF(expandable->at), result);
        }
      }
    }
    if(AT_WEST_EDGE(expandable->at) == FALSE) {
      if(IsPointInList(WEST_OF(expandable->at), mine) == FALSE) {
        if(HowNear(WEST_OF(expandable->at), opponent) > 1) {
          result = AddPoint2List(WEST_OF(expandable->at), result);
        }
      }
    }
    if(AT_EAST_EDGE(expandable->at) == FALSE) {
      if(IsPointInList(EAST_OF(expandable->at), mine) == FALSE) {
        if(HowNear(EAST_OF(expandable->at), opponent) > 1) {
          result = AddPoint2List(EAST_OF(expandable->at), result);
        }
      }
    }
    expandable = expandable->next;
  }
  return result;
}

void ExpandTerritories(Pointlist * first, Pointlist * second)
{
  Pointlist * result = NULL;
  Pointlist * swap = NULL;
  Pointlist * exp1st = NULL;
  Pointlist * exp2nd = NULL;
  Pointlist * last = NULL;
  gint stabileCount = 0;
  
  if(first == NULL) {
    return;
  }
  if(second == NULL) {
    return;
  }
  exp1st = first;
  exp2nd = second;
  while(stabileCount < 2) {
    result = ExpandTerritory(exp1st, first, second);
    last = first;
    while(last->next != NULL) {
      last = last->next;
    }
    last->next = result;
    exp1st = exp2nd;
    exp2nd = result;
    swap = first;
    first = second;
    second = swap;
    if(result == NULL) {
      stabileCount++;
    }
    else {
      stabileCount = 0;
    }
  }
}

void PessimisticExpand(Node * node)
{
  Pointlist * copyTB = NULL;
  Pointlist * copyTW = NULL;

  if(node == NULL) {
    return;
  }
  copyTB = CopyList(node->sgfTB); /* Copy black territory */
  copyTW = CopyList(node->sgfTW); /* Copy white territory */
  ExpandTerritories(node->sgfTB, node->sgfTW); /* Expand territories if black gets everything contested */
  ExpandTerritories(copyTW, copyTB); /* Expand territories if white gets everything contested */
  DestroyPointlist(node->sgfTB); /* Get rid of optimistic black territory estimate */
  DestroyPointlist(copyTW); /* Get rid of optimistic white territory estimate */
  node->sgfTB = copyTB; /* Set black territory estimate be the one that was optimistic for white */
}

void MarkGroupDead(MainView * main, gint at)
{
  if(main->currentNode == NULL) {
    return;
  }
  if(oneDgoban[at] == STONE_BLACK) {
    if(IsPointInList(at, main->currentNode->sgfTW) == TRUE) {
      return;
    }
    main->currentNode->sgfTW = AddPoint2List(at, main->currentNode->sgfTW);
  }
  else {
    if(IsPointInList(at, main->currentNode->sgfTB) == TRUE) {
      return;
    }
    main->currentNode->sgfTB = AddPoint2List(at, main->currentNode->sgfTB);
  }
  if(AT_NORTH_EDGE(at) == FALSE) {
    if(oneDgoban[at] == oneDgoban[NORTH_OF(at)]) {
      MarkGroupDead(main, NORTH_OF(at));
    }
  }
  if(AT_SOUTH_EDGE(at) == FALSE) {
    if(oneDgoban[at] == oneDgoban[SOUTH_OF(at)]) {
      MarkGroupDead(main, SOUTH_OF(at));
    }
  }
  if(AT_WEST_EDGE(at) == FALSE) {
    if(oneDgoban[at] == oneDgoban[WEST_OF(at)]) {
      MarkGroupDead(main, WEST_OF(at));
    }
  }
  if(AT_EAST_EDGE(at) == FALSE) {
    if(oneDgoban[at] == oneDgoban[EAST_OF(at)]) {
      MarkGroupDead(main, EAST_OF(at));
    }
  }
}

void MarkGroupAlive(MainView * main, gint at)
{
  if(main->currentNode == NULL) {
    return;
  }
  if(oneDgoban[at] == STONE_BLACK) {
    if(IsPointInList(at, main->currentNode->sgfTW) == FALSE) {
      return;
    }
    main->currentNode->sgfTW = RemovePoint(main->currentNode->sgfTW, at);
  }
  else {
    if(IsPointInList(at, main->currentNode->sgfTB) == FALSE) {
      return;
    }
    main->currentNode->sgfTB = RemovePoint(main->currentNode->sgfTB, at);
  }
  if(AT_NORTH_EDGE(at) == FALSE) {
    if(oneDgoban[at] == oneDgoban[NORTH_OF(at)]) {
      MarkGroupAlive(main, NORTH_OF(at));
    }
  }
  if(AT_SOUTH_EDGE(at) == FALSE) {
    if(oneDgoban[at] == oneDgoban[SOUTH_OF(at)]) {
      MarkGroupAlive(main, SOUTH_OF(at));
    }
  }
  if(AT_WEST_EDGE(at) == FALSE) {
    if(oneDgoban[at] == oneDgoban[WEST_OF(at)]) {
      MarkGroupAlive(main, WEST_OF(at));
    }
  }
  if(AT_EAST_EDGE(at) == FALSE) {
    if(oneDgoban[at] == oneDgoban[EAST_OF(at)]) {
      MarkGroupAlive(main, EAST_OF(at));
    }
  }
}

void SwapGroupStatus(MainView * main, gint at)
{
  if(main->currentNode == NULL) {
    return;
  }
  if(oneDgoban[at] == STONE_BLACK) {
    if(IsPointInList(at, main->currentNode->sgfTW) == TRUE) {
      MarkGroupAlive(main, at);
    }
    else {
      MarkGroupDead(main, at);
    }
  }
  if(oneDgoban[at] == STONE_WHITE) {
    if(IsPointInList(at, main->currentNode->sgfTB) == TRUE) {
      MarkGroupAlive(main, at);
    }
    else {
      MarkGroupDead(main, at);
    }
  }
  SetTerritoryAtGroups(main);
  PessimisticExpand(main->currentNode);
  /*
  if(main->currentNode->stone == STONE_BLACK) {
    ExpandTerritories(main->currentNode->sgfTW, main->currentNode->sgfTB);
  }
  else {
    ExpandTerritories(main->currentNode->sgfTB, main->currentNode->sgfTW);
  } */
  ClearTerritoryAtGroups(main);
}

void MarkDeadListedGroupsOfColor(MainView * main, Pointlist * lethalArea, Stone color)
{
  while(lethalArea != NULL) {
    if(oneDgoban[lethalArea->at] == color) {
      MarkGroupDead(main, lethalArea->at);
    }
    lethalArea = lethalArea->next;
  }
}

void DragLabellist(Labellist * llist, gint offset, MainView * main)
{
  gint x = 0, y = 0;
  while(llist != NULL) {
    x = LOCATION2X(llist->at);
    y = LOCATION2Y(llist->at);
    if((x >= main->select_left) && (x <= main->select_right) && (y >= main->select_top) && (y <= main->select_bottom)) {
      llist->at -= offset;
    }
    llist = llist->next;
  }
}

void DragPointlist(Pointlist * plist, gint offset, MainView * main)
{
  gint x = 0, y = 0;
  while(plist != NULL) {
    x = LOCATION2X(plist->at);
    y = LOCATION2Y(plist->at);
    if((x >= main->select_left) && (x <= main->select_right) && (y >= main->select_top) && (y <= main->select_bottom)) {
      plist->at -= offset;
    }
    plist = plist->next;
  }
}

void DragArea(MainView * main)
{
  gint offset = 0, x = 0, y = 0;
  Node * node = NULL;
  
  offset = main->mouseGobanClickX - main->mouseGobanCurrentX + (main->sgfSZ * (main->mouseGobanClickY - main->mouseGobanCurrentY));
  node = main->currentNode;
  while(node != NULL) {
    x = LOCATION2X(node->at);
    y = LOCATION2Y(node->at);
    if((x >= main->select_left) && (x <= main->select_right) && (y >= main->select_top) && (y <= main->select_bottom)) {
      node->at -= offset;
    }
    DragPointlist(node->sgfAB, offset, main);
    DragPointlist(node->sgfAW, offset, main);
    DragPointlist(node->sgfAE, offset, main);
    DragPointlist(node->sgfTR, offset, main);
    DragPointlist(node->sgfCR, offset, main);
    DragPointlist(node->sgfSQ, offset, main);
    DragPointlist(node->sgfMA, offset, main);
    DragLabellist(node->sgfLB, offset, main);
    node = node->parent;
  }
  DragPointlist(main->sgfAB, offset, main);
  DragPointlist(main->sgfAW, offset, main);
  DragPointlist(main->sgfTR, offset, main);
  DragPointlist(main->sgfCR, offset, main);
  DragPointlist(main->sgfSQ, offset, main);
  DragPointlist(main->sgfMA, offset, main);
  DragLabellist(main->sgfLB, offset, main);
}
