/*
 * This file is part of goban770
 *
 * Copyright (C) 2006,2007 Jarmo Ahosola.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <gtk/gtk.h>
#include <ui/interface.h>
#include <game/gametree.h>
#include <game/notes.h>
#include <game/goban.h>
#include <utils/log.h>
#include <libintl.h>
#include <stdlib.h>

/* privates */
static gint gametree_h_treeWidth = 0;
static gint gametree_h_treeHeight = 0;
static gint gametree_h_treeNodes = 0;
static gint gametree_h_treeLeafs = 0;

void GetTreeMeasures(gint * width, gint * height, gint * nodes, gint * leafs)
{
  width[0] = gametree_h_treeWidth;
  height[0] = gametree_h_treeHeight;
  nodes[0] = gametree_h_treeNodes;
  leafs[0] = gametree_h_treeLeafs;
}

Pointlist * NewPoint(gint point)
{
  /* Zero memory with g_new0 */
  Pointlist * result = g_new0( Pointlist, 1 );
  if(result == NULL) {
#ifndef PRODUCTION_VERSION  
    msg2log("Reserving Pointlist fails", LOGLEVEL_ERROR, LOGSCOPE_MEMORY);
#endif
    return NULL;
  }
#ifndef PRODUCTION_VERSION  
  LogReserveMemory("Pointlist");
#endif
  result->next = NULL;
  result->at = point;
  return result;
}

/* Add point last in the list in case it is not in the list already. Returns root of the new list */
Pointlist * AddPoint2List(gint at, Pointlist * list)
{
  Pointlist * result = NULL;
  
  if(list == NULL) {
    return NewPoint(at);
  }
  result = list;
  while(list->next != NULL) {
    if(list->at == at) {
      return result;
    }
    list = list->next;
  }
  if(list->at != at) {
    list->next = NewPoint(at);
  }
  return result;
}

/* Copy list to unique list of points */
Pointlist * CopyList(Pointlist * list)
{
  Pointlist * result = NULL;

  while(list != NULL) {
    result = AddPoint2List(list->at, result);
    list = list->next;
  }
  return result;
}

Labellist * NewLabel(gint point)
{
  gint i = 0;
  /* Zero memory with g_new0 */
  Labellist * result = g_new0( Labellist, 1 );
  if(result == NULL) {
#ifndef PRODUCTION_VERSION  
    msg2log("Reserving Labellist fails", LOGLEVEL_ERROR, LOGSCOPE_MEMORY);
#endif
    return NULL;
  }
#ifndef PRODUCTION_VERSION  
  LogReserveMemory("Labellist");
#endif
  result->next = NULL;
  result->at = point;
  for(i = 0; i <= MAX_LABEL_LENGTH; i++) {
    result->label[i] = '\0';
  }
  return result;
}

void DestroyPointlist(Pointlist * plist)
{
  if(plist == NULL) {
    return;
  }
  DestroyPointlist(plist->next);
  g_free(plist);
#ifndef PRODUCTION_VERSION  
  LogReleaseMemory("Pointlist");
#endif
}

void DestroyLabellist(Labellist * llist)
{
  if(llist == NULL) {
    return;
  }
  DestroyLabellist(llist->next);
  g_free(llist);
#ifndef PRODUCTION_VERSION  
  LogReleaseMemory("Labellist");
#endif
}

/* Creates and initialized new gametree node */
/* Node * NewNode( Node * parent, gint x, gint y, Stone stone, Node * bigSister, gint playAt, gint koAt ) */
Node * NewNode( Node * parent, Stone stone, Node * bigSister, gint playAt, gint koAt )
{
	 Node * tmp;
    /* Zero memory with g_new0 */
    Node * result = g_new0( Node, 1 );
    
  if(result == NULL) {
#ifndef PRODUCTION_VERSION  
    msg2log("Reserving Node fails", LOGLEVEL_ERROR, LOGSCOPE_MEMORY);
#endif
    return NULL;
  }
#ifndef PRODUCTION_VERSION  
  LogReserveMemory("Node");
#endif
    result->parent = parent;
    if(parent != NULL) {
      if(parent->child == NULL) {
        parent->child = result;
      }
    }
    result->child = NULL;
    if(bigSister == NULL) {
      result->sibling = result;
      result->activeBranch = TRUE; /* If there are no sisters, then this must be active as it is the only branch */
    }
    else {
      tmp = bigSister->sibling;
      bigSister->sibling = result;
      result->sibling = tmp;
      result->activeBranch = FALSE; /* If there are any sisters, one of them is active already, so this may not be */
    }
    if(parent == NULL) {
	   result->nodeTurnNumber = 1;
	   result->nodeDrawX = 1;
    }
    else {
	   result->nodeTurnNumber = parent->nodeTurnNumber + 1;
	   result->nodeDrawX = parent->nodeDrawX + 1;
    }
    result->nodeHeight = -1;
    result->sgfVx2 = 0;
    result->sgfC = NULL;
    result->unsupportedSgf = NULL;
    /* result->x = x;
    result->y = y; */
    result->at = playAt; /* XY2Location(x,y); */
    result->stone = stone;
	 result->sgfTB = NULL;
	 result->sgfTW = NULL;
	 result->sgfAB = NULL;
	 result->sgfAW = NULL;
	 result->sgfAE = NULL;
	 result->sgfMA = NULL;
	 result->sgfTR = NULL;
	 result->sgfCR = NULL;
	 /* Cgoban2 style last move marking 
	 if(playAt > -1) {
	   result->sgfCR = NewPoint(playAt);
	 } */
	 if(koAt != -1) {
	   result->sgfSQ = NewPoint(koAt);
	 }
	 else {
	   result->sgfSQ = NULL;
	 }
	 result->sgfLB = NULL;
	 result->underTheStones = FALSE;
    return result;
}

/* clean up the allocated memory */
Node * node_destroy(  Node * node )
{
    Node * bigSister;
    Node * replacer;
    gint familySize = 0;
        
    if(node->child != NULL) {
      kill_family(node->child); /* Kill all descendants of the node */
      node->child = NULL;
    }
    bigSister = node;
    while (bigSister->sibling != NULL) {
      if(bigSister->sibling == node) {
        break; /* Found big sister */
      }
      bigSister = bigSister->sibling;
      familySize++;
      if(familySize == 0) {
#ifndef PRODUCTION_VERSION  
        msg2log("Corrupted family ties at node_destroy", LOGLEVEL_ERROR, LOGSCOPE_LOOP);
#endif
        return NULL; /* corrupted family ties */
      }
    }
    if(bigSister == NULL) {
#ifndef PRODUCTION_VERSION  
      msg2log("Deny children of parent at node_destroy", LOGLEVEL_ERROR, LOGSCOPE_MEMORY);
#endif
      return NULL; /* sister does not acknowledge existance of children of her parent */ 
    }
    bigSister->sibling = node->sibling; /* Expell node from family */
    if(node->activeBranch == TRUE) { /* If destroyed node was active */
      node->sibling->activeBranch = TRUE; /* it has to give it's active status to it's sibling */
    }
    if(node->parent != NULL) {
      if(node->parent->child == node) { /* If this is the first child */
        node->parent->child = node->sibling; /* Make sibling of this to be first child */
      }
      if(node->parent->child == node) { /* If that was self */
        node->parent->child = NULL; /* Then set parent to have no children */
      }
      replacer = node->parent->child;
    }
    else {
      replacer = node->sibling;
      if(replacer == node) {
        replacer = NULL;
      }
    }
    /* MISSING: memory release for usupported tags. What is the correct way to release that memory? */
    DestroyNote(node->sgfC);
	 DestroyPointlist(node->sgfTB);
	 DestroyPointlist(node->sgfTW);
	 DestroyPointlist(node->sgfAB);
	 DestroyPointlist(node->sgfAW);
	 DestroyPointlist(node->sgfAE);
	 DestroyPointlist(node->sgfMA);
	 DestroyPointlist(node->sgfTR);
	 DestroyPointlist(node->sgfCR);
	 DestroyPointlist(node->sgfSQ);
	 DestroyLabellist(node->sgfLB);
    g_free( node );
#ifndef PRODUCTION_VERSION  
  LogReleaseMemory("Node");
#endif
    return replacer;
}

/* destroy all sisters, their children, own children and self */
void kill_family(  Node * node )
{
    if(node == NULL) return;
    while(node != node->sibling) {
      node_destroy(node->sibling);
    }
    node_destroy(node);
}

Node * back_in_tree( Node * current )
{
    if(current == NULL) return NULL;
    return current->parent;
}

Node * forward_in_tree( Node * current, Node * childOfRoot )
{
    Node * child;
    if(current == NULL) {
      child = childOfRoot;
    }
    else {
      if(current->child == NULL) {
        return current;
      }
      child = current->child;
    }
    return get_active( child );
}

Node * upper_branch_in_tree( Node * node )
{
	 Node * bigSister;
    Node * current;
	 if(node == NULL) {
#ifndef PRODUCTION_VERSION  
      msg2log("Node is NULL at upper_branch_in_tree", LOGLEVEL_WARNING, LOGSCOPE_MEMORY);
#endif
	   return NULL;
	 }
    current = get_active(node);
	 if(current == NULL) {
#ifndef PRODUCTION_VERSION  
      msg2log("Current is NULL at upper_branch_in_tree", LOGLEVEL_WARNING, LOGSCOPE_MEMORY);
#endif
	   return NULL;
	 }
	 bigSister = get_bigsister( current );
	 if(bigSister == NULL) {
#ifndef PRODUCTION_VERSION  
      msg2log("Corrupted family ties at upper_branch_in_tree", LOGLEVEL_ERROR, LOGSCOPE_MEMORY);
#endif
	   return NULL;
	 }
	 /* Must be in order below in case of only child */
	 current->activeBranch = FALSE; /* deselect old active branch */
	 bigSister->activeBranch = TRUE; /* select new active branch */
	 return bigSister;
}

Node * lower_branch_in_tree( Node * node )
{
    Node * current;
	 if(node == NULL) {
#ifndef PRODUCTION_VERSION  
      msg2log("Node is NULL at lower_branch_in_tree", LOGLEVEL_WARNING, LOGSCOPE_MEMORY);
#endif
	   return NULL;
	 }
    current = get_active(node);
	 if(current == NULL) {
#ifndef PRODUCTION_VERSION  
      msg2log("Current is NULL at lower_branch_in_tree", LOGLEVEL_WARNING, LOGSCOPE_MEMORY);
#endif
	   return NULL;
	 }
	 if(current->sibling == NULL) {
#ifndef PRODUCTION_VERSION  
      msg2log("Corrupted family ties at lower_branch_in_tree", LOGLEVEL_ERROR, LOGSCOPE_MEMORY);
#endif
	   return NULL;
	 }
	 /* Must be in order below in case of only child */
	 current->activeBranch = FALSE; /* deselect old active branch */	 
	 current->sibling->activeBranch = TRUE; /* select new active branch */
	 return current->sibling;
}

/* Return sibling or self if such has stone at x,y */
Node * get_at_xy( Node * sister, gint x, gint y)
{
    Node * tmp;
    gint at = 0;
    if(x < 0) {
      at = y;
    }
    else {
      at = XY2Location(x,y);
    }
    if(sister == NULL) return NULL;
    tmp = sister;
    do{
      if(tmp->at == at) return tmp;
      tmp = tmp->sibling;
    } while(tmp != sister);
    return NULL;
}

/* Return the sibling with active branch */ 
Node * get_active( Node * node ) {
    while(node != NULL && node->activeBranch == FALSE) {
      node = node->sibling;
    }
    return node;
}

/* Set first child as active branch for all nodes in the tree */
void ResetActive( Node * firstChild )
{
  Node * node = NULL;
  if(firstChild == NULL) {
    return;
  }
  firstChild->activeBranch = TRUE;
  ResetActive(firstChild->child);
  node = firstChild->sibling;
  while(node != firstChild) {
    node->activeBranch = FALSE;
    ResetActive(node->child);
    node = node->sibling;
  }
}

/* Set the given node to be on active branch */
void set_active( Node * node )
{
    Node * previousActive;
    if(node == NULL) {
      return;
    }
    if(node->activeBranch == TRUE) {
      set_active(node->parent);
      return;
    }
    previousActive = get_active(node);
    if(previousActive != NULL) {
      previousActive->activeBranch = FALSE;
    }
    else {
#ifndef PRODUCTION_VERSION  
      msg2log("No previous active found at set_active", LOGLEVEL_WARNING, LOGSCOPE_MEMORY);
#endif
    }
    node->activeBranch = TRUE;
    set_active(node->parent);
}

/* Return node which sibling pointer points to the given node */
Node * get_bigsister( Node * node )
{
    Node * tmp;
    tmp = node;
    while(tmp != NULL && tmp->sibling != node) {
      tmp = tmp->sibling;
    }
    return tmp;
}

Node * AddNonmove2GameTree(Node * parent, Node * childOfRoot, gint nonstone, Stone color)
{
  Node * node;
  if(childOfRoot == NULL) { /* First move of the game */
    node = NewNode( NULL, color, NULL, nonstone, -1 );
    return node; /* Return node into which first move of the game is made */
  }
  if(parent == NULL) { /* In case we were at the root of the game */
    node = get_at_xy( childOfRoot, nonstone, nonstone); /* Try to find existing gametree branch for the move */
  }
  else { /* In case we were not at the root of the game */
    node = get_at_xy( parent->child, nonstone, nonstone); /* Try to find existing gametree branch for the move */
  }
  if(node == NULL) { /* If that fails add a new node for the move */
#ifndef PRODUCTION_VERSION  
    msg2log("Add a new node for the move", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
    if(parent == NULL) {
      node = NewNode( parent, color, get_bigsister( childOfRoot ), nonstone, -1 );
    }
    else {
      node = NewNode( parent, color, get_bigsister( parent->child ), nonstone, -1 );
    }
  }
  return node; /* Return node into which move is made */
}

Node * AddMove2GameTree(Node * parent, Node * childOfRoot, gint x, gint y, Stone color, gint koAt)
{
  Node * node;
  if(childOfRoot == NULL) { /* First move of the game */
#ifndef PRODUCTION_VERSION  
    msg2log("First move of the game", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
    node = NewNode( NULL, color, NULL, XY2Location(x,y), -1 );
#ifndef PRODUCTION_VERSION  
    msg2log("Exit make_move", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
    return node; /* Return node into which first move of the game is made */
  }
  if(parent == NULL) { /* In case we were at the root of the game */
    node = get_at_xy( childOfRoot, x, y); /* Try to find existing gametree branch for the move */
  }
  else { /* In case we were not at the root of the game */
    node = get_at_xy( parent->child, x, y); /* Try to find existing gametree branch for the move */
  }
  if(node == NULL) { /* If that fails add a new node for the move */
#ifndef PRODUCTION_VERSION  
    msg2log("Add a new node for the move", LOGLEVEL_INFO, LOGSCOPE_NONE);
#endif
    if(parent == NULL) {
      node = NewNode( parent, color, get_bigsister( childOfRoot ), XY2Location(x,y), koAt );
    }
    else {
      node = NewNode( parent, color, get_bigsister( parent->child ), XY2Location(x,y), koAt );
    }
  }
#ifndef PRODUCTION_VERSION  
  msg2log("Exit make_move", LOGLEVEL_INFO, LOGSCOPE_PATH);
#endif
  return node; /* Return node into which move is made */
}

static gint levelHeight[MAX_TREELEVELS];

/* Define drawing height for node in game tree
  node is node which height we try to determine
  firstChild is the first child of the family, the one that is on the local main variation
  lowerBound is minimum height for the node. This is horizontal line
  upperBound is recommended upper line. This is diagonal line
  The goal is to make a branch so that
  1) First node height after branch parent >= branch parent height
  2) After first node, N next nodes height is height[n] = height[n-1]+1
  3) After Nth node height of all nodes are height[n] = height[n-1]
  4) Node must have bigger height than all branches above it
  5) With restrictions above, last node of branch has MIN(possible height)
*/
gint NodeHeight(Node * node, Node * firstChild, gint treeLevel, gint lowerBound, gint upperBound)
{
  gint ch = 0; /* Child height */
  gint h = levelHeight[treeLevel] + 1; /* First, try to set the drawing height to the smallest available height for the tree level. */
  if(h < lowerBound) { /* If that is smaller than lowerBound, */
    h = lowerBound; /* use lowerBound instead so that branch do not start to go upwards. */
  }
  if(upperBound < h) { /* If determined height is bigger than upperBound, */
    upperBound = h; /* the upperBound has to be scretched to fit the needed height. */
  }
  if(node->child != NULL) { /* If node is not branch terminal leaf, */
    ch = NodeHeight(node->child, node->child, treeLevel+1, h, upperBound+1); /* the height of the remaining branch has to be determined. */
    if(ch > h) { /* If child need to have higher height, */
      h = ch - 1; /* The parent is moved closer the child so that branch does not has steap level changes. */
      if(h < upperBound) { /* If parent is not set to it's maximum height */
        h = ch; /* Move it to the same level with the child */
      }
	 }
  }
  if(h > levelHeight[treeLevel]) { /* If space has not yet been reserved */  
    levelHeight[treeLevel] = h; /* Let's reserve place for this node. */
  }
  if(treeLevel > 0) {
    if(levelHeight[treeLevel - 1] < h) {
      levelHeight[treeLevel - 1] = h - 1; /* Let's reserve place for line from stone to its parent */
    }
  }
  if(node->sibling != firstChild) { /* If node has sister which height has not been determined, */
    NodeHeight(node->sibling, firstChild, treeLevel, h+1, h+1); /* try to position it at next available height. */
  }
  node->nodeHeight = h; /* Store the calculated height for the node. */
  return h; /* Return height of the node so that earlier branch can be adjusted if needs to. */
}

void MakeTreeLayout(Node * childOfRoot) {
  gint i;
  for(i=0;i<MAX_TREELEVELS;i++) { /* Initialise all tree levels */
    levelHeight[i] = -1; /* to have height -1 as biggest occupied height */
  }
  if(childOfRoot != NULL) {
    NodeHeight(childOfRoot, childOfRoot, 0, 0, 0); /* Determine position for child of root, it's all descendants, it's sisters and their descendants. */
  }
  UpdateTreeDimensions(childOfRoot);
}



gint GetTreeWidth()
{
  return gametree_h_treeWidth;
}

gint GetTreeHeight()
{
  return gametree_h_treeHeight;
}

void UpdateTreeDimensions(Node * node)
{
  gametree_h_treeWidth = 1;
  gametree_h_treeHeight = 1;
  gametree_h_treeNodes = 0;
  gametree_h_treeLeafs = 0;
  MeasureTree(node);
}

/* Calculate dimensions of a game tree */
void MeasureTree(Node * node)
{
  Node * firstChild = node;
  if(node == NULL) {
    return;
  }
  do {
    if(node->child != NULL) {
      MeasureTree(node->child);
    }
    else {
      gametree_h_treeLeafs++;
    }
    if(node->nodeTurnNumber >= gametree_h_treeWidth) {
      gametree_h_treeWidth = node->nodeTurnNumber + 1;
    }
    if(node->nodeHeight >= gametree_h_treeHeight) {
      gametree_h_treeHeight = node->nodeHeight + 1;
    }
    node = node->sibling;
    gametree_h_treeNodes++;
  } while(node != firstChild);
}

/* Search for node with given tree coordinates */
Node * GetNodeAt(Node * node, gint height, gint turn, gboolean fullTree)
{
  Node * result = NULL;
#ifndef PRODUCTION_VERSION  
  msg2log("GetNodeAt", LOGLEVEL_INFO, LOGSCOPE_PATH | LOGSCOPE_GAMETREE);
#endif
  if(node == NULL) {
#ifndef PRODUCTION_VERSION  
  msg2log("node = NULL", LOGLEVEL_WARNING, LOGSCOPE_PATH | LOGSCOPE_GAMETREE);
#endif
    return NULL;
  }
  /*
  if(node->nodeHeight > height) {
#ifndef PRODUCTION_VERSION  
  msg2log("No more height investigated", LOGLEVEL_INFO, LOGSCOPE_GAMETREE);
#endif
    return NULL;
  } */
  if(node->nodeDrawX > turn) {
#ifndef PRODUCTION_VERSION  
  msg2logGint("No more turns investigated (%d)", LOGLEVEL_INFO, LOGSCOPE_GAMETREE, node->nodeDrawX);
#endif
    return NULL;
  }
  if(node->nodeHeight == height && node->nodeDrawX == turn) {
#ifndef PRODUCTION_VERSION  
  msg2log("Node at coordinates found", LOGLEVEL_INFO, LOGSCOPE_GAMETREE);
#endif
    return node;
  }
  if((fullTree == TRUE) || (node->activeBranch == TRUE)) {
    result = GetNodeAt(node->child, height, turn, fullTree);
    if(result != NULL) {
      return result;
    }
  }
  if(node->sibling->nodeHeight > node->nodeHeight) {
#ifndef PRODUCTION_VERSION  
  msg2logGint("Sibling branch investigated at height %d", LOGLEVEL_INFO, LOGSCOPE_GAMETREE, node->sibling->nodeHeight);
#endif
    return GetNodeAt(node->sibling, height, turn, fullTree);
  }
  return NULL;
}

void MoveBranchTo(Node * what,Node * where) {
  Node * aid = NULL;
  aid = where;
  while(aid->sibling != what) {
    aid = aid->sibling;
  }
  aid->sibling = what->sibling; /* Take moved branch out of family */
  if(what->nodeHeight > where->nodeHeight) { /* Move upwards */
    if(where->parent->child == where) { /* If branch is moved to the upmost */
      what->parent->child = what; /* it is going to be parent's first child */
    }
    while(aid->sibling != where) {
      aid = aid->sibling;
    }
    what->sibling = where; /* Connect moved branch above position to where it is moved */
    aid->sibling = what; /* Connect position above new location to the moved branch */
  }
  else { /* Move downwards */
    if(what->parent->child == what) { /* If branch was he upmost */
      what->parent->child = what->sibling; /* The next child is going to be parent's first child */
    }
    what->sibling = where->sibling; /* Connect moved branch above position to where it is moved */
    where->sibling = what; /* Connect position above new location to the moved branch */
  }
}

Node * LiftBranchUpwards(Node * node, gint height) {
  if(node == NULL) {
    return NULL;
  }
  while (node->parent != NULL) {
    if(node->parent->nodeHeight < height) { /* If parent is above target height stop searching */
      return NULL;
    }
    node->parent->child = node; /* Set this node to be first child of it's parent */
    node = node->parent; /* Do same for all ancestors */
  }
  if(height == 0) { /* If lifting at the main line */
    return node; /* Return new child of root */
  }
  return NULL;
}

void MovePoint(Pointlist * point, gint fromAt, gint toAt) {
  while(point != NULL) {
    if(point->at == fromAt) {
      point->at = toAt;
    }
    point = point->next;
  }
}

gboolean IsLabelInList(gint at, Labellist * llist) {
  if(llist == NULL) {
    return FALSE;
  }
  if(llist->at == at) {
    return TRUE;
  }
  return IsLabelInList(at,llist->next);
}

gboolean IsPointInList(gint at, Pointlist * point) {
  if(point == NULL) {
    return FALSE;
  }
  if(point->at == at) {
    return TRUE;
  }
  return IsPointInList(at,point->next);
}

Pointlist * RemovePoint(Pointlist * point, gint at) {
  Pointlist * result = NULL;
  if(point == NULL) {
    return NULL;
  }
  point->next = RemovePoint(point->next, at);
  if(point->at == at) {
    result = point->next;
    g_free(point);
    return result;
  }
  return point;
}

/* Return closest ancestor node which location is at given co-ordinaes, or NULL if such is not found */ 
Node * FindAncestorAtXY( Node * descendant , gint x, gint y)
{
  gint at = 0;
  at = XY2Location(x,y);
  while(descendant != NULL) { /* Search upto root */
    if(descendant->at == at) { /* if location is wanted */
      break; /* quit searching */
    }
    descendant = descendant->parent; /* else look for parent */
  }
  return descendant; /* and return what is found */
}

/* Given node from which move is searched, searches last match before given move that is not under the stones play */
/* Returns TRUE if manages to move stone, FALSE othervise */
gboolean MoveStoneInHistory(Node * node,gint fromX,gint fromY,gint toX,gint toY) {
  gint fromAt = 0, toAt = 0;
  fromAt = XY2Location(fromX,fromY);
  toAt = XY2Location(toX,toY);
  if(node == NULL) {
#ifndef PRODUCTION_VERSION  
  msg2log("MoveStoneInHistory with NULL", LOGLEVEL_ERROR, LOGSCOPE_GAMETREE);
#endif
    return FALSE;
  }
  /*
  while((node->child != NULL) && (node->at != fromAt)) {
    node = node->child;
    while(node->activeBranch == FALSE) {
      node = node->sibling;
    }
  }
  */
  while(node != NULL) { /* Up to root of the game */
    if((node->at == fromAt) && (node->underTheStones == FALSE)) { /* If we are at the node of the moved stone */
      node->at = toAt; /* Put the stone in the new wanted position */
      /* Let's remove all markings from the new location */
      node->sgfCR = RemovePoint(node->sgfCR, toAt);
      node->sgfSQ = RemovePoint(node->sgfSQ, toAt);
      node->sgfTR = RemovePoint(node->sgfTR, toAt);
      node->sgfMA = RemovePoint(node->sgfMA, toAt);
      /* Let's move all markings with the stone */
      MovePoint(node->sgfCR, fromAt, toAt);
      MovePoint(node->sgfSQ, fromAt, toAt);
      MovePoint(node->sgfTR, fromAt, toAt);
      MovePoint(node->sgfMA, fromAt, toAt);
      return TRUE;
    }
    node = node->parent;
  }
#ifndef PRODUCTION_VERSION  
  msg2log("MoveStoneInHistory stone not found", LOGLEVEL_ERROR, LOGSCOPE_GAMETREE);
#endif
  return FALSE;
}

void CircleLastPlayed(Node * node)
{
  Pointlist * point;
  if(node == NULL) {
    return;
  }
  CircleLastPlayed(node->child);
  if(node->sibling->nodeHeight > node->nodeHeight) {
    CircleLastPlayed(node->sibling);
  }
  point = node->sgfCR;
  while(point != NULL) {
    if(point->at == node->at) {
      return;
    }
    point = point->next;
  }
  point = node->sgfCR;
  node->sgfCR = NewPoint(node->at);
  if(node->sgfCR != NULL) {
    node->sgfCR->next = point;
  }
}

void RemoveCustomMarkings(Node * node)
{
  if(node == NULL) {
    return;
  }
  RemoveCustomMarkings(node->child);
  if(node->sibling->nodeHeight > node->nodeHeight) {
    RemoveCustomMarkings(node->sibling);
  }
  if(node->sgfC == NULL) {
	 DestroyPointlist(node->sgfTB);
	 node->sgfTB = NULL;
	 DestroyPointlist(node->sgfTW);
	 node->sgfTW = NULL;
	 DestroyPointlist(node->sgfMA);
	 node->sgfMA = NULL;
	 DestroyPointlist(node->sgfTR);
	 node->sgfTR = NULL;
	 DestroyPointlist(node->sgfCR);
	 node->sgfCR = NULL;
	 DestroyPointlist(node->sgfSQ);
	 node->sgfSQ = NULL;
	 DestroyLabellist(node->sgfLB);
	 node->sgfLB = NULL;
  }
}

Node * SelectBestLineOfPlay(Node * node, gint branchCount)
{
  Node * bestFound = NULL;
  Node * firstChild = node;
  Node * subresult = NULL;
  Node * newActive = NULL;
  
  if(node == NULL) {
    return NULL;
  }
  do {
    node->activeBranch = FALSE; /* Clear old active branch indication */
    if(node->child == NULL) {
      subresult = node;
    }
    else {
      if(node->child == node->child->sibling) {
        subresult = SelectBestLineOfPlay(node->child, branchCount);
      }
      else {
        subresult = SelectBestLineOfPlay(node->child, branchCount + 1);
      }
    }
    if(bestFound == NULL) { /* If nothing has been found before */
      bestFound = subresult; /* the found is then new best */
      newActive = node;
    }
    else {
      if((bestFound->stone != subresult->stone) && (subresult->stone == node->stone)) { /* If previous best is better for opponent and new found is better for self */
        bestFound = subresult; /* the found is then new best */
        newActive = node;
      }
      if((bestFound->stone == subresult->stone) && (subresult->stone == node->stone)) { /* If previous best and new found are good for self */
        /* Here is missing condition to pick brach with less variations */
        if (subresult->nodeTurnNumber < bestFound->nodeTurnNumber) { /* and new found has less potential for reading error */
          bestFound = subresult; /* the found is then new best */
          newActive = node;
        }
      }
      if((bestFound->stone == subresult->stone) && (subresult->stone != node->stone)) { /* If previuos best and new found are good for opponent */
        /* Here is missing condition to pick brach with more variations */
        if(subresult->nodeTurnNumber > bestFound->nodeTurnNumber) { /* and new found has more potential for reading error */
          bestFound = subresult; /* the found is then new best */
          newActive = node;
        }
      }
    }
    node = node->sibling;
  } while(node != firstChild);
  newActive->activeBranch = TRUE; /* Select new active branch */
  return bestFound;
}

Node * RandomSibling(Node * oneOfThem)
{
  Node * node = NULL;
  gint howMany = 0;
  gint changes = 0;
  
  node = oneOfThem;
  do {
    howMany += CountLeafs(node);
    node = node->sibling;
  } while(node != oneOfThem);
  changes = rand()%howMany;
  while(node != NULL) {
    changes -= CountLeafs(node);
    if(changes < 0) {
      return node;
    }
    node = node->sibling;
  }
  return node;
}

gint CountLeafs(Node * subTreeRoot)
{
  gint result = 0;
  Node * first = NULL;
  Node * toTry = NULL;
  
  if(subTreeRoot == NULL) {
    return 0;
  }
  if(subTreeRoot->child == NULL) {
    return 1;
  }
  first = subTreeRoot->child;
  toTry = first;
  do {
    result += CountLeafs(toTry);
    toTry = toTry->sibling;
  } while ((first != toTry) && (toTry != NULL));
  return result;
}

/* Find out to which player given position is favorable */
Stone GetGoodForColor(Node * position)
{
  Node * child = NULL;

  if(position == NULL) { /* If there is no position */
    return STONE_NONE; /* it is not good to either player */
  }
  child = position->child;
  while (child != NULL) { /* If any of the child positions of the position */
    if(GetGoodForColor(child) != position->stone) { /* are bad for the player who made the last play */
      return OPPONENT(position->stone); /* oponent could chose that continuation and thus postion is better for him */
    }
    child = child->sibling; /* Check all children */
    if(child == position->child) { /* if we are about to check first child again */
      break; /* we give up checking more instead */
    }
  }
  return position->stone; /* As opponent did not find good line of play, the position is good for the player who made the last play */
}

/* Tries to refute chosen play with the existing analysis. If succeeds, returns the refutation, if not, returns the move itself */
Node * RefutePlay(Node * play)
{
  Node * child = NULL;

  if(play == NULL) { /* If there is no play */
    return NULL; /* it cannot be refuted */
  }
  child = play->child;
  while (child != NULL) { /* If any child of the play */
    if(GetGoodForColor(child) == child->stone) { /* is beneficial for that player */
      return child; /* we have found refutation */
    }
    child = child->sibling; /* Check all children */
    if(child == play->child) { /* if we are about to check first child again */
      break; /* we give up checking more instead */
    }
  }
  return play; /* As no refutation were found, we believe there isn't any */
}

void ReNumberMovesOfTree(Node * node, gint moveNumber)
{
  Node * first;
  if(node == NULL) { /* If parent had no children */
    return; /* we are done for this branch */
  }
  first = node; /* Remember were we started to check these siblings */
  do {
    node->nodeTurnNumber = moveNumber; /* Set wanted move number to the node */
    ReNumberMovesOfTree(node->child, moveNumber + 1); /* and increased move numbers to it's children */
    node = node->sibling; /* and same move number to it's sisters */
  } while (first != node); /* until for all siblings have been set new move numbers */
}

void MakePrunedLayout(Node * node, gint moveNumber)
{
  Node * sibling = NULL;

  /* return MakeTreeLayout(node); Testing code, REMOVE */
  if(node == NULL) {
    return;
  }
  if(node->parent == NULL) {
    node->nodeTurnNumber = 1;
  }
  else {
    node->nodeTurnNumber = node->parent->nodeTurnNumber + 1;
  }
  node->nodeDrawX = moveNumber;
  if(node->parent == NULL) {
    node->nodeHeight = 0;
  }
  else {
    node->nodeHeight = node->parent->nodeHeight - 1;
    if(node->nodeHeight < 0) {
      node->nodeHeight = 0;
    }
  }
  sibling = node;
  if(node->sibling == node) {
    MakePrunedLayout(node->child,moveNumber + 1);
  }
  else {
    if(node->child != NULL) {
      if(node->child == node->child->sibling) {
        MakePrunedLayout(node->child,moveNumber + 1);
      }
      else {
        MakePrunedLayout(node->child,moveNumber + 2);
      }
    }
  }
  while(sibling->sibling != node) {
    sibling->sibling->nodeHeight = sibling->nodeHeight + 1;
    sibling = sibling->sibling;
    sibling->nodeTurnNumber = node->nodeTurnNumber;
    sibling->nodeDrawX = node->nodeDrawX;
    if(sibling->child != NULL) {
      if(sibling->child == sibling->child->sibling) {
        MakePrunedLayout(sibling->child,moveNumber + 1);
      }
      else {
        MakePrunedLayout(sibling->child,moveNumber + 2);
      }
    }
  }
}
