/*****************************************************************************
 * Copyright: 2010-2011 Christian Fetzer <fetzer.ch@googlemail.com>          *
 * Copyright: 2010-2011 Michael Zanetti <mzanetti@kde.org>                   *
 *                                                                           *
 * This program is free software: you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation, either version 3 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This program is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.     *
 *                                                                           *
 ****************************************************************************/

#include "itemdialog.h"
#include "core.h"
#include "maps/gmwmap.h"
#include "data/gmwitemmodel.h"
#include "data/gmwvehicle.h"
#include "data/gmwparkingspot.h"
#include "engines/gmwengine.h"

#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QDebug>
#include <QPushButton>
#include <QStackedWidget>
#include <QMap>
#include <QLabel>
#include <QAction>
#include <QMessageBox>
#include <QTimer>

#ifdef Q_WS_MAEMO_5
#include <QMaemo5InformationBox>
#endif

ItemDialog::ItemDialog(QWidget *parent, Qt::WindowFlags f) :
    QDialog(parent, f)
{
    setAttribute(Qt::WA_DeleteOnClose, true);
    setMaximumSize(parentWidget()->size());

    m_masterLayout = new QBoxLayout(QBoxLayout::TopToBottom);
    m_masterLayout->setMargin(0);

    m_stackWidget = new QStackedWidget(this);

    m_masterLayout->addWidget(m_stackWidget);

    setLayout(m_masterLayout);

    // Add the buttons for scrolling
    m_buttonLayout = new QBoxLayout(QBoxLayout::RightToLeft);

    m_pbNext = new QPushButton();
    m_pbNext->setIcon(QIcon(":/arrow-right.png"));
    m_pbNext->setEnabled(false);
    connect(m_pbNext, SIGNAL(clicked()), this, SLOT(nextPage()));
    m_buttonLayout->addWidget(m_pbNext);

#ifndef Q_OS_SYMBIAN
    m_pbBook = new QPushButton();
    m_pbBook->setIcon(QIcon(":/car-booked.png"));
    connect(m_pbBook, SIGNAL(clicked()), this, SLOT(book()));
    m_buttonLayout->addWidget(m_pbBook);

    m_pbGoTo = new QPushButton();
    m_pbGoTo->setIcon(QIcon(":/go-last.png"));
    connect(m_pbGoTo, SIGNAL(clicked()), this, SLOT(slotGoTo()));
    m_buttonLayout->addWidget(m_pbGoTo);
#else
    QAction* backButton = new QAction(tr("Back"), this);
    connect(backButton, SIGNAL(triggered()), SLOT(close()));
    backButton->setSoftKeyRole(QAction::NegativeSoftKey);

    QAction* gotoButton = new QAction(tr("Go To"), this);
    connect(gotoButton, SIGNAL(triggered()), SLOT(slotGoTo()));
    gotoButton->setSoftKeyRole(QAction::PositiveSoftKey);

    QList<QAction *> softKeys;
    softKeys.append(backButton);
    softKeys.append(gotoButton);

    /* set new soft keys */
    addActions(softKeys);

    setMaximumSize(parent->size());
#endif

    m_pbPrevious = new QPushButton();
    m_pbPrevious->setIcon(QIcon(":/arrow-left.png"));
    m_pbPrevious->setEnabled(false);
    connect(m_pbPrevious, SIGNAL(clicked()), this, SLOT(previousPage()));
    m_buttonLayout->addWidget(m_pbPrevious);
    m_masterLayout->addLayout(m_buttonLayout);    
}

ItemDialog::~ItemDialog() {
    qDebug() << "~ItemDialog";
}

QString PageWidget::smileyName(GMWVehicle::State state) {
    switch(state) {
    case GMWVehicle::StateExcellent:
        return ":/face-smile-big.png";
    case GMWVehicle::StateGood:
        return ":/face-smile.png";
    case GMWVehicle::StateSatisfied:
        return ":/face-plain.png";
    case GMWVehicle::StateIssuesExist:
        return ":/face-sad.png";
    case GMWVehicle::StateUnacceptable:
        return ":/face-surprise.png";
    case GMWVehicle::StateUnknown:
        return ":/unknown.png";
    }
    return QString();
}

QString PageWidget::statusText(GMWVehicle::State state) {
    switch(state) {
    case GMWVehicle::StateExcellent:
        return tr("Excellent");
    case GMWVehicle::StateGood:
        return tr("Good");
    case GMWVehicle::StateSatisfied:
        return tr("Satisfied");
    case GMWVehicle::StateIssuesExist:
        return tr("Bad");
    case GMWVehicle::StateUnacceptable:
        return tr("Unacceptable");
    case GMWVehicle::StateUnknown:
        return tr("Unknown");
    }
    return QString();
}

void ItemDialog::addObject(GMWItem *object) {
    int index = m_stackWidget->insertWidget(m_stackWidget->count(), new PageWidget(object));

    if(m_stackWidget->count() > 1) {
        m_pbNext->setEnabled(true);
    }
    if(m_stackWidget->count() == 1) {
        goToPage(0);
    }
}


void ItemDialog::nextPage(){
    if(m_stackWidget->currentIndex() < (m_stackWidget->count() - 1)) {
        goToPage(m_stackWidget->currentIndex() + 1);
    }
}

void ItemDialog::previousPage() {
    if(m_stackWidget->currentIndex() > 0) {
        goToPage(m_stackWidget->currentIndex() - 1);
    }
}

void ItemDialog::updateWindowTitle(GMWItem *object) {
    switch(object->objectType()){
    case GMWItem::TypeGasStation:
        setWindowTitle(tr("Gas Station"));
        break;
    case GMWItem::TypeParkingLot:
        setWindowTitle(tr("Parking Spot"));
        break;
    case GMWItem::TypeVehicle:
        setWindowTitle(tr("Vehicle"));
        break;
    case GMWItem::TypeUnknown:
        setWindowTitle(tr("Unknown"));
        break;
    }
    setWindowIcon(QIcon(object->image()));
}

void ItemDialog::slotGoTo() {
    accept();
    GMWItem *currentItem = dynamic_cast<PageWidget*>(m_stackWidget->currentWidget())->item();
    Core::map()->disableTracking();
    Core::map()->moveTo(currentItem->location());
    Core::map()->routeTo(currentItem->location());
    Core::map()->setActiveItem(currentItem);
    emit zoomTo(currentItem);
}

void ItemDialog::goToPage(int page) {
    if(m_stackWidget->count() > page) {
        m_stackWidget->setCurrentIndex(page);
        GMWItem *item = dynamic_cast<PageWidget*>(m_stackWidget->currentWidget())->item();
        updateWindowTitle(item);

        if(page > 0) {
            m_pbPrevious->setEnabled(true);
        } else {
            m_pbPrevious->setEnabled(false);
        }
        if(page < m_stackWidget->count() - 1) {
            m_pbNext->setEnabled(true);
        } else {
            m_pbNext->setEnabled(false);
        }
        m_pbBook->setIcon(QIcon(":/car-booked.png"));
        if(item->objectType() == GMWItem::TypeVehicle) {
            GMWVehicle *vehicle = dynamic_cast<GMWVehicle*>(item);
            if(vehicle->booking().isValid() && ! vehicle->booking().isExpired()) {
                m_pbBook->setIcon(QIcon(":/cancel-booking.png"));
            }
        }
    }
}

void ItemDialog::book()
{
    if(!Core::engine()->defaultAccount().isValid()) {
        // Try to setup account
        emit setupAccount();
        reject();
        return;
    }

    GMWVehicle *currentVehicle = dynamic_cast<GMWVehicle*>(dynamic_cast<PageWidget*>(m_stackWidget->currentWidget())->item());
    if(!currentVehicle) {
        return;
    }

    if(currentVehicle->booking().isValid()) {
        // car already booked. cancel booking
        if(QMessageBox::question(this, tr("Cancel booking"), tr("Cancel booking?") + "<br>" + tr("Note: Cancelling or missing a booking is associated with additional costs."), QMessageBox::Yes | QMessageBox::No) == QMessageBox::No) {
            return;
        }

        if(Core::engine()->cancelBooking(currentVehicle)) {
            m_pbBook->setIcon(QIcon(":/car-booked.png"));

#ifdef Q_WS_MAEMO_5
            QMaemo5InformationBox::information(this, "Booking cancelled successfully.");
#else
            QMessageBox::critical(this, "Booking cancelled successfully.", "Booking cancelled successfully.");
#endif
        } else {
#ifdef Q_WS_MAEMO_5
            QMaemo5InformationBox::information(this, "Error cancelling booking.");
#else
            QMessageBox::critical(this, "Error cancelling booking.", "Error cancelling booking.");
#endif
        }
        return;
    }


    if(QMessageBox::question(this, tr("Create booking"), tr("Create booking?") + "<br>" + tr("Note: The booking will be valid for 30 minutes from now. Cancelling or missing a booking is associated with additional costs."), QMessageBox::Yes | QMessageBox::No) == QMessageBox::No) {
        return;
    }
    if(Core::engine()->createBooking(currentVehicle)) {

        m_pbBook->setIcon(QIcon(":/cancel-booking.png"));
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(this, "The car has been bookes successfully");
#else
        QMessageBox::information(this, "Car booked successfully", "Car booked successfully");
#endif
//        accept();
        emit zoomTo(currentVehicle);
        Core::map()->moveTo(currentVehicle->location());
        Core::map()->routeTo(currentVehicle->location());
    } else {
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(this, "Error booking car");
#else
        QMessageBox::critical(this, "Error booking car", "Error booking car");
#endif
    }

}

PageWidget::PageWidget(GMWItem *item, QWidget *parent):
    QWidget(parent),
    m_item(item)
{
//    setMaximumSize(parentWidget()->size());

    QHBoxLayout *layout = new QHBoxLayout();
    setLayout(layout);
    QVBoxLayout *vLayout1 = new QVBoxLayout();
    QVBoxLayout *vLayout2 = new QVBoxLayout();

    vLayout1->addWidget(new QLabel(tr("Name:")));
    QLabel *nameLabel = new QLabel(item->name());
    nameLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Minimum);
    vLayout2->addWidget(nameLabel);

    vLayout1->addWidget(new QLabel(tr("Address:")));
    vLayout2->addWidget(new QLabel(item->address()));

    vLayout1->addWidget(new QLabel(tr("Distance:")));

    QHBoxLayout *distanceLayout = new QHBoxLayout();

    m_azimuthLabel = new QLabel();
    m_azimuthLabel->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Minimum);
    distanceLayout->addWidget(m_azimuthLabel);

    m_distanceLabel = new QLabel(item->distanceString());
    distanceLayout->addWidget(m_distanceLabel);

    vLayout2->addLayout(distanceLayout);

    switch(item->objectType()){
    case GMWItem::TypeGasStation:
        vLayout1->addWidget(new QLabel(this));
        vLayout1->addWidget(new QLabel(this));
        vLayout1->addWidget(new QLabel(this));
        break;
    case GMWItem::TypeParkingLot: {
        GMWParkingSpot *parking = dynamic_cast<GMWParkingSpot*>(item);
        QHBoxLayout *parkLayout = new QHBoxLayout();
        vLayout1->addWidget(new QLabel(tr("Capacity:")));
        QLabel *parkIcon = new QLabel();
        parkIcon->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Minimum);
        if(parking->capacityUsed() < parking->capacityTotal()){
            parkIcon->setPixmap(QPixmap(":/parking_free"));
        } else {
            parkIcon->setPixmap(QPixmap(":/parking_full"));
        }
        parkLayout->addWidget(parkIcon);
        parkLayout->addWidget(new QLabel(
                                  QString::number(parking->capacityTotal() - parking->capacityUsed())
                                  + '/' + QString::number(parking->capacityTotal())
                                  + " (" + tr("free/total") + ')'));
        vLayout2->addLayout(parkLayout);
        vLayout1->addWidget(new QLabel(this));
        vLayout1->addWidget(new QLabel(this));
    }
    break;
    case GMWItem::TypeVehicle: {
        GMWVehicle *car = dynamic_cast<GMWVehicle*>(item);

        vLayout1->addWidget(new QLabel(tr("Availability:")));
        QHBoxLayout *availabilityLayout = new QHBoxLayout();
        m_availabilityImage = new QLabel();
        m_availabilityImage->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Minimum);
        availabilityLayout->addWidget(m_availabilityImage);
        m_availabilityLabel = new QLabel();
        availabilityLayout->addWidget(m_availabilityLabel);
        vLayout2->addLayout(availabilityLayout);

        vLayout1->addWidget(new QLabel(tr("Fuel level:")));
        QHBoxLayout *fuelLayout = new QHBoxLayout();
        QLabel *fuelLabel = new QLabel();
        if(car->fuelLevel() < 10){
            fuelLabel->setPixmap(QPixmap(":/fuel_red.png"));
        } else if (car->fuelLevel() < 25) {
            fuelLabel->setPixmap(QPixmap(":/fuel_orange.png"));
        } else {
            fuelLabel->setPixmap(QPixmap(":/fuel_green.png"));
        }
        fuelLabel->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Minimum);
        fuelLayout->addWidget(fuelLabel);
        fuelLayout->addWidget(new QLabel(QString::number(car->fuelLevel()) + "%"));
        vLayout2->addLayout(fuelLayout);

        vLayout1->addWidget(new QLabel(tr("Interior state: ")));

        QHBoxLayout *statusLayout1 = new QHBoxLayout();
        QLabel *interiorLabel = new QLabel();
        interiorLabel->setPixmap(QPixmap(smileyName(car->interiorState())));
        interiorLabel->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Minimum);
        statusLayout1->addWidget(interiorLabel);
        statusLayout1->addWidget(new QLabel(statusText(car->interiorState())));
        vLayout2->addLayout(statusLayout1);

        vLayout1->addWidget(new QLabel(tr("Exterior state: ")));

        QHBoxLayout *statusLayout2 = new QHBoxLayout();
        QLabel *exteriorLabel = new QLabel();
        exteriorLabel->setPixmap(QPixmap(smileyName(car->exteriorState())));
        exteriorLabel->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Minimum);
        statusLayout2->addWidget(exteriorLabel);
        statusLayout2->addWidget(new QLabel(statusText(car->exteriorState())));
        vLayout2->addLayout(statusLayout2);
    }
    break;

    case GMWItem::TypeUnknown:
        break;
    }

    vLayout1->addStretch(0);
    vLayout2->addStretch(0);

    layout->addLayout(vLayout1);
    layout->addLayout(vLayout2);

    itemChanged();
    m_refreshTimer.setInterval(30000);
    connect(&m_refreshTimer, SIGNAL(timeout()), SLOT(itemChanged()));
    connect(m_item, SIGNAL(changed()), SLOT(itemChanged()));
}

GMWItem* PageWidget::item()
{
    return m_item;
}

void PageWidget::itemChanged()
{
    qDebug() << "item has changed";
    if (!m_item->azimuthImage().isNull()) {
        m_azimuthLabel->setPixmap(m_item->azimuthImage());
    } else {
        m_azimuthLabel->setVisible(false);
    }
    m_distanceLabel->setText(m_item->distanceString());
    if(m_item->objectType() == GMWItem::TypeVehicle) {
        GMWVehicle *car = dynamic_cast<GMWVehicle*>(m_item);
        qDebug() << "got car:" << car->booking().isValid();
        qDebug() << "exp:" << car->booking().toString();
        if(car->booking().isValid()){
//            qDebug() << "expirationime is:" << car->booking().expirationTime();
            if(!car->booking().isExpired()) {
                m_availabilityImage->setPixmap(QPixmap(":/car-booked.png"));
            } else {
                m_availabilityImage->setPixmap(QPixmap(":/booking-expired.png"));
            }
            m_availabilityLabel->setText(car->booking().toString());
            m_refreshTimer.start();
        } else {
            m_availabilityImage->setPixmap(QPixmap(":/parking_free"));
            m_availabilityLabel->setText(tr("Free"));
            m_refreshTimer.stop();
        }
    }
}
