/*****************************************************************************
 * Copyright: 2010-2011 Christian Fetzer <fetzer.ch@googlemail.com>          *
 * Copyright: 2010-2011 Michael Zanetti <mzanetti@kde.org>                   *
 *                                                                           *
 * This program is free software: you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation, either version 3 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This program is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.     *
 *                                                                           *
 ****************************************************************************/

#include <QtGui>
#include <QDebug>

#include <QtNetwork/QNetworkAccessManager>
#include <QtNetwork/QNetworkRequest>
#include <QNetworkReply>

#include "mainwindow.h"
#include "core.h"

#include "maps/gmwmap.h"
#include "engines/gmwengine.h"
#include "data/gmwitemmodel.h"
#include "data/gmwitemsortfilterproxymodel.h"
#include "data/gmwvehicle.h"
#include "data/gmwgasstation.h"
#include "ui/menubar.h"
#include "ui/settingsdialog.h"
#include "ui/itemlist.h"
#include "ui/itemdialog.h"
#include "ui/oauthwizard.h"
#include "ui/accountselectordialog.h"

#include <QGeoPositionInfoSource>
#include <QMenuBar>
#include <QTimer>
#include <QMessageBox>

#ifdef Q_WS_MAEMO_5
    #include <QtGui/QX11Info>
    #include <X11/Xlib.h>
    #include <X11/Xatom.h>
    #include <QtDBus/QDBusConnection>
    #include <mce/dbus-names.h>
    #include <mce/mode-names.h>
    #include <QMaemo5InformationBox>
    #include <QMaemo5ListPickSelector>
#endif

MainWindow::MainWindow() :
    m_direction(0)
{
#ifdef Q_WS_MAEMO_5
    m_orientation = Qt::Horizontal;
    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif

    setWindowTitle("Get Me Wheels");
}

void MainWindow::initialize()
{
    connect(qApp, SIGNAL(onlineStateChanged(bool)), SLOT(onlineStateChanged(bool)));

    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));

#ifdef Q_OS_SYMBIAN
    showMaximized();
#else
#ifndef Q_WS_MAEMO_5
   resize(800,480);
#endif
#endif

    // select the map displaying engine. Hardcoded for now
    setCentralWidget(Core::map());
    centralWidget()->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding);
    connect(Core::map(), SIGNAL(objectClicked(GMWItem*)), this, SLOT(objectClicked(GMWItem*)));
    connect(Core::map(), SIGNAL(showList()), this, SLOT(showList()));
    connect(Core::map(), SIGNAL(resized()), SLOT(mapResized()));
    connect(Core::map(), SIGNAL(panned()), SLOT(mapPanned()));
    connect(Core::map(), SIGNAL(mapError(QString)), SLOT(mapError(QString)));

    // create Engine
    connect(Core::engine(), SIGNAL(downloadStarted()), SLOT(downloadStarted()));
    connect(Core::engine(), SIGNAL(downloadFinished(GMWItem::Type, bool , const QString &)), SLOT(downloadFinished(GMWItem::Type, bool, const QString &)));
    connect(Core::engine(), SIGNAL(loadedFromCache(GMWItem::Type,QDateTime)), SLOT(loadedFromCache(GMWItem::Type,QDateTime)));

    // Menubar
    setMenuBar(new MenuBar(this));

    // Status Bar
    createStatusBar();

    // create Object List
    m_objectList = new GMWObjectList(Core::proxyModel(), this);
    connect(m_objectList, SIGNAL(setupAccount()), SLOT(setupAccount()));
    m_objectList->layout()->setMenuBar(new MenuBar(m_objectList));

#ifdef Q_WS_MAEMO_5
    m_objectList->setAttribute(Qt::WA_Maemo5StackedWindow);
#endif

    // GPS
    m_gps = QGeoPositionInfoSource::createDefaultSource(this);
    if (m_gps) {
        m_gps->setUpdateInterval(5000);
        connect(m_gps, SIGNAL(positionUpdated(const QGeoPositionInfo&)), this, SLOT(positionUpdated(const QGeoPositionInfo&)));
        m_gps->startUpdates();
    } else {
        qDebug() << "No GPS available";
        positionUpdated(QGeoPositionInfo(QGeoCoordinate(48.41000, 9.96000), QDateTime::currentDateTime()));
    }
    connect(this, SIGNAL(positionUpdated(const QGeoCoordinate&, qreal)), Core::model(), SLOT(currentPositionChanged(const QGeoCoordinate&, qreal)));
//    Core::map()->setTracking(true);

    grabZoomKeys(true);

#ifdef Q_WS_MAEMO_5
    QDBusConnection bus = QDBusConnection::systemBus();
    bus.connect(MCE_SERVICE, MCE_SIGNAL_PATH, MCE_SIGNAL_IF, MCE_INACTIVITY_SIG , this, SLOT(setInactive(bool)));
#endif

    // load Settings
    QMetaObject::invokeMethod(this, "loadSettings", Qt::QueuedConnection);
//    qDebug() << "settings loaded";
    //refreshAll(true);

}

void MainWindow::setupAccount()
{

    if(m_settingsDialog) {
        Core::engine()->setLocation(m_settingsDialog->location());
    }

    if(!Core::engine()->authenticated()) {
        OAuthSetupUi oauthsetup(this);
        oauthsetup.exec();
    }

    if(!Core::engine()->authenticated()) {
        // Still not authenticated... print error message
        return;
    }

    QList<GMWAccount> accounts = Core::engine()->accounts();
    if(accounts.count() > 0) {
#ifdef Q_WS_MAEMO_5
        QMaemo5ListPickSelector *selector = new QMaemo5ListPickSelector();
        QStandardItemModel *model = new QStandardItemModel();
        foreach(const GMWAccount &account, accounts) {
            model->appendRow(new QStandardItem(account.description()));
        }
        selector->setModel(model);
        QDialog *listPicker = dynamic_cast<QDialog*>(selector->widget(this));
        listPicker->exec();
        foreach(const GMWAccount &account, accounts) {
            if(selector->currentValueText() == account.description()) {
                Core::engine()->setDefaultAccount(account);
            }
        }
#else
        AccountSelectorDialog accountSelector(accounts, this);
        accountSelector.exec();
        Core::engine()->setDefaultAccount(accountSelector.selectedAccount());
#endif
        Core::model()->clearVehicles();
        Core::engine()->refreshVehicles(false);
    } else {
        // No accounts for location... Print error message
        QString message = tr("No accounts avaialble for this location");
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(NULL, message, QMaemo5InformationBox::DefaultTimeout);
#else
        QMessageBox::information(this, "Error", message);
#endif
    }

    if(m_settingsDialog) {
        m_settingsDialog->setAccountName(Core::engine()->defaultAccount().description());
    }

}

MainWindow::~MainWindow(){
    grabZoomKeys(false);
    delete m_objectList;
}

void MainWindow::closeEvent(QCloseEvent *event) {
    if (true) {
        event->accept();
    } else {
        event->ignore();
    }
}

void MainWindow::positionUpdated(const QGeoPositionInfo &info) {
    qDebug() << "GPS Position updated:" << QDateTime::currentDateTime().toString();
    Core::map()->positionUpdated(info);
    statusBarGPS->setText("");

    // Cache the direction (not included in every gps update)
    if (info.hasAttribute(QGeoPositionInfo::Direction)) {
        m_direction = info.attribute(QGeoPositionInfo::Direction);
        //qDebug() << "GPS direection received:" << m_direction;
    }
    emit positionUpdated(info.coordinate(), m_direction);

    // Disable the fitInViewport workaround
    mapPanned();
}

void MainWindow::timerEvent(QTimerEvent *event)
{
    qDebug() << "Timer ID:" << event->timerId();
    //qDebug() << qrand() % 180 - 90;
    QGeoPositionInfo position(QGeoCoordinate(qrand() % 180 - 90, qrand() % 360 - 180), QDateTime::currentDateTime());

    if (!position.isValid()) {
        qDebug() << position;
    }
    position.setAttribute(QGeoPositionInfo::Direction, qrand() % 360);
    positionUpdated(position);
}

void MainWindow::about() {
   QMessageBox::about(this, tr("Get Me Wheels - Car finder"),
                      tr("Get Me Wheels - Car finder <br> " \
                         "Authors: <br>Christian Fetzer (fetzer.ch@googlemail.com) <br>" \
                         "Michael Zanetti (mzanetti@kde.org) <br><br>" \
                         "Helps you finding the fastest way to the next free car2go (This " \
                         "product uses the car2go API but is not endorsed or certified by " \
                         "car2go - www.car2go.com)."));
}

void MainWindow::createStatusBar() {
    statusBarGPS = new QLabel(tr("Waiting for GPS..."));
    statusBarDownload = new QLabel(tr("No data available"));
    statusBarDownload->setAlignment(Qt::AlignRight);

    statusBar()->addWidget(statusBarGPS, 1);
    statusBar()->addWidget(statusBarDownload, 1);
}

void MainWindow::refreshAll(bool useCache) {
    Core::model()->clearAll();
    Core::engine()->refreshStationary(useCache);
    Core::engine()->refreshVehicles(useCache);
}

void MainWindow::refreshVehicles() {
    Core::model()->clearVehicles();
    Core::engine()->refreshVehicles(false);
}

void MainWindow::objectClicked(GMWItem *object) {
    if(!m_itemDialog) {
        m_itemDialog = new ItemDialog(this);
        connect(m_itemDialog, SIGNAL(setupAccount()), SLOT(setupAccount()));
        m_itemDialog->addObject(object);

#ifdef Q_WS_MAEMO_5
        QSettings settings("getmewheels", "getmewheels");
        m_itemDialog->setAttribute(Qt::WA_Maemo5AutoOrientation, settings.value("Autorotation", true).toBool());
#endif
#ifdef Q_OS_SYMBIAN
        m_itemDialog->showMaximized();
        m_itemDialog->setFocus();
#else
        m_itemDialog->show();
#endif
    } else {
        m_itemDialog->addObject(object);
    }
}

void MainWindow::openSettings() {
    m_settingsDialog = new SettingsDialog(this);
    QSettings settings("getmewheels", "getmewheels");

    m_settingsDialog->setLocations(Core::engine()->supportedLocations(), settings.value("Location").toString());
    m_settingsDialog->setMapProviders(Core::map()->supportedMapProviders(), settings.value("MapProvider").toString());
    m_settingsDialog->setRotationEnabled(settings.value("Autorotation", true).toBool());
    m_settingsDialog->setAccountName(Core::engine()->defaultAccount().description());
    qDebug() << "got default account:" << Core::engine()->defaultAccount().description();

    connect(m_settingsDialog, SIGNAL(setupAccount()), SLOT(setupAccount()));

    if(m_settingsDialog->exec() == QDialog::Accepted) {
        qDebug() << "dialog accepted. location is" << m_settingsDialog->location();
        settings.setValue("Location", m_settingsDialog->location());
        settings.setValue("MapProvider", m_settingsDialog->mapProvider());
        settings.setValue("Autorotation", m_settingsDialog->rotationEnabled());
        // If a user sets the location explicitly he most likely wants to see the location instead his current position => disable tracking
        Core::map()->setTracking(false);
        loadSettings(false);
//        refreshAll(false);
    }
    delete m_settingsDialog;
}

void MainWindow::loadSettings(bool useCache)
{
    QSettings settings("getmewheels", "getmewheels");

    // Check if all settings are here
    if(!settings.contains("MapProvider") || !settings.contains("Location")) {
        openSettings();
    }

    bool changed = false;

    QString mapProvider = settings.value("MapProvider").toString();
    if(mapProvider != Core::map()->mapProvider()) {
        if(!Core::map()->setMapProvider(mapProvider)) {
            if(!Core::map()->supportedMapProviders().isEmpty()) {
                settings.setValue("MapProvider", Core::map()->supportedMapProviders().first());
            }
        }
        changed = true;
    }

    QString location = settings.value("Location").toString();
    qDebug() << "loading settings location:" << location << ". Old location is:" << Core::engine()->location();
    if(Core::engine()->supportedLocations().contains(location) && (Core::engine()->location() != location)) {
        changed = true;
        Core::model()->clearAll();
        Core::engine()->setLocation(location);
    }

    if(changed) {
    qDebug() << "settings business area. map is:" << Core::map()->mapProvider() << "location is" << Core::engine()->location();
        Core::map()->setBusinessArea(Core::engine()->businessArea());
        refreshAll(useCache);
        qDebug() << "centering!";
    }
    centerOnStartCoordinates();

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5AutoOrientation, settings.value("Autorotation", true).toBool());
#endif
}

void MainWindow::centerOnStartCoordinates()
{
    // Center Map
    qDebug() << "zooming map to " << Core::engine()->startingBounds().topLeft() << Core::engine()->startingBounds().bottomRight();
    Core::map()->zoomTo(Core::engine()->startingBounds());
}

void MainWindow::showList() {
#ifdef Q_OS_SYMBIAN
    m_objectList->showMaximized();
    m_objectList->setFocus();
#else
#ifndef Q_WS_MAEMO_5
    m_objectList->resize(800, 600);
#endif
    m_objectList->show();
#endif
}

void MainWindow::orientationChanged(){
#ifdef Q_WS_MAEMO_5
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    if (screenGeometry.width() > screenGeometry.height()) {
        qDebug() << "In Landscape Mode";
        m_orientation = Qt::Horizontal;
    } else {
        qDebug() << "In Portrait Mode";
        m_orientation = Qt::Vertical;
    }
//    if(m_itemDialog) {
//        m_itemDialog->rotate(m_orientation);
//    }
#endif
}

void MainWindow::onlineStateChanged(bool online) {
    if (online) {
//        qDebug() << "Now online, refreshing map";
//        Core::map()->refresh();
    }
}

void MainWindow::grabZoomKeys(bool grab) {
#ifndef Q_WS_MAEMO_5
    Q_UNUSED(grab)
#else
    if (!winId()) {
        qWarning("Can't grab keys unless we have a window id");
        return;
    }

    unsigned long val = (grab) ? 1 : 0;
    Atom atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", False);
    if (!atom) {
        qWarning("Unable to obtain _HILDON_ZOOM_KEY_ATOM. This example will only work "
                 "on a Maemo 5 device!");
        return;
    }

    XChangeProperty (QX11Info::display(),
             winId(),
             atom,
             XA_INTEGER,
             32,
             PropModeReplace,
             reinterpret_cast<unsigned char *>(&val),
             1);
#endif
 }

void MainWindow::downloadStarted() {
    statusBarDownload->setText(tr("Downloading data..."));
}

void MainWindow::downloadFinished(GMWItem::Type type, bool success, const QString &errorString) {
    Q_UNUSED(type)
    qDebug() << "download finished";
    if(success){
        statusBarDownload->setText(tr("Data download successful @ %1").arg(QDateTime::currentDateTime().toString("h:mm")));
    } else {
        statusBarDownload->setText(tr("Data download failed: %1").arg(errorString));
    }
}

void MainWindow::loadedFromCache(GMWItem::Type type, const QDateTime &cacheDate) {
    QString typeString;
    switch (type) {
    case GMWItem::TypeVehicle:
        typeString = tr("Vehicles");
        break;
    case GMWItem::TypeGasStation:
        typeString = tr("Gas Stations");
        break;
    case GMWItem::TypeParkingLot:
        typeString = tr("Parking Lots");
        break;
    default:
        qWarning("Unknown data loaded from cache");
    }
    qint64 age = cacheDate.msecsTo(QDateTime::currentDateTime()) / 1000;
    QString ageString;
    if (age < 60) ageString = QString::number(age) + "s";
    else if (age < 60*60) ageString = QString::number(age / 60) + "min";
    else if (age < 60*60*24) ageString = QString::number(age / (60*60)) + "h";
    else ageString = QString::number(age / (60*60*24)) + "day(s)";

    statusBarDownload->setText(tr("Loaded %1 from cache (%2)").arg(typeString).arg(ageString));
}

/* The next two functions are used as a workaround to center the map correctly on
 * startup. QGraphicsGeoMap has an issue in fitInViewport(). To make the map
 * nicely usable I add this workaround. Should be removed once QGraphicsGeoMap
 * is fixed:
 * On startup I connect the signal resized() to the map. When the map is
 * placed in the window, it will this emit this signal multiple times.
 * after that we are ready to call fitInViewPort(). However, the signal will
 * also be emitted on a screen rotate or window resize. To prevent the map
 * beeing centered on every rotation, the signal is disconnected once tha map has
 * been panned
 */
void MainWindow::mapResized()
{
    centerOnStartCoordinates();
    qDebug() << "***** resized" << Core::map()->size();
//    mapPanned();
}

void MainWindow::mapPanned()
{
    disconnect(Core::map(), SIGNAL(resized()), this, SLOT(mapResized()));
    disconnect(Core::map(), SIGNAL(panned()), this, SLOT(mapPanned()));
}

void MainWindow::setInactive(bool inactive)
{
    if(inactive) {
        qDebug() << "Application now inactive";
        m_gps->stopUpdates();
    } else {
        qDebug() << "Application now active";
        m_gps->startUpdates();
    }
}

void MainWindow::mapError(const QString &message)
{
#ifdef Q_WS_MAEMO_5
    QMaemo5InformationBox::information(NULL, message, QMaemo5InformationBox::DefaultTimeout);
#else
    statusBarDownload->setText(message);
#endif
}
