/*****************************************************************************
 * Copyright: 2010-2011 Christian Fetzer <fetzer.ch@googlemail.com>          *
 * Copyright: 2010-2011 Michael Zanetti <mzanetti@kde.org>                   *
 *                                                                           *
 * This program is free software: you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation, either version 3 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This program is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.     *
 *                                                                           *
 ****************************************************************************/

#include "gmwitemmodel.h"

#include <QDebug>
#include <QPaintEngine>
#include <QRectF>

GMWItemModel::GMWItemModel(QObject *parent) :
    QAbstractTableModel(parent)
{
    qRegisterMetaType<GMWItem*>("GMWObject*");
}

void GMWItemModel::addObject(GMWItem *item) {
    addObjects(QList<GMWItem*>() << item);
}

void GMWItemModel::addObjects(QList<GMWItem*> items) {
    qDebug() << "Adding " << items.count() << " items to model; Current size is" << m_objects.count();
    beginInsertRows(QModelIndex(), m_objects.size(), m_objects.size() + items.count() - 1);
    m_objects.append(items);
    endInsertRows();
    qDebug() << "Items added; New size is " << m_objects.count();
}

void GMWItemModel::clearAll() {
    if (m_objects.isEmpty()) return;

    qDebug() << "Removing all items from model; Current size is" << m_objects.count();
    beginRemoveRows(QModelIndex(), 0, m_objects.size()-1);
    qDeleteAll(m_objects);
    m_objects.clear();
    endRemoveRows();
    qDebug() << "All items removed";
}

void GMWItemModel::clearVehicles() {
    qDebug() << "Removing vehicles from model; Current size is" << m_objects.count();
    QMutableListIterator<GMWItem*> i(m_objects);
    while (i.hasNext()) {
        if (i.next()->objectType() == GMWItem::TypeVehicle) {
            beginRemoveRows(QModelIndex(), m_objects.indexOf(i.value()), m_objects.indexOf(i.value()));
            delete i.value();
            i.remove();
            endRemoveRows();
        }
    }
    qDebug() << "Vehicles removed; Size:" << m_objects.count();
}

void GMWItemModel::currentPositionChanged(const QGeoCoordinate &position, qreal direction)
{
    foreach (GMWItem* obj, m_objects) {
         obj->calculateDistance(position, direction);
    }
    emit dataChanged(index(0, 1), index(m_objects.size()-1, 2));
}

int GMWItemModel::rowCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent)

    return m_objects.count();
}

int GMWItemModel::columnCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent)

    if (m_currentPosition.isValid()) {
        return 1;
    } else {
        return 3;
    }
}

QVariant GMWItemModel::data(const QModelIndex &index, int role) const
{
    GMWItem *object = m_objects.at(index.row());

    if (role == Qt::DecorationRole) {
        switch (index.column()) {
            case 0: return object->image();
            case 2: return object->azimuthImage();
        }
    }

    if (role == Qt::DisplayRole) {
        switch (index.column()) {
            case 0: return object->address();
            case 1: return object->distanceString();
        }
    }

    if (role == Qt::UserRole) {
        return qVariantFromValue(object);
    }

    if (role == Qt::TextAlignmentRole) {
        switch (index.column()) {
            case 1:
                return QVariant(Qt::AlignRight | Qt::AlignVCenter);
        }
    }

    return QVariant();
}

GMWItem *GMWItemModel::nearestItem(GMWItem::Type type) const
{
    GMWItem *nearest = 0;
    foreach(GMWItem *item, m_objects) {
        if(item->objectType() == type) {
            if(nearest == 0 || item->distance() < nearest->distance()) {
                nearest = item;
            }
        }
    }
    return nearest;
}
