/*****************************************************************************
 * Copyright: 2010-2011 Christian Fetzer <fetzer.ch@googlemail.com>          *
 * Copyright: 2010-2011 Michael Zanetti <mzanetti@kde.org>                   *
 *                                                                           *
 * This program is free software: you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation, either version 3 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This program is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.     *
 *                                                                           *
 ****************************************************************************/

#include "mobilitymap.h"

#include "geomap.h"
#include "gmwmarker.h"
#include "navigator.h"

#include <QtCore/QDebug>
#include <QtCore/QPropertyAnimation>
#include <QtCore/QParallelAnimationGroup>

#include <QtGui/QVBoxLayout>

#include <QtLocation/QGeoBoundingBox>

/**
 * Private class ensures that map always fills view.
 */
class MapView : public QGraphicsView
{
public:
    MapView(QGraphicsScene *scene, QWidget *parent = 0) : QGraphicsView(scene, parent), m_map(0) {}
    void setMap(GeoMap* map) {
        if(m_map) {
            scene()->removeItem(m_map);
        }
        m_map = map;
        scene()->addItem(m_map);
        m_map->resize(viewport()->size());
    }

private:
    GeoMap* m_map;

protected:
    void resizeEvent(QResizeEvent *) {
        qDebug() << "resize event";
        if(m_map) {
            m_map->resize(viewport()->size());
        }
    }
};

MobilityMap::MobilityMap(GMWItemSortFilterProxyModel *model, QWidget *parent) :
    GMWMap(model, parent),
    m_serviceProvider(0),
    m_map(0)
{
    QGraphicsScene *scene = new QGraphicsScene;
    m_view = new MapView(scene, this);
    m_view->setVisible(true);
    m_view->setInteractive(true);
    m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_view->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_view->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    //m_map->scale(2,2);

    layout()->addWidget(m_view);

}

QStringList MobilityMap::supportedMapProviders() const
{
    return QGeoServiceProvider::availableServiceProviders();
}

QString MobilityMap::mapProvider() const
{
    return m_providerName;
}

bool MobilityMap::setMapProvider(const QString &provider)
{
    if(supportedMapProviders().contains(provider)) {
        if(m_serviceProvider != 0) {
            delete m_serviceProvider;
        }
        m_serviceProvider = new QGeoServiceProvider(provider);
        m_providerName = provider;
        initialize();
        return true;
    } else {
        qCritical("GeoServiceProvider not found! Falling back to first available one...");
        if(!QGeoServiceProvider::availableServiceProviders().isEmpty()) {
            m_serviceProvider = new QGeoServiceProvider(QGeoServiceProvider::availableServiceProviders().first());
            m_providerName = provider;
            initialize();
        }
        return false;
    }
}

void MobilityMap::initialize()
{
    // Create Map
    m_mappingManager = m_serviceProvider->mappingManager();
    qDebug() << "Mapping Manager supported types and connectivity modes:" << m_mappingManager->supportedMapTypes() << m_mappingManager->supportedConnectivityModes();


    GeoMap *map = new GeoMap(m_mappingManager);
    m_view->setMap(map);

    if(m_map) {
        delete m_map;
    }
    m_map = map;

    connect(map, SIGNAL(clicked(QGeoMapObject*)), SLOT(geoObjectClicked(QGeoMapObject*)));
    connect(map, SIGNAL(panned()), SLOT(disableTracking()));

    // Current Position
    QPixmap positionMarker(":/currentLoc.png");
    m_positionMarker = new QGeoMapPixmapObject(QGeoCoordinate(0,0), QPoint(-positionMarker.width()/2,-positionMarker.height()/2), positionMarker);
    m_positionMarker->setVisible(false);
    m_positionMarker->setZValue(1);
    map->addMapObject(m_positionMarker);

    rowsInserted(QModelIndex(), 0, m_model->rowCount() - 1);
}

void MobilityMap::positionUpdated(const QGeoPositionInfo &info)
{
    if (!m_positionMarker->isVisible()) {
        m_positionMarker->setVisible(true);
    }
    GMWMap::positionUpdated(info);
    m_positionMarker->setCoordinate(info.coordinate());
}

void MobilityMap::moveTo(const QGeoCoordinate &point)
{
    animatedPanTo(point);
}

void MobilityMap::routeTo(const QGeoCoordinate &point)
{
    qDebug() << m_view->size();
    //Navigator *navigator = new Navigator(m_serviceProvider->routingManager(), m_mapWidget, point);
    //m_lastNavigator = navigator;

    //connect(navigator, SIGNAL(routingError(QGeoRouteReply::Error,QString)), this, SLOT(showErrorMessage(QGeoRouteReply::Error,QString)));

    //mapsWidget->statusBar()->setText("Routing...");
    //mapsWidget->statusBar()->show();

    //navigator->start();
}

void MobilityMap::zoomIn()
{
    m_map->setZoomLevel(m_map->zoomLevel() + 1);
}

void MobilityMap::zoomOut()
{
    m_map->setZoomLevel(m_map->zoomLevel() - 1);
}

void MobilityMap::zoomTo(const QGeoBoundingBox &bounds)
{
    m_map->fitInViewport(bounds);

    // This works around some strange behavior in QGraphicsGeoMap:
    // fitInViewPort sometimes does not zoom in into the map but only centers
    // to the given bounds. We zoom in to level 12 for now as this
    // seems to be best suited to display an entire city
    m_map->setZoomLevel(12);

}

void MobilityMap::refresh()
{

}

void MobilityMap::rowsInserted(const QModelIndex &parent, int start, int end)
{
    Q_UNUSED(parent)

    qDebug() << "Inserting markers into map" << start << "-" << end << "Current markers:" << m_items.count() << "New markers:" << end-start+1;
    QTime timer;
    QTime ges;
    ges.start();
    timer.start();
    for(int i = start; i <= end; ++i) {
        GMWItem *item = m_model->data(m_model->index(i, 0), Qt::UserRole).value<GMWItem*>();
        GMWMarker *marker = new GMWMarker(item);
        marker->setCoordinate(item->location());
        m_map->addMapObject(marker);
        m_items.insert(item, marker);
//        qDebug() << "inserted item" << i << timer.restart() << "ms";
    }
    qDebug() << "Markers inserted; Markers:" << m_items.count() << "time:" << ges.elapsed();
}

void MobilityMap::rowsAboutToBeRemoved(const QModelIndex &parent, int start, int end)
{
    Q_UNUSED(parent)

    qDebug() << "Removing markers from map" << start << "-" << end << "Current markers:" << m_items.count() << "Markers to remove:" << end-start+1;
    for(int i = start; i <= end; ++i) {
        QModelIndex index = m_model->index(i, 0);
        GMWItem *item = m_model->data(index, Qt::UserRole).value<GMWItem*>();
        GMWMarker *marker = m_items.value(item);
        m_map->removeMapObject(marker);
        m_items.remove(item);
        delete marker;
    }
//    qDebug() << "items removed; Items:" << m_items.count();
}

void MobilityMap::geoObjectClicked(QGeoMapObject *object)
{
    GMWMarker *marker = dynamic_cast<GMWMarker*>(object);
    if (!marker) return;

    GMWItem *item = m_items.key(marker);
    emit objectClicked(item);
}

void MobilityMap::animatedPanTo(const QGeoCoordinate &center)
{
    if (!m_map) return;

    QPropertyAnimation *latAnim = new QPropertyAnimation(m_map, "centerLatitude");
    latAnim->setEndValue(center.latitude());
    latAnim->setDuration(200);
    QPropertyAnimation *lonAnim = new QPropertyAnimation(m_map, "centerLongitude");
    lonAnim->setEndValue(center.longitude());
    lonAnim->setDuration(200);

    QParallelAnimationGroup *group = new QParallelAnimationGroup;
    group->addAnimation(latAnim);
    group->addAnimation(lonAnim);
    group->start(QAbstractAnimation::DeleteWhenStopped);
}
