#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QDebug>
#include <QtWebKit/QWebView>
#include <QWebFrame>
#include <QMenuBar>

#include <QApplication>
#include <QMessageBox>
#include <QDir>
#include <QNetworkDiskCache>
#include <QToolButton>
#include <QResizeEvent>
#include <QTimer>

#include <QStringBuilder>

#include <QSettings>
#include <QShortcut>

#include <pickselectoraction.h>

#ifdef Q_WS_MAEMO_5
    #include <QMaemo5InformationBox>
    #define BASECLASSWND Maemo5Wnd
#else
    #define BASECLASSWND QMainWindow
#endif

#include "geocoderequester.h"
#include "googledirections.h"

#include "finddlg.h"
#include "directionsdlg.h"
#include "currdirectionsdlg.h"

MainWindow::MainWindow(QWidget *parent) :
    BASECLASSWND(parent),
    ui(new Ui::MainWindow)
{
    m_pViewGroup = NULL;
    m_pRequester = NULL;
    m_pNetAccessMngr = NULL;
    m_pDirectionsDlg = NULL;
    m_pCurrDirectionsDlg = NULL;

    m_pGoogleDirections = NULL;
    m_pAutoRotateAction = NULL;
    m_pTrafficLayer = NULL;
    m_pCyclingLayer = NULL;

    m_isAutoCenter = true;
    m_isAutoRotate = false;
    m_angle =0;

    for ( int c=0; c<4; c++ ) m_pToolButtons[c] = NULL;

#ifdef Q_WS_MAEMO_5
    m_pSourceUpdatesBtt = NULL;
#endif

    ui->setupUi(this);

    m_startAddress.setCoordinates( 35.8185 ,  -78.8641);

    init();
    // createMenu();
}

MainWindow::~MainWindow() {
    delete ui;

    if ( m_pRequester ) { delete m_pRequester; m_pRequester=NULL; }
    if ( m_pGoogleDirections ) { delete m_pGoogleDirections; m_pGoogleDirections=NULL; }
    if ( m_pDirectionsDlg ) { delete m_pDirectionsDlg; m_pDirectionsDlg=NULL; }
    if ( m_pCurrDirectionsDlg ) { delete m_pCurrDirectionsDlg; m_pCurrDirectionsDlg=NULL; }
}

void MainWindow::init() {
#ifdef Q_WS_MAEMO_5
    QString cachePath = "/home/user/.geeps/cache";

    setAttribute(Qt::WA_Maemo5StackedWindow);
    setWindowFlags(windowFlags() | Qt::Window);

#else
    QString cachePath = QApplication::applicationDirPath() + "/cache";

#endif

    QDir dir(cachePath);
    if ( !dir.exists() ) {
        if ( !dir.mkpath(dir.path()) ) {
            qCritical() << "error creating app folder: " << dir.path();
        }
    }

    qDebug() << "cache folder:" << cachePath;
    QNetworkDiskCache *pCache = new QNetworkDiskCache(this);
    pCache->setCacheDirectory(cachePath);

    dir = QDir(cachePath + "/persistent");
    if ( !dir.exists() ) {
        if ( !dir.mkpath(dir.path()) ) {
            qCritical() << "error creating app folder: " << dir.path();
        }
    }
    QWebSettings::enablePersistentStorage(dir.path());

    m_pNetAccessMngr = new QNetworkAccessManager(this);
    m_pNetAccessMngr->setCache(pCache);

    m_pRequester = new GeocodeRequester(m_pNetAccessMngr, this);
    connect(m_pRequester, SIGNAL(finish(bool,QString)), SLOT(onFindDestinationAddress(bool,QString)));

    m_pGoogleDirections = new GoogleDirections(m_pNetAccessMngr, this);
    connect(m_pGoogleDirections, SIGNAL(finish(bool,QString)), SLOT(onDirectionsFinished(bool,QString)));

    // connect(m_pNetAccessMngr, SIGNAL(finished(QNetworkReply*)), SLOT(onRequestfinished(QNetworkReply*)) );

    QWebPage *pPage = ui->webView->page();
    QWebFrame *pFrame = pPage->mainFrame();

    pFrame->setScrollBarPolicy(Qt::Vertical, Qt::ScrollBarAlwaysOff);
    pFrame->setScrollBarPolicy(Qt::Horizontal, Qt::ScrollBarAlwaysOff);

    pPage->setNetworkAccessManager(m_pNetAccessMngr);
    pPage->settings()->setAttribute(QWebSettings::DeveloperExtrasEnabled, true);

    dir = QDir(cachePath + "/local");
    if ( !dir.exists() ) {
        if ( !dir.mkpath(dir.path()) ) {
            qCritical() << "error creating app folder: " << dir.path();
        }
    }
    pPage->settings()->setLocalStoragePath(dir.path());

    // offline?
    dir = QDir(cachePath + "/offline");
    if ( !dir.exists() ) {
        if ( !dir.mkpath(dir.path()) ) {
            qCritical() << "error creating app folder: " << dir.path();
        }
    }
    QWebSettings::setOfflineStoragePath(dir.path());

    pPage->settings()->setAttribute(QWebSettings::OfflineStorageDatabaseEnabled, true);
    pPage->settings()->setAttribute(QWebSettings::OfflineWebApplicationCacheEnabled, true);
    connect(pFrame, SIGNAL(javaScriptWindowObjectCleared()), SLOT(javaScriptWindowObjectCleared()));

#ifdef Q_WS_MAEMO_5
    connect(&m_geoLocation, SIGNAL(updated()), SLOT(onPositionUpdated()));
    grabZoomKeys(true);
#endif

    MouseSuppressor *pSupperssor = new MouseSuppressor(ui->webView);
    connect(pSupperssor, SIGNAL(click(QMouseEvent*)), SLOT(onClick(QMouseEvent*)));

    // tool buttons
    m_pToolButtons[0] = new ToolButton(QIcon(":/images/west.png"), 0, this);
    m_pToolButtons[1] = new ToolButton(QIcon(":/images/north.png"), 0, this);
    m_pToolButtons[2] = new ToolButton(QIcon(":/images/east.png"), 0, this);
    m_pToolButtons[3] = new ToolButton(QIcon(":/images/south.png"), 0, this);
    connect(m_pToolButtons[0], SIGNAL(clicked()), SLOT(onWest()));
    connect(m_pToolButtons[1], SIGNAL(clicked()), SLOT(onNorth()));
    connect(m_pToolButtons[2], SIGNAL(clicked()), SLOT(onEast()));
    connect(m_pToolButtons[3], SIGNAL(clicked()), SLOT(onSouth()));
    for ( int c=0; c<4; c++ ) m_pToolButtons[c]->setVisible(false);

    connect(ui->webView, SIGNAL(loadFinished(bool)), SLOT(onLoadFinished(bool)));
    load();
}

void MainWindow::load() {
    ui->webView->setUrl( QUrl::fromUserInput("http://www.crochik.com/gps2.html") );
}

void MainWindow::onLoadFinished(bool isSuccess) {
    if ( !isSuccess ) {
        if ( QMessageBox::question(
            this, "Failed to Load Map", "Would you like to try again?",
            QMessageBox::Yes, QMessageBox::No)
            == QMessageBox::Yes
        ){
            load();
            return;
        }
        // abort
        close();
        return;
    }

    // ready!
    qDebug() << "loaded html page";
    ui->webView->disconnect(this, SLOT(onLoadFinished(bool)));
    createMenu();
}

void MainWindow::createMenu() {
    QSettings settings;

    // views
    int viewMode = settings.value("Mode", 1).toInt();
    m_pViewGroup = new QActionGroup(this);
    m_pViewGroup->setExclusive(true);

    QAction *pAction;
    pAction = new QAction("Hybrid", m_pViewGroup); pAction->setCheckable(true); pAction->setChecked(viewMode==0);
    pAction = new QAction("Road Map", m_pViewGroup); pAction ->setCheckable(true); pAction->setChecked(viewMode==1);
    pAction = new QAction("Satellite", m_pViewGroup); pAction->setCheckable(true); pAction->setChecked(viewMode==2);
    pAction = new QAction("Terrain", m_pViewGroup); pAction->setCheckable(true); pAction->setChecked(viewMode==3);
    menuBar()->addActions(m_pViewGroup->actions());
    connect(m_pViewGroup, SIGNAL(triggered(QAction*)), this, SLOT(onViewChanged(QAction*)));
    setView(viewMode);

    // menuBar()->addAction("Reload", this, SLOT(reload()));
    // menuBar()->addAction("Find Address", this, SLOT(find()));

    menuBar()->addAction("Directions", this, SLOT(directions()));

    // menuBar()->addAction("Reset Tracking", this, SLOT(resetRoute()));

    /*
    m_pAutoRotateAction = menuBar()->addAction("Auto Rotate", this, SLOT(onAutoRotate()));
    m_pAutoRotateAction->setCheckable(true);
    m_pAutoRotateAction->setChecked(false);
    */

    bool traffic = settings.value("TrafficLayer", false).toBool();
    bool bicycle = settings.value("BicyclingLayer", false).toBool();
    m_pTrafficLayer = menuBar()->addAction("Traffic Layer", this, SLOT(onTrafficLayer()));
    m_pTrafficLayer->setCheckable(true);
    m_pTrafficLayer->setChecked(traffic);

    m_pCyclingLayer = menuBar()->addAction("Bicycling Layer", this, SLOT(onBicyclingLayer()));
    m_pCyclingLayer->setCheckable(true);
    m_pCyclingLayer->setChecked(bicycle);

    m_pAccuracyAction = menuBar()->addAction("Show Accuracy", this, SLOT(onShowAccuracy()));
    m_pAccuracyAction->setCheckable(true);
    m_pAccuracyAction->setChecked(false);

    if ( traffic ) onTrafficLayer();
    if ( bicycle ) onBicyclingLayer();


#ifdef Q_WS_MAEMO_5
    addFullscreenToMenu();
    addOrientationToMenu();
    // new QShortcut(QKeySequence("F"), this, SLOT(onFullscreen()));

    int updates = settings.value("UpdateMode", 0).toInt();

    m_pSourceUpdatesBtt = new PickSelectorAction(this);
    m_pSourceUpdatesBtt->setText("Position Updates");
    m_pSourceUpdatesBtt->items() << "Off" << "every second" << "every 2 seconds" << "every 5 seconds"
            << "every 10 seconds" << "every 15 seconds" << "every 30 seconds";

            // << "10 per second" << "5 per second"
            // << "every minute" << "every 5 minutes" << "every 10 minutes" << "every 15 minutes" << "every half hour" << "every hour";

    m_pSourceUpdatesBtt->setSelectedIndex(updates);

    connect(m_pSourceUpdatesBtt, SIGNAL(selectionChanged(int)), this, SLOT(onSourceUpdates(int)));
    menuBar()->addAction(m_pSourceUpdatesBtt);
    onSourceUpdates(updates);
#endif
}

void MainWindow::onShowAccuracy() {
    if ( !m_pAccuracyAction->isChecked() ) runJavascript("radius=null;");
}

void MainWindow::onTrafficLayer() {
    QSettings settings;
    settings.setValue("TrafficLayer", m_pTrafficLayer->isChecked());

    if ( m_pTrafficLayer->isChecked() ) {
        runJavascript("trafficLayer = new google.maps.TrafficLayer(); trafficLayer.setMap(map);");
        return;
    }

    runJavascript("if (trafficLayer) { trafficLayer.setMap(null); trafficLayer=null; }");
}

void MainWindow::onBicyclingLayer() {
    QSettings settings;
    settings.setValue("BicyclingLayer", m_pCyclingLayer->isChecked());

    if ( m_pCyclingLayer->isChecked() ) {
        runJavascript("bikeLayer = new google.maps.BicyclingLayer(); bikeLayer.setMap(map);");
        return;
    }

    runJavascript("if (bikeLayer) { bikeLayer.setMap(null); bikeLayer=null; }");
}

void MainWindow::onAutoRotate() {
    m_isAutoRotate = m_pAutoRotateAction->isChecked();
    if ( !m_isAutoRotate ) {
        // north
        QString script = QString("document.getElementById(\"map_canvas\").style[\"WebkitTransform\"] = \"rotate(%1deg)\";").arg(0);
        runJavascript(script);

        runJavascript("document.getElementById(\"map_canvas\").style[\"width\"] = \"100%\";");
        runJavascript("document.getElementById(\"map_canvas\").style[\"height\"] = \"100%\";");
    } else {
        m_angle = 0;
        // resize to square
        QString sz = QString::number(ui->webView->width()>ui->webView->height() ? ui->webView->height() : ui->webView->width());
        runJavascript(QString("document.getElementById(\"map_canvas\").style[\"width\"] = \"%1px\";").arg(sz));
        runJavascript(QString("document.getElementById(\"map_canvas\").style[\"height\"] = \"%1px\";").arg(sz));
    }
}

void MainWindow::onSourceUpdates(int index) {
    int updates = 0;
    switch (index) {
        // case 1: updates = 10*60*60; break;
        // case 2: updates = 5*60*60; break;
        case 1: updates = 60*60; break;
        case 2: updates = 30*60; break;
        case 3: updates = 12*60; break;
        case 4: updates = 6*60; break;
        case 5: updates = 4*60; break;
        case 6: updates = 2*60; break;
        case 7: updates = 60; break;
        case 8: updates = 12; break;
        case 9: updates = 6; break;
        case 10: updates = 4; break;
        case 11: updates = 2; break;
        case 12: updates = 1; break;
    }

    QSettings settings;
    settings.setValue("UpdateMode", index);

#ifdef Q_WS_MAEMO_5
    m_geoLocation.setUpdatesPerHour(updates);
#endif
}

void MainWindow::onRequestfinished(QNetworkReply*) {
    // qDebug() << "request finished";
}

void MainWindow::onClick(QMouseEvent *) {
    if ( m_pToolButtons[0] && !m_pToolButtons[0]->isVisible() ) m_timerCount = 0;

    // show buttons
    for ( int c=0; c<4; c++ ) if ( m_pToolButtons[c] ) m_pToolButtons[c]->setVisible(true);
    incrementTimer();
}

void MainWindow::incrementTimer() {
    // start timer
    m_timerCount++;
    QTimer::singleShot(5000, this, SLOT(onHideButtons()));
}

void MainWindow::onHideButtons() {
    if ( --m_timerCount ) return;

    for ( int c=0; c<4; c++ ) if ( m_pToolButtons[c] ) m_pToolButtons[c]->setVisible(false);
    showFullScreen();
}

void MainWindow::onViewChanged(QAction* pAction) {
    setView(m_pViewGroup->actions().indexOf(pAction));
}

void MainWindow::setView(int index) {
    QString value;
    switch ( index ) {
        case 0: value = "HYBRID"; break;
        case 1: value = "ROADMAP"; break;
        case 2: value = "SATELLITE"; break;
        case 3: value = "TERRAIN"; break;
        default:
            return;
    }

    QSettings settings;
    settings.setValue("Mode", index);
    runJavascript("map.setMapTypeId(google.maps.MapTypeId." + value + ");");
}

void MainWindow::runJavascript(QString script) {
    QWebFrame *pFrame = ui->webView->page()->mainFrame();
    qDebug() << script;
    pFrame->evaluateJavaScript(script);
}

void MainWindow::resetRoute() {
    runJavascript("if ( polyLine ) polyLine.setMap(null); polyLine=null;");
}

void MainWindow::directions() {
    if ( m_pCurrDirectionsDlg ) {
        showCurrentDirections();
        return;
    }

    if ( !m_pDirectionsDlg ) {
        m_pDirectionsDlg = new DirectionsDlg(this);
#ifdef Q_WS_MAEMO_5
        m_pDirectionsDlg->setAttribute(Qt::WA_Maemo5StackedWindow);
        m_pDirectionsDlg->setWindowFlags(Qt::Window);
#endif
    }

    m_pDirectionsDlg->setStartAddress(m_startAddress.description()); // !!!!
    m_pDirectionsDlg->setDestinationAddress(m_destination.description());

#ifdef Q_WS_MAEMO_5
    m_pDirectionsDlg->setStartAddressToCurrentPos(m_geoLocation.isUpdating());
#endif

    if ( m_pDirectionsDlg->exec() != QDialog::Accepted ) return;

    m_startAddress.setDescription(m_pDirectionsDlg->startAddress());
    m_startAddress.setCoordinates(0,0); // ???

    /*
    m_destination.setDescription(m_pDirectionsDlg->destinationAddress());
    m_destination.setCoordinates(0,0); // ???
    */

    showProgressIndicator(true);
    m_pRequester->query(m_pDirectionsDlg->destinationAddress());
}

void MainWindow::onDirectionsFinished(bool isSuccess, QString errorMessage) {
    showProgressIndicator(false);

    if ( !isSuccess ) {
        QMessageBox::warning(this, "Directions", errorMessage);
        return;
    }

    /*
    Directions& dir(m_pGoogleDirections->m_directions[0]);
    qDebug() << "Direction: " << dir.m_summary;
    */

    if ( m_pCurrDirectionsDlg ) {
        delete m_pCurrDirectionsDlg;
        m_pCurrDirectionsDlg = NULL;
    }

    //
    // todo: fix to show the alternative selected on the dialog
    // runJavascript("getDirections(\"" + m_startAddress.description() + "\", \"" + m_destination.description() + "\");");
    getDirectionsOnMap();

    showCurrentDirections();

    // only the turning points
    /*
    QString script;
    script.append("if (directionsPoly) { directionsPoly.setMap(null); directionsPoly = null; }");
    script.append("var arr = new Array();");
    for ( int c=0; c<dir.m_steps.length(); c++ ) {
        Step step = dir.m_steps[c];
        script.append(QString("arr.push( new google.maps.LatLng(%1, %2));").arg(QString::number(step.latitude()), QString::number(step.longitude())));
    }
    script.append("directionsPoly = new google.maps.Polyline({ map: map, path: arr, strokeColor: \"#0000ff\", strokeOpacity: .7, strokeWeight: 5});");
    runJavascript(script);
    */
}

void MainWindow::getDirectionsOnMap() {
    QString script = "var request = {";

    // m_pDirectionsDlg->showAlternatives();
    // m_pDirectionsDlg->units();

    script += "travelMode: google.maps.DirectionsTravelMode." % m_pDirectionsDlg->mode().toUpper() % ",";
    foreach (QString str, m_pDirectionsDlg->avoid() ) {
        if ( str=="highways" ) script += "avoidHighways: true,";
        if ( str=="tolls" ) script += "avoidTolls: true,";
    }

    if ( m_pDirectionsDlg->showAlternatives() ) script += "provideRouteAlternatives: true,";

    script +=
            "origin: \"" % m_startAddress.description()% "\","
            "destination: \"" % m_destination.description()% "\"};"

        "directionsService.route("
            "request,"
            "function(result, status) {"
                "if (status == google.maps.DirectionsStatus.OK) {"
                      "directionsDisplay.setMap(map);"
                      "directionsDisplay.setDirections(result);"
                      "directionsDisplay.setRouteIndex(0);"
                "}"
            "}"
        ");";

    runJavascript(script);
}

void MainWindow::resetDirections() {
    resetRoute();

    runJavascript("directionsDisplay.setMap(null);");

    if ( m_pCurrDirectionsDlg ) {delete m_pCurrDirectionsDlg; m_pCurrDirectionsDlg = NULL;}
}

void MainWindow::showCurrentDirections() {
    if ( !m_pCurrDirectionsDlg ) {
        m_pCurrDirectionsDlg = new CurrDirectionsDlg(m_pGoogleDirections->m_directions, this);
        m_pCurrDirectionsDlg->setWindowTitle(m_destination.description());

#ifdef Q_WS_MAEMO_5
        m_pCurrDirectionsDlg->setAttribute(Qt::WA_Maemo5StackedWindow);
        m_pCurrDirectionsDlg->setWindowFlags(Qt::Window);
#endif
    }

    int res = m_pCurrDirectionsDlg->exec();
    runJavascript(QString("directionsDisplay.setRouteIndex(%1);").arg(m_pCurrDirectionsDlg->selectedIndex()));

    // todo: fix to show the alternative selected on the dialog
    // runJavascript("getDirections(\"" + m_startAddress.description() + "\", \"" + m_destination.description() + "\");");

#ifdef GOOGLEMAPS_JSON
    QString script = "var json = " % m_pGoogleDirections->m_directions[m_pCurrDirectionsDlg->selectedIndex()].m_json % ";"
        "directionsDisplay.setDirections(json); alert(\"ok!\");";

    runJavascript(script);
#endif

    if (  res!= QMessageBox::Accepted ) {
        return;
    }

    switch ( m_pCurrDirectionsDlg->m_button ) {
        case CurrDirectionsDlg::ButtonClear:
            resetDirections();
            break;

        case CurrDirectionsDlg::ButtonNew:
            resetDirections();
            directions();
            break;

        case CurrDirectionsDlg::ButtonRecalculate:
            resetDirections();
            showProgressIndicator(true);
            m_pRequester->query(m_pDirectionsDlg->destinationAddress());
            break;

        default:
            // todo: fix to show the alternative selected on the dialog
            // runJavascript("getDirections(\"" + m_startAddress.description() + "\", \"" + m_destination.description() + "\");");
            break;
    }
}

void MainWindow::onFindDestinationAddress(bool isSuccess, QString errorMessage) {
    showProgressIndicator(false);

    if ( !isSuccess ) {
        QMessageBox::warning(this, "Directions - Destination", errorMessage);
        return;
    }

    if ( m_pRequester->m_address.length()!=1 ) {
        FindDlg dlg(*m_pRequester, m_pDirectionsDlg->destinationAddress(), this);
        dlg.setWindowTitle("Destination Address");
#ifdef Q_WS_MAEMO_5
        dlg.setAttribute(Qt::WA_Maemo5StackedWindow);
        dlg.setWindowFlags(Qt::Window);
#endif
        if ( dlg.exec()!=QDialog::Accepted ) return;
        m_destination = dlg.m_selectedAddress;

    } else {
        m_destination = m_pRequester->m_address[0];

    }

    if ( m_pDirectionsDlg ) m_pDirectionsDlg->setDestinationAddress(m_destination.description());

    // save poi ?
    // ...

    resetRoute();

    showProgressIndicator(true);
    m_pGoogleDirections->query(
        m_startAddress.description(),
        m_destination.description(),
        m_pDirectionsDlg->mode(),
        m_pDirectionsDlg->showAlternatives(),
        m_pDirectionsDlg->units(),
        m_pDirectionsDlg->avoid()
    );
}

void MainWindow::find() {
    FindDlg dlg(m_pNetAccessMngr, this);
#ifdef Q_WS_MAEMO_5
    dlg.setAttribute(Qt::WA_Maemo5StackedWindow);
    dlg.setWindowFlags(Qt::Window);
#endif

    if ( dlg.exec()!=QDialog::Accepted ) return;

    QWebFrame *pFrame = ui->webView->page()->mainFrame();
    pFrame->evaluateJavaScript(QString("addMark(%1, %2, \"%3\");").arg(
            QString::number(dlg.m_selectedAddress.latitude()),
            QString::number(dlg.m_selectedAddress.longitude()),
            dlg.m_selectedAddress.description()
        )
    );
}

void MainWindow::reload() {
    ui->webView->reload();
    // ui->webView->setUrl( QUrl::fromUserInput("http://192.168.1.100:8080/gps.html") );
}

void MainWindow::javaScriptWindowObjectCleared() {
#ifdef Q_WS_MAEMO_5
    QWebFrame *pFrame = ui->webView->page()->mainFrame();
    pFrame->addToJavaScriptWindowObject("crochik", &m_geoLocation);
#endif
}

void MainWindow::changeEvent(QEvent *e) {
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void MainWindow::onPositionUpdated() {
#ifdef Q_WS_MAEMO_5
    Location last = m_startAddress;

    // reset start
    m_startAddress.setCoordinates( m_geoLocation.latitude(), m_geoLocation.longitude());
    m_startAddress.setDescription(QString("%1 %2").arg(QString::number(m_startAddress.latitude()), QString::number(m_startAddress.longitude())));

    // m_startAddress.setCoordinates(last.latitude()+.001, last.longitude());

    if ( m_isAutoCenter && m_isAutoRotate ) {
        double dx = m_geoLocation.longitude() - last.longitude();
        double dy = m_geoLocation.latitude() - last.latitude();

        int angle = 0;
        if ( (dx>0 && dy>0 && dx>dy) || (dx<0 && dy<0 && dx<dy) || (dx>0 && dy<0 && dx>-dy) || (dx<0 && dy>0 && dx<-dy) ) {
            // horizontal
            angle = dx>0 ? 90 : 270;
        } else {
            // vertical
            angle = dy<0 ? 0 : 180;
        }

        if ( dx || dy ) {
            if ( angle==m_angle ) {
                QString script = QString("document.getElementById(\"map_canvas\").style[\"WebkitTransform\"] = \"rotate(%1deg)\";").arg(angle);
                runJavascript(script);
            } else {

                // resetRoute();
            }
            m_angle = angle;
        }
    }

    runJavascript(
        QString("updateCenter(%1, %2, %3, %4);").arg(
            m_isAutoCenter?"true":"false",
            QString::number(m_geoLocation.latitude()),
            QString::number(m_geoLocation.longitude()),
            m_pAccuracyAction->isChecked() ? QString::number(m_geoLocation.horizontalAccuracy()) : "0"
        )
    );

    if ( !m_pCurrDirectionsDlg ) return;

    runJavascript(
        QString("track(%1, %2);").arg(
            QString::number(m_startAddress.latitude()),
            QString::number(m_startAddress.longitude())
        )
    );
#endif
}

void MainWindow::resizeEvent(QResizeEvent *pEvent) {
    BASECLASSWND::resizeEvent(pEvent);

    if ( m_pToolButtons[0] ) m_pToolButtons[0]->move(0, (pEvent->size().height()-m_pToolButtons[0]->height())/2);
    if ( m_pToolButtons[1] ) m_pToolButtons[1]->move((pEvent->size().width()-m_pToolButtons[1]->width())/2,0);
    if ( m_pToolButtons[2] ) m_pToolButtons[2]->move(pEvent->size().width()-m_pToolButtons[2]->width(), (pEvent->size().height()-m_pToolButtons[2]->height())/2);
    if ( m_pToolButtons[3] ) m_pToolButtons[3]->move((pEvent->size().width()-m_pToolButtons[3]->width())/2, pEvent->size().height()-m_pToolButtons[3]->height());
}

void MainWindow::keyPressEvent(QKeyEvent *event) {
    QString str;
    switch ( event->key() ) {
        case Qt::Key_F7:
            str = "map.setZoom(map.getZoom()+1);";
            break;

        case Qt::Key_F8:
            str = "map.setZoom(map.getZoom()-1);";
            break;

        case Qt::Key_Left:
            onWest();
            return;

        case Qt::Key_Right:
            onEast();
            return;

        case Qt::Key_Up:
            onNorth();
            return;

        case Qt::Key_Down:
            onSouth();
            return;

        case Qt::Key_Enter:
            m_isAutoCenter = true;
            onPositionUpdated();
            // str = "updateCenter(true);";
            return;

        case Qt::Key_F:
            onFullscreen();
            return;

        case Qt::Key_0: // = 0x30,
        case Qt::Key_1: // = 0x31,
        case Qt::Key_2: // = 0x32,
        case Qt::Key_3: // = 0x33,
        case Qt::Key_4: // = 0x34,
        case Qt::Key_5: // = 0x35,
        case Qt::Key_6: // = 0x36,
        case Qt::Key_7: // = 0x37,
        case Qt::Key_8: // = 0x38,
        case Qt::Key_9: // = 0x39,
            str = QString("map.setZoom(%1);").arg((event->key()-Qt::Key_0)*2);
            break;

        default:
            return;
    }

    runJavascript(str);
}

void MainWindow::onWest() {
    incrementTimer();
    m_isAutoCenter = false;
    runJavascript(QString("map.panBy(%1,0);").arg(-ui->webView->width()/2));
}
void MainWindow::onEast() {
    incrementTimer();
    m_isAutoCenter = false;
    runJavascript(QString("map.panBy(%1,0);").arg(ui->webView->width()/2));
}
void MainWindow::onNorth() {
    incrementTimer();
    m_isAutoCenter = false;
    runJavascript(QString("map.panBy(0,%1);").arg(-ui->webView->height()/2));
}
void MainWindow::onSouth() {
    incrementTimer();
    m_isAutoCenter = false;
    runJavascript(QString("map.panBy(0,%1);").arg(ui->webView->height()/2));
}

void MainWindow::showProgressIndicator(bool show) {
#ifdef Q_WS_MAEMO_5
    // http://qt.nokia.com/doc/qt-maemo-4.6/maemo5-windowstates.html
    this->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, show);
#endif

    QApplication::processEvents();
}

MouseSuppressor::MouseSuppressor(QWebView *pView)
    : QObject(pView)
{
    pView->installEventFilter(this);
}

bool MouseSuppressor::eventFilter(QObject *, QEvent *e) {
    switch (e->type()) {
        case QEvent::MouseButtonPress:
            if (static_cast<QMouseEvent *>(e)->button() == Qt::LeftButton) m_isMousePressed = true;
            return true;

        case QEvent::MouseButtonRelease:
            if ( m_isMousePressed ) {
                QMouseEvent *pEvent = static_cast<QMouseEvent *>(e);
                click(pEvent);
                m_isMousePressed = false;
            }
            return true;

        case QEvent::MouseMove:
            return true;

        default:
            break;
    }
    return false;
}

