#include "googledirections.h"

#include <QUrl>
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QDebug>
#include <QMessageBox>

#include <QDomDocument>
#include <QStringBuilder>

GoogleDirections::GoogleDirections(QNetworkAccessManager *pNetAccessMngr, QObject *parent) :
    QObject(parent)
{
    m_pNetAccessMngr = pNetAccessMngr;
    m_pReply = NULL;

}

GoogleDirections::~GoogleDirections() {
    if ( m_pReply ) { delete m_pReply; m_pReply = NULL; }
}

bool GoogleDirections::query(QString origin, QString destination,
    QString mode/*="driving"*/, bool alternatives/*=false*/, QString units/*="metric"*/,
    QStringList avoid/*=QStringList()*/)
{
    QUrl url("http://maps.googleapis.com/maps/api/directions/xml");
    url.addQueryItem("sensor", "false");
    url.addQueryItem("origin", origin);
    url.addQueryItem("destination", destination);

    /*
    * driving (default) indicates standard driving directions using the road network.
    * walking requests walking directions via pedestrian paths & sidewalks (where available).
    * bicycling requests bicycling directions via bicycle paths & preferred streets (currently only available in the US).
    */
    url.addQueryItem("mode", mode);
    url.addQueryItem("alternatives", alternatives?"true":"false");

    /*
    * metric specifies usage of the metric system. Textual distances are returned using kilometers and meters.
    * imperial specifies usage of the Imperial (English) system. Textual distances are returned using miles and feet.
    */
    url.addQueryItem("units", units);

    /*
    * avoid=tolls
    * avoid=highways
    */
    foreach ( QString str, avoid ) url.addQueryItem("avoid", str);

    qDebug() << QString(url.toEncoded());

    if ( m_pReply ) { delete m_pReply; m_pReply = NULL; }

    m_directions.clear();
    m_pReply = m_pNetAccessMngr->get(QNetworkRequest(url));
    connect(m_pReply, SIGNAL(finished()), this, SLOT(onFinished()));

    qDebug() << "Directions Requested";
    return true;
}

void GoogleDirections::onFinished() {
    qDebug() << "reply finished";
    QDomDocument doc;
    QString error;
    int line, col;
    if ( !doc.setContent(m_pReply, false, &error, &line, &col) ) {
        qCritical() << "DOM error: " << error << line << col;
        return;
    }

    qDebug() << doc.toString();

    QDomNodeList list = doc.elementsByTagName("status");
    if ( list.length()!=1 ) {
        qCritical() << "Error";
        return;
    }

    /* STATUS
    * OK indicates the response contains a valid result.
    * NOT_FOUND indicates at least one of the locations specified in the requests's origin, destination, or waypoints could not be geocoded.
    * ZERO_RESULTS indicates no route could be found between the origin and destination.
    * MAX_WAYPOINTS_EXCEEDED indicates that too many waypointss were provided in the request The maximum allowed waypoints is 8, plus the origin, and destination. ( Google Maps Premier customers may contain requests with up to 23 waypoints.)
    * INVALID_REQUEST indicates that the provided request was invalid.
    * OVER_QUERY_LIMIT indicates the service has received too many requests from your application within the allowed time period.
    * REQUEST_DENIED indicates that the service denied use of the directions service by your application.
    * UNKNOWN_ERROR indicates a directions request could not be processed due to a server error. The request may succeed if you try again.
    */
    QDomNode node = list.at(0);
    QString status = node.firstChild().nodeValue();
    qDebug()<< "-- status: " << status;
    if ( status == "ZERO_RESULTS")  {
        emit finish(false, "No results found.");
        return;
    }

    if ( status != "OK" ) {
        emit finish(false, "Service returned status: " + status);
        return;
    }

    list = doc.elementsByTagName("route");
    for ( int c=0; c<list.count(); c++ ) {
        QDomElement route = list.at(c).toElement();

        Directions dir;
        dir.m_summary = route.firstChildElement("summary").firstChild().nodeValue();
        dir.m_copyrights = route.firstChildElement("copyrights").firstChild().nodeValue();

        qDebug() << "Route: " << dir.m_summary;

#ifdef GOOGLEMAPS_JSON
        dir.m_json = "{"
            "\"status\": \"OK\","
            "\"routes\": [ {"
            "\"summary\": \"Summary\","
            "\"legs\": [";

        QString end_location_lat;
        QString end_location_lng;
#endif

        // steps
        QDomNodeList steps = route.elementsByTagName("step");
        for ( int d=0; d<steps.count(); d++ ) {
            node = steps.at(d).toElement();

            QDomNode startNode= node.firstChildElement("start_location");
            QDomNode durNode = node.firstChildElement("duration");
            QDomNode distNode = node.firstChildElement("distance");

            QString html_instructions = node.firstChildElement("html_instructions").firstChild().nodeValue();
            QString start_location_lat = startNode.firstChildElement("lat").firstChild().nodeValue();
            QString start_location_lng = startNode.firstChildElement("lng").firstChild().nodeValue();
            QString duration_value = durNode.firstChildElement("value").firstChild().nodeValue();
            QString duration_text = durNode.firstChildElement("text").firstChild().nodeValue();
            QString distance_value = distNode.firstChildElement("value").firstChild().nodeValue();
            QString distance_text = distNode.firstChildElement("text").firstChild().nodeValue();

#ifdef GOOGLEMAPS_JSON
            QDomNode endNode = node.firstChildElement("end_location");
            end_location_lat = endNode.firstChildElement("lat").firstChild().nodeValue();
            end_location_lng = endNode.firstChildElement("lng").firstChild().nodeValue();

            QDomNode polylineNode = node.firstChildElement("polyline");
            QString polyline_points = polylineNode.firstChildElement("points").firstChild().nodeValue();
            QString polyline_levels = polylineNode.firstChildElement("levels").firstChild().nodeValue();
            QString travel_mode = node.firstChildElement("travel_mode").firstChild().nodeValue();

            if (d>0) dir.m_json += ",";
            dir.m_json += "{"
                "\"travel_mode\": \"" % travel_mode % "\","
                "\"start_location\": {"
                "\"lat\": " % start_location_lat % ","
                "\"lng\": " % start_location_lng % "},"
                "\"end_location\": {"
                "\"lat\": " % end_location_lat % ","
                "\"lng\": " % end_location_lng % "},"
                "\"polyline\": {"
                "\"points\": \"" % polyline_points % "\","
                "\"levels\": \"" % polyline_levels % "\"},"
                "\"duration\": {"
                "\"value\": " % duration_value % ","
                "\"text\": \"" % duration_text % "\"},"
                "\"html_instructions\": \"html\","
                "\"distance\": {"
                "\"value\": " % distance_value % ","
                "\"text\": \"" % distance_text % "\"} "
                "}";
#endif

            Step step;
            step.setDescription(html_instructions);
            step.setCoordinates( start_location_lat.toDouble(), start_location_lng.toDouble() );
            step.m_duration = duration_value.toInt();
            step.m_durationStr = duration_text;
            step.m_distance = distance_value.toInt();
            step.m_distanceStr = distance_text;

            dir.m_steps.append(step);
            qDebug() << step.toString();
        }

#ifdef GOOGLEMAPS_JSON
        start_location_lat = QString::number(dir.m_steps[0].latitude()); //startNode.firstChildElement("lat").firstChild().nodeValue();
        start_location_lng = QString::number(dir.m_steps[0].longitude()); // startNode.firstChildElement("lng").firstChild().nodeValue();

        //end location = last end location

        dir.m_json += "],"
            "\"duration\": {"
            "\"value\": 0,"
            "\"text\": \"duration_text\"},"
            "\"start_location\": {"
            "\"lat\": " % start_location_lat % ","
            "\"lng\": " % start_location_lng % "},"
            "\"end_location\": {"
            "\"lat\": " % end_location_lat % ","
            "\"lng\": " % end_location_lng % "},"
            "\"start_address\": \"start_address\","
            "\"end_address\": \"end_address\","
            "\"via_waypoint\": []"
            "}";

        QDomNode overviewNode =  route.firstChildElement("overview_polyline");

        dir.m_json += "], "
            "\"copyrights\": \"" % dir.m_copyrights % "\", "
            "\"overview_polyline\": {"
            "\"points\": \"" % overviewNode.firstChildElement("points").firstChild().nodeValue() % "\", "
            "\"levels\": \"" % overviewNode.firstChildElement("levels").firstChild().nodeValue() % "\"}, "
            "\"warnings\": [], "
            "\"waypoint_order\": []"
            "}";

        qDebug() << dir.m_json;
#endif

        m_directions.append(dir);
    }
    // ...
    emit finish(true, QString::null);
}
