/*
 * Copyright (C) 2001, 2002 Anders Carlsson <andersca@gnu.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>

#include <gconf/gconf.h>
#include <libintl.h>
#include <glib/gi18n-lib.h>

#if WITH_GNOME
#include <libgnome/gnome-program.h>
#include <libgnomeui/gnome-ui-init.h>
#include <libgnomeui/gnome-app-helper.h>
#endif

#if USE_HILDON
#include <libosso.h>
#include <hildon/hildon-program.h>
#include <hildon/hildon-window.h>
#endif

#include "gconf-editor-application.h"
#include "gconf-stock-icons.h"
#include "gconf-editor-window.h"

gint
main (gint argc, gchar **argv)
{
#if WITH_GNOME
	GnomeProgram *program = NULL;
#elif USE_HILDON
	HildonProgram *program = NULL;
	osso_context_t *osso_context;
	GtkSettings *settings;
#endif
	GOptionContext *context;
	GtkWidget *window;
	char *description;
	static gchar **remaining_args = NULL;
	gchar *initial_key = NULL;
	GError *error;
		
	const GOptionEntry entries[] = 
	{
	  { G_OPTION_REMAINING, '\0', 0, G_OPTION_ARG_STRING_ARRAY, &remaining_args, NULL, N_("[KEY]") },
	  { NULL }
	};

	bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	description = g_strdup_printf ("- %s", _("Directly edit your entire configuration database"));
	context = g_option_context_new (description);
	g_free (description);

	g_option_context_add_main_entries (context, entries, GETTEXT_PACKAGE);

#if WITH_GNOME
	program = gnome_program_init (PACKAGE, VERSION, LIBGNOMEUI_MODULE,
									argc, argv,
									GNOME_PARAM_GOPTION_CONTEXT, context,								
									GNOME_PARAM_APP_DATADIR, DATADIR,
									GNOME_PARAM_NONE);
#else
	g_option_context_add_group (context, gtk_get_option_group (TRUE));
	if (!g_option_context_parse (context, &argc, &argv, &error)) {
		g_warning ("Cannot parse arguments: %s", error->message);
		g_error_free (error);
		return 1;
	}
	g_option_context_free (context);
#endif

	/* Register our stock icons */
	gconf_stock_icons_register ();

	window = gconf_editor_application_create_editor_window (GCONF_EDITOR_WINDOW_TYPE_NORMAL);
	gtk_widget_show_now (window);

#if USE_HILDON
	settings = gtk_widget_get_settings (GTK_WIDGET (window));
	g_object_set (settings, "gtk-button-images", FALSE, "gtk-menu-images", FALSE, NULL);

	osso_context = osso_initialize ("gconf_editor", VERSION, FALSE, NULL);

	g_set_application_name (_("Configuration Editor"));

	program = hildon_program_get_instance ();
	hildon_program_add_window (HILDON_PROGRAM (program), HILDON_WINDOW (window));
#endif

	/* get the key specified on the command line if any. Ignore the rest */
	initial_key = remaining_args != NULL ? remaining_args[0] : NULL;

	if (initial_key != NULL)
		gconf_editor_window_go_to (GCONF_EDITOR_WINDOW (window),initial_key);
	
	gtk_main ();

#if WITH_GNOME
	gnome_accelerators_sync();
#endif
	
#if WITH_GNOME || USE_HILDON
	g_object_unref (program);
#endif

#if USE_HILDON
	osso_deinitialize (osso_context);
#endif

	g_strfreev (remaining_args);

	return 0;
}
