/*
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007-2010 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "uilogview.h"
#include "uidrivinglog.h"
#include "uisettings.h"

/*******************************************************************
 *
 * Private definitions
 *
 *******************************************************************/

/*******************************************************************
 *
 * Private variables
 *
 *******************************************************************/

gint logdispcol=0xffff;

static
gchar *column_headers[] = {gettext_noop("Starttime"),
			   gettext_noop("Endtime"),
			   gettext_noop("Start lat"),
			   gettext_noop("Start lon"),
			   gettext_noop("Start place"),
			   gettext_noop("End lat"),
			   gettext_noop("End lon"),
			   gettext_noop("End place"),
			   gettext_noop("Distance"),
			   gettext_noop("Effectivetime"),
			   gettext_noop("Explanation"),
			   "icon", // for icon
			   "fancy", // for the fancy combined column
};

/*******************************************************************
 *
 * Prototypes of private functions
 *
 *******************************************************************/

static gboolean
visible_func (GtkTreeModel *model,
              GtkTreeIter  *iter,
              gpointer      data);
static
void cell_data_func (GtkTreeViewColumn *col,
		      GtkCellRenderer   *renderer,
		      GtkTreeModel      *model,
		      GtkTreeIter       *iter,
		     gpointer           user_data);
static
GtkTreeModel *create_and_fill_logmodel (void);

/*******************************************************************
 *
 * Private functions
 *
 *******************************************************************/

static gboolean
visible_func (GtkTreeModel *model,
              GtkTreeIter  *iter,
              gpointer      data)
{
  gchar *str;
  char *date;
  gdouble colval;
  gboolean visible = FALSE;
  gboolean compres;
  GList *filt;
  GList *lp;
/*   recordfilter_t *p; */
  gboolean first;
  gboolean and;

/*   filt=*(GList **)data; */

  gtk_tree_model_get (model, iter, LOG_COL_STARTTIME, &str, -1);

  if (str) {
    visible = TRUE;
  }

  return visible;
}

static
void cell_data_func (GtkTreeViewColumn *col,
		      GtkCellRenderer   *renderer,
		      GtkTreeModel      *model,
		      GtkTreeIter       *iter,
		      gpointer           user_data)
{
  double  value;
  gchar   buf[20];

  gtk_tree_model_get(model, iter, ((data_func_t *)user_data)->col, &value, -1);

  switch (((data_func_t *)user_data)->col) {
  default:
    g_snprintf(buf, sizeof(buf), ((data_func_t *)user_data)->format, value);
  }

  g_object_set(renderer, "text", buf, NULL);
}

static
void new_cell_data_func (GtkTreeViewColumn *col,
			 GtkCellRenderer   *renderer,
			 GtkTreeModel      *model,
			 GtkTreeIter       *iter,
			 gpointer           user_data)
{
  gchar *markup;
  gchar format_with_notes[] = "<span foreground=\"%s\">%s: %.1f %s, %s (%s)</span>\n<span foreground=\"%s\" size=\"xx-small\">%s %s %s %s</span>";
  gchar format_without_notes[] = "<span foreground=\"%s\">%s: %.1f %s, %s</span>\n<span foreground=\"%s\" size=\"xx-small\">%s %s %s %s</span>";

  char* starttime;
  char* endtime;
  char* startplace;
  char* endplace;
  char* explanation;
  double distance;
  double startlat;
  double startlon;
  double endlat;
  double endlon;
  char* effectivetime;
  sqlite_int64 id;

  /* Note: data in model is already in user units, no need to use conversion functions */
  gtk_tree_model_get(model, iter, 
		     LOG_COL_STARTTIME, &starttime, LOG_COL_ENDTIME, &endtime, 
		     LOG_COL_STARTLAT, &startlat, LOG_COL_STARTLON, &startlon,
		     LOG_COL_STARTPLACE, &startplace,
		     LOG_COL_ENDLAT, &endlat, LOG_COL_ENDLON, &endlon,
		     LOG_COL_ENDPLACE, &endplace, LOG_COL_DISTANCE, &distance,
		     LOG_COL_EFFECTIVETIME, &effectivetime, LOG_COL_EXPLANATION, &explanation,
		     -1);

  if (strlen(explanation) == 0) {
    markup=g_markup_printf_escaped(format_without_notes,
				   get_default_text_color(),
				   starttime, 
				   distance, 
				   ((curunit.lengthunit==SI)? _("km") : _("mi")),
				   effectivetime,
				   get_secondary_text_color(),
				   _("from"),
				   startplace,
				   _("to"),
				   endplace);
  }
  else {
    markup=g_markup_printf_escaped(format_with_notes,
				   get_default_text_color(),
				   starttime, 
				   distance, 
				   ((curunit.lengthunit==SI)? _("km") : _("mi")),
				   effectivetime,
				   explanation,
				   get_secondary_text_color(),
				   _("from"),
				   startplace,
				   _("to"),
				   endplace);
  }
  g_object_set(renderer, "markup", markup, NULL);
  g_free(markup);
  g_free(starttime);
  g_free(endtime);
  g_free(startplace);
  g_free(endplace);
  g_free(explanation);
  g_free(effectivetime);
}

static
GtkTreeModel *create_and_fill_logmodel (void)
{
  GtkListStore  *store;
  GtkTreeIter    iter;
  GtkTreeModel        *filter;
  GtkTreeSortable     *sortable;
  const unsigned char* dbstarttime;
  const unsigned char* dbendtime;
  const unsigned char* startplace;
  const unsigned char* endplace;
  const unsigned char* explanation;
  double distance;
  double startlat;
  double startlon;
  double endlat;
  double endlon;
  double effectivetime;
  sqlite_int64 id;
  char *timestr;

  if (db_open()) {
    store = gtk_list_store_new (LOG_NUM_COLS, G_TYPE_STRING, G_TYPE_STRING,
				G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_STRING,
				G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_STRING,
				G_TYPE_DOUBLE, G_TYPE_STRING, G_TYPE_STRING,
				GDK_TYPE_PIXBUF, G_TYPE_STRING, 
				G_TYPE_INT64, G_TYPE_BOOLEAN);

    if (DB_OK == db_init_log_query(currentcar)) {
      while(DB_ROW == db_step_log_query_data(&distance, &dbstarttime, &dbendtime, 
					     &startlat, &startlon,
					     &startplace, &endlat, &endlon, &endplace,
					     &effectivetime, &explanation, &id)) {
/* 	convdate(&starttime, datefmtstr[curunit.dateformat], "%Y-%m-%d", dbstarttime); */
/* 	convdate(&endtime, datefmtstr[curunit.dateformat], "%Y-%m-%d", dbendtime); */
	convunixtime(&timestr, "%T", (time_t)effectivetime);

	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    LOG_COL_STARTTIME, dbstarttime,
			    LOG_COL_ENDTIME, dbendtime,
			    LOG_COL_STARTLAT, startlat,
			    LOG_COL_STARTLON, startlon,
			    LOG_COL_STARTPLACE, startplace,
			    LOG_COL_ENDLAT, endlat,
			    LOG_COL_ENDLON, endlon,
			    LOG_COL_ENDPLACE, endplace,
			    LOG_COL_DISTANCE, SIlength2user(distance),
			    LOG_COL_EFFECTIVETIME, timestr,
			    LOG_COL_EXPLANATION, explanation,
			    LOG_COL_ID, id,
			    LOG_COL_VISIBLE, TRUE,
			    -1);

	free(timestr);

      }
    }
    else {
      PDEBUG("db_init_log_query returned != DB_OK\n");
    }

    db_reset_log_query();

    filter = gtk_tree_model_filter_new(GTK_TREE_MODEL(store), NULL);
/*     gtk_tree_model_filter_set_visible_column(GTK_TREE_MODEL_FILTER(filter), */
/* 					     COL_VISIBLE); */

    gtk_tree_model_filter_set_visible_func(GTK_TREE_MODEL_FILTER(filter),
					   visible_func,
/*                                            (gpointer)&filterqueue, */
                                           NULL,
					   NULL);
    sortable = GTK_TREE_SORTABLE(gtk_tree_model_sort_new_with_model(filter));

/*     gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_DAY,  */
/* 				    sort_iter_compare_func, */
/* 				    GINT_TO_POINTER(COL_DAY), NULL); */

/*     gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_TRIP, */
/* 				    sort_iter_compare_func, */
/* 				    GINT_TO_POINTER(COL_TRIP), NULL); */

/*     gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_CONSUM, */
/* 				    sort_iter_compare_func, */
/* 				    GINT_TO_POINTER(COL_CONSUM), NULL); */

/*     gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_PRICEPERTRIP, */
/* 				    sort_iter_compare_func, */
/* 				    GINT_TO_POINTER(COL_PRICEPERTRIP), NULL); */

/*     gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_PRICEPERLITRE, */
/* 				    sort_iter_compare_func, */
/* 				    GINT_TO_POINTER(COL_PRICEPERLITRE), NULL); */

/*     /\* set initial sort order *\/ */
    gtk_tree_sortable_set_sort_column_id(sortable, LOG_COL_STARTTIME, GTK_SORT_DESCENDING);

/*     g_signal_connect(sortable, "sort-column-changed", */
/* 		     G_CALLBACK(sort_column_changed), &ui); */

    return GTK_TREE_MODEL (sortable);
  }
  else
    return NULL;
}


/*******************************************************************
 *
 * Public functions
 *
 *******************************************************************/

void update_logview_row_data(GtkListStore  *store,
			     GtkTreeIter   *iter,
			     double distance,
			     char *starttime,
			     double startlat,
			     double startlon,
			     char *startplace,
			     char *endtime,
			     double endlat,
			     double endlon,
			     char *endplace,
			     char* effectivetime,
			     char *explanation,
			     sqlite_int64 id)
{
  PDEBUG("Updating logview\n");

  if (distance != -1) {
    gtk_list_store_set(store, iter, LOG_COL_DISTANCE, SIlength2user(distance), -1);
  }
  if (starttime != NULL) {
    gtk_list_store_set(store, iter, LOG_COL_STARTTIME, starttime, -1);
  }
  if (startlat != -1) {
    gtk_list_store_set(store, iter, LOG_COL_STARTLAT, startlat, -1);
  }
  if (startlon != -1) {
    gtk_list_store_set(store, iter, LOG_COL_STARTLON, startlon, -1);
  }
  if (startplace != NULL) {
    gtk_list_store_set(store, iter, LOG_COL_STARTPLACE, startplace, -1);
  }
  if (endlat != -1) {
    gtk_list_store_set(store, iter, LOG_COL_ENDLAT, endlat, -1);
  }
  if (endlon != -1) {
    gtk_list_store_set(store, iter, LOG_COL_ENDLON, endlon, -1);
  }
  if (endplace != NULL) {
    gtk_list_store_set(store, iter, LOG_COL_ENDPLACE, endplace, -1);
  }
  if (effectivetime != NULL) {
    gtk_list_store_set(store, iter, LOG_COL_EFFECTIVETIME, effectivetime, -1);
  }
  if (explanation != NULL) {
    gtk_list_store_set(store, iter, LOG_COL_EXPLANATION, explanation, -1);
  }
  gtk_list_store_set(store, iter, LOG_COL_ID, id, -1);
}

GtkWidget *create_logview_and_model (void)
{
  GtkWidget *view;
/*   static data_func_t dfdata1; */
  static data_func_t dfdata2;
  static data_func_t dfdata3;
  static data_func_t dfdata4;
  static data_func_t dfdata5;
  static data_func_t dfdata6;
  static data_func_t dfdata7;
  static data_func_t dfdata8;
  static data_func_t dfdata9;
  static data_func_t dfdata10;
  static data_func_t dfdata11;
  static data_func_t dfdata12;

  GtkTreeViewColumn   *col;
  GtkCellRenderer     *renderer;
  GtkTreeModel        *model;
  GtkTreeSelection    *select;

  GtkWidget           *label;
  PangoAttrList       *attrs;
  PangoAttribute      *attr;

  /* Scale texts smaller: create pango attributes */
  attrs = pango_attr_list_new();
  attr = pango_attr_scale_new(fontscalefactors[fontsize]);
  attr->start_index = 0;
  attr->end_index = -1;
  pango_attr_list_insert(attrs, attr);

  model = create_and_fill_logmodel ();
#if MAEMO_VERSION_MAJOR == 5
  view = hildon_gtk_tree_view_new_with_model (HILDON_UI_MODE_EDIT, 
					      model);
#else
  view = gtk_tree_view_new_with_model (model);
#endif


  /* --- Column #1 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[LOG_COL_STARTTIME]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOG_COL_STARTTIME);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), LOG_COL_STARTTIME);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOG_COL_STARTTIME), logdispcol & (1<<LOG_COL_STARTTIME) && (uistyle==UICLASSIC));

  /* --- Column #2 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[LOG_COL_ENDTIME]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOG_COL_ENDTIME);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), LOG_COL_ENDTIME);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOG_COL_ENDTIME), logdispcol & (1<<LOG_COL_ENDTIME) && (uistyle==UICLASSIC));

  /* --- Column #3 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[LOG_COL_STARTLAT]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOG_COL_STARTLAT);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  dfdata2.col = LOG_COL_STARTLAT;
  strncpy(dfdata2.format,"%.5f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata2, NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), LOG_COL_STARTLAT);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOG_COL_STARTLAT), logdispcol & (1<<LOG_COL_STARTLAT) && (uistyle==UICLASSIC));

  /* --- Column #4 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[LOG_COL_STARTLON]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOG_COL_STARTLON);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  dfdata3.col = LOG_COL_STARTLON;
  strncpy(dfdata3.format,"%.5f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata3, NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), LOG_COL_STARTLON);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOG_COL_STARTLON), logdispcol & (1<<LOG_COL_STARTLON) && (uistyle==UICLASSIC));

  /* --- Column #5 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[LOG_COL_STARTPLACE]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOG_COL_STARTPLACE);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), LOG_COL_STARTPLACE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOG_COL_STARTPLACE), logdispcol & (1<<LOG_COL_STARTPLACE) && (uistyle==UICLASSIC));

  /* --- Column #6 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[LOG_COL_ENDLAT]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOG_COL_ENDLAT);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), LOG_COL_ENDLAT);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOG_COL_ENDLAT), logdispcol & (1<<LOG_COL_ENDLAT) && (uistyle==UICLASSIC));

  dfdata4.col = LOG_COL_ENDLAT;
  strncpy(dfdata4.format,"%.5f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata4, NULL);

  /* --- Column #7 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[LOG_COL_ENDLON]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOG_COL_ENDLON);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), LOG_COL_ENDLON);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOG_COL_ENDLON), logdispcol & (1<<LOG_COL_ENDLON) && (uistyle==UICLASSIC));

  dfdata5.col = LOG_COL_ENDLON;
  strncpy(dfdata5.format,"%.5f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata5, NULL);

  /* --- Column #8 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[LOG_COL_ENDPLACE]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOG_COL_ENDPLACE);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), LOG_COL_ENDPLACE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOG_COL_ENDPLACE), logdispcol & (1<<LOG_COL_ENDPLACE) && (uistyle==UICLASSIC));

  /* --- Column #9 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[LOG_COL_DISTANCE]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOG_COL_DISTANCE);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), LOG_COL_DISTANCE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOG_COL_DISTANCE), logdispcol & (1<<LOG_COL_DISTANCE) && (uistyle==UICLASSIC));

  /* --- Column #10 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[LOG_COL_EFFECTIVETIME]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOG_COL_EFFECTIVETIME);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), LOG_COL_EFFECTIVETIME);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOG_COL_EFFECTIVETIME), dispcol & (1<<LOG_COL_EFFECTIVETIME) && (uistyle==UICLASSIC));

  dfdata6.col = LOG_COL_EFFECTIVETIME;
  strncpy(dfdata6.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata6, NULL);

  /* --- Column #11 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[LOG_COL_EXPLANATION]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOG_COL_EXPLANATION);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), LOG_COL_EXPLANATION);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOG_COL_EXPLANATION), logdispcol & (1<<LOG_COL_EXPLANATION) && (uistyle==UICLASSIC));

  /* Icon column */
  /* --- Column #0 --- */
  col = gtk_tree_view_column_new();

//  gtk_tree_view_column_set_title(col, "Icon");

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_pixbuf_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "pixbuf", LOG_COL_ICON);
/*   gtk_tree_view_column_set_cell_data_func(col, renderer, pixbuf_cell_data_func, view, NULL); */

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOG_COL_ICON), uistyle==UIFANCY);
  /* --- Column #0.5 --- */

  col = gtk_tree_view_column_new();

/*   label=gtk_label_new(gettext(column_headers[COL_COMBINED])); */
/*   gtk_tree_view_column_set_widget(col, GTK_WIDGET(label)); */
/*   gtk_label_set_attributes(GTK_LABEL(label), attrs); */
/*   gtk_widget_show(label); */

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "markup", LOG_COL_COMBINED);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_cell_data_func(col, renderer, new_cell_data_func, NULL, NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), LOG_COL_COMBINED);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOG_COL_COMBINED), uistyle==UIFANCY);

  /* Invisible column to distinguish the results */
  /* --- Column #last --- */

  col = gtk_tree_view_column_new();

  gtk_tree_view_column_set_title(col, "Id");

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOG_COL_ID);

/*   renderer = gtk_cell_renderer_text_new (); */
/*   gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view), */
/*                                                -1,       */
/*                                                "Id",   */
/*                                                renderer, */
/*                                                "text", LOG_COL_ID, */
/*                                                NULL); */
  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOG_COL_ID), FALSE);

  /* Invisible visibility column */
  /* --- Column #last+1 --- */

  col = gtk_tree_view_column_new();

  gtk_tree_view_column_set_title(col, "Visible");

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOG_COL_VISIBLE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOG_COL_VISIBLE), FALSE);

/*   model = create_and_fill_model (); */

/*   /\* Change the unit dependent headers *\/ */
/*   update_column_headers(view); */

/*   gtk_tree_view_set_model (GTK_TREE_VIEW (view), model); */

  if (uistyle==UIFANCY) {
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), FALSE);
  }
  else {
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), TRUE);
  }

  /* The tree view has acquired its own reference to the
   *  model, so we can drop ours. That way the model will
   *  be freed automatically when the tree view is destroyed */

  g_object_unref (model);

  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
  gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
/*   g_signal_connect(view, "row-activated", */
/* 		   G_CALLBACK (callback_recordactivated), */
/* 		   &ui); */

  return view;
}
