/* 
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007-2010 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "uifilter.h"
#include "uiview.h"

/*******************************************************************
 *
 * Private defines
 *
 *******************************************************************/


/*******************************************************************
 *
 * Public variables
 *
 *******************************************************************/

recordfilter_t recfilter;
GList *filterqueue=NULL;

/*******************************************************************
 *
 * Private function prototypes
 *
 *******************************************************************/
static void filter_update(recordfilter_t *filt);
static void draw_filter_selector(AppUIData *pui, GtkWidget *table, gint i, guint nele, recordfilter_t *p);
static GtkTreeModel *get_filter(GtkWidget *view);
static void filter_changed_callback( GtkAction * action, AppUIData *pui);

/*******************************************************************
 *
 * Private functions
 *
 *******************************************************************/

/**
 * \fn GString *get_filter_string(GList *flist)
 * \brief Returns the current filter as a string
 * \param *flist pointer to the filter list
 * \return Pointer to a newly allocated filter string
 *
 * This functions returns the filter list as a filter string. The
 * string is allocated from the heap. After usage, the string must be
 * freed with g_string_free.
 *
 */
static
GString *get_filter_string(GList *flist)
{
  GString *filtstring;
  GList *lp;
  recordfilter_t *p;
  gboolean first;

  filtstring=g_string_new(NULL);
  lp=g_list_first(flist);
  first=TRUE;
  while (lp != NULL) {
    p=(recordfilter_t *)lp->data;
    if (!first) {
      if (p->and) 
	g_string_append_printf(filtstring, " AND "); 
      else
	g_string_append_printf(filtstring, " OR ");
    }
    else
      first=FALSE;
    if (p->column==COL_DAY)
      g_string_append_printf(filtstring, "(%s < Date < %s)",p->mindate->str, p->maxdate->str);
    else if (p->column==COL_NOTES)
      g_string_append_printf(filtstring,"(Notes==%s)",p->notes->str);
    else
      g_string_append_printf(filtstring,"(%.2f < Column %d < %.2f)",p->minval, p->column, p->maxval);
    lp=g_list_next(lp);
  };

  return (filtstring);

}

static
void filter_update(recordfilter_t *filt)
{
  char *date;

  db_get_min_date(&date);
  if (date != NULL) {
    g_string_append_printf(filt->mindate,"%s",date);
    free(date);
  }
  db_get_max_date(&date);
  if (date != NULL) {
    g_string_append_printf(filt->maxdate,"%s",date);
    free(date);
  }
}


static
void filter_free (gpointer data, gpointer user_data)
{
  recordfilter_t *p;

  p=(recordfilter_t *)data;
  filter_clear(p);
  free(p);
}

static
void filter_filterqueue_empty(GList **list)
{
  g_list_foreach(*list, filter_free, NULL);
  g_list_free(*list);
  *list=NULL;
}

static
void filter_records_update(void)
{
  gint active;
  gint andactive;
  GList *lp;
  recordfilter_t *p;
  char *mindate;
  char *maxdate;
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#endif
  GString *s;

  lp=g_list_first(filterqueue);
  while (lp != NULL) {
    p=(recordfilter_t *)lp->data;

    /* Mandatory or optional */
#if MAEMO_VERSION_MAJOR == 5
    andactive = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (p->entry.andorcombo), 0);
#else
    andactive = gtk_combo_box_get_active (GTK_COMBO_BOX (p->entry.andorcombo));
#endif
    p->and= andactive == 0;

    if (p->column==COL_DAY) {
      /* Get the minimum date */
#if PLAINGTK == 0
    /* Missing: reading gtk_entry and converting to year, month, day */
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_get_date(HILDON_DATE_BUTTON(p->entry.minentry), &year, &month, &day); /* Month is betweewn 0 and 11 */
      month++; 
#else
      hildon_date_editor_get_date(HILDON_DATE_EDITOR(p->entry.minentry), &year, &month, &day);
#endif
      mindate=(char *)malloc(ENTRYDATEMAX*sizeof(char));
      g_snprintf(mindate, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
      date2sqlite(&mindate, datefmtstr[curunit.dateformat], 
		  gtk_entry_get_text(GTK_ENTRY(p->entry.minentry)));
#endif
      g_string_assign(p->mindate, mindate);
      free(mindate);

      /* Get the maximum date */
#if PLAINGTK == 0
    /* Missing: reading gtk_entry and converting to year, month, day */
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_get_date(HILDON_DATE_BUTTON(p->entry.maxentry), &year, &month, &day); /* Month is betweewn 0 and 11 */
      month++; 
#else
      hildon_date_editor_get_date(HILDON_DATE_EDITOR(p->entry.maxentry), &year, &month, &day);
#endif
      maxdate=(char *)malloc(ENTRYDATEMAX*sizeof(char));
      g_snprintf(maxdate, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
      date2sqlite(&maxdate, datefmtstr[curunit.dateformat], 
		  gtk_entry_get_text(GTK_ENTRY(p->entry.maxentry)));
#endif
      g_string_assign(p->maxdate, maxdate);
      free(maxdate);
    }
    else if (active==COL_NOTES) {
      /* Get the notes filter */
      g_string_assign(p->notes, gtk_entry_get_text(GTK_ENTRY(p->entry.minentry)));
      g_pattern_spec_free(p->pattern);
      p->pattern=g_pattern_spec_new(p->notes->str);
    }
    else {
      /* Get the other fields */
      s=g_string_new("");
      g_string_assign(s, gtk_entry_get_text(GTK_ENTRY(p->entry.minentry)));
      p->minval=atof(s->str);

      g_string_assign(s, gtk_entry_get_text(GTK_ENTRY(p->entry.maxentry)));
      p->maxval=atof(s->str);
      g_string_free(s, TRUE);
    }

    lp=g_list_next(lp);
  };
}

static
void delete_filter_records( GtkWidget *widget,
			    GdkEvent  *event,
			    AppUIData *pui )
{

  filter_records_update();

  update_filter_button(pui, filter_filterqueue_length() > 0);

#if MAEMO_VERSION_MAJOR == 5
  /* This is a work around since the apply button has some issues */
  gtk_tree_model_filter_refilter(GTK_TREE_MODEL_FILTER(get_filter(pui->view)));
#endif
  gtk_widget_destroy(GTK_WIDGET(widget));

  return FALSE;
}

/**
 * \fn GString* get_filter_entry_userunit(gdouble value, gint col)
 * \brief Returns the value as a string in user unit
 * \param value the value
 * \param col the column number
 * \return Pointer to a newly allocated string
 *
 * This functions returns the given value in the appropriate user
 * unit. The string is allocated from the heap. After usage, the
 * string must be freed with g_string_free.
 *
 */
static
GString* get_filter_entry_userunit(gdouble value, gint col)
{
  GString *text;

  text=g_string_new(NULL);

  switch (col) {
  case COL_PRICE:
  case COL_SERVICE:
  case COL_OIL:
  case COL_TIRES:
    g_string_printf(text, "%.2f",value);
    break;
  case COL_KM:
  case COL_TRIP:
    g_string_printf(text, "%.2f",SIlength2user(value));
    break;
  case COL_PRICEPERTRIP:
    g_string_printf(text, "%.2f",SIpricepertrip2user(value));
    break;
  case COL_PRICEPERLITRE:
    g_string_printf(text, "%.2f",SIppl2user(value));
    break;
  case COL_FILL:
    g_string_printf(text, "%.2f",SIvolume2user(value));
    break;
  case COL_CONSUM:
    g_string_printf(text, "%.2f",SIconsumption2user(value));
    break;
  case COL_CO2EMISSION:
    g_string_printf(text, "%.2f",SIemission2user(value));
    break;
  }

  return text;
}

static
void fill_filter_minmax_entrys(AppUIData *pui, gint active, GtkWidget *minentry, GtkWidget *maxentry)
{
  GString *s;
  GString *value;
  char *newdate;
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#else
  char *date;
#endif
  int fueltype;
  gfloat emisperlitre;
  gfloat minval;
  gfloat maxval;
  gfloat temp;

  if (active==COL_CO2EMISSION) {
    fueltype = db_get_fueltype(currentcar);
    emisperlitre = get_emission_per_litre(fueltype);
  }

  s=g_string_new("");
  if (active==COL_DAY) {
    db_get_min_date(&newdate);
#if PLAINGTK == 1
    convdate(&date, datefmtstr[curunit.dateformat], "%Y-%m-%d", newdate);

    gtk_entry_set_text(GTK_ENTRY(minentry),date);
    free(date);
#else
    getyearmonthday(newdate, 
		    &year,
		    &month,
		    &day);
#if MAEMO_VERSION_MAJOR == 5
    hildon_date_button_set_date(HILDON_DATE_BUTTON(minentry), year, month-1, day); /* Month is betweewn 0 and 11 */
#else
    hildon_date_editor_set_date(HILDON_DATE_EDITOR(minentry), year, month, day);
#endif
#endif
    free(newdate);

      db_get_max_date(&newdate);
#if PLAINGTK == 1
      convdate(&date, datefmtstr[curunit.dateformat], "%Y-%m-%d", newdate);
      gtk_entry_set_text(GTK_ENTRY(maxentry),date);
      free(date);
#else
      getyearmonthday(newdate, 
		      &year,
		      &month,
		      &day);
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_set_date(HILDON_DATE_BUTTON(maxentry), year, month-1, day); /* Month is betweewn 0 and 11 */
#else
      hildon_date_editor_set_date(HILDON_DATE_EDITOR(maxentry), year, month, day);
#endif
#endif
      free(newdate);

  }
  else if (active!=COL_NOTES) {
    minval=db_get_min(active);
    if (active==COL_CO2EMISSION)
      minval=minval*emisperlitre;
    maxval=db_get_max(active);
    if (active==COL_CO2EMISSION)
      maxval=maxval*emisperlitre;

    if (active==COL_CONSUM && (curunit.consumeunit==US || curunit.consumeunit==IMPERIAL)) {
      temp=maxval;
      maxval=minval;
      minval=temp;
    }

    value=get_filter_entry_userunit(minval, active);
    gtk_entry_set_text(GTK_ENTRY(minentry),value->str);
    g_string_free(value, TRUE);

    value=get_filter_entry_userunit(maxval, active);
    gtk_entry_set_text(GTK_ENTRY(maxentry),value->str);
    g_string_free(value, TRUE);
  }
  else {
    gtk_entry_set_text(GTK_ENTRY(minentry),recfilter.notes->str);
  }
  g_string_free(s, TRUE);
  
}

static
void show_filter_minmax_entrys(AppUIData *pui, GtkWidget *vbox)
{
  GtkWidget *hbox;
  GtkWidget *label;
  gint active;

#if MAEMO_VERSION_MAJOR == 5
  active=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(pui->filtercombo), 0)+COL_DAY;
#else
  active=gtk_combo_box_get_active(GTK_COMBO_BOX(pui->filtercombo))+COL_DAY;
#endif

#if MAEMO_VERSION_MAJOR == 5
  pui->filteraddandorbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->filteraddandorbutton), _("Necessity"));
  gtk_box_pack_start (GTK_BOX(vbox), pui->filteraddandorbutton, TRUE, TRUE, 0);
  gtk_widget_show (pui->filteraddandorbutton);
  pui->filteraddandorcombo=create_andor_combo (pui->filteraddandorbutton);
#else
  pui->filteraddandorcombo=create_andor_combo ();
  gtk_box_pack_start (GTK_BOX(vbox), pui->filteraddandorcombo, TRUE, TRUE, 0);
  gtk_widget_show (pui->filteraddandorcombo);
#endif

  hbox=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

if (active!=COL_NOTES)
  label=gtk_label_new(_("Minimum value"));
 else
  label=gtk_label_new(_("Search pattern"));

  gtk_box_pack_start (GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);

  if (active==COL_DAY) {
#if PLAINGTK == 1
    pui->filteraddminentry=gtk_entry_new();
#else
#if MAEMO_VERSION_MAJOR == 5
    pui->filteraddminentry = hildon_date_button_new(HILDON_SIZE_FINGER_HEIGHT,
							  HILDON_BUTTON_ARRANGEMENT_VERTICAL);
#else
    pui->filteraddminentry = hildon_date_editor_new();
#endif
#endif
  }
  else if (active!=COL_NOTES) {
#if MAEMO_VERSION_MAJOR == 5
    pui->filteraddminentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
    pui->filteraddminentry=gtk_entry_new();
#endif
  }
  else {
#if MAEMO_VERSION_MAJOR == 5
    pui->filteraddminentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
    pui->filteraddminentry=gtk_entry_new();
#endif
  }

  gtk_box_pack_start (GTK_BOX(hbox), pui->filteraddminentry, TRUE, TRUE, 0);
  gtk_widget_show(pui->filteraddminentry);

  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 0);


  if (active!=COL_NOTES) {
    label=gtk_label_new(_("Maximum value"));
    gtk_box_pack_start (GTK_BOX(hbox), label, TRUE, TRUE, 0);
    gtk_widget_show(label);
    if (active==COL_DAY) {
#if PLAINGTK == 1

#if MAEMO_VERSION_MAJOR == 5
      pui->filteraddmaxentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
      pui->filteraddmaxentry=gtk_entry_new();
#endif
#else
#if MAEMO_VERSION_MAJOR == 5
      pui->filteraddmaxentry = hildon_date_button_new(HILDON_SIZE_FINGER_HEIGHT,
							  HILDON_BUTTON_ARRANGEMENT_VERTICAL);
#else
      pui->filteraddmaxentry = hildon_date_editor_new();
#endif
#endif

    }
    else {
#if MAEMO_VERSION_MAJOR == 5
      pui->filteraddmaxentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
      pui->filteraddmaxentry=gtk_entry_new();
#endif
    }

  }
  else { /* active==COL_NOTES */
    /* Create an empty label to keep widget the packing ok */ 
    label=gtk_label_new("");
    gtk_box_pack_start (GTK_BOX(hbox), label, TRUE, TRUE, 0);
    gtk_widget_show(label);
  }

  gtk_box_pack_start (GTK_BOX(hbox), pui->filteraddmaxentry, TRUE, TRUE, 0);
  gtk_widget_show(pui->filteraddmaxentry);

  gtk_widget_show(hbox);

  fill_filter_minmax_entrys(pui, active, pui->filteraddminentry, pui->filteraddmaxentry);
}

static
GtkWidget *create_filter_combo(AppUIData *pui)
{
  GtkWidget *combo;
  gint i;
#if MAEMO_VERSION_MAJOR == 5
  GRegex *regex;
  gchar *pattern="\\n";
#endif
  gchar *newstr;
  GString *header;

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
#else
  combo = gtk_combo_box_new_text();
#endif

#if MAEMO_VERSION_MAJOR == 5
  regex = g_regex_new(pattern, G_REGEX_DOTALL, 0, NULL);
#endif

  for (i=COL_DAY;i<COL_ID;i++) {

    header = get_column_header(i);
#if MAEMO_VERSION_MAJOR == 5
    newstr = g_regex_replace(regex, header->str, -1, 1, " ", 0, NULL);
#else
    newstr = gettext(header->str);
#endif

#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (combo), newstr);
#else
    gtk_combo_box_append_text(GTK_COMBO_BOX (combo), newstr);
#endif
    g_string_free(header, TRUE);

#if MAEMO_VERSION_MAJOR == 5
    g_free(newstr);
#endif
  }

#if MAEMO_VERSION_MAJOR == 5
  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR (combo), 0, 0);
#else
  gtk_combo_box_set_active(GTK_COMBO_BOX (combo), 0);
#endif

#if MAEMO_VERSION_MAJOR == 5
  g_regex_unref(regex);
#endif

  return combo;

}

static
void add_filter_cb (GtkWidget *widget, gpointer data )
{
  gint active;
  gint andactive;
  recordfilter_t *p;
  AppUIData *pui;
  char *mindate;
  char *maxdate;
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#endif
  GString *filtstring;
  GString *s;

  pui=&ui;

#if MAEMO_VERSION_MAJOR == 5
  active=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(pui->filtercombo), 0)+COL_DAY;
#else
  active=gtk_combo_box_get_active(GTK_COMBO_BOX(pui->filtercombo))+COL_DAY;
#endif

  g_print("Active = %d", active);

  p=(recordfilter_t *)malloc(sizeof(recordfilter_t));
  filter_init(p);
  draw_filter_selector(pui, pui->filtertable, active, filter_filterqueue_length(), p);

  if (filterqueue != NULL) {
#if MAEMO_VERSION_MAJOR == 5
    andactive = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (pui->filteraddandorcombo), 0);
    p->and= andactive == 0;
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR (p->entry.andorcombo), 0, andactive);
#else
    andactive = gtk_combo_box_get_active (GTK_COMBO_BOX (pui->filteraddandorcombo));
    p->and= andactive == 0;
    gtk_combo_box_set_active(GTK_COMBO_BOX (p->entry.andorcombo), andactive);
#endif
  }
  p->column=active;
  filterqueue=g_list_append(filterqueue, p);
  if (active==COL_DAY) {

      /* Get the minimum date */
#if PLAINGTK == 0
    /* Missing: reading gtk_entry and converting to year, month, day */
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_get_date(HILDON_DATE_BUTTON(pui->filteraddminentry), &year, &month, &day); /* Month is betweewn 0 and 11 */
      month++; 
      hildon_date_button_set_date(HILDON_DATE_BUTTON(p->entry.minentry), year, month-1, day);
#else
      hildon_date_editor_get_date(HILDON_DATE_EDITOR(pui->filteraddminentry), &year, &month, &day);
      hildon_date_editor_set_date(HILDON_DATE_EDITOR(p->entry.minentry), year, month, day);
#endif
      mindate=(char *)malloc(ENTRYDATEMAX*sizeof(char));
      g_snprintf(mindate, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
      date2sqlite(&mindate, datefmtstr[curunit.dateformat], 
		      gtk_entry_get_text(GTK_ENTRY(pui->filteraddminentry)));
#endif
      g_string_assign(p->mindate, mindate);
      free(mindate);

      /* Get the maximum date */
#if PLAINGTK == 0
    /* Missing: reading gtk_entry and converting to year, month, day */
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_get_date(HILDON_DATE_BUTTON(pui->filteraddmaxentry), &year, &month, &day); /* Month is betweewn 0 and 11 */
      month++; 
      hildon_date_button_set_date(HILDON_DATE_BUTTON(p->entry.maxentry), year, month-1, day);
#else
      hildon_date_editor_get_date(HILDON_DATE_EDITOR(pui->filteraddmaxentry), &year, &month, &day);
      hildon_date_editor_set_date(HILDON_DATE_EDITOR(p->entry.maxentry), year, month, day);
#endif
      maxdate=(char *)malloc(ENTRYDATEMAX*sizeof(char));
      g_snprintf(maxdate, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
      date2sqlite(&maxdate, datefmtstr[curunit.dateformat], 
		      gtk_entry_get_text(GTK_ENTRY(pui->filteraddmaxentry)));
#endif
      g_string_assign(p->maxdate, maxdate);
      free(maxdate);

  }
  else if (active==COL_NOTES) {
    /* Get the notes filter */
    g_string_assign(p->notes, gtk_entry_get_text(GTK_ENTRY(pui->filteraddminentry)));
    gtk_entry_set_text(GTK_ENTRY(p->entry.minentry),p->notes->str);
    g_pattern_spec_free(p->pattern);
    p->pattern=g_pattern_spec_new(p->notes->str);
  }
  else {
    /* Get the other fields */
    s=g_string_new("");
    g_string_assign(s, gtk_entry_get_text(GTK_ENTRY(pui->filteraddminentry)));
    gtk_entry_set_text(GTK_ENTRY(p->entry.minentry),s->str);
    p->minval=atof(s->str);

    g_string_assign(s, gtk_entry_get_text(GTK_ENTRY(pui->filteraddmaxentry)));
    gtk_entry_set_text(GTK_ENTRY(p->entry.maxentry),s->str);
    p->maxval=atof(s->str);
    g_string_free(s, TRUE);
  }

#if MAEMO_VERSION_MAJOR == 5
  /* This is a work around since the apply button has some issues */
//  gtk_tree_model_filter_refilter(GTK_TREE_MODEL_FILTER(get_filter(pui->view)));
#endif
}

static
void add_filter_callback( GtkAction * action, AppUIData *pui)
{
  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkWidget *label;
  GtkWidget *hbox;

  pui->addfilterdialog = gtk_dialog_new_with_buttons(_("Add a new filter"),
						     GTK_WINDOW(pui->filterwindow),
						     GTK_DIALOG_MODAL,
						     GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
						     GTK_STOCK_CANCEL,
						     GTK_RESPONSE_REJECT,
						     NULL);

  vbox = GTK_WIDGET(GTK_DIALOG(pui->addfilterdialog)->vbox);

#if MAEMO_VERSION_MAJOR == 5
  pui->filterbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->filterbutton), _("Filter"));
  gtk_box_pack_start (GTK_BOX(vbox), pui->filterbutton, TRUE, TRUE, 0);
  gtk_widget_show(pui->filterbutton);
#else
  hbox=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
  label=gtk_label_new(_("Filter"));
  gtk_box_pack_start (GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);
#endif

  pui->filtercombo = create_filter_combo(pui);
#if MAEMO_VERSION_MAJOR == 5
  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (pui->filterbutton),
                                     HILDON_TOUCH_SELECTOR (pui->filtercombo));
#else
  gtk_box_pack_start (GTK_BOX(hbox), pui->filtercombo, TRUE, TRUE, 0);
  gtk_widget_show(pui->filtercombo);
  gtk_widget_show(hbox);
#endif

  pui->filtervbox=gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start (GTK_BOX(vbox), pui->filtervbox, TRUE, TRUE, 0);
  show_filter_minmax_entrys(pui, pui->filtervbox);
  gtk_widget_show(pui->filtervbox);

#if MAEMO_VERSION_MAJOR == 5
  g_signal_connect(G_OBJECT(pui->filterbutton), "value-changed",
		   G_CALLBACK(filter_changed_callback), pui);
#else
  g_signal_connect (GTK_OBJECT (pui->filtercombo), "changed",
		    G_CALLBACK(filter_changed_callback), pui);
#endif

  gtk_widget_show(pui->addfilterdialog);

  if (gtk_dialog_run(GTK_DIALOG(pui->addfilterdialog))  == GTK_RESPONSE_ACCEPT) {
    add_filter_cb(pui->addfilterdialog, 0);
  }
  gtk_widget_destroy(GTK_WIDGET(pui->addfilterdialog));
}


void callback_clear_filter( GtkAction * action, AppUIData *pui )
{
#if MAEMO_VERSION_MAJOR == 5
  GtkTextBuffer *textbuffer;
#endif
  filter_filterqueue_empty(&filterqueue);
/* #if MAEMO_VERSION_MAJOR == 5 */
/*   textbuffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(pui->filterentry)); */
/*   gtk_text_buffer_set_text(textbuffer, "", -1); */
/* #else */
/*   gtk_entry_set_text(GTK_ENTRY(pui->filterentry), ""); */
/* #endif */
#if MAEMO_VERSION_MAJOR == 5
  /* This is a work around since the apply button has some issues */
  gtk_tree_model_filter_refilter(GTK_TREE_MODEL_FILTER(get_filter(pui->view)));
#endif
}


static
void filter_records_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  filter_records_update();

  gtk_tree_model_filter_refilter(GTK_TREE_MODEL_FILTER(get_filter(pui->view)));

  gtk_widget_destroy(GTK_WIDGET(pui->filterwindow));
}

static
void remove_filter_callback( GtkAction * action, recordfilter_t *p )
{
  gtk_widget_destroy(p->entry.removebutton);
#if MAEMO_VERSION_MAJOR == 5
  gtk_widget_destroy(p->entry.andorbutton);
#else
  gtk_widget_destroy(p->entry.andorcombo);
  gtk_widget_destroy(p->entry.label);
#endif
  gtk_widget_destroy(p->entry.minentry);
  gtk_widget_destroy(p->entry.maxentry);
  filterqueue = g_list_remove(filterqueue, p);
  filter_clear(p);
  free(p);

  /* The following is propably not necessary */
  if (filter_filterqueue_length()==0) {
    filter_filterqueue_empty(&filterqueue);
  }
}

static
void filter_andor_changed_cb( GtkAction * action, recordfilter_t *p )
{
  gint andactive;

#if MAEMO_VERSION_MAJOR == 5
  HildonTouchSelector *selector;
  selector = hildon_picker_button_get_selector(HILDON_PICKER_BUTTON(p->entry.andorbutton));
  andactive = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (selector), 0);
#else
  andactive = gtk_combo_box_get_active (GTK_COMBO_BOX (p->entry.andorcombo));
#endif
  p->and=andactive == 0;
}


static
void fill_filter_entrys(AppUIData *pui, recordfilter_t *p)
{
  gint andactive;
  guint year;
  guint month;
  guint day;
  GString *s1;
  GString *s2;

  if (p->and)
    andactive = 0;
  else
    andactive = 1;
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR (p->entry.andorcombo), 0, andactive);
#else
    gtk_combo_box_set_active(GTK_COMBO_BOX (p->entry.andorcombo), andactive);
#endif

    if (p->column == COL_DAY) {
      getyearmonthday(p->mindate->str,
		      &year,
		      &month,
		      &day);
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_set_date(HILDON_DATE_BUTTON(p->entry.minentry), year, month-1, day);
#else
      hildon_date_editor_set_date(HILDON_DATE_EDITOR(p->entry.minentry), year, month, day);
#endif

      getyearmonthday(p->maxdate->str,
		      &year,
		      &month,
		      &day);
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_set_date(HILDON_DATE_BUTTON(p->entry.maxentry), year, month-1, day);
#else
      hildon_date_editor_set_date(HILDON_DATE_EDITOR(p->entry.maxentry), year, month, day);
#endif
    }
    else if (p->column == COL_NOTES) {
      gtk_entry_set_text(GTK_ENTRY(p->entry.minentry),p->notes->str);
    }
    else { /* Normal fields */
      s1=g_string_new("");
      s2=g_string_new("");
      g_string_printf(s1, "%.2f",p->minval);
      g_string_printf(s2, "%.2f",p->maxval);
      gtk_entry_set_text(GTK_ENTRY(p->entry.minentry),s1->str);
      gtk_entry_set_text(GTK_ENTRY(p->entry.maxentry),s2->str);
      g_string_free(s2, TRUE);
      g_string_free(s1, TRUE);
    }

}

static
void draw_filter_selector(AppUIData *pui, GtkWidget *table, gint i, guint nele, recordfilter_t *p)
{
#if MAEMO_VERSION_MAJOR == 5
  GRegex *regex;
  gchar *pattern="\\n";
#else
  GtkWidget *label;
#endif
  gchar *newstr;
  GString *header;

#if MAEMO_VERSION_MAJOR == 5
  regex = g_regex_new(pattern, G_REGEX_DOTALL, 0, NULL);
#endif

    header = get_column_header(i);
#if MAEMO_VERSION_MAJOR == 5
    newstr = g_regex_replace(regex, header->str, -1, 1, " ", 0, NULL);
#else
    newstr = gettext(header->str);
#endif

#if MAEMO_VERSION_MAJOR == 5
    p->entry.removebutton=hildon_button_new_with_text(HILDON_SIZE_AUTO,
						  HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						  _("Remove"),
						  newstr);
#else
    p->entry.removebutton=gtk_button_new_from_stock (GTK_STOCK_REMOVE);
#endif
    gtk_table_attach(GTK_TABLE(table), p->entry.removebutton, 0, 1, nele, nele+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    g_signal_connect (p->entry.removebutton, "clicked",
		      G_CALLBACK (remove_filter_callback),
		      p);
    gtk_widget_show (p->entry.removebutton);

#if MAEMO_VERSION_MAJOR == 5
#else
    p->entry.label = gtk_label_new(newstr);
    gtk_table_attach(GTK_TABLE(table), p->entry.label, 1, 2, nele, nele+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show (p->entry.label);
#endif

    g_string_free(header, TRUE);
#if MAEMO_VERSION_MAJOR == 5
    g_free(newstr);
#endif

#if MAEMO_VERSION_MAJOR == 5
    p->entry.andorbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    hildon_button_set_title (HILDON_BUTTON (p->entry.andorbutton), _("Necessity"));
    gtk_table_attach(GTK_TABLE(table), p->entry.andorbutton, 2, 3, nele, nele+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show (p->entry.andorbutton);
    p->entry.andorcombo=create_andor_combo (p->entry.andorbutton);
/*     g_signal_connect (GTK_OBJECT (p->entry.andorbutton), "value-changed", */
/* 		      G_CALLBACK(filter_andor_changed_cb), p); */
#else
    p->entry.andorcombo=create_andor_combo ();
/*     g_signal_connect (GTK_OBJECT (p->entry.andorcombo), "changed", */
/* 		      G_CALLBACK(filter_andor_changed_cb), p); */
    gtk_table_attach(GTK_TABLE(table), p->entry.andorcombo, 2, 3, nele, nele+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show (p->entry.andorcombo);
#endif

    if (i==COL_DAY) {
#if PLAINGTK == 1
	  p->entry.minentry=gtk_entry_new();
#else
#if MAEMO_VERSION_MAJOR == 5
	  p->entry.minentry = hildon_date_button_new(HILDON_SIZE_FINGER_HEIGHT,
						     HILDON_BUTTON_ARRANGEMENT_VERTICAL);
#else
	  p->entry.minentry = hildon_date_editor_new();
#endif
#endif
    }
    else if (i!=COL_NOTES) {
#if MAEMO_VERSION_MAJOR == 5
      p->entry.minentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
      p->entry.minentry=gtk_entry_new();
#endif
    }
    else {
#if MAEMO_VERSION_MAJOR == 5
      p->entry.minentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
      p->entry.minentry=gtk_entry_new();
#endif
    }
    gtk_table_attach(GTK_TABLE(table), p->entry.minentry, 3, 4, nele, nele+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(p->entry.minentry);

    if (i!=COL_NOTES) {
      if (i==COL_DAY) {
#if PLAINGTK == 1

#if MAEMO_VERSION_MAJOR == 5
	  p->entry.maxentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
	  p->entry.maxentry=gtk_entry_new();
#endif
#else
#if MAEMO_VERSION_MAJOR == 5
	  p->entry.maxentry = hildon_date_button_new(HILDON_SIZE_FINGER_HEIGHT,
						     HILDON_BUTTON_ARRANGEMENT_VERTICAL);
#else
	  p->entry.maxentry = hildon_date_editor_new();
#endif
#endif

      }
      else {
#if MAEMO_VERSION_MAJOR == 5
	p->entry.maxentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
	p->entry.maxentry=gtk_entry_new();
#endif
      }
      gtk_table_attach(GTK_TABLE(table), p->entry.maxentry, 4, 5, nele, nele+1,
		       GTK_EXPAND|GTK_FILL,
		       0, 0, 5);
      gtk_widget_show(p->entry.maxentry);
    }

#if MAEMO_VERSION_MAJOR == 5
    g_regex_unref(regex);
#endif

}



static
void filter_changed_callback( GtkAction * action, AppUIData *pui)
{
  GtkWidget *vbox;

  vbox = GTK_WIDGET(GTK_DIALOG(pui->addfilterdialog)->vbox);

  gtk_widget_destroy(pui->filtervbox);
  pui->filtervbox=gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start (GTK_BOX(vbox), pui->filtervbox, TRUE, TRUE, 0);
  show_filter_minmax_entrys(pui, pui->filtervbox);
  gtk_widget_show(pui->filtervbox);
}


/*******************************************************************
 *
 * Public functions
 *
 *******************************************************************/

void filter_init(recordfilter_t *filt)
{
  filt->mindate=g_string_new("");
  filt->maxdate=g_string_new("");
  filt->notes=g_string_new("*");
  filt->pattern=g_pattern_spec_new(filt->notes->str);
  filt->column=0;

  filter_update(filt);

  filt->and = FALSE;

}

void filter_clear(recordfilter_t *filt)
{
  g_string_free(filt->mindate, TRUE);
  g_string_free(filt->maxdate, TRUE);
  g_string_free(filt->notes, TRUE);
  g_pattern_spec_free(filt->pattern);
}

guint filter_filterqueue_length(void)
{
  return g_list_length(filterqueue);
}

void callback_filter_records( GtkAction * action, AppUIData *pui )
{
#if MAEMO_VERSION_MAJOR == 5
  GtkTextBuffer *textbuffer;
#endif
  gchar *newstr;
  GtkWidget *vbox;
  GtkWidget *scrollwin;
  GtkWidget *label;
  GtkWidget *image;
  GtkWidget *button;
  gint i;
  GList *lp;
  recordfilter_t *p;

#if MAEMO_VERSION_MAJOR == 5
  pui->filterwindow = hildon_stackable_window_new();
  hildon_program_add_window(pui->app->program, pui->filterwindow);
  g_signal_connect(G_OBJECT(pui->filterwindow), "destroy",
		   G_CALLBACK(gtk_widget_destroy), NULL);
  g_signal_connect(G_OBJECT(pui->filterwindow), "delete_event",
		   G_CALLBACK(delete_filter_records), pui);
  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (pui->filterwindow), vbox);
#else
  pui->filterwindow = gtk_dialog_new_with_buttons(_("Filter records"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  vbox = GTK_WIDGET(GTK_DIALOG(pui->filterwindow)->vbox);
#endif

  label=gtk_label_new(_("Active filters"));
  gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, TRUE, 0);
  gtk_widget_show(label);

#if MAEMO_VERSION_MAJOR == 5
  button = hildon_button_new_with_text(HILDON_SIZE_AUTO,
				       HILDON_BUTTON_ARRANGEMENT_HORIZONTAL,
				       _("Add new"),
				       NULL);
#else
  button = gtk_button_new_with_label(_("Add new"));
#endif
#if MAEMO_VERSION_MAJOR == 5
  image = gtk_image_new_from_icon_name ("general_add", GTK_ICON_SIZE_BUTTON);
  hildon_button_set_image (HILDON_BUTTON (button), image);
  hildon_button_set_image_position (HILDON_BUTTON (button), GTK_POS_LEFT);
#else
  image = gtk_image_new_from_stock (GTK_STOCK_ADD, GTK_ICON_SIZE_BUTTON);
  gtk_button_set_image (GTK_BUTTON (button), image);
  gtk_button_set_image_position (GTK_BUTTON (button), GTK_POS_LEFT);
#endif
  gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, TRUE, 0);
  g_signal_connect (button, "clicked",
		    G_CALLBACK (add_filter_callback),
		    pui);
  gtk_widget_show(button);

#if MAEMO_VERSION_MAJOR == 5
  scrollwin = hildon_pannable_area_new();
  g_object_set(G_OBJECT(scrollwin), "mov-mode", HILDON_MOVEMENT_MODE_BOTH, NULL);
  gtk_widget_set_size_request(scrollwin, DIALOG_MIN_WIDTH1, DIALOG_MIN_HEIGHT0);
#else
  scrollwin=gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_size_request(scrollwin, DIALOG_MIN_WIDTH1, DIALOG_MIN_HEIGHT2);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin), 
				 GTK_POLICY_AUTOMATIC, 
				 GTK_POLICY_AUTOMATIC);
#endif
/*   gtk_box_pack_start (GTK_BOX(vbox), scrollwin, TRUE, TRUE, 0); */
  gtk_box_pack_start (GTK_BOX(vbox), scrollwin, FALSE, TRUE, 0);

  pui->filtertable = gtk_table_new(5, COL_ID+1, FALSE);
#if MAEMO_VERSION_MAJOR == 5
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(scrollwin),pui->filtertable);
#else
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollwin),pui->filtertable);
#endif

  i=0;
  lp=g_list_first(filterqueue);
  while (lp != NULL) {
    p=(recordfilter_t *)lp->data;
    draw_filter_selector(pui, pui->filtertable, p->column, i, p);
    fill_filter_entrys(pui, p);
    i++;
    lp=g_list_next(lp);
  };

  gtk_widget_show(pui->filtertable);
  gtk_widget_show(scrollwin);

#if LIBHELP == 1
    help_dialog_help_enable(GTK_DIALOG(pui->filterwindow),
				   HELP_ID_FILTER,
				   pui->app->osso);
#endif

#if MAEMO_VERSION_MAJOR == 5
/*     button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT); */
/*     gtk_button_set_label(GTK_BUTTON(button), _("Apply")); */
/*     gtk_box_pack_start (GTK_BOX(vbox), button, TRUE, TRUE, 0); */
/*     g_signal_connect(G_OBJECT(button), "clicked", */
/* 		     G_CALLBACK(filter_records_response), */
/* 		     pui); */
/*     gtk_widget_show(button); */

    gtk_widget_show(vbox);
    gtk_widget_show_all(GTK_WIDGET(pui->filterwindow));
#else
    g_signal_connect (G_OBJECT(pui->filterwindow), "response",
		      G_CALLBACK (filter_records_response),
		      pui);
    gtk_widget_show(pui->filterwindow);
#endif

}
