/*
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007-2010 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <glib.h>
#include <glib-object.h>
#include <conic.h>

/*******************************************************************
 *
 * Public variables
 *
 *******************************************************************/

volatile gboolean netconnected = FALSE;
volatile gboolean netconnecting = FALSE;

/*******************************************************************
 *
 * Private definitions
 *
 *******************************************************************/

/*******************************************************************
 *
 * Private variables
 *
 *******************************************************************/
ConIcConnection *connection;
ConIcConnectionStatus status = 0xFFFF;

/*******************************************************************
 *
 * Prototypes of private functions
 *
 *******************************************************************/
static void connection_handler(ConIcConnection *connection,
			       ConIcConnectionEvent *event,
			       gpointer user_data);
static void connection_info(ConIcConnection *connection,
                            ConIcConnectionEvent *event,
                            gpointer user_data);

/*******************************************************************
 *
 * Private functions
 *
 *******************************************************************/
static void connection_handler(ConIcConnection *connection,
			       ConIcConnectionEvent *event,
			       gpointer user_data)
{
  ConIcConnectionStatus status = con_ic_connection_event_get_status(event);
  ConIcConnectionStatus *status_ptr = (ConIcConnectionStatus*)user_data;
  ConIcConnectionError error;

  const gchar *iap_id = con_ic_event_get_iap_id(CON_IC_EVENT(event));
  const gchar *bearer = con_ic_event_get_bearer_type(CON_IC_EVENT(event));

  *status_ptr = status;

  switch(status) {
  case CON_IC_STATUS_CONNECTED:
    g_debug("Hey, we are connected to IAP %s with bearer %s!", iap_id, bearer);
    netconnected = TRUE;
    netconnecting = FALSE;
    break;
  case CON_IC_STATUS_DISCONNECTING:
    g_debug("We are disconnecting...");
    netconnected = FALSE;
    netconnecting = FALSE;
    break;
  case CON_IC_STATUS_DISCONNECTED:
    netconnected = FALSE;
    netconnecting = FALSE;
    g_debug("And we are disconnected. Let's see what went wrong...");
    error = con_ic_connection_event_get_error(event);
    switch(error) {
    case CON_IC_CONNECTION_ERROR_NONE:
      g_debug("Libconic thinks there was nothing wrong.");
      break;
    case CON_IC_CONNECTION_ERROR_INVALID_IAP:
      g_debug("Invalid (non-existing?) IAP was requested.");
      break;
    case CON_IC_CONNECTION_ERROR_CONNECTION_FAILED:
      g_debug("Connection just failed.");
      break;
    case CON_IC_CONNECTION_ERROR_USER_CANCELED:
      g_debug("User canceled the connection attempt");
      break;
    }
    break;
  default:
    netconnected = FALSE;
    g_debug("Unknown connection status received");
  }
}

static void connection_info(ConIcConnection *connection,
                            ConIcConnectionEvent *event,
                            gpointer user_data)
{
}

/*******************************************************************
 *
 * Public functions
 *
 *******************************************************************/

void connect_init(void)
{
  /* Create connection object */
  connection = con_ic_connection_new();

  /* Connect signal to receive connection events */
  g_object_set(G_OBJECT(connection), "automatic-connection-events", TRUE, NULL);
  g_signal_connect(G_OBJECT(connection), "connection-event",
		   G_CALLBACK(connection_handler), &status);

  netconnected = FALSE;
  netconnecting = FALSE;
}

gboolean connect_request(void)
{
  gboolean success = FALSE;

  /* Request connection and check for the result */
  success = con_ic_connection_connect(connection, CON_IC_CONNECT_FLAG_NONE);
  if (!success) g_warning("Request for connection failed");

/* Iterate main loop for the first connection event */
  while (status == 0xFFFF) g_main_context_iteration(NULL, TRUE);
  if (status == CON_IC_STATUS_CONNECTED) {
    g_debug("We are connected!");
    netconnected = TRUE;
  }
  else {
    g_debug("We are not connected!");
    netconnected = FALSE;
  }

  return success;
}

gboolean connect_ensure(void)
{
  gboolean success = FALSE;

  if (!netconnected) {
    g_debug("Not connected.");
    if (!netconnecting) {
      g_debug("Trying to establish connection.");
      netconnecting = TRUE;
      success = con_ic_connection_connect(connection, CON_IC_CONNECT_FLAG_NONE);
      if (!success) {
	g_warning("Request for connection failed");
	netconnecting = FALSE;
      }
    }
  }

  return success;
}
