/* 
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007-2010 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef _FUELPAD_H_
#define _FUELPAD_H_

#include <sys/time.h>

#include <sqlite3.h>

//#define DEBUG
#ifdef DEBUG
#if (__GNUC__ > 2) && ((__GNUC__ > 3) || (__GNUC_MINOR__ > 2))
#define PDEBUG(fmt...) do {						\
		if (debug_level) {					\
			struct timeval tv;				\
			gettimeofday(&tv, 0);				\
			printf("DEBUG[%d]:%ld.%ld:%s:%s():%d: ",	\
			       getpid(),				\
			       tv.tv_sec, tv.tv_usec,			\
			       __FILE__, __FUNCTION__, __LINE__);	\
			printf(fmt);					\
			fflush(stdout);					\
		}							\
	}while(0)
#else
#define PDEBUG(fmt...) do {						\
		if (debug_level) {					\
			struct timeval tv;				\
			gettimeofday(&tv, 0);				\
			printf("DEBUG[%d]:%ld.%ld:%s:%s():%d: ",	\
			       getpid(),				\
			       tv.tv_sec, tv.tv_usec,			\
			       __FILE__, __FUNCTION__, __LINE__);	\
			printf(fmt);					\
			fflush(stdout);					\
		}							\
	}while(0)
#endif
#else
#define PDEBUG(fmt...)
#endif

/*******************************************************************
 *
 * Public defines
 *
 *******************************************************************/

#define DB_OK SQLITE_OK
#define DB_ROW SQLITE_ROW
#define DB_DONE SQLITE_DONE

/*******************************************************************
 *
 * Public type definitions
 *
 *******************************************************************/

enum dbtimespan {
  OVERALL,
  LASTYEAR,
  LASTMONTH,
  SPANEND
};

/*******************************************************************
 *
 * Public variables
 *
 *******************************************************************/

extern int debug_level;
extern unsigned int currentdriver;
extern unsigned int currentcar;
extern sqlite3_stmt *ppStmtOneDriver;
extern sqlite3_stmt *ppStmtCurCar;
extern sqlite3_stmt *ppStmtOneCar;
extern sqlite3_stmt *ppStmtAddDriver;
extern sqlite3_stmt *ppStmtUpdateDriver;
extern sqlite3_stmt *ppStmtAddCar;
extern sqlite3_stmt *ppStmtUpdateCar;
extern sqlite3_stmt *ppStmtOneRecord;
extern sqlite3_stmt *ppStmtRecords;
extern sqlite3_stmt *ppStmtExport;
extern sqlite3_stmt *ppStmtExportCar;
extern sqlite3_stmt *ppStmtDeleteRecord;
extern sqlite3_stmt *ppStmtUpdateRecord;
extern sqlite3_stmt *ppStmtGetReport;
extern sqlite3_stmt *ppStmtAddAlarmtype;
extern sqlite3_stmt *ppStmtGetAlarmtype;
extern sqlite3_stmt *ppStmtGetOneAlarmtype;
extern sqlite3_stmt *ppStmtUpdateAlarmtype;
extern sqlite3_stmt *ppStmtAddEvent;
extern sqlite3_stmt *ppStmtGetEvents;
extern sqlite3_stmt *ppStmtGetOneEvent;
extern sqlite3_stmt *ppStmtDeleteEvent;
extern sqlite3_stmt *ppStmtDeleteEventwithRecordid;
extern sqlite3_stmt *ppStmtUpdateEvent;

/*******************************************************************
 *
 * Public functions
 *
 *******************************************************************/

float db_get_fueltype(int carid);
float calc_co2_emission(float consum, int carid);
void db_setfilename(char *fn);
char *db_getfilename(void);
void db_connect(void);
void db_close(void);
int db_open(void);
float db_get_last_km(void);
float db_get_last_refill(double newkm);
void db_find_prev_full(double km, double *fullfill, double *fullkm, double *fullprice);
sqlite_int64 db_find_next_full(double km, double *fullfill, double *fullkm, double *fullprice);
int db_get_monthly_data(int year, double **x, double **fill, double **trip, double **consum, double **priceperlitre);
float db_get_totalkm(enum dbtimespan timespan);
float db_get_totalfillkm(enum dbtimespan timespan);
float db_get_totalfill(enum dbtimespan timespan);
int db_get_yearly_consum(int **year, int **numfills, double **minconsum, double **avgconsum, double **maxconsum);
int db_get_consum_bin(int year, double minconsum, double maxconsum);
float db_get_totalcost(void);
sqlite_int64 db_get_last_insert_rowid(void);
void db_get_driver(char **driver, int id);
void db_get_current_driver(char **driver);
sqlite_int64 db_get_driverid(int index);
int db_get_numof_drivers(void);
int db_get_numof_records(int id);
int db_get_numof_events(int id);
int db_get_numof_events_with_record(int id);
int db_get_numof_cars(void);
int db_get_numof_carrecords(int id);
int db_get_numof_carevents(int id);
int db_get_numof_alarmtypes(int id);
int db_delete_driver(int id);
int db_delete_car(int id);
void db_get_current_car(char **car);
sqlite_int64 db_get_carid(int index);
int db_estimate_date(float km, char **estdate, char *lastdate, double interval);
float db_estimate_km(char *lastdate, double interval);
void db_get_last_event(int alarmid, char **day, double *km);
float db_get_max(int fieldnum);
void db_get_max_date(char **date);
float db_get_min(int fieldnum);
void db_get_min_date(char **date);
int db_events_exist(int alarmid);
void db_calc_date(char **newdate, char *date, double num, char *unit);
int db_delete_events(int alarmid);
int db_delete_event_with_record_id(int recordid);
int db_delete_reminder(int id);

/*******************************************************************
 * Database access functions for querying the years in the database
 *******************************************************************/
int db_init_years_query(unsigned int carid);
void db_reset_years_query(void);
int db_step_years_query(void);
const unsigned char *db_years_column_text(int col);

/*******************************************************************
 * Database access functions for querying the car details
 *******************************************************************/
int db_init_current_car_query(unsigned int carid);
void db_reset_current_car_query(void);
int db_step_current_car_query(void);
const unsigned char *db_current_car_column_text(int col);

/*******************************************************************
 * Database access functions for querying the drivers in the db
 *******************************************************************/
int db_init_driver_query(void);
void db_reset_driver_query(void);
int db_step_driver_query(void);
const unsigned char *db_driver_column_text(int col);
sqlite_int64 db_driver_column_int64(int col);

/*******************************************************************
 * Database access functions for querying the cars in the db
 *******************************************************************/
int db_init_car_query(void);
void db_reset_car_query(void);
int db_step_car_query(void);
const unsigned char *db_car_column_text(int col);
sqlite_int64 db_car_column_int64(int col);

/*******************************************************************
 * Database access functions for querying the records
 *******************************************************************/
int db_init_record_query(unsigned int carid);
void db_reset_record_query(void);
int db_step_record_query(void);
const unsigned char *db_record_column_text(int col);
sqlite_int64 db_record_column_int64(int col);
double db_record_column_double(int col);
int db_step_record_query_data(const unsigned char **date, /* column 0 */
			      double *km,                 /* column 1 */
			      double *trip,               /* column 2 */
			      double *fill,               /* column 3 */
			      double *consum,             /* column 4 */
			      double *price,              /* column 5 */
			      double *ppl,                /* column 6 */
			      double *ppt,                /* column 7 */
			      double *service,            /* column 8 */
			      double *oil,                /* column 9 */
			      double *tires,              /* column 10 */
			      const unsigned char **notes, /* column 11 */
			      sqlite_int64 *id            /* column 12 */
			      );

/*******************************************************************
 * Database access functions for querying data for a report
 *******************************************************************/
int db_init_report_query(unsigned int carid);
void db_reset_report_query(void);
int db_step_report_query(void);
const unsigned char *db_report_column_text(int col);
double db_report_column_double(int col);

sqlite_int64 db_add_record(char * date,
			   double km,
			   double trip,
			   double fill,
			   double consum,
			   double price,
			   double priceperlitre,
			   double pricepertrip,
			   double service,
			   double oil,
			   double tires,
			   char *notes);
sqlite_int64 db_update_record(sqlite_int64 id,
			      char * date,
			      double km,
			      double trip,
			      double fill,
			      double consum,
			      double price,
			      double priceperlitre,
			      double pricepertrip,
			      double service,
			      double oil,
			      double tires,
			      char *notes);
int db_delete_record(sqlite_int64 id);
void db_close_report_query(void);

/*******************************************************************
 * Database access functions for driving log
 *******************************************************************/

sqlite_int64 db_add_drivinglog(double distance,
			       char *starttime,
			       double startlat,
			       double startlon,
			       char *startplace,
			       char *endtime,
			       double endlat,
			       double endlon,
			       char *endplace,
			       double effectivetime,
			       char *explanation);
sqlite_int64 db_update_trip(sqlite_int64 id,
			    char *startplace,
			    char *endplace,
			    char *explanation);
int db_delete_trip(sqlite_int64 id);
int db_init_log_query(unsigned int carid);
void db_reset_log_query(void);
int db_step_log_query(void);
const unsigned char *db_log_column_text(int col);
sqlite_int64 db_log_column_int64(int col);
double db_log_column_double(int col);
int db_step_log_query_data(double *distance,                  /* column 0 */
			   const unsigned char **starttime,   /* column 1 */
			   const unsigned char **endtime,     /* column 2 */
			   double *startlat,                  /* column 3 */
			   double *startlon,                  /* column 4 */
			   const unsigned char **startplace,  /* column 5 */
			   double *endlat,                    /* column 6 */
			   double *endlon,                    /* column 7 */
			   const unsigned char **endplace,    /* column 8 */
			   double *effectivetime,             /* column 9 */
			   const unsigned char **explanation, /* column 10 */
			   sqlite_int64 *id                   /* column 11 */
			   );

/*******************************************************************
 * Database access functions for location alias
 *******************************************************************/

sqlite_int64 db_add_locationalias(double latitude,
				  double longitude,
				  char *place,
				  char *alias);
sqlite_int64 db_update_location(sqlite_int64 id, char *alias);
int db_delete_location(sqlite_int64 id);
sqlite_int64 db_find_nearest_location(double latitude,
				      double longitude,
				      double radius,
				      char **place,
				      char **alias);
int db_init_location_query(void);
void db_reset_location_query(void);
int db_step_location_query(void);
const unsigned char *db_location_column_text(int col);
sqlite_int64 db_location_column_int64(int col);
double db_location_column_double(int col);
int db_step_location_query_data(
				double *latitude,             /* column 0 */
				double *longitude,            /* column 1 */
				const unsigned char **place,  /* column 2 */
				const unsigned char **alias,  /* column 3 */
				sqlite_int64 *id              /* column 4 */
				);


void fuelpad_exit(void);

#endif
