/* 
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007, 2008 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/**
 *
 * \file
 * \brief  Utility functions to fuelpad
 * \author Julius Luukko
 *
 *
 */

#define _XOPEN_SOURCE
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#define TIMELENMAX 12
#define DATELENMAX 20


/**
 * \fn void convdate(char **s, const char *format, const char *date)
 * \brief Converts date with given format to %Y-%m-%d format
 * \param **s pointer to buffer where converted date will be stored
 * \param *format input date format in the format undestood by strptime
 * \param *date input date
 *
 * This function is used to convert a date in given format to a format
 * that is understood as a date by sqlite3.
 *
 */
void convdate(char **s, const char *outformat, 
	      const char *informat, const char *date)
{
  time_t t1;
  struct tm *tm;

  *s=(char *)malloc((DATELENMAX+1)*sizeof(char));

  time(&t1);
  tm=localtime(&t1);
  strptime(date, informat, tm);
  strftime(*s, DATELENMAX, outformat, tm);
}

void date2sqlite(char **s, const char *format, const char *date)
{
  convdate(s, "%Y-%m-%d", format, date);
}

void gettimefmt(char **aika, const char *format)
{
  time_t t1;
  struct tm *lctime;

  *aika=(char *)malloc((TIMELENMAX+1)*sizeof(char));

  time(&t1);
  lctime=localtime(&t1);
  strftime(*aika, TIMELENMAX, format, lctime);
}

void gettime(char **aika)
{
  gettimefmt(aika, "%F");
}

int getyear(void)
{
  char *aika;
  int ret;

  gettimefmt(&aika, "%Y");
  if (strlen(aika)>0)
    ret=atof(aika);
  else
    ret=2009; /* fallback to something reasonable */
  free(aika);
  return ret;
}

void getyearmonthday(const char *datestr, int *year, int *month, int *day)
{
  char *date;

  convdate(&date, "%Y", "%Y-%m-%d", datestr);
  *year=atoi(date);
  if (date != NULL) free(date);

  convdate(&date, "%m", "%Y-%m-%d", datestr);
  *month=atoi(date);
  if (date != NULL) free(date);

  convdate(&date, "%d", "%Y-%m-%d", datestr);
  *day=atoi(date);
  if (date != NULL) free(date);
}

/**
 * \fn double doubleornothing(double input)
 * \brief Returns zero if input is nan or inf, otherwise returns input
 * \param input input value
 *
 * This function returns zero (0.0) if the input is nan or inf, otherwise
 * the input is returned as is.
 *
 */
double doubleornothing(double input)
{
  double ret;

  if (isnan(input) || isinf(input))
    ret=0.0;
  else
    ret=input;

  return ret;
}
